;;;; -*- Mode:LISP; Syntax:COMMON-LISP; Package:GBB-GRAPHICS; Base:10; Fonts:(MEDFNT) -*-
;;;; *-* File: VAX6:DIS$DISK:[GBB.V-120.LOCAL.GRAPHICS]EXPLORER-GRAPHICS.LISP *-*
;;;; *-* Last-Edit: Friday, July 14, 1989  15:00:16; Edited-By: Gallagher *-*
;;;; *-* Machine: Gilgamesh (Explorer II, Microcode 416) *-*
;;;; *-* Software: TI Common Lisp System 4.105 *-*
;;;; *-* Lisp: TI Common Lisp System 4.105 (0.0) *-*

;;;; ==========================================================================
;;;;
;;;;                    GBB GRAPHICS FUNCTIONS FOR EXPLORER
;;;;
;;;; ==========================================================================
;;;
;;; Written by: Kevin Gallagher and Daniel Corkill
;;;             Department of Computer and Information Science
;;;             University of Massachusetts
;;;             Amherst, Massachusetts 01003.
;;;
;;; This code was written as part of the GBB (Generic Blackboard) system at
;;; the Department of Computer and Information Science (COINS), University of 
;;; Massachusetts, Amherst, Massachusetts 01003.
;;;
;;; Copyright (c) 1987, 1988 COINS.  
;;; All rights are reserved.
;;;
;;; Development of this code was partially supported by:
;;;    Donations from Texas Instruments, Inc.;
;;;    NSF CER grant DCR-8500332;
;;;    ONR URI contract N00014-86-K-0764.
;;;
;;; Permission to copy this software, to redistribute it, and to use it for
;;; any purpose is granted, subject to the following restrictions and
;;; understandings.
;;;
;;; 1.  Title and copyright to this software and any material associated
;;; therewith shall at all times remain with COINS.  Any copy made of this
;;; software must include this copyright notice in full.
;;;
;;; 2.  The user acknowledges that the software and associated materials
;;; are provided as a research tool that remains under active development
;;; and is being supplied ``as is'' for the purposes of scientific
;;; collaboration aimed at further development and application of the
;;; software and the exchange of technical data.
;;;
;;; 3.  All software and materials developed as a consequence of the use of
;;; this software shall duly acknowledge such use, in accordance with the
;;; usual standards of acknowledging credit in academic research.
;;;
;;; 4.  Users of this software agree to make their best efforts to inform
;;; the COINS GBB Development Group of noteworthy uses of this software.
;;; The COINS GBB Development Group can be reached at:
;;;
;;;     GBB Development Group
;;;     C/O Dr. Daniel D. Corkill
;;;     Department of Computer and Information Science
;;;     Lederle Graduate Research Center
;;;     University of Massachusetts
;;;     Amherst, Massachusetts 01003
;;;
;;;     (413) 545-0156
;;;
;;; or via electronic mail:
;;;
;;;     GBB@CS.UMass.Edu
;;;
;;; Users are further encouraged to make themselves known to this group so
;;; that new releases, bug fixes, and tutorial information can be
;;; distributed as they become available.
;;;
;;; 5.  COINS makes no representations or warranties of the merchantability
;;; or fitness of this software for any particular purpose; that uses of
;;; the software and associated materials will not infringe any patents,
;;; copyrights, trademarks, or other rights; nor that the operation of this
;;; software will be error-free.  COINS is under no obligation to provide
;;; any services, by way of maintenance, update, or otherwise.  
;;;
;;; 6.  In conjunction with products or services arising from the use of
;;; this material, there shall be no use of the name of the Department of
;;; Computer and Information Science or the University of Massachusetts in
;;; any advertising, promotional, or sales literature without prior written
;;; consent from COINS in each case.
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;;;
;;;  04-16-87 File Created.  (Cork)
;;;  05-19-87 Extensively revised and extended.  (Gallagher)
;;;
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(in-package 'GBB-GRAPHICS)

(export '(setup-pane))

(proclaim '(optimize (speed 3) (safety 1)))


;;; ---------------------------------------------------------------------------
;;;   Global Variables
;;; ---------------------------------------------------------------------------

(defvar *GBB-GRAPHICS-FRAMES* nil

  "List of GBB grapics panes.")

;;; -----------------------------------------------------------------------

(defvar *UNIT-COLOR-ALIST* nil
  
  "Association list of unit type and color.")

;;; -----------------------------------------------------------------------

(defconstant *LEGEND-LEFT-MARGIN-SIZE* 10

  "Minimum size (in pixels) of the legend left margin.")

;;; -----------------------------------------------------------------------

(defconstant *LEGEND-TOP-MARGIN-SIZE* 2

  "Minimum size (in pixels) of the legend top margin.")

;;; -----------------------------------------------------------------------

(defconstant *LEGEND-RIGHT-MARGIN-SIZE* 4

  "Minimum size (in pixels) of the legend right margin.")

;;; -----------------------------------------------------------------------

(defconstant *LEGEND-BOTTOM-MARGIN-SIZE* 10

  "Minimum size (in pixels) of the legend bottom margin.")

;;; -----------------------------------------------------------------------

(defparameter *LABEL-FONT* fonts:hl12b

  "Font to use for displaying the space label.")

;;; -----------------------------------------------------------------------

(defparameter *PLOT-BAR-SIZE* 1

  "This variable determines the width of the bar used to indicate a scalar
   in one dimension and a range in another.  It's value is one less than
   the width of the bar in pixels.")

;;; -----------------------------------------------------------------------

(defparameter *PLOT-SCALE-MINIMUM-TIC-SEPARATION* 50

  "Minimum number of pixels between adjacent tic marks.")

;;; -----------------------------------------------------------------------

(defparameter *TIC-SIZE* 5

  "Length of a tic mark.")

;;; -----------------------------------------------------------------------

(defparameter *TIC-LABEL-SEPARATION* 4

  "Number of pixels separating the tic label from the tic mark.")

;;; -----------------------------------------------------------------------

(defparameter *FILL-REGIONS* nil

  "Flag to indicate whether 2-D regions should be drawn as filled
   rectangles or outlines.")

;;; -----------------------------------------------------------------------

(defparameter *LINE-THICKNESS* 1

  "Thickness of the lines used to draw blackboard objects.")

;;; -----------------------------------------------------------------------

(defparameter *REFRESH-OFTEN* t

  "If this flag is true, the display will be redrawn when units are
   moved or deleted.")

;;; -----------------------------------------------------------------------

(defparameter *SHOW-SET-LINKS* t

  "Flag to indicate whether links between set elements should be drawn.")

;;; -----------------------------------------------------------------------

(defparameter *SHOW-UNRELATED-LINKS* :minimum

  "This variable indicates whether links between unrelated index elements
   should be drawn.  possible values are as follows:

       :NONE, NIL   No links are drawn.
       :MINIMUM, T  One line is drawn connecting all the index elements.
       :MAXIMUM     A line from each index element is drawn to every other
                    index element.")  

;;; -----------------------------------------------------------------------

(defparameter *FIND-POINT-FUZZ* 0.02

  "Factor used to determine the range of values to search when finding
   units near a point.")

;;; -----------------------------------------------------------------------

(defparameter %%STEPPING-ENABLED%% t

  "Flag which allows stepping to be disabled.  If %%STEPPING-ENABLED%% is
   NIL then no stepping will be done.")


;;; ---------------------------------------------------------------------------
;;;   Flavor Definitions
;;; ---------------------------------------------------------------------------
	     
(defflavor GBB-GRAPHICS-FRAME
           ((graphics-panes nil))
           (tv:window-with-typeout-mixin 
            w:io-buffer-sharers-mixin
            w:choose-configuration-mixin
            w:alias-for-inferiors-mixin
            w:inferiors-not-in-select-menu-mixin
            tv:essential-mouse
            w:bordered-constraint-frame)
  (:gettable-instance-variables graphics-panes)
  (:default-init-plist
    :borders              1
    :save-bits            t
    :activate-p           t
    :expose-p             nil
    :io-buffer-provider   'graphics-1
    :io-buffer-sharers    *GRAPHICS-IO-BUFFER-SHARERS*
    :panes                *GRAPHICS-FRAME-PANES*
    :constraints          (reverse *GRAPHICS-FRAME-CONFIGURATIONS*)
    :typeout-window      `(temporary-typeout-window-with-mouse-sensitive-items
                            :item-type-alist
                            ((unit
                               :UNIT
                               (:mouse-l-1 "Describe this unit"
                                :mouse-m-1 "Set USER::=="))
                             (structure
                               :STRUCTURE
                               (:mouse-l-1 "Describe this unit"
                                :mouse-m-1 "Set USER::=="))))
    ))


(defflavor GBB-GRAPHICS-WINDOW
           ((space-selected-p		nil)
	    (space-instances		nil)
            (user-title         	nil)
	    (dimension-info		nil)
            (x-dim			nil)
	    (x-type			nil)
            (x-min 			0)
            (x-max 			0)
            (x-precision                1)
	    (x-labels 			nil)
	    (x-nlabels 			0)
	    (x-test 			'eq)
            (y-dim 			nil)
	    (y-type 			nil)
            (y-min			0)
            (y-max			0) 
            (y-precision                1)
	    (y-labels			nil)
	    (y-nlabels			0)
	    (y-test			'eq)
	    (legend-left-margin 	0)
	    (legend-top-margin  	0)
	    (legend-right-margin 	0)
	    (legend-bottom-margin 	0)
	    (legend-left-size   	0)
	    (legend-top-size    	0)
	    (legend-right-size  	0)
	    (legend-bottom-size 	0)
            (graph-width		0)
            (graph-height		0)
	    (display-mode 		:all-units)
	    (current-units 		nil)
	    (unit-types 		t)
	    (filter-function		nil)
            (sort-function      	nil)
	    (step-modes 		nil)
            (draw-line-endpoints        t)
            (reverse-x-axis             nil)
            (reverse-y-axis             nil)
            )
           (w:borders-mixin
	    w:top-label-mixin
	    legend-margin-hack-mixin
	    w:locatable-scroll-bar-mixin
	    w:mouse-sensitive-text-scroll-window
            inside-margin-mixin
	    w:process-mixin
	    w:not-externally-selectable-mixin
 	    w:window)
  (:gettable-instance-variables space-instances
				space-selected-p)
  (:default-init-plist
    :save-bits		t
    :blinker-p		nil
    :borders		0
    :label		'(:string "No Current Space"
			  :top :centered :font fonts:hl12b)
    :font-map		'(fonts:cptfont)
    :scroll-bar-side    :left
    :scroll-bar-on-off  :off))

(defflavor gbb-graphics-pane
	   ()
           (gbb-component-window-mixin gbb-graphics-window)
  )


(defmethod (gbb-graphics-frame :after :init) (init-plist)

  "Initializations specific to the GBB graphics frame."

  (declare (ignore init-plist))

  (pushnew self *gbb-graphics-frames*)

  ;; Save list of the graphics panes.
  (setf graphics-panes
	(mapcar #'(lambda (pane)
		      (send self :get-pane pane))
		  *graphics-pane-names*))

  ;; Start the background process running.  This process handles
  ;; blips from the graphics panes and the command-menu.
  (let ((blip-process (send self :send-pane 'graphics-1 :process)))
    (send blip-process :run-reason self)
    (send blip-process :set-priority 5))

  (send self :set-configuration *graphics-initial-configuration*))


(defmethod (gbb-graphics-frame :after :kill) ()
  (setf *gbb-graphics-frames* (delete self *gbb-graphics-frames*)))


(defmethod (gbb-graphics-frame :after :set-configuration) (&rest ignore)
  (declare (ignore ignore))
  (let ((ll (send self :get-pane 'll))
        (g1 (send self :get-pane 'graphics-1)))
    (cond ((send ll :exposed-p)
           (send self :set-selection-substitute ll))
          ((send g1 :exposed-p)
           (send self :set-selection-substitute g1))
          ;; Neither exposed?  Punt.
          (t nil))))

(defmethod (gbb-graphics-frame :name-for-selection) ()
  (or (send self :name)
      "GBB Graphics Frame"))

(defmethod (gbb-graphics-frame :which-pane-am-i) (pane)
  (or (car (rassoc pane tv:internal-panes :test #'eq))
      (error "~s is not a pane in ~s.")))

(defun get-frame-from-window (window)
  (cond ((typep window 'gbb-graphics-frame)
         window)
        ((typep (send window :superior) 'gbb-graphics-frame)
         (send window :superior))
        (t (error "Can't find a GBB Graphics Frame from ~s."
                  window))))


(defflavor gbb-lisp-listener-pane
	()
	(gbb-component-window-mixin w:lisp-listener)
  (:default-init-plist
    :save-bits t))

;;; This avoids the problem that if two frames have been created then
;;; SELECT-L will never get you to another lisp listener.
(defmethod (gbb-lisp-listener-pane :lisp-listener-p) ()
  nil)

(defmethod (gbb-lisp-listener-pane :mouse-click)
           (button x y)
  (declare (ignore x y))
  (cond ((= button #\MOUSE-R-1)
         (process-run-function "GBB Menu"
                               #'funcall-with-top-level-catcher
                               #'do-menu-for-frame
                               w:superior)
         t)
        ((= button #\MOUSE-M-1)
         (process-run-function "GBB Menu"
                               #'funcall-with-top-level-catcher
                               #'(lambda (frame)
                                   (send frame :choose-configuration))
                               w:superior)
         t)
        (t nil)))

;;; -----------------------------------------------------------------------

(defflavor gbb-logo-pane ((first-line "GBB")
			  (more-lines '("University of Massachusetts"
					"Amherst")))
	   (gbb-component-window-mixin
            w:dont-select-with-mouse-mixin
	    w:window-pane)
  (:default-init-plist
    :font-map '(fonts:cmr18 fonts:hl12b)
    :more-p nil
    :blinker-p nil
    :label nil))

(defmethod (gbb-logo-pane :after :refresh) (&optional type)

  (declare (ignore type))

  (when w:restored-bits-p (send self :clear-window))

  (let* ((char-height-0 (w::font-char-height (aref w:font-map 0)))
	 (char-height-1 (w::font-char-height (aref w:font-map 1)))
         (vsp-0 10)
         (vsp-1 2)
	 window-width window-height top-0 top-1)
    (multiple-value-setq (window-width window-height)
      (send self :inside-size))
    ;; Top of line 0.
    (setf top-0 (floor (* (if more-lines .3 .2) window-height)))
    ;; Top of line 1.
    (setf top-1 (floor (min (+ top-0 char-height-0 vsp-0)
                            (* .7 window-height))))

    ;; Display first line.
    (send self :set-current-font 0)
    (send self :string-out-centered-explicit first-line 0 top-0 window-width)
    
    ;; Display additional lines if any.
    (send self :set-current-font 1)
    (do ((current-y top-1
		    (+ current-y char-height-1 vsp-1))
	 (lines more-lines (cdr lines)))
	((or (endp lines)
	     (> (+ current-y char-height-1) window-height)))
      (send self :string-out-centered-explicit
            (first lines) 0 current-y window-width))))

;;; ----------------------------------------------------------------------- 

(defflavor gbb-menu-pane
           ()
           (gbb-component-window-mixin
            w:command-menu)
  (:default-init-plist
    :save-bits t
    :item-list '((" Choose Display Modes All "
		  :value :choose-display-mode-all
		  :documentation "Choose the display mode for all panes.")
		 (" Choose Step Modes All "
		  :value :choose-step-modes-all
		  :documentation "Turn stepping on or off for all panes.")
		 (" Display Current Units All "
		  :value :display-current-units-all
		  :documentation "Display the current units on all the spaces.")
		 (" Display Space Contents All "
		  :value :display-space-contents-all
		  :documentation "Display all the units on all the spaces.")
                 (" Change Configuration "
                  :value :change-configuration
                  :documentation "Select a new frame configuration.")
		 )))


;;; ---------------------------------------------------------------------------
;;;   Mixins:
;;; ---------------------------------------------------------------------------


;;; This mixin exists only for it's :compute-margins method.  I use it
;;; so that I can put it after the border and label mixins so that the
;;; margin regions it establishes are inside those for borders and
;;; labels, but outside the scroll bar margins.  Note that I can't just
;;; put this on the base flavor, gbb-graphics-window, because it's
;;; compute margins will run either first or last, but not in the middle.

(defflavor legend-margin-hack-mixin () ()
  (:required-methods :compute-legend-margins)
  (:documentation
    "Hack to get the legend margins inside the label margins."))

(defmethod (legend-margin-hack-mixin :compute-margins) (lm tm rm bm)
  (send self :compute-legend-margins lm tm rm bm))

;;; -----------------------------------------------------------------------

;;; Another mixin for adjusting margins.  Because I've moved the scroll bar
;;; out of the bordrr margin region, there is no space between the scroll bar
;;; and the text displayed in the window (the items).  This mixin adds an
;;; extra pixel between the scroll bar on the left and the graph axes on the
;;; top, right, and bottom.

(defflavor inside-margin-mixin () ()
  (:required-methods :scrolling-list-of-units?)
  (:documentation
    "Hack to get another margin inside the scroll bar margins."))

(defmethod (inside-margin-mixin :compute-margins) (lm tm rm bm)
  "Leave a little extra space in the case that a list of units
   is being displayed."
  (if (send self :scrolling-list-of-units?)
      (values (1+ lm) (1+ tm) (1+ rm) (1+ bm))
      (values lm tm rm bm)))

;;; -----------------------------------------------------------------------

;;; My own pane mixin.

(defflavor gbb-component-window-mixin () ()
  (:required-flavors w:essential-window)
  (:documentation
    "This mixin provides methods that are required for all the panes."))

(defvar *recursion* nil "Flag used to avoid infinite recursion.")

;;; Normally, if a pane (other than the listener) has no selection
;;; substitute and it receives the :mouse-select method the frame will
;;; get into a weird state.  This wrapper avoids that problem.

(defwrapper (gbb-component-window-mixin :mouse-select) (ignore . body)
  "On mouse-select send the select message to the frame.  There the
   frame will select the proper selection substitute."
  `(if *recursion*
       (progn ,@body)
       (let ((*recursion* t))
         (send w:superior :select))))


;;; ---------------------------------------------------------------------------
;;;   Typeout Window Stuff:
;;; --------------------------------------------------------------------------- 

(defflavor temporary-typeout-window-with-mouse-sensitive-items
	   ()
	   (w:basic-mouse-sensitive-items
            w:temporary-window-mixin
            w:typeout-window))

(defmethod (temporary-typeout-window-with-mouse-sensitive-items :mouse-click)
           (button x y)
  (let ((item (send self :mouse-sensitive-item x y)))
    (or
      (when item
        (let ((item-type (assoc (w:typeout-item-type item) tv:item-type-alist
                                :test #'eq)))
          (when item-type
            ;; Form the blip and stuff it into the keyboard buffer.
            (send self :force-kbd-input
                  (list :typeout-execute (w:typeout-item-item item) self button))
            t)))
      ;; Return T unless this is double-right, to inhibit the blip made by default.
      (not (= button #\Mouse-r-2)))))

(defmethod (temporary-typeout-window-with-mouse-sensitive-items
             :who-line-documentation-string)
           (&aux item item-type x y)
  (multiple-value-setq (x y)
    (w:sheet-calculate-offsets self w:mouse-sheet))
  (setq x (- w:mouse-x x)
	y (- w:mouse-y y))
  (and (setq item (send self :mouse-sensitive-item x y))
       (setq item-type (w:typeout-item-type item))
       (setq item-type (assoc item-type tv:item-type-alist :test #'eq))
       (append (third item-type)
               `(:documentation
                  ,(format nil "~:@(~s~)" (w:typeout-item-item item))))))

(deff process-mouse-sensitive-item 'process-mouse-sensitive-unit)

(defmacro with-typeout-stream ((stream source-window) &body body)
  ;; Note special methods :EXPOSE-FOR-TYPEOUT and :DEACTIVATE
  ;; which are used with typeout windows.
  `(let ((temp-window (send (get-frame-from-window ,source-window) :typeout-window)))
     (cond ((send temp-window :exposed-p)
            (format temp-window "~%~78,,,'-<-~>~%")
            (with-open-stream (,stream temp-window)
              ,@body)
            (format temp-window "~%[Type any character to remove]")
            (read-and-process-typeout-blips temp-window)
            (throw 'top-level-typeout nil))
           (t
            (catch 'top-level-typeout
              (unwind-protect
                  (progn
                    (send temp-window :expose-for-typeout)
                    (send temp-window :clear-input)
                    (with-open-stream (,stream temp-window)
                      ,@body)
                    (format temp-window "~%[Type any character to remove]")
                    (read-and-process-typeout-blips temp-window))
                (send temp-window :deactivate)))))))

(defun read-and-process-typeout-blips (window)
  (let (blip)
    (loop
      (setf blip (send window :any-tyi))
      (cond ((consp blip)
             (process-mouse-sensitive-item blip))
            ((fixnump blip)
             (case blip
               (#.(char-int #\clear-screen)
                (send window :clear-screen))
               (otherwise (return nil))))
            (t (return nil))))))

(defun describe-item-in-typeout-window (item pane)
  (with-typeout-stream (w pane)
    (describe-item item w)))

(defun describe-in-typeout-window (item pane)
  (with-typeout-stream (w pane)
    (describe-item item w)))

(defun describe-item (item &optional (stream *standard-output*))
  "Print a description of ITEM. ITEM should be a unit or a structure."
  (flet ((print-a-slot (slot)
           (let ((value (gbb:get-structure-slot item slot)))
             (format stream "~%~a~36,4t" slot)
             (print-item value stream))))
    (cond ((gbb:unit-instance-p item)
           (let ((description (gbb:get-unit-description item)))
             (format stream "Unit of type ~s.~%~@[~%~s~%~]"
                     (gbb:unit.name description)
                     (gbb:unit.documentation description))
             (print-a-slot 'name)
             (dolist (slot (gbb:unit.slots description))
               (print-a-slot (gbb:slot.name slot)))
             (dolist (link (gbb:unit.links description))
               (print-a-slot (gbb:link.name link)))
             (terpri stream)))
          ((typep item 'structure)
           (let ((structure-type (type-of item)))
             (format stream "Structure of type ~s.~%~@[~%~s~%~]"
                     structure-type
                     (documentation structure-type 'structure))
             (dolist (slot-name (gbb:structure-slot-names (type-of item)))
               (print-a-slot slot-name))
             (terpri stream)))
          (t nil))
    item))

(defparameter *string-buffer* (make-array 255 :element-type 'string-char :fill-pointer 0))

(defun print-item (item stream &optional space-p)
  (let ((*print-array* t)
        (*print-structure* nil))

    (labels ((check-y-bump (item)
               (setf (fill-pointer *string-buffer*) 0)
               (format *string-buffer* "~A " item)
               (multiple-value-bind (x y)
                   (send stream :compute-motion *string-buffer* 0 nil
                         (send stream :cursor-x)
                         (send stream :cursor-y))
                 (declare (ignore x))
                 (when (/= y (send stream :cursor-y))
                   (format stream "~%~36,4t"))))
             (sensitive-print (item type)
               (check-y-bump item)
               (format stream "~VM" type item)
               (when space-p (write-char #\space stream)))
             (normal-print (item)
               (check-y-bump item)
               (format stream "~S" item)
               (when space-p (write-char #\space stream))))
        
      (cond ((or (consp item)
                 (and (vectorp item) (not (or (stringp item) (typep item 'structure)))))
             (when (vectorp item)
               (write-char #\# stream))
             (write-char #\( stream)
             ;; Write all elements except last.
             (map nil #'(lambda (it)
                          (print-item it stream t))
                      (subseq item 0 (1- (length item))))
             ;; Write last element.
             (print-item (elt item (1- (length item))) stream)
             (write-char #\) stream))
            ((gbb:unit-instance-p item)
             (sensitive-print item 'unit))
            ((typep item 'structure)
             (sensitive-print item 'structure))
            (t
             (normal-print item))))))


;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :after :init) (&rest ignore)
  (send self :set-title nil)
  (send self :resize-graph))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :background-process) ()
  (send w:superior :send-pane 'graphics-1 :process))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :my-pane-number) ()
  "Returns the number of this pane as a string."
  (let ((pane-name (send w:superior :which-pane-am-i self)))
    (subseq (symbol-name pane-name) 9)))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :after :refresh) (&optional type)

  (cond
    ;; If the old bits are ok then don't do anything.
    ((eq type :use-old-bits)
     nil)
    ;; Otherwise, we need to refresh the window...
    (t
     ;; If the old bit array was displayed then erase it.
     (when (and w:restored-bits-p
                (not (send self :scrolling-list-of-units?)))
       (send self :clear-graph-region))
     (send self :refresh-contents nil))))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :after :refresh-margins) (&rest ignore)
  (send self :refresh-legends))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :after :change-of-size-or-margins) (&rest ignore)
  (send self :set-title user-title)
  (send self :resize-graph)
  ;; Fix up the transformation matrix so that (0, 0) corresponds
  ;; to the top left of the inside margins.
  (send self :default-window)
  
  (send self :pan (w:sheet-left-margin-size self)
                  (w:sheet-top-margin-size self)))

;;; -----------------------------------------------------------------------

;;#+EXPLORER
;;(ticl::defmethod (gbb-graphics-window :after :draw-rectangle)
;;           (left top r-width r-height &optional
;;                                      (thickness 1)
;;                                      (color (if (w:color-system-p self)
;;                                                 (tv:sheet-foreground-color self)
;;                                                 w:black))
;;                                      (alu w:normal))
;;  ;; The Explorer draw-rectangle method has a bug which leaves out
;;  ;; the final pixel in the rectanlge.
;;
;;  (let ((x (+ left r-width))
;;        (y (+ top r-height)))
;;    (send self :draw-line (1- x) y x y thickness color alu)))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :resize-graph) ()
  "Save the width and height of the data portion of the graph."
  (multiple-value-bind (display-width display-height)
      (send self :inside-size)
    (setf graph-width display-width)
    (setf graph-height display-height)))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :clear-graph-region) ()
  "Clear the data region of the graph."
  ;; :Clear-Window only clear the inside portion of the window.
  (send self :clear-window)
  (send self :draw-axes))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :compute-legend-margins) (lm tm rm bm)

  "Add additional room for legends, etc."

  (let ((char-height 11)
	(char-width 8))

    ;; :COMPUTE-MARGINS is called very early in the window's
    ;; initialization process but the instance variable CURRENT-FONT
    ;; isn't initialized until later.  So, if CURRENT-FONT isn't bound
    ;; then assume the font will be cptfont (height=11, width=8).

    (when (variable-boundp w:current-font)
      (setf char-height (w::font-char-height w:current-font)
	    char-width (w::font-char-width w:current-font)))

    ;; Save the margins preceeding (outside of) the legends.
    (setf legend-left-margin lm
	  legend-top-margin tm
	  legend-right-margin rm
	  legend-bottom-margin bm)
  
    ;; Compute the left legend size.  It must be at least
    ;; *legend-left-margin-size* but not more than 20% of the
    ;; width of the pane.
    (setf legend-left-size
	  (cond ((null y-dim)
		 *legend-left-margin-size*)
		(t
		 (let ((max-length
			 (ecase y-type
			   ((:ordered) (max (length (format nil "~d" y-min))
					    (length (format nil "~d" y-max))))
			   ((:enumerated)
			    (mapc-max #'(lambda (label)
                                          (length (format nil "~a" label)))
				      y-labels)))))
		   (max *legend-left-margin-size*
			(+ (min (floor w:width 5)  ; 20%
				(+ 2 (* max-length char-width)))
			   (* 2 char-width)         ; width of the axis label
			   *tic-size*
			   *tic-label-separation*))))))

    ;; The rest of the legends...
    (setf legend-top-size
	  (if (send self :scrolling-list-of-units?)
	      (+ *legend-top-margin-size* 4)
	      *legend-top-margin-size*))

    (setf legend-right-size *legend-right-margin-size*)

    (setf legend-bottom-size
	  (if (null x-dim)
	      *legend-bottom-margin-size*
	      (max *legend-bottom-margin-size*
		   (+ (* 2 char-height)
		      *tic-size*
		      *tic-label-separation*))))

    ;; Return the new margins.
    (values
      (+ lm legend-left-size)
      (+ tm legend-top-size)
      (+ rm legend-right-size)
      (+ bm legend-bottom-size))))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :draw-axes) ()

  ;; The axes are drawn on the innermost bits that are
  ;; still in the margins.
  (let* ((left   (+ legend-left-margin legend-left-size -1))
         (top    (+ legend-top-margin legend-top-size -1))
         (right  (- w:width legend-right-margin legend-right-size))
         (bottom (- w:height legend-bottom-margin legend-bottom-size)))
    (send self :draw-line-explicit left top right top)
    (send self :draw-line-explicit right top right bottom)
    (send self :draw-line-explicit right bottom left bottom)
    (send self :draw-line-explicit left bottom left top)))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :refresh-legends) ()

  ;; The legend region (which is a margin) should already be cleared 
  ;; by (:method w:sheet :refresh).

  ;; Draw the axes:
  (send self :draw-axes)

  ;; Draw the axis labels only if a space is selected.
  (when space-selected-p

    (let* ((font w:current-font)
	   (left (1- (w:sheet-inside-left)))
	   (top (1- (w:sheet-inside-top)))
	   (right (w:sheet-inside-right))
	   (bottom (w:sheet-inside-bottom))
	   (x-legend-top (+ bottom *tic-size* *tic-label-separation*
			    (w::font-char-height font))))

      ;; X Axis Labels ::
      (when x-dim
	(send self :draw-x-tics left top right bottom)
	(send self :STRING-OUT-EXPLICIT
	      (symbol-name x-dim)
	      (- (floor (+ left right) 2)
		 (floor (send self :string-length (symbol-name x-dim)) 2))
	      x-legend-top
	      nil nil font w:char-aluf))
      
      ;; Y Axis Labels ::
      (when y-dim
	(send self :draw-y-tics left top right bottom)
	(send self :draw-y-legend
	      (symbol-name y-dim)
	      (+ legend-left-margin 3)
	      (- (floor (+ top bottom) 2) 6)))
      )))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :draw-y-legend) (string x-pos y-pos
						 &optional (font w:current-font)
							   (alu w:char-aluf))
  "Draws string centered on y-pos.  The string is drawn in `vertical'
   format."
  (let* ((length (length (string string)))
	 (char-height (w::font-char-height font))
	 (total-height (* length char-height)))
    (do ((y (- y-pos (floor total-height 2))
	    (+ y char-height 1))
	 (i 0 (1+ i)))
	((>= i length))
      (send self :string-out-explicit
	    string
	    x-pos y nil nil
	    font alu i (1+ i)))))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :draw-x-tics) (left top right bottom)
  (send self
	(ecase x-type
	  ((:ordered) :draw-ordered-x-tics)
	  ((:enumerated) :draw-enumerated-x-tics))
	left top right bottom))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :draw-ordered-x-tics) (left top right bottom)
  (declare (ignore top))
  (let* ((tic-increment (compute-tic-interval graph-width x-min x-max x-precision))
	 (tic-top bottom)
	 (tic-limit (- right *plot-scale-minimum-tic-separation* -1)))

    ;; Tics for minimum and maximum::
    (send self :draw-x-tic (tic-label-string (if reverse-x-axis x-max x-min))
                           left tic-top :left)
    (send self :draw-x-tic (tic-label-string (if reverse-x-axis x-min x-max))
                           right tic-top :right)

    ;; Intervening tics.
    (do* ((x-value (+ x-min tic-increment) (+ x-value tic-increment))
	  (x-pixel (+ left (send self :x-world-to-pixel x-value))
		   (+ left (send self :x-world-to-pixel x-value))))
	 ((or (>= x-value x-max)
	      (>= x-pixel tic-limit)))
      (send self :draw-x-tic (tic-label-string x-value) x-pixel tic-top))))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :draw-enumerated-x-tics) (left top right bottom)
  (declare (ignore top right))
  (let* ((tic-increment (floor graph-width (1+ x-nlabels)))
	 (tic-top bottom))
    (do ((labels x-labels (cdr labels))
	 (x-pixel (+ left tic-increment) (+ x-pixel tic-increment)))
	((endp labels))
      (send self :draw-x-tic
	    (format nil "~a" (first labels)) x-pixel tic-top))))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :draw-x-tic) (tic-label tic-x tic-top
					      &optional (place :center)
					                (alu w:char-aluf))
  "Draw and label a single tic mark."
  (let* ((tic-bottom (+ tic-top *tic-size*))
	 (tic-label-width (send self :string-length tic-label))
	 (label-left (case place
			((:left)   tic-x)
			((:right)  (- tic-x tic-label-width))
			(otherwise (- tic-x (ceiling tic-label-width 2)))))
	 (label-top (+ tic-bottom 3)))
    (send self :draw-line-explicit tic-x tic-top tic-x tic-bottom alu)
    (send self :string-out-explicit
	  tic-label label-left label-top nil nil w:current-font alu)))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :draw-y-tics) (left top right bottom)
  (send self
	(ecase y-type
	  ((:ordered) :draw-ordered-y-tics)
	  ((:enumerated) :draw-enumerated-y-tics))
	left top right bottom))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :draw-ordered-y-tics) (left top right bottom)

  (declare (ignore right))
  (let* ((tic-increment (compute-tic-interval graph-height y-min y-max y-precision))
	 (tic-right left)
	 (tic-limit (+ top *plot-scale-minimum-tic-separation* -1)))

    ;; Tics for minimum and maximum::
    (send self :draw-y-tic (tic-label-string (if reverse-y-axis y-max y-min))
                           bottom tic-right :bottom)
    (send self :draw-y-tic (tic-label-string (if reverse-y-axis y-min y-max))
                           top tic-right :top)
	  
    ;; Intervening tics.
    (do* ((y-value (+ y-min tic-increment) (+ y-value tic-increment))
	  (y-pixel (+ top (send self :y-world-to-pixel y-value))
		   (+ top (send self :y-world-to-pixel y-value))))
	 ((or (>= y-value y-max)
	      (<= y-pixel tic-limit)))
      (send self :draw-y-tic (tic-label-string y-value) y-pixel tic-right))))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :draw-enumerated-y-tics) (left top right bottom)
  (declare (ignore top right))
  (let* ((tic-increment (floor graph-height (1+ y-nlabels)))
	 (tic-right left))
    (do ((labels y-labels (cdr labels))
	 (y-pixel (- bottom tic-increment) (- y-pixel tic-increment)))
	((endp labels))
      (send self :draw-y-tic
	    (format nil "~a" (first labels)) y-pixel tic-right))))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :draw-y-tic) (tic-label tic-y tic-right
					      &optional (place :center)
					                (alu w:char-aluf))
  "Draw and label a single tic mark."
  (let* ((tic-left (- tic-right *tic-size*))
	 (font-height (w::font-char-height w:current-font))
	 (font-width (w::font-char-width w:current-font))
	 (tic-label-width (send self :string-length tic-label))
	 (label-top (case place
		       ((:top)    tic-y)
		       ((:bottom) (- tic-y font-height))
		       (otherwise (- tic-y (ceiling font-height 2)))))
	 (label-left-margin (+ legend-left-margin (* 2 font-width)))
	 (label-right-max (- tic-left *tic-label-separation*))
	 (label-left (max label-left-margin
			  (- label-right-max
			     tic-label-width))))
    (send self :draw-line-explicit tic-left tic-y tic-right tic-y alu)
    (send self :string-out-explicit
	  tic-label label-left label-top label-right-max nil w:current-font alu)))

;;; -----------------------------------------------------------------------

(defun tic-label-string (number)
  "Return a string for NUMBER suitable for use as a tic label."
  (cond ((typep number 'integer)
         (format nil "~d" number))
        (t (let ((string (format nil "~f" number)))
             (if (< (length string) 7)
                 string
                 (format nil "~6f" number))))))

;;; -----------------------------------------------------------------------

(defun compute-tic-interval (axis-length data-min data-max precision)

  "Compute the distance (in data units) between tic marks.  AXIS-LENGTH
   is the length of the axis in pixels.  DATA-MIN and DATA-MAX are the
   lower and upper bounds of the data to be displayed.  PRECISION is the
   precision of the bounds.  Returns the increment between tic marks."

  (let* ((data-interval (* (- data-max data-min) (/ 10 precision)))
	 (start-n-tics (floor axis-length *plot-scale-minimum-tic-separation*))
	 (n-tics start-n-tics)
	 quotient remainder)
    (when (zerop n-tics)
      (return-from compute-tic-interval (- data-max data-min)))
    (loop
      (multiple-value-setq (quotient remainder)
        (floor data-interval n-tics))
      (cond 
        ;; If the remainder is zero then this is a
        ;; `good' choice of tic interval.
        ((zerop remainder)
         (return (* quotient (/ precision 10))))
        ;; If n-tics is 2 (or less) then there is no `good'
        ;; choice for the tic intervals; so make do with
        ;; the original guess. (The scale won't be uniform.)
        ((<= n-tics 2)
         (return (if (= precision 1)
                     (ceiling data-interval start-n-tics)
                     (compute-tic-interval axis-length data-min data-max 1))))
        ;; Else, try one fewer tics.
        (t (decf n-tics))))))

;;; -----------------------------------------------------------------------

(defun fixup-bounds (min max)

  "Compute `nice' values for the upper and lower bounds of a dimension.
   MIN and MAX are the lower and upper bounds of the data to be displayed.
   Returns three values: the new lower bound, the new upper bound, and the
   precision."

  (flet ((float-if-necessary (number)
           (if (typep number 'integer)
               number
               (float number))))
    (let* ((width (abs (- max min)))
           power precision new-max new-min)
      (cond ((< width 1)
             (setf power (ceiling (log (/ 1 width) 10)))
             (setf precision (/ 1 (expt 10 power))))
            (t
             (setf power (floor (log width 10)))
             (setf precision (expt 10 power))))
      (setf new-min (* (floor min precision) precision))
      (setf new-max (* (ceiling max precision) precision))
      (values (float-if-necessary new-min)
              (float-if-necessary new-max)
              (float-if-necessary precision)))))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :set-title) (&optional new-title)
  (let ((title-string
          (cond ((not space-selected-p)
                 "No Current Space")
                ((not new-title)
                 (pretty-space-path space-instances self *label-font*))
                ((stringp new-title)
                 new-title)
                (t (format nil "~s" new-title)))))
    (send self :SET-LABEL
          `(:TOP :CENTERED :FONT ,*label-font* :STRING ,title-string))))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :refresh-contents) (&optional
						    (clear-first? t)
						    (step? nil))
  "Redraw the contents of the pane."

  (flet ((display-and-wait (unit)
           (send self :display-unit unit nil t :refresh)
           (when (and %%stepping-enabled%% step?)
             (wait-for-step-proceed self))))

    (when (and space-selected-p w:exposed-p)

      ;; Clear the data region of the graph
      (when (and clear-first? (not (send self :scrolling-list-of-units?)))
        (send self :clear-graph-region))

      ;; Display the unit(s).
      (ecase display-mode
        ((:last-unit)
         (when (and current-units (first current-units))
           (send self :display-unit (first current-units) nil t :refresh)))
        ((:recently-added-units)
         (mapc #'display-and-wait current-units))
        ((:all-units :whole-space)
         (mapc #'(lambda (si)
                   (gbb::map-space-instance #'display-and-wait unit-types si))
               space-instances)))

      (when (send self :scrolling-list-of-units?)
        (send self :put-item-in-window 1)))))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :refresh-entire-display)
           (&optional stepping-enabled?)
  (let ((%%stepping-enabled%% stepping-enabled?))
    (send self :refresh :complete-redisplay)))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :set-space)
           (new-space-instances
	    new-dimension-info
            new-x-dim
            new-y-dim
            &optional
	    (new-display-mode 	   :all-units)
	    (new-step-modes 	   nil)
	    (new-unit-types 	   t)
            (new-filter-function   nil)
            (new-sort-function     nil)
            (new-user-title        nil)
            new-x-min new-x-max new-x-labels
	    new-y-min new-y-max new-y-labels)

  "Associate this pane with a new space instance."

  (when (and space-selected-p
             (send self :scrolling-list-of-units?))
    (send self :delete-all-items))

  (setf space-instances (gbb::assure-list new-space-instances))
  (setf dimension-info new-dimension-info)

  (send self :set-dimensions new-x-dim new-y-dim
                             new-x-min new-x-max new-x-labels
                             new-y-min new-y-max new-y-labels)
  (send self :set-attributes new-display-mode
                             new-step-modes
                             new-unit-types
                             new-filter-function
                             new-sort-function)
  (setf space-selected-p t)

  (setf user-title new-user-title)
  (send self :set-title user-title)
  (send self :redefine-margins)
  (send self :set-current-units nil)

  (when w:exposed-p
    (send self :refresh-entire-display)))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :same-space) (new-space-instances)

  "Make this pane display NEW-SPACE-INSTANCES which must all be
   have at least the two current dimensions."

  (unless space-selected-p
    (command-loop-user-error
      "No space associated with this pane, ~s." self))

  (setf new-space-instances (gbb::assure-list new-space-instances))

  (let ((new-dimension-info (get-common-dimensions new-space-instances)))

    (unless (and
              (or (null x-dim)
                  (find x-dim new-dimension-info :key #'first :test #'string=))
              (or (null y-dim)
                  (find y-dim new-dimension-info :key #'first :test #'string=)))
      (command-loop-user-error
        "Some spaces don't have the necessary dimensions.~@
         The current dimensions are ~a and ~a.~@
         The new space instances are ~s."
        x-dim y-dim new-space-instances))

    (send self :set-space new-space-instances
                          new-dimension-info
                          x-dim y-dim
                          display-mode
                          step-modes
                          unit-types
                          filter-function
                          sort-function
                          user-title
                          x-min x-max x-labels
                          y-min y-max y-labels)))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :set-attributes)
           (&optional
	    (new-display-mode    :all-units)
	    (new-step-modes      nil)
	    (new-unit-types      t)
            (new-filter-function nil)
            (new-sort-function   nil))

  "Set attributes such as display mode for this pane.  This doesn't
   alter the dimensions, or redisplay.  If any argument is omitted
   that attribute is set to its default value."

  (send self :set-display-mode new-display-mode)
  (send self :set-step-modes new-step-modes)
  (send self :set-unit-types new-unit-types)
  (send self :set-filter-function new-filter-function)
  (send self :set-sort-function new-sort-function))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :set-current-units) (new-current-units)

  "Update the list of current units for this pane.  The argument,
   NEW-CURRENT-UNITS, is copied and any units that don't satisfy
   UNIT-TYPES are removed."

  (setf current-units
	(cond ((null new-current-units)
	       nil)
	      ((eq unit-types t)
	       (copy-list new-current-units))
	      (t
	       (remove-if-not #'(lambda (u) (member (type-of u) unit-types))
			      new-current-units))))

  (when (send self :scrolling-list-of-units?)
    (send self :delete-all-items)
    (send self :set-items current-units)))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :set-dimensions)
           (new-x-dim new-y-dim
	    &optional
	    new-x-min new-x-max new-x-labels
	    new-y-min new-y-max new-y-labels)            

  "Set which dimensions to display.  This doesn't redisplay or change
   any other attributes.  If any of the optional arguments are omitted
   or nil maximal values will be used."

  (let ((x-dim-info (assoc new-x-dim dimension-info :test #'string=))
	(y-dim-info (assoc new-y-dim dimension-info :test #'string=)))
    
    (when (or (and new-x-dim (not x-dim-info))
              (and new-y-dim (not y-dim-info)))
      (error "~s and/or ~s cannot be displayed on this pane.~@
              The dimensions that can be displayed are:~:{ ~a~}."
             new-x-dim new-y-dim dimension-info))

    (flet
      ((set-dimension (dim-info new-min new-max new-labels)
         (let ((new-type (dimension-info.dimension-type dim-info))
               new-precision new-test new-nlabels)
           (ecase new-type
             ((:ordered)
              (unless new-min (setf new-min (first (dimension-info.dimension-arg dim-info))))
              (unless new-max (setf new-max (second (dimension-info.dimension-arg dim-info))))
              (multiple-value-bind (i j precision)
                  (fixup-bounds new-min new-max)
                (declare (ignore i j))
                (setf new-precision precision))
              ;; Reset these for neatness.
              (setf new-labels nil new-test 'eq new-nlabels 0))
             ((:enumerated)
              (when (null new-labels)
                (setf new-labels (dimension-info.dimension-arg dim-info)))
              (setf new-test (getopt (dimension-info.dimension-options dim-info)
                                     :test))
              (setf new-nlabels (length new-labels))
              ;; MIN is 0 and MAX is (1 + N).  These correspond to the
              ;; edges -- no enumerated labels appear on the axes.
              (setf new-min 0 new-max (1+ new-nlabels))))
           (values new-type new-min new-max new-precision new-labels new-test new-nlabels))))
         
      (setf x-dim new-x-dim)
      (when x-dim
        (multiple-value-bind (new-type new-min new-max new-precision new-labels new-test new-nlabels)
            (set-dimension x-dim-info new-x-min new-x-max new-x-labels)
          (setf x-type      new-type
                x-min       new-min
                x-max       new-max
                x-precision new-precision
                x-labels    new-labels
                x-test      new-test
                x-nlabels   new-nlabels)))

      (setf y-dim new-y-dim)
      (when y-dim
        (multiple-value-bind (new-type new-min new-max new-precision new-labels new-test new-nlabels)
            (set-dimension y-dim-info new-y-min new-y-max new-y-labels)
          (setf y-type      new-type
                y-min       new-min
                y-max       new-max
                y-precision new-precision
                y-labels    new-labels
                y-test      new-test
                y-nlabels   new-nlabels)))

      (send self :set-scrolling-list-of-units
            (and (null x-dim) (null y-dim)))
      (unless (send self :scrolling-list-of-units?)
        (warn-about-unit/dimension-mismatch (if (eq unit-types t)
                                                (all-space-unit-types space-instances)
                                                unit-types)
                                            x-dim
                                            y-dim)))))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :choose-dimensions) ()

  "Pop up a menu to select which dimensions to display."

  (unless space-selected-p
    (command-loop-user-error "This pane is not associated with any space."))
  (let ((scrolling-unit-list? (and (null x-dim) (null y-dim))))
    (multiple-value-bind (new-x-dim new-y-dim)
        (select-dimensions dimension-info
                           space-instances
                           x-dim
                           y-dim)
      ;; If this was a list of units before and is not now then
      ;; we have to delete all the items from the text scroll window.
      (when (and scrolling-unit-list?
                 (or new-x-dim new-y-dim))
        (send self :delete-all-items))
      (send self :set-dimensions new-x-dim new-y-dim)
      (send self :redefine-margins)
      (send self :refresh-entire-display))))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :set-axis-bounds) ()

  (unless space-selected-p
    (command-loop-user-error "This pane is not associated with any space."))
  (when (and (null x-dim) (null y-dim))
    (command-loop-user-error "This pane is not displaying any dimensions."))

  (let* ((new-x-min x-min)
         (new-x-max x-max)
         (new-x-labels x-labels)
         (new-y-min y-min)
         (new-y-max y-max)
         (new-y-labels y-labels)
         (cvv-items (list "")))
    (declare (special new-x-min new-x-max new-x-labels
		      new-y-min new-y-max new-y-labels))

    (flet ((add-cvv-items (dimension type min-sym max-sym labels-sym)
	    (when dimension
	      (let* ((info (assoc dimension dimension-info :test #'string=))
		     min max labels doc)
		(ecase type
		  ((:ordered)
		   (setf min (first (dimension-info.dimension-arg info))
			 max (second (dimension-info.dimension-arg info))
			 doc (format nil "Enter a number between ~d and ~d."
				     min max))
		   (push `(,min-sym
			    ,(format nil "~a Dimension Lower Bound"
				     (symbol-name dimension))
			    :number-between ,min ,max
			    :documentation ,doc)
			 cvv-items)
		   (push `(,max-sym
			    ,(format nil "~a Dimension Upper Bound"
				     (symbol-name dimension))
			    :number-between ,min ,max
			    :documentation ,doc)
			 cvv-items))
		  ((:enumerated)
		   (setf labels (dimension-info.dimension-arg info))
		   (push `(,labels-sym
			    ,(format nil "~a Dimension Labels" (symbol-name dimension))
			    :set ,labels
			    :documentation "Click left to add or delete a label.")
			 cvv-items)))
		(push "" cvv-items)))))

      (add-cvv-items x-dim x-type 'new-x-min 'new-x-max 'new-x-labels)
      (add-cvv-items y-dim y-type 'new-y-min 'new-y-max 'new-y-labels)

      (setf cvv-items (nreverse cvv-items))

      (loop
	(w:choose-variable-values
	  cvv-items
	  :label `(:string ,(format nil " Bounds for ~a "
				    (gbb::space-instance.name
                                      (first space-instances)))
		   :centered :font fonts:medfnb)
	  :margin-choices '("Exit" ("Abort" (throw 'top-level nil)))
	  :extra-width 10)
	(unless (or (and (eq x-type :enumerated) (null new-x-labels))
		    (and (eq y-type :enumerated) (null new-y-labels)))
          (send self :set-dimensions x-dim y-dim
                new-x-min new-x-max new-x-labels
                new-y-min new-y-max new-y-labels)
          (send self :redefine-margins)
          (send self :refresh-entire-display)
          (return t))
	(gbb-graphics-warning
	  "You must choose at least one label for ~a."
	  (if (and (eq x-type :enumerated) (null new-x-labels))
	      x-dim
	      y-dim))))))
	    
;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :copy-dimensions) ()
	    
  (unless space-selected-p
    (command-loop-user-error "This pane is not associated with any space."))
  (let ((other-pane
	  (w:point-at-a-window '(:mouse-l-1 "Choose a pane"
                                 :mouse-m-1 "Abort"
                                 :mouse-r-1 "Choose a pane"))))
    (when (null other-pane)
      (throw 'top-level nil))
    (unless (member other-pane (send w:superior :graphics-panes))
      (command-loop-user-error "Choose a graphing pane."))
    (when (and (send other-pane :scrolling-list-of-units?)
               (not (send self :scrolling-list-of-units?)))
      (send other-pane :delete-all-items))
    (send other-pane :set-dimensions
          x-dim
          y-dim
          x-min x-max x-labels
          y-min y-max y-labels)
    (send other-pane :redefine-margins)
    (send other-pane :refresh-entire-display)))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :set-unit-types) (new-unit-types)
  (setf unit-types
        (cond ((or (listp new-unit-types) (eq new-unit-types t))
               new-unit-types)
              ((symbolp new-unit-types)
               (list new-unit-types))
              (t (command-loop-user-error
                   "~s is not a unit type." new-unit-types))))
  (unless (send self :scrolling-list-of-units?)
    (warn-about-unit/dimension-mismatch (if (eq unit-types t)
                                            (all-space-unit-types space-instances)
                                            unit-types)
                                        x-dim
                                        y-dim)))

(defmethod (gbb-graphics-window :choose-unit-types) ()
  (unless space-selected-p
    (command-loop-user-error "This pane is not associated with any space."))
  (let* ((all-units (all-space-unit-types space-instances))
         (item-list `((" Select All Unit Types "
                       :execute
                       (funcall-self :set-highlighted-items
                                     (copy-list ',all-units))
                       :documentation "Select all Unit Types."
                       :font fonts:hl12i)
                      ("" :no-select t)
                      ,@all-units))
         (highlighted-items (copy-list (if (eq unit-types t)
                                           all-units
                                           unit-types)))
         (choices
           (w:menu-choose
             item-list
             :label '(:string " Unit Types " :centered :font fonts:medfnb)
             :highlighting t
             :highlighted-items highlighted-items
             :menu-margin-choices '(:doit))))
    (when choices
      (setf unit-types choices)
      (send self :refresh-contents))))

;;; -----------------------------------------------------------------------

(defun warn-about-unit/dimension-mismatch (unit-types x-dim y-dim)
  "Print a warning if the units in UNIT-TYPES do not have both
   dimensional indexes X-DIM and Y-DIM."

  (let ((bad-units (remove-if
                     #'(lambda (unit)
                         (and (if x-dim (unit-has-dimension unit x-dim) t)
                              (if y-dim (unit-has-dimension unit y-dim) t)))
                     unit-types)))
    (cond ((null bad-units)
           nil)
          ((or (null x-dim) (null y-dim))
           (gbb-graphics-warning
             "The following units are missing the dimensional index ~a~@
              so they will not be displayed.~{~%~8t~s~}"
             (or x-dim y-dim) bad-units))
          (t
           (gbb-graphics-warning
             "The following units are missing one or both~@
              dimensional indexes ~a and ~a~@
              so they will not be displayed.~{~%~8t~s~}"
              x-dim y-dim bad-units)))))

;;; -----------------------------------------------------------------------

(defun function-or-nil-p (x)
  (or (null x) (functionp x)))

(defmethod (gbb-graphics-window :set-filter-function) (new-filter-function)
  (unless (function-or-nil-p new-filter-function)
    (command-loop-user-error "~s is not a function."))
  (setf filter-function new-filter-function))

(defmethod (gbb-graphics-window :choose-filter-function) ()
  (let ((new-filter-function filter-function))
    (declare (special new-filter-function))
    (w:choose-variable-values
      '(""
	(new-filter-function "Filter Function"
	  :documentation "Enter a function or NIL."
	  :test function-or-nil-p)
	"")
      :label '(:string "Choose a filter function."
	       :centered :font fonts:medfnt)
      :extra-width 55)
    (setf filter-function new-filter-function)))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :set-sort-function) (new-sort-function)
  (unless (function-or-nil-p new-sort-function)
    (command-loop-user-error "~s is not a function."))
  (setf sort-function new-sort-function))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :set-display-mode) (new-display-mode)
  (setf display-mode new-display-mode)
  (when (eq display-mode :last-unit)
    (if (consp current-units)
	(setf (cdr current-units) nil)
	(setf current-units (cons nil nil)))))

(defmethod (gbb-graphics-window :choose-display-mode) ()
  (let ((choice (choose-display-mode-menu)))
    (when choice
      (send self :set-display-mode choice))))
	    
(defmethod (gbb-graphics-window :choose-display-mode-all) ()
  (let ((choice (choose-display-mode-menu)))
    (when choice
      (dolist (pane (send w:superior :graphics-panes))
	(send pane :set-display-mode choice)))))

(defun choose-display-mode-menu ()
  (w:menu-choose
    '((" Last Unit Added " :value :last-unit
       :documentation "Display only the last unit added to the space.")
      (" Recently Added Units " :value :recently-added-units
       :documentation "Display each unit as it is added to the space.")
      (" All Units on Space " :value :all-units
       :documentation "Display all the units on the space."))
    :label '(:string " Display Mode " :centered :font fonts:medfnb)
    :item-alignment :center))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :set-step-modes) (new-step-modes)
  (setf step-modes (gbb::assure-list new-step-modes)))

(defmethod (gbb-graphics-window :choose-step-modes) ()
  (multiple-value-bind (choices choice-was-made)
      (choose-step-mode-menu step-modes)
    (when choice-was-made
      (send self :set-step-modes choices))))
	    
(defmethod (gbb-graphics-window :choose-step-modes-all) ()
  (multiple-value-bind (choices choice-was-made)
      (choose-step-mode-menu step-modes)
    (when choice-was-made
      (dolist (pane (send w:superior :graphics-panes))
	(send pane :set-step-modes choices)))))

(defun choose-step-mode-menu (step-modes)
  (let* ((some-items
	   '((" Step After Unit Addition "
	      :value :unit-addition
	      :documentation "Pause after a unit is added to this space.")
	     (" Step During Refresh "
	      :value :display-all
	      :documentation
	      "Pause after each unit is displayed during any `Display' operation."))))
    (w:mmc-all/none some-items
                    :highlighted-values step-modes 
                    :label '(:string " Step Modes " :centered :font fonts:medfnb)
                    :all-item " Turn on all stepping. "
                    :none-item " Turn off all stepping. ")))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :set-pane-options) (options)
  (gbb::dolist-by-twos (key value options)
    (ecase key
      (:reverse-x-axis (setf reverse-x-axis value))
      (:reverse-y-axis (setf reverse-y-axis value))
      (:draw-line-endpoints (setf draw-line-endpoints value)))))

(defmethod (gbb-graphics-window :choose-pane-options) ()
  (let ((new-options (choose-pane-options-menu reverse-x-axis
                                               reverse-y-axis
                                               draw-line-endpoints)))
    (when new-options
      (send self :set-pane-options new-options)
      (send self :refresh-entire-display))))

(defmethod (gbb-graphics-window :choose-pane-options-all) ()
  (let ((new-options (choose-pane-options-menu reverse-x-axis
                                               reverse-y-axis
                                               draw-line-endpoints)))
    (when new-options
      (dolist (pane (send w:superior :graphics-panes))
        (send pane :set-pane-options new-options)
        (send pane :refresh-entire-display)))))

(defun choose-pane-options-menu (reverse-x reverse-y draw-endpoints)
  (declare (special reverse-x reverse-y draw-endpoints))
  (let ((cvv-items
          `((reverse-x "X Axis Origin"
                       :assoc (("Left" . nil) ("Right" . t)) princ
                       :documentation "Select direction of the X axis origin.")
            (reverse-y "Y Axis Origin"
                       :assoc (("Bottom" . nil) ("Top" . t)) princ
                       :documentation "Select direction of the Y axis origin.")
            (draw-endpoints "Draw Line Endpoints" :boolean
                            :documentation "Select whether to draw endpoints as large dots."))))
    (catch 'no-options
      (w:choose-variable-values
        cvv-items
        :label `(:string " Options " :centered :font fonts:medfnb)
        :margin-choices '("Exit" ("Abort" (throw 'no-options nil)))
        :extra-width 10)
      (list :reverse-x-axis reverse-x
            :reverse-y-axis reverse-y
            :draw-line-endpoints draw-endpoints))))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :copy-pane) ()
  "Copy the attributes of this pane to another pane."
  (unless space-selected-p
    (command-loop-user-error "This pane is not associated with any space."))
  (let ((other-pane (point-at-a-graphics-pane tv:superior)))
    (send other-pane :set-space
	  space-instances
	  dimension-info
	  x-dim
	  y-dim
	  display-mode
	  step-modes
	  unit-types
          filter-function
          sort-function
          user-title
          x-min x-max x-labels
	  y-min y-max y-labels)
    (send other-pane :set-current-units current-units)))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :reset-pane) ()
  "Disassociate this pane from any space."
  (setf space-selected-p nil)
  (setf user-title nil)
  (send self :delete-all-items)
  (send self :set-scrolling-list-of-units nil)
  (send self :set-title nil)
  ;; (send self :redefine-margins)
  (when w:restored-bits-p
    (tv:sheet-force-access (self)
      (send self :refresh :complete-redisplay))))

(defmethod (gbb-graphics-window :reset-all-panes) ()
  "Disassociate this all panes from any space."
  (dolist (pane (send w:superior :graphics-panes))
    (send pane :reset-pane))
  ;; Force recomputation of the space instance menu item list.
  (build-space-menu-items)
  nil)

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :clear-pane) ()
  "Forget any units that are currently displayed by this pane."
  (send self :set-current-units nil)
  (send self :delete-all-items)
  (send self :refresh-contents))

(defmethod (gbb-graphics-window :clear-all-panes) ()
  "Forget any units that are currently displayed by any pane."
  (dolist (pane (send w:superior :graphics-panes))
    (send pane :clear-pane)))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :pane-status) ()
  
  (unless space-selected-p
    (command-loop-user-error "This pane is not associated with any space."))
  (using-resource (*standard-output* w:pop-up-text-window)
    (let ((save-selected-window w:selected-window)
	  (w *standard-output*))
      (send w :activate)
      (send w :set-size-in-characters 75 20)
      (send w :set-label nil)
      (send w :expose-near '(:mouse))
      (send w :select)
      (send w :clear-window)
      (format w "~&Pane:~20t~a" (send w:superior :which-pane-am-i self))
      (format w "~%Space:~20t~a" (pretty-space-path (first space-instances)))
      (dolist (si (rest space-instances))
        (format w "~%~20t~a" (pretty-space-path si)))
      (terpri w)
      (format w "X Dimension:~20t~a (~s)~%" x-dim x-type)
      (format w "      Min - Max:~20t(~d - ~d)~:[~;  Reversed~]~%"
              x-min x-max reverse-x-axis)
      (format w "      Labels:~20t~s~%" x-labels)
      (format w "Y Dimension:~20t~a (~s)~%" y-dim y-type)
      (format w "      Min - Max:~20t(~d - ~d)~:[~;  Reversed~]~%"
              y-min y-max reverse-y-axis)
      (format w "      Labels:~20t~s~%" y-labels)
      (format w "Units Types:~20t~s~%" unit-types)
      (format w "Filter Function:~20t~s~%" filter-function)
      (format w "Sort Function:~20t~s~%" sort-function)
      (format w "Display Mode:~20t~s~%" display-mode)
      (format w "Step Mode:~20t~s~%" step-modes)
      (unless (eq display-mode :all-units)
	(format w "Units Displayed:~{~20t~s~%~}" current-units))
      (format t "~2&[Type any character to remove this display.]~%")
      (send w :tyi)
      (send w :deexpose)
      (send save-selected-window :select))))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :display-current-units) ()

  (send self :refresh-contents t (member :display-all step-modes)))
  
;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :display-space-contents) ()
  "Display all the units that are stored on the space."
  (when (and w:exposed-p
             space-selected-p
             (not (send self :scrolling-list-of-units?)))
    (send self :clear-graph-region)
    (dolist (si space-instances)
      (gbb::map-space-instance #'(lambda (unit)
                                   (send self :display-unit
                                         unit
                                         nil
                                         (not (eq display-mode :last-unit))
                                         :display-all)
                                   (when (member :display-all step-modes)
                                     (wait-for-step-proceed self)))
                               unit-types
                               si))))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :display-unit) (unit
						&optional
						(remember-unit t)
						(dont-erase nil)
						(due-to :unit-addition))

  "Display a single unit.

   If REMEMBER-UNIT is true then the unit is saved as the current unit
   or added to the list of current units.  If REMEMBER-UNIT is false
   then the unit is simply displayed.

   Normally, if the display mode is :last-unit, the graph is cleared
   before the unit is displayed.  If DONT-ERASE is true then the graph
   is not cleared first -- the unit may overwrite whatever else may
   already be displayed.

   DUE-TO specifies the context in which this method is being called."

  (when (and space-selected-p
	     (gbb::intersectp space-instances (gbb::basic-unit.%%space-instances%% unit))
             (or (eq unit-types t)
                 (member (type-of unit) unit-types))
             (or (null filter-function)
                 (funcall filter-function unit)))

    (when remember-unit
      (ecase display-mode
	((:recently-added-units)
	 (pushnew unit current-units :test #'eq))
	((:last-unit)
	 (if (consp current-units)
	     (setf (first current-units) unit)
	     (setf current-units (list unit))))
	((:all-units :whole-space) nil)))
    (when (and w:exposed-p
	       (if x-dim
		   (unit-has-dimension unit x-dim)
		   t)
	       (if y-dim
		   (unit-has-dimension unit y-dim)
		   t))
      (when (and (eq display-mode :last-unit)
		 (not dont-erase))
	(send self :clear-graph-region))
      (cond ((and x-dim y-dim)
	     (send self :2d-display-unit unit))
	    ((or x-dim y-dim)
	     (send self :1d-display-unit unit))
	    (t
	     (send self :0d-display-unit unit)))
      (when (and %%stepping-enabled%%
                 (eq due-to :unit-addition)
		 (member :unit-addition step-modes))
	(wait-for-step-proceed self)))))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :2d-display-unit) (unit)

  (let* ((mutual (mutual-composite-dimensions-p unit x-dim y-dim))
         (index-info (get-unit-dimension-info unit mutual x-dim y-dim)))

    (cond (mutual
	   (send self :DISPLAY-MUTUALLY-COMPOSITE-LINKS unit index-info)
	   (dolist (index-element index-info)
	     (send self :2D-DISPLAY-ELEMENT
		   unit
		   (first index-element)
		   (second index-element))))

          ;; Disjoint composites ::
          ((and (composite-dimension-p unit x-dim)
                (composite-dimension-p unit y-dim))
           ;; Maybe draw links between elements.
           (send self :DISPLAY-NON-MUTUAL-COMPOSITE-LINKS
                 unit index-info t t)
           ;; Scatter plot
           (dolist (x-element (first index-info))
             (dolist (y-element (second index-info))
               (send self :2D-DISPLAY-ELEMENT unit x-element y-element))))

          ;; Only X is a composite ::
          ((composite-dimension-p unit x-dim)
	   (send self :DISPLAY-NON-MUTUAL-COMPOSITE-LINKS
                 unit index-info t nil)
	   (let ((y-element (second index-info)))
	     (dolist (x-element (first index-info))
	       (send self :2D-DISPLAY-ELEMENT unit x-element y-element))))

          ;; Only Y is a composite ::
          ((composite-dimension-p unit y-dim)
	   (send self :DISPLAY-NON-MUTUAL-COMPOSITE-LINKS
                 unit index-info nil t)
	   (let ((x-element (first index-info)))
	     (dolist (y-element (second index-info))
	       (send self :2D-DISPLAY-ELEMENT unit x-element y-element))))

          ;; Neither are composites ::
          (t
           (let ((x-element (first index-info))
		 (y-element (second index-info)))
             (send self :2D-DISPLAY-ELEMENT unit x-element y-element))))))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :2d-display-element) (unit element-x element-y)

  (let ((color (get-unit-color unit)))

    (cond
    
      ;; Range in both dimensions ::
      ((and (consp element-x)
            (consp element-y))
       (let ((xmin (send self :x-world-to-pixel-clipped (first element-x)))
             (xmax (send self :x-world-to-pixel-clipped (rest element-x)))
             (ymin (send self :y-world-to-pixel-clipped (first element-y)))
             (ymax (send self :y-world-to-pixel-clipped (rest element-y))))
         (when reverse-x-axis (rotatef xmin xmax))
         (when reverse-y-axis (rotatef ymin ymax))
         ;; Don't draw if completely outside window.
         (unless (or (< xmax 0) (> xmin graph-width)
                     (< ymin 0) (> ymax graph-height))
           ;; Clear the inside of the rectangle ::
           (send self :draw-filled-rectangle
                 xmin ymax (- xmax xmin) (- ymin ymax)
                 w:black w:erase-aluf)
           ;; Draw the rectangle ::
           (send self
                 (if *fill-regions* :draw-filled-rectangle :draw-rectangle)
                 xmin ymax (- xmax xmin) (- ymin ymax)
                 *line-thickness* color w:char-aluf))))
    
      ;; Range in only x-dimension ::
      ((consp element-x)
       (let ((xmin (send self :x-world-to-pixel-clipped (first element-x)))
             (xmax (send self :x-world-to-pixel-clipped (rest element-x)))
             (y (send self :y-world-to-pixel element-y)))
         (when reverse-x-axis (rotatef xmin xmax))
         ;; Don't draw if completely outside window.
         (unless (or (< xmax 0) (> xmin graph-width)
                     (not (<= 0 y graph-height)))
           (send self :draw-filled-rectangle
                 xmin
                 (1- y)
                 (max (- xmax xmin) *plot-bar-size*)
                 *plot-bar-size*
                 color w:char-aluf))))
    
      ;; Range in only y-dimension ::
      ((consp element-y)
       (let ((x (send self :x-world-to-pixel element-x))
             (ymin (send self :y-world-to-pixel-clipped (first element-y)))
             (ymax (send self :y-world-to-pixel-clipped (rest element-y))))
         (when reverse-y-axis (rotatef ymin ymax))
         ;; Don't draw if completely outside window.
         (unless (or (< ymin 0) (> ymax graph-width)
                     (not (<= 0 x graph-width)))
           (send self :draw-filled-rectangle
                 (1- x)
                 ymax
                 *plot-bar-size*
                 (max (- ymin ymax) *plot-bar-size*)
                 color w:char-aluf))))
    
      ;; Simple point ::
      (t                
       (let ((x (send self :x-world-to-pixel element-x))
             (y (send self :y-world-to-pixel element-y)))
         ;; Don't draw if completely outside window.
         (when (and (<= 0 x graph-width) (<= 0 graph-height))
           (draw-point self x y w:char-aluf draw-line-endpoints color)))))))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :1d-display-unit) (unit)

  (let* ((dimension (or x-dim y-dim))
	 (axis (if x-dim :X :Y))
	 (index-info (first (get-unit-dimension-info unit nil dimension))))
    (cond 
      ;; Its a composite ::
      ((composite-dimension-p unit dimension)
       ;; Skip the links in the one dimensional case.
       ;; Do the elements ::
       (dolist (element index-info)
	 (if (eq axis :x)
	     (send self :X-1D-DISPLAY-ELEMENT unit element)
	     (send self :Y-1D-DISPLAY-ELEMENT unit element))))
      ;; It's a scalar ::
      (t
       (if (eq axis :x)
	   (send self :X-1D-DISPLAY-ELEMENT unit index-info)
	   (send self :Y-1D-DISPLAY-ELEMENT unit index-info))))))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :x-1d-display-element) (unit element-x)

  (let ((y-pixel-position (floor graph-height 2))
        (color (get-unit-color unit)))

    (ecase x-type

      ((:ordered)
       (cond
	 ;; Range ::
	 ((consp element-x)
	  (unless (or (> (first element-x) x-max)
		      (< (rest element-x) x-min))
	    ;; Draw unless it's completely outside window.
	    (let ((xmin (send self :x-world-to-pixel-clipped (first element-x)))
		  (xmax (send self :x-world-to-pixel-clipped (rest element-x))))
	      (send self :draw-filled-rectangle
		    xmin
		    y-pixel-position
		    (max (- xmax xmin) *plot-bar-size*)
		    *plot-bar-size*
		    color w:char-aluf))))
	 ;; Simple point ::
	 (t                
	  (when (<= x-min element-x x-max)
	    ;; Draw unless it's outside the window.
	    (draw-point self
                        (send self :x-world-to-pixel element-x)
                        y-pixel-position
                        draw-line-endpoints
                        color)))))

      ((:enumerated)
       ;; Can only be a simple value.
       (draw-point self
                   (send self :x-world-to-pixel element-x)
                   y-pixel-position
                   draw-line-endpoints
                   color))
      )))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :y-1d-display-element) (unit element-y)

  (let ((x-pixel-position (floor graph-width 2))
        (color (get-unit-color unit)))

    (ecase y-type

      ((:ordered)
       (cond
	 ;; Range ::
	 ((consp element-y)
	  (unless (or (> (first element-y) y-max)
		      (< (rest element-y) y-min))
	    ;; Draw unless it's completely outside window.
	    (let ((ymin (send self :y-world-to-pixel-clipped (first element-y)))
		  (ymax (send self :y-world-to-pixel-clipped (rest element-y))))
	      (send self :draw-filled-rectangle
		    x-pixel-position
		    ymax
		    *plot-bar-size*
		    (max (- ymin ymax) *plot-bar-size*)
		    color w:char-aluf))))
	 ;; Simple point ::
	 (t                
	  (when (<= y-min element-y y-max)
	    ;; Draw unless it's outside the window.
	    (draw-point self
                        x-pixel-position
                        (send self :y-world-to-pixel element-y)
                        draw-line-endpoints
                        color)))))

      ((:enumerated)
       ;; Can only be a simple value.
       (draw-point self
                   x-pixel-position
                   (send self :y-world-to-pixel element-y)
                   draw-line-endpoints
                   color))
      )))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :0d-display-unit) (unit)
  (let ((items (send self :items))
        index)
    (cond ((find unit items :test #'eq)
           ;; Unit is already there -- don't do anything.
           nil)
          ((and sort-function
                (setf index (position unit items :test-not sort-function)))
           ;; There is a sort function and 
           ;; the unit should go somewhere in the middle.
           (send self :insert-item index unit))
          (t
           (send self :append-item unit)))
    (when *refresh-often*
      (send self :put-item-in-window unit))))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :scrolling-list-of-units?) ()
  "Returns true if the space is displaying a list of units
   rather than a graph."
  (and space-selected-p
       (null x-dim)
       (null y-dim)))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :set-scrolling-list-of-units)
           (&optional (mode :on))
  "Turns scrolling on and off."
  (ecase mode
    ((:on t)    (send self :set-scroll-bar-mode :maximum))    
    ((:off nil) (send self :set-scroll-bar-mode nil)))
  (send self :decide-if-scrolling-necessary))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :decide-if-scrolling-necessary) ()
  "Turn the scroll-bar region on or off."
  (let ((old-scroll-bar-on-off w:scroll-bar-on-off))
    ;; Bind this to avoid infinite recursion.  Other scroll bar methods
    ;; check this variable.
    (bind (locate-in-instance self 'w:scroll-bar-making-decision) t)
    (if (send self :scrolling-list-of-units?)
        (setf w:scroll-bar-on-off :on)
        (setf w:scroll-bar-on-off :off))
    (unless (eq w:scroll-bar-on-off old-scroll-bar-on-off)
      (send self :redefine-margins))))
     
;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :delete-unit) (unit)

  (cond ((null current-units)
	 nil)
	((listp current-units)
	 (setf current-units (delete unit current-units :test #'eq)))
	((eq current-units unit)
	 (setf current-units nil))
	(t nil))

  (when (send self :scrolling-list-of-units?)
    (let ((index (send self :number-of-item unit)))
      (when index
	(send self :delete-item index)))))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :delete-all-items) ()

  (let ((all-items (send self :items)))
    (cond ((arrayp all-items)
	   (setf (fill-pointer all-items) 0))
	  (t
	   (do ((i (send self :number-of-items) (1- i)))
	       ((>= 0 i))
	     (send self :delete-item i))))))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :display-mutually-composite-links) (unit index-info)
  
  (let ((color (get-unit-color unit)))
    (when (or *show-set-links*
              (not (gbb::set-dimension-p unit x-dim)))

      (let ((element-x-1 (first (first index-info)))
            (element-y-1 (second (first index-info)))
            element-x-2 element-y-2 x1 y1 x2 y2)

        (setf x1 (send self :x-world-to-pixel element-x-1)
              y1 (send self :y-world-to-pixel element-y-1))
        (dolist (elements (rest index-info))
          (setf element-x-2 (first elements)
                element-y-2 (second elements)
                x2 (send self :x-world-to-pixel element-x-2)
                y2 (send self :y-world-to-pixel element-y-2))
          (send self :DRAW-LINE x1 y1 x2 y2 *line-thickness* color)
          (setf x1 x2
                y1 y2))))))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :display-non-mutual-composite-links)
	   (unit index-info &optional (x-composite t) (y-composite t))

  "Draw the links between non mutually composite indexes (indexes that
   don't share the same composite element).  One of the indexes may be
   a scalar."

  (let ((x-elements (first index-info))
	(y-elements (second index-info))
        (color (get-unit-color unit))
        (x1) (x2) (y1) (y2))

    (ecase *show-unrelated-links*

      ;; Don't draw any links.
      ((:NONE NIL)
       nil)

      ;; Draw a single line linking all the index elements.
      ((:MINIMUM T)
       (gbb::dolist-or-atom (element-x-1 x-elements x-composite)
         (setf x2 x1)
         (setf x1 (send self :x-world-to-pixel element-x-1))
         (gbb::dolist-or-atom (element-y-1 y-elements y-composite)
           (setf y1 (send self :y-world-to-pixel element-y-1))
           (gbb::dolist-or-atom (element-y-2 y-elements y-composite)
             (when x2
               (send self :DRAW-LINE x1 y1 x2 y2 *line-thickness* color)
               (setf x2 nil))
             (setf y2 (send self :y-world-to-pixel element-y-2))
             (send self :DRAW-LINE x1 y1 x1 y2 *line-thickness* color)))))

      ;; Draw a line from every element to every other element.
      ((:MAXIMUM)
       (gbb::dolist-or-atom (element-x-1 x-elements x-composite)
         (setf x1 (send self :x-world-to-pixel element-x-1))
         (gbb::dolist-or-atom (element-y-1 y-elements y-composite)
           (setf y1 (send self :y-world-to-pixel element-y-1))
           (gbb::dolist-or-atom (element-x-2 x-elements x-composite)
             (setf x2 (send self :x-world-to-pixel element-x-2))
             (gbb::dolist-or-atom (element-y-2 y-elements y-composite)
               (setf y2 (send self :y-world-to-pixel element-y-2))
               (send self :DRAW-LINE x1 y1 x2 y2 *line-thickness* color)))))))))


;;; ---------------------------------------------------------------------------
;;;   Finding Units
;;; ---------------------------------------------------------------------------

(defmethod (gbb-graphics-window :find-units-at-point) (x y)

  (flet ((compute-find-values (world-value dim type min max labels)
           (when dim
             (ecase type
               ((:ordered)
                (let ((delta (* *find-point-fuzz* (- max min))))
                  (send self :check-value-in-bounds dim world-value)
                  (values (list (list (- world-value delta) (+ world-value delta)))
                          (format nil "~,1f" world-value))))
               ((:enumerated)
                (let ((result (list (elt labels (1- (round world-value))))))
                  (values result
                          (format nil "~a" result))))))))

    (let ((world-x (send self :x-pixel-to-world x))
          (world-y (send self :y-pixel-to-world y))
          x-pattern y-pattern x-string y-string)

      (multiple-value-setq (x-pattern x-string)
        (compute-find-values world-x x-dim x-type x-min x-max x-labels))
      (multiple-value-setq (y-pattern y-string)
        (compute-find-values world-y y-dim y-type y-min y-max y-labels))
        
      (send self :find-units
            x-pattern y-pattern
            (format nil "Units near~@[ ~a~]~@[ ~a~]." x-string y-string)))))

;;; ----------------------------------------------------------------------- 

(defmethod (gbb-graphics-window :find-units-in-region) (x y)

  (declare (ignore x y))
  (flet ((compute-find-values (region-min region-max dim type labels)
           (when dim
             (ecase type
               ((:ordered)
                (send self :check-value-in-bounds dim region-min)
                (send self :check-value-in-bounds dim region-max)
                (values (list (list region-min region-max))
                        (format nil "(~,1f, ~,1f)" region-min region-max)))
               ((:enumerated)
                (let ((result (subseq labels (1- (round region-min))
                                             (round region-max))))
                  (values result
                          (format nil "~a" result))))))))

    (let ((lm (w:sheet-inside-left))
          (tm (w:sheet-inside-top))
          (rm (w:sheet-inside-right))
          (bm (w:sheet-inside-bottom))
          window-mouse-x window-mouse-y
          x-pattern x-string y-pattern y-string)

      ;; Get mouse coordinates relative to this windows outside edges.
      (multiple-value-bind (x-offset y-offset)
          (w:sheet-calculate-offsets self w:mouse-sheet)
        (setf window-mouse-x (- w:mouse-x x-offset)
              window-mouse-y (- w:mouse-y y-offset)))
      ;; Mark a region.
      (multiple-value-bind (region-left region-top region-right region-bottom)
          (w:mouse-mark-rectangle self lm tm rm bm window-mouse-x window-mouse-y)

        (when region-left
          ;; If either of the axes are reversed then switch the corresponding
          ;; pixel values so that (e.g.) region-left will always be the pixel
          ;; value for the minimum world value.
          (when reverse-x-axis
            (rotatef region-left region-right))
          (when reverse-y-axis
            (rotatef region-top region-bottom))
          (multiple-value-setq (x-pattern x-string)
            (compute-find-values (send self :x-pixel-to-world region-left)
                                 (send self :x-pixel-to-world region-right)
                                 x-dim x-type x-labels))
          (multiple-value-setq (y-pattern y-string)
            (compute-find-values (send self :y-pixel-to-world region-bottom)
                                 (send self :y-pixel-to-world region-top)
                                 y-dim y-type y-labels))
          (send self :find-units
                x-pattern y-pattern
                (format nil "Units in Region ~@[ ~a~]~@[ ~a~]."
                        x-string y-string)))))))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :find-units) (x-pattern y-pattern
                                              &optional label)

  (labels ((filter-unit-types ()
             (let ((filtered-unit-types
                     (if (eq unit-types t)
                         (all-space-unit-types space-instances)
                         unit-types)))
               (remove-if-not #'unit-has-both-dimensions filtered-unit-types)))
           (unit-has-both-dimensions (unit)
             (and (if x-dim (unit-has-dimension unit x-dim) t)
                  (if y-dim (unit-has-dimension unit y-dim) t))))

    (let* ((pattern (build-find-pattern x-dim x-type x-pattern
                                        y-dim y-type y-pattern))
           (units (find-units-semi-internal
                    (filter-unit-types)
                    space-instances
                    pattern
                    (if (eq display-mode :all-units)
                        nil
                        #'(lambda (u)
                            (member u current-units :test #'eq)))
                    filter-function))
           (item-list nil))

      (cond
        ((null units)
         (gbb-graphics-warning "No Units Found."))
        (t
         (setf item-list
               (mapcar #'(lambda (u)
                           `(,(format nil " ~s " u)
                             :buttons
                             ((nil :eval (describe-item-in-typeout-window ',u ',self))
                              (nil :eval (setq user::== ',u))
                              (nil :eval (send ',self :unit-operations-menu ',u)))
                             :documentation
                             (:mouse-l-1 "Describe this unit"
                              :mouse-m-1 "Set USER::==")))
                       units))
         (w:menu-choose
           item-list
           :label (and label
                       `(:string ,label :centered :font fonts:medfnb))
           :columns 1))))))

(defmethod (gbb-graphics-window :check-value-in-bounds) (dim value)
  (let* ((dim-info (assoc dim dimension-info :test #'string=))
         (type (dimension-info.dimension-type dim-info))
         min max)
    (when (eq type :ordered)
      (setf min (first (dimension-info.dimension-arg dim-info))
            max (second (dimension-info.dimension-arg dim-info)))
      (unless (and (<= min value) (< value max))
        (command-loop-user-error
          "Find out of range.~@
           The actual ~a dimension bounds are ~d - ~d"
          dim min max)))))


;;; ---------------------------------------------------------------------------
;;;   Zooming and Panning
;;; ---------------------------------------------------------------------------

(defmethod (gbb-graphics-window :zoom-in) ()

  (labels ((scale-x (new-pixel)
             (send self :x-pixel-to-world new-pixel))
	   (scale-y (new-pixel)
             (send self :y-pixel-to-world new-pixel)))

    (block zoom-in
      (let ((lm (w:sheet-inside-left))
            (tm (w:sheet-inside-top))
            (rm (w:sheet-inside-right))
            (bm (w:sheet-inside-bottom))
            window-mouse-x window-mouse-y
            new-x-min new-x-max new-y-min new-y-max new-x-labels new-y-labels)
        
        ;; Get mouse coordinates relative to this windows outside edges.
        (multiple-value-bind (x-offset y-offset)
            (w:sheet-calculate-offsets self w:mouse-sheet)
          (setf window-mouse-x (- w:mouse-x x-offset)
                window-mouse-y (- w:mouse-y y-offset)))
        (multiple-value-bind (new-left new-top new-right new-bottom)
            (w:mouse-mark-rectangle
              self lm tm rm bm window-mouse-x window-mouse-y)
          (unless new-left (return-from zoom-in nil))
          ;; If either of the axes are reversed then switch the corresponding
          ;; pixel values so that (e.g.) new-left will always be the pixel
          ;; value for the minimum world value.
          (when reverse-x-axis
            (rotatef new-left new-right))
          (when reverse-y-axis
            (rotatef new-bottom new-top))
          (when x-dim
            (multiple-value-setq (new-x-min new-x-max)
              (fixup-bounds (scale-x new-left) (scale-x new-right))))
          (when y-dim
            (multiple-value-setq (new-y-min new-y-max)
              (fixup-bounds (scale-y new-bottom) (scale-y new-top)))))
        (when (eq x-type :enumerated)
          (setf new-x-labels (subseq x-labels (max 0 (1- new-x-min)) (1- new-x-max))))
        (when (eq y-type :enumerated)
          (setf new-y-labels (subseq y-labels (max 0 (1- new-y-min)) (1- new-y-max))))
        (send self :set-dimensions x-dim y-dim
              new-x-min new-x-max new-x-labels
              new-y-min new-y-max new-y-labels)
        (send self :refresh-entire-display)))))

				      
(defmethod (gbb-graphics-window :zoom-out) ()
  (send self :set-dimensions x-dim y-dim)
  (send self :refresh-entire-display))


;;; ---------------------------------------------------------------------------
;;;   Operations on Units
;;; ---------------------------------------------------------------------------

(defvar user::== nil)

(defun process-mouse-sensitive-unit (blip)
  (let ((type (first blip))
	(unit (second blip))
	(window (third blip))
	(mouse-char (int-char (fourth blip))))
    (declare (ignore type))
    (case mouse-char
      (#\mouse-l-1 (describe-item-in-typeout-window unit window))
      (#\mouse-l-2 (when (typep window 'w:typeout-window)
                     (send window :clear-screen))
                   (describe-item-in-typeout-window unit window))
      (#\mouse-m-1 (setf user::== unit))
      (#\mouse-r-1 (if (typep window 'gbb-graphics-pane)
                       (do-menu-for-pane window)
                       (w:beep)))
      (otherwise (w:beep)))))

(defun describe-in-pop-up-window (unit)
  (using-resource (*standard-output* w:pop-up-text-window)
    (let ((save-selected-window w:selected-window)
	  (w *standard-output*))
      (send w :activate)
      (send w :set-size-in-characters 75 32)
      (send w :set-label nil)
      (send w :set-border-margin-width 3)
      (send w :expose-near '(:mouse))
      (send w :select)
      (send w :clear-window)
      (describe-unit-instance unit w)
      (format w "~2&[Type any character to remove this display.]~%")
      (send w :tyi)
      (send w :deexpose)
      (send save-selected-window :select))))


(defparameter *unit-operations-menu-item-list*
  '((" Describe Unit "
     :buttons ((nil :value :describe-unit-in-typeout-window)
               (nil :value :describe-unit-in-typeout-window)
               (nil :value :describe-unit-in-pop-up-window))
     :documentation "Describe this unit.")
    (" Set USER::== "
     :value :set-==
     :documentation "Set the value of the symbol USER::== to this unit.")
    (" Display Links "
     :value :display-unit-links
     :documentation "Display the units linked to this unit.")))

(defmethod (gbb-graphics-window :unit-operations-menu) (unit)

  (let ((choice (w:menu-choose
                  *unit-operations-menu-item-list*
                  :label `(:string ,(format nil "Operations on ~a" (gbb::-> unit))
                           :centered :font fonts:medfnb)
                  :columns 1)))
    (ecase choice
      (:describe-unit-in-typeout-window
       (describe-item-in-typeout-window unit self))
      (:describe-unit-in-pop-up-window
       (describe-in-pop-up-window unit))
      (:set-==
       (setq user::== unit))
      (:display-unit-links
       (send self :display-unit-links unit))
      (nil nil))))

(defmethod (gbb-graphics-window :display-unit-links) (unit)
  (let* ((all-links (gbb::link-names (gbb::get-unit-description unit)))
         (some-links (w:mmc-all/none all-links
                                     :label "Links to Display")))
    (gbb::with-events-disabled
      (dolist (link-name some-links)
        (gbb::dolist-or-atom (l (gbb::get-unit-link unit link-name))
          (send self :display-unit l nil t :display-link))))))


;;; ---------------------------------------------------------------------------
;;;   Data Value to Pixel Value Conversions
;;; ---------------------------------------------------------------------------

(defun average2 (n1 n2)
  "Compute the average of two numbers.  The result is a float."
  (/ (+ n1 n2) 2.0))

(defmethod (gbb-graphics-window :x-world-to-pixel-clipped) (x)
  (max 0
       (min (1- graph-width)
	    (send self :x-world-to-pixel x))))

(defmethod (gbb-graphics-window :y-world-to-pixel-clipped) (y)
  (max 0
       (min (1- graph-height)
	    (send self :y-world-to-pixel y))))

(defmethod (gbb-graphics-window :x-world-to-pixel)
	   (x &optional (min x-min) (max x-max))
  "This method translates from world coordinates to screen coordinates
   (w.r.t. the inside margins) for the X axis."
  (case x-type
     (:enumerated 
      (let ((x-index (position x x-labels :test x-test)))
	(if x-index
	    (round (* graph-width (- (1+ x-index) min))
		   (- max min))
	    -10)))
     (:ordered
      (when (consp x)
        (setf x (/ (+ (car x) (cdr x)) 2.0)))
      (let ((pixel (round (* graph-width (- x min))
                          (- max min))))
        (if reverse-x-axis
            (- graph-width pixel)
            pixel)))))

(defmethod (gbb-graphics-window :y-world-to-pixel)
	   (y &optional (min y-min) (max y-max))
  (case y-type
     (:enumerated
      (let ((y-index (position y y-labels :test y-test)))
	(if y-index
	    (- graph-height (round (* graph-height (- (1+ y-index) min))
				   (- max min)))
	    -10)))
     (:ordered
      (when (consp y)
        (setf y (/ (+ (car y) (cdr y)) 2.0)))
      (let ((pixel (round (* graph-height (- y min))
			     (- max min))))
        (if reverse-y-axis
            pixel
            (- graph-height pixel))))))


(defmethod (gbb-graphics-window :x-pixel-to-world) (x-pixel)

  "Convert a window x coordinate into a world x coordinate.  X-PIXEL
   is relative to the outside of the window -- not the mouse-sheet, or
   the window's superior."

  (and x-dim
       (let* ((lm (w:sheet-inside-left))
              (world (* (- x-max x-min)
                        (/ (- x-pixel lm) (float graph-width)))))
         ;; WORLD is the distance (in world coordinates) of X-PIXEL
         ;; from the left margin.
         (if reverse-x-axis
             (- x-max world)
             (+ x-min world)))))

(defmethod (gbb-graphics-window :y-pixel-to-world) (y-pixel)
  "Convert a window y coordinate into a world y coordinate.  Y-PIXEL
   is relative to the outside of the window -- not the mouse-sheet, or
   the window's superior."
  (and y-dim
       (let* ((tm (w:sheet-inside-top))
              (world (* (- y-max y-min)
                        (/ (- y-pixel tm) (float graph-height)))))
         ;; WORLD is the distance (in world coordinates) of Y-PIXEL
         ;; from the top margin.
         (if reverse-y-axis
             (+ y-min world)
             (- y-max world)))))


;;; ---------------------------------------------------------------------------
;;;   Random Graphics Functions
;;; ---------------------------------------------------------------------------

(defmethod (gbb-graphics-window :draw-line-explicit) (x0 y0 x1 y1
						      &optional
						      (alu w:char-aluf)
						      (draw-end-point-p t))
  "Draw line with respect to the outside edges."
  (w:sheet-force-access (self)
    (w:prepare-sheet (self)
      (w:%draw-line x0 y0 x1 y1 alu draw-end-point-p self))))

;;; -----------------------------------------------------------------------

(defparameter *PLOT-POINT-ARRAY*
     (make-array '(5 32) :element-type 'bit
                 :initial-contents '(#*01110000000000000000000000000000
                                     #*11111000000000000000000000000000
                                     #*11111000000000000000000000000000
                                     #*11111000000000000000000000000000
                                     #*01110000000000000000000000000000))
  "Bit array which represents a point.")

(defparameter *PLOT-POINT-X-OFFSET* 2
  "X offset, in pixels, to the center of the point in *PLOT-POINT-ARRAY*.")

(defparameter *PLOT-POINT-Y-OFFSET* 2
  "Y offset, in pixels, to the center of the point in *PLOT-POINT-ARRAY*.")

(defparameter *PLOT-POINT-WIDTH* 5)
(defparameter *PLOT-POINT-HEIGHT* 5)

;;; -----------------------------------------------------------------------

(defun draw-point (window x y &optional (alu w:normal)
                                        (big-point t)
                                        (color w:black))
  "Draw a point."

  (if big-point
      (tv:prepare-color (self color)
        (send window :bitblt alu *plot-point-width* *plot-point-height*
              *plot-point-array* 0 0
              (- x *plot-point-x-offset*)
              (- y *plot-point-y-offset*)))
      (send window :draw-point x y alu color)))

;;; -----------------------------------------------------------------------

(defun get-unit-color (unit)
  (let ((type (etypecase unit
                (gbb::basic-unit (type-of unit))
                (symbol unit))))
    (cond ((cdr (assoc type *unit-color-alist* :test #'eq)))
          (t
           (pushnew-acons *unit-color-alist* type w:black :test #'eq)
           w:black))))

;;; -----------------------------------------------------------------------

(defun unit-colors-menu ()

  (labels ((color-unit-item-list (units)
             (let ((item-list nil)
                   color)
               (dolist (unit units)
                 (setf color (get-unit-color unit))
                 (push `(,(car (find color w:color-alist :key #'cdr))
                         :no-select t :color ,color)
                       item-list)
                 (push `(,(symbol-name unit) :value ,unit
                         :color ,(get-unit-color unit))
                       item-list))
               item-list)))

    (let (unit color)
      (loop
        (setf unit (w:menu-choose
                     (list* '("All Units" :value :all :font fonts:hl12i
                              :documentation "Set all units to the same color.")
                            '("" :no-select t)
                            (color-unit-item-list gbb::*all-units*))
                     :item-alignment :left
                     :columns 2
                     :label '(:string " Choose a Unit Type "
                                      :centered :font fonts:medfnt)))
        (when (null unit) (return nil))
        (setf color (w:menu-choose
                      w:color-alist
                      :label (format nil " Color for ~a " unit)
                      :columns 1))
        (cond ((null color) nil)
              ((eq unit :all)
               (dolist (pair *unit-color-alist*)
                 (setf (cdr pair) color)))
              (t
               (setf (cdr (assoc unit *unit-color-alist* :test #'eq))
                     color)))))))


;;; ---------------------------------------------------------------------------
;;;   Mouse Methods
;;; ---------------------------------------------------------------------------

(defmethod (gbb-graphics-window :point-in-margin?) (x y)
  (or (< x (w:sheet-inside-left))
      (< y (w:sheet-inside-top))
      (> x (w:sheet-inside-right))
      (> y (w:sheet-inside-bottom))))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :MOUSE-LEFT-1) (x y)

  (if (or (not space-selected-p)
          (send self :scrolling-list-of-units?)
          (send self :point-in-margin? x y))
      (do-select-space w:superior self)
      (send self :find-units-at-point x y)))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :mouse-left-2) (x y)

  (if (or (not space-selected-p)
          (send self :scrolling-list-of-units?)
          (send self :point-in-margin? x y))
      (send self :choose-dimensions)
      (send self :find-units-in-region x y)))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :mouse-middle-1) (x y)
  
  (if (or (not space-selected-p)
          (send self :scrolling-list-of-units?)
          (send self :point-in-margin? x y))
      (send w:superior :choose-configuration)
      (send self :zoom-in)))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :mouse-middle-2) (x y)
  
  (if (or (not space-selected-p)
          (send self :scrolling-list-of-units?)
          (send self :point-in-margin? x y))
      (send self :choose-unit-types)
      (send self :zoom-out)))

;;; -----------------------------------------------------------------------

(defmethod (gbb-graphics-window :mouse-right-1) (x y)
  (declare (ignore x y))
  (do-menu-for-pane self))

;;; -----------------------------------------------------------------------

(defparameter *inside-who-line-doc*
    '(:mouse-l-1 "Find Units Near Point"
      :mouse-l-2 "Find Units in Region"
      :mouse-m-1 "Zoom In"
      :mouse-m-2 "Zoom Out"
      :mouse-r-1 "GBB Graphics Menu"
      :mouse-r-2 "System Menu")
  "Who line documentation string for cases where the mouse is on the
   inside of the window.")

(defparameter *margins-who-line-doc*
    '(:mouse-l-1 "Select a Space"
      :mouse-l-2 "Choose Dimensions"
      :mouse-m-1 "Change Configuration"
      :mouse-m-2 "Choose Unit Types"
      :mouse-r-1 "GBB Graphics Menu"
      :mouse-r-2 "System Menu")
  "Who line documentation string for cases where the mouse is in the
   margins of the window.")

(defparameter *mouse-sensitive-unit-who-line-doc*
    '(:mouse-l-1 "Describe this unit"
      :mouse-m-1 "Set USER::=="
      :mouse-r-1 "GBB Graphics Menu"
      :mouse-r-2 "System Menu"))

(defvar *last-sensitive-unit* nil
  "Records the last mouse sensitive item displayed in the who line doc string.")

(defmethod (gbb-graphics-window :who-line-documentation-string) ()

  (let ((x sys:mouse-x)
        (y sys:mouse-y)
        x-offset y-offset item item-type)
    (multiple-value-setq (x-offset y-offset)
      (w:sheet-calculate-offsets self w:mouse-sheet))
    (setf x (- x x-offset)
          y (- y y-offset))

    (cond
      ((or (not space-selected-p)
           (send self :point-in-margin? x y))
       *margins-who-line-doc*)
      ((not (send self :scrolling-list-of-units?))
       *inside-who-line-doc*)
      (t
       (multiple-value-setq (item item-type)
         (send self :mouse-sensitive-item x y))
       (cond ((null item-type)
              *margins-who-line-doc*)
             (item
              (unless (eq item *last-sensitive-unit*)
                (w:who-line-clobbered))
              (setq *last-sensitive-unit* item)
              `(,@*mouse-sensitive-unit-who-line-doc*
                :documentation ,(format nil "~s" item)))
             (t *margins-who-line-doc*))))))

(defwrapper (gbb-graphics-window :mouse-click) ((mouse-char x y) . body)
  (let ((result-v (gensym)))
    `(let ((,result-v (progn ,@body)))
       (or ,result-v
           w:scroll-bar-active-state
           (send self :force-kbd-input
                 `(:mouse-click ,(int-char mouse-char) ,self ,x ,y))))))


(defmethod (gbb-graphics-window :print-item) (the-item screen-line-number item-number)
  (declare (ignore screen-line-number item-number))
  (send self :item :ms-gbb-unit the-item))


;;; ---------------------------------------------------------------------------
;;;   <SYSTEM>-Gamma
;;; ---------------------------------------------------------------------------

;;; Make the system easy to get to:
;;; SYSTEM Symbol-Shift-G will it bring up.

(w:add-system-key
 #\GAMMA
 'gbb-graphics-frame
 "GBB Graphics - Graphical display of GBB stuff.")


;;; ---------------------------------------------------------------------------
;;;   Interface to GBB:
;;; ---------------------------------------------------------------------------

;; Take care that these functions work even if the GBB graphics frame
;; isn't instantiated.


(pushnew 'insert-unit-graphics-hook
	 gbb::*insert-unit-hook-functions*)

(defun insert-unit-graphics-hook (unit-instance space-instance)

  "INSERT-UNIT-ON-SPACE-GRAPHICS-HOOK unit-instance space-instance

   This function keeps the graphics panes up to date.  When a unit
   is instantiated it is displayed in the appropriate panes (if any)."
  
  (dolist (frame *gbb-graphics-frames*)
    (dolist (pane (send frame :graphics-panes))
      (when (and (send pane :space-selected-p)
                 (member space-instance (send pane :space-instances)))
        (send pane :display-unit unit-instance)))))


(pushnew 'delete-unit-graphics-hook
	 gbb::*delete-unit-hook-functions*)

(defun delete-unit-graphics-hook (unit-instance space-instance)

  "DELETE-UNIT-GRAPHICS-HOOK unit-instance space-instance

   This function keeps the graphics panes up to date.  When a unit
   is deleted from a space it should be deleted from any panes that
   may be displaying it."

  (dolist (frame *gbb-graphics-frames*)
    (dolist (pane (send frame :graphics-panes))
      (when (and (send pane :space-selected-p)
                 (member space-instance (send pane :space-instances)))
        (send pane :delete-unit unit-instance)
        (when *refresh-often*
          (send pane :refresh-contents))))))


(pushnew 'move-unit-graphics-hook
         gbb::*move-unit-hook-functions*)

(defun move-unit-graphics-hook (unit-instance space-instance)

  "MOVE-UNIT-GRAPHICS-HOOK unit-instance space-instance

   This function keeps the graphics panes up to date.  When a unit
   is moved on a space it should be redrawn in it's new position."

  (dolist (frame *gbb-graphics-frames*)
    (dolist (pane (send frame :graphics-panes))
      (when (and (send pane :space-selected-p)
                 (member space-instance (send pane :space-instances)))
        (if *refresh-often*
            (send pane :refresh-contents)
            (send pane :display-unit unit-instance))))))


(pushnew 'instantiate-bb-db-graphics-hook
	 gbb::*instantiate-bb-db-hook-functions*)

(defun instantiate-bb-db-graphics-hook ()

  "INSTANTIATE-BB-DB-GRAPHICS-HOOK

   This function is called each time instantiate-blackboard-database
   is called.  Among other things, it updates the panes to point to
   the new space instances."

  ;; Build the menu items list for the bb-db menu.
  (when *gbb-graphics-frames*
    (build-space-menu-items))

  (dolist (frame *gbb-graphics-frames*)
    ;; If any panes are already displaying a space then
    ;; make the pane display the corresponding new space.
    (dolist (pane (send frame :graphics-panes))
      (when (send pane :space-selected-p)
	(let* ((old-space-instances (send pane :space-instances))
	       (new-space-instances
                 (mapc-condcons #'(lambda (si)
                                    (gbb::get-space-instance-from-path
                                      (gbb::get-path-from-space-instance si)
                                      nil))
                                old-space-instances)))
	  (cond (new-space-instances
		 (send pane :set-current-units nil)
		 (send pane :same-space new-space-instances))
		(t
		 (send pane :reset-pane))))))))


(pushnew 'clear-space-graphics-hook
	 gbb::*clear-space-hook-functions*)

(defun clear-space-graphics-hook (cleared-space-instance)

  "CLEAR-SPACE-GRAPHICS-HOOK cleared-space-instance

   Clears all panes that are displaying the contents
   of CLEARED-SPACE-INSTANCE."

  (dolist (frame *gbb-graphics-frames*)

    ;; If any panes are already displaying a space then
    ;; make the pane display the corresponding new space.
    (dolist (pane (send frame :graphics-panes))
      (when (send pane :space-selected-p)
	(let* ((displayed-space-instances (send pane :space-instances)))
	  (when (member cleared-space-instance displayed-space-instances)
	    (send pane :clear-pane)))))))

(defun clear-graphics-hooks ()
  (setf gbb::*insert-unit-hook-functions*
        (delete 'insert-unit-graphics-hook
                gbb::*insert-unit-hook-functions*))
  (setf gbb::*delete-unit-hook-functions*
        (delete 'delete-unit-graphics-hook
                gbb::*delete-unit-hook-functions*))
  (setf gbb::*move-unit-hook-functions*
        (delete 'move-unit-graphics-hook
                gbb::*move-unit-hook-functions*))
  (setf gbb::*instantiate-bb-db-hook-functions*
        (delete 'instantiate-bb-db-graphics-hook
                gbb::*instantiate-bb-db-hook-functions*))
  (setf gbb::*clear-space-hook-functions*
        (delete 'clear-space-graphics-hook
                gbb::*clear-space-hook-functions*)))

(defun add-graphics-hooks ()
  (pushnew 'insert-unit-graphics-hook
           gbb::*insert-unit-hook-functions*)
  (pushnew 'delete-unit-graphics-hook
           gbb::*delete-unit-hook-functions*)
  (pushnew 'move-unit-graphics-hook
           gbb::*move-unit-hook-functions*)
  (pushnew 'instantiate-bb-db-graphics-hook
           gbb::*instantiate-bb-db-hook-functions*)
  (pushnew 'clear-space-graphics-hook
           gbb::*clear-space-hook-functions*))

;;; -----------------------------------------------------------------------

(defun setup-pane (pane paths x-dimension y-dimension
                   &key
                   (title               nil)
                   (frame               nil)
                   (display-mode        :all-units)
                   (unit-types          t)
                   (step-modes          nil)
                   (filter-function     nil)
                   (sort-function       nil)
                   (reverse-x-axis      nil)
                   (reverse-y-axis      nil)
                   (draw-line-endpoints t))


  "SETUP-PANE pane paths x-dimension y-dimension
           &key (title               nil)
                (frame               nil)
                (step-modes          nil)
                (display-mode        :all-units)
                (unit-types          t)
                (filter-function     nil)
                (sort-function       nil)
                (reverse-x-axis      nil)
                (reverse-y-axis      nil)
                (draw-line-endpoints t)

   Set up a pane to display some data.  PANE is a number which indicates
   which pane to set up."

  (when (null frame)
    (cond ((null *gbb-graphics-frames*)
           (setf frame (tv:make-window 'gbb-graphics-frame)))
          ((= 1 (length *gbb-graphics-frames*))
           (setf frame (first *gbb-graphics-frames*)))
          (t (error "No frame supplied and more than one graphics frame exists.~@
                     You must supply a frame to SETUP-PANE.~@
                     The available frames are:~%~s."
                    *gbb-graphics-frames*))))

  (check-type frame gbb-graphics-frame)
  (check-type x-dimension symbol)
  (check-type y-dimension symbol)

  (let* ((pane-name (intern (etypecase pane
                              (number (format nil "GRAPHICS-~d" pane))
                              (symbol (symbol-name pane))
                              (string pane))
                            "GBB-GRAPHICS"))
         (the-pane (send frame :get-pane pane-name))
         (space-instances
           (if (listp paths)
               (gbb::get-space-instances-from-path-structures paths)
               (list (gbb::get-space-instance-from-path-structure paths))))
         (dimension-info (get-common-dimensions space-instances)))

    (send the-pane :set-pane-options `(:reverse-x-axis      ,reverse-x-axis
                                       :reverse-y-axis      ,reverse-y-axis
                                       :draw-line-endpoints ,draw-line-endpoints))
    (send the-pane :set-space space-instances
                              dimension-info
                              x-dimension
                              y-dimension
                              display-mode
                              step-modes
                              unit-types
                              filter-function
                              sort-function
                              title)))

(defun send-pane (frame pane message &rest args)

  (when (null frame)
    (cond ((null *gbb-graphics-frames*)
           (setf frame (tv:make-window 'gbb-graphics-frame)))
          ((= 1 (length *gbb-graphics-frames*))
           (setf frame (first *gbb-graphics-frames*)))
          (t (error "No frame supplied and more than one graphics frame exists.~@
                     You must supply a frame to SETUP-PANE.~@
                     The available frames are:~%~s."
                    *gbb-graphics-frames*))))

  (check-type frame gbb-graphics-frame)

  (let ((pane-name (intern (etypecase pane
                             (number (format nil "GRAPHICS-~d" pane))
                             (symbol (symbol-name pane))
                             (string pane))
                           "GBB-GRAPHICS")))
    (apply frame :send-pane pane-name message args)))


(defun gbb-graphics-frame-process (window)
  ;; Bind *terminal-io* so errors will show up there (most of the time).
  (let* ((frame (send window :superior))
         (ll-pane (send frame :get-pane 'll))
         (*terminal-io* ll-pane)
         blip)
    (ucl::ignore-errors-query-loop
      (catch 'top-level
        (setf blip (send window :any-tyi))
        (process-blip blip)
        ))))

(defun reset-background-process (&optional frame)
  "Reset the background process for the frame."
  (when (null frame)
    (setf frame
          (if (typep (send w:selected-window :superior) 'gbb-graphics-frame)
              (send w:selected-window :superior)
              (first *gbb-graphics-frames*))))
  (send (send frame :send-pane 'graphics-1 :process) :reset))

(deff user::rgr 'reset-background-process)

(defun funcall-with-top-level-catcher (fn &rest args)
  (ucl::ignore-errors-query
    (catch 'top-level
      (apply fn args))))

(defun process-blip (blip)
  (cond ((listp blip)
         (case (first blip)
           ((:menu)
            (process-menu-item blip))
           ((:mouse-click :mouse-button)
            (process-mouse-click blip))
           ((:ms-gbb-unit)
            (process-mouse-sensitive-unit blip))
           (otherwise
            (w:notify nil "~&Unknown Blip: ~s~%" blip))))
        (t (w:notify nil "~&Unknown Blip: ~s~%" blip))))

(defun process-menu-item (blip)
  (let* ((item (second blip))
         (operation (get item :value))
         ;; (mouse (third blip))
         (menu (fourth blip))
	 (frame (send menu :superior)))
    (process-menu-operation operation frame)))

(defun process-menu-operation (operation frame &optional pane)

  (flet ((send-all-panes (message)
           ;; Send a message to all exposed graphics panes.
           (dolist (pane (send frame :graphics-panes))
             (when (send pane :exposed-p)
               (send pane message)))))

    ;; Select a pane if necessary:
    (cond
      ;; Pane selected or no pane needed.
      ((or (typep pane 'gbb-graphics-window)
           (member operation '(:display-current-units-all
                               :display-space-contents-all
                               :change-configuration
                               :choose-unit-colors)
                       :test #'eq))
       nil)
      ;; Any pane will do.
      ((member operation '(:choose-display-mode-all
                           :choose-step-modes-all
                           :clear-all-panes
                           :reset-all-panes)
               :test #'eq)
       (setf pane (send frame :get-pane 'graphics-1)))
      ((null pane)
       (setf pane (select-graphics-pane frame)))
      ((symbolp pane)
       (setf pane (send frame :get-pane pane))))

    (case operation
      (:select-space		     (do-select-space frame pane))
      (:display-current-units-all    (send-all-panes :display-current-units))
      (:display-space-contents-all   (send-all-panes :display-space-contents))
      (:change-configuration	     (send frame :choose-configuration))
      (:options-menu                 (do-options-menu frame pane))
      (:choose-unit-colors           (unit-colors-menu))
      (otherwise                     (send pane operation)))))


(defun process-mouse-click (blip)
  (let* ((type (first blip))
	 (mouse-char (if (eq type :mouse-click)
			 (second blip)
			 (int-char (second blip))))
         (window (third blip))
         (x (fourth blip))
         (y (fifth blip)))
    (case mouse-char
      ((#\mouse-l-1) (send window :mouse-left-1 x y))
      ((#\mouse-l-2) (send window :mouse-left-2 x y))
      ((#\mouse-m-1) (send window :mouse-middle-1 x y))
      ((#\mouse-m-2) (send window :mouse-middle-2 x y))
      ((#\mouse-r-1) (send window :mouse-right-1 x y))
      (otherwise (w:notify nil "~&Unknown Blip: ~s~%" blip)))))


(defparameter *pane-menu-item-list*
  '((" Select Space "
     :value :select-space
     :documentation "Select a space to display in this pane.")
    (" Copy Pane "
     :value :copy-pane
     :documentation "Copy this pane to another pane.")
    ("" :no-select t)
    (" Choose Dimensions "
     :value :choose-dimensions
     :documentation "Choose which dimension to display on each axis.")
    (" Copy Dimensions "
     :value :copy-dimensions
     :documentation "Copy this pane's dimensions to another pane.")
    (" Choose Dimension Bounds "
     :value :set-axis-bounds
     :documentation "Set the bounds for each dimension.")
    ("" :no-select t)
    (" Options Menu "
     :value :options-menu
     :documentation "Menu of many options for both this pane and all panes.")
    (" Choose Unit Colors "
     :value :choose-unit-colors
     :documentation "Choose what colors to use for each unit type.")
    ("" :no-select t)
    (" Zoom In/Out "
     :buttons ((nil :value :zoom-in)
               (nil :value :zoom-in)
               (nil :value :zoom-out))
     :documentation "L, M: Zoom in on a portion of the space, R: Display the entire space.")
    ("" :no-select t)
    (" Display Current Units "
     :buttons ((nil :value :display-current-units)
               (nil :value :display-current-units)
               (nil :value :display-current-units-all))
     :documentation "Display the current units.  L, M: This pane only, R: All panes.")
    (" Display Space Contents "
     :buttons ((nil :value :display-space-contents)
               (nil :value :display-space-contents)
               (nil :value :display-space-contents-all))
     :documentation "Display all the units on a space.  L, M: This pane only, R: All panes.")
    ("" :no-select t)
    (" Clear Pane "
     :buttons ((nil :value :clear-pane)
               (nil :value :clear-pane)
               (nil :value :clear-all-panes))
     :documentation "Clear this pane. This resets the current units of the pane.  L, M: This pane only, R: All panes.")
    (" Reset Pane "
     :buttons ((nil :value :reset-pane)
               (nil :value :reset-pane)
               (nil :value :reset-all-panes))
     :documentation "Disassociate this pane from any space.  L, M: This pane only, R: All panes.")
    (" Change Configuration "
     :value :change-configuration
     :documentation "Select a new frame configuration.")
    ))

(defparameter *options-menu-item-list*
  '((" Choose Unit Types "
     :value :choose-unit-types
     :documentation "Select what unit types to display in this pane.")
    (" Choose Filter Function "
     :value :choose-filter-function
     :documentation "Select a function to filter units to be displayed in this pane.")
    (" Choose Display Mode "
     :buttons ((nil :value :choose-display-mode)
               (nil :value :choose-display-mode)
               (nil :value :choose-display-mode-all))
     :documentation "Choose the display mode for this pane.  L, M: This pane only, R: All panes.")
    (" Choose Step Mode "
     :buttons ((nil :value :choose-step-modes)
               (nil :value :choose-step-modes)
               (nil :value :choose-step-modes-all))
     :documentation "Turn stepping on or off for this pane.  L, M: This pane only, R: All panes.")
    (" Choose Pane Options "
     :buttons ((nil :value :choose-pane-options)
               (nil :value :choose-pane-options)
               (nil :value :choose-pane-options-all))
     :documentation "Choose pane options.  L, M: This pane only, R: All panes.")
    (" Show Pane Status "
     :value :pane-status
     :documentation "Show information about this pane.")
    ))

(defun do-menu-for-pane (pane &optional (item-list *pane-menu-item-list*))
  (let* ((label `(:string ,(format nil " Operations on Pane ~a "
                                   (send pane :my-pane-number))
                  :font fonts:medfnb
                  :centered))
         (operation (w:menu-choose item-list
                                   :item-alignment :center
                                   :columns 1
                                   :label label))
         (frame (send pane :superior)))
    (when operation
      (process-menu-operation operation frame pane))))

(defun do-menu-for-frame (frame &optional (item-list *pane-menu-item-list*))
  (let* ((label `(:string "GBB Graphics Menu"
                  :font fonts:medfnb
                  :centered))
         (operation (w:menu-choose item-list
                                   :item-alignment :center
                                   :columns 1
                                   :label label)))
    (when operation
      (process-menu-operation operation frame))))

(defun do-options-menu (frame pane)
  (let ((label `(:string ,(format nil " Operations on Pane ~a "
                                  (send pane :my-pane-number))
                 :font fonts:medfnb
                 :centered))
        operation)
    (loop
      (setf operation (w:menu-choose *options-menu-item-list*
                                     :item-alignment :center
                                     :columns 1
                                     :label label))
      (when (null operation)
        (return nil))
      (process-menu-operation operation frame pane))))


(defun command-loop-user-error (string &rest format-args)
  "Notify the user of an error and return to top level in the
   command loop.  This function does not return."
  (apply #'w:notify nil string format-args)
  ;; This catch tag is established in the function
  ;; gbb-graphics-frame-process.
  (throw 'top-level nil))

(defun gbb-graphics-warning (string &rest format-args)
  "Warn the user about some problem."
  (apply #'w:notify nil string format-args))


(defun do-select-space (frame pane)

  (declare (ignore frame))

  (let* ((space-items (space-menu-item-list))
	 (space-instances (w:mmc-all/none
                            space-items
                            :highlighted-values (send pane :space-instances)
                            :label '(:string  " Current Spaces "
                                     :centered :font fonts:medfnb)
                            :all-item " Select All Spaces "
                            :none-item " Deselect All Spaces "
                            :item-alignment :left
                            :columns 1))
	 (x-dim nil)
	 (y-dim nil)
	 dimension-info)

    (unless space-instances
      (return-from do-select-space nil))

    (setf dimension-info (get-common-dimensions space-instances))

    (when (and (null dimension-info)
               (gbb::space.dimensions
                 (gbb::space-instance.space (first space-instances)))
               (not (w:mouse-confirm
                      (format nil
                        "These spaces have no dimensions in common:~%~s."
                        (remove-duplicates
                          (mapcar #'gbb::space-instance.name space-instances)))
                      #.(format nil
                          "Click mouse here or hit <END> if that's what you intended.~@
                           Move the mouse out of this window or hit `N' to abort.")))
        (throw 'top-level nil)))

    (unless (null dimension-info)
      (multiple-value-setq (x-dim y-dim)
	(select-dimensions dimension-info
                           space-instances)))

    (send pane :set-space
               space-instances
               dimension-info
               x-dim
               y-dim)))


(defun select-graphics-pane (frame)

  "SELECT-GRAPHICS-PANE frame

   Select an exposed graphics pane from `frame'.  If there is only
   one pane exposed, it is simply returned.  If more than one pane
   is exposed then the user selects one from a menu.  If she doesn't
   make a selection, then this function returns nil."

  (let* ((exposed-panes (mapc-condcons
                          #'(lambda (pane)
                              (and (send pane :exposed-p) pane))
                          (send frame :graphics-panes))))
    (case (length exposed-panes)
       ((0) (command-loop-user-error "No graphics panes exposed."))
       ((1) (return-from select-graphics-pane (first exposed-panes)))
       (otherwise (point-at-a-graphics-pane frame)))))

(defun point-at-a-graphics-pane (frame)
  "Point at a graphics pane with the mouse.  Returns the pane selected."
  (do ((pane nil))
      (nil)
    (setf pane (w:point-at-a-window '(:mouse-l-1 "Choose a pane"
                                      :mouse-m-1 "Abort"
                                      :mouse-r-1 "Choose a pane")))
    (cond ((null pane)
           (throw 'top-level nil))
          ((member pane (send frame :graphics-panes))
           (return pane))
          (t (gbb-graphics-warning "Choose a graphing pane.")))))

(defun choose-graphics-pane (frame)
  "Pop up a menu of the exposed panes.  Returns the pane choosen."
  (let* ((exposed-panes (mapc-condcons
                          #'(lambda (pane)
                              (and (send pane :exposed-p) pane))
                          (send frame :graphics-panes)))
         (item-list (mapcar #'(lambda (pane)
                                `(,(send pane :name) :value ,pane))
                            exposed-panes)))
    (w:menu-choose
      item-list
      :label '(:string " Choose a Pane " :centered :font fonts:medfnb)
      :item-alignment :left)))

(defun select-dimensions (dimension-info space-instances
			  &optional (starting-x 'x) (starting-y 'y))

  "Choose which two dimensions of a space to display.  Returns
   two values: the new x dimension and the new y dimension.
   SPACE-INSTANCES is only used for the menu label."

  (let* ((dimensions (mapcar #'car dimension-info))
	 (x-present (member starting-x dimensions :test #'string=))
	 (y-present (member starting-y dimensions :test #'string=))
	 choices cvv-item-list new-x-dim new-y-dim)
    (declare (special new-x-dim new-y-dim))

    (cond ((null dimensions)
	   (return-from select-dimensions (values nil nil)))
	  ((and x-present y-present)
	   (setf new-x-dim (first x-present)
		 new-y-dim (first y-present)))
	  (x-present
	   (setf new-x-dim (first x-present)
		 new-y-dim (if (eq new-x-dim (first dimensions))
			       (second dimensions)
			       (first dimensions))))
	  (y-present
	   (setf new-y-dim (first y-present)
		 new-x-dim (if (eq new-y-dim (first dimensions))
			       (second dimensions)
			       (first dimensions))))
	  (t (setf new-x-dim (first dimensions)
		   new-y-dim (second dimensions))))
    
    ;; NIL is used to indicate choosing no dimension here.  This works
    ;; even if there is a dimension named nil because the dimension
    ;; will be stored stored as a keyword.
    (setf choices `(,@(mapcar #'(lambda (d) (cons (symbol-name d) d)) dimensions)
		    ("No Dimension" . nil)))
    (setf cvv-item-list `((new-x-dim "X Axis Dimension" :assoc ,choices)
			  (new-y-dim "Y Axis Dimension" :assoc ,choices)))

    (loop
      (w:choose-variable-values
	cvv-item-list
	:label `(:string ,(format nil " Dimensions for ~s~@[, etc.~] "
                                  (gbb::space.name
                                    (gbb::space-instance.space
                                      (first space-instances)))
                                  (< 1 (length space-instances)))
		 :centered
		 :font fonts:medfnb)
	:margin-choices '("Exit"
			  ("Abort" (throw 'top-level nil))))
      (cond ((and (null new-x-dim)
		  (null new-y-dim))
	     (return-from select-dimensions (values nil nil)))
	    ((eq new-x-dim new-y-dim)
	     (gbb-graphics-warning "Choose two different dimensions."))
	    (t
	     (return-from select-dimensions (values new-x-dim new-y-dim))))
      )))


(defvar *space-instance-menu-item-list* nil
  "Variable used to cache the item list for SPACE-MENU-ITEM-LIST.")

(defun space-menu-item-list ()
  (cond ((null gbb::*blackboard-database*)
	 (command-loop-user-error "No Blackboard Available."))
	((null *space-instance-menu-item-list*)
	 (build-space-menu-items))
	(t *space-instance-menu-item-list*)))
       
(defun build-space-menu-items ()
  (let ((item-list (mapcan #'build-space-menu-items-1
			   gbb::*blackboard-database*)))
    (setf *space-instance-menu-item-list* item-list)
    item-list))

(defun left-justify-item-list (item-list)
  ;; Pad each menu item with blanks at the end so that the indentation
  ;; is preserved.  This is necessary because tv:menu-choose centers
  ;; all the items.
  ;; (This is not necessary for Explorer Release 3.1 and greater
  ;; because w:menu-choose can left justify.)
  (let* ((longest-string (mapc-max #'(lambda (item)
                                       (length (string (first item))))
                                   item-list))
         len)
    (dolist (item item-list)
      (setf len (length (string (first item))))
      (setf (first item)
	    (format nil "~a~[ ~:;~:1*~v@t ~]"
		    (first item) (- longest-string len))))
    item-list))

(defun build-space-menu-items-1 (node &optional (depth 0))

  (let ((indent (1+ (* 3 depth))))

    (cond
      ;; Base Case
      ((null node)
       nil)
      
      ;; No index necessary
      ((and (= (gbb::db-node.start node) 0)
	    (= (gbb::db-node.end node) 1))
       (if (gbb::space-type-p (gbb::db-node.type node))
	   ;; Leaf
	   `((,(format nil "~v@t~s" indent (gbb::db-node.name node))
	      :value ,(svref (gbb::db-node.vector node) 0)
              :font fonts:cptfontb))
	   ;; Not a Leaf
	   (cons
	     `(,(format nil "~v@t~s" indent (gbb::db-node.name node))
	       :no-select t :font fonts:cptfont)
	     (mapcan #'(lambda (node)
			 (build-space-menu-items-1 node (1+ depth)))
		     (svref (gbb::db-node.vector node) 0)))))
      
      ;; Index required
      (t
       (do* ((start (gbb::db-node.start node))
             (end (gbb::db-node.end node))
             (i start (1+ i))
             (result nil))
	   ((>= i end)
	    result)
	 (setf result
	       (nconc result
		      (if (gbb::space-type-p (gbb::db-node.type node))
			  ;; Leaf
			  `((,(format nil "~v@t~s [~d]" indent (gbb::db-node.name node) i)
			     :value ,(svref (gbb::db-node.vector node) (- i start))
                             :font fonts:cptfontb))
			  ;; Not a Leaf
			  (cons
			    `(,(format nil "~v@t~s [~d]" indent (gbb::db-node.name node) i)
			      :no-select t
                              :font fonts:cptfont)
			    (mapcan #'(lambda (node)
					(build-space-menu-items-1 node (1+ depth)))
				    (svref (gbb::db-node.vector node) (- i start))))))))))))


(defun pretty-space-path (space-instances &optional window font)

  "Return a string for SPACE-INSTANCES, which may be a list of
   space instances or a single space instance."

  (labels ((string-width-in-pixels (string &optional (start 0))
	    (send window :string-length
		  string start nil nil font)))

    (let* ((space-path
	     (format nil "~a" (gbb::get-path-from-space-instance
                                (if (listp space-instances)
                                    (first space-instances)
                                    space-instances))))
	   (window-width
	     ;; Inside width in pixels
	     (and window (send window :inside-width)))
	   (string-width
	     ;; String width in pixels
	     (and window (string-width-in-pixels space-path)))
           (more-marker
             ;; More than one space?
             (if (and (listp space-instances) (> (length space-instances) 1))
                 "++"
                 ""))
           (more-width
             (if window (string-width-in-pixels more-marker) 0)))

      (when (or (null window)
		(> window-width (+ string-width more-width)))
	(return-from pretty-space-path
          (concatenate 'string space-path more-marker)))

      ;; If the whole path won't fit then try taking elements off the
      ;; front of the path until it does fit.
      (do* ((start-index 0)
	    (next-space (position #\space space-path :start start-index)
			(position #\space space-path :start start-index))
	    (goal-width (- window-width
                           (string-width-in-pixels "... ")
                           more-width)))
	   ((null next-space)
	    ;; This will only happen if we can't fit any reasonable part
	    ;; of the space path into the width.
	    "??????????")
	(setf start-index (1+ next-space))
	(setf string-width (string-width-in-pixels space-path start-index))
	(when (<= string-width goal-width)
	  (return-from pretty-space-path
	    (concatenate 'string "... "
                         (subseq space-path start-index)
                         more-marker)))))))


(defvar *step-message* " Step.")

(defun wait-for-step-proceed (pane)

  "Wait for a keystroke in the ll pane."

  (let* ((frame (send pane :superior))
         (graphics-1 (send frame :get-pane 'graphics-1))
         (blip nil)
         (height (send pane :height))
         (font fonts:hl12b)
         (pane-step-message "Step")
         (message-width (+ 2 (send pane :string-length
                                   pane-step-message 0 nil nil font)))
         (message-height (w:font-char-height font))
         (start-y (- height message-height 2))
         (alu-on #+TI w:alu-add #+SYMBOLICS w:alu-xor)
         (alu-off #+TI w:alu-sub #+SYMBOLICS w:alu-xor)
         (color (- tv:blue (tv:sheet-background-color pane))))
    (labels
      ((block-on ()
         (w:sheet-force-access (pane)
           (w:prepare-sheet (pane)
             (tv:prepare-color (pane color)
               (sys::%draw-rectangle message-width message-height
                                     2 start-y
                                     alu-on pane))))
         (send pane :string-out-explicit
               pane-step-message 2 start-y nil nil fonts:hl12b alu-off))
       (block-off ()
         (w:sheet-force-access (pane)
           (w:prepare-sheet (pane)
             (tv:prepare-color (pane color)
               (sys::%draw-rectangle message-width message-height
                                     2 start-y
                                     alu-off pane))))
         (send pane :string-out-explicit
               pane-step-message 2 start-y nil nil fonts:hl12b alu-on))
       (background-process ()
         ;; Step inside the background process.
         ;; Temporarily change the frame's selection substitute to
         ;; be Graphics-1.  This causes all input to the frame to
         ;; go to Graphics-1's i/o buffer.  Note that graphics-1
         ;; is always exposed in any configuration.
         (w:with-selection-substitute (graphics-1 frame)
           (block-on)
           ;; Trap the <abort> key.
           (condition-case ()
               (loop
                (setq blip (send graphics-1 :any-tyi))
                (block-off)
                (cond ((fixnump blip)
                       ;; A character
                       (return-from wait-for-step-proceed nil))
                      (t ;; A blip
                       (process-blip blip)
                       (block-on))))
             ((sys:abort)
              (block-off)
              (throw 'top-level nil)))))
       (other-process ()
         ;; Step while running in another process.
         (unwind-protect
             (progn (block-on)
                    (format t *step-message*)
                    (read-char *standard-input*))
           (block-off))))

      (if (eq w:current-process (send pane :background-process))
          (background-process)
          (other-process)))))

#+IGNORE
(compile-flavor-methods gbb-graphics-frame
                        gbb-graphics-pane
                        gbb-lisp-listener-pane
                        gbb-logo-pane
                        gbb-menu-pane)

;;; ---------------------------------------------------------------------------
;;;                                End of File
;;; ---------------------------------------------------------------------------
