;;;; -*- Mode:LISP; Syntax:COMMON-LISP; Package:TSP; Base:10 -*-
;;;; *-* File: VAX6:DIS$DISK:[GBB.V-120.DISTRIBUTION.GBB1]TSP.LISP *-*
;;;; *-* Edited-By: Gallagher *-*
;;;; *-* Last-Edit: Sunday, February 12, 1989  21:45:08 *-*
;;;; *-* Machine: Gilgamesh (Explorer II, Microcode 308) *-*
;;;; *-* Software: TI Common Lisp System 4.61 *-*
;;;; *-* Lisp: TI Common Lisp System 4.61 (0.0) *-*

;;;; **************************************************************************
;;;; **************************************************************************
;;;; *
;;;; *       GBB1 KS Shell: Travelling Salesman Example
;;;; *
;;;; **************************************************************************
;;;; **************************************************************************
;;;
;;; Written by: Philip Johnson
;;;             Department of Computer and Information Science
;;;             University of Massachusetts
;;;             Amherst, Massachusetts 01003.
;;;
;;; This code was written as part of the GBB (Generic Blackboard) system at
;;; the Department of Computer and Information Science (COINS), University of 
;;; Massachusetts, Amherst, Massachusetts 01003.
;;;
;;; Copyright (c) 1986, 1987, 1988 COINS.  
;;; All rights are reserved.
;;;
;;; Development of this code was partially supported by:
;;;    Donations from Texas Instruments, Inc.;
;;;    NSF CER grant DCR-8500332;
;;;    ONR URI contract N00014-86-K-0764.
;;;
;;; Permission to copy this software, to redistribute it, and to use it for
;;; any purpose is granted, subject to the following restrictions and
;;; understandings.
;;;
;;; 1.  Title and copyright to this software and any material associated
;;; therewith shall at all times remain with COINS.  Any copy made of this
;;; software must include this copyright notice in full.
;;;
;;; 2.  The user acknowledges that the software and associated materials
;;; are provided as a research tool that remains under active development
;;; and is being supplied ``as is'' for the purposes of scientific
;;; collaboration aimed at further development and application of the
;;; software and the exchange of technical data.
;;;
;;; 3.  All software and materials developed as a consequence of the use of
;;; this software shall duly acknowledge such use, in accordance with the
;;; usual standards of acknowledging credit in academic research.
;;;
;;; 4.  Users of this software agree to make their best efforts to inform
;;; the COINS GBB Development Group of noteworthy uses of this software.
;;; The COINS GBB Development Group can be reached at:
;;;
;;;     GBB Development Group
;;;     C/O Dr. Daniel D. Corkill
;;;     Department of Computer and Information Science
;;;     Lederle Graduate Research Center
;;;     University of Massachusetts
;;;     Amherst, Massachusetts 01003
;;;
;;;     (413) 545-0156
;;;
;;; or via electronic mail:
;;;
;;;     GBB@CS.UMass.Edu
;;;
;;; Users are further encouraged to make themselves known to this group so
;;; that new releases, bug fixes, and tutorial information can be
;;; distributed as they become available.
;;;
;;; 5.  COINS makes no representations or warranties of the merchantability
;;; or fitness of this software for any particular purpose; that uses of
;;; the software and associated materials will not infringe any patents,
;;; copyrights, trademarks, or other rights; nor that the operation of this
;;; software will be error-free.  COINS is under no obligation to provide
;;; any services, by way of maintenance, update, or otherwise.  
;;;
;;; 6.  In conjunction with products or services arising from the use of
;;; this material, there shall be no use of the name of the Department of
;;; Computer and Information Science or the University of Massachusetts in
;;; any advertising, promotional, or sales literature without prior written
;;; consent from COINS in each case.
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;;;
;;;  06-18-87 File Created.  (Johnson)
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

(in-package "TSP")

(use-package '(lisp umass-extended-lisp gbb gbb1))

(export '(tsp))


;;;-----------------------------------------------------------------------------
;;;
;;;  Load the PRESCRIPTION control KSs.
;;;
;;;-----------------------------------------------------------------------------

(defparameter user::*gbb1-example-directory*
              (user::gbb-subdirectory "gbb1")
  "Directory where the GBB1 example files will be stored.")

(require "GBB1-PRESCRIPTIONS"
         (user::gbb-file "prescription-ks" "gbb1"))


;;;-----------------------------------------------------------------------------
;;;
;;;  The Traveling Salesman Problem
;;;
;;;-----------------------------------------------------------------------------

#||

The travelling salesman problem is to find the shortest path between a set
of cities, which is a very difficult problem indeed.  This application does
not attempt to find *the* shortest path, but rather *a* short path.  To do
this, it uses two heuristics in deciding how to construct a path.  The first
is to prefer short arcs (connect each city to its two closest neighbors),
and the second is to prefer outer arcs (don't zigzag).  The structure of
this application, described below, is taken directly from the BB1 TSP
system.

This system has two domain-specific blackboards: TOUR-INFO and SOLUTION.
The TOUR-INFO blackboard has a single space, CITIES, which stores a
blackboard object containing information about each city to be visited.  The
SOLUTION blackboard contains a single space, PATHS, which stores blackboard
objects containing information about the current status of the solution
path.

The domain blackboard objects are CITY and PATH.  The CITY object contains
the name of a city, its location in terms of X and Y coordinates, and the
number of times it appears in the current solution path.  The PATH object
contains the cities left to visit, the cities which have been visited, the
arcs, and information about the global "geography" (the average, minimum,
and maximum X and Y values.)

The domain KSs are START-TSP, ADD-ARC, and ADD-FINAL-ARC.  START-TSP is the
initial KS, and simply initializes the the objects on the domain
blackboards.  ADD-ARC is triggered by the initialization of the SOLUTION
blackboard, and creates one KSAR for every city to city combination.  These
Ksars remain active (i.e.  are not obviated) only as long as their cities
have not been already used in the solution path, and the addition of the arc
would not cause a loop to develop.  ADD-FINAL-ARC is similar to ADD-ARC,
except that it is triggered when there is only one more arc to be added to
complete the solution path (which will cause a loop in the path).

The domain KSs described above are sufficient to construct a solution path.
The remaining control KSs provide the information required for deciding
which arc to add to the solution path at any point in time.  As stated
above, the control plan uses two heuristic KSs in rating the arcs available:
PREFER-SHORT-ARCS and PREFER-OUTER-ARCS.  These heuristics implement the
control focus KS called CREATE-GOOD-PATH, which implements the strategy KS
called FIND-SHORT-PATH.  To construct this hierarchical control plan from
these control KSs, this system uses the "prescription" metacontrol KSs which
are described in detail in the GBB1 chapter of the GBB reference manual.

||#

;;;-----------------------------------------------------------------------------
;;;
;;;  Variables and constants
;;;
;;;-----------------------------------------------------------------------------

;; The Y dimension is represented as two times the number of degrees
;; lattitude so that the Y distances are approximately the same distance
;; as X distances.  (Obviously, longitude and latitude are not the best
;; measures to use here.)

(defconstant *y-factor* 1.5)
(defconstant *min-lattitude* #.(* 20 *y-factor*))
(defconstant *max-lattitude* #.(* 60 *y-factor*))
(defconstant *lattitude-range* '(#.*min-lattitude* #.*max-lattitude*))

;; The X dimension is represented as degress _East_ of Greenwich rather
;; than West of Greenwich as is the normal convention.  This is so the
;; numbers will increase from left to right and the map will look
;; reasonable when graphed.

(defconstant *min-longitude* #.(- 360 140))
(defconstant *max-longitude* #.(- 360 60))
(defconstant *longitude-range* '(#.*min-longitude* #.*max-longitude*))


;;;-----------------------------------------------------------------------------
;;;
;;;  Define the domain bb objects.
;;;
;;;-----------------------------------------------------------------------------

(format t "~%~a ~a" (short-date-and-time) "Defining Domain Objects")

(define-gbb1-unit CITY

  "CITY contains all the necessary information about a particular city."

  :SLOTS ((x             0)
	  (y             0)
	  (times-in-path 0))

  :LINKS ((free (path cities-to-visit))
	  (used (path cities-visited)))

  :DIMENSIONAL-INDEXES
         ((x x :type :point)
	  (y y :type :point))

  :PATHS ((:path '(tour-info cities))))  

;;;-----------------------------------------------------------------------------

(define-gbb1-unit PATH

  "PATH contains the information about the current state of the solution
   path and about the distribution of the cities that are to be visited."

  :EVENT-CLASSES  (:creation-events)
  
  :SLOTS ((arcs nil)
	  (x-ave 0)	  (y-ave 0)
	  (x-min 0)	  (y-min 0)
	  (x-max 0)	  (y-max 0)
	  (length 0))

  :LINKS ((cities-to-visit (city free))
	  (cities-visited  (city used)))

  :PATHS ((:path '(solution paths))))

;;;-----------------------------------------------------------------------------
;;;
;;;  Define the domain BB structures
;;;
;;;-----------------------------------------------------------------------------

(define-space PATHS
  "The space which holds the solution path object"
  :units (path))

(define-space CITIES
  "The space which holds the city objects."
  :units (city)
  :dimensions ((x :ordered #.*longitude-range*)
	       (y :ordered #.*lattitude-range*)))

(define-blackboard SOLUTION (paths)
  "The blackboard to hold the solution path")

(define-blackboard TOUR-INFO (cities)
  "The blackboard to hold the info about cities.")

;;;-----------------------------------------------------------------------------
;;;
;;;  Define domain KSs: START-TSP
;;;
;;;-----------------------------------------------------------------------------

(format t "~%~a ~a" (short-date-and-time) "Defining START-TSP")

(define-gbb1-domain-ks START-TSP
   
  "START TSP is the initial KS to be run by the TSP system.
   It creates the city objects and a solution path object with the
   appropriate initial values."

  :TRIGGER-CONDITIONS (())
  :ACTION-FUNCTION #'(lambda (start-tsp-ksar)
		       (declare (ignore start-tsp-ksar))
		       (create-city-objects)
		       (create-path-object)))

;;;-----------------------------------------------------------------------------

;; Note: this algorithm really loses when cities in the middle
;; of the continent are included.

(defvar *the-cities*
        '(;;               Latitude   Longitude
         ;("Montreal"         46         74)
          ("Boston"           42         71)
          ("New York"         41         74)
          ("Toronto"          44         79)
          ("Washington DC"    39         75)
          ("Atlanta"          34         84)
          ("Miami"            26         80)
          ("Chicago"          42         88)
         ;("Kansas City"      39         95)
          ("Dallas"           33         96)
         ;("Denver"           40        105)
          ("Calgary"          51        114)
          ("Las Vegas"        36        115)
          ("Los Angeles"      34        118)
          ("San Francisco"    38        123)
          ("Seattle"          47        122)
          )
  "List of cities through which the salesman must travel.")


(defun CREATE-CITY-OBJECTS ()

  "CREATE-CITY-OBJECTS nil

   This function creates a whole bunch of CITY blackboard objects."

  (dolist (city-info *the-cities*)
    (let ((city-name (first  city-info))
          (latitude (second city-info))
          (longitude (third  city-info)))
      (make-city
        :name city-name
        :x    (- 360 longitude)        ;; Convert west longitude to east longitude.
        :y    (* latitude *y-factor*)  ;; Convert lattitude to Y units.
        :times-in-path 0))))

;;;-----------------------------------------------------------------------------

(defun CREATE-PATH-OBJECT ()

  "CREATE-PATH-OBJECT nil

   This function sets up the solution path blackboard object."

  (let* ((city-list (find-units 'city (make-paths :paths '(tour-info cities)) :all))
	 (num-cities (length city-list))
	 (x-average 0.0)
	 (y-average 0.0)
	 (x-max     *min-longitude*)
	 (y-max     *min-lattitude*)
	 (x-min     *max-longitude*)
	 (y-min     *max-lattitude*))

    ;; Compute the city "geography" info for later use by the heuristics.
    ;; The min and max values are initialised (above) to the max and min
    ;; (note the reversal) values for the space.
    (dolist (city city-list)
      (incf x-average  (city$x city))
      (incf y-average  (city$y city))
      (setf x-max (max x-max (city$x city))
            y-max (max y-max (city$y city))
            x-min (min x-min (city$x city))
            y-min (min y-min (city$y city))))
    
    (setf x-average (/ x-average num-cities)
	  y-average (/ y-average num-cities))

   (make-path
    :name            "Solution Path"
    :cities-to-visit  city-list
    :x-ave            x-average
    :y-ave            y-average		   
    :x-min            x-min
    :y-min            y-min
    :x-max            x-max 
    :y-max            y-max
    :length           num-cities)))

;;;-----------------------------------------------------------------------------
;;;
;;;  Define domain KSs: ADD-ARC
;;;
;;;-----------------------------------------------------------------------------

(format t "~%~a ~a" (short-date-and-time) "Defining ADD-ARC")

(define-gbb1-domain-ks ADD-ARC
	   
  "ADD-ARC is triggered by the initialization of the system, and triggers
   KSARs to add arcs for all pair-wise combinations of cities in the system.
   These KSARs are obviated when the addition either of the cities at each 
   end of the arc have already been used in the path, or if the addition of
   the arc would make a loop."

  :TRIGGER-CONDITIONS   ((trigger-event-class-p :unit-creation)
			 (trigger-event-level-p (make-paths :paths '(solution paths))))
  
  :PRECONDITIONS        ((and (< (city$times-in-path (add-arc-ksar$from-city *this-ksar*)) 2)
                              (< (city$times-in-path (add-arc-ksar$to-city *this-ksar*)) 2)))
  
  :OBVIATION-CONDITIONS ((or (>= (city$times-in-path (add-arc-ksar$from-city *this-ksar*)) 2)
                             (>= (city$times-in-path (add-arc-ksar$to-city *this-ksar*)) 2)
			     (makes-loop (add-arc-ksar$from-city *this-ksar*)
					 (add-arc-ksar$to-city *this-ksar*))))
  
  :CONTEXT-SLOTS      ((from-city to-city) (pairwise-cities-list))
  :ACTION-FUNCTION    #'add-arc-to-path
  :FROM-BB            (make-paths :paths '(solution paths))
  :TO-BB              (make-paths :paths '((solution paths) (tour-info cities)))
  :COST               15
  :RELIABILITY        100
  :AUTHOR             "Philip Johnson")
  
;;;-----------------------------------------------------------------------------

(defun PAIRWISE-CITIES-LIST ()

  "PAIRWISE-CITIES-LIST nil

   This function returns a list of all from- and to- city combinations."

  (let ((city-list (find-units 'city (make-paths :paths '(tour-info cities)) :all))
	(pairwise-list nil))

    (dolist (city1 city-list)
      (setf pairwise-list
	    (nconc (mapc-condcons #'(lambda (city2)
				      (unless (or (eq city1 city2)
						  (string> (city$name city1) 
							   (city$name city2)))
					(list city1 city2)))
				  city-list)
                   pairwise-list)))
    
    pairwise-list))

;;;-----------------------------------------------------------------------------

(defun ONLY-ONCE-IN-PATH (city)
  (= (city$times-in-path city) 1))

(defun ADD-ARC-TO-PATH (add-arc-ksar)

  "ADD-ARC-TO-PATH add-arc-ksar

   This action adds the arc indicated by the from and to slot of the ksar
   to the solution path."

  (let ((solution (find-path "Solution Path"))
	(from-city (add-arc-ksar$from-city add-arc-ksar))
	(to-city   (add-arc-ksar$to-city add-arc-ksar)))

    (incf (path$length solution) (distance-between from-city to-city))
    (linkf (path$cities-visited solution) from-city)
    (linkf (path$cities-visited solution) to-city)
    (push (list from-city to-city) (path$arcs solution))
    (incf (city$times-in-path from-city))
    (incf (city$times-in-path to-city))))

;;;-----------------------------------------------------------------------------

(defun MAKES-LOOP (from-city to-city
		   &optional (existing-arcs (path$arcs (find-path "Solution Path"))))

  "MAKES-LOOP from-city to-city &optional existing-arcs

   This function taken straight out of the BB1 TSP system.
   Note - this only works for graphs which have < 2 arcs out of each node.
   Check that condition beforehand!"


  (let ((next-city-1 nil)
	(next-city-2 nil))

    (cond ((or (null from-city) (null to-city))
	   nil)
	  ((eq from-city to-city)
	   t)
	  (t (setq next-city-1 (find from-city existing-arcs :key #'car))
	     (setq next-city-2 (find from-city existing-arcs :key #'cadr))
	     (or (makes-loop (cadr next-city-1) to-city
			     (remove from-city existing-arcs :key #'car))
		 (makes-loop (car next-city-2) to-city
			     (remove from-city existing-arcs :key #'cadr)))))))

;;;-----------------------------------------------------------------------------

(defun DISTANCE-BETWEEN (city1 city2)

  "DISTANCE-BETWEEN city1 city2

   Computes the distance between 2 cities."

  (let ((from-x (City$x city1))
	(to-x   (City$x city2))
        (from-y (City$y city1))
	(to-y   (City$y city2)))

    (sqrt (+ (* (- to-x from-x) (- to-x from-x))
	     (* (- to-y from-y) (- to-y from-y))))))

;;;-----------------------------------------------------------------------------
;;;
;;;  Define domain KSs: ADD-FINAL-ARC
;;;
;;;-----------------------------------------------------------------------------

(format t "~%~a ~a" (short-date-and-time) "Defining ADD-FINAL-ARC")

(define-gbb1-domain-ks ADD-FINAL-ARC

  "Adds the final arc to the path.  This case must be distinguished from
   the general ADD-ARC ks because it will cause a loop in the path."


  :TRIGGER-CONDITIONS ((trigger-event-class-p :unit-creation)
		       (trigger-event-level-p (make-paths :paths '(solution paths))))
  :CONTEXT-SLOTS      ((num-cities) `((,(length (path$cities-to-visit (find-path "Solution Path"))))))
  :PRECONDITIONS      ((= (1- (add-final-arc-ksar$num-cities *this-ksar*))
			      (length (path$arcs (find-path "Solution Path")))))
  :ACTION-FUNCTION    #'add-final-arc-to-path
  :FROM-BB            (make-paths :paths '(solution paths))
  :TO-BB              (make-paths :paths '((solution paths) (tour-info cities)))
  :COST               15
  :RELIABILITY        100
  :AUTHOR             "Philip Johnson")

;;;-----------------------------------------------------------------------------

(defun ADD-FINAL-ARC-TO-PATH (add-final-arc-ksar)

  "ADD-FINAL-ARC-TO-PATH add-final-arc-ksar

   Adds the final arc to the path and prints out a message saying so."

  (declare (ignore add-final-arc-ksar))

  (let* ((solution (find-path "Solution Path"))
         (remaining-cities (find-units t (make-paths :paths '(tour-info cities) )
                                         :all
                                         :filter-before '(only-once-in-path)))
	 (from-city (first remaining-cities))
	 (to-city (second remaining-cities)))

    (incf (path$length solution) (distance-between from-city to-city))
    (linkf (path$cities-visited solution) from-city)
    (linkf (path$cities-visited solution) to-city)
    (push (list from-city to-city) (path$arcs solution))
    (incf (city$times-in-path from-city))
    (incf (city$times-in-path to-city))

    (format t "~2%--------------------------------~@
               The salesman has completed his travels.~@
               The route is:~%")
    (show-solution solution)
    (format t "--------------------------------~%")))

(defun show-solution (solution &optional (indent 6))
  (let* ((arc (first (path$arcs solution)))
         (next-city (second arc))
         (remaining-arcs (copy-list (cdr (path$arcs solution)))))
    (format t "~v@t~a~%" indent (city$name (first arc)))
    (format t "~v@t~a~%" indent (city$name next-city))
    (do ()
        ((null remaining-arcs))
      (setf arc (find next-city remaining-arcs :test #'member))
      (unless arc
        (error "Some arc is missing.  Can't find another two arcs for ~a."
               next-city))
      (setf remaining-arcs (delete arc remaining-arcs :test #'eq))
      (setf next-city (if (eq next-city (first arc))
                          (second arc)
                          (first arc)))
      (format t "~v@t~a~%" indent (city$name next-city)))))
     

;;;-----------------------------------------------------------------------------
;;;
;;;  Define Control KSs: Strategy FIND-SHORT-PATH
;;;
;;;-----------------------------------------------------------------------------

(format t "~%~a ~a" (short-date-and-time) "Defining FIND-SHORT-PATH")

(define-gbb1-control-ks FIND-SHORT-PATH

  "Strategy to find a short path.  It is implemented by CREATE-GOOD-PATH."

  :CONTROL-TYPE          :STRATEGY
  :TRIGGER-CONDITIONS    ((trigger-event-level-p (make-paths :paths '(solution paths)))
			  (trigger-event-class-p :unit-creation))
  :ACTION-FUNCTION       #'CREATE-FIND-PATH-PO
  :FROM-BB               (make-paths :paths '(solution paths))
  :TO-BB                 (make-paths :paths '(gbb1 control control-plan))
  :AUTHOR                "Philip Johnson")

;;;-----------------------------------------------------------------------------

(defun NO-MORE-PRESCRIPTION (strategy-po)

  "NO-MORE-PRESCRIPTION strategy-po

   The goal of the strategy is satisfied when the future prescription slot is
   nil."

  (null (basic-strategy-po$future-prescription strategy-po)))

;;;-----------------------------------------------------------------------------

(defun CREATE-FIND-PATH-PO (find-path-ksar)

  "CREATE-FIND-PATH-PO find-path-ksar

   Adds a find-short-path strategy PO to the control plan blackboard."

  (declare (ignore find-path-ksar))

  (make-find-short-path-po
    :NAME                "Find Short Path"
    :STATUS              :OPERATIVE
    :FUTURE-PRESCRIPTION '(create-good-path)
    :GOAL-FUNCTION       #'no-more-prescription))


;;;-----------------------------------------------------------------------------
;;;
;;; Define Control KSs: CREATE-GOOD-PATH
;;;
;;;-----------------------------------------------------------------------------

(format t "~%~a ~a" (short-date-and-time) "Defining CREATE-GOOD-PATH")

(define-gbb1-control-ks CREATE-GOOD-PATH

  "This focus creates a good path by using heuristics to prefer outer and 
   shorter arcs. Its goal is achieved when the path is created."

  :CONTROL-TYPE        :FOCUS
  :TRIGGER-CONDITIONS  ((strategy-po-p (trigger-unit))
			(trigger-event-class-p :link-update)
			(current-prescription-p 'create-good-path (trigger-unit)))
  :CONTEXT-SLOTS       ((triggering-strategy) `((,(trigger-unit))))
  :ACTION-FUNCTION     #'create-good-path-po
  :FROM-BB             (make-paths :paths '(gbb1 control control-plan))
  :TO-BB               (make-paths :paths '(gbb1 control control-plan))
  :COST                25
  :RELIABILITY         80)

;;;-----------------------------------------------------------------------------

(defun CREATE-GOOD-PATH-PO (create-path-ksar)

  "CREATE-GOOD-PATH-PO create-path-ksar

   This action creates a focus plan object for creating a path."

  (make-create-good-path-po
    :NAME                        "Create Good Path"
    :STATUS                      :OPERATIVE
    :STABILITY                   :STABLE
    :IMPLEMENTOR-OF              (create-good-path-ksar$triggering-strategy
                                   create-path-ksar)
    :WEIGHT                      1
    :GOAL-FUNCTION               #'create-good-path-goal
    :INTEGRATED-RATING-FUNCTION  #'sum-of-weights-times-ratings
    :HEURISTICS                  '(prefer-short-arcs prefer-outer-arcs)))

;;;-----------------------------------------------------------------------------

(defun CREATE-GOOD-PATH-GOAL (create-good-path-po)

  "CREATE-GOOD-PATH-GOAL create-good-path-po

   Returns T if the number of arcs in the solution path equals the 
   number of cities to be visited."

  (declare (ignore create-good-path-po))

  (let ((solution (find-path "Solution Path")))
    (= (length (path$arcs solution))
       (length (path$cities-to-visit solution)))))

;;;-----------------------------------------------------------------------------
;;;
;;; Define control KSs: Heuristic PREFER-OUTER-ARCS
;;;
;;;-----------------------------------------------------------------------------

(format t "~%~a ~a" (short-date-and-time) "Defining PREFER-OUTER-ARCS")

(define-gbb1-control-ks PREFER-OUTER-ARCS

  "Prefers outer arcs."

  :CONTROL-TYPE       :HEURISTIC
  :TRIGGER-CONDITIONS ((focus-po-p (trigger-unit))
		       (trigger-event-class-p :unit-creation)
		       (member 'prefer-outer-arcs (basic-focus-po$heuristics (trigger-unit))))
  :CONTEXT-SLOTS      ((triggering-focus) `((,(trigger-unit))))
  :ACTION-FUNCTION    #'CREATE-OUTER-ARCS-PO
  :FROM-BB            (make-paths :paths '(gbb1 control control-plan))
  :TO-BB              (make-paths :paths '(gbb1 control control-plan))
  :COST               25
  :RELIABILITY        100
  :AUTHOR             "Philip Johnson")

;;;-----------------------------------------------------------------------------

(defun CREATE-OUTER-ARCS-PO (prefer-outer-arcs-ksar)

  "CREATE-OUTER-ARCS-PO prefer-outer-arcs-ksar

   Creates a heuristic PO which prefers outer arcs."

  (make-prefer-outer-arcs-po
    :NAME            "Prefer Outer Arcs"
    :STATUS          :OPERATIVE
    :STABILITY       :STABLE
    :IMPLEMENTOR-OF  (prefer-outer-arcs-ksar$triggering-focus prefer-outer-arcs-ksar)
    :WEIGHT          2
    :RATING-FUNCTION #'rate-outer-arcs-higher))

;;;-----------------------------------------------------------------------------

(defun RATE-OUTER-ARCS-HIGHER (ksar)

  "RATE-OUTER-ARCS-HIGHER ksar

   If the ksar is an ADD-ARC, it rates it more highly if it's an outer arc."

  (if (typep ksar 'add-arc-ksar)
      (outer-arc-rating (add-arc-ksar$from-city ksar) (add-arc-ksar$to-city ksar))
      0))

;;;-----------------------------------------------------------------------------

(defun OUTER-ARC-RATING (from-city to-city)
  
  "OUTER-ARC-RATING from-city to-city

   Returns a rating of the arc based upon its outsideness."
  
  ;; this is another modification of code from the BB1 TSP system.
  
  (let* ((x1       (city$x from-city))
	 (y1       (city$y from-city))
	 (x2       (city$x to-city))
	 (y2       (city$y to-city))
	 (path     (find-path "Solution Path"))
 	 (center-x (path$x-ave path))
	 (center-y (path$y-ave path))
	 (delta-x  (path$x-max path))
	 (delta-y  (path$y-max path)))
    
    (* 250 (+ (float (/ (abs (+ (- x1 center-x)
                                (- x2 center-x)))
                        delta-x))
              (float (/ (abs (+ (- y1 center-y)
                                (- y2 center-y)))
                        delta-y))))))

;;;-----------------------------------------------------------------------------
;;;
;;; Define control KSs: Heuristic PREFER-SHORT-ARCS
;;;
;;;-----------------------------------------------------------------------------

(format t "~%~a ~a" (short-date-and-time) "Defining PREFER-SHORT-ARCS")

(define-gbb1-control-ks PREFER-SHORT-ARCS

  "Prefers short arcs."

  :CONTROL-TYPE       :HEURISTIC
  :TRIGGER-CONDITIONS ((focus-po-p (trigger-unit))
		       (trigger-event-class-p :unit-creation)
		       (member 'prefer-short-arcs (basic-focus-po$heuristics (trigger-unit))))
  :CONTEXT-SLOTS      ((triggering-focus) `((,(trigger-unit))))
  :ACTION-FUNCTION    #'CREATE-SHORT-ARCS-PO
  :FROM-BB            (make-paths :paths '(gbb1 control control-plan))
  :TO-BB              (make-paths :paths '(gbb1 control control-plan))
  :COST               25
  :RELIABILITY        100
  :AUTHOR             "Philip Johnson")

;;;-----------------------------------------------------------------------------

(defun CREATE-SHORT-ARCS-PO (prefer-short-arcs-ksar)

  "CREATE-SHORT-ARCS-PO prefer-short-arcs-ksar

   Creates a heuristic PO which prefers short arcs."

  (make-prefer-short-arcs-po
    :NAME            "Prefer Short Arcs"
    :STATUS          :OPERATIVE
    :STABILITY       :STABLE
    :IMPLEMENTOR-OF  (prefer-short-arcs-ksar$triggering-focus prefer-short-arcs-ksar)
    :WEIGHT          3
    :RATING-FUNCTION #'rate-short-arcs-higher))

;;;-----------------------------------------------------------------------------

(defun RATE-SHORT-ARCS-HIGHER (ksar)

  "RATE-SHORT-ARCS-HIGHER ksar

   If the ksar is an ADD-ARC, it rates it more highly if it's a short arc."

  (if (typep ksar 'add-arc-ksar)
      (/ 1000 (distance-between (add-arc-ksar$from-city ksar) (add-arc-ksar$to-city ksar)))
      0))

;;;-----------------------------------------------------------------------------
;;;
;;; Start things up.
;;;
;;;-----------------------------------------------------------------------------

(defun TSP-PRIORITY (ksar)

  "TSP-PRIORITY nil

   The tsp priority  is the sum of the integrated ratings."
  
  (let ((total 0))
    (map-focus-ratings #'(lambda (integrated-rating po)
			   (declare (ignore po))
			   (incf total integrated-rating))
 		      ksar)
    
    total))

;;;-----------------------------------------------------------------------------
;;;
;;;   Create the domain blackboards
;;;
;;;-----------------------------------------------------------------------------

(define-gbb1-parameters
  :priority-fn                   #'tsp-priority
  :priority-fn-stability         :stable
  :recommendation-fn             #'first
  :termination-fn                #'(lambda () nil)
  :precondition-recheck-interval nil      
  :obviation-recheck-interval    1
  :max-execution-cycles          25)

(define-gbb1-output
  :trace-fn           nil
  :trace-print-points '(:before-agenda-update :final-state)
  :print-unit-width   25
  :output-stream      *standard-output*)

(defun tsp ()
  (instantiate-blackboard-database 'tour-info 'solution :mode :overwrite)  
  (run-gbb1 :initial-ks 'start-tsp))

;;;-----------------------------------------------------------------------------
;;;                                End of file
;;;-----------------------------------------------------------------------------
