;;;; -*- Mode:LISP; Syntax:COMMON-LISP; Package:GBB; Base:10 -*-
;;;; *-* File: VAX6:DIS$DISK:[GBB.V-120.LOCAL]FIND-UNIT.LISP *-*
;;;; *-* Edited-By: Gallagher *-*
;;;; *-* Last-Edit: Monday, July 31, 1989  17:28:45 *-*
;;;; *-* Machine: Gilgamesh (Explorer II, Microcode 416) *-*
;;;; *-* Software: TI Common Lisp System 4.105 *-*
;;;; *-* Lisp: TI Common Lisp System 4.105 (0.0) *-*

;;;; **************************************************************************
;;;; **************************************************************************
;;;; *
;;;; *                        FIND UNIT FUNCTIONS
;;;; *
;;;; **************************************************************************
;;;; **************************************************************************
;;;
;;; Written by: Kevin Gallagher
;;;             Department of Computer and Information Science
;;;             University of Massachusetts
;;;             Amherst, Massachusetts 01003.
;;;
;;; This code was written as part of the GBB (Generic Blackboard) system at
;;; the Department of Computer and Information Science (COINS), University of 
;;; Massachusetts, Amherst.
;;;
;;; Copyright (c) 1986, 1987, 1988 COINS.  
;;; All rights reserved.
;;;
;;; Development of this code was partially supported by:
;;;    NSF CER grant DCR-8500332;
;;;    Donations from Texas Instruments, Inc.;
;;;    ONR URI grant N00014-86-K-0764.
;;;
;;; Permission to copy this software, to redistribute it, and to use it for
;;; any purpose is granted, subject to the following restrictions and
;;; understandings.
;;;
;;; 1.  Title and copyright to this software and any material associated
;;; therewith shall at all times remain with COINS.  Any copy made of this
;;; software must include this copyright notice in full.
;;;
;;; 2.  The user acknowledges that the software and associated materials
;;; are provided as a research tool that remains under active development
;;; and is being supplied ``as is'' for the purposes of scientific
;;; collaboration aimed at further development and application of the
;;; software and the exchange of technical data.
;;;
;;; 3.  All software and materials developed as a consequence of the use of
;;; this software shall duly acknowledge such use, in accordance with the
;;; usual standards of acknowledging credit in academic research.
;;;
;;; 4.  Users of this software agree to make their best efforts to inform
;;; the COINS GBB Development Group of noteworthy uses of this software.
;;; The COINS GBB Development Group can be reached at:
;;;
;;;     GBB Development Group
;;;     C/O Dr. Daniel D. Corkill
;;;     Department of Computer and Information Science
;;;     Lederle Graduate Research Center
;;;     University of Massachusetts
;;;     Amherst, Massachusetts 01003
;;;
;;;     (413) 545-0156
;;;
;;; or via electronic mail:
;;;
;;;     GBB@CS.UMass.Edu
;;;
;;; Users are further encouraged to make themselves known to this group so
;;; that new releases, bug fixes, and tutorial information can be
;;; distributed as they become available.
;;;
;;; 5.  COINS makes no representations or warranties of the merchantability
;;; or fitness of this software for any particular purpose; that uses of
;;; the software and associated materials will not infringe any patents,
;;; copyrights, trademarks, or other rights; nor that the operation of this
;;; software will be error-free.  COINS is under no obligation to provide
;;; any services, by way of maintenance, update, or otherwise.  
;;;
;;; 6.  In conjunction with products or services arising from the use of
;;; this material, there shall be no use of the name of the Department of
;;; Computer and Information Science or the University of Massachusetts in
;;; any advertising, promotional, or sales literature without prior written
;;; consent from COINS in each case.
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;;;
;;;  01-08-86 File Created.  (Gallagher)
;;;  07-16-86 Added all the functions from NEW-FIND-UNIT, which implements the
;;;           new pattern language, to this file.  (Gallagher)
;;;  07-16-86 Fixed retrieval from unstructured spaces.  (Gallagher)
;;;  08-25-86 Redo handling of unstructured spaces and unstructured unit
;;;           mappings.  (Gallagher)
;;;  10-13-86 Completely reimplemented FIND-UNITS.  (Gallagher)
;;;  01-05-87 Added :AND concatenation of patterns.  (Gallagher)
;;;  01-16-87 Added the ability to retrieve set index-structure types.  Also
;;;           Updated MAP-SPACE and FIND-UNITS to use path structures and
;;;           replaced some MAPL's with functions to make the code clearer.
;;;           (Gallagher)
;;;  01-29-87 Added MAP-UNIT-TYPE (Johnson)
;;;  03-11-87 More work on sets.  Added the ability to compare scalar unit
;;;           instances with set patterns and scalar patterns with set unit
;;;           instances.  (Gallagher)
;;;  01-19-88 Cleaned up array region determination code to eliminate
;;;           overlapping regions.  (Gallagher)
;;;  01-25-88 Added array intersection.  (Gallagher)
;;;  03-10-88 Improved efficiency of find algorithms.  (Gallagher)
;;;  06-16-88 Changed FIND-UNITS to be a function.  The macro had function
;;;           semantics anyway and any optimizations we might do based on the
;;;           pattern can be done by a compiler transform.  (Gallagher)
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

(in-package 'gbb :USE '(lisp umass-extended-lisp))

(export '(find-units
          map-space
          map-unit-type
          map-unit-types))

(use-package '(umass-extended-lisp))

(proclaim `(optimize (speed ,*gbb-optimize-speed*)
                     (safety ,*gbb-optimize-safety*)))


;;;; --------------------------------------------------------------------------
;;;;   Unit Marking
;;;; --------------------------------------------------------------------------

;;; These numbers are used to mark unit instances as they are being
;;; processed by find-units and map-space.

(defconstant *find-rejected*   255
  "Number used to indicate that a unit has failed some test and should
   be eliminated from further consideration.")

(defconstant *find-succeeded*   254
  "Number used to indicate that a unit has passed all tests.")

(defconstant *find-max-counts* 127
  "Highest number that will be used in marking successful units.")


;;;; --------------------------------------------------------------------------
;;;;   Mapping
;;;; --------------------------------------------------------------------------

(defun map-unit-types (fcn unit-types)

  "MAP-UNIT-TYPES fcn unit-types

   Apply a function to all named unit instances of a given set of types.
   UNIT-TYPES is a unit type or list of unit types.  UNIT-TYPES may also
   be T, in which case FCN is applied to all named unit instances in
   the database.  FCN is a function of one argument that is applied
   to each instance in turn.  MAP-UNIT-TYPES returns NIL --- FCN is
   applied for side effect only."

  (let ((all-units (eq unit-types t))
        (start-runtime (gbb-runtime)))

    (setf unit-types (if all-units
                         *all-units*
                         (expand-unit-type-list unit-types)))

    (dolist-or-atom (unit-type unit-types)
      (cond ((anonymous-unit-type-p unit-type)
             (unless all-units
               (cerror "Skip ~s unit types."
                       "~s unit types are anonymous and so MAP-UNIT-TYPES ~
                        doesn't work with them."
                       unit-type)))
            (t (maphash #'(lambda (key value)
                            (declare (ignore key))
                            (funcall fcn value))
                        (unit-info.hash-table (get-unit-info unit-type))))))
    (update-runtime *map-unit-types-meter* start-runtime)
    (update-count *map-unit-types-count*)
    nil))


(proclaim '(inline map-unit-type))

(defun map-unit-type (fcn unit-types)

  "MAP-UNIT-TYPE fcn unit-types

   Old name for MAP-UNIT-TYPES.  Retained for compatiblity."

  (map-unit-types fcn unit-types))

;;; ----------------------------------------------------------------------

(defun map-space (function unit-types paths)

  "MAP-SPACE function unit-types paths

   Apply a function to all unit instances on a space.  PATHS can be a
   path-structure or a list of path-structures.  FUNCTION is a function
   of one argument that is applied to each instance in turn.  UNIT-TYPES
   restricts the types of unit instances that are considered; it may a
   symbol or a list.  UNIT-TYPES may also be T, in which case FUNCTION
   will be applied to all the instances regardless of their type.
   MAP-SPACE returns nil --- FUNCTION is applied for side effect only."

  (unless (eq unit-types t)
    (setf unit-types (expand-unit-type-list unit-types)))

  (let ((start-runtime (gbb-runtime)))
    (dolist-or-atom (path paths)
      (error-unless (path-structure-p path)
         "The PATHS argument to MAP-SPACE, ~s,~@
          was not a path structure or a list of path structures."
         paths)
      (map-space-instance
        function
        unit-types
        (get-space-instance-from-path-structure path)))
    (update-runtime *map-space-meter* start-runtime)
    (update-count *map-space-count*)))


;; See also find-units-all (below).  These two functions are very similar
;; --- but different enough to require them to be separate functions.


(defun map-space-instance (function unit-types space-instance
                           &optional no-subtypes)

  "MAP-SPACE-INSTANCE function unit-types space-instance

   Apply FUNCTION to the units on a space-instance."

  ;; Don't do anything to t.  Saves a MEMBER search below.
  (cond ((eq unit-types t) nil)
        (no-subtypes
         (setf unit-types (assure-list unit-types)))
        (t (setf unit-types (expand-unit-type-list unit-types))))

  (let* ((space-instance-data (space-instance.data space-instance))
	 unit-mapping data array region)
    ;; Loop though the data alist
    (dolist (storage-entry space-instance-data)
      (setf unit-mapping (storage-entry.unit-mapping storage-entry)
	    data (storage-entry.data storage-entry))
      (cond (;; Skip if this unit-mapping doesn't store these units.
	     (not (or (eq unit-types t)
		      (overlapp unit-types (unit-mapping.units unit-mapping))))
	     nil)
	    ((unstructured-unit-mapping-p unit-mapping)
	     ;; There should be no duplicates in this list so there is
	     ;; no need to mark the units.
	     (dolist (unit data)
	       (when (or (eq unit-types t)
			 (member (unit-type-of unit) unit-types :test #'eq))
		 (funcall function unit))))
	    (t
	     (setf array (smallest-space-data-array data)
		   region (entire-array-region array))
	     ;; Clear the mark for all the units.
	     (mark-all-units array 0 region)
	     ;; Apply the function to each unit.
	     (do-array-region (subscripts region)
	       (dolist (unit (apply #'aref array subscripts))
		 (when (and (zerop (get-unit-mark unit))
			    (or (eq unit-types t)
				(member (unit-type-of unit) unit-types :test #'eq)))
		   (funcall function unit)
		   (set-unit-mark unit 1)))))))))

;;; ----------------------------------------------------------------------

;; See also map-space (above).  These two functions are very similar
;; --- but different enough to require them to be separate functions.

(defun find-units-all (unit-types space-instance options)

  "FIND-UNITS-ALL unit-types space-instance options

   Map through all the units on a space applying the filters and
   accumulating the result."

  (macrolet ((looking-for (unit)
               `(or (eq unit-types t)
		    (member (unit-type-of ,unit) unit-types :test #'eq))))
    (let* ((start-runtime (gbb-runtime))
           (space-instance-data (space-instance.data space-instance))
	   (result nil)
	   (filter-before (find-options.filter-before options))
	   (filter-after (find-options.filter-after options))
	   unit-mapping data array region)

      ;; Loop though the data alist
      (dolist (storage-entry space-instance-data)
	(setf unit-mapping (storage-entry.unit-mapping storage-entry)
	      data (storage-entry.data storage-entry))
	(cond (;; Skip if this unit-mapping doesn't store these units.
	       (not (or (eq unit-types t)
			(overlapp unit-types (unit-mapping.units unit-mapping))))
	       nil)
	      ((unstructured-unit-mapping-p unit-mapping)
	       ;; There should be no duplicates in this list so no need
	       ;; to mark them as unseen first.
	       (dolist (unit data)
		 (when (and (looking-for unit)
			    (do-filters filter-before unit)
			    (do-filters filter-after unit))
		   (push unit result))))
	      (t
	       (setf array (smallest-space-data-array data)
		     region (entire-array-region array))
	       ;; Clear the mark for all the units.
	       (mark-all-units array 0 region)
	       ;; Apply the filters to each unit.
	       (do-array-region (subscripts region)
		 (dolist (unit (apply #'aref array subscripts))
		   (when (and (zerop (get-unit-mark unit))
			      (looking-for unit)
			      (do-filters filter-before unit)
			      (do-filters filter-after unit))
		     (push unit result)
		     (set-unit-mark unit 1)))))))
      (update-runtime *find-units-all-meter* start-runtime)
      (update-count *find-units-all-count*)
      result)))

;;; ----------------------------------------------------------------------

(defun smallest-space-data-array (storage-alist)
  "Return the smallest array from the storage-alist (an alist of
   indexes and arrays from the space-instance-data alist)."
  ;; The storage alist is ordered so that the smallest
  ;; (coarsest grained) array is first.
  (storage-element.array (first storage-alist)))

(defun largest-space-data-array (storage-alist)
  "Return the largest array from the storage-alist (an alist of
   indexes and arrays from the space-instance-data alist)."
  ;; The storage alist is ordered so that the largeset
  ;; (finest grained) array is last.
  (storage-element.array (first (last storage-alist))))


;;;; --------------------------------------------------------------------------
;;;;   Find-Units
;;;; --------------------------------------------------------------------------

(defun find-units (types paths pattern
                   &key filter-before filter-after)
  
  "FIND-UNITS types paths pattern &key filter-before filter-after

   Search the space specified by `Paths' for units which match `Pattern'.
   `Types' can be either symbol or a list specifying what types of units to
   consider.  `Paths' can be a path-structure or a list of path-structures.

   `Pattern' is a list of alternating keywords and values.  The keywords and
   their possible values are:

       :PATTERN-OBJECT   another list of options (:INDEX-TYPE,
                         :INDEX-OBJECT, :SELECT, :SUBSEQ, :DELTA,
                         :DISPLACE)

       :ELEMENT-MATCH    :EXACT, :OVERLAPS,
                         :INCLUDES (unit must include the pattern),
                         :WITHIN (unit must be within the pattern)

       :MATCH            a number indicated as (:PERCENTAGE ..)
       :MISMATCH         (:COUNT ..) or (:ALL-BUT ..)
       :SKIPPED

       :BEFORE-EXTRAS    a range (e.g., (0 5), a single number
       :AFTER-EXTRAS     or :dont-care

       :CONTIGUOUS       t or nil
   
   An example call is:

     (find-units '(ghyp hyp)
                  (make-paths :unit-instances goal-unit)
                 '(:pattern-object
                     (:index-type       time-location-list
                      :index-object     (#<TIME-LOCATION 3 (4 4)>
                                         #<TIME-LOCATION 4 (3 5)>
                                         #<TIME-LOCATION 5 (2 7)>)
                      :delta            ((x 2) (y 1)))
                   :element-match    :includes
                   :match            (:percentage 75)
                   :mismatch         2
                   :before-extras    (0 5)
                   :after-extras     (0 0)
                   :contiguous       t)
                :filter-before '(doubled-p)
                :filter-after '(vulnerable-p))

   The filters are predicates which take one argument, a unit instance, and
   return either T or NIL.  If the filter returns NIL then the instance is not
   added to the list of accumulated instances; if it returns T, it is.  The
   filters are appiled in order, as if by AND."

  (let ((search-pattern (parse-pattern pattern))
        (options (make-find-options
                   :filter-before filter-before
                   :filter-after  filter-after))
        (unit-types types)
        (result nil)
        (the-paths paths)
        space-instance all-unit-types)

    (cond ((eq unit-types t)
           (setf all-unit-types t))
          (t
           (setf unit-types (if (or (atom unit-types)
                                    (x-unit-type-of unit-types t))
                                (list unit-types)
                                unit-types))
           (setf all-unit-types (expand-unit-type-list unit-types))))
    
    (dolist-or-atom (path the-paths)
      (error-unless (path-structure-p path)
         "The PATHS argument to FIND-UNITS, ~s,~@
           is not a path structure or a list of path structures."
         the-paths)
      (setf space-instance (get-space-instance-from-path-structure path))
      (check-find-reasonability space-instance search-pattern unit-types)
      (setf result
            (npush-list
              (find-units-internal all-unit-types
                                   space-instance
                                   search-pattern
                                   options)
              result)))
    result))



(defun find-units-internal
       (unit-types space-instance search-pattern options)

  (when (eq search-pattern :all)
    (return-from find-units-internal
      (find-units-all unit-types space-instance options)))

  (let* ((start-runtime (gbb-runtime))
         (units-to-return nil)
	 (space (space-instance.space space-instance))
	 unit-mapping)
    (declare (special units-to-return))

    ;; Each space instance can have several unit-mappings associated with it.
    ;; A `storage entry' is a list:
    ;;    (unit-mapping . <alist of (index-symbols . storage array)>).
    (dolist (storage-entry (space-instance.data space-instance))
      
      (setf unit-mapping (storage-entry.unit-mapping storage-entry))
      (cond
	;; Skip if this unit-mapping doesn't store these units.
	((not (or (eq unit-types t)
		  (overlapp unit-types (unit-mapping.units unit-mapping))))
	 nil)
	;; Unstructured space.
	((unstructured-unit-mapping-p (storage-entry.unit-mapping storage-entry))
	 (find-units-on-unstructured-mapping
	   unit-types space
	   (storage-entry.data storage-entry)
	   search-pattern
	   options))
	;; Normal, structured space.
	(t (find-units-on-structured-mapping
	     unit-types space storage-entry search-pattern options))))

    (update-runtime *find-units-meter* start-runtime space-instance)
    (update-count *find-units-count* space-instance)
    units-to-return))


(defun check-find-reasonability (space-instance search-pattern unit-types)

  "Signal an error if the search pattern is inappropriate for
   these unit-types."

  (cond ((eq search-pattern :all)
	 t)
        ((simple-search-pattern-p search-pattern)
         (check-find-reasonability-1 space-instance search-pattern unit-types))
	((eq (compound-search-pattern.combination search-pattern) :and)
	 (mapc #'(lambda (p)
		   (check-find-reasonability-1 space-instance p unit-types))
	       (compound-search-pattern.patterns search-pattern)))
        (t (error "Bad pattern for FIND-UNITS:~%Can't combine patterns with ~s."
                  (compound-search-pattern.combination search-pattern)))))


(defun check-find-reasonability-1 (space-instance search-pattern unit-types)

  "Signal an error if the simple search pattern, SEARCH-PATTERN, is
   inappropriate for UNIT-TYPES.  In particular, it is an error to:
     1. Use an index in the pattern that is not an index in the
        unit type;
     2. Use an index that is a series in the pattern but a
        scalar in the the unit or vice-versa."

  (let* ((pattern-object (pattern.pattern-object search-pattern))
         (all-indexes (pattern-object.all-indexes pattern-object))
         (c-indexes (pattern-object.c-indexes pattern-object))
         (n-indexes (pattern-object.n-indexes pattern-object))
         (match (pattern.match search-pattern))
         (error-messages nil)
         d-indexes d-index)
    
    (labels

      ((index-names-unit-index? (index unit-index)
         ;; Returns true if INDEX (which is a keyword) is the name of UNIT-INDEX.
         (eq index (unit-index.name unit-index)))

       (save-error (message &rest args)
         (push (apply #'format nil message args) error-messages))

       (all-indexes-present? (unit-type unit-d-indexes)
         ;; Returns true if the unit contains all the indexes in the pattern.
         ;; If the unit is missing some indexes then the error message is saved
         ;; and nil is returned.
         (cond ((subsetp all-indexes unit-d-indexes :test #'index-names-unit-index?)
                t)
               (t
                (let ((difference (set-difference all-indexes unit-d-indexes
                                                  :test #'index-names-unit-index?)))
                  (save-error
                    (if (= (length difference) 1)
                        "~a is not a dimensional index in ~s."
                        "~a are not a dimensional indexes in ~s.")
                    difference unit-type)
                  nil)))))

      (when (and (zerop (pattern-object.length pattern-object))
                 (not (member match '((:count 1) (:all-but 0) (:percentage 100))
                              :test #'equal)))
        (save-error "A scalar pattern has an inappropriate :MATCH option: ~s.~@
                     Valid :MATCH options for scalar patterns are:~@
                     (:COUNT 1), (:ALL-BUT 0), or (:PERCENTAGE 100)."
                    match))

      (dolist (unit-type (if (eq unit-types t)
                             (space.units (space-instance.space space-instance))
                             unit-types))
        (setf d-indexes
              (unit.d-indexes (get-unit-description (x-unit-type-of unit-type))))

        ;; Unit must contain all the indexes in the
        ;; pattern for the following tests.
        (when (all-indexes-present? unit-type d-indexes)
          
          (unless (set-pattern-object-p pattern-object)
            (dolist (index c-indexes)
              (setf d-index (find index d-indexes :key #'unit-index.name :test #'eq))
              (unless (composite-index-structure-p (unit-index.index-structure d-index))
                (save-error
                  "~a is a scalar index in ~s~%but it is a series index in the pattern."
                  index unit-type))))
	     
          (dolist (index n-indexes)
            (setf d-index (find index d-indexes :key #'unit-index.name :test #'eq))
            (unless (or (set-unit-index-p d-index)
                        (scalar-unit-index-p d-index))
              (save-error
                "~a is a series index in ~s~%but it is a scalar index in the pattern."
                index unit-type)))))

      ;; Report any errors
      (error-when error-messages
         "The following errors were found parsing the pattern for FIND-UNITS:~@
          ~{=> ~a~^~%~}"
         error-messages))))


(defun find-units-on-unstructured-mapping
             (unit-types space unit-instances search-pattern options)

  ;; Find units on an unstructured space.

  (declare (special units-to-return))

  (dolist (unit-instance unit-instances)
    (when (or (eq unit-types t)
	      (member (unit-type-of unit-instance) unit-types :test #'eq))
      (when (and
	      (do-filters (find-options.filter-before options) unit-instance)
	      (unstructured-compare-unit-instance-to-search-pattern
		unit-instance search-pattern space)
	      (do-filters (find-options.filter-after options)
			  unit-instance))
	(push unit-instance units-to-return))))

  units-to-return)


(defun unstructured-compare-unit-instance-to-search-pattern (unit-instance search-pattern space)

  ;; Compare a unit instance to the search pattern.
  ;; Used by find-units-on-unstructured-mapping.
  ;; Returns true if it passed.

  (cond ((simple-search-pattern-p search-pattern)
	 (compare-unit-instance-to-pattern-1
	   unit-instance search-pattern space))
	((eq (compound-search-pattern.combination search-pattern) :and)
	 (every #'(lambda (pattern)
		    (compare-unit-instance-to-pattern-1
		      unit-instance pattern space))
		(compound-search-pattern.patterns search-pattern)))
	(t (error "Internal GBB Error: ~
                   Bad pattern given to COMPARE-UNIT-INSTANCE-TO-PATTERN."))))


(defun filter-and-compare-unit-instance
       (unit-instance candidate-mark search-pattern space unit-types options)

  (declare (special units-to-return))

  (macrolet ((reject (instance)
               `(progn (setf (get-unit-mark ,instance) *find-rejected*)
                       (return-from filter-and-compare-unit-instance nil)))
             (skip (instance)
               (declare (ignore instance))
               '(return-from filter-and-compare-unit-instance nil)))

    ;; If the mark is not equal to the previous count then skip this
    ;; instance.  This will be the case if the unit has already been
    ;; rejected, or it's already passed this iteration or it wasn't
    ;; selected by the primary retrieval in some earlier iteration.
    (unless (= candidate-mark (get-unit-mark unit-instance))
      (skip unit-instance))

    ;; Make sure that this is one of the desired units.
    (unless (or (eq unit-types t)
                (member (unit-type-of unit-instance) unit-types :test #'eq))
      (reject unit-instance))

    ;; Do the before filters.  If the any fail then mark this unit
    ;; as rejected because we might come across it again in another
    ;; bucket.
    (unless (do-filters (find-options.filter-before options) unit-instance)
      (reject unit-instance))

    ;; Now, if we get here then this unit is a candidate.
    ;; It is the right type, and it's passed the before filters; so
    ;; compare the unit to the pattern.
    (cond ((simple-search-pattern-p search-pattern)
           (unless (compare-unit-instance-to-pattern-1
                     unit-instance search-pattern space)
             (reject unit-instance)))
          (t ;; Compound Pattern:
           (dolist (pattern (compound-search-pattern.patterns search-pattern))
             (unless (compare-unit-instance-to-pattern-1
                       unit-instance pattern space)
               (reject unit-instance)))))

    ;; Finally, do the after filters.
    (cond ((do-filters (find-options.filter-after options) unit-instance)
           (push unit-instance units-to-return)
           (setf (get-unit-mark unit-instance) *find-succeeded*))
          (t
           (reject unit-instance)))))


(defun compare-unit-instance-to-pattern-1 (unit-instance search-pattern space)

  "Compare a unit instance to a simple pattern.
   Returns true if UNIT-INSTANCE matches SEARCH-PATTERN."

  (let ((description (get-unit-description unit-instance)))
    (and
      (check-noncomposite-indexes
        unit-instance description search-pattern space)
      (cond ((set-pattern-object-p (pattern.pattern-object search-pattern))
             (check-set-pattern
               unit-instance description search-pattern space))
            (t
             (check-composite-indexes
               unit-instance description search-pattern space))))))


(defun check-noncomposite-indexes (unit-instance description search-pattern space)

  (let* ((unit-indexes (unit.d-indexes description))
	 (pattern-object (pattern.pattern-object search-pattern))
	 unit-index)

    (dolists ((pattern-symbol (pattern-object.n-indexes pattern-object))
	      ;; type of this datapoint
	      (pattern-type (pattern-object.n-types pattern-object))
	      ;; value of this datapoint
	      (value (pattern-object.n-data pattern-object)))

      (setf unit-index (find pattern-symbol unit-indexes
			     :key #'unit-index.name :test #'eq))
      (unless (cond ((set-unit-index-p unit-index)
		     (compare-scalar-pattern-index-to-set-unit-index
		       pattern-symbol
		       pattern-type
		       value
		       (nth (unit-index.offset unit-index)
			    (basic-unit.%%indexes%% unit-instance))
		       space
		       (pattern.match-function search-pattern)))
		    (t (compare-index-values
			 pattern-symbol
			 pattern-type
			 value
			 (nth (unit-index.offset unit-index)
			      (basic-unit.%%indexes%% unit-instance))
			 space
			 (pattern.match-function search-pattern))))
	(return-from check-noncomposite-indexes nil)))

    t))


(defun check-composite-indexes (unit-instance description search-pattern space)

  "Returns true if the unit satisfies the composite indexes in the pattern."

  (let ((pattern-object (pattern.pattern-object search-pattern)))

    (when (null (pattern-object.c-indexes pattern-object))
      (return-from check-composite-indexes t))

    (let ((composite-index (get-unit-composite-index
			      description
			      pattern-object)))

      (cond (;; No composite indexes for this unit-instance
	     ;; -- I may want to make this an error.
	     (null composite-index) t)
	    (t
	     (and
	       (check-before-extras
		 unit-instance description composite-index pattern-object
		 (pattern.before-extras search-pattern))
	       (check-after-extras
		 unit-instance description composite-index pattern-object
		 (pattern.after-extras search-pattern))
	       (check-within-index
		 unit-instance description composite-index search-pattern space))))

      )))


(defun check-set-pattern (unit-instance description search-pattern space)

  ;; This function is called when the search pattern is a set.  We
  ;; know that the pattern object is a set.  We also know that a
  ;; corresponding index exists in the unit (because it was
  ;; checked by check-find-reasonability).  The question that
  ;; remains is whether or not the unit-instance's indexes are
  ;; composite indexes (sets or series).

  (and (check-set-pattern-with-composite-unit
         unit-instance description search-pattern space)
       (check-set-pattern-with-scalar-unit
         unit-instance description search-pattern space)))


;;; Functions to compare a set pattern with set or series unit.

(defun check-set-pattern-with-composite-unit
          (unit-instance description search-pattern space)

  (let* ((pattern-object (pattern.pattern-object search-pattern))
         (pattern-c-indexes (pattern-object.c-indexes pattern-object)))

    ;; If the unit and the pattern have no composite indexes in common
    ;; then simply return true.
    (unless (find-if #'(lambda (unit-index)
                         (and (member (unit-index.name unit-index) pattern-c-indexes
                                      :test #'eq)
                              (composite-unit-index-p unit-index)))
                     (unit.d-indexes description))
      (return-from check-set-pattern-with-composite-unit t))

    ;; Else compare the unit and the pattern w.r.t. the composites in each.
    (if (related-indexes-p pattern-c-indexes description)
        (check-set-pattern-with-related-composite-unit
          unit-instance description search-pattern space)
        (check-set-pattern-with-unrelated-composite-unit
          unit-instance description search-pattern space))))



(defun check-set-pattern-with-related-composite-unit
          (unit-instance description search-pattern space)

  (let* ((pattern-object (pattern.pattern-object search-pattern))
	 (element-match-fn (pattern.match-function search-pattern))
	 (n-match (pattern.n-match search-pattern))
	 (n-mismatch (pattern.n-mismatch search-pattern))
	 (matched 0)
	 (mismatched 0)
	 (pattern-symbols (pattern-object.c-indexes pattern-object))
	 (all-unit-indexes (unit.d-indexes description))
	 (unit-index-list
           (remove-if-not
             #'(lambda (unit-index)
                 (and (composite-unit-index-p unit-index)
                      (member (unit-index.name unit-index) pattern-symbols
                              :test #'eq)))
             all-unit-indexes))
	 (index-slots-list
	   (mapcar #'(lambda (unit-index)
		       (nth (unit-index.offset unit-index)
			    (basic-unit.%%indexes%% unit-instance)))
		   unit-index-list))
	 (index-row (make-list (length pattern-symbols))))

    ;; Step through each index element of the pattern...
    (dolist-slices (pattern-row (pattern-object.c-data pattern-object))

      (ncopy-list index-row index-slots-list)
      ;; Step through each index element in the unit...
      (do ()
	  ((endp (first index-row))
	   (incf mismatched))
	;; When a pattern element matches then go on to the next pattern element.
	(when (compare-pattern-row element-match-fn pattern-row pattern-object
				   unit-index-list index-row space)
	  (incf matched)
	  (return t))
	(pop-slices index-row))
      )

    (and (>= matched n-match)
	 (<= mismatched n-mismatch))))


(defun check-set-pattern-with-unrelated-composite-unit
          (unit-instance description search-pattern space)

  (let* ((pattern-object (pattern.pattern-object search-pattern))
	 (element-match-fn (pattern.match-function search-pattern))
	 (n-match (pattern.n-match search-pattern))
	 (unit-indexes (unit.d-indexes description))
         (unit%%indexes%% (basic-unit.%%indexes%% unit-instance))
         unit-index unit-index-data)

    (when (> n-match 0)
      (dolists (;; index for this datapoint
                (pattern-symbol (pattern-object.c-indexes pattern-object))
                ;; type of this datapoint
                (pattern-type (pattern-object.c-types pattern-object))
                ;; value of this datapoint
                (pattern-data (pattern-object.c-data pattern-object)))

        (setf unit-index (find pattern-symbol unit-indexes
                               :key #'unit-index.name :test #'eq))

        (when (composite-unit-index-p unit-index)
          (setf unit-index-data (nth (unit-index.offset unit-index) unit%%indexes%%))
          ;; At least one of the elements of the pattern set must match
          ;; an element from the unit.  If not we return NIL (failure)
          ;; from the function.
          (unless
            (dolist (pattern-value pattern-data nil)
              (when (compare-scalar-pattern-index-to-set-unit-index
                      pattern-symbol
                      pattern-type
                      pattern-value
                      unit-index-data
                      space
                      element-match-fn)
                (return t)))
            (return-from check-set-pattern-with-unrelated-composite-unit nil)))))

    t))


(defun check-set-pattern-with-scalar-unit
	  (unit-instance description search-pattern space)

  (let* ((pattern-object (pattern.pattern-object search-pattern))
         (element-match-fn (pattern.match-function search-pattern))
	 (n-match (pattern.n-match search-pattern))
	 (n-mismatch (pattern.n-mismatch search-pattern))
	 (unit-indexes (unit.d-indexes description))
         (unit%%indexes%% (basic-unit.%%indexes%% unit-instance))
         (matched 0)
         (mismatched 0))

    (labels
      ((compare-row (pattern-row &aux unit-index)
        ;; Compare all indexes for one `row' (index element)
        (block compare-slice
          (dolists
            (;; Index name for this datapoint
             (pattern-symbol (pattern-object.c-indexes pattern-object))
             ;; Type of this datapoint
             (pattern-type (pattern-object.c-types pattern-object))
             ;; Value of this datapoint
             (value pattern-row))

            (setf unit-index (find pattern-symbol unit-indexes
                                   :key #'unit-index.name :test #'eq))

            ;; Ignore composite indexes.  They will be handled elsewhere. 
            (or 
              (composite-unit-index-p unit-index)
              (compare-index-values
                pattern-symbol
                pattern-type
                value
                (nth (unit-index.offset unit-index) unit%%indexes%%)
                space
                element-match-fn)
              (return-from compare-slice nil)))

          ;; If we get here it means that all the indexes pass.
          t)))
     
      (when (> n-match 0)
        (dolist-slices (pattern-row (pattern-object.c-data pattern-object))
          (if (compare-row pattern-row)
              (incf matched)
              (incf mismatched))))

      ;; Return true if satisfies match; false otherwise.
      (and (>= matched n-match) (<= mismatched n-mismatch)))))



(defun find-units-on-structured-mapping
       (unit-types space storage-entry search-pattern options)

  "Find units stored w.r.t. a structured unit mapping.
   The argument STORAGE-ENTRY is a list of the form:
      (unit-mapping . <alist of (index-symbols . storage array)>).
   Each element of the alist is called a `storage-element.'"

  (let* ((unit-mapping (storage-entry.unit-mapping storage-entry))
         (storage-alist (storage-entry.data storage-entry))
         (simple-pattern? (simple-search-pattern-p search-pattern))
         applicable-array-count)

    (labels ((indexes-overlapp (storage-indexes)
               (if simple-pattern?
                   (overlapp storage-indexes (search-pattern-indexes search-pattern))
                   (find storage-indexes
                         (compound-search-pattern.patterns search-pattern)
                         :key #'search-pattern-indexes
                         :test #'overlapp))))

      (setf applicable-array-count
            (count-if #'indexes-overlapp storage-alist :key #'storage-element.indexes))

      (cond
        ((zerop applicable-array-count)
         ;; This case means that there is no overlap between the indexes in the
         ;; pattern and the indexes in the unit-mapping.  (i.e., there has been
         ;; no unit-mapping defined for this combination of space, unit-type and
         ;; index.)
         (let* ((smallest-array (smallest-space-data-array storage-alist))
                (region (entire-array-region smallest-array)))
           (mark-all-units smallest-array 0 region)
           (find-units-in-array unit-types space smallest-array
                                search-pattern options 0 region)))

        (t
         ;; In this case there is some overlap between the pattern indexes and
         ;; the unit mapping indexes.
         (let* ((storage-element (find-if #'indexes-overlapp storage-alist
                                          :key #'storage-element.indexes))
                (storage-indexes (storage-element.indexes storage-element))
                (storage-array (storage-element.array storage-element))
                (pattern-object
                  (pattern.pattern-object
                    (if simple-pattern?
                        search-pattern
                        (find storage-indexes
                              (compound-search-pattern.patterns search-pattern)
                              :key #'search-pattern-indexes
                              :test #'overlapp))))
                (regions (get-regions-from-pattern-object
                           pattern-object storage-indexes space unit-mapping))
                (candidate-count (1- applicable-array-count)))

           ;; Clear the mark for all the units that fall into the correct
           ;; buckets of the first array.
           (dolist (region regions)
             (mark-all-units storage-array 0 region))

           ;; If there is more than one array which has overlapping indexes
           ;; then intersect the arrays.  Intersection doesn't look at the
           ;; first array.  That is saved for the final sweep when the pattern
           ;; is checked.
           (when (> applicable-array-count 1)
             (intersect-storage-arrays
               space #'indexes-overlapp unit-mapping
               storage-alist search-pattern simple-pattern?))

           ;; Finally check the units from the correct buckets of the first
           ;; array against the pattern.
           (dolist (region regions)
             (find-units-in-array unit-types space storage-array search-pattern
                                  options candidate-count region)))))

      ;; The return value isn't used.  The result is accumulated in
      ;; the special variable UNITS-TO-RETURN.
      nil)))


#+OBSOLETE
(defun find-units-on-structured-mapping-1
       (unit-types space unit-mapping storage-element search-pattern
        options untouched-count current-count final-count)
        
  (let* ((storage-indexes (storage-element.indexes storage-element))
         (pattern-object
           (pattern.pattern-object
             (if (simple-search-pattern-p search-pattern)
                 search-pattern
                 (find storage-indexes
                       (compound-search-pattern.patterns search-pattern)
                       :key #'search-pattern-indexes
                       :test #'overlapp)))))

    (cond ((overlapp storage-indexes (pattern-object.n-indexes pattern-object))
	     ;; Only indexes of like kinds (i.e., composite or non-composite)
	     ;; can be grouped together in a unit mapping, if one index is
	     ;; non-composite then they all must be.
	     (find-units-noncomposite
	       unit-types space unit-mapping storage-element
	       pattern-object search-pattern options
               untouched-count current-count final-count))
	    
	    (t
	     ;; If none of the indexes are non-composite then they are all composite.
	     (find-units-composite
	       unit-types space unit-mapping storage-element
	       pattern-object search-pattern options
               untouched-count current-count final-count)))))



#+OBSOLETE
(defun find-units-composite
          (unit-types space unit-mapping storage-element
	   pattern-object search-pattern options
           untouched-count current-count final-count)

  (let* ((storage-indexes (storage-element.indexes storage-element))
	 (space-array (storage-element.array storage-element))
	 (previous-count (1- current-count))
	 (regions (get-regions-from-pattern-object
                    pattern-object storage-indexes space unit-mapping)))

    (dolist (region regions)
      (do-array-region (subscripts region)
	(dolist (unit-instance (apply #'aref space-array subscripts))

	  (filter-and-compare-unit-instance
	    unit-instance untouched-count previous-count final-count
	    search-pattern space unit-types options)

	  ))

      ;; The return value isn't used.
      nil)))


#+OBSOLETE
(defun find-units-noncomposite
          (unit-types space unit-mapping storage-element
	   pattern-object search-pattern options
           untouched-count current-count final-count)

  "Find units via indexes that aren't composite."

  (let* ((array-indexes (storage-element.indexes storage-element))
	 (space-array (storage-element.array storage-element))
	 (previous-count (1- current-count))
	 (region (get-region-from-pattern-row
		   (pattern-object.n-data pattern-object)
		   (pattern-object.n-indexes pattern-object)
		   (pattern-object.n-types pattern-object)
		   space array-indexes unit-mapping)))

    (do-array-region (subscripts region)
      (dolist (unit-instance (apply #'aref space-array subscripts))

	(filter-and-compare-unit-instance
	  unit-instance untouched-count previous-count final-count
	  search-pattern space unit-types options)))

    nil))


(defun find-units-in-array (unit-types space array search-pattern
                            options candidate-mark region)

  "Find units in the case that there is no index that can be used
   to narrow the primary retrieval."

  (do-array-region (subscripts region)
    (dolist (unit-instance (apply #'aref array subscripts))
      (filter-and-compare-unit-instance
        unit-instance candidate-mark search-pattern
        space unit-types options)))
  nil)



;;; Assumptions in the check functions:
;;;   1. The composite index is a :point index.
;;;   2. The elements of the unit indexes are ordered by the composite index.
;;;   3. There are no gaps in the composite index.

(defun check-before-extras (unit unit-description composite-symbol
			    pattern-object before-extras)
  
  "Return true if the unit satisfies the before-extras constraint."
  
  (when (eq before-extras :dont-care)
    (return-from check-before-extras t))
  
  (let* ((pattern-symbols (pattern-object.c-indexes pattern-object))
	 (composite-index-offset
	   (position composite-symbol pattern-symbols :test #'eq))
	 (first-composite-index-value
	   (first (nth composite-index-offset
		       (pattern-object.c-data pattern-object))))
	 (unit-composite-index
	   (find composite-symbol (unit.d-indexes unit-description)
		 :test #'eq :key #'unit-index.name))
	 (unit-composite-values
	   (nth (unit-index.offset unit-composite-index) (basic-unit.%%indexes%% unit)))
	 (position (position first-composite-index-value unit-composite-values
			     :key #'index-element.point-value)))
    (cond ((not (null position))
	   ;; The unit indexes and pattern object overlap.
	   (<= (first before-extras) position (second before-extras)))
	  ((< (index-element.point-value (first unit-composite-values))
	      first-composite-index-value)
	   ;; The first unit index is before the pattern so all unit
	   ;; indexes will be  before the pattern.
	   (<= (first before-extras)
	       (length unit-composite-values)
	       (second before-extras)))
	  (t ;; All the unit indexes are after the pattern.
	   (<= (first before-extras) 0 (second before-extras))))))


(defun check-after-extras (unit unit-description composite-symbol
			   pattern-object after-extras)

  "Return true if the unit satisfies the after-extras constraint."

  (when (eq after-extras :dont-care)
    (return-from check-after-extras t))
  
  (let* ((pattern-symbols (pattern-object.c-indexes pattern-object))
	 (composite-index-offset
	   (position composite-symbol pattern-symbols :test #'eq))
	 (last-composite-index-value
	   (first (last (nth composite-index-offset
			     (pattern-object.c-data pattern-object)))))
	 (unit-composite-index
	   (find composite-symbol (unit.d-indexes unit-description)
		 :test #'eq :key #'unit-index.name))
	 (unit-composite-values
	   (nth (unit-index.offset unit-composite-index) (basic-unit.%%indexes%% unit)))
	 (position (position last-composite-index-value unit-composite-values
			     :key #'index-element.point-value)))
    (cond ((not (null position))
	   ;; The unit indexes and pattern object overlap.
	   (<= (first after-extras)
	       (- (length unit-composite-values) (1+ position))
	       (second after-extras)))
	  ((> (index-element.point-value (first (last unit-composite-values)))
	      last-composite-index-value)
	   ;; The last unit index is after the pattern so all unit
	   ;; indexes will be after the pattern.
	   (<= (first after-extras)
	       (length unit-composite-values)
	       (second after-extras)))
	  (t ;; All the unit indexes are before the pattern.
	   (<= (first after-extras) 0 (second after-extras))))))


(defun check-within-index (unit unit-description
			   composite-symbol search-pattern space)

  "Returns true if the unit satisfies the constraints within the range of
   the pattern."

  (let* ((pattern-object (pattern.pattern-object search-pattern))
	 (element-match-fn (pattern.match-function search-pattern))
	 (pattern-length (pattern-object.length pattern-object))
	 (n-match (pattern.n-match search-pattern))
	 (n-mismatch (pattern.n-mismatch search-pattern))
	 (n-skipped (pattern.n-skipped search-pattern))
	 (contiguous (pattern.contiguous search-pattern))
	 (pattern-symbols (pattern-object.c-indexes pattern-object))
	 (all-unit-indexes (unit.d-indexes unit-description))
	 (unit-index-list
	   (mapcar #'(lambda (name)
		       (find name all-unit-indexes
			     :key #'unit-index.name :test #'eq))
		   pattern-symbols))
	 (index-slots-list
	   (mapcar #'(lambda (unit-index)
		       (nth (unit-index.offset unit-index) (basic-unit.%%indexes%% unit)))
		   unit-index-list))
	 (pattern-composite-index-offset
	   (position composite-symbol pattern-symbols))
	 (pattern-composite-index-list
	   (nth pattern-composite-index-offset (pattern-object.c-data pattern-object)))
	 (first-pattern-composite-index
	   (first pattern-composite-index-list))
	 (unit-composite-index-list
	   (nth (position composite-symbol unit-index-list
			  :test #'eq :key #'unit-index.name)
		index-slots-list))
	 (matched 0)
	 (mismatched 0)
	 (skipped 0)
	 (contiguous-state (initialize-contiguous-state)))

    ;; Move the unit's index pointers forward to beginning of pattern
    (let ((position (position first-pattern-composite-index
			      unit-composite-index-list
			      :key #'index-element.point-value))
	  (first-unit-composite-index
	    (index-element.point-value (first unit-composite-index-list))))
      (cond (;; Pattern and unit overlap
	     (not (null position))
	     (mapl #'(lambda (index-slots)
		       (setf (car index-slots)
			     (nthcdr position (car index-slots))))
		   index-slots-list)
	     (setf unit-composite-index-list
		   (nthcdr position unit-composite-index-list)))
	    (;; Entire unit is before the pattern.
	     (< first-unit-composite-index first-pattern-composite-index)
	     (if (> n-match 0)
		 (return-from check-within-index nil)
		 (setf index-slots-list
		         (nsubstitute-if nil #'identity index-slots-list)
		       unit-composite-index-list nil)))
	    (t ;; Entire unit is after the first element of the pattern.
	     nil)))

    ;; Quick length check
    (when (< (length unit-composite-index-list) n-match)
      (return-from check-within-index nil))

    ;; For each element in the pattern object (a list of ``locations'')
    (do* ((columns (copy-list (pattern-object.c-data pattern-object)))
	  (pattern-row (make-list (length columns)))
	  (length pattern-length (1- length))
	  (pci-list pattern-composite-index-list (cdr pci-list))
	  ;; pattern composite-index value
	  (pci (first pci-list) (first pci-list))
	  ;; unit composite-index value
	  (uci))
	 ((endp pci-list))

      ;; Step pattern-row to the next row.
      (mapl #'(lambda (columns-list row-list)
		;; Set pattern-row to be a single row
		(setf (car row-list) (caar columns-list)))
	    columns pattern-row)
      ;; Pop columns down to the next row
      (pop-slices columns)

      ;; Get the unit composite-index value
      (setf uci (first unit-composite-index-list))
      (cond (;; No unit index for this pattern element (i.e., this unit has
	     ;; no index at this composite index.
	     (or (null uci)
		 (/= pci (index-element.point-value uci)))
	     (incf skipped)
	     (setf contiguous-state
		   (update-contiguous-state contiguous-state :skipped pci)))
	    (t ;; There is a unit index for this pattern element.  Now
	       ;; determine if it is a match or mismatch.
	     (cond ((compare-pattern-row
		      element-match-fn pattern-row pattern-object
		      unit-index-list index-slots-list space)
		    (incf matched)
		    (setf contiguous-state
			  (update-contiguous-state contiguous-state :match pci)))
		   (t (incf mismatched)
		      (setf contiguous-state
			    (update-contiguous-state
			      contiguous-state :mismatch pci))))
	     ;; Move the unit's index pointers forward one.
	     (pop-slices index-slots-list)
	     (setf unit-composite-index-list (cdr unit-composite-index-list))
	     ;; Test for an early exit
	     (when (< (+ matched length) n-match)
	       (return-from check-within-index nil)))))
    (and (>= matched n-match)
	 (<= mismatched n-mismatch)
  	 (<= skipped n-skipped)
	 (or (null contiguous)
	     (contiguous-state-contiguous-p contiguous-state)))))



;;;; --------------------------------------------------------------------------
;;;;   Pattern Parsing
;;;; --------------------------------------------------------------------------


(defun parse-pattern (pattern &aux the-pattern)

  "Parse pattern takes the pattern argument from find-units and returns
   a `search-pattern' containing all the information from the pattern.
   A pattern is either a simple pattern (a simple list of alternating
   keyword/value pairs) or a compound pattern (a list beginning with
   the keyword :AND followed by simple patterns).  A simple pattern is
   represented as a single search-pattern.  A compound pattern is
   represented as a list beginning with the keyword :and followed by
   the constituent search-patterns."

  (when (eq pattern :all)
    (return-from parse-pattern :all))

  (setf the-pattern
	(case (first pattern)
	   (:and (make-compound-search-pattern
		   :and (mapcar #'parse-pattern-1 (rest pattern))))
	   (:or  (error "Combining patterns with :OR is not implemented yet.~@
                         The pattern is ~s."
			pattern))
	   (otherwise (parse-pattern-1 pattern))))

  (when (compound-search-pattern-p the-pattern)
    (error-when (> (compound-search-pattern.length the-pattern) *find-max-counts*)
        "Too many components in the pattern ~s.~@
         The maximum number of components is ~d."
        pattern *find-max-counts*)
    (setf (compound-search-pattern.patterns the-pattern)
          (sort (compound-search-pattern.patterns the-pattern)
                #'(lambda (p1 p2)
                    (if (composite-pattern-object-p p1)
                        (if (composite-pattern-object-p p2)
                            ;; P1 and P2 are both composite:
                            (< (pattern-object.length p1)
                               (pattern-object.length p2))
                            ;; P1 is composite but P2 is not:
                            nil)
                        ;; P1 is not composite.  Is P2 composite?
                        (composite-pattern-object-p p2)))
                :key #'pattern.pattern-object)))

  the-pattern)


(defun parse-pattern-1 (pattern)

  (let (search-pattern the-pattern-object pattern-length)
    (with-keywords-bound (((pattern-object nil)
			   (element-match :exact)
			   (match '(:percentage 100))
			   (mismatch '(:count 0))
			   (skipped '(:percentage 100))
			   (before-extras 0)
			   (after-extras 0)
			   (contiguous t))
			  pattern)

      (unless pattern-object
	(error "There is no :PATTERN-OBJECT specified in the pattern: ~s." 
	       pattern))

      (setf the-pattern-object (build-pattern-object pattern-object))
      (setf search-pattern
	    (make-search-pattern
	      :match-function (get-element-match-function element-match)
	      :before-extras  (fixup-extras before-extras)
	      :after-extras   (fixup-extras after-extras)
	      :match          match
	      :mismatch       mismatch
	      :skipped        skipped
	      :contiguous     contiguous
	      :pattern-object the-pattern-object))
      (setf pattern-length (pattern-object.length the-pattern-object))
      (when (> pattern-length 0)
	(setf (pattern.n-match search-pattern)
	      (convert-match-spec match pattern-length))
	(setf (pattern.n-mismatch search-pattern)
	      (convert-match-spec mismatch pattern-length))
	(setf (pattern.n-skipped search-pattern)
	      (convert-match-spec skipped pattern-length)))
      search-pattern)))


(defun build-pattern-object (pattern-spec)

  "Transform the pattern object into a canonical representation.  The
   argument is a pattern object specification -- a list.   This function
   returns a pattern object -- a structure."

  (cond ((eq (first pattern-spec) :concatenate)
	 (error-unless (cdr pattern-spec)
	    ":CONCATENATE must be given at least one argument.")
	 (concatenate-pattern-objects (cdr pattern-spec)))
	((eq (first pattern-spec) :merge)
	 (error ":MERGE doesn't work yet."))
	(t (build-pattern-object-1 pattern-spec))))


(defun build-pattern-object-1 (pattern-spec)
  "Build a pattern object from a simple (i.e., not compound)
   pattern spec."
  (let ((the-pattern-object nil))
    (with-keywords-bound ((index-type index-object select subseq delta displace)
			  pattern-spec)
      (or
        ;; INDEX-OBJECT is supplied and has a value.
        index-object
        ;; INDEX-OBJECT is explicitly nil.
        (member :index-object pattern-spec :test #'eq)
        ;; No INDEX-OBJECT specified.
        (error "There is no :INDEX-OBJECT specified in ~s." pattern-spec))
      (and select subseq
	   (error "Can't specify both :SELECT and :SUBSEQ in ~s." pattern-spec))
      (setf the-pattern-object
	    (cond ((index-structure-name-p index-type)
		   (make-pattern-from-index-structure
		     (get-index-structure index-type)
		     index-object select subseq delta displace))
		  ((pattern-object-dimension-type-p index-type)
		   (make-dimension-pattern
		     index-type index-object select subseq delta displace))
		  (t (error "Bad INDEX-TYPE, ~s, in the PATTERN-OBJECT ~s."
			    index-type pattern-spec))))
      (setf (pattern-object.length the-pattern-object)
	    (pattern-length the-pattern-object)))
    the-pattern-object))


(defun concatenate-pattern-objects (pattern-specs)
  
  (let ((the-pattern-object nil)
	(next nil))
    (dolist (pattern-spec pattern-specs)
      (setf next (build-pattern-object-1 pattern-spec))
      ;; Must be composite.
      (error-unless (composite-pattern-object-p next)
	 ":CONCATENATE only works for composite indexes structures.~@
          ~s is not a composite."
	 pattern-spec)
      (setf the-pattern-object
	    (if (null the-pattern-object)
		next
		(concatenate-two-pattern-objects
		  the-pattern-object next pattern-specs))))
    the-pattern-object))


(defun concatenate-two-pattern-objects (pattern-object next source)

  "Destructively modify `pattern-object' to include the
   index data from `next'.  Returns the updated pattern-object."

  ;; Both pattern objects must have the same indexes.
  (error-unless (set-equal (pattern-object.c-indexes pattern-object)
			   (pattern-object.c-indexes next))
     "Incompatible pattern objects while concatenating:~{~%    ~s~}.~@
      Some indexes are ~a and some are ~a."
     source
     (pattern-object.c-indexes pattern-object)
     (pattern-object.c-indexes next))
  ;; Composite indexes must match.
  (error-unless (eq (pattern-object.c-index pattern-object)
		    (pattern-object.c-index next))
     "Incompatible pattern objects while concatenating:~{~%    ~s~}.~@
      The composite index changes from ~a to ~a."
     source (pattern-object.c-index pattern-object)
     (pattern-object.c-index next))
  ;; Must be the same sorts of composites
  (error-unless (eq (pattern-object.c-type pattern-object)
		    (pattern-object.c-type next))
     "Incompatible pattern objects while concatenating:~{~%    ~s~}.~@
      The composite type changes from ~s to ~s."
     source (pattern-object.c-type pattern-object)
     (pattern-object.c-type next))

  (let ((next-indexes (pattern-object.c-indexes next))
	(next-data (pattern-object.c-data next))
	(next-types (pattern-object.c-types next))
	(composite-index (pattern-object.c-index next))
	(series? (eq :series (pattern-object.c-type next)))
	offset)
    (dolists ((index (pattern-object.c-indexes pattern-object))
	      (data (pattern-object.c-data pattern-object))
	      (type (pattern-object.c-types pattern-object)))
      (setf offset (position index next-indexes :test #'eq))

      ;; Index element types must be consistent
      (error-unless (eq type (nth offset next-types))
	 "Incompatible pattern objects while concatenating:~{~%    ~s~}.~@
          The element type of ~a changes from ~s to ~s."
	 source index type (nth offset next-types))
      ;; Series composites must be sequential
      (error-when (and series?
		       (eq index composite-index)
		       (not (< (first (last data))
			       (first (nth offset next-data)))))
	 "Incompatible pattern objects while concatenating:~{~%    ~s~}.~@
          The value of the composite index, ~a, is not monotonically increasing."
	 source composite-index)	 

      ;; Note that the variable, `data', is guaranteed to be non-nil
      ;; so that this nconc can work purely by side effect.
      (nconc data (nth offset next-data))))

  (setf (pattern-object.length pattern-object)
	(+ (pattern-object.length pattern-object)
	   (pattern-object.length next)))
  pattern-object)


(defun pattern-object-dimension-type-p (form)
  "Returns true if `Form' is a literal index type.
   E.g., (:dimension 'classification :type :label)."
  (and (listp form)
       (getf form :dimension)
       (index-element-type-p (getf form :type))))

(defun make-dimension-pattern
		    (index-type index-object select subseq delta displace)
  "Make a pattern object in the case that the index-type is a literal
   index specifier."
  (declare (ignore select subseq))
  (let ((index-name (form-keyword (getf index-type :dimension)))
	(index-element-type (getf index-type :type)))
    (multiple-value-bind (datapoint type)
	(compute-pattern-datapoint
	  index-object
	  index-element-type
	  (ecase index-element-type
	     ((:point :label) (list #'identity))
	     (:range          (list #'first #'second)))
	  (second (assoc index-name displace :test #'string=))
	  (second (assoc index-name delta :test #'string=)))
      (make-pattern-object
	:all-indexes (list index-name)
	:n-indexes   (list index-name)
	:n-types     (list type)
	:n-data      (list datapoint)))))


(defun make-pattern-from-index-structure
                      (index-type index-object select subseq delta displace)
  "Make a pattern-object in the case that the index-type is an index-structure."
  (or (typep index-object (index-structure.name index-type))
      (error "While processing the pattern,~@
              ~s was supposed to be a ~s but it's a ~s."
	     index-object (index-structure.name index-type)
	     (type-of index-object)))
  (cond ((composite-index-structure-p index-type)
	 (make-composite-pattern index-type index-object select subseq delta displace))
	(t 
	 (make-noncomposite-pattern index-type index-object delta displace))))


(defun make-composite-pattern (index-type index-object select subseq delta displace)
  (let* ((start 0)
	 (end (length index-object))
	 (composite-type (index-structure.type index-type))
	 (composite-index (index-structure.composite-index index-type))
	 (functions (index-structure.functions index-type))
	 (accessor
	   (first (index-structure-function.functions
		    (get-index-function composite-index index-type))))
	 (pattern-object (make-pattern-object)))
    (setf (pattern-object.c-type pattern-object) composite-type)
    (setf (pattern-object.c-index pattern-object) composite-index)
    (when (and select (eq composite-type :set))
      (error ":SELECT doesn't make sense with ~s because~@
              ~1:*~s is a set type of index structure."
             (index-structure.name index-type)))
    (when select
      (setf start (position (first select) index-object :key accessor)
	    end (1+ (position (second select) index-object :key accessor))))
    (when subseq
      (setf start (first subseq)
	    end (second subseq)))
    (dolist (fn functions)
      (make-composite-pattern-index-element
	pattern-object fn index-object start end delta displace))
    (setf (pattern-object.all-indexes pattern-object)
	  (copy-list (pattern-object.c-indexes pattern-object)))
    pattern-object))


(defun make-composite-pattern-index-element
               (pattern-object index-fn index-object start end delta displace)
  
  (let* ((name (index-structure-function.index-name index-fn))
	 (type (index-structure-function.index-type index-fn))
	 (functions (index-structure-function.functions index-fn))
	 (displace-amount (second (assoc name displace :test #'string=)))
	 (delta-amount (second (assoc name delta :test #'string=)))
	 (data nil))
    ;; One would like to be able to do a map here
    ;; (e.g., (mapcar #'(lambda ...) (subseq list start end)) )
    ;; but subseq always conses a new list.
    (do ((i start (1+ i))
	 (iteml (nthcdr start index-object) (cdr iteml)))
	((>= i end) (setf data (nreverse data)))
      (push (compute-pattern-datapoint
	      (first iteml) type functions displace-amount delta-amount)
	    data))
    (setf (pattern-object.c-indexes pattern-object)
	  (nconc (pattern-object.c-indexes pattern-object) (list name)))
    (setf (pattern-object.c-types pattern-object)
	  (nconc (pattern-object.c-types pattern-object)
		 (list (if (and (eq type :point) delta-amount)
			   :range
			   type))))
    (setf (pattern-object.c-data pattern-object)
	  (nconc (pattern-object.c-data pattern-object) (list data)))
    pattern-object))


(defun make-noncomposite-pattern (index-type index-object delta displace)
  (let* ((functions (index-structure.functions index-type))
	 (pattern-object (make-pattern-object)))
    (dolist (fn functions)
      (make-noncomposite-pattern-index-element
	pattern-object fn index-object delta displace))
    (setf (pattern-object.all-indexes pattern-object)
	  (copy-list (pattern-object.n-indexes pattern-object)))
    pattern-object))


(defun make-noncomposite-pattern-index-element
                    (pattern-object index-fn index-object delta displace)
  
  (let* ((name (index-structure-function.index-name index-fn))
	 (type (index-structure-function.index-type index-fn))
	 (functions (index-structure-function.functions index-fn))
	 (displace-amount (second (assoc name displace :test #'string=)))
	 (delta-amount (second (assoc name delta :test #'string=)))
	 )
    (setf (pattern-object.n-indexes pattern-object)
	  (nconc (pattern-object.n-indexes pattern-object) (list name)))
    (setf (pattern-object.n-types pattern-object)
	  (nconc (pattern-object.n-types pattern-object)
		 (list (if (and (eq type :point) delta-amount)
			   :range
			   type))))
    (setf (pattern-object.n-data pattern-object)
	  (nconc (pattern-object.n-data pattern-object)
		 (list
		   (compute-pattern-datapoint
		     index-object type functions displace-amount delta-amount))))
    pattern-object))


(defun compute-pattern-datapoint (object type functions displace-amount delta-amount)
  "Compute a single datapoint for the pattern object.  Returns two
   values: the datapoint and it's type (which will be an index-element-type)."
  (let ((datapoint nil)
	(datapoint-type nil)
	temp)
    (cond
      ;; Points may be transformed into ranges with the
      ;; delta option in the pattern-spec.
      ((or (eq type :range)
	   (and (eq type :point) delta-amount))
       (cond ((eq type :point)
	      (setf temp (funcall (first functions) object)
		    datapoint (list temp temp)))
	     ((eq type :range)
	      (setf datapoint (list (funcall (first functions) object)
				    (funcall (second functions) object)))))
       (when displace-amount
	 (incf (first datapoint) displace-amount)
	 (incf (second datapoint) displace-amount))
       (when delta-amount
	 (decf (first datapoint) delta-amount)
	 (incf (second datapoint) delta-amount))
       (when (> (first datapoint) (second datapoint))
	 (setf (first datapoint) (- (first datapoint)
				    (floor (- (first datapoint) (second datapoint)) 2))
	       (second datapoint) (first datapoint)))
       (setf datapoint-type :range))
      ;; Point
      ((eq type :point)
       (setf datapoint (funcall (first functions) object))
       (when displace-amount
	 (incf datapoint displace-amount)))
      ;; Label
      ((eq type :label)
       (setf datapoint (funcall (first functions) object))))
    (values datapoint (or datapoint-type type))))


(defun get-regions-from-pattern-object (pattern-object storage-indexes space unit-mapping)

  "Arguments:
     PATTERN-OBJECT   - a pattern-object
     STORAGE-INDEXES  - list of indexes that are used in the particular array
     SPACE            - a space type
     UNIT-MAPPING     - a unit-mapping"

  (let ((regions nil))

    (cond
      ((overlapp storage-indexes (pattern-object.n-indexes pattern-object))
       ;; Only indexes of like kinds (i.e., only composite or only
       ;; non-composite) can be grouped together in a unit mapping, if
       ;; one index is non-composite then they all must be.
       (setf regions
             (list (get-region-from-pattern-row
                     (pattern-object.n-data pattern-object)
                     (pattern-object.n-indexes pattern-object)
                     (pattern-object.n-types pattern-object)
                     space storage-indexes unit-mapping))))

      ((overlapp storage-indexes (pattern-object.c-indexes pattern-object))
       ;; The overlapping indexes are composite.  We have to look at all
       ;; the elements of the composite.
       (dolist-slices (pattern-row (pattern-object.c-data pattern-object))
         (push (get-region-from-pattern-row
                 pattern-row
                 (pattern-object.c-indexes pattern-object)
                 (pattern-object.c-types pattern-object)
                 space storage-indexes unit-mapping)
               regions))
       (setf regions (optimize-regions regions)))

      (t
       ;; Finally, in this case there is no overlap between the indexes
       ;; in the pattern and the indexes for the array.
       (setf regions (list (get-entire-region-for-indexes
                             storage-indexes unit-mapping)))))

    regions))


(defun get-region-from-pattern-row
           (pattern-row pattern-symbols pattern-types
	    space array-index-symbols unit-mapping)

  ;; Returns the subscripts of the region of the space array for
  ;; PATTERN-ROW.  The return value is in this form:
  ;; ((x1 . x2) (y1 . y2) ...)."
  ;;
  ;; The elements in PATTERN-ROW are in the same order as the
  ;; symbols in PATTERN-SYMBOLS.  Any index in the pattern (i.e., symbol in
  ;; PATTERN-SYMBOLS) that is not in ARRAY-INDEX-SYMBOLS is ignored.

  (let* ((space-dimensions (space.dimensions space))
	 (umi-list (unit-mapping.umi-list unit-mapping))
	 (region nil)
	 dimension pattern-offset umi bucket-list)

    (dolist (index-name array-index-symbols)
      (setf dimension (find index-name space-dimensions
			    :key #'dimension.name :test #'string=)
	    umi (find index-name umi-list :key #'umi.name)
	    ;; bucket-list will either contain a hash table or a list
	    ;; of buckets.
	    bucket-list (umi.buckets umi)
	    pattern-offset (position index-name pattern-symbols :test #'string-equal))

      (if (null pattern-offset)
	  ;; This index isn't in the pattern so include the whole dimension.
	  (push (cons 0 (umi.array-size umi)) region)
	  ;; The index is in the pattern so compute the appropriate region.
	  (ecase (umi.type umi)
	    (:subranges
	     (multiple-value-bind (start end)
		 (pattern-datapoint-range (nth pattern-offset pattern-row)
					  (nth pattern-offset pattern-types)
					  (odim.range dimension))
	       (push (bucket-indexes start end bucket-list t) region)))
	    (:groups
	     ;; bucket-list contains a table in this case
	     (let ((index (get-label-index bucket-list
					   (nth pattern-offset pattern-row))))
	       (push (cons index (1+ index)) region))))))

    (nreverse region)))

(defun get-entire-region-for-indexes (storage-indexes unit-mapping)
  "Return a region that selects the entire range for the
   indexes in the argument, STORAGE-INDEXES."
  (let ((umi-list (unit-mapping.umi-list unit-mapping))
        (region nil)
        umi)
    (dolist (index-name storage-indexes)
      (setf umi (find index-name umi-list :key #'umi.name))
      (push (cons 0 (umi.array-size umi)) region))
    (nreverse region)))


;;; --------------------------------------------------------------------------
;;;   Array Region Hacking
;;; --------------------------------------------------------------------------

(defun optimize-regions (regions)

  "OPTIMIZE-REGIONS regions

   Scans REGIONS, which is a list of array regions acceptable to
   DO-ARRAY-REGION, to eliminate any overlap among the element
   regions.  Returns the new list of regions.  The argument, REGIONS,
   may be modified."

  (labels
    ((adjust-regions (region1 region2 following-region-list)
       ;; Compare REGION2 with REGION1, possibly deleting or splitting
       ;; REGION2.  Returns true if any changes were made; nil otherwise.
       (ecase (compare-regions region1 region2)
         ;; No action
         (:disjoint nil)
         ;; Delete region2 from the list
         (:enclose
          (setf (cdr following-region-list) (cddr following-region-list))
          t)
         ;; Split region2 into smaller regions
         (:overlap
          (setf (cdr following-region-list)
                (sort (nconc (split-region region1 region2)
                             (cddr following-region-list))
                      #'>
                      :key #'region-size))
          t))))

    ;; 1. Sort by decreasing size of region
    (setf regions (sort regions #'> :key #'region-size))

    ;; 2. Scan the regions adjusting each region to eliminate any
    ;;    overlap.
    (do* ((region-list-1 regions             (cdr region-list-1))
          (region1       (car region-list-1) (car region-list-1)))
         ((null region-list-1))

      ;; Compare REGION1 to all the remaining regions.
      (do* ((following-list region-list-1)
            (region-list-2 (cdr following-list) (cdr following-list))
            (region2 (car region-list-2) (car region-list-2)))
           ((null region-list-2))

        ;; Adjust REGION2 with respect to REGION1.  If no adjustments were
        ;; made then go on to the next region.  If an adjustment was made
        ;; then don't bump the pointer, FOLLOWING-LIST, because the second
        ;; element of FOLLOWING-LIST may no longer be REGION2.  Even if it
        ;; is the value may have been changed by ADJUST-REGION.
        (unless (adjust-regions region1 region2 following-list)
          (setf following-list (cdr following-list)))))

    ;; 3. Return the new region list
    regions))


(defun compare-regions (r1 r2)

  "COMPARE-REGIONS r1 r2

   Compare two regions.  Returns one of three symbols:
     :ENCLOSE    if R1 encloses R2
     :OVERLAP    if the two regions overlap
     :DISJOINT   if the two regions don't overlap at all.

   =>> Important: R1 must be larger than R2 <<="
  
  (let ((encloses t)
        (overlaps t)
        r1-lower r1-upper r2-lower r2-upper)
    (dolists ((r1-dim r1)
              (r2-dim r2))
      (region-bounds-setf r1-lower r1-upper r1-dim)
      (region-bounds-setf r2-lower r2-upper r2-dim)
      (unless (and (<= r1-lower r2-lower) (< r2-lower r1-upper)
                   (< r1-lower r2-upper) (<= r2-upper r1-upper))
        (setf encloses nil))
      (when (or (<= r1-upper r2-lower) (>= r1-lower r2-upper))
        (setf overlaps nil)))
    (cond (encloses :enclose)
          (overlaps :overlap)
          (t :disjoint))))


(defun split-region (r1 r2)

  "SPLIT-REGION r1 r2

   Splits R2 into two regions such that one of them will no longer
   overlap with R1.  R1 and R2 must overlap.  The result is a list
   of two regions.  R2 is modified and used in the returned list.
   For example:
     (split-region '((4 . 8) (2 . 7)) '((6 . 10) (5 . 8)))
     ==>   (((6 . 8) (5 . 8))
            ((8 . 10) (5 . 8)))"
  
  (do ((r1-dims r1 (cdr r1-dims))
       (r2-dims r2 (cdr r2-dims))
       (n 0 (1+ n))
       (split (copy-tree r2))
       r1-dim r2-dim r1-lower r1-upper r2-lower r2-upper lower upper)
      ((null r1-dims)
       (error
         "The first region completely encloses the second region:~%~5t~s~%~5t~s"
         r1 r2))
    (setf r1-dim (car r1-dims)
          r2-dim (car r2-dims)
          r1-lower (car r1-dim)
          r1-upper (cdr r1-dim)
          r2-lower (car r2-dim)
          r2-upper (cdr r2-dim)
          lower (and (<= r1-lower r2-lower) (< r2-lower r1-upper))
          upper (and (< r1-lower r2-upper) (<= r2-upper r1-upper)))
    (cond ((and lower upper)
           ;; No action
           )
          (lower
           ;; The upper bound of this dimension for R2 is outside R1's
           ;; upper bound.
           (setf (car (nth n split)) r1-upper)
           (setf (cdr r2-dim) r1-upper)
           (return-from split-region (list r2 split)))
          (upper
           ;; The lower bound of this dimension for R2 is outside R1's
           ;; lower bound.
           (setf (cdr (nth n split)) r1-lower)
           (setf (car r2-dim) r1-lower)
           (return-from split-region (list r2 split)))
          (t (error "These two regions are disjoint:~%~5t~s~%~5t~s"
                    r1 r2)))))

(defun region-size (region)
  "Return the total size of an array region."
  (let ((size 1))
    (dolist (dim region)
      ;; Remember n == (n) == (n . n+1)
      (when (and (consp dim) (cdr dim))
        (setf size (* size (- (rest dim) (first dim))))))
    size))


;;; --------------------------------------------------------------------------
;;;   Array Intersection
;;; --------------------------------------------------------------------------

(defun intersect-storage-arrays
       (space overlap-predicate unit-mapping
        storage-alist search-pattern simple-pattern?)
  "Do the intersection of several storage arrays."
  
  (flet ((mark-units-in-pattern (storage-element search-pattern current-count)
           (let* ((storage-indexes  (storage-element.indexes storage-element))
                  (storage-array (storage-element.array storage-element))
                  (sp (if simple-pattern?
                          search-pattern
                          (find storage-indexes
                                (compound-search-pattern.patterns search-pattern)
                                :key #'search-pattern-indexes
                                :test #'overlapp)))
                  (pattern-object  (pattern.pattern-object sp))
                  (previous-count (1- current-count))
                  (regions (get-regions-from-pattern-object
                             pattern-object storage-indexes space unit-mapping)))
             (dolist (region regions)
               (do-array-region (subscripts region)
                 (dolist (unit (apply #'aref storage-array subscripts))
                   (when (= (get-unit-mark unit) previous-count)
                     (setf (get-unit-mark unit) current-count))))))))

    (let ((storage-entry-list
            ;; Skip over the first entry because that is the one that
            ;; will be used in the secondary retreival (when we actually
            ;; look at the unit to see if it satisfies the pattern).
            (cdr (member-if overlap-predicate storage-alist
                            :key #'storage-element.indexes))))

      (do ((storage-entries (member-if overlap-predicate storage-entry-list
                                       :key #'storage-element.indexes)
                            (member-if overlap-predicate (cdr storage-entries)
                                       :key #'storage-element.indexes))
           (current-count 1 (1+ current-count)))

          ((null storage-entries)
           current-count)
        ;; Now:
        ;;   1. Get a search pattern.
        ;;   2. Mark the units that may satisfy the pattern
        ;;       2a. Compute the region based on the pattern.
        ;;       2b. Mark the units that fall into the region.
        (mark-units-in-pattern (car storage-entries)
                               search-pattern
                               current-count)))))


;;; --------------------------------------------------------------------------
;;;   Contigous State
;;; --------------------------------------------------------------------------
;;; To keep track of whether the matching index elements are contigous or not
;;; I define an abstract data type that implements the FSA below.  I don't
;;; check that the values of successive elements are actually contiguous.
;;; A sequence is contiguous unless it fails.  Specifically, is if there are
;;; no matches it is still considered contiguous.
;;;
;;;           |  :MATCH     |  :MISMATCH, :SKIPPED
;;;   --------|-------------|---------------------
;;;   OUT     |  IN         |  OUT
;;;   IN      |  IN         |  WAS-IN
;;;   WAS-IN  |  FAIL       |  WAS-IN
;;;   FAIL    |  FAIL       |  FAIL


(defun initialize-contiguous-state ()
  :out)

(defun update-contiguous-state (state input value)
  (declare (ignore value))
  (case state
     (:out     (if (eq input :match) :in :out))
     (:in      (if (eq input :match) :in :was-in))
     (:was-in  (if (eq input :match) :fail :was-in))
     (:fail    :fail)))

(defun contiguous-state-contiguous-p (state)
  (not (eq state :fail)))
 


(defun compare-pattern-row (element-match-fn pattern-row pattern-object
			    unit-index-list index-slots-list space)

  "Compare a pattern row (from check-within-index) to the corresponding
   elements from a unit-instance."

  ;; INDEX-SLOTS is a list of pointers to the lists of unit index
  ;; values for each index.

  (let ((unit-index nil))
    
    (dolists ((pattern-symbol (pattern-object.c-indexes pattern-object))
	      (pattern-type (pattern-object.c-types pattern-object))
	      (pattern-value pattern-row))
      
      (setf unit-index (find pattern-symbol unit-index-list
			     :key #'unit-index.name :test #'eq))
      (and unit-index
	   (not (compare-index-values
		  pattern-symbol
		  pattern-type
		  pattern-value
		  (first (nth (position unit-index unit-index-list :test #'eq)
			      index-slots-list))
		  space
		  element-match-fn))
	   (return-from compare-pattern-row nil)))

    t))


(defun compare-scalar-pattern-index-to-set-unit-index
         (pattern-symbol pattern-type pattern-value
	  unit-values space element-match-fn)

  (dolist (unit-value unit-values nil)
    (when (compare-index-values
	    pattern-symbol
	    pattern-type
	    pattern-value
	    unit-value
	    space
	    element-match-fn)
      (return t))))


(defun compare-index-values
            (index pattern-type pattern-value unit-value space match-function)

  "Arguments:
     INDEX          - a keyword (e.g., :time).
     PATTERN-TYPE   - index-element-type of the pattern (e.g., :point).
     PATTERN-VALUE  - the value of the index from the pattern.
     UNIT-VALUE     - the value of the index for the unit-instance from.
                      the %%indexes%% slot (a list like (:range nil 2 6)).
     SPACE          - the space -- used for comparing labels.
     MATCH-FUNCTION - the function to use to compare pattern with unit."

  (let (;; Type of the value from the unit
	(unit-type (index-element.element-type unit-value))
	(the-unit-value (index-element.value unit-value)))
    ;; Hack around non-optimal representation of unit index values
    #+IGNORE
    (when (member unit-type '(:point :label))
      (setf the-unit-value (first the-unit-value)))
    (funcall match-function
	     index unit-type the-unit-value pattern-type pattern-value space)))



(defun convert-match-spec (spec length)
  "Convert (:percentage x), (:all-but x), (:count x) to an integer."
  (error-unless (and (two-element-list-p spec)
		     (symbolp (first spec))
		     (numberp (second spec)))
     "Bad format for a :MATCH, :MISMATCH, or :SKIPPED specification.~@
      Something like (:PERCENTAGE 100) was expected.~@
      ~s was seen."
     spec)

  (let ((result (ecase (car spec)
		  (:count (second spec))
		  (:all-but (- length (second spec)))
		  (:percentage (floor (* (second spec) length) 100)))))
    (when (< result 0)
      (error "~s, a :MATCH, :MISMATCH, or :SKIPPED specification,~@
              results in a length less than zero." spec))
    (when (> result length)
      (error "~s, a :MATCH, :MISMATCH, or :SKIPPED specification, results in a number
              greater than the length of the pattern object."
	     spec))
    result))


(defun fixup-extras (extras)
  "Check the extras spec and return a range or :dont-care."
  (cond ((eq extras :dont-care) :dont-care)
	((integerp extras) (list extras extras))
	((and (listp extras)
	      (= (length extras) 1)
	      (integerp (first extras)))
	 (append extras extras))
	((and (two-element-list-p extras)
	      (integerp (first extras))
	      (integerp (second extras)))
	 extras)
	(t (error "Bad before-extras or after-extras specification: ~s."
		  extras))))
	   
(defun mark-all-units (array value
		       &optional (region (entire-array-region array)))

  "MARK-ALL-UNITS array value &optional region

   Set the mark of every unit stored in a region of ARRAY to VALUE.
   REGION is a list of upper and lower bound pairs suitable for
   DO-ARRAY-REGION which specifies what part of the array is
   affected.  If REGION is not supplied it defaults to the entire
   array."

  (do-array-region (subscripts region)
    (dolist (unit (apply #'aref array subscripts))
      (setf (basic-unit.%%mark-1%% unit) value))))


;;;
;;; Match Functions:
;;;

(defun match-overlaps (index unit-type unit-value pattern-type pattern-value space)

  (ecase unit-type
    (:point
     (ecase pattern-type
       (:point (= unit-value pattern-value))
       (:range (and (>= unit-value (pattern-range-start pattern-value))
		    (<= unit-value (pattern-range-end pattern-value))))
       (:label nil)))
    (:range
     (ecase pattern-type
       (:point (and (<= (pattern-range-start unit-value) pattern-value)
		    (>= (pattern-range-end unit-value) pattern-value)))
       (:range (not (or (< (pattern-range-end unit-value)
			   (pattern-range-start pattern-value))
			(> (pattern-range-start unit-value)
			   (pattern-range-end pattern-value)))))
       (:label nil)))
    (:label
     (ecase pattern-type
       (:point nil)
       (:range nil)
       (:label (compare-labels unit-value pattern-value index space))))))


(defun match-within (index unit-type unit-value pattern-type pattern-value space)

  "Unit's indexes are within the pattern's."

  (ecase unit-type
    (:point
     (ecase pattern-type
       (:point (= unit-value pattern-value))
       (:range (and (>= unit-value (pattern-range-start pattern-value))
		    (<= unit-value (pattern-range-end pattern-value))))
       (:label nil)))
    (:range
     (ecase pattern-type
       (:point (and (= (pattern-range-start unit-value) pattern-value)
		    (= (pattern-range-end unit-value) pattern-value)))
       (:range (and (<= (pattern-range-start pattern-value)
			(pattern-range-start unit-value))
		    (>= (pattern-range-end pattern-value)
			(pattern-range-end unit-value))))
       (:label nil)))
    (:label
     (ecase pattern-type
       (:point nil)
       (:range nil)
       (:label (compare-labels unit-value pattern-value index space))))))


(defun match-includes (index unit-type unit-value pattern-type pattern-value space)

  "Unit's indexes include the pattern's."

  (ecase unit-type
    (:point
     (ecase pattern-type
       (:point (= unit-value pattern-value))
       (:range (and (= unit-value (pattern-range-start pattern-value))
		    (= unit-value (pattern-range-end pattern-value))))
       (:label nil)))
    (:range
     (ecase pattern-type
       (:point (and (<= (pattern-range-start unit-value) pattern-value)
		    (>= (pattern-range-end unit-value) pattern-value)))
       (:range (and (<= (pattern-range-start unit-value)
			(pattern-range-start pattern-value))
		    (>= (pattern-range-end unit-value)
			(pattern-range-end pattern-value))))
       (:label nil)))
    (:label
     (ecase pattern-type
       (:point nil)
       (:range nil)
       (:label (compare-labels unit-value pattern-value index space))))))


(defun match-exact (index unit-type unit-value pattern-type pattern-value space)

  (ecase unit-type
    (:point
     (ecase pattern-type
       (:point (= unit-value pattern-value))
       (:range (and (= unit-value (pattern-range-start pattern-value))
		    (= unit-value (pattern-range-end pattern-value))))
       (:label nil)))
    (:range
     (ecase pattern-type
       (:point (and (= (pattern-range-start unit-value) pattern-value)
		    (= (pattern-range-end unit-value) pattern-value)))
       (:range (and (= (pattern-range-start unit-value)
		       (pattern-range-start pattern-value))
		    (= (pattern-range-end unit-value)
		       (pattern-range-end pattern-value))))
       (:label nil)))
    (:label
     (ecase pattern-type
       (:point nil)
       (:range nil)
       (:label (compare-labels unit-value pattern-value index space))))))


(defun get-element-match-function (element-match-keyword)
  (ecase element-match-keyword
    (:overlaps #'match-overlaps)
    (:within #'match-within)
    (:includes #'match-includes)
    (:exact #'match-exact)))


(defun compare-labels (v1 v2 index space)
  (let ((dim (find index (space.dimensions space)
		   :key #'dimension.name :test #'eq)))
    (if dim
	(funcall (edim.test dim) v1 v2)
	(error "Space ~S does not have the dimension ~a."
               (space.name space) index))))



(defun do-filters (filters unit-instance)
  "Applies `Filters' to `Unit-Instance'.
   `Filters' may be a function or a list of functions."
  (cond ((null filters)
	 t)
	((functionp filters)
	 (funcall filters unit-instance))
	(t
	 ;; If it's not a function then it must be a list of functions.
	 (dolist (f filters t)
	   (unless (funcall f unit-instance)
	     (return nil))))))

(defun pattern-datapoint-range (datapoint type range)

  "Return two values corresponding to the limits of the pattern
   datapoint.  Type is an index-element-type."
  (declare (ignore range))

  (ecase type
    (:point (values datapoint datapoint))
    (:range (values (pattern-range-start datapoint)
		    (pattern-range-end datapoint)))))


;;; --------------------------------------------------------------------------
;;;				  End of File
;;; --------------------------------------------------------------------------

