;;;; -*- Mode:LISP; Syntax:COMMON-LISP; Package:USER; Base:10 -*-
;;;; *-* File: VAX11:DIS$DISK:[GBB.V-120.LOCAL]EXAMPLES.LISP *-*
;;;; *-* Edited-By: Cork *-*
;;;; *-* Last-Edit: Thursday, September 15, 1988  12:36:04 *-*
;;;; *-* Machine: Caliban (Explorer II,  Microcode EXP2-UCODE 308 for the Explorer Lisp Microprocessor) *-*
;;;; *-* Software: TI Common Lisp System 4.61 *-*
;;;; *-* Lisp: TI Common Lisp System 4.61 (1.0) *-*

;;;; **************************************************************************
;;;; **************************************************************************
;;;; *
;;;; *                     SIMPLE GBB EXAMPLES
;;;; *
;;;; **************************************************************************
;;;; **************************************************************************
;;;
;;; Written by: Kevin Gallagher
;;;             Department of Computer and Information Science
;;;             University of Massachusetts
;;;             Amherst, Massachusetts 01003.
;;;
;;; This code was written as part of the GBB (Generic Blackboard) system at
;;; the Department of Computer and Information Science (COINS), University of 
;;; Massachusetts, Amherst.
;;;
;;; Copyright (c) 1986, 1987, 1988 COINS.  
;;; All rights reserved.
;;;
;;; Development of this code was partially supported by:
;;;    NSF CER grant DCR-8500332;
;;;    Donations from Texas Instruments, Inc.;
;;;    ONR URI grant N00014-86-K-0764.
;;;
;;; Permission to copy this software, to redistribute it, and to use it for
;;; any purpose is granted, subject to the following restrictions and
;;; understandings.
;;;
;;; 1.  Title and copyright to this software and any material associated
;;; therewith shall at all times remain with COINS.  Any copy made of this
;;; software must include this copyright notice in full.
;;;
;;; 2.  The user acknowledges that the software and associated materials
;;; are provided as a research tool that remains under active development
;;; and is being supplied ``as is'' for the purposes of scientific
;;; collaboration aimed at further development and application of the
;;; software and the exchange of technical data.
;;;
;;; 3.  All software and materials developed as a consequence of the use of
;;; this software shall duly acknowledge such use, in accordance with the
;;; usual standards of acknowledging credit in academic research.
;;;
;;; 4.  Users of this software agree to make their best efforts to inform
;;; the COINS GBB Development Group of noteworthy uses of this software.
;;; The COINS GBB Development Group can be reached at:
;;;
;;;     GBB Development Group
;;;     C/O Dr. Daniel D. Corkill
;;;     Department of Computer and Information Science
;;;     Lederle Graduate Research Center
;;;     University of Massachusetts
;;;     Amherst, Massachusetts 01003
;;;
;;;     (413) 545-0156
;;;
;;; or via electronic mail:
;;;
;;;     GBB@CS.UMass.Edu
;;;
;;; Users are further encouraged to make themselves known to this group so
;;; that new releases, bug fixes, and tutorial information can be
;;; distributed as they become available.
;;;
;;; 5.  COINS makes no representations or warranties of the merchantability
;;; or fitness of this software for any particular purpose; that uses of
;;; the software and associated materials will not infringe any patents,
;;; copyrights, trademarks, or other rights; nor that the operation of this
;;; software will be error-free.  COINS is under no obligation to provide
;;; any services, by way of maintenance, update, or otherwise.  
;;;
;;; 6.  In conjunction with products or services arising from the use of
;;; this material, there shall be no use of the name of the Department of
;;; Computer and Information Science or the University of Massachusetts in
;;; any advertising, promotional, or sales literature without prior written
;;; consent from COINS in each case.
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;;;
;;;  01-08-86 File Created.  (Gallagher)
;;;  07-15-88 Adding some units to multiple spaces (Brolio)
;;;  
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

#+SYMBOLICS
(in-package "CL-USER")
#-SYMBOLICS
(in-package "USER")

(use-package '(lisp umass-extended-lisp gbb))


;;;; --------------------------------------------------------------------------
;;;;   Blackboard and Space Definitions
;;;; --------------------------------------------------------------------------

(format t "~2&;;; Defining SPACES, BLACKBOARDS, INDEX-STRUCTURES...~%")

;;;  This defines a blackboard hierarchy that looks like this:
;;;
;;;           root
;;;           / \
;;;          /   \
;;;         B     A
;;;        / \     \
;;;       /   \     \
;;;     sp1   sp2    B
;;;                 / \
;;;                /   \
;;;              sp1   sp2


(define-spaces (SPACE1 SPACE2)
  "Some spaces."
  :units      (unit1 unit2 unit3 unit5)
  :dimensions ((time :ordered (0 25))
               (x    :ordered (0 30))
               (y    :ordered (0 30))
               (type :enumerated (dog cat mouse lion tiger bear racoon opossum)
                     :test equal)))

(define-blackboards (B) (SPACE1 SPACE2)
  "Basic blackboard structure.")

(define-blackboards (A) (B))

(define-blackboards (ROOT) (A B)
  "Another layer of BB.")


;;;; --------------------------------------------------------------------------
;;;;   Types, Defstructs, and Index Structures
;;;; --------------------------------------------------------------------------


;;; This type is used in lieu of fixnum.

(deftype small-positive-integer ()
  '(integer 0 1000))


;;; Defstructs used by the index-structures below.

(defstruct (time-and-region
             (:print-function time-and-region-printer))
  time
  region)
  
;;; The constructor, boa-make-2d-region, takes its arguments in the form
;;; lower left point and upper right point, rather than the keyword
;;; format.  This is just for convenience.
;;;
;;; Note that the default constructor is needed by save-blackboard.

(defstruct (2d-region (:print-function 2d-region-printer)
                      ;; Make sure that the default constructor is available
                      ;; for save-bb-db.
                      (:constructor make-2d-region)
                      (:constructor boa-make-2d-region
                                    (&optional xmin ymin xmax ymax)))
  xmin  xmax
  ymin  ymax)
  
(defstruct (time-point (:print-function time-point-printer))
  time)
 

;;; Print functions for the defstructs.

(defun print-defstruct-readably (stream struct)
  
  "PRINT-DEFSTRUCT-READABLY stream struct

   Prints simple structures using #S notation. Useful when
   :print-function is supplied to defstruct."
  
  (cond ((defstruct-p (type-of struct))
         (format stream " #S(~s" (type-of struct))
         (dolist (slot (structure-slot-names (type-of struct)))
           (format stream " :~a" slot)
           (print-defstruct-readably stream (get-structure-slot struct slot)))
         (format stream ")"))
        ((listp struct)
           (format stream " (list*")
           (do ((tail struct (cdr tail)))
               ((atom tail)
                (cond ((null tail)
                       (format stream " nil"))
                      (t (print-defstruct-readably stream tail))))
             (print-defstruct-readably stream (first tail)))
           ;; Close the LIST* form
           (format stream ")"))
        (t
         ;; Other atoms must know how to print themselves readably.
         ;; Note that no quote is needed because the #S read macro doesn't
         ;; evaluate the slot values.
         (format stream " ~s" struct))))

(defun time-and-region-printer (object stream depth)
  (declare (ignore depth))
  (format stream "<T~d ~s>"
          (time-and-region-time object) (time-and-region-region object)))

(defun 2d-region-printer (object stream depth)
  (declare (ignore depth))
  (format stream "(~d,~d) (~d,~d)"
          (2d-region-xmin object) (2d-region-ymin object)
          (2d-region-xmax object) (2d-region-ymax object)))

(defun time-point-printer (object stream depth)
  (declare (ignore depth))
  (format stream "T~d" (time-point-time object)))


(defun make-t&r (time xmin ymin xmax ymax)
  (make-time-and-region
    :time time
    :region (boa-make-2d-region xmin ymin xmax ymax)))


;;; The Index Structures.

(define-index-structure TIME-INDEX
  "A single point in time."
  :type time-point
  :indexes ((time :point time-point-time)))

(define-index-structure X-Y-INDEX
  "A region of the x y plane."
  :type 2d-region
  :indexes ((x :range (:min 2d-region-xmin) (:max 2d-region-xmax))
            (y :range (:min 2d-region-ymin) (:max 2d-region-ymax))))

(define-index-structure TIMES-AND-REGIONS
  "A series of regions on the plane at sucessive points in time."
  :composite-type   list
  :composite-index  time
  :element-type     time-and-region
  :indexes ((time :point time-and-region-time)
            (x :range
              (:min 2d-region-xmin time-and-region-region)
              (:max 2d-region-xmax time-and-region-region))
            (y :range
              (:min 2d-region-ymin time-and-region-region)
              (:max 2d-region-ymax time-and-region-region))))

(define-index-structure TIMES-AND-POINTS
  "A series of points on the plane at a successive points in time."
  :composite-type   list
  :composite-index  time
  :element-type     list
  :indexes ((time :point first)
            (x    :point first second)
            (y    :point rest second)))

(define-index-structure POINT-SET
  "A set of points on the plane which are unordered with respect to time."
  :composite-type  list
  :composite-index :none
  :element-type    cons
  :indexes         ((x :point first)
                    (y :point rest)))

(define-index-structure ANIMAL-KINDS
  "A set of Animal Types."
  :composite-type  list
  :composite-index :none
  :element-type    symbol
  :indexes         ((type :label)))

(define-index-structure SPACE-INDEX
  "This index structure provides a space name (SP-NAME) and a
   space count (SP-COUNT)."
  :type    list
  :indexes ((sp-name  :label first)
            (sp-count :point second)))

;;; Dummy initial values to use in the unit definitions below.

(defparameter *initial-time-index*
              (make-time-point :time 0))

(defparameter *initial-x-y-index*
              (boa-make-2d-region 0 0 0 0))

(defparameter *initial-times-and-regions*
              (list (make-time-and-region
                      :time 0
                      :region (boa-make-2d-region 0 0 0 0))))

(defparameter *initial-times-and-points*
              '((0 (0 . 0))))

(defparameter *initial-point-set*
              '((0 . 0)))

(defparameter *initial-animal-kinds*
              '(dog))

;;;; --------------------------------------------------------------------------
;;;;   Simple Events
;;;; --------------------------------------------------------------------------

;;; These are just to illustrate the arguments to the different event
;;; types (and check that the system is working).  Real events would
;;; communicate with your control shell.

(defmacro format-event (name &rest args)
  (flet ((format-string (name args)
           (if (< (length args) 2)
               (format nil "~~&~a:~~@{ ~~s~~}~~%"
                       (string-upcase name))
               (format nil "~~&~a:~~%~{  ~a=~~s~~%~}"
                       (string-upcase name) args))))
    `(format t ,(format-string name args) ,@args)))

(defun print-creation (unit)
  (format-event creation-event unit))

(defun print-add-to-space (unit path-structure)
  (format-event add-to-space-event unit path-structure))

(defun print-deletion (unit)
  (format-event deletion-event unit))

(defun print-delete-from-space (unit path-structure)
  (format-event delete-from-space-event unit path-structure))

(defun print-access (unit slot current-value)
  (format-event access-event unit slot current-value))

(defun print-initialization (unit slot init-value)
  (format-event initialization-event unit slot init-value))

(defun print-slot-update (unit slot new-value old-value)
  (format-event slot-update-event unit slot new-value old-value))

(defun print-link-update (unit link new-value link-added)
  (format-event link-update-event unit link new-value link-added))

(defun print-unlink (unit link new-value deleted-unit)
  (format-event unlink-event unit link new-value deleted-unit))


;;;; --------------------------------------------------------------------------
;;;;   Unit Definitions
;;;; --------------------------------------------------------------------------

(format t "~2&;;; Defining UNITS...~%")

(define-unit (UNIT1 :name-function
                    (:creation-events print-creation)
                    (:add-to-space-events print-add-to-space)
                    (:deletion-events print-deletion)
                    (:delete-from-space-events print-delete-from-space))

  "Unit Number 1"

  :slots
    ((time-region-list *initial-times-and-regions*
                       :read-only t
                       :type times-and-regions
                       :save-print-function print-defstruct-readably)
     (next-goal nil :initialization-events (print-initialization)
                :access-events (print-access)
                :update-events (print-slot-update))
     (animal-kinds '(dog) :type animal-kinds
                   :dynamic t)
     (space-name 'space1)
     (bb-count   0))

  :dimensional-indexes
    ((time time-region-list :type :point)
     (x    time-region-list :type :range)
     (y    time-region-list :type :range)
     (type animal-kinds :type :label))

  :links
    ((link1 (unit2 link1.)
            :access-events (print-access)
            :update-events (print-link-update)
            :unlink-events (print-unlink))
     (link2 :reflexive :initialization-events (print-initialization))
     (link3 :reflexive :singular)
     (link4 :singular (unit2 link4.))
     (link5 :singular (unit2 link5. :singular)
            :access-events (print-access)
            :update-events (print-link-update)
            :unlink-events (print-unlink))
     (link-a. (unit3 link-a)))

  :path-indexes
    ((sp-name  space-name :type :label)
     (bb-count bb-count   :type :point))

  :paths
    ((:path `(root 0 A 0 B ,bb-count ,sp-name 0))
     (:path-structure (make-paths :paths `(root 0 A 1 B ,bb-count ,sp-name 0))))

  ) ;; End of Unit1

;;; -------------------------------------------------------------------

(define-unit (UNIT2 :export
                    :name-function
                    (:slot-initialization-events print-initialization)
                    (:link-initialization-events print-initialization))

   "Unit Number 2"

   :slots
     ((position *initial-x-y-index*
                :type x-y-index
                :save-print-function print-defstruct-readably)
      (time 0 :type small-positive-integer)
      (animal-kind 'dog)
      (space-name 'space1)
      (bb-count   nil))

   :dimensional-indexes
     ((x position)
      (y position)
      (time time :type :point)
      (type animal-kind :type :label))

   :links
     ((link1. (unit1 link1))
      (link4. (unit1 link4 :singular))
      (link5. :singular (unit1 link5 :singular))
      (link-b. (unit4 link-b)))

   :path-indexes
     ((sp-name  space-name :type :label)
      (bb-count bb-count   :type :point))

   :paths ((:path (if bb-count
                      `(root A 0 B ,bb-count ,sp-name 0)
                      `(root     B         0 ,sp-name 0)))))
   
;;; -------------------------------------------------------------------

;;; UNIT3 and UNIT4 are to check the :include option.

(define-unit (UNIT3 (:creation-events print-creation)
                    (:deletion-events print-deletion))
  :slots
    ((slot1 'unit3 :private t
                   :access-events (print-access)
                   :update-events (print-slot-update))
     (slot2 nil :access-events (print-access)
                :update-events (print-slot-update))
     (position *initial-x-y-index*
               :type x-y-index
               :save-print-function print-defstruct-readably)
     (space '(space1 0) :type space-index))
  :dimensional-indexes
    ((x position)
     (y position))
  :links
    ((link-a (unit1 link-a.) :access-events (print-access)
                             :update-events (print-link-update))
     (self-link :reflexive))
  :path-indexes
    ((sp-name  space)
     (sp-count space))
  :paths
    ((:paths `(root 0 a 0 b 0 ,sp-name ,sp-count))))

;;; -------------------------------------------------------------------

(define-unit (UNIT4 (:include unit3)
                    :export
                    (:creation-events print-creation)
                    (:deletion-events print-deletion))
  :slots ((slot1 'unit4) ;; change the default for slot1
          slot3
          (time 0 :type small-positive-integer
                :access-events ((lambda (obj slot value)
                                  (format-event *access obj slot value)))))
  :dimensional-indexes
         ((time time :type :point))
  :links ((link-b (unit2 link-b.)
                  :access-events ((lambda (obj slot value)
                                    (format-event *access obj slot value))))))

;;; -------------------------------------------------------------------

(define-unit (UNIT5 :unnamed
                    (:creation-events print-creation))

  :slots
    ((position *initial-point-set* :type point-set)
     (kinds '(lion) :type animal-kinds))

  :dimensional-indexes
    ((x position)
     (y position)
     (type kinds))

  :paths
    ((:path '(root 0 b 0 space1 0))))


;;;; --------------------------------------------------------------------------
;;;;   Unit Mappings
;;;; --------------------------------------------------------------------------

(format t "~2&;;; Defining the UNIT-MAPPINGS...~%")

;;; The define-unit-mapping forms set up storage on the spaces
;;; as shown in this table:
;;;
;;; UNIT1  ( time  x y  type)   SPACE1
;;; UNIT2  ((time  x y) type)   SPACE1
;;; UNIT3  (      (x y))        SPACE1
;;; UNIT4  (      (x y))        SPACE1   ; inherited from unit3
;;; UNIT5  (      (x y))        SPACE1
;;;
;;; UNIT1  ()                   SPACE2
;;; UNIT2  ()                   SPACE2
;;; UNIT3  (      (x y))        SPACE2
;;; UNIT4  ( time  x y )        SPACE2
;;; UNIT5  (      (x y))        SPACE2


(define-unit-mapping (unit1) (space1)
  "Mapping UNIT1 instances onto SPACE1."
             :indexes (time x y type)
             :index-structure
             ((time :subranges
                    (:start 5)
                    (5 15 (:width 5))
                    (15 :end (:width 2)))
              (x    :subranges (:start :end (:width 5)))
              (y    :subranges (:start :end (:width 10)))
              (type :groups)))

(define-unit-mapping (unit2) (space1)
  "Mapping UNIT2 instances onto SPACE1."
             :indexes ((time x y) type)
             :index-structure
             ((time :subranges (:start :end (:width 5)))
              (x    :subranges (:start :end (:width 5)))
              (y    :subranges (:start :end (:width 5)))
              (type :groups (dog cat) (lion tiger))))

;; An unstructured unit mapping.
(define-unit-mapping (unit1 unit2) (space2))

(define-unit-mapping (unit3) (space1 space2)
             :indexes ((x y))
             :index-structure
             ((x    :subranges (:start :end (:width 5)))
              (y    :subranges (:start :end (:width 5)))))

(define-unit-mapping (unit4) (space2)
             :indexes (time x y)
             :index-structure
             ((time :subranges (:start 10) (10 12) (12 16) (16 20) (20 :end))
              (x    :subranges (:start :end (:width 5)))
              (y    :subranges (:start :end (:width 5)))))

(define-unit-mapping (unit5) (space1 space2)
             :indexes ((x y))
             :index-structure
             ((x    :subranges (:start :end (:width 5)))
              (y    :subranges (:start :end (:width 5)))))


;;;; --------------------------------------------------------------------------
;;;;   Unit and Database Creation
;;;; --------------------------------------------------------------------------

(format t "~2&;;; Instantiating the database...~%")

(instantiate-blackboard-database '(root (a 2 (b 2 (space1 2)))
                                        (b 2 (space1 2)))
                                 :mode :overwrite)

;;; -------------------------------------------------------------------

(format t "~2&;;; Making the units...~%")

;;; Make some units.  First, build a time-region-list.

(defparameter time-region-list-1
  (list
    (make-t&r 1  4 10  6 12)
    (make-t&r 2  5 13  7 15)
    (make-t&r 3  6 16  8 18)
    (make-t&r 4  7 19  9 21)
    (make-t&r 5  8 22 10 24)
    (make-t&r 6  9 25 11 27)
    (make-t&r 7 10 28 12 29)))

(defparameter time-region-list-2
  (list
    (make-t&r 1  7 10  9 12)
    (make-t&r 2  6 13  8 15)
    (make-t&r 3  7 16  9 18)
    (make-t&r 4  8 19 10 21)
    (make-t&r 5 11 22 13 24)))

;;; -------------------------------------------------------------------

;;; Unit1 has a default bb path of BB1, so we only need to specify the space...
;;;
;;; Name the first group (all the units of type unit1) explicitly
;;; so I can check find-units.

;; The units are put on the spaces automatically so I don't really need
;; to save them.  However, it makes it more convenient to check the link
;; macros below.
(defparameter units-one
   (list
     (make-unit1 :name "One"
                 :time-region-list time-region-list-1
                 :animal-kinds '(dog cat)
                 :next-goal 'another-goal)
     (make-unit1 :name "Two"
                 :time-region-list (cdr time-region-list-1)
                 :animal-kinds '(cat lion))
     (make-unit1 :name "Three"
                 :time-region-list (list (second time-region-list-1)
                                         (third time-region-list-1)
                                         (fourth time-region-list-1))
                 :animal-kinds '(dog cat opossum))))

(make-unit1 :name "Four"
            :time-region-list (list (fourth time-region-list-1)
                                    (fifth time-region-list-1))
            :animal-kinds '(racoon))

(make-unit1 :name "Five"
            :time-region-list (list (first time-region-list-1)
                                    (second time-region-list-1)
                                    (third time-region-list-1))
            :animal-kinds '(dog))

(make-unit1 :name "Six"
            :time-region-list time-region-list-1
            :animal-kinds '(bear))


;; These go on space2, which is unstructured.  They have the same index
;; values as the units above.
(make-unit1 :name "Un"
            :space-name 'space2
            :time-region-list time-region-list-1
            :animal-kinds '(dog cat))
(make-unit1 :name "Deus"
            :space-name 'space2
            :time-region-list (cdr time-region-list-1)
            :animal-kinds '(cat lion))
(make-unit1 :name "Trois"
            :space-name 'space2
            :time-region-list (list (second time-region-list-1)
                                    (third time-region-list-1)
                                    (fourth time-region-list-1))
            :animal-kinds '(dog cat opossum))
(make-unit1 :name "Quatre"
            :space-name 'space2
            :time-region-list (list (fourth time-region-list-1)
                                    (fifth time-region-list-1))
            :animal-kinds '(racoon))
(make-unit1 :name "Cinq"
            :space-name 'space2
            :time-region-list (list (first time-region-list-1)
                                    (second time-region-list-1)
                                    (third time-region-list-1))
            :animal-kinds '(dog))

(make-unit1 :name "Seis"
            :space-name 'space2
            :time-region-list time-region-list-1
            :animal-kinds '(bear))


;;;-----------------------------------------------------------------
;;; Add some units to another space
;;; to make sure save-blackboard-database captures multi-space units

(let ((sp1-path (make-paths :paths '(root 0 b 1 space2 0))))
  (dolist (unit units-one)
    (add-unit-to-space unit sp1-path)))

;;; -------------------------------------------------------------------

;; These units have no relation to one another, I just need a list
;; of units later on.
(defparameter units-two
   (list
     (make-unit2 :position (boa-make-2d-region 2 3 4 3)
                 :time 5
                 :animal-kind 'dog)
     (make-unit2 :position (boa-make-2d-region 4 2 8 6)
                 :time 1
                 :animal-kind 'cat)
     (make-unit2 :position (boa-make-2d-region 16 5 24 9)
                 :time 5
                 :animal-kind 'opossum)))

(make-unit2 :position (boa-make-2d-region 6 6 10 6)
            :time 6
            :animal-kind 'racoon)

(make-unit2 :position (boa-make-2d-region 8 5 12 8)
            :time 7
            :animal-kind 'dog)

(make-unit2 :position (boa-make-2d-region 10 10 15 12)
            :time 8
            :animal-kind 'cat)

(make-unit2 :position (boa-make-2d-region 12 9 14 10)
            :time 9
            :animal-kind 'racoon)

;;; -------------------------------------------------------------------

(defparameter units-three
   (list (make-unit3 :position (boa-make-2d-region 8 5 12 8))
         (make-unit3 :position (boa-make-2d-region 18 10 19 12))))

;;; -------------------------------------------------------------------

(defparameter units-four
   (list
     (make-unit4 :position (boa-make-2d-region 8 5 12 8)
                 :time 7)
     (make-unit4 :position (boa-make-2d-region 10 10 14 12)
                 :time 10)))

;;; -------------------------------------------------------------------

(defparameter some-points
              '((1 . 1) (2 . 3) (6 . 3) (7 . 5)))

(make-unit5 :position some-points)
(make-unit5 :position (cdr some-points))
(make-unit5 :position (cddr some-points))
(make-unit5 :position (cdddr some-points))


;;;; --------------------------------------------------------------------------
;;;;   Check Random Things
;;;; --------------------------------------------------------------------------

(format t "~2&;;; Check events...~%")

;;; Check slot events...

;; Update event
(setf (unit1$next-goal (first units-one)) 'new-goal)
;; Access event
(print (unit1$next-goal (first units-one)))

(format t "~2&;;; Check Links...~%")

(linkf (unit1$link5 (first units-one)) (first units-two))
(unlinkf (unit1$link5 (first units-one)) (first units-two))
(linkf (unit1$link5 (first units-one)) (second units-two))
(linkf-list (unit1$link1 (first units-one)) units-two)

(unlinkf (unit1$link5 (first units-one)) (second units-two))
(unlinkf-all (unit1$link1 (first units-one)))

(format t "~2&;;; Check included slots and links...~%")

(linkf (unit4$link-a (first units-four)) (first units-one))
(linkf (unit1$link-a. (second units-one)) (second units-four))
(linkf (unit3$link-a (first units-four)) (second units-one))
(linkf (unit3$self-link (first units-three)) (first units-four))

(setf (unit3$slot2 (first units-four)) '(random slot value))
(print (unit4$slot2 (first units-four)))

;; Check consistency of the unit links.
(check-unit-links)

(format t "~2&;;; Check adding/deleteing units to/from a space...~%")

(let ((path-struct (make-paths :paths '(root a 1 b 1 space1))))
  ;; Add a unit to another space.
  (add-unit-to-space (first units-two) path-struct)
  ;; Check that it's there.
  (catch 'found
    (find-units 'unit2 path-struct :all
                :filter-after #'(lambda (u)
                                  (when (eq u (first units-two))
                                    (format t "~&OK.~%")
                                    (throw 'found t))))
    (format t "~2&*** Unit didn't get added to the space! ***~2%"))
  ;; Delete the unit from the space.
  (delete-unit-from-space (first units-two) path-struct)
  ;; Check that it's gone.
  (catch 'found
    (find-units 'unit2 path-struct :all
                :filter-after
                #'(lambda (u)
                    (when (eq u (first units-two))
                      (format t "~2&*** Unit didn't get deleted from the space! ***~2%")
                      (throw 'found t))))
    (format t "~&OK.~%")))


;;;; --------------------------------------------------------------------------
;;;;   Blackboard Retrieval
;;;; --------------------------------------------------------------------------

(format t "~2&;;; Retrieval: FIND-UNITS and MAP-SPACES...~%")

;;; Print the contents of a space.
(format t "~%;;; Units in (root a b space1):")
(map-space #'print t (make-paths :paths '(root a b space1)))


(format t "~2&;;; Running FIND-UNITS...~%")

(defun f1 (unit)
  "A Trivial Filter"
  (format t "~%F1:  Unit = ~s" unit)
  t)

;;; Check Find...
         

;; The first time this file is loaded this should find
;; (#<UNIT2 "UNIT2-2" xxxx>)

(format t "~&;;; (\"UNIT2-2\")...~% => ~s~2%"
        (find-units 'unit2
           (make-paths :paths '(root b space1))
           `(:element-match :includes
             :pattern-object (:index-type x-y-index
                              :index-object ,(boa-make-2d-region 5 3 8 5)))
           :filter-before 'f1))

;; This should find:
;;     (#<UNIT1 "One" xxxx> #<UNIT1 "Two" xxxx>
;;      #<UNIT1 "Three" xxxx> #<UNIT1 "Six" xxxx>)
;;
;; Notice that "Four" and "Five" get eliminated because they don't
;; have three matching points.


(format t "~&;;; (\"One\" \"Two\" \"Three\" \"Six\")...~% => ~s~2%"
        (find-units
          'unit1
          (make-paths :paths '(root a 0 b 0 space1 0))
          `(:element-match :overlaps
            :pattern-object (:index-type   times-and-regions
                             :index-object ,time-region-list-2)
            :after-extras  :dont-care
            :mismatch      (:count 2)
            :match         (:count 3))
          :filter-before 'f1))

;;; The whole point of :CONCATENATE is to avoid having to do this
;;; kind of stuff.  However, to check that it's working I construct
;;; an object of type TIMES-AND-POINTS and then concatenate it with
;;; an object of type TIMES-AND-REGIONS.
(let* ((time-region (first time-region-list-2))
       (region (time-and-region-region time-region))
       (x-min (2d-region-xmin region))
       (x-max (2d-region-xmax region))
       (x-delta (/ (- x-max x-min) 2))
       (y-min (2d-region-ymin region))
       (y-max (2d-region-ymax region))
       (y-delta (/ (- y-max y-min) 2))
       (tll
         `((,(time-and-region-time time-region)
            (,(+ x-min x-delta) . ,(+ y-min y-delta))))))
  (format t "~&;;; :CONCATENATE two pattern objects:~@
               ;;; (\"One\" \"Two\" \"Three\" \"Six\")...~% => ~s~2%"
     (find-units
       'unit1
       (make-paths :paths '(root a 0 b 0 space1 0))
       `(:element-match :overlaps
         :pattern-object
            (:concatenate
              (:index-type   times-and-points
               :index-object ,tll 
               :delta ((x ,x-delta) (y ,y-delta)))
              (:index-type   times-and-regions
               :index-object ,(cdr time-region-list-2)))
         :after-extras  :dont-care
         :mismatch      (:count 2)
         :match         (:count 3))
       :filter-before 'f1)))

(let ((pattern `(:AND
                  (:element-match   :exact
                   :pattern-object  (:index-type   times-and-regions
                                     :index-object ,time-region-list-1
                                     :select       (2 4))
                   :before-extras   :dont-care
                   :after-extras    :dont-care
                   :match           (:count 3))
                  (:element-match   :exact
                   :pattern-object  (:index-type   animal-kinds
                                     :index-object (dog))
                   :match           (:count 1)
                   :mismatch        (:all-but 1))))
      (saved-animal-kinds (unit1$animal-kinds (first units-one))))

  (format t "~&;;; (\"One\" \"Three\")...~% => ~s~2%"
        (find-units 'unit1
                    (make-paths :paths '(root a 0 b 0 space1 0))
                    pattern))

  (format t "~2&;;; Check dynamic indexes...~%")
  (setf (unit1$animal-kinds (first units-one)) '(racoon mouse))
  (format t "~&;;; (\"Three\")...~% => ~s~2%"
          (find-units 'unit1
                      (make-paths :paths '(root a 0 b 0 space1 0))
                      pattern))
  ;; Restore saved ANIMAL-KINDS slot value.
  (setf (unit1$animal-kinds (first units-one)) saved-animal-kinds))


(format t "~&;;; Find with a `constructed' index-type... ~s~2%"
        (find-units
          'unit2
          (make-paths :paths '(root b 0 space1 0))
          '(:element-match     :exact
            :pattern-object    (:index-type (:dimension type :type :label)
                                :index-object dog))))


(format t "~&;;; Find with a SET pattern.~@
             ;;; Should find three instances of UNIT5... ~s~2%"
        (find-units
          'unit5
          (make-paths :paths '(root b 0 space1 0))
          `(:element-match  :exact
            :pattern-object (:index-type   point-set
                             :index-object ,some-points
                             :subseq       (1 3))
            :match          (:count 1)
            :mismatch       (:count 1))))


(format t "~&;;; 'Save' the blackboard...~%")
(save-blackboard-database *standard-output*
                          (make-paths :paths '(root))
                          :comment "Test of Blackboard Save")

;;; -----------------------------------------------------------------
;;;                             End of File
;;; -----------------------------------------------------------------
