/* File: hetis93.pl
 * Created:  19-Sep-87
 * Last Update:  3-Mar-93
 * Author: Afzal Ballim (C) 1988-1993
 * Purpose: The inheritance reasoner described in my
 *	    doctoral dissertation for stereotype reasoning.
 *	    Based on the HETIS system.
 *
 * Needs: Libraries synread, set, max
 */

:- dynamic node/2.

:- user_library(maxmin),
   user_library(set),
   user_library(synread).

:- loadsyntax('node.syn').

/*===========================================================
 * Predicate: load_hetis_db/1
 * Modes: load_hetis_db(+File)
 * 
 * Purpose: File contains descriptions of nodes that are
 *	    loaded into the system.
 * 
 * Licensed Use: 
 * 
 * Created:  3-Mar-93
 * Last Update: 
 *=========================================================*/
load_hetis_db(File) :-
	load_obj_file(File,node,L),
	asserting(L).

asserting([]).
asserting([[H]|T]) :-
	assertz(H),
	asserting(T).

% NOTE: this only handles heterogeneous with strict and defeasible links.
%       Exception links are not considered in this (yet). 

% Following have arity of two: next-node and conjecture-level
% strict positive membership      == memSp
% strict negative membership      == memSn
% strict class inclusion          == subS
% strict positive class inclusion == subSp
% strict negative class inclusion == subSn
% defeasible inclusion            == subD
% defeasible positive inclusion   == subDp
% defeasible negative inclusion   == subDn

% also
%    source node to start things  == source
%    a bottomed out path          == bottom

%%% SOME PREDICATES FOR DEMONSTRATING THE THINGS YOU CAN DO %%%


% ancestors: returns the set of objects that an object
% may inherit information from.
% NOTE: this works by constantly failing inherit. A faster form
% can be written based on path, which rather than matching the node
% against everything, would follow all branches from it to determine
% what it can inherit from.

ancestors(G,L) :- setof(X,inherit(G,X),L).

% descendants: as ancestors, except finds everything that can inherit from D
%
descendants(D,L) :- setof(X,inherit(X,D),L).


% inherit: is a predicate that determines
% whether there is evidence for one object inheriting information 
% from another object.

inherit(S,G) :- 
    node(S,_),
    node(G,_),                % allows you to produce minimal spanning tree
    paths(S,G,Interp,_,_,_,_,_,_),
    not_all_neg(Interp).      % fail inherit if all paths are negative


% not_all_neg: succeeds if any interpreted path is non-negative
%
not_all_neg([[_,_,subSp(_,_)|_]|_]).

not_all_neg([[_,_,subDp(_,_)|_]|_]).

not_all_neg([[_,_,memSp(_,_)|_]|_]).

not_all_neg([[_,_,source(_,_)|_]|_]).

not_all_neg([_|R]) :- not_all_neg(R).

% "paths" is a predicate with arguments:
%      SOURCE = source node
%      GOAL   = goal node
%      INTERP = list of interpretations of equal precedence
%    PRECEDENCE = the precedence level of the interpretations
%
% Paths expands from the source to find all paths that lead to 
% the goal; it validates the paths and returns the "best" paths.
%
% a path is stored as a LIFO list,i.e., last node first. The first element 
% in any path is its precedence level, each subsequent item is a node with 
% the interpretation and conjecture level to that point.
%
% an interpreted path is alist consisting of an ambiguity level, and then
% the rest of a path as above, e.g.,
%    [ ambiguity(3), 3, subDp(n4), subDp(n7), subS(n9), source(n1) ]
%

paths(SOURCE,GOAL,INTERP,PRECEDENCE,
      Goal_complete,
      Neg_complete,
      Cycle_complete,
      Leaf_complete,
      Bottom_complete) :-
    expand(GOAL,
	   [ [0,source(SOURCE,0)] ],
	   [],[],[],[],[],[],
	   Goal_complete,
	   Neg_complete,
	   Cycle_complete,
	   Leaf_complete,
	   Bottom_complete),!,
    interp(SOURCE,GOAL,Goal_complete,Neg_complete,INTERP,PRECEDENCE).



% interp: takes the source, goal, goal completed paths, neg completed paths,
% and produces an interpretation of source to goal. Prints a report that warns
% of ambiguous/contradictory interpretations, whether the !best! interpretation
% is derived from a path that is itself ambiguous (i.e., a sub-part of it is
% one half of an ambiguity, etc.

% no paths, but silent mode
%
interp(_,_,[],_,_,_) :-
    silent_mode,!,fail.         


% no paths, loud mode so inform then fail
%
interp(S,G,[],_,_,_) :-
    nl,write('There are no valid paths from '),
    write(S),
    write(' to '),
    write(G),!,fail.         

% silent mode
%
interp(_,_,GC,NC,B,P) :-
    silent_mode,!,
    highest_paths(GC,[],100000000,Btmp,P),
    ambig_paths_count(Btmp,NC,Btmp,B).

% loud mode
%
interp(S,G,GC,NC,B,P) :-
    highest_paths(GC,[],100000000,Btmp,P),
    nl,write('Paths from '),write(S),write(' to '),write(G),write(':'),
    nl,write('    Paths have precedence level = '),write(P),nl,nl,
    write_paths(Btmp,NC,Btmp,B).

% ambig_paths_count: count the number of ambiguities in a set of paths
%
% ARGS: paths in, neg_complete, all paths in, paths out

% ambiguous at the goal
%
ambig_paths_count([[Plev|Pth]|Btmp],NC,B_all,[[Plev,ambiguities(Pamb)|Pth]|B]) :-
    Pth = [Phead|_],
    negate_in(Phead,B_all),
    ambig_one_path(Pth,1,NC,Pamb),
    ambig_paths_count(Btmp,NC,B_all,B),!.

%
%
ambig_paths_count([[Plev|Pth]|Btmp],NC,B_all,[[Plev,ambiguities(Pamb)|Pth]|B]) :-
    ambig_one_path(Pth,0,NC,Pamb),
    ambig_paths_count(Btmp,NC,B_all,B),!.

ambig_paths_count([],_,_,[]).

% ambig_one_path: see if there is an ambiguity in a path
%
% ARGS = Path, ambiguity so far, neg_comp, total ambiguity.

% terminating
%
ambig_one_path([source(_,_)],A,_,A).

% ambiguous here
%
ambig_one_path([LL|Lrest],AmbI,NC,AmbO) :-
    negate_in(LL,NC),
    AmbT is AmbI +1,
    ambig_one_path(Lrest,AmbT,NC,AmbO).

% unambiguous to this node.
%
ambig_one_path([_|Lrest],AmbI,NC,AmbO) :-
    ambig_one_path(Lrest,AmbI,NC,AmbO).

%**************************************************

% write_paths: takes a path and writes it out, checking to see if it  
% has an ambiguous section (i.e, a sub-part is ambiguous)

% ambiguous at the goal
%
write_paths([[Plev|Pth]|Btmp],NC,B_all,[[Plev,ambiguities(Pamb)|Pth]|B]) :-
    Pth = [Phead|_],
    negate_in(Phead,B_all),
    write_one_path(Pth,1,NC,Pamb),write(' *ambiguous at the goal*'),
    nl,nl,
    write_paths(Btmp,NC,B_all,B),!.

%
%
write_paths([[Plev|Pth]|Btmp],NC,B_all,[[Plev,ambiguities(Pamb)|Pth]|B]) :-
    write_one_path(Pth,0,NC,Pamb),nl,nl,
    write_paths(Btmp,NC,B_all,B),!.

write_paths([],_,_,[]).

% write_one_path: writes out one path, seeing if there is an ambiguity in it.
%
% ARGS = Path, ambiguity so far, neg_comp, total ambiguity.

% terminating (actually, the first thing printed)
%
write_one_path([source(S,_)],A,_,A) :-
    write(A),write(' ambiguous subpath(s) in: '),nl,
    tab(5),write(S).

% ambiguous here
%
write_one_path([LL|Lrest],AmbI,NC,AmbO) :-
    negate_in(LL,NC),
    AmbT is AmbI +1,
    write_one_path(Lrest,AmbT,NC,AmbO),
    LL =.. [LinkType,Node,Conj],
    write(' --[ '),
    write('L^ '),write(Conj),
    write_link_type(LinkType),
    write(' ]--> '),write(' *ambiguous here* '),write(Node).

% unambiguous to this node.
%
write_one_path([LL|Lrest],AmbI,NC,AmbO) :-
    write_one_path(Lrest,AmbI,NC,AmbO),
    LL =.. [LinkType,Node,Conj],
    write(' --[ '),
    write('L^'),write(Conj),
    write_link_type(LinkType),
    write(' ]--> '),write(Node).

write_link_type(memSp) :- write('+sm').
write_link_type(memSn) :- write('-sm').
write_link_type(subSp) :- write('+si').
write_link_type(subSn) :- write('-si').
write_link_type(subDp) :- write('+di').
write_link_type(subDn) :- write('-di').

% expand: takes the paths with the highest precedence levels (most preferred)
% and expands them, checking to see which are completed in certain ways.
%
% NOTE: the lower the number, the higher the precedence.
%

% Args are:
%         Goal = goal node
%       Source = source node
%       PrecPs = highest precedence paths to be expanded
%    NonPrecPs = lowerprecedence paths, to be considered for later expansion
%        IpGC  = goal completed paths found so far
%        IpNC  = last link negative completed paths found so far
%        IpCC  = cyclic completed paths found so far
%        IpLC  = leaf completed paths found so far
%        IpBC  = bottom completed paths found so far (i.e, uninformative)
%        OpGC  = goal completed paths
%        OpNC  = last link negative completed paths
%        OpCC  = cyclic completed paths
%        OpLC  = leaf completed paths
%        OpBC  = bottom completed paths (i.e, uninformative)

% stop expanding if it is to itself
%
expand(G,[[0,source(G,_)]],_,_,_,_,_,_,[[0,source(G,0)]],[],[],[],[]).

% stop expanding if the highest paths have a lower precedence than the highest
% precedence amongst the goal completed paths
%
expand(_,[[PrecPV|_]|_],_,OpGC,OpNC,OpCC,OpLC,OpBC,OpGC,OpNC,OpCC,OpLC,OpBC) :-
    highest_precedence(OpGC,100000000,PrecGV),
    PrecGV < PrecPV.


% stop expanding when no paths left to expand
%
expand(_,[],[],OpGC,OpNC,OpCC,OpLC,OpBC,OpGC,OpNC,OpCC,OpLC,OpBC).

% otherwise, expand the highest precedence paths, find the subsequent
% highest precedence paths, and continue expanding with them.
% Remove, from consideration, paths that have been invalidated, using negative
% completed paths.
%
expand(Goal,PrecPs,NonPrecPs,IpGC,IpNC,IpCC,IpLC,IpBC,OpGC,OpNC,OpCC,OpLC,OpBC) :-
    expand_level(Goal,PrecPs,GC,NC,CC,LC,BC,Expanded), % expand a level
    union(IpNC,NC,IpNC1),                           % IpNC1 = all neg_com
    remove_invalidated(NonPrecPs,NC,NonVal),        % check old against only new NC
    remove_invalidated(Expanded,IpNC1,ExVal),       % check new against all
    union(NonVal,ExVal,TmpPs),                  % current valid paths
    highest_paths(TmpPs,[],100000000,NewPrecPs,_),
    minus(TmpPs,NewPrecPs,NewNonPrecPs),
    union(IpGC,GC,IpGC1),
    union(IpCC,CC,IpCC1),
    union(IpLC,LC,IpLC1),
    union(IpBC,BC,IpBC1),
    expand(Goal,NewPrecPs,NewNonPrecPs,IpGC1,IpNC1,IpCC1,IpLC1,IpBC1,
	   OpGC,OpNC,OpCC,OpLC,OpBC).

% or no valid paths
%
expand(_,_,_,_,_,_,_,_,[],[],[],[],[]).


% expand_level: take a set of paths, with the same level of precedence
% and expand them out. Check to see which resulting paths are complete
% due to having reached the goal node, last link being a negation (except,
% in the case of e links, there negating has a different effect), path
% cycles back on itself, last node is a leaf (i.e., out degree of zero),
% or the path does not reduce (bottoms out, noninformative).
%
% This is where to start when doing optimisation.
%
% expand_level(Goal,PrecPs,OpGC,OpNC,OpCC,OpLC,OpBC,OpEx)
%


expand_level(Goal,[Path|Rest],OpGC,OpNC,OpCC,OpLC,OpBC,OpEx) :-
    Path = [_,Ln|_], arg(1,Ln,Node),
    node(Node,Olinks),
    expand1(Goal,Path,Olinks,GC1,NC1,CC1,LC1,BC1,Ex1),
    expand_level(Goal,Rest,GCn,NCn,CCn,LCn,BCn,Exn),
    union(GC1,GCn,OpGC),
    union(NC1,NCn,OpNC),
    union(CC1,CCn,OpCC),
    union(LC1,LCn,OpLC),
    union(BC1,BCn,OpBC),
    union(Ex1,Exn,OpEx).



% nothing left to expand
%
expand_level(_,[],[],[],[],[],[],[]).



% expand1: expands one node, which is connected to from the source
% by a path. E.g., say the source is node0, and that there is a path
% to node15 such that the interpretation is subDp(node15,3), i.e, 
% this path from node0 to node15 is interpreted to be an inclusion D
% positive, with conjecture level 3. The predicate takes the path
% and extends it to all of the nodes to which node15 leads.
%
% ARGS:
%     Goal    = the goal node
%     Path    = the path to the current node
%     Olinks  = the links out of the current node
%     GC1     = returned goal completed paths
%     NC1     =    "     negation completed paths
%     CC1     =    "     cyclic    "       "
%     LC1     =    "     leaf        "       "
%     BC1     =    "     bottom      "       "
%     Ex1     = other returned paths
%


expand1(Goal,Path,[Ol1|OlRest],GC1,NC1,CC1,LC1,BC1,Ex1) :-
    Path = [_,LastLink|PathRest],
    reduce(LastLink,Ol1,NewLast),
    precedence_of(NewLast,Prec),
    NewPath = [Prec,NewLast,LastLink|PathRest],
    place(Goal,NewPath,GCp,NCp,CCp,LCp,BCp,Exp),!,
    expand1(Goal,Path,OlRest,GCr,NCr,CCr,LCr,BCr,Exr),
    union(GCp,GCr,GC1),
    union(NCp,NCr,NC1),
    union(CCp,CCr,CC1),
    union(LCp,LCr,LC1),
    union(BCp,BCr,BC1),
    union(Exp,Exr,Ex1).

% nothing to expand1
%
expand1(_,_,[],[],[],[],[],[],[]).

%
% remove_invalidated: takes a list of paths, and removes from it
% paths that have been invalidated by negative completed paths.
% 
% ARGS:   PIn  = initial paths
%         NC   = negative completed paths
%         Pout = resultant paths
%

% first path is invalid
%
remove_invalidated([P|PinR],NC,Pout) :-
    negates(P,NC),
    remove_invalidated(PinR,NC,Pout).


% first path is not invalidated
%
remove_invalidated([P1|PinR],NC,[P1|Pout]) :-
    remove_invalidated(PinR,NC,Pout).


% termination case
%
remove_invalidated([],_,[]).


% negates: if the path handed to it is negated by one in th
% list of paths
% 
% ARGS:  Path, Neg_complete_paths


negates([Plev,L1|_],[[Nlev,L2|_]|_]) :-
    Nlev < Plev,
    arg(1,L1,N),arg(1,L2,N),
    functor(L1,P1,_),
    functor(L2,P2,_),
    ((P1=memSp,P2=memSn);
     (P1=subSp,(P2=subSn;P2=subDn));
     (P1=subDp,(P2=subSn;P2=subDn))).

/*
negates([Plev,memSp(N,_)|_],[[Nlev,memSn(N,_)|_]|_]) :-
    Nlev < Plev.

negates([Plev,subSp(N,_)|_],[[Nlev,subSn(N,_)|_]|_]) :-
    Nlev < Plev.

negates([Plev,subDp(N,_)|_],[[Nlev,subSn(N,_)|_]|_]) :-
    Nlev < Plev.

negates([Plev,subDp(N,_)|_],[[Nlev,subDn(N,_)|_]|_]) :-
    Nlev < Plev.

negates([Plev,subSp(N,_)|_],[[Nlev,subDn(N,_)|_]|_]) :-
    Nlev < Plev.
*/

negates(P,[_|R]):- negates(P,R).

% negate_in: takes a linked-to node, e.g., subSp(n1,5), and
% determines whether it's negation occurs in a list of 
% negative completed paths.
%

negate_in(memSp(N,P),[[_,memSn(N,P)|_]|_]).

negate_in(subDp(N,P),[[_,subDn(N,P)|_]|_]).

negate_in(subSp(N,P),[[_,subSn(N,P)|_]|_]).

negate_in(memSn(N,P),[[_,memSp(N,P)|_]|_]).

negate_in(subDn(N,P),[[_,subDp(N,P)|_]|_]).

negate_in(subSn(N,P),[[_,subSp(N,P)|_]|_]).

negate_in(P,[_|L]) :-
    negate_in(P,L).




% place: decides whether a path is at the goal, is negative complete,
% is leaf complete, is bottom complete, or is an ordinary path to be
% further expanded.
%

% bottom complete
%
place(_,Path,[],[],[],[],[Path],[]) :-
    Path = [_,bottom(_,_)|_].

% goal complete
%
place(Goal,Path,[Path],[],[],[],[],[]) :-
    Path = [_,LL|_],
    arg(1,LL,Goal).

% negative complete
%
place(_,Path,[],[Path],[],[],[],[]) :-
    (Path = [_,subDn(_,_)|_];
     Path = [_,subSn(_,_)|_];
     Path = [_,memSn(_,_)|_]).

% cyclic complete (i.e., path cycles back on itself
%
place(_,Path,[],[],[Path],[],[],[]) :-
    Path = [_,LL|Prest],
    arg(1,LL,Node),
    node_in(Node,Prest).


% leaf complete
%
place(_,Path,[],[],[],[Path],[],[]) :-
    Path = [_,LL|_],
    arg(1,LL,Node),
    node(Node,[]).

% otherwise, the path is just ordinary
%
place(_,Path,[],[],[],[],[],[Path]).


% node_in: takes a node and a path, and determines whether the node lies
% on the path.

% node occurs at the head of the path
%
node_in(Node,[Ln|_]) :-
    arg(1,Ln,Node).

% is node in the rest of the path?
%
node_in(Node,[_|Lrest]) :-
    node_in(Node,Lrest).


% The reduction functions. Exception links are not yet included.
%

% inclusion s+ to inclusion "s"'s
%
reduce(subSp(_,C1),subS(D,C2),subS(D,Cr)) :-
    max(C1,C2,Cr),!.
    
reduce(subSp(_,C1),subSp(D,C2),subSp(D,Cr)) :-
    max(C1,C2,Cr),!.

reduce(subSp(_,C1),subSn(D,C2),subSn(D,Cr)) :-
    max(C1,C2,Cr),!.


% inclusion d+ to inclusion "s"'s
%
reduce(subDp(_,C1),subS(D,C2),subD(D,Cr)) :-
    max(C1,C2,Ctmp), Cr is Ctmp + 1,!.

reduce(subDp(_,C1),subSp(D,C2),subDp(D,Cr)) :-
%    max(C1,C2,Ctmp), Cr is Ctmp + 1,!.
    max(C1,C2,Cr),!.

reduce(subDp(_,C1),subSn(D,C2),subDn(D,Cr)) :-
%    max(C1,C2,Ctmp), Cr is Ctmp + 1,!.
    max(C1,C2,Cr),!.

% inclusion s+ to inclusion "d"'s
% 
reduce(subSp(_,C1),subD(D,C2),subD(D,Cr)) :-
    max(C1,C2,Ctmp), Cr is Ctmp + 1,!.

reduce(subSp(_,C1),subDp(D,C2),subDp(D,Cr)) :-
    max(C1,C2,Ctmp), Cr is Ctmp + 1,!.

reduce(subSp(_,C1),subDn(D,C2),subDn(D,Cr)) :-
    max(C1,C2,Ctmp), Cr is Ctmp + 1,!.



% inclusion D= to inclusion "d"'s
%
reduce(subDp(_,C1),subD(D,C2),subD(D,Cr)) :-
    max(C1,C2,Ctmp), Cr is Ctmp + 1,!.

reduce(subDp(_,C1),subDp(D,C2),subDp(D,Cr)) :-
    max(C1,C2,Ctmp), Cr is Ctmp + 2,!.

reduce(subDp(_,C1),subDn(D,C2),subDn(D,Cr)) :-
    max(C1,C2,Ctmp), Cr is Ctmp + 2,!.


reduce(source(_,_),First,First).




% member s+ to inclusion "s"'s and "d"'s
%
reduce(memSp(_,C1),subSp(D,C2),memSp(D,Cr)) :-
    max(C1,C2,Cr),!.

reduce(memSp(_,C1),subSn(D,C2),memSn(D,Cr)) :-
    max(C1,C2,Cr),!.

reduce(memSp(_,C1),subDp(D,C2),memSp(D,Cr)) :-
    max(C1,C2,Ctmp), Cr is Ctmp + 1,!.

reduce(memSp(_,C1),subDn(D,C2),memSn(D,Cr)) :-
    max(C1,C2,Ctmp), Cr is Ctmp + 1,!.



% and of course, all others
% reduce to bottom
%
reduce(_,P2,bottom(D,0)) :-
    arg(1,P2,D).


% precedence_of is a predicate that returns the precedence level
% of a link-to-node, e.g., subDp(n,12) has precedence level 38.
% Define the PRECEDENCE of a path to be
%
%	 if interp = s+ or s- then  I=1
%	  elseif interp = s or d+ or d- of j+ or j- or j0 then I=2
%	  elseif interp = d then I=3
%	 fi
%	 PRECEDENCE_LEVEL= (3 * conjecture_level)+I
%

precedence_of(memSp(_,C),P) :-
    P is C * 3 + 1.

precedence_of(memSn(_,C),P) :-
    P is C * 3 + 1.

precedence_of(subSp(_,C),P) :-
    P is C * 3 + 1.

precedence_of(subSn(_,C),P) :-
    P is C * 3 + 1.

precedence_of(subS(_,C),P) :-
    P is C * 3 + 2.

precedence_of(subDp(_,C),P) :-
    P is C * 3 + 2.

precedence_of(subDn(_,C),P) :-
    P is C * 3 + 2.

precedence_of(subD(_,C),P) :-
    P is C * 3 + 3.

precedence_of(source(_,_),0).

precedence_of(bottom(_,_),100000000).

% find the paths with highest precedence level
% amongst a set of paths.
% a call should be of the form: highest_paths(Path_in,[],Init_Prec_level,PrecP,PrecV).
% no paths left
%
highest_paths([],MP,C,MP,C).

% next path has precedence of the current high
%
highest_paths([Phead|Ptail],Prec_pathsofar,Current_highest,PrecP,Precv) :-
    [Current_highest|_] = Phead,!,
    highest_paths(Ptail,[Phead|Prec_pathsofar],Current_highest,PrecP,Precv).
    
% next path is new highest
%
highest_paths([Phead|Ptail],_,Current_highest,PrecP,Precv) :-
    [New_highest|_] = Phead,
    New_highest < Current_highest,!,
    highest_paths(Ptail,[Phead],New_highest,PrecP,Precv).

% next path is discarded
%
highest_paths([_|Ptail],Prec_pathsofar,Current_highest,PrecP,Precv) :-
    highest_paths(Ptail,Prec_pathsofar,Current_highest,PrecP,Precv).


% highest_precedence, takes a list of paths, and gives the highest precedence
% value from that list.
% NOTE: it needs an initial precedence value to start (2nd arg)
%       e.g., 100000000
%
highest_precedence([[V|_]|Rest],Precsofar,Prec) :-
    V < Precsofar,
    highest_precedence(Rest,V,Prec).


%
highest_precedence([_|Rest],Precsofar,Prec) :-
    highest_precedence(Rest,Precsofar,Prec).


% no paths left
%
highest_precedence([],Prec,Prec).




%% silent: set do-not-print-out-paths mode
%

silent :-
    printing_mode(on),retractall(printing_mode(on)).

%% loud: set print-out-paths mode
%

loud :-
    silent_mode,asserta(printing_mode(on)).

silent_mode:- printing_mode(on),!,fail.
    
silent_mode.

:- asserta(printing_mode(on)).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SWITCH_POSN: switches the position of two objects in a list, indexed from
%              1
%
% Args: X  = position of first object
%       Y  = position of second object
%       L  = the initial list
%       Lo = the final list

switch_posn(X,Y,L,Lo)
:-
    X > Y,
    switchff(1,Y,X,L,Lo).

switch_posn(X,Y,L,Lo)
:-
   switchff(1,X,Y,L,Lo).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SWITCHFF: scans through a list to a position indicated, then sets up the
%           switching of an item at that point with one further on.
%
% Args: C  = current position
%       X = first position
%       Y  = position of second object
%       L  = the initial list
%       Lo = the final list

switchff(X,X,Y,[First|L],[Second|Lo])
:-
    Nc is X+1,
    switchwith(Nc,Y,L,First,Second,Lo).

switchff(C,X,Y,[H|L],[H|Lo])
:-
    Nc is C +1,
    switchff(Nc,X,Y,L,Lo).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SWITCHWITH: scans through a list to find the second of two positions, then
%             performs unification which completes the switching
% Args: C  = current position
%       Y  = position of second object
%       L  = the initial list
%       F  = the first object
%       S  = the second object
%       Lo = the final list

switchwith(Y,Y,[S|Lo],F,S,[F|Lo]).

switchwith(C,Y,[H|L],F,S,[H|Lo])
:-
    Nc is C+1,
    switchwith(Nc,Y,L,F,S,Lo).

