/* XRF.PL :  Prolog Cross-Reference Program 

						Dave Bowen
						Updated: 1 June 1983
	This is the main module of XREF.

*/

% Compiler declarations %

 :- public go/0.			% User entry.
 :- public xrf/1, indirect/1.		% These are public for internal Call.

 :- op(1050,xfy,->).
 :- op(500,fy,@).

 :- mode caller(+,+,+,+).
 :- mode crecord(+,+).
 :- mode definition(+,+).
 :- mode do_file(+,+).
 :- mode exhaust(+). 
 :- mode getfrom(+).
 :- mode goal(?,-). 
 :- mode goal0(+,-).
 :- mode has_ext(+,+).
 :- mode head(+,+,-,-).  
 :- mode indirect(+).    
 :- mode mark_interpreted(+).
 :- mode process(+,+).   
 :- mode reply(+).  
 :- mode widen(?,-).
 :- mode xrf(+).

/******************************************************************************

Data

   $caller(Called_Predicate, Calling_Functor, Calling_Arity, Where_Defined)
			  Held on key: Called_Predicate.
			  $caller(P,F,N,I) means P is called by F/N in file I.
   $defn(File, Predicate)
			  Held on key: Predicate.
			  $defn(I,P) means P is defined in I.
   $file(File)
			  Held on key: $file(_).
			  $file(F) means F is a file.
   Predicate
			  Held on key: $pred
			  pred(G,M) means G/M was defined OR used.

******************************************************************************/

save_it(X) :-
	['/home/afzal/lib/prolog/Boot'],
	save(X),go.

				/* Top level */
go :-   %'LC',
	repeat,
	    ttynl, display('Next file: '), ttyflush,
	    readtonl(String),
	    reply(String), !.

				/* Check for termination of input: just 
				   <newline> typed? */
reply([]) :- !,
	(collect ; true),halt.	% Yes: go and start output phase
reply(String) :-
	getfrom(String), !,	% Something input: what is it?
	fail.			% Go back to repeat

				/* Prolog, indirect or definition file? */
getfrom([64|S]) :- !,
	do_file(indirect,S).		% Indirect file introduced by '@'
getfrom([42|S]) :- !,
	do_file(load_file,S).		% Definition file introduced by '*'
getfrom(S) :- 
	has_ext(S,".ccl"), !,		% Indirect file indicated by ".CCL"?
	do_file(indirect,S).
getfrom(S) :- 
	has_ext(S,".def"), !,		% Definition file indicated by ".DEF"?
	do_file(load_file,S).
getfrom(S) :-				% Must be Prolog file for cross ref
	do_file(xrf,S).

				/* Takes two character lists as arguments, and
				   tests whether the 1st ends with the 2nd */
has_ext(Ext,Ext) :- !.
has_ext([_|L],Ext) :- has_ext(L,Ext).


				/* Open up the file, call processing procedure,
				   and close file again. Calls indirect(File)
				   load_file(File) or xrf(File). */
do_file(Predicate,String) :-
	name(File,String),		% Convert char list to atom (File)
	seeing(Old_file),		% Save currently open file
	see_chek(File),			% Open required file (may fail)
	P =.. [Predicate,File],		% Construct call to reqd procedure
	call(P),			% Call it
	seen,				% Close file
	see(Old_file).			% Re-open original file
do_file(_,_).				% Always succeed

				/* Get file names from indirect file */
indirect(F) :- 
	readtonl(S),			% Read a line, fail at end_of_file(^Z)
	( (S="")			% Ignore blank lines
	;   display('File: '),		% (S is char list)
	    writes(S), ttynl,		% Echo on terminal
	    getfrom(S)			% Process (may be indirect or def)
	), !,
	indirect(F).			% Loop to get next line
indirect(F) :-
	ttynl,				% Tell user when indirect file finished
	display('Indirect file '),
	display(F),
	display(' processed'), ttynl.

				/* Cross reference processing for a particular
				   file F */
xrf(F) :-
	recordz('$file'(_),'$file'(F),_),	% record F under $file
	exhaust(F).				% go through F term by term

				/* Process each clause, T, in file F */
exhaust(F) :- 
	repeat,				% Iterate till end_of_file
	    read(T),			% Read a clause
	    expand_term(T,T1),		% Pre-translation of grammar rules
	    process(T1,F),		% Cross-ref processing
	T=(end_of_file), !.

				/* Process clause (1st arg). 2nd arg is file */

process((P:-Q),I) :- !, 		% Non-unit clause
	head(P,I,F,N), !,
	goal(Q,G),			% Process successive goals by 
	caller(G,F,N,I).		%   backtracking.
process((:-G),_) :- !, call(G).		% goal clause, call it.
process((?-G),_) :- !, call(G).		% question, ditto
process(end_of_file,_) :- !.		% eof so succeed and exit repeat loop
process(P,I) :- head(P,I,_,_).		% unit clause

				/* Record the fact that P is a predicate & that
				   it is defined in file I. Return principal
				   functor of P (F) & its arity (N). */
head(P,I,F,N) :-
	functor(P,F,N),			% P has name F and arity N
	functor(G,F,N),			% G is most general term F/N
	definition(G,I).		% Record F/N is pred & defined in I

				/* Fail if goal is a variable. */
goal(G,_) :- var(G), !, fail.
goal(G,G1) :- goal0(G,G1).

				/* Returns most general term having the
				   principal functor & arity of each goal in 
				   the clause (successively on backtracking).
				   Ignores system predicates. */
goal0((G,_),G1) :-
   goal(G,G1).
goal0((_,G),G1) :- !,
   goal(G,G1).
goal0((G;_),G1) :-
   goal(G,G1).
goal0((_;G),G1) :- !,
   goal(G,G1).
goal0(G1,G2) :- 
   recorded(G1,'$applies'(G1,P),_),
   widen(P,P1),
   goal(P1,G2),
   mark_interpreted(G2).
goal0(G,_) :- 
   recorded(G,'$system',_), !,
   fail.
goal0(G,G).

				/* Record that P is a predicate and that it is
				   defined in file I */
definition(P,I) :-
	recorded(P,'$system',_), !,	% But not if P is a system predicate
	warn(P,'already defined as a system predicate'), fail.
definition(P,I) :-
	crecord('$pred',P),
	crecord(P,'$defn'(I,P)).

				/* Record that P is a predicate called by
				   F/N in file I */
caller(P,F,N,I) :- 
	functor(P,Pf,Pn),
	functor(P1,Pf,Pn),
	crecord('$pred',P1),
	crecord(P1,'$caller'(P1,F,N,I)).

				/* Record that P is called by the user or
				   outside its file of definition, and hence
				   must be public. */
mark_interpreted(P) :- 
	caller(P,'<user>',0,undefined).

				/* Record term Q on key P unless already
				   recorded. */
crecord(P,Q) :- recorded(P,Q,_), !.
crecord(P,Q) :- recordz(P,Q,_), !.


				/* Increase arity of predicate by specified
				   amount */
widen(A+_,_) :-
	var(A), !, fail.	% NB also covers variable as first arg
widen(A+Offset,A1) :- !,
	functor(A,F,N1),
	N2 is N1+Offset,
	functor(A1,F,N2).
widen(A,A).
/* XRFCOL.PL :  Collecting up module of XREF.

						Dave Bowen
						Updated: 11 March 82
*/

 :- mode callers(+,+,-).   
 :- mode defn_file(+,-).
 :- mode entries(+,+).   
 :- mode exts(+,+,+).    
 :- mode fentries(-). 
 :- mode got_defn(+,-).
 :- mode lt2(+,+,+,+).   
 :- mode lte(+,+).  
 :- mode multiple_defn(+,+,-,-).
 :- mode notin(+,+).
 :- mode partition(+,+,-,-).  
 :- mode qsort(+,-,+).   


/******************************************************************************

Data for collecting up

   $ext(File, Predicate)
			  Held on key: File.
			  $ext(I,P) means P is an external (import) of I

   $entry(File, Predicate)
			  Held on key: File.
			  $entry(I,P) means P is an entry (export) of I

******************************************************************************/


collect :-
   fentries(L),		%  Make list of all predicates with all associated data
   qsort(L,L1,[]),	%  Sort them
   do_output(L1).

				/* Find entries. Search through all encountered
				   predicates */
fentries([e(F,N,f(I,Cs))|L]) :- 
	recorded('$pred',P,Ptr),		% Find a predicate P
	erase(Ptr),
	functor(P,F,N),				% P is F/N
	defn_file(P,I),				% P is defined in file I
	callers(P,I,Cs),			% Cs is list of callers c(F,N)
	((I=undefined,warn(P,'not defined')); true),
	((Cs=[],warn(P,'not called')); true),
	multiple_defn(P,[I],L,L1),		% Any multiple definitions?
	((nonvar(L),warn(P,'multiply defined')); true), !,
	fentries(L1).
fentries([]).

				/* Find I where P is defined, or else
				   set I=undefined.*/
defn_file(P,I) :- 
	got_defn(P,I), !.
defn_file(P,undefined).

				/* Is P defined in a file, or is it known? */
got_defn(P,I) :- 
	recorded(P,'$defn'(I,P),_).
got_defn(P,I) :- 
	recorded(P,'$known'(I),_).

				/* Look for multiple defns of P. List contains
				   all places P is already known to be defined.
				   3rd arg is var & tail of e-list. Instantiate
				   its head to any multiple defn, and return
				   4th arg as new var tail of e-list */

multiple_defn(P,List,[e(F,N,f(I,[]))|L],L1) :-
   got_defn(P,I), notin(I,List), !,
   functor(P,F,N),
   multiple_defn(P,[I|List],L,L1).
multiple_defn(P,_,L,L).

				/* True if X is not in List */
notin(X,List) :- 
	\+ member(X,List).

				/* Return a (possibly empty) list of all
				   callers of the procedure P */
callers(P,I,[c(F,N)|Cs]) :-
	recorded(P,'$caller'(P,F,N,J),Ptr),
	erase(Ptr), !,
	exts(I,J,P),
	callers(P,I,Cs).
callers(_,_,[]).

				/* Record externals. P is defined in I (entry/
				   export), used in J (external/import) */

exts(I,I,_) :- !.			% Do nothing if defn & use in same file
exts(I,undefined,P) :- !,		% Get here if P was mark_interpreted
	entries(I,P).			% Record P is exported from I
exts(I,J,P) :-
	entries(I,P),			% Record P is exported from I
	crecord(J,'$ext'(J,P)).		% Record P is imported to J

				/* Record exports */
entries(undefined,_) :- !.
entries(I,P) :- crecord(I,'$entry'(I,P)).


				/* Quick sort of functor entries. qsort(A,B,C)
				   returns B as the concatenation of sorted
				   A and C (which should already be sorted). */
qsort([X|L],R,R0) :-
	partition(L,X,L1,L2),
	qsort(L2,R1,R0),
	qsort(L1,R,[X|R1]).
qsort([],R,R).

				/* Partition list arg1 on the value of Y
				   into arg3 and arg4 */
partition([X|L],Y,[X|L1],L2) :-
	lte(X,Y), !,
	partition(L,Y,L1,L2).
partition([X|L],Y,L1,[X|L2]) :-
	partition(L,Y,L1,L2).
partition([],_,[],[]).

				/* Comparison predicate for sort */
lte(e(F1,N1,_),e(F2,N2,_)) :-
	lt2(F1,N1,F2,N2).

				/* Order first by functor name, then arity */
lt2(F,N1,F,N2) :- !, N1=<N2.
lt2(F1,_,F2,_) :- F1 @< F2.
%   File   : XRFDEF.PL
%   Author : Dave Bowen
%   Updated: 15 June 1984
%   Purpose: Handles .DEFinition files for XREF.

:- public
	load/1,			% User entry
	load_file/1.		% This is called by XRF.PL 

:- mode
	load(+),
	load_file(+),
	note(+),
	get_pred_spec(+, -),
	check_width(+),
	check_filename(+),
	check_yesorno(+).


/*-----------------------------------------------------------------------------
Data for user definitions.

The following terms may be recorded using a predicate as the key:

	$system		for built-in predicates.
	$known(Where)	for predicates known to be defined in "Where".
	$applies(P,T)	for predicates P which apply one of their arguments T
	$called		for predicates which are called from other places

The following types of term may be recorded under the key $define:

	width(N)	where N is the required page width for the 
			cross-reference listing.
	title(X)	where X is the title for the listing.
	cross_ref_file(F) where F is the filename for the listing.
	globals_file(G)	where G is the filename for the listing of the imports
			and exports of each file. If G='no' then no such 
			listing will be produced.
	update_globals(YesOrNo) where YesOrNo must be 'yes' or 'no'.  If it is
			'yes' each file referenced will have its imports and
			exports updated (these must be in a particular format -
			see below).

-----------------------------------------------------------------------------*/

				% Load in definition file containing system
				% or known predicates, or operators.
load([F|L]) :- !,
	load(F),
	load(L).
load([]) :- !.
load(File) :-
	see_chek(File),
	!,
	load_file(File),
	seen.
load(_).			% if see_chek failed.

				% Load given DEFinition file.
load_file(File) :-
	repeat,
	    read(T),
	    (   T = end_of_file
	    ;   note(T), fail
	    ),
	!,
	write('Definition file '), write(File), write(' loaded'), nl.


				% Process terms in definition file.
note(system(P)) :-
	get_pred_spec(P, S), !,
	crecord(S, '$system').		% P is a system (built-in) predicate
note(known(P,Where)) :-
	get_pred_spec(P, S), !,
	crecord(S, '$known'(Where)).	% P is known to be defined in Where
note(op(Prec,Assoc,Name)) :- !,
	op(Prec, Assoc, Name).		% operator defns handled as normally
note(applies(P,T)) :- !,
	% BEWARE; get_pred_spec is NOT called here!
	recorda(P, '$applies'(P,T), _).	% P must contain T which it Calls
note(called(P)) :-
	get_pred_spec(P, S), !,
	mark_interpreted(S).		% P is called from somewhere
%  The following clauses deal with answers to questions about the layout
%  of output, what files to use, and so on, which the user would otherwise
%  have to type in.
note(width(N)) :- !,
	check_width(N),			% Paper width for cross-ref listing
	recorda('$define', width(N), _).
note(title(T)) :- !,			% Title of cross-ref listing
	recorda('$define', title(T), _).
note(cross_ref_file(F)) :- !, 		% File name for cross-ref listing
	check_filename(F),
	recorda('$define', cross_ref_file(F), _). 
note(globals_file(G)) :- !,		% File name for imports/exports listing
	check_filename(G),		% - may be 'no' meaning no listing reqd
	recorda('$define', globals_file(G), _).
note(update_globals(Yes_or_No)) :- !,	% If imports/exports listed, do you
	check_yesorno(Yes_or_No),	% want your files updated using TECO?
	recorda('$define', update_globals(Yes_or_No), _).
note(Botched) :-
	write('! Unrecognisable definition '),
	write(Botched), write(' -- ignored.'), nl.


% Routines for checking validity of arguments 

%   get_pred_spec lets the user specify a predicate either by giving a
%   most general term (which is what XREF has always wanted in the past)
%   or by giving a Functor/Arity pair (which is more consistent with the
%   other things that want to know about predicates).  If the specifier
%   is not recognisable, get_pred_spec fails so that note/1 can report it.

get_pred_spec(Functor/Arity, MGT) :-
	atom(Functor),
	integer(Arity),
	Arity >= 0,
	!,
	functor(MGT, Functor, Arity).
get_pred_spec(Term, MGT) :-
	nonvar(Term),
	functor(Term, Functor, Arity),
	atom(Functor),	% don't accept known(17) !
	!,
	functor(MGT, Functor, Arity).

				/* Check Width is in range */
check_width(Width) :-  
	integer(Width), Width >= 50, Width =< 150, !.
check_width(Width) :-
	write('! Width ('), write(Width),
	write(') should be between 50 and 150.'), nl,
	fail.

				/* Check legal file name */
check_filename(F) :-
	atom(F), !.
check_filename(F) :-
	write('! File name ('), write(F),
	write(') ill-formed.'), nl,
	fail.

				/* Check for yes/no */
check_yesorno(Y_or_N) :-
	(  Y_or_N == yes ; Y_or_N == no  ), !.
check_yesorno(Y_or_N) :-
	write('! Flag ('), write(Y_or_N),
	write(') should be ''yes'' or ''no''.'), nl,
	fail.
%   File   : XRFMOD.PL
%   Author : Richard A. O'Keefe
%   Updated: 19 September 1984
%   Purpose: Update the declarations in Prolog source files.

/*  This file replaces a TECO program.
    The idea is that the cross-referencer can create import-export
    declarations for Prolog source files, and we can use this to
    maintain such declarations in the source files automatically.
    If a source file contains
		%here%
    we replace that string by the declarations, in the layout
		%declarations%\n
		the actual declarations\n
	        %end%
    If the source file contains
		%declarations% ... %end%
    we replace that text by the new block.  If it contains neither
    %here% nor %declarations%, or if it contains %declarations% but
    that is not followed by %end%, an error message will be printed
    and the file will not be changed.

    What update_declarations(File) does is this.  First it creates a
    new file File.TMP (after discarding the extension if any of File).
    Then it copies characters from File to File.TMP looking for %here%
    or %declarations%.  If it finds %declarations% it will skip until
    it finds %end%.  If anything goes wrong it will print its error
    message and delete File.TMP.  If all goes well, it will write out
    the new declarations, copy the rest of File to File.TMP, and then
    it will do some complicated file juggling.

    First it deletes File.BAK, if there is such a file.
    Then it renames File to File.BAK.
    Then it renames File.TMP to File
    The point of all this juggling is to try to ensure that you won't
    lose your original file.  Something like this ought to be built in.

    I'm sorry about the amazing mess this file is in.  To some extent,
    that's what file hacking does to you, but the main problem is that
    I wrote this in a tearing hurry and didn't stop to design it.
*/

:- public
	update_declarations/3.

/* import
	do_publics/1		from 'XRFOUT.PL',
	do_imports/1		from 'XRFOUT.PL'.
*/
:- mode
	abandon_update(+),
	copy_skipped(+),
	copy_to_percent(-),
	copy_to_percent(+, -),
	finish_update(+, +, +, +),
	new_extension(+, +, -),
	skip_to_percent(+),
	try_percent(+),
	try_percent(+, +, +, -),
	update_declarations(+, +, +).


update_declarations(File, Exports, Imports) :-
	seeing(OldSee),
	telling(OldTell),
	nofileerrors,
	(   new_extension(File, "TMP", TmpFile),
	    see(File),
	    tell(TmpFile),
	    fileerrors,
	    !,
	    copy_to_percent(Which),
	    (   Which = 0, !,	%  neither 'here' nor 'declarations'
		    abandon_update('%here% nor %declarations%')
	    ;   Which = 1, !,	%  $here$ found
		    finish_update(File, TmpFile, Exports, Imports)
	    ;   Which = 2, 	%  $declaratiop
		    skip_to_percent(32), !,
		    finish_update(File, TmpFile, Exports, Imports)
	    ;    abandon_update('%end% after %declarations%')
	    )
	;   fileerrors,
	    close(File),		% ok even if File wasn't open
	    tell(user),
	    write('! problem opening '), write(File),
	    write(' or its .TMP copy'), nl
	),
	see(OldSee),
	tell(OldTell).


%   finish_update writes out the new declarations between new
%   %declarations% and %end% brackets.  It then copies the rest of
%   the original file into the temporary copy, and closes it.
%   Finally, it juggles the files around as described above.  Note
%   that this is for Dec-10 Prolog running on Bottoms-10, when you
%   do rename/2 the file has to be open, and rename/2 will close it.
%   I have no idea whether this will run under C Prolog or not, and
%   I strongly suspect that it won't.

finish_update(File, TmpFile, Exports, Imports) :-
	write('%declarations%'), nl, nl,
	do_publics(Exports),
	do_imports(Imports),
	write('%end%'),  % NO nl
	repeat,
	    get0(C),
	    ( C = 26 ; put(C), fail ),
	!,
	told,			%  close the .TMP file
	new_extension(File, "BAK", BakFile),
	nofileerrors,
	(   see(BakFile), rename(BakFile, [])
	;   true		%  Delete the backup file
	),  !,			%  if it already exists.
	fileerrors,
	see(File),    rename(File, BakFile),
	see(TmpFile), rename(TmpFile, File).


abandon_update(Error) :-
	telling(TmpFile),
	rename(TmpFile, []),		% delete the .TMP file
	seeing(File),
	seen,				% close the original
	tell(user),
	write('! Warning: '), write(File), write(' has no '),
	write(Error), write(' -- not changed.'), nl.


copy_to_percent(Which) :-
	get0(C),
	copy_to_percent(C, Which).


copy_to_percent(26, 0) :- !.		% didn't find a percent at all.
copy_to_percent(0'%, Which) :- !,
	get0(C),
	(   C = 0'h, !, try_percent("ere%", "h%", 1, Which)
	;   C = 0'd, !, try_percent("eclarations%", "d%", 2, Which)
	;   put(0'%), copy_to_percent(Which)
	).
copy_to_percent(C, Which) :-
	put(C),
	get0(D),
	copy_to_percent(D, Which).


try_percent([], _, Which, Which).
try_percent([Char|Chars], Skipped, WillBe, Which) :-
	get0(C),
	(   C = Char, !, try_percent(Chars, [Char|Skipped], WillBe, Which)
	;   copy_skipped([C|Skipped]), copy_to_percent(Which)
	).


copy_skipped([]).
copy_skipped([Char|Chars]) :-
	copy_skipped(Chars),
	put(Char).


%   skip_to_percent skips characters looking for %end%.
%   It succeeds if it finds it, fails if it hits end of file.

skip_to_percent(26) :- !, fail.
skip_to_percent(0'%) :- !,
	try_percent("end%").
skip_to_percent(_) :-
	get0(C),
	skip_to_percent(C).


try_percent([]).
try_percent([Char|Chars]) :-
	get0(C),
	(   C = Char, !, try_percent(Chars)
	;   skip_to_percent(C)
	).



%   new_extension('device:filnam.ext', "NEW", 'device:filnam.NEW')
%   new_extension('device:filnam.',    "NEW", 'device:filnam.NEW')
%   new_extension('device:filnam',     "NEW", 'device:filnam.NEW')

new_extension(File, Extension, NewFile) :-
	name(File, Name),
	(   append(Prefix, [46|_], Name)
	;   Prefix = Name
	),
	append(Prefix, [46|Extension], NewName), !,
	name(NewFile, NewName).

%   File   : XRFOUT.PL
%   Author : Dave Bowen
%   Updated: 22 August 1984
%   Purpose: Output module for XREF.

:- public			% for setof/3
	erase_file_record/1,
	erase_export_record/2,
	erase_import_record/2.
:- mode
	charlength(+, -),
	do_imports(+),
	do_imports(+, +),
	do_output(+),
	do_publics(+),
	do_publics(+, +),
	erase_file_record(-),
	erase_export_record(+, -),
	erase_import_record(+, -),
	export_records(+, -),
	f_back(+, +, -, +),
	f_front(+, +, +, +),
	f_output(+, +),
	f_write(+, +, -, +),
	f_writepred(+, +, +, -, +),
	import_records(+, -),
	makeroom(+, +, -, +),
	number_length(+, -),
	out_globals,
	writes(+),
	write_callers(+, +, -, +),
	write_centre(+, +).


				/* O/P cross-ref listing in reqd format.*/
do_output(Dbase) :-
	get_crf(File, Chars),		%  Get filename for cross-ref listing
	tell_chek(File), !,		%  Get another if can't 'tell' it
	get_title(Title),
	get_width(Width),
	write_centre('**********************************', Width), nl,
	write_centre('* PROLOG CROSS REFERENCE LISTING *', Width), nl,
	write_centre('**********************************', Width), nl,
	nl,
	write_centre(Title, Width), nl,
	nl, nl,
	write('PREDICATE               FILE          CALLED BY'), nl,
	nl, nl,
	f_output(Dbase, Width), !,	%  Output cross-reference list
	told,
	nl, write('Cross-reference listing written to '), write(File), nl,
	out_globals.			%  Output import/export lists if reqd


				/* formatted output of cross refs */
f_output(Dbase, Width) :-
	member(e(F,N,f(I,Cs)), Dbase),
	f_front(F, N, I, Width),
	f_back(Cs, 38, _, Width),
	fail.
f_output(_, _).

				/* Write predicate and file where defined */
f_front(F, N, I, W) :-
	nl, nl,
	f_writepred(F, N, 0, C1, W),
	pos(C1, 24),
	f_write(I, 24, C2, W),
	pos(C2, 38).

				/* Either write callers or 'NOT CALLED' */
f_back([], C1, C2, W) :-
	f_write('NOT CALLED', C1, C2, W).
f_back([H|T], C1, C2, W) :-
	sort([H|T], Sorted),
	write_callers(Sorted, C1, C2, W).

				/* List callers of the predicate */
write_callers([], C, C, _) :- !.
write_callers([c(F,N)|Zs], C1, C4, W) :-
	f_writepred(F, N, C1, C2, W),
	put(32),
	C3 is C2+1,
	write_callers(Zs, C3, C4, W).



				/* L is the length in chars of X */
charlength(X, L) :-
	atomic(X), !,			% Is X an atom or integer?
	name(X, Chars),
	length(Chars, L).
charlength(F/N, L) :-			% Is it of the form F/N
	atom(F), integer(N), !,		% with F/N a plausible functor?
	name(F, S1),	length(S1, L1),
	name(N, S2),	length(S2, L2),
	L is L1+L2+1.
charlength(X, L) :-			% Must be a list (string)
	length(X, L).			% is this used at all?
   
				/* Write predicate remembering format info */
f_writepred('<user>', 0, C1, C2, W) :- !,
	makeroom(6, C1, C2, W),
	write('<user>').
f_writepred(F, N, C1, C2, W) :-
	charlength(F/N, L),
	makeroom(L, C1, C2, W),
	write(F), put(47), write(N).

				/* Write atom ditto */
f_write(X, C1, C2, W) :-
	charlength(X, L),
	makeroom(L, C1, C2, W),
	write(X).

				/* Make sure there is room to write L */ 
makeroom(L, C1, C2, W) :-
	(   L+C1 < W, C2 is C1+L
	;   C1 =< 38, C2 is C1+L
	;   nl, tab(38), C2 is L+38
	),  !.


				/* Write X in the centre of current line */
write_centre(X, W) :-
	charlength(X, L),
	Space is (W-L)/2,
	tab(Space),
	writes(X).

				/* Move from column From to To */
pos(From, To) :-
	(   To > From, Space is To-From
	;   nl, Space = To
	),  !,
	tab(Space).

				/* Write out list of chars (string), or atom */
writes([]) :- !.
writes([C|Cs]) :- !,
	put(C),
	writes(Cs).
writes(X) :-
	write(X).


out_globals :-
	setof(File, erase_file_record(File), Files),
	(   get_globals_file(Globals),	% if we are to write a globals file
	    tell_chek(Globals),		% and we can open it
	    !,				% commit to that file
	    (   member(file(FileName,Exports,Imports), Files),
		write('%   FILE:  '), write(FileName), nl, nl,
		do_publics(Exports),	% write the exported predicates
		do_imports(Imports),	% and imported predicates
		nl, fail		% for each File we've read
	    ;   true			% this'd be "forall" but the Dec-10
	    ),				% compiler doesn't understand that
	    told,			% close the Globals file
	    write('Globals listing written to '), write(Globals), nl
	;   true			% do nothing if no file wanted
	),
	(   get_update_globals(yes),	% if we are to update the declarations
	    !,				% in each source file
	    (   member(file(FileName,Exports,Imports), Files),
		update_declarations(FileName, Exports, Imports),
		fail
	    ;   true
	    ),
	    write('Source files updated.'), nl
	;   true
	).


/*  erase_file_record(File)
    enumerates triples file(FileName,Exports,Imports) where FileName is the
    name of a file that XREF has looked at, Exports is a list of
    export(Functor,Arity) pairs naming predicates exported from that File,
    and imports is a list of import(Functor,Arity,FromFile) triples naming
    predicates imported from other files and indicating which.

    It has the side effect of erasing all this information from the data
    base, which only matters if you're using XREF inside something else.
*/
erase_file_record(file(FileName,Exports,Imports)) :-
	recorded('$file'(_), '$file'(FileName), Ref),
	erase(Ref),
	export_records(FileName, Exports),
	import_records(FileName, Imports).


export_records(FileName, Exports) :-
	setof(Export, erase_export_record(FileName,Export), Exports), !.
export_records(_, []).


erase_export_record(FileName, export(Symbol,Arity)) :-
	recorded(FileName, '$entry'(FileName, Pred), Ref),
	erase(Ref),
	functor(Pred, Symbol, Arity).


import_records(FileName, Imports) :-
	setof(Import, erase_import_record(FileName,Import), Imports), !.
import_records(_, []).


erase_import_record(FileName, import(Symbol,Arity,FromFile)) :-
	recorded(FileName, '$ext'(FileName, Pred), Ref),
	erase(Ref),
	functor(Pred, Symbol, Arity),
	defn_file(Pred, FromFile).


				/* Output public decls (exports) for a file */
do_publics([]) :- !.
do_publics(L) :-
	do_publics(L, ':- public').

do_publics([], L) :- !,	
	put(46), nl, nl.
do_publics([export(F,N)|L], Atom) :-
	write(Atom), nl,
	put(9), writeq(F/N),
	do_publics(L, (',')).

				/* Output import decls for a file */
do_imports([]) :- !.
do_imports(L) :-
	do_imports(L, '%- import').

do_imports([], _) :- !,
	put(46), nl, nl.
do_imports([import(F,N,I)|L], Atom) :-
	write(Atom), nl,
	put(37), put(9), writeq(F/N),
	charlength(F/N, W), Space is 32-W,
	tab(Space), write('from '), write(I),
	do_imports(L, (',')).

%   File   : XRFTTY.PL
%   Author : Dave Bowen
%   Updated: 19 September 1984
%   Purpose: Terminal interaction for XREF

:- public
	member/2.				% this is Called

:- mode
	actionchar(+, +, -),
	append(?, ?, ?),
	get_crf(-, -),
	get_globals_file(-),
	get_title(-),
	get_update_globals(?),
	get_width(-),
	member(?, +),
	readtonl(-),
	readtonl0(-),
	readln(+, -),
	see_chek(+),
	tell_chek(+),
	warn(+, +),
	writepred(+, +),
	writes(+),
	yesno(+, ?).


				/* Get name for cross-refs file: instantiate
				   both File and Chars */

get_crf(File, Chars) :- 			%  Name specified in DEF file?
	recorded('$define', cross_ref_file(File), Ref),
	erase(Ref), !,				%  Yes, remove from database
	name(File, Chars).
get_crf(File, Chars) :-
	repeat,
	    ttynl, display('Filename for Cross-Reference listing: '), ttyflush, 
	    readtonl(Chars),
	!,
	name(File, Chars),			%  Read file name from terminal
	check_filename(File).			%  Check name is appropriate

				/* Get required page width */
get_width(W) :-
 	recorded('$define', width(W), Ref),	% Was it specified in DEF file?
 	erase(Ref), !.
get_width(W) :-					% No, prompt for it.
 	repeat,
	    ttynl, display('Width: '), ttyflush,
	    readtonl(Chars),
	!,
	name(W, Chars),
	check_width(W).

				/* Get title for Cross-Reference listing */
get_title(T) :-
	recorded('$define',title(T),Ref),	% Specified in a DEF file?
	erase(Ref), !.
get_title(T) :-
	ttynl, display('Title: '), ttyflush,	% Not in DEF file, ask for it.
	readtonl0(T).				% Empty line is allowed.

			/* Gets filename (or "no") for import/export lists */

get_globals_file(File) :-			% Specified in DEF file?
	recorded('$define', globals_file(File), Ref),
	erase(Ref), !,
	File \== no.
get_globals_file(File) :-			% No, ask for it.
	yesno('Do you want a listing of imports/exports', yes),
	repeat,
	    ttynl, display('Filename for imports/exports: '), ttyflush, 
	    readtonl(Chars),			% No cut; if we can't write
	    name(File, Chars).			% the file, try another name.

				/* Does the user want us to update the
				   Import/Export lists in all the files? */
get_update_globals(Yes_or_No) :-
	recorded('$define', update_globals(Y_or_N), Ref),
	erase(Ref), !,
	Y_or_N = Yes_or_No.
get_update_globals(Yes_or_No) :-
	yesno('Alter the import/export declarations in your files', Yes_or_No).


% Utilities for input/output 

member(X, [X|_]).
member(X, [_|L]) :-
	member(X, L).


append([], L, L).
append([H|T], L, [H|R]) :-
	append(T, L, R).


yesno(Question, Answer) :-
	repeat,
	    ttynl, display(Question), display('? '), ttyflush,
	    readtonl(Ans),
	    (   Ans = [0'y|_], !, Answer = yes
	    ;   Ans = [0'n|_], !, Answer = no
	    ;   display('! Answer y(es) or n(o)'), fail
	    ).

			/* Give a warning about a predicate */
warn(Pred, State) :-
	functor(Pred, F, N),
	telling(File), tell(user),
	write('** WARNING: '), write(F/N is State), nl,
	tell(File).


writepred('<user>', 0) :- !,
	write('<user>').
writepred(F, N) :-
	writeq(F/N).


				/* See file or complain if it doesn't exist */
see_chek(File) :-
	(   nofileerrors, see(File), !, fileerrors
	;   fileerrors,
	    display('! Can''t read '), display(File), ttynl,
	    fail
	).

				/* Open file for output or complain */
tell_chek('TTY:') :- !,
	tell(user).
tell_chek(File) :-
	(   nofileerrors, tell(File), !, fileerrors
	;   fileerrors,
	    display('! Can''t write '), display(File), ttynl,
	    fail
	).


% Low level input routines


				/* Read a line, returning character list */
readtonl(Cs) :-
	readln(ignore_blanks,Cs).		% - for reading file names

readtonl0(Cs) :-
	readln(keep_blanks,Cs).			% - for reading text

readln(Flag,Cs) :- 
	get0(C),!,
	actionchar(Flag,C,Cs).

actionchar(_,31,[]) :- !.			% newline: return []
actionchar(_,10,[]) :- !.			% ditto
actionchar(_,26,_) :- !, fail.			% ^Z: fail
actionchar(_,4,_) :- !, fail.			% ^D: fail
actionchar(_,-1,_) :- !, fail.			% ^D: fail
actionchar(_,end_of_file,_) :- !, fail.			% ^D: fail
actionchar(ignore_blanks,C,Cs) :-		% ignore layout characters
	C=<32, !, 				%  (incl. space) if reqd
	readln(ignore_blanks,Cs).
actionchar(Flag,C,[Cfirst|Crest]) :- !,		% other: construct list
%	("a"=<C, C=<"z", !, 			% convert lower to upper case
%	    Cfirst is C+"A"-"a"
%	;   Cfirst=C),
	Cfirst=C,
	readln(Flag,Crest).
