% File: Check.pl
% Created: 17-Feb-93
% Last Update: 18-Feb-93
% Author: Afzal Ballim
% Purpose: A library checking facility.
% Needs: -

:- dynamic 'lib:::check:::isasp'/1.

%============================================================
% Predicate: check_library(Lib)
% Modes: check_library(+Lib)
% 
% Purpose: performs a check on the given library. The check
%	   has 4 parts:
%		1) is each of the "public" predicates in fact
%		   defined in one of the "include" files.
%		2) Are there any "include" files that do not
%		   contain "public" functions.
%		3) Are the dependency files all necessary.
%		4) Are there any undefined predicates.
% 
% Licensed Use: 
% 
% Created: 17-Feb-93
% Last Update: 18-Feb-93
%============================================================

check_library(Lib) :-
    name(Lib,LibSuffix),			    % generate
    name(LibPreSuf,[108,105,98|LibSuffix]),	    % path to
    absolute_file_name(library(LibPreSuf),LibFull), % lib.
    format("Checking ~w for consistency...~n",[Lib]),
    format("Processing ~w...~n",[LibFull]),
    'lib:::process_lib'(Lib,LibFull).

%============================================================
% Predicate: lib:::process_lib/2
% Modes: lib:::process_lib(+LibName,+LibFile)
% 
% Purpose: This is the top of the actual processing of the
%	   library. Reads the library file, chasing the 
%	   directives to read the necessary include and
%	   dependency files, producing data that is then
%	   analysed.
% 
% Licensed Use: 
% 
% Created: 18-Feb-93
% Last Update: 
%============================================================

'lib:::process_lib'(Lib,LibFile) :-
    open(LibFile,read,Stream),
    read(Stream,Term),!,
    'lib:::check_lib_directives'(Lib,Term,Stream,Data),!,
    close(Stream),
    'lib:::check_lib_data'(Lib,Data),
    retractall('lib:::check:::isasp'(_)).

/* The data entries have the following format:

    PubPreds = a list of predicate specs
    IncPreds = a list of items of the form:
		    ph(PredHead,PredCalls,File)
    DepPreds = also of the form ph(PredHead,PredCalls,File)
    Synopsis = the synopsis
    Files    = files(IncLst,DepList)

	PredCalls is simply a list of non-built-ins that are 
	called by the predicate.
*/

%============================================================
% Predicate: 'lib:::check_lib_directives'/4
% Modes: 
% 
% Purpose: check the directives in the library file
% 
% Licensed Use: 
% 
% Created: 18-Feb-93
% Last Update: 
%============================================================
'lib:::check_lib_directives'(_,end_of_file,_,_) :- !.
'lib:::check_lib_directives'(Lib,Term,Stream,Data) :-
	expand_term(Term,Form),
	'lib:::check_lib_directive'(Lib,Form,Data),
	read(Stream,NextTerm),!,
	'lib:::check_lib_directives'(Lib,NextTerm,Stream,Data).

%============================================================
% Predicate: lib:::check_lib_directive/3
% Modes: 
% 
% Purpose: check a Term read from the library file.
% 
% Licensed Use: 
% 
% Created: 18-Feb-93
% Last Update: 
%============================================================
'lib:::check_lib_directive'(Lib,
	(:-library_description(Lib,Synopsis)),
	[_,_,_,_,Synopsis]) 
:- !.

'lib:::check_lib_directive'(Lib,
	(:-library_publics(Lib,Publics)),
	[Publics,_,_,_,_]) 
:- !.

'lib:::check_lib_directive'(Lib,
	(:-library_includes(Lib,Includes)),
	[_,I,_,files(Includes,_),_]) :- 
    !,'lib:::find_file_preds'(Includes,[],I).

'lib:::check_lib_directive'(Lib,
	(:-library_dependencies(Lib,Dependencies)),
	[_,_,Dep,files(_,Dependencies),_]) :- 
    !,'lib:::find_file_preds'(Dependencies,[],Dep).

'lib:::lib_check_directive'(_,(:-X),_):-
    (X=library_synopsis(_,_);
     X=library_publics(_,_);
     X=library_includes(_,_);
     X=library_dependencies(_,_)),!,
    format("**ERROR** Repeated directive ~w~n",[X]).

'lib:::check_lib_directive'(_,Form,_) :- 
    !,format("**ERROR** Unknown library entry ~w~n",[Form]).

%============================================================
% Predicate: lib:::find_file_preds/3
% Modes: 
% 
% Purpose: get all the predicates defined in file list.
% 
% Licensed Use: 
% 
% Created: 18-Feb-93
% Last Update: 
%============================================================
'lib:::find_file_preds'([],P,P).

'lib:::find_file_preds'([F1|Fs],Fsf,FPs):-
    nofileerrors,
    absolute_file_name(library(F1),File),
    open(File,read,Stream),
    read(Stream,Trm),expand_term(Trm,Term),
    'lib:::find_afiles_preds'(Stream,F1,Term,Fsf,FsfA),!,
    'lib:::find_file_preds'(Fs,FsfA,FPs).

% the file is a library file
'lib:::find_file_preds'([F1|Fs],Fsf,FPs):-
    name(F1,Libname),
    name(F1lib,[108,105,98|Libname]),
    nofileerrors,
    absolute_file_name(library(F1lib),File),
    open(File,read,Stream),
    read(Stream,Trm),expand_term(Trm,Term),
    'lib:::find_afiles_preds'(Stream,F1,Term,Fsf,FsfA),!,
    'lib:::find_file_preds'(Fs,FsfA,FPs).


'lib:::find_file-preds'([F1|Fs],Fsf,FPs) :-
	format("**ERROR** Cannot find file ~w~n",[F1]),!,
	'lib:::find_file_preds'(Fs,Fsf,FPs).

%============================================================
% Predicate: lib:::find_afiles_preds/5
% Modes: 
% 
% Purpose: get all the predicates fro a file, does not handle
%	   indirection at the moment.
% 
% Licensed Use: 
% 
% Created: 18-Feb-93
% Last Update: 
%============================================================


'lib:::find_afiles_preds'(Stream,_,end_of_file,FP,FP) :-
    !,close(Stream).

% A library file.... include its includes
'lib:::find_afiles_preds'(Stream,File,(:- library_includes(_,FF)),
	Fin,Fout) :-
	'lib:::find_file_preds'(FF,[],Flibs),
	'lib:::append_as_lib'(File,Flibs,Fin,FNew),
	read(Stream,Term),expand_term(Term,NewTerm),
	'lib:::find_afiles_preds'(Stream,File,NewTerm,FNew,Fout).

% A library file.... include its dependencies
'lib:::find_afiles_preds'(Stream,File,(:- library_dependencies(_,FF)),
	Fin,Fout) :-
	'lib:::find_file_preds'(FF,[],Flibs),
	'lib:::append'(Fin,Flibs,FNew),
	read(Stream,Term),expand_term(Term,NewTerm),
	'lib:::find_afiles_preds'(Stream,File,NewTerm,FNew,Fout).

'lib:::find_afiles_preds'(Stream,File,Term,Fin,Fout) :-
    functor(Term,':-',NArgs),!,
    if(NArgs=1,
       (format("**WARNING** Directive ~w ignored~n",[Term]),
	FNew=Fin),
       (arg(1,Term,Arg1),
	arg(2,Term,Arg2),
	'lib:::find_calls'(Arg2,[],Calls),
	functor(Arg1,Arg1Fn,Arg1Ar),
	FNew=[ph(Arg1Fn/Arg1Ar,Calls,File)|Fin])),
    read(Stream,Trm),
    expand_term(Trm,NewTerm),
    'lib:::find_afiles_preds'(Stream,File,NewTerm,FNew,Fout).

'lib:::find_afiles_preds'(Stream,File,Term,Fin,Fout) :-
    functor(Term,TermFn,TermAr),
    FNew=[ph(TermFn/TermAr,[],File)|Fin],
    read(Stream,Trm),expand_term(Trm,NewTerm),
    'lib:::find_afiles_preds'(Stream,File,NewTerm,FNew,Fout).

    
%============================================================
% Predicate: 'lib:::find_calls'/2
% Modes: 'lib:::find_calls'(+Body,-CallList)
% 
% Purpose: CallList is a list of non-built-in calls that
%	   appear in body.
% 
% Licensed Use: 
% 
% Created: 19-Feb-93
% Last Update: 
%============================================================

'lib:::find_calls'(Goal,Calls,Calls) :-
	var(Goal),!.

'lib:::find_calls'((Goal1,Goal2),Csofar,Calls) :-
	'lib:::find_calls'(Goal1,Csofar,C1),!,
	'lib:::find_calls'(Goal2,C1,Calls).

'lib:::find_calls'((Goal1;Goal2),Csofar,Calls) :-
	'lib:::find_calls'(Goal1,Csofar,C1),!,
	'lib:::find_calls'(Goal2,C1,Calls).

'lib:::find_calls'((Goal1 -> Goal2),Csofar,Calls) :-
	'lib:::find_calls'(Goal1,Csofar,C1),!,
	'lib:::find_calls'(Goal2,C1,Calls).

'lib:::find_calls'((Goal1 -> Goal2 ; Goal3),Csofar,Calls) :-
	'lib:::find_calls'(Goal1,Csofar,C1),!,
	'lib:::find_calls'(Goal2,C1,C2),!,
	'lib:::find_calls'(Goal3,C2,Calls).

'lib:::find_calls'(if(Goal1,Goal2),Csofar,Calls) :-
	'lib:::find_calls'(Goal1,Csofar,C1),!,
	'lib:::find_calls'(Goal2,C1,Calls).

'lib:::find_calls'(if(Goal1,Goal2,Goal3),Csofar,Calls) :-
	'lib:::find_calls'(Goal1,Csofar,C1),!,
	'lib:::find_calls'(Goal2,C1,C2),!,
	'lib:::find_calls'(Goal3,C2,Calls).

'lib:::find_calls'((\+Goal1),Csofar,Calls) :-
	'lib:::find_calls'(Goal1,Csofar,Calls).

'lib:::find_calls'(call(Goal1),Csofar,Calls) :-
	'lib:::find_calls'(Goal1,Csofar,Calls).

'lib:::find_calls'(Goal,Csofar,Calls) :-
	functor(Goal,Fn,Ar),
	functor(GenGoal,Fn,Ar),
	((predicate_property(GenGoal,built_in),
	  Calls=Csofar);
	 Calls=[Fn/Ar|Csofar]).

'lib:::find_calls'(_,Calls,Calls).

%============================================================
% Predicate: 'lib:::check_lib_data'/2
% Modes: 'lib:::check_lib_data'(+Lib,+Data)
% 
% Purpose: verify that the library is correct.
% 
% Licensed Use: 
% 
% Created: 18-Feb-93
% Last Update: 
%============================================================

/* when we get here Data is a 5 element list as follows:

	Elem 1 =    List of public predspecs, should be non-empty
	Elem 2 =    List of included predicates**
	Elem 3 =    List of dependency predicates**
	Elem 4 =    file(IncFiles,DepFiles)
	Elem 5 =    Synopsis of the library (if defined)

    ** The included and dependecy predicates are of the form
	ph(Pred/Arity,CallList,File)
       where File is the file where Pred/Arity is defined, and
       CallList is alist of non-built-in predicates that Pred
       calls. This list will have an entry for EACH clause of
       a predicate, although all we really need is one for each
       predicate (per file, i.e., multifile entries should be kept
       separate).
*/
	

'lib:::check_lib_data'(Lib,
	    [Pub,IncP,DepP,files(IncF,DepF),Syn]) :- 
    format("Report on library '~w':~n~n",[Lib]),
    'lib:::check_lib_data_Synopsis'(Syn),	% synopsis
    'lib:::check_lib_data_Pubs'(Pub,IncP), % pubs defined
    'lib:::append'(IncP,DepP,AllPs),
    'lib:::check_lib_data_Incs'(Pub,IncP,AllPs,IncF), % includes
    'lib:::check_lib_data_Deps'(Pub,IncP,DepP,AllPs,DepF). % depends

/* The synopsis must be defined */
'lib:::check_lib_data_Synopsis'(Syn) :-
    ((nonvar(Syn),
     format("Synopsis: ~w~n====~n",[Syn]));
     format("**ERROR** The synopsis is not defined.~n",[])),!.

/*  All the public predicates must be defined in included files
    (not dependency files).
*/

'lib:::check_lib_data_Pubs'([],_) :- format("====~n~n",[]),!.
'lib:::check_lib_data_Pubs'([P|Pubs],IncP) :-
    format("Public predicate ~w...",[P]),
    (('lib:::member'(ph(P,_,F),IncP),
      format("defined in ~w~n",[F]));
     format("**ERROR** not defined in an included file~n",[])),!,
    'lib:::check_lib_data_Pubs'(Pubs,IncP).


/* Include files should contain nothing but public predicates,
   and support for public predicates. So, for each included file
   look for predicates that are neither public, nor supports for
   a public.

   Note: could get rid of the setof and improve performance by
         noting that IncP is ordered in the reverse order of IncF.

*/

'lib:::check_lib_data_Incs'(_,_,_,[]) :- format("====~n~n",[]),!.
'lib:::check_lib_data_Incs'(Pub,IncP,AllPs,[F|IncF]) :-
    format("Checking included file ~w...~n",[F]),
    setof(FP,(ZZZ)^
	'lib:::member'(ph(FP,ZZZ,F),IncP),FPreds), % each must have a path
					      % to a public pred
    'lib:::check_lib_data_Inc_preds'(FPreds,Pub,IncP,AllPs),!,
    format("Checked included file ~w~n~n",[F]),
    'lib:::check_lib_data_Incs'(Pub,IncP,AllPs,IncF).
    

'lib:::check_lib_data_Inc_preds'([],_,_,_) :- !.

'lib:::check_lib_data_Inc_preds'([FP|FPreds],Pub,IncP,AllPs) :-
    % if FP is a public, then we are done.
    'lib:::member'(FP,Pub),!,
    format("    ~w is a public predicate~n",[FP]),
    'lib:::member'(ph(FP,Calls,_),IncP),
    'lib:::check_calls'(Calls,AllPs),
    'lib:::check_lib_data_Inc_preds'(FPreds,Pub,IncP,AllPs).

'lib:::check_lib_data_Inc_preds'([FP|FPreds],Pub,IncP,AllPs) :-
    'lib:::isa_support_predicate'(FP,Pub,IncP),!,
    format("    ~w is a support predicate~n",[FP]),
    'lib:::member'(ph(FP,Calls,_),IncP),
    'lib:::check_calls'(Calls,AllPs),
    'lib:::check_lib_data_Inc_preds'(FPreds,Pub,IncP,AllPs).

'lib:::check_lib_data_Inc_preds'([FP|FPreds],Pub,IncP,AllPs) :-
    format("**WARNING** ~w is neither public nor a support predicate~n",
	   [FP]),!,
    'lib:::check_lib_data_Inc_preds'(FPreds,Pub,IncP,AllPs).


/* depend files should contain nothing but support for public
   predicates
*/

'lib:::check_lib_data_Deps'(_,_,_,_,[]) :- format("====~n~n",[]),!.

'lib:::check_lib_data_Deps'(Pub,IncP,DepP,AllPs,[DF|DepF]) :-
    format("Checking dependency file ~w...~n",[DF]),
    setof(DFP,(ZZZ)^'lib:::member'(ph(DFP,ZZZ,DF),DepP),DFPs),
    'lib:::check_lib_data_Dep_preds'(DFPs,Pub,AllPs),!,
    format("Checked dependency file ~w~n~n",[DF]),
    'lib:::check_lib_data_Deps'(Pub,IncP,DepP,AllPs,DepF).


'lib:::check_lib_data_Dep_preds'([],_,_) :- !.

'lib:::check_lib_data_Dep_preds'([P|Preds],Pub,AllPs) :-
    'lib:::isa_support_predicate'(P,Pub,AllPs),!,
    format("    ~w is a support predicate~n",[P]),
    'lib:::member'(ph(P,Calls,_),AllPs),
    'lib:::check_calls'(Calls,AllPs),
    'lib:::check_lib_data_Dep_preds'(Preds,Pub,AllPs).

'lib:::check_lib_data_Dep_preds'([P|Preds],Pub,AllPs) :-
    !,format("**WARNING**  ~w is not a support predicate~n",[P]),
    'lib:::check_lib_data_Dep_preds'(Preds,Pub,AllPs).

'lib:::isa_support_predicate'(P,_,_) :-
	'lib:::check:::isasp'(P).

'lib:::isa_support_predicate'(_,[],_) :- !,fail.

'lib:::isa_support_predicate'(Pred,Pub,IncP) :-
    'lib:::member'(P,Pub),
    'lib:::member'(ph(P,PSup,_),IncP),
    'lib:::member'(Pred,PSup),!,
% UGLY HACK
    asserta('lib:::check:::isasp'(Pred)).

'lib:::isa_support_predicate'(Pred,Pub,All) :-
    'lib:::member'(ph(PCaller,PSup,_),All),
    \+ PCaller = Pred,				% don't recurse on same
    'lib:::member'(Pred,PSup),
% prevent cycles
    'lib:::remove_all_Psups_refs'(Pred,All,Alln),
    'lib:::isa_support_predicate'(PCaller,Pub,Alln),
% UGLY HACK
    asserta('lib:::check:::isasp'(Pred)).

'lib:::remove_all'(_,[],[]).
'lib:::remove_all'(P,[P|T],L) :- 
	!,'lib:::remove_all'(P,T,L).
'lib:::remove_all'(P,[H|T],[H|L]) :-
	!,'lib:::remove_all'(P,T,L).

'lib:::remove_all_Psups_refs'(_,[],[]).
'lib:::remove_all_Psups_refs'(P,[H1|T],[H1p|L]) :-
	H1=ph(Pr,Pss,L1),
	H1p=ph(Pr,Pssp,L1),
	'lib:::remove_all'(P,Pss,Pssp),!,
	'lib:::remove_all_Psups_refs'(P,T,L).

'lib:::append'([],L3,L3).

'lib:::append'([H1|T1],L2,[H1|T3]) :-
	'lib:::append'(T1,L2,T3).

'lib:::append_as_lib'(_,[],L3,L3).

'lib:::append_as_lib'(File,[ph(X,Y,_)|T1],L2,[ph(X,Y,File)|T3]) :-
	'lib:::append_as_lib'(File,T1,L2,T3).

'lib:::member'(Elem, [Elem | _]).
'lib:::member'(Elem, [_ | Rest_of_list]) :- 'lib:::member'(Elem, Rest_of_list).

'lib:::check_calls'([],_).

'lib:::check_calls'([H|T],X) :-
	'lib:::member'(ph(H,_,_),X),!,
	'lib:::check_calls'(T,X).

'lib:::check_calls'([H|T],X) :-
	format("**ERROR** ~w is called but not defined.~n",[H]),	
	'lib:::check_calls'(T,X).
