
/* This file contains functions to setup/initialise stuff for the Rosenblatt
& Payton mechanism */


#include <stdio.h>
#include <math.h>
/*gn*/
#include <suntool/sunview.h>
/*gn*/
#include <suntool/panel.h>
/*gn*/
#include <suntool/canvas.h>
#include "../sim_env/structures.h"
#include "../sim_env/util_fns.h"
#include "../sim_env/env_defs.h"
#include "../sim_env/env_spec.h"
#include "../sim_env/animal_defs.h"
#include "asm_defs.h"
#include "asm_decs.h"


/* this function gets run every timestep in order to control the animal's
behaviour */
int select_action(arg1, arg2, arg3, arg4, arg5_1, arg5_2, arg6_1, arg6_2,
arg7, arg8, arg9, arg10, arg11, arg12, arg13, arg14, arg15, arg16, arg17,
arg18, arg19, arg20, arg21, arg22, arg23, arg24, arg25, arg26, arg27, arg28,
arg29, arg30, arg31, arg32)
double arg1;			/* shortage of fat */
double arg2;			/* shortage of carbo */
double arg3;			/* shortage of protein */
double arg4;			/* shortage of water */
double arg5_1;			/* how much animal temp is below desirable */
double arg5_2;			/* how much animal temp is above desirable */
double arg6_1;			/* how much square temp is below desirable */
double arg6_2;			/* how much square temp is above desirable */
double arg7;			/* 'courtedness' of mate */
double arg8;			/* how clean the animal is */
double arg9;			/* closeness of nightfall */
double arg10;			/* estimated variance of the animal's position */
double arg11;			/* estimated distance from the animal to its den */
double *arg12;			/* stimulus of perceived food in all directions */
double *arg13;			/* stimulus of remembered food in all directions */
double *arg14;			/* stimulus of perceived water in all directions */
double *arg15;			/* stimulus of remembered water in all directions */
double *arg16;			/* stimulus of perceived shelter in all directions */
double *arg17;			/* stimulus of perceived shade in all directions */
double *arg18;			/* stimulus of perceived p1 in all directions */
double *arg19;			/* stimulus of perceived p2 in all directions */
double *arg20;			/* stimulus of perceived dp in all directions
						   (adjacent squares) */
double *arg21;			/* stimulus of perceived irr in all directions
						   (adjacent squares) */
double *arg22;			/* stimulus of perceived mate in all directions */
double *arg23;			/* stimulus of perceived den in all directions */
double *arg24;			/* stimulus of remembered den in all directions */
double *arg25;			/* stimulus of perceived edge in all directions */
double arg26;			/* how long it is since scanning for predators */
double arg27;			/* distance from cover */
double arg28;			/* time since predator last perceived */
double arg29;			/* animal health */
double arg30;			/* animal permanent injury */
double *arg31;			/* stimulus of perceived dp in all directions
						   (non-adjacent squares) */
double *arg32;			/* stimulus of perceived irr in all directions
						   (non-adjacent squares) */
{

  int i;

  /* assign parameters passed to behave_choice to global variables to be
  used for all behaviour functions */
  fat_minus = arg1;
  carbo_minus = arg2;
  protein_minus = arg3;
  water_minus = arg4;
  an_temp_minus= arg5_1;
  an_temp_plus = arg5_2;
  sqr_temp_minus= arg6_1;
  sqr_temp_plus = arg6_2;
  mate_courtedness = arg7;
  cleanliness = arg8;
  night_prox = arg9;
  variance = arg10;
  den_distance = arg11;
  scan_time = arg26;
  cover_distance = arg27;
  time_since_predator = arg28;
  health = arg29;
  p_injury = arg30;

  /* get the maximums (and directions of them) for all direction dependent
  stimuli */
  get_maxs_and_dirs(&(arg12[0]), &(arg13[0]), &(arg14[0]), &(arg15[0]),
	&(arg16[0]), &(arg17[0]), &(arg18[0]), &(arg19[0]), &(arg20[0]),
	&(arg21[0]), &(arg22[0]), &(arg23[0]), &(arg24[0]), &(arg25[0])); 

  /* initialise all nodes */
  for (i = 0; i < num_nodes; i++) {
	nodes[i].all_act_received = FALSE;
	nodes[i].activation = 0.0;
  }

  /* calculate activations of sensors of internal and external environment*/
  calc_internal_sensors();
  calc_external_sensors(arg12, arg14, arg16, arg17, arg18, arg19, arg20,
	arg21, arg22, arg23);
  calc_indeterminate_sensors();
  calc_directed_external_sensors(arg12, arg13, arg14, arg15, arg16, arg17,
	arg18, arg19, arg20, arg21, arg22, arg23, arg24, arg25, arg31, arg32);

  /* pass activation down through network from sensors and top-level nodes
  to intermediate nodes and finally bottom-level nodes */
  pass_activation_through_network();

  /* choose the action associated with the lowest level node chosen */
  action = get_action();

  return(action);
}


void set_up_strategy()
{

  /* input a list of descriptors of the animal's internal state */
  init_internal_sensors();

  /* input a list of descriptors of the environment */
  init_external_sensors();

  /* input a list of descriptors of the environment */
  init_indeterminate_sensors();

  /* input a list of directed descriptors of the environment */
  init_directed_external_sensors();

  /* input a set of nodes */
  init_nodes();

  /* input all the connections between nodes */
  init_links();
/*g*/
/*g*/  /* also initialise a longitudinal profiles display */
/*g*/  set_up_long_profile();
/*g*/  set_up_long_profile2();
}


/* function to input a list of sensors of the internal state of the animal
from a file */
void init_internal_sensors()
{

  int i;
  FILE *f;

  f = fopenwithcheck(INT_SENSORS_FNAME, "r");

  /* read in names of internal sensors */
  num_int_sensors = 0;
  while (getline(f, int_sensors[num_int_sensors], MAX_LINE_LENGTH) != 0)
	num_int_sensors++;

  fclose(f);

/*  printf("read in %d internal sensors\n", num_int_sensors);
*/}


/* function to input a list of sensors of the external environment from a
file.  The format of the file is:
<external sensor name>
..[repeated]
*/
void init_external_sensors()
{

  int i;
  FILE *f;

  f = fopenwithcheck(EXT_SENSORS_FNAME, "r");

  /* read in names of external sensors */
  num_ext_sensors = 0;
  while (getline(f, ext_sensors[num_ext_sensors], MAX_LINE_LENGTH) != 0)
	num_ext_sensors++;

  fclose(f);

/*  printf("read in %d external sensors\n", num_ext_sensors);
*/}


/* function to input a list of sensors of indeterminate stimuli of the
environment from a file.  The format of the file is:
<indeterminate sensor name>
..[repeated]
*/
void init_indeterminate_sensors()
{

  int i;
  FILE *f;

  f = fopenwithcheck(INDET_SENSORS_FNAME, "r");

  /* read in names of indeterminate sensors */
  num_indet_sensors = 0;
  while (getline(f, indet_sensors[num_indet_sensors], MAX_LINE_LENGTH) != 0)
	num_indet_sensors++;

  fclose(f);

/*  printf("read in %d indeterminate sensors\n", num_indet_sensors);
*/}


/* function to input a list of directed sensors of the external environment
from a file.  The format of the file is:
<directed external sensor name>
..[repeated]
*/
void init_directed_external_sensors()
{

  int i;
  FILE *f;

  f = fopenwithcheck(DIRECTED_EXT_SENSORS_FNAME, "r");

  /* read in names of directed external sensors */
  num_directed_ext_sensors = 0;
  while (getline(f, directed_ext_sensors[num_directed_ext_sensors],
		 MAX_LINE_LENGTH) != 0)
	num_directed_ext_sensors++;

  fclose(f);

/*  printf("read in %d directed external sensors\n", num_directed_ext_sensors);
*/}


/* this function reads in the names of all the different types of nodes */
void init_nodes()
{

  num_nodes = 0;

  init_top_level_nodes();

  init_intermediate_nodes();

  init_bottom_level_nodes();
}


/* this function reads in a list of all the top level (= sub_systems /
behaviours / activities) nodes (and their names). 
The file will have the format:
<node name>
...[repeated as many times as necessary]
*/
void init_top_level_nodes()
{

  int i;
  FILE *f;

  f = fopenwithcheck(TOP_LEVEL_NODES_FNAME, "r");

  num_top_level_nodes = 0;
  while (getline(f, nodes[num_top_level_nodes].name, MAX_LINE_LENGTH) != 0)
	num_top_level_nodes++;

  num_nodes = num_top_level_nodes;

  /* initialise fields in node structures */
  for (i = 0; i < num_top_level_nodes; i++) {
	nodes[i].activation = 0.0;
	nodes[i].num_in_links = 0;
	nodes[i].num_out_links = 0;
	nodes[i].type = TOP_LEVEL;
  }

  fclose(f);

/*  printf("%d top level nodes read in\n", num_top_level_nodes);
*/}


/* this function reads in a list of all the intermediate level nodes (and
their names).  The file will have the format:
<node name>
...[repeated as many times as necessary]
*/
void init_intermediate_nodes()
{

  int i;
  FILE *f;

  f = fopenwithcheck(INTERMEDIATE_NODES_FNAME, "r");

  num_intermediate_nodes = 0;
  while (getline(f, nodes[num_top_level_nodes+num_intermediate_nodes].name,
	  MAX_LINE_LENGTH) != 0)
	num_intermediate_nodes++;

  num_nodes += num_intermediate_nodes;

  /* initialise fields in node structures */
  for (i = num_top_level_nodes; i < num_nodes; i++) {
	nodes[i].activation = 0.0;
	nodes[i].num_in_links = 0;
	nodes[i].num_out_links = 0;
	nodes[i].type = INTERMEDIATE;
  }

  fclose(f);

/*  printf("%d intermediate nodes read in\n", num_intermediate_nodes);
*/}


/* this function reads in a list of all the bottom level nodes (and their
names).  The file will have the format:
<node name>
...[repeated as many times as necessary]
*/
void init_bottom_level_nodes()
{

  int i;
  FILE *f;
  char node_name[50];
  int node_num;
  char eol;
  int dir_len;

  f = fopenwithcheck(BOTTOM_LEVEL_NODES_FNAME, "r");

  node_num = num_top_level_nodes + num_intermediate_nodes;

  while (getline(f, node_name, MAX_LINE_LENGTH) != 0) {

	if (node_name[0] != '8') {
	  strcpy(nodes[node_num++].name, node_name); 
	}

	else {
	  dir_len = strlen(nodes[node_num].name);
	  eol = nodes[node_num].name[dir_len];
	  strcpy(nodes[node_num].name, &(node_name[2]));
	  dir_len = strlen(nodes[node_num].name);
	  nodes[node_num].name[dir_len] = ' ';
	  nodes[node_num].name[dir_len+1] = 'n';
	  nodes[node_num].name[dir_len+2] = eol;
	  node_num++;
	  strcpy(nodes[node_num].name, &(node_name[2]));
	  nodes[node_num].name[dir_len] = ' ';
	  nodes[node_num].name[dir_len+1] = 'n';
	  nodes[node_num].name[dir_len+2] = 'e';
	  nodes[node_num].name[dir_len+3] = eol;
	  node_num++;
	  strcpy(nodes[node_num].name, &(node_name[2]));
	  nodes[node_num].name[dir_len] = ' ';
	  nodes[node_num].name[dir_len+1] = 'e';
	  nodes[node_num].name[dir_len+2] = eol;
	  node_num++;
	  strcpy(nodes[node_num].name, &(node_name[2]));
	  nodes[node_num].name[dir_len] = ' ';
	  nodes[node_num].name[dir_len+1] = 's';
	  nodes[node_num].name[dir_len+2] = 'e';
	  nodes[node_num].name[dir_len+3] = eol;
	  node_num++;
	  strcpy(nodes[node_num].name, &(node_name[2]));
	  nodes[node_num].name[dir_len] = ' ';
	  nodes[node_num].name[dir_len+1] = 's';
	  nodes[node_num].name[dir_len+2] = eol;
	  node_num++;
	  strcpy(nodes[node_num].name, &(node_name[2]));
	  nodes[node_num].name[dir_len] = ' ';
	  nodes[node_num].name[dir_len+1] = 's';
	  nodes[node_num].name[dir_len+2] = 'w';
	  nodes[node_num].name[dir_len+3] = eol;
	  node_num++;
	  strcpy(nodes[node_num].name, &(node_name[2]));
	  nodes[node_num].name[dir_len] = ' ';
	  nodes[node_num].name[dir_len+1] = 'w';
	  nodes[node_num].name[dir_len+2] = eol;
	  node_num++;
	  strcpy(nodes[node_num].name, &(node_name[2]));
	  nodes[node_num].name[dir_len] = ' ';
	  nodes[node_num].name[dir_len+1] = 'n';
	  nodes[node_num].name[dir_len+2] = 'w';
	  nodes[node_num].name[dir_len+3] = eol;
	  node_num++;
	}
  }

  num_bottom_level_nodes = node_num - num_top_level_nodes -
	num_intermediate_nodes;
  num_nodes = node_num;

  /* initialise fields in node structures */
  for (i = (num_top_level_nodes+num_intermediate_nodes); i < num_nodes; i++) {
	nodes[i].activation = 0.0;
	nodes[i].num_in_links = 0;
	nodes[i].num_out_links = 0;
	nodes[i].type = BOTTOM_LEVEL;
  }

  fclose(f);

/*  printf("%d bottom level nodes read in\n", num_bottom_level_nodes);
  printf("%d nodes of any sort read in\n", num_nodes);
*/}


/* this function reads in the senders and receivers of connections in the
network and the strengths thereof from a file.  A sender can be a sensor
(internal or external or indeterminate or directed external) or a node,
whereas a receiver has to be a node.  The file has the format: 
<sender name>
<receiver name>
<strength>
..
[repeated]
*/
void init_links()
{

  char sender_name[50];
  char receiver_name[50];
  char dir_name[50];
  char eol;
  int dir_length;
  int sender_type;
  int receiver_type;
  int sender;
  int receiver;
  int i;
  FILE *f;
  double link_strength;

  f = fopenwithcheck(LINKS_FNAME, "r");

  num_links = 0;

  /* read in list of links */
  while (getline(f, receiver_name, MAX_LINE_LENGTH) != 0) {

	/* calculate number of receiver node */
	receiver = -1;
	for (i = 0; i < num_nodes; i++)
	  if (strcmp(receiver_name, nodes[i].name) == 0) {
		receiver = i;
		receiver_type = NODE;
/*		printf("receiver is %s (type = NODE)\n", receiver_name); 
*/	  }

	if ((receiver == -1) && (receiver_name[0] == '8')) {
	  strcpy(dir_name, &(receiver_name[2]));
	  dir_length = strlen(dir_name);
	  eol = dir_name[dir_length];
	  dir_name[dir_length] = ' ';
	  dir_name[dir_length+1] = 'n';
	  dir_name[dir_length+2] = eol;
	  for (i = 0; i < num_nodes; i++)
		if (strcmp(dir_name, nodes[i].name) == 0) {
		  receiver = i;
		  receiver_type = MULTIPLE_NODE;
/*		  printf("receiver is %s (type = MULTIPLE_NODE)\n", receiver_name); 
*/		}
	}

	if (receiver == -1) {
	  printf("**error**: receiver %s not recognised\n", receiver_name);
	  exit(1);
	}

	getline(f, sender_name, MAX_LINE_LENGTH);

	/* until end of list of inputs to this receiver */
	while (strcmp(sender_name, "END") != 0) {

	  sender = -1;
	  sender_type = -1;

	  for (i = 0; i < num_nodes; i++)

	    if (strcmp(sender_name, nodes[i].name) == 0) {
		  sender = i;
		  sender_type = NODE;
/*		  printf("sender is %s (type = NODE), ", sender_name);
*/	    }

	  /* if the name given does not match any node */
	  if (sender == -1)
	    for (i = 0; i < num_int_sensors; i++)
		  if (strcmp(sender_name, int_sensors[i]) == 0) {
		    sender = i;
		    sender_type = INTERNAL_SENSOR;
/*		    printf("sender is %s (type = INTERNAL_SENSOR), ", sender_name);
*/		  }

	  if (sender == -1)
	    for (i = 0; i < num_ext_sensors; i++)
		  if (strcmp(sender_name, ext_sensors[i]) == 0) {
		    sender = i;
		    sender_type = EXTERNAL_SENSOR;
/*		    printf("sender is %s (type = EXTERNAL_SENSOR), ", sender_name);
*/		  }

	  if (sender == -1)
	    for (i = 0; i < num_indet_sensors; i++)
		  if (strcmp(sender_name, indet_sensors[i]) == 0) {
		    sender = i;
		    sender_type = INDETERMINATE_SENSOR;
/*		    printf("sender is %s (type = INDETERMINATE_SENSOR), ",sender_name);
*/		  }

	  if (sender == -1)
	    for (i = 0; i < num_directed_ext_sensors; i++)
		  if (strcmp(sender_name, directed_ext_sensors[i]) == 0) {
		    sender = i;
		    sender_type = DIRECTED_EXTERNAL_SENSOR;
/*		    printf("sender is %s (type = DIRECTED_EXTERNAL_SENSOR), ",
			  sender_name); 
*/		  }

	  if (sender == -1) {
	    printf("**error**: sender %s not recognised\n", sender_name);
		exit(1);
	  }

	  fscanf(f, "%lf\n", &link_strength);
/*	  printf("link_strength is %g\n", link_strength);
*/
	  /* deal with ordinary connection to node first of all */
	  if (receiver_type == NODE) {
	    links[num_links].sender_number = sender;
	    links[num_links].sender_type = sender_type;
	    links[num_links].receiver_number = receiver;
	    links[num_links].receiver_type = receiver_type;
	    links[num_links].strength = link_strength;
	    nodes[sender].out_links[nodes[sender].num_out_links] = num_links;
	    nodes[receiver].in_links[nodes[receiver].num_in_links] = num_links;
	    nodes[sender].num_out_links++;
	    nodes[receiver].num_in_links++;
	    num_links++;
	  }

	  /* then deal with special cases when links are to whole sets of nodes */
	  if (receiver_type == MULTIPLE_NODE) {		/* to all move actions */
	    for (i = 0; i < 8; i++) {
		  links[num_links].sender_number = sender;
		  links[num_links].sender_type = sender_type;
		  links[num_links].receiver_number = (receiver + i);
		  links[num_links].receiver_type = receiver_type;
		  links[num_links].strength = link_strength;
		  nodes[sender].out_links[nodes[sender].num_out_links] = num_links;
		  nodes[receiver+i].in_links[nodes[receiver+i].num_in_links]
		    = num_links; 
		  nodes[sender].num_out_links++;
		  nodes[receiver+i].num_in_links++;
		  num_links++;
	    }
	  }

	  getline(f, sender_name, MAX_LINE_LENGTH);
	}
  }

  fclose(f);

/*  printf("%d links read in\n", num_links);
*/}


/* function to read in a line from a file and put it in the array of chars
whose address is provided.  The function returns the length of the line, or
0 if the end of the file has been reached */
int getline(f, line, max_length)
FILE *f;
char *line;
int max_length;
{

  char c;
  int i, scan_result;

  i = 0;
  scan_result = fscanf(f, "%c", &c);

  /* while the first character is a percentage sign (comment) then read rest
  of line (without storing) and start another */
  while ((c == '%') && (scan_result != EOF)) {
	/* read rest of line without putting in buffer */
	while ((c != '\n') && (scan_result != EOF))
	  scan_result = fscanf(f, "%c", &c);
	/* read in the first character of the next line */
	scan_result =fscanf(f, "%c", &c);
  }

  while ((i < max_length) && (c != '\n') && (scan_result != EOF)) {
	line[i] = c;
	i++;
	scan_result = fscanf(f, "%c", &c);
  }

  line[i] = '\0';

  return(i);
}


void calc_internal_sensors()
{

  int i;

  calc_food_minus();
  calc_water_minus();
  calc_an_temp_minus();
  calc_sqr_temp_minus();
  calc_an_temp_plus();
  calc_sqr_temp_plus();
  calc_dirtiness();
  calc_low_health();
  calc_temporal_discount_penalty();
  calc_uncertainty_penalty();
  calc_one_point_zero();

/*  for (i = 0; i < num_int_sensors; i++)
	printf("%s act. = %.2f\n", int_sensors[i], int_sensor_acts[i]);
*/}


void calc_external_sensors(a12, a14, a16, a17, a18, a19, a20, a21, a22, a23) 
double *a12, *a14, *a16, *a17, *a18, *a19, *a20, *a21, *a22, *a23;
{

  int i;

  calc_cf_in_sq(a12);
  calc_ff_in_sq(a12);
  calc_prey_in_sq(a12);
  calc_food_in_sq(a12);
  calc_water_in_sq(a14);
  calc_shade_in_sq(a17);
  calc_shelter_in_sq(a16);
  calc_p1_in_sq(a18);
  calc_max_p1(a18);
  calc_max_p1_adjusted(a18);
  calc_p2_in_sq(a19);
  calc_max_p2(a19);
  calc_max_p2_adjusted(a18);
  calc_dp_in_sq(a20);
  calc_irr_in_sq(a21);
  calc_receptive_mate_in_sq(a22);
  calc_courted_mate_in_sq(a22);
  calc_den_in_sq(a23);
  calc_no_den_in_sq(a23);

/*  printf("\n");
  for (i = 0; i < num_ext_sensors; i++)
	printf("%s act. = %.2f\n", ext_sensors[i], ext_sensor_acts[i]);
*/}


void calc_indeterminate_sensors()
{

  int i;

  calc_variance();
  calc_night_prox();
  calc_den_distance();
  calc_time_since_scanned();
  calc_predator_since_scanned();
  calc_distance_from_cover();

/*  printf("\n");
  for (i = 0; i < num_indet_sensors; i++)
	printf("%s act. = %.2f\n", indet_sensors[i], indet_sensor_acts[i]);
*/}


void calc_directed_external_sensors(a12, a13, a14, a15, a16, a17, a18, a19,
a20, a21, a22, a23, a24, a25, a31, a32)
double *a12, *a13, *a14, *a15, *a16, *a17, *a18, *a19, *a20, *a21, *a22,
	   *a23, *a24, *a25, *a31, *a32; 
{

  int i;

  calc_food(a12);
  calc_food_r(a13);
  calc_water(a14);
  calc_water_r(a15);
  calc_shade(a17);
  calc_shelter(a16);
  calc_dp(a20);
  calc_dp2(a31);
  calc_opp_dp(a20);
  calc_irr(a21);
  calc_irr2(a32);
  calc_opp_irr(a21);
  calc_p1(a18);
  calc_opp_p1(a18);
  calc_p1_adjusted(a18);
  calc_p2(a19);
  calc_opp_p2(a19);
  calc_p2_adjusted(a19);
  calc_receptive_mate(a22);
  calc_den(a23);
  calc_den_r(a24);
  calc_edge(a25);
  calc_explore_direction();
  calc_all_directions();
  calc_cover(a16);

/*  printf("\n");
  for (i = 0; i < num_directed_ext_sensors; i++)
	printf("%s acts. = %.2f %.2f %.2f %.2f %.2f %.2f %.2f %.2f\n",
	  directed_ext_sensors[i], 
	  directed_ext_sensor_acts[i][0], directed_ext_sensor_acts[i][1],
	  directed_ext_sensor_acts[i][2], directed_ext_sensor_acts[i][3],
	  directed_ext_sensor_acts[i][4], directed_ext_sensor_acts[i][5],
	  directed_ext_sensor_acts[i][6], directed_ext_sensor_acts[i][7]);     
*/}


/* function to spread activation through the network, down from the
top-level nodes through intermediate ones to the bottom-level nodes */
void pass_activation_through_network()
{

  int i, j;
  int l;		/* link number */
  int node_inputs_ready;

  /* zero node activations first of all */
  /* for each node */
  for (i = 0; i < num_nodes; i++)
	nodes[i].activation = 0.0;

  while (activation_spreading_finished() == FALSE) {

	/* for each node */
	for (i = 0; i < num_nodes; i++) {

	  /* if the node has not yet received all of its excitation */
	  if (nodes[i].all_act_received == FALSE) {

		/* check to see if all inputs to node are 'ready' (i.e. are either
		sensors or nodes which have received all of their activation
		already) */
		node_inputs_ready = TRUE;
		for (j = 0; j < nodes[i].num_in_links; j++) {
		  l = nodes[i].in_links[j];
		  if ((links[l].sender_type == NODE) &&
			  (nodes[links[l].sender_number].all_act_received == FALSE))
			node_inputs_ready = FALSE;
		}

		/* if all inputs to the node are ready then feed in their
		activations along the links to the node */
		if (node_inputs_ready == TRUE)

		  input_excitation_to_node(i);
	  }
	}
  }
}


int activation_spreading_finished()
{

  int i;

  /* if any node has still not received all of its activation, then
  spreading of activation process not finished */ 
  for (i = 0; i < num_nodes; i++)
	if (nodes[i].all_act_received == FALSE)
	  return(FALSE);
  return(TRUE);
}


/* select an action by choosing the bottom level node which has
"accumulated" the most activation */
int get_action()
{

  int i;
  int winning_node;
  double highest_activation = -1000000.0;

  /* find the node with the highest activation */
  for (i = (num_top_level_nodes+num_intermediate_nodes); i <
	num_nodes; i++) 

	if (nodes[i].activation > highest_activation) {
	  highest_activation = nodes[i].activation;
	  winning_node = i;
	}

/*  printf(">>>> winning node is %s, with activation %g\n",
	nodes[winning_node].name, nodes[winning_node].activation);
*/
  /* return the appropriate low-level action */
  if (strcmp("eat cf", nodes[winning_node].name) == 0)
	return(EATING_CF);
  else if (strcmp("eat ff", nodes[winning_node].name) == 0)
	return(EATING_FF);
  else if (strcmp("pounce", nodes[winning_node].name) == 0)
	return(POUNCING);
  else if (strcmp("drink (bl)", nodes[winning_node].name) == 0)
	return(DRINKING);
  else if (strcmp("rest (bl)", nodes[winning_node].name) == 0)
	return(RESTING);
  else if (strcmp("freeze", nodes[winning_node].name) == 0)
	return(FREEZING);
  else if (strcmp("court (bl)", nodes[winning_node].name) == 0)
	return(COURT_DISPLAYING);
  else if (strcmp("mate (bl)", nodes[winning_node].name) == 0)
	return(MATING);
  else if (strcmp("sleep (bl)", nodes[winning_node].name) == 0)
	return(SLEEPING);
  else if (strcmp("clean (bl)", nodes[winning_node].name) == 0)
	return(CLEANING);
  else if (strcmp("look n", nodes[winning_node].name) == 0)
	return(LOOKING_N);
  else if (strcmp("look ne", nodes[winning_node].name) == 0)
	return(LOOKING_NE);
  else if (strcmp("look e", nodes[winning_node].name) == 0)
	return(LOOKING_E);
  else if (strcmp("look se", nodes[winning_node].name) == 0)
	return(LOOKING_SE);
  else if (strcmp("look s", nodes[winning_node].name) == 0)
	return(LOOKING_S);
  else if (strcmp("look sw", nodes[winning_node].name) == 0)
	return(LOOKING_SW);
  else if (strcmp("look w", nodes[winning_node].name) == 0)
	return(LOOKING_W);
  else if (strcmp("look nw", nodes[winning_node].name) == 0)
	return(LOOKING_NW);
  else if (strcmp("move n", nodes[winning_node].name) == 0)
	return(MOVE_N);
  else if (strcmp("move ne", nodes[winning_node].name) == 0)
	return(MOVE_NE);
  else if (strcmp("move e", nodes[winning_node].name) == 0)
	return(MOVE_E);
  else if (strcmp("move se", nodes[winning_node].name) == 0)
	return(MOVE_SE);
  else if (strcmp("move s", nodes[winning_node].name) == 0)
	return(MOVE_S);
  else if (strcmp("move sw", nodes[winning_node].name) == 0)
	return(MOVE_SW);
  else if (strcmp("move w", nodes[winning_node].name) == 0)
	return(MOVE_W);
  else if (strcmp("move nw", nodes[winning_node].name) == 0)
	return(MOVE_NW);
  else if (strcmp("move fast n", nodes[winning_node].name) == 0)
	return(MOVE2_N);
  else if (strcmp("move fast ne", nodes[winning_node].name) == 0)
	return(MOVE2_NE);
  else if (strcmp("move fast e", nodes[winning_node].name) == 0)
	return(MOVE2_E);
  else if (strcmp("move fast se", nodes[winning_node].name) == 0)
	return(MOVE2_SE);
  else if (strcmp("move fast s", nodes[winning_node].name) == 0)
	return(MOVE2_S);
  else if (strcmp("move fast sw", nodes[winning_node].name) == 0)
	return(MOVE2_SW);
  else if (strcmp("move fast w", nodes[winning_node].name) == 0)
	return(MOVE2_W);
  else if (strcmp("move fast nw", nodes[winning_node].name) == 0)
	return(MOVE2_NW);
  else if (strcmp("look around (bl)", nodes[winning_node].name) == 0)
	return(LOOKING_AROUND);
  else {
	printf("error: invalid action chosen (%s)\n", nodes[winning_node].name);
	exit(1);
  }
}


void calc_food_minus()
{

  int i;
  int number = -1;

  for (i = 0; i < num_int_sensors; i++)
    if (strcmp(int_sensors[i], "food minus") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  /* food deficit some combination of need for fat, carbohydrate and
  protein */
  int_sensor_acts[number] = fat_minus;
  if ((carbo_minus > fat_minus) && (carbo_minus > protein_minus))
    int_sensor_acts[number] = carbo_minus;
  else if ((protein_minus > fat_minus) && (protein_minus > carbo_minus))
    int_sensor_acts[number] = protein_minus;
}


void calc_water_minus()
{

  int i;
  int number = -1;

  for (i = 0; i < num_int_sensors; i++)
    if (strcmp(int_sensors[i], "water minus") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  int_sensor_acts[number] = water_minus;
}


void calc_an_temp_minus()
{

  int i;
  int number = -1;
  char name[30];

  strcpy(name, "an temp minus");
  for (i = 0; i < num_int_sensors; i++)
    if (strcmp(int_sensors[i], name) == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  int_sensor_acts[number] = an_temp_minus;
}


void calc_sqr_temp_minus()
{

  int i;
  int number = -1;
  char name[30];

  strcpy(name, "sqr temp minus");
  for (i = 0; i < num_int_sensors; i++)
    if (strcmp(int_sensors[i], name) == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  int_sensor_acts[number] = sqr_temp_minus;
}


void calc_an_temp_plus()
{

  int i;
  int number = -1;

  for (i = 0; i < num_int_sensors; i++)
    if (strcmp(int_sensors[i], "an temp plus") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  int_sensor_acts[number] = an_temp_plus;
}


void calc_sqr_temp_plus()
{

  int i;
  int number = -1;

  for (i = 0; i < num_int_sensors; i++)
    if (strcmp(int_sensors[i], "sqr temp plus") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  int_sensor_acts[number] = sqr_temp_plus;
}


void calc_dirtiness()
{

  int i;
  int number = -1;

  for (i = 0; i < num_int_sensors; i++)
    if (strcmp(int_sensors[i], "dirtiness") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  int_sensor_acts[number] = (1.0 - cleanliness);
}


void calc_low_health()
{

  int i;
  int number = -1;

  for (i = 0; i < num_int_sensors; i++)
    if (strcmp(int_sensors[i], "low health") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

	int_sensor_acts[number] = 1.0 - health;
}


void calc_temporal_discount_penalty()
{

  int i;
  int number = -1;

  for (i = 0; i < num_int_sensors; i++)
    if (strcmp(int_sensors[i], "temporal discount penalty") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  int_sensor_acts[number] = 1.0;
}


void calc_uncertainty_penalty()
{

  int i;
  int number = -1;

  for (i = 0; i < num_int_sensors; i++)
    if (strcmp(int_sensors[i], "uncertainty penalty") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  int_sensor_acts[number] = 1.0;
}


void calc_one_point_zero()
{

  int i;
  int number = -1;

  for (i = 0; i < num_int_sensors; i++)
    if (strcmp(int_sensors[i], "one point zero") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  int_sensor_acts[number] = 1.0;
}


void calc_cf_in_sq(food_p_stimuli)
double *food_p_stimuli;
{

  int i;
  int number = -1;

  for (i = 0; i < num_ext_sensors; i++)
    if (strcmp(ext_sensors[i], "cf in sq") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  if (cf_in_animal_square == TRUE)
	ext_sensor_acts[number] = pow(food_p_stimuli[SAME_SQUARE], 0.2);
  else
	ext_sensor_acts[number] = 0.0;
}


void calc_ff_in_sq(food_p_stimuli)
double *food_p_stimuli;
{

  int i;
  int number = -1;

  for (i = 0; i < num_ext_sensors; i++)
    if (strcmp(ext_sensors[i], "ff in sq") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  if (ff_in_animal_square == TRUE)
	ext_sensor_acts[number] = pow(food_p_stimuli[SAME_SQUARE], 0.2);
  else
	ext_sensor_acts[number] = 0.0;
}


void calc_prey_in_sq(food_p_stimuli)
double *food_p_stimuli;
{

  int i;
  int number = -1;

  for (i = 0; i < num_ext_sensors; i++)
    if (strcmp(ext_sensors[i], "prey in sq") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  if (prey_in_animal_square == TRUE)
	ext_sensor_acts[number] = pow(food_p_stimuli[SAME_SQUARE], 0.2);
  else
	ext_sensor_acts[number] = 0.0;
}


void calc_food_in_sq(food_p_stimuli)
double *food_p_stimuli;
{

  int i;
  int number = -1;

  for (i = 0; i < num_ext_sensors; i++)
    if (strcmp(ext_sensors[i], "food in sq") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  ext_sensor_acts[number] = pow(food_p_stimuli[SAME_SQUARE], 0.2);
}


void calc_water_in_sq(water_p_stimuli)
double *water_p_stimuli;
{

  int i;
  int number = -1;

  for (i = 0; i < num_ext_sensors; i++)
    if (strcmp(ext_sensors[i], "water in sq") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  ext_sensor_acts[number] = pow(water_p_stimuli[SAME_SQUARE], 0.2);
}


void calc_shade_in_sq(shelter_p_stimuli)
double *shelter_p_stimuli;
{

  int i;
  int number = -1;

  for (i = 0; i < num_ext_sensors; i++)
    if (strcmp(ext_sensors[i], "shade in sq") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  ext_sensor_acts[number] = pow(shelter_p_stimuli[SAME_SQUARE], 0.5);
}


void calc_shelter_in_sq(shelter_p_stimuli)
double *shelter_p_stimuli;
{

  int i;
  int number = -1;

  for (i = 0; i < num_ext_sensors; i++)
    if (strcmp(ext_sensors[i], "shelter in sq") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  ext_sensor_acts[number] = pow(shelter_p_stimuli[SAME_SQUARE], 0.5);
}


void calc_dp_in_sq(dp_p_stimuli)
double *dp_p_stimuli;
{

  int i;
  int number = -1;

  for (i = 0; i < num_ext_sensors; i++)
    if (strcmp(ext_sensors[i], "dp in sq") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  ext_sensor_acts[number] = dp_p_stimuli[SAME_SQUARE];
}


void calc_irr_in_sq(irr_p_stimuli)
double *irr_p_stimuli;
{

  int i;
  int number = -1;

  for (i = 0; i < num_ext_sensors; i++)
    if (strcmp(ext_sensors[i], "irr in sq") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  ext_sensor_acts[number] = irr_p_stimuli[SAME_SQUARE];
}


void calc_p1_in_sq(p1_p_stimuli)
double *p1_p_stimuli;
{

  int i;
  int number = -1;

  for (i = 0; i < num_ext_sensors; i++)
    if (strcmp(ext_sensors[i], "p1 in sq") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  ext_sensor_acts[number] = p1_p_stimuli[SAME_SQUARE];
}


void calc_max_p1(p1_p_stimuli)
double *p1_p_stimuli;
{

  int i;
  int number = -1;

  for (i = 0; i < num_ext_sensors; i++)
    if (strcmp(ext_sensors[i], "max p1") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  ext_sensor_acts[number] = max_p1_perc_stimulus;
}


void calc_max_p1_adjusted(p1_p_stimuli)
double *p1_p_stimuli;
{

  int i;
  int number = -1;

  for (i = 0; i < num_ext_sensors; i++)
    if (strcmp(ext_sensors[i], "max p1 (adjusted)") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  if (max_p1_perc_stimulus > 0.30)
	ext_sensor_acts[number] = max_p1_perc_stimulus;
  else
	ext_sensor_acts[number] = 0.0;
}


void calc_p2_in_sq(p2_p_stimuli)
double *p2_p_stimuli;
{

  int i;
  int number = -1;

  for (i = 0; i < num_ext_sensors; i++)
    if (strcmp(ext_sensors[i], "p2 in sq") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  ext_sensor_acts[number] = p2_p_stimuli[SAME_SQUARE];
}


void calc_max_p2(p2_p_stimuli)
double *p2_p_stimuli;
{

  int i;
  int number = -1;

  for (i = 0; i < num_ext_sensors; i++)
    if (strcmp(ext_sensors[i], "max p2") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  ext_sensor_acts[number] = max_p2_perc_stimulus;
}


void calc_max_p2_adjusted(p2_p_stimuli)
double *p2_p_stimuli;
{

  int i;
  int number = -1;

  for (i = 0; i < num_ext_sensors; i++)
    if (strcmp(ext_sensors[i], "max p2 (adjusted)") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  if (max_p2_perc_stimulus > 0.30)
	ext_sensor_acts[number] = max_p2_perc_stimulus;
  else
	ext_sensor_acts[number] = 0.0;
}


void calc_receptive_mate_in_sq(mate_p_stimuli)
double *mate_p_stimuli;
{

  int i;
  int number = -1;

  for (i = 0; i < num_ext_sensors; i++)
    if (strcmp(ext_sensors[i], "receptive mate in sq") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  ext_sensor_acts[number] = mate_p_stimuli[SAME_SQUARE];
}


void calc_courted_mate_in_sq(mate_p_stimuli)
double *mate_p_stimuli;
{

  int i;
  int number = -1;

  for (i = 0; i < num_ext_sensors; i++)
    if (strcmp(ext_sensors[i], "courted mate in sq") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  ext_sensor_acts[number] = mate_p_stimuli[SAME_SQUARE];
  if (mate_courtedness < 0.5)
	ext_sensor_acts[number] *= mate_courtedness;
}


void calc_den_in_sq(den_p_stimuli)
double *den_p_stimuli;
{

  int i;
  int number = -1;

  for (i = 0; i < num_ext_sensors; i++)
    if (strcmp(ext_sensors[i], "den in sq") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  ext_sensor_acts[number] = den_p_stimuli[SAME_SQUARE];
}


void calc_no_den_in_sq(den_p_stimuli)
double *den_p_stimuli;
{

  int i;
  int number = -1;

  for (i = 0; i < num_ext_sensors; i++)
    if (strcmp(ext_sensors[i], "no den in sq") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  if (den_p_stimuli[SAME_SQUARE] > 0.2)
	ext_sensor_acts[number] = 0.0;
  else
	ext_sensor_acts[number] = 1.0;
}


void calc_variance()
{

  int i;
  int number = -1;

  for (i = 0; i < num_indet_sensors; i++)
    if (strcmp(indet_sensors[i], "variance") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  indet_sensor_acts[number] = variance;
}


void calc_night_prox()
{

  int i;
  int number = -1;

  for (i = 0; i < num_indet_sensors; i++)
    if (strcmp(indet_sensors[i], "night prox") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  indet_sensor_acts[number] = pow(night_prox, 10.0);
}


void calc_den_distance()
{

  int i;
  int number = -1;

  for (i = 0; i < num_indet_sensors; i++)
    if (strcmp(indet_sensors[i], "distance from den") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  indet_sensor_acts[number] = den_distance;
}


void calc_time_since_scanned()
{

  int i;
  int number = -1;

  for (i = 0; i < num_indet_sensors; i++)
    if (strcmp(indet_sensors[i], "time since scanned") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  indet_sensor_acts[number] = scan_time;
}


void calc_predator_since_scanned()
{

  int i;
  int number = -1;

  for (i = 0; i < num_indet_sensors; i++)
    if (strcmp(indet_sensors[i], "predator since scanned") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  if ((time_since_predator < scan_time) && (time_since_predator < 0.4))
    indet_sensor_acts[number] = 1.0;
  else if ((time_since_predator < scan_time) && (time_since_predator < 0.8))
    indet_sensor_acts[number] = 0.5;
  else
	indet_sensor_acts[number] = 0.0;
}


void calc_distance_from_cover()
{

  int i;
  int number = -1;

  for (i = 0; i < num_indet_sensors; i++)
    if (strcmp(indet_sensors[i], "distance from cover") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  indet_sensor_acts[number] = pow(cover_distance, 0.70);
}


void calc_food(food_p_stimuli)
double *food_p_stimuli;
{

  int i;
  int number = -1;

  for (i = 0; i < num_directed_ext_sensors; i++)
    if (strcmp(directed_ext_sensors[i], "food") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  for (i = 0; i < 8; i++)
	directed_ext_sensor_acts[number][i] = pow(food_p_stimuli[i+1], 0.20);
}


void calc_food_r(food_r_stimuli)
double *food_r_stimuli;
{

  int i;
  int left1, right1;
  int number = -1;

  for (i = 0; i < num_directed_ext_sensors; i++)
    if (strcmp(directed_ext_sensors[i], "food r") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  for (i = 0; i < 8; i++) {
	left1 = ((i+7) % 8);
	right1 = ((i+1) % 8);
	directed_ext_sensor_acts[number][i] = pow(food_r_stimuli[i+1], 0.20) +
								   0.75 * pow(food_r_stimuli[left1+1], 0.20) +
								   0.75 * pow(food_r_stimuli[right1+1], 0.20);
  }
}


void calc_water(water_p_stimuli)
double *water_p_stimuli;
{

  int i;
  int number = -1;

  for (i = 0; i < num_directed_ext_sensors; i++)
    if (strcmp(directed_ext_sensors[i], "water") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  for (i = 0; i < 8; i++)
	directed_ext_sensor_acts[number][i] = pow(water_p_stimuli[i+1], 0.20);
}


void calc_water_r(water_r_stimuli)
double *water_r_stimuli;
{

  int i;
  int left1, right1;
  int number = -1;

  for (i = 0; i < num_directed_ext_sensors; i++)
    if (strcmp(directed_ext_sensors[i], "water r") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  for (i = 0; i < 8; i++) {
	left1 = ((i+7) % 8);
	right1 = ((i+1) % 8);
	directed_ext_sensor_acts[number][i] = pow(water_r_stimuli[i+1], 0.20) +
								   0.75 * pow(water_r_stimuli[left1+1], 0.20) +
								   0.75 * pow(water_r_stimuli[right1+1], 0.20);
  }
}


void calc_shade(shade_p_stimuli)
double *shade_p_stimuli;
{

  int i;
  int number = -1;

  for (i = 0; i < num_directed_ext_sensors; i++)
    if (strcmp(directed_ext_sensors[i], "shade") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  for (i = 0; i < 8; i++)
	directed_ext_sensor_acts[number][i] = sqrt(shade_p_stimuli[i+1]);
}


void calc_shelter(shelter_p_stimuli)
double *shelter_p_stimuli;
{

  int i;
  int number = -1;

  for (i = 0; i < num_directed_ext_sensors; i++)
    if (strcmp(directed_ext_sensors[i], "shelter") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  for (i = 0; i < 8; i++)
	directed_ext_sensor_acts[number][i] = sqrt(shelter_p_stimuli[i+1]);
}


void calc_dp(dp_p_stimuli)
double *dp_p_stimuli;
{

  int i;
  int number = -1;

  for (i = 0; i < num_directed_ext_sensors; i++)
    if (strcmp(directed_ext_sensors[i], "dp") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  for (i = 0; i < 8; i++)
	directed_ext_sensor_acts[number][i] = dp_p_stimuli[i+1];
}


void calc_dp2(dp_p_stimuli2)
double *dp_p_stimuli2;
{

  int i;
  int number = -1;

  for (i = 0; i < num_directed_ext_sensors; i++)
    if (strcmp(directed_ext_sensors[i], "dp2") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  for (i = 0; i < 8; i++)
	directed_ext_sensor_acts[number][i] = dp_p_stimuli2[i+1];
}


void calc_opp_dp(dp_p_stimuli)
double *dp_p_stimuli;
{

  int i;
  int number = -1;
  int opp_dir, opp_dir_l, opp_dir_r, opp_dir_l2, opp_dir_r2;

  for (i = 0; i < num_directed_ext_sensors; i++)
    if (strcmp(directed_ext_sensors[i], "opp dp") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  for (i = 0; i < 8; i++) {
	opp_dir = ((i+4) % 8);
	opp_dir_r = ((i+5) % 8);
	opp_dir_l = ((i+3) % 8);
	opp_dir_r2 = ((i+6) % 8);
	opp_dir_l2 = ((i+2) % 8);
	directed_ext_sensor_acts[number][i] = (1.0 * dp_p_stimuli[opp_dir+1]) +
										  (1.0 * dp_p_stimuli[opp_dir_l+1]) +
										  (1.0 * dp_p_stimuli[opp_dir_r+1]) +
										  (0.5 * dp_p_stimuli[opp_dir_l2+1]) +
										  (0.5 * dp_p_stimuli[opp_dir_r2+1]);
  }
}


void calc_irr(irr_p_stimuli)
double *irr_p_stimuli;
{

  int i;
  int number = -1;

  for (i = 0; i < num_directed_ext_sensors; i++)
    if (strcmp(directed_ext_sensors[i], "irr") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  for (i = 0; i < 8; i++)
	directed_ext_sensor_acts[number][i] = irr_p_stimuli[i+1];
}


void calc_irr2(irr_p_stimuli2)
double *irr_p_stimuli2;
{

  int i;
  int number = -1;

  for (i = 0; i < num_directed_ext_sensors; i++)
    if (strcmp(directed_ext_sensors[i], "irr2") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  for (i = 0; i < 8; i++)
	directed_ext_sensor_acts[number][i] = irr_p_stimuli2[i+1];
}


void calc_opp_irr(irr_p_stimuli)
double *irr_p_stimuli;
{

  int i;
  int number = -1;
  int opp_dir, opp_dir_l, opp_dir_r, opp_dir_l2, opp_dir_r2;

  for (i = 0; i < num_directed_ext_sensors; i++)
    if (strcmp(directed_ext_sensors[i], "opp irr") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  for (i = 0; i < 8; i++) {
	opp_dir = ((i+4) % 8);
	opp_dir_r = ((i+5) % 8);
	opp_dir_l = ((i+3) % 8);
	opp_dir_r2 = ((i+6) % 8);
	opp_dir_l2 = ((i+2) % 8);
	directed_ext_sensor_acts[number][i] = (1.0 * irr_p_stimuli[opp_dir+1]) +
										  (1.0 * irr_p_stimuli[opp_dir_l+1]) +
										  (1.0 * irr_p_stimuli[opp_dir_r+1]) +
										  (0.5 * irr_p_stimuli[opp_dir_l2+1]) +
										  (0.5 * irr_p_stimuli[opp_dir_r2+1]);
  }
}


void calc_p1(p1_p_stimuli)
double *p1_p_stimuli;
{

  int i;
  int left1, right1;
  int number = -1;

  for (i = 0; i < num_directed_ext_sensors; i++)
    if (strcmp(directed_ext_sensors[i], "p1") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  for (i = 0; i < 8; i++) {
	left1 = ((i+7)%8);
	right1 = ((i+1)%8);
	directed_ext_sensor_acts[number][i] = (1.0 * p1_p_stimuli[i+1]) +
										  (0.5 * p1_p_stimuli[left1+1]) +
										  (0.5 * p1_p_stimuli[right1+1]);
  }
}


void calc_opp_p1(p1_p_stimuli)
double *p1_p_stimuli;
{

  int i;
  int number = -1;
  int opp_dir, opp_dir_l, opp_dir_r, opp_dir_l2, opp_dir_r2;

  for (i = 0; i < num_directed_ext_sensors; i++)
    if (strcmp(directed_ext_sensors[i], "opp p1") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  for (i = 0; i < 8; i++) {
	opp_dir = ((i+4) % 8);
	opp_dir_r = ((i+5) % 8);
	opp_dir_l = ((i+3) % 8);
	opp_dir_r2 = ((i+6) % 8);
	opp_dir_l2 = ((i+2) % 8);
	directed_ext_sensor_acts[number][i] = (1.0 * p1_p_stimuli[opp_dir+1]) +
										  (1.0 * p1_p_stimuli[opp_dir_l+1]) +
										  (1.0 * p1_p_stimuli[opp_dir_r+1]) +
										  (0.5 * p1_p_stimuli[opp_dir_l2+1]) +
										  (0.5 * p1_p_stimuli[opp_dir_r2+1]);
  }
}


void calc_p1_adjusted(p1_p_stimuli)
double *p1_p_stimuli;
{

  int i;
  int number = -1;

  for (i = 0; i < num_directed_ext_sensors; i++)
    if (strcmp(directed_ext_sensors[i], "p1 (adjusted)") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  for (i = 0; i < 8; i++)
	if (p1_p_stimuli[i+1] > 0.2)
	  directed_ext_sensor_acts[number][i] = 1.0;
	else
	  directed_ext_sensor_acts[number][i] = p1_p_stimuli[i+1] / 0.2;
}


void calc_p2(p2_p_stimuli)
double *p2_p_stimuli;
{

  int i;
  int left1, right1;
  int number = -1;

  for (i = 0; i < num_directed_ext_sensors; i++)
    if (strcmp(directed_ext_sensors[i], "p2") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  for (i = 0; i < 8; i++) {
	left1 = ((i+7)%8);
	right1 = ((i+1)%8);
	directed_ext_sensor_acts[number][i] = (1.0 * p2_p_stimuli[i+1]) +
										  (0.5 * p2_p_stimuli[left1+1]) +
										  (0.5 * p2_p_stimuli[right1+1]);
  }
}


void calc_opp_p2(p2_p_stimuli)
double *p2_p_stimuli;
{

  int i;
  int number = -1;
  int opp_dir, opp_dir_l, opp_dir_r, opp_dir_l2, opp_dir_r2;

  for (i = 0; i < num_directed_ext_sensors; i++)
    if (strcmp(directed_ext_sensors[i], "opp p2") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  for (i = 0; i < 8; i++) {
	opp_dir = ((i+4) % 8);
	opp_dir_r = ((i+5) % 8);
	opp_dir_l = ((i+3) % 8);
	opp_dir_r2 = ((i+6) % 8);
	opp_dir_l2 = ((i+2) % 8);
	directed_ext_sensor_acts[number][i] = (1.0 * p2_p_stimuli[opp_dir+1]) +
										  (1.0 * p2_p_stimuli[opp_dir_l+1]) +
										  (1.0 * p2_p_stimuli[opp_dir_r+1]) +
										  (0.5 * p2_p_stimuli[opp_dir_l2+1]) +
										  (0.5 * p2_p_stimuli[opp_dir_r2+1]);
  }
}


void calc_p2_adjusted(p2_p_stimuli)
double *p2_p_stimuli;
{

  int i;
  int number = -1;

  for (i = 0; i < num_directed_ext_sensors; i++)
    if (strcmp(directed_ext_sensors[i], "p2 (adjusted)") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  for (i = 0; i < 8; i++)
	if (p2_p_stimuli[i+1] > 0.2)
	  directed_ext_sensor_acts[number][i] = 1.0;
	else
	  directed_ext_sensor_acts[number][i] = p2_p_stimuli[i+1] / 0.2;
}


void calc_receptive_mate(mate_p_stimuli)
double *mate_p_stimuli;
{

  int i;
  int number = -1;

  for (i = 0; i < num_directed_ext_sensors; i++)
    if (strcmp(directed_ext_sensors[i], "receptive mate") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  for (i = 0; i < 8; i++)
	directed_ext_sensor_acts[number][i] = pow(mate_p_stimuli[i+1], 0.20);;
}


void calc_den(den_p_stimuli)
double *den_p_stimuli;
{

  int i;
  int number = -1;

  for (i = 0; i < num_directed_ext_sensors; i++)
    if (strcmp(directed_ext_sensors[i], "den") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  for (i = 0; i < 8; i++)
	directed_ext_sensor_acts[number][i] = den_p_stimuli[i+1];
}


void calc_den_r(den_r_stimuli)
double *den_r_stimuli;
{

  int i;
  int left1, right1;
  int number = -1;

  for (i = 0; i < num_directed_ext_sensors; i++)
    if (strcmp(directed_ext_sensors[i], "den r") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  for (i = 0; i < 8; i++) {
	left1 = ((i+7) % 8);
	right1 = ((i+1) % 8);
	directed_ext_sensor_acts[number][i] = pow(den_r_stimuli[i+1], 0.20) +
								   0.75 * pow(den_r_stimuli[left1+1], 0.20) +
								   0.75 * pow(den_r_stimuli[right1+1], 0.20);
  }
}


void calc_edge(edge_p_stimuli)
double *edge_p_stimuli;
{

  int i;
  int number = -1;

  for (i = 0; i < num_directed_ext_sensors; i++)
    if (strcmp(directed_ext_sensors[i], "edge") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  for (i = 0; i < 8; i++)
	directed_ext_sensor_acts[number][i] = edge_p_stimuli[i+1];
}


void calc_explore_direction()
{

  static int counter;
  static int base_direction;
  int direction;
  int d_left, d_right;
  int i;
  int number = -1;
  double r;

  for (i = 0; i < num_directed_ext_sensors; i++)
    if (strcmp(directed_ext_sensors[i], "explore direction") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  /* choose random direction to head in which is likely to be similar to the
  last random direction */
  if (counter == 0)
	base_direction = ((int) rnd(8.0));
  if (counter > 4)
	counter = 0;
  counter++;
  r = rnd(1.0);
  if (r < 0.10)
	direction = ((base_direction+2) % 8);
  else if (r < 0.20)
	direction = ((base_direction+6) % 8);
  else if (r < 0.40)
	direction = ((base_direction+1) % 8);
  else if (r < 0.60)
	direction = ((base_direction+7) % 8);
  else 
	direction = base_direction;

  /* calculate which directions are neighbouring to the chosen one */
  d_left = (direction+7) % 8;
  d_right = (direction+1) % 8;

  /* all directions = 0.0 except randomly chosen one (1.0) and two adjacent
  ones (0.5) */ 
  for (i = 0; i < 8; i++)
	directed_ext_sensor_acts[number][i] = 0.0;
  directed_ext_sensor_acts[number][direction] = 1.0;
  directed_ext_sensor_acts[number][d_left] = 0.5;
  directed_ext_sensor_acts[number][d_right] = 0.5;

/*  if ((direction+1) == N)
	printf("                     explore direction =   N\n");
  else if ((direction+1) == NE)
	printf("                     explore direction =   NE\n");
  else if ((direction+1) == E)
	printf("                     explore direction =   E\n");
  else if ((direction+1) == SE)
	printf("                     explore direction =   SE\n");
  else if ((direction+1) == S)
	printf("                     explore direction =   S\n");
  else if ((direction+1) == SW)
	printf("                     explore direction =   SW\n");
  else if ((direction+1) == W)
	printf("                     explore direction =   W\n");
  else if ((direction+1) == NW)
	printf("                     explore direction =   NW\n");
  else
	printf("                     explore direction =   ** ERROR ** (%d)\n",
	  direction);
*/}


void calc_all_directions()
{

  int i;
  int number = -1;

  for (i = 0; i < num_directed_ext_sensors; i++)
    if (strcmp(directed_ext_sensors[i], "all directions") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  for (i = 0; i < 8; i++)
    directed_ext_sensor_acts[number][i] = 1.0;
}


void calc_cover(shelter_p_stimuli)
double *shelter_p_stimuli;
{

  int i;
  int number = -1;
  int left, right;

  for (i = 0; i < num_directed_ext_sensors; i++)
    if (strcmp(directed_ext_sensors[i], "cover") == 0)
      number = i;

  if (number == -1) {
	printf("error in sensor name\n");
	exit(1);
  }

  for (i = 0; i < 8; i++) {
	left = ((i+7)%8);
	right = ((i+1)%8);
	directed_ext_sensor_acts[number][i] = pow(shelter_p_stimuli[i+1], 0.33)
	    + (0.65 * pow(shelter_p_stimuli[left+1], 0.33))
		+ (0.65 * pow(shelter_p_stimuli[right+1], 0.33));
  }
}


/* function to look through the stimuli strengths for each type of stimulus
in each direction, and pick out the maximums of each type (over all
directions).  The direction of the maximum in each case is also calculated */
void get_maxs_and_dirs(food_perc_stimulus, food_memory_stimulus,
water_perc_stimulus, water_memory_stimulus, shelter_perc_stimulus,
shade_perc_stimulus, p1_perc_stimulus, p2_perc_stimulus, dp_perc_stimulus,
irr_perc_stimulus, mate_perc_stimulus, den_perc_stimulus,
den_memory_stimulus, edge_perc_stimulus)
double *food_perc_stimulus;
double *food_memory_stimulus;
double *water_perc_stimulus;
double *water_memory_stimulus;
double *shelter_perc_stimulus;
double *shade_perc_stimulus;
double *p1_perc_stimulus;
double *p2_perc_stimulus;
double *dp_perc_stimulus;
double *irr_perc_stimulus;
double *mate_perc_stimulus;
double *den_perc_stimulus;
double *den_memory_stimulus;
double *edge_perc_stimulus;
{

  int i;

  max_food_perc_stimulus = 0.00001;
  max_food_memory_stimulus = 0.00001;
  max_water_perc_stimulus = 0.00001;
  max_water_memory_stimulus = 0.00001;
  max_shelter_perc_stimulus = 0.00001;
  max_shade_perc_stimulus = 0.00001;
  max_p1_perc_stimulus = 0.00001;
  max_p2_perc_stimulus = 0.00001;
  max_dp_perc_stimulus = 0.00001;
  max_irr_perc_stimulus = 0.00001;
  max_mate_perc_stimulus = 0.00001;
  max_den_perc_stimulus = 0.00001;
  max_den_memory_stimulus = 0.00001;
  max_edge_perc_stimulus = 0.00001;

  for (i = 0; i < NUM_DIRECTIONS; i++) {

    if (food_perc_stimulus[i] > max_food_perc_stimulus)
      max_food_perc_stimulus = food_perc_stimulus[i];
    if (food_memory_stimulus[i] > max_food_memory_stimulus)
      max_food_memory_stimulus = food_memory_stimulus[i];
    if (water_perc_stimulus[i] > max_water_perc_stimulus)
      max_water_perc_stimulus = water_perc_stimulus[i];
    if (water_memory_stimulus[i] > max_water_memory_stimulus)
      max_water_memory_stimulus = water_memory_stimulus[i];
    if (shelter_perc_stimulus[i] > max_shelter_perc_stimulus)
      max_shelter_perc_stimulus = shelter_perc_stimulus[i];
    if (shade_perc_stimulus[i] > max_shade_perc_stimulus)
      max_shade_perc_stimulus = shade_perc_stimulus[i];
    if (p1_perc_stimulus[i] > max_p1_perc_stimulus)
      max_p1_perc_stimulus = p1_perc_stimulus[i];
    if (p2_perc_stimulus[i] > max_p2_perc_stimulus)
      max_p2_perc_stimulus = p2_perc_stimulus[i];
    if (dp_perc_stimulus[i] > max_dp_perc_stimulus)
      max_dp_perc_stimulus = dp_perc_stimulus[i];
    if (irr_perc_stimulus[i] > max_irr_perc_stimulus)
      max_irr_perc_stimulus = irr_perc_stimulus[i];
    if (mate_perc_stimulus[i] > max_mate_perc_stimulus)
      max_mate_perc_stimulus = mate_perc_stimulus[i];
    if (den_perc_stimulus[i] > max_den_perc_stimulus)
      max_den_perc_stimulus = den_perc_stimulus[i];
    if (den_memory_stimulus[i] > max_den_memory_stimulus)
      max_den_memory_stimulus = den_memory_stimulus[i];
    if (edge_perc_stimulus[i] > max_edge_perc_stimulus)
      max_edge_perc_stimulus = edge_perc_stimulus[i];
  }

  /* make very small stimuli equal to zero */
  if (max_food_perc_stimulus <= 0.00002)
    max_food_perc_stimulus = 0.0;
  if (max_food_memory_stimulus <= 0.00002)
    max_food_memory_stimulus = 0.0;
  if (max_water_perc_stimulus <= 0.00002)
    max_water_perc_stimulus = 0.0;
  if (max_water_memory_stimulus <= 0.00002)
    max_water_memory_stimulus = 0.0;
  if (max_shelter_perc_stimulus <= 0.00002)
    max_shelter_perc_stimulus = 0.0;
  if (max_shade_perc_stimulus <= 0.00002)
    max_shade_perc_stimulus = 0.0;
  if (max_p1_perc_stimulus <= 0.00002)
    max_p1_perc_stimulus = 0.0;
  if (max_p2_perc_stimulus <= 0.00002)
    max_p2_perc_stimulus = 0.0;
  if (max_dp_perc_stimulus <= 0.00002)
    max_dp_perc_stimulus = 0.0;
  if (max_irr_perc_stimulus <= 0.00002)
    max_irr_perc_stimulus = 0.0;
  if (max_mate_perc_stimulus <= 0.00002)
    max_mate_perc_stimulus = 0.0;
  if (max_den_perc_stimulus <= 0.00002)
    max_den_perc_stimulus = 0.0;
  if (max_den_memory_stimulus <= 0.00002)
    max_den_memory_stimulus = 0.0;
  if (max_edge_perc_stimulus <= 0.00002)
    max_edge_perc_stimulus = 0.0;
}


void add_directed_external_sensor_input(receiver_num, link_strength,
d_e_s_num) 
int receiver_num;
double link_strength;
int d_e_s_num;
{

  int dir;

  if ((strcmp(nodes[receiver_num].name, "move n") == 0) ||
	  (strcmp(nodes[receiver_num].name, "look n") == 0) ||
	  (strcmp(nodes[receiver_num].name, "move fast n") == 0))
	dir = N;
  else if ((strcmp(nodes[receiver_num].name, "move ne") == 0) ||
	  (strcmp(nodes[receiver_num].name, "look ne") == 0) ||
	  (strcmp(nodes[receiver_num].name, "move fast ne") == 0))
	dir = NE;
  else if ((strcmp(nodes[receiver_num].name, "move e") == 0) ||
	  (strcmp(nodes[receiver_num].name, "look e") == 0) ||
	  (strcmp(nodes[receiver_num].name, "move fast e") == 0))
	dir = E;
  else if ((strcmp(nodes[receiver_num].name, "move se") == 0) ||
	  (strcmp(nodes[receiver_num].name, "look se") == 0) ||
	  (strcmp(nodes[receiver_num].name, "move fast se") == 0))
	dir = SE;
  else if ((strcmp(nodes[receiver_num].name, "move s") == 0) ||
	  (strcmp(nodes[receiver_num].name, "look s") == 0) ||
	  (strcmp(nodes[receiver_num].name, "move fast s") == 0))
	dir = S;
  else if ((strcmp(nodes[receiver_num].name, "move sw") == 0) ||
	  (strcmp(nodes[receiver_num].name, "look nsw") == 0) ||
	  (strcmp(nodes[receiver_num].name, "move fast SW") == 0))
	dir = SW;
  else if ((strcmp(nodes[receiver_num].name, "move w") == 0) ||
	  (strcmp(nodes[receiver_num].name, "look w") == 0) ||
	  (strcmp(nodes[receiver_num].name, "move fast e") == 0))
	dir = W;
  else if ((strcmp(nodes[receiver_num].name, "move nw") == 0) ||
	  (strcmp(nodes[receiver_num].name, "look nw") == 0) ||
	  (strcmp(nodes[receiver_num].name, "move fast nw") == 0))
	dir = NW;

  nodes[receiver_num].activation +=
	(directed_ext_sensor_acts[d_e_s_num][(dir-1)] * link_strength);
}


void input_excitation_to_node(n)
int n;			/* number of node */
{

  int i, j;
  int l;		/* link number */
  int node_type;

  node_type = NODE;

  /* initialise node input values */
  for (i = 0; i < MAX_NUM_IN_LINKS; i++) {
	is[i] = 0.0;
	for (j = 0; j < 8; j++)
	  m_is[i][j] = 0.0;
  }

  for (i = 0; i < nodes[n].num_in_links; i++) {

	l = nodes[n].in_links[i];

	/* if the link is going to a normal node */
	if (links[l].receiver_type == NODE) {

/*	  printf("%s node activation = %g, ", nodes[n].name,
		nodes[n].activation); 
*/
	  if (links[l].sender_type == NODE)
		is[i] = (nodes[links[l].sender_number].activation *
		  links[l].strength);

	  else if (links[l].sender_type == INTERNAL_SENSOR)
		is[i] = (int_sensor_acts[links[l].sender_number] *
		  links[l].strength);

	  else if (links[l].sender_type == EXTERNAL_SENSOR)
		is[i] = (ext_sensor_acts[links[l].sender_number] *
		  links[l].strength);

	  else if (links[l].sender_type == INDETERMINATE_SENSOR)
		is[i] = (indet_sensor_acts[links[l].sender_number] *
		  links[l].strength);

	  else
		printf("link %d sender of unknown or wrong type (%d)\n", l,
		  links[l].sender_type);

/*	  printf("new = %g\n", is[i]);
*/	}

	/* if the link is going to a multiple node */
	else if (links[l].receiver_type == MULTIPLE_NODE) {

	  node_type = MULTIPLE_NODE;

/*	  printf("%s, etc node activations = %g %g %g %g %g %g %g %g  ",
		nodes[n].name, nodes[n+0].activation, nodes[n+1].activation,
		nodes[n+2].activation, nodes[n+3].activation, nodes[n+4].activation,
		nodes[n+5].activation, nodes[n+6].activation,
		nodes[n+7].activation);
*/
	  for (j = 0; j < 8; j++) {

		if (links[l].sender_type == NODE)
		  m_is[i][j] = (nodes[links[l].sender_number].activation
			* links[l].strength);

		else if (links[l].sender_type == DIRECTED_EXTERNAL_SENSOR)
		  m_is[i][j] =
			(directed_ext_sensor_acts[links[l].sender_number][j] *
			links[l].strength);

		else if (links[l].sender_type == INDETERMINATE_SENSOR)
		  m_is[i][j] = (indet_sensor_acts[links[l].sender_number]
			* links[l].strength);

		else if (links[l].sender_type == EXTERNAL_SENSOR)
		  m_is[i][j] = (ext_sensor_acts[links[l].sender_number]
			* links[l].strength);

		else if (links[l].sender_type == INTERNAL_SENSOR)
		  m_is[i][j] = (int_sensor_acts[links[l].sender_number]
			* links[l].strength);

		else {
		  printf("link %d sender of unknown or wrong type (%d)\n", l,
		    links[l].sender_type);
		  exit(1);
		}
	  }

/*	  printf("new = %g %g %g %g %g %g %g %g\n", nodes[n+0].activation,
		nodes[n+1].activation, nodes[n+2].activation, nodes[n+3].activation,
		nodes[n+4].activation, nodes[n+5].activation, nodes[n+6].activation,
		nodes[n+7].activation); 
*/	}

	else {
	  printf("receiver of connection not a node or multiple node\n");
	  exit(1);
	}

/*	if (links[l].sender_type == NODE)
	  printf("link processed from %s (NODE) ",
		nodes[links[l].sender_number].name); 
	else if (links[l].sender_type == INTERNAL_SENSOR)
	  printf("link processed from %s (INTERNAL SENSOR) ",
		int_sensors[links[l].sender_number]); 
	else if (links[l].sender_type == EXTERNAL_SENSOR)
	  printf("link processed from %s (EXTERNAL SENSOR) ",
		ext_sensors[links[l].sender_number]);
	else if (links[l].sender_type == INDETERMINATE_SENSOR)
	  printf("link processed from %s (INDETERMINATE SENSOR) ",
		indet_sensors[links[l].sender_number]);
	else if (links[l].sender_type == DIRECTED_EXTERNAL_SENSOR)
	  printf("link processed from %s (DIRECTED_EXTERNAL SENSOR) ",
		directed_ext_sensors[links[l].sender_number]);

	if (links[l].receiver_type == NODE)
	  printf(" to %s (NODE).\n", nodes[n].name);
	else if (links[l].receiver_type == MULTIPLE_NODE)
	  printf(" to %s (MULTIPLE NODE).\n", nodes[n].name);
*/  }

  /* now that sizes of inputs have been calculated, decide how to combine
  them */
  combine_inputs(n, node_type);

  if (node_type == NODE)
	nodes[n].all_act_received = TRUE;
  else if (node_type == MULTIPLE_NODE)
	for (j = 0; j < 8; j++)
	  nodes[n+j].all_act_received = TRUE;
}


/*g*//* function to update the slider values on the strategy display */
/*g*/void update_strategy_display()
/*g*/{
/*g*/
/*g*/  update_long_profile();
/*g*/  update_long_profile2();
/*g*/}
/*g*/
/*g*/
/*g*//* function to initialise a display for longitudinal profiles for each */
/*g*//* sub-problem, showing motivational levels and occurrences of relevant */
/*g*//* actions */
/*g*/void set_up_long_profile()
/*g*/{
/*g*/
/*g*/  int i;
/*g*/
/*g*/  lpd_position = 0;
/*g*/
/*g*/  /* define a font for the display */
/*g*/  screen_r_7 = pf_open("/usr/lib/fonts/fixedwidthfonts/screen.r.7");
/*g*/
/*g*/  long_profile_frame = window_create(NULL, FRAME,
/*g*/    WIN_SHOW, FALSE,
/*g*/    FRAME_LABEL, "Longitudinal Profiles (values scaled to fit in graphs)",
/*g*/    WIN_X, 400,
/*g*/    WIN_Y, 0,
/*g*/    WIN_WIDTH, LDP_WIDTH+10+EXTRA,
/*g*/    WIN_HEIGHT, ((NUM_SYSTEMS_FOR_LPD+1)/2*(LPD_ROW_HEIGHT+2)+2+25),
/*g*/    0);
/*g*/
/*g*/  long_profile_canvas = window_create(long_profile_frame, CANVAS,
/*g*/    WIN_SHOW, TRUE,
/*g*/    CANVAS_AUTO_SHRINK, TRUE,
/*g*/    CANVAS_AUTO_EXPAND, TRUE,
/*g*/    WIN_X, 0,
/*g*/    WIN_Y, 3,
/*g*/    WIN_WIDTH, LDP_WIDTH+6+EXTRA,
/*g*/    WIN_HEIGHT, ((NUM_SYSTEMS_FOR_LPD+1)/2*(LPD_ROW_HEIGHT+2)+2),
/*g*/    0);
/*g*/
/*g*/  long_profile_pw = canvas_pixwin(long_profile_canvas);
/*g*/
/*g*/  /* attach colourmap to pixwin */
/*g*/  pw_setcmsname(long_profile_pw, "doesntmatter");
/*g*/  pw_putcolormap(long_profile_pw, 0, CMAP_SIZE, red, green, blue);
/*g*/
/*g*/  /* draw in blank graphs for each system */
/*g*/  for (i = 0; i < NUM_SYSTEMS_FOR_LPD; i++) {
/*g*/
/*g*/	if (i <= 17) {
/*g*/	  pw_vector(long_profile_pw, 3, (2+i*(2+LPD_ROW_HEIGHT)), (LDP_WIDTH-3),
/*g*/		(2+i*(2+LPD_ROW_HEIGHT)), (PIX_SRC|PIX_COLOR(40)), 0);
/*g*/	  pw_vector(long_profile_pw, 3, (2+i*(2+LPD_ROW_HEIGHT)), 3, 
/*g*/		(2+LPD_ROW_HEIGHT+i*(2+LPD_ROW_HEIGHT)), (PIX_SRC|PIX_COLOR(40)), 0);
/*g*/	  pw_vector(long_profile_pw, 3, (2+LPD_ROW_HEIGHT+i*(2+LPD_ROW_HEIGHT)),
/*g*/		(LDP_WIDTH-3), (2+LPD_ROW_HEIGHT+i*(2+LPD_ROW_HEIGHT)),
/*g*/		(PIX_SRC|PIX_COLOR(40)), 0); 
/*g*/	  pw_vector(long_profile_pw, (LDP_WIDTH-3), (2+i*(2+LPD_ROW_HEIGHT)),
/*g*/		(LDP_WIDTH-3), (2+LPD_ROW_HEIGHT+i*(2+LPD_ROW_HEIGHT)),
/*g*/		(PIX_SRC|PIX_COLOR(40)), 0);
/*g*/	  pw_vector(long_profile_pw, 3, (2+i*(2+LPD_ROW_HEIGHT)+LPD_GRAPH_HEIGHT),
/*g*/		(LDP_WIDTH-3), (2+i*(2+LPD_ROW_HEIGHT)+LPD_GRAPH_HEIGHT),
/*g*/		(PIX_SRC|PIX_COLOR(40)), 0);
/*g*/	}
/*g*/	else {
/*g*/	  pw_vector(long_profile_pw, 3+EXTRA, (2+(i-18)*(2+LPD_ROW_HEIGHT)), (LDP_WIDTH-3+EXTRA),
/*g*/		(2+(i-18)*(2+LPD_ROW_HEIGHT)), (PIX_SRC|PIX_COLOR(40)), 0);
/*g*/	  pw_vector(long_profile_pw, 3+EXTRA, (2+(i-18)*(2+LPD_ROW_HEIGHT)), 3+EXTRA, 
/*g*/		(2+LPD_ROW_HEIGHT+(i-18)*(2+LPD_ROW_HEIGHT)), (PIX_SRC|PIX_COLOR(40)), 0);
/*g*/	  pw_vector(long_profile_pw, 3+EXTRA, (2+LPD_ROW_HEIGHT+(i-18)*(2+LPD_ROW_HEIGHT)),
/*g*/		(LDP_WIDTH-3+EXTRA), (2+LPD_ROW_HEIGHT+(i-18)*(2+LPD_ROW_HEIGHT)),
/*g*/		(PIX_SRC|PIX_COLOR(40)), 0); 
/*g*/	  pw_vector(long_profile_pw, (LDP_WIDTH-3+EXTRA), (2+(i-18)*(2+LPD_ROW_HEIGHT)),
/*g*/		(LDP_WIDTH-3+EXTRA), (2+LPD_ROW_HEIGHT+(i-18)*(2+LPD_ROW_HEIGHT)),
/*g*/		(PIX_SRC|PIX_COLOR(40)), 0);
/*g*/	  pw_vector(long_profile_pw, 3+EXTRA, (2+(i-18)*(2+LPD_ROW_HEIGHT)+LPD_GRAPH_HEIGHT),
/*g*/		(LDP_WIDTH-3+EXTRA), (2+(i-18)*(2+LPD_ROW_HEIGHT)+LPD_GRAPH_HEIGHT),
/*g*/		(PIX_SRC|PIX_COLOR(40)), 0);
/*g*/	}
/*g*/  }
/*g*/
/*g*/  /* write in the names of each profile */
/*g*/  pw_text(long_profile_pw, 5, (2+LPD_ROW_HEIGHT+0*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "sleep");
/*g*/  pw_text(long_profile_pw, 5, (2+LPD_ROW_HEIGHT+1*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "rest");
/*g*/  pw_text(long_profile_pw, 5, (2+LPD_ROW_HEIGHT+2*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "freeze");
/*g*/  pw_text(long_profile_pw, 5, (2+LPD_ROW_HEIGHT+3*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "eat cf");
/*g*/  pw_text(long_profile_pw, 5, (2+LPD_ROW_HEIGHT+4*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "eat ff");
/*g*/  pw_text(long_profile_pw, 5, (2+LPD_ROW_HEIGHT+5*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "drink");
/*g*/  pw_text(long_profile_pw, 5, (2+LPD_ROW_HEIGHT+6*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "clean");
/*g*/  pw_text(long_profile_pw, 5, (2+LPD_ROW_HEIGHT+7*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "court");
/*g*/  pw_text(long_profile_pw, 5, (2+LPD_ROW_HEIGHT+8*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "mate");
/*g*/  pw_text(long_profile_pw, 5, (2+LPD_ROW_HEIGHT+9*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "pounce");
/*g*/  pw_text(long_profile_pw, 5, (2+LPD_ROW_HEIGHT+10*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "look around");
/*g*/  pw_text(long_profile_pw, 5, (2+LPD_ROW_HEIGHT+11*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "look n");
/*g*/  pw_text(long_profile_pw, 5, (2+LPD_ROW_HEIGHT+12*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "look ne");
/*g*/  pw_text(long_profile_pw, 5, (2+LPD_ROW_HEIGHT+13*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "look e");
/*g*/  pw_text(long_profile_pw, 5, (2+LPD_ROW_HEIGHT+14*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "look se");
/*g*/  pw_text(long_profile_pw, 5, (2+LPD_ROW_HEIGHT+15*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "look s");
/*g*/  pw_text(long_profile_pw, 5, (2+LPD_ROW_HEIGHT+16*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "look sw");
/*g*/  pw_text(long_profile_pw, 5, (2+LPD_ROW_HEIGHT+17*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "look w");
/*g*/  pw_text(long_profile_pw, 5+EXTRA, (2+LPD_ROW_HEIGHT+0*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "look nw");

/*g*/  pw_text(long_profile_pw, 5+EXTRA, (2+LPD_ROW_HEIGHT+1*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "move n");
/*g*/  pw_text(long_profile_pw, 5+EXTRA, (2+LPD_ROW_HEIGHT+2*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "move ne");
/*g*/  pw_text(long_profile_pw, 5+EXTRA, (2+LPD_ROW_HEIGHT+3*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "move e");
/*g*/  pw_text(long_profile_pw, 5+EXTRA, (2+LPD_ROW_HEIGHT+4*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "move se");
/*g*/  pw_text(long_profile_pw, 5+EXTRA, (2+LPD_ROW_HEIGHT+5*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "move s");
/*g*/  pw_text(long_profile_pw, 5+EXTRA, (2+LPD_ROW_HEIGHT+6*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "move sw");
/*g*/  pw_text(long_profile_pw, 5+EXTRA, (2+LPD_ROW_HEIGHT+7*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "move w");
/*g*/  pw_text(long_profile_pw, 5+EXTRA, (2+LPD_ROW_HEIGHT+8*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "move nw");
/*g*/  pw_text(long_profile_pw, 5+EXTRA, (2+LPD_ROW_HEIGHT+9*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "move fast n");
/*g*/  pw_text(long_profile_pw, 5+EXTRA, (2+LPD_ROW_HEIGHT+10*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "move fast ne");
/*g*/  pw_text(long_profile_pw, 5+EXTRA, (2+LPD_ROW_HEIGHT+11*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "move fast e");
/*g*/  pw_text(long_profile_pw, 5+EXTRA, (2+LPD_ROW_HEIGHT+12*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "move fast se");
/*g*/  pw_text(long_profile_pw, 5+EXTRA, (2+LPD_ROW_HEIGHT+13*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "move fast s");
/*g*/  pw_text(long_profile_pw, 5+EXTRA, (2+LPD_ROW_HEIGHT+14*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "move fast sw");
/*g*/  pw_text(long_profile_pw, 5+EXTRA, (2+LPD_ROW_HEIGHT+15*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "move fast w");
/*g*/  pw_text(long_profile_pw, 5+EXTRA, (2+LPD_ROW_HEIGHT+16*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "move fast nw");
/*g*/
/*g*/  /* initialise old_lpd_x and old_lpd_y */
/*g*/  for (i = 0; i < NUM_SYSTEMS_FOR_LPD; i++) {
/*g*/    if (i <= 17)
/*g*/      old_lpd_x[i] = 3;
/*g*/    else
/*g*/      old_lpd_x[i] = 3+EXTRA;
/*g*/    if (i <= 17)
/*g*/      old_lpd_y[i] = 2 + i*(2+LPD_ROW_HEIGHT) + LPD_GRAPH_HEIGHT;
/*g*/    else
/*g*/      old_lpd_y[i] = 2 + (i-18)*(2+LPD_ROW_HEIGHT) + LPD_GRAPH_HEIGHT;
/*g*/  }
/*g*/}
/*g*/
/*g*/
/*g*//* function to update each of the longitudinal profiles */
/*g*/void update_long_profile()
/*g*/{
/*g*/
/*g*/  int i, lpd_x, lpd_y;
/*g*/  double max_action_activation;
/*g*/
/*g*/  /* for each profile */
/*g*/  for (i = 0; i < NUM_SYSTEMS_FOR_LPD; i++) {
/*g*/
/*g*/	if (i <= 17)
/*g*/	  lpd_x = 4 + lpd_position;
/*g*/	else
/*g*/	  lpd_x = 4 + lpd_position + EXTRA;
/*g*/
/*g*/	/* calculate y-position of graph point (dependent on some measure */
/*g*/	/* of activation of the system) */
/*g*/
/*g*/	lpd_acts[i] =
/*g*/      nodes[i+num_top_level_nodes+num_intermediate_nodes].activation/2.0;
/*g*//*	lpd_acts[i] =*/
/*g*//*      nodes[i+num_top_level_nodes+num_intermediate_nodes].activation / */
/*g*//*      max_action_activation; */
/*g*/	if (i <= 17)
/*g*/	  lpd_y = 2 + i*(2+LPD_ROW_HEIGHT) + LPD_GRAPH_HEIGHT -
/*g*/	    ((int) (lpd_acts[i] * ((double) (LPD_GRAPH_HEIGHT-2))));
/*g*/	else 
/*g*/	  lpd_y = 2 + (i-18)*(2+LPD_ROW_HEIGHT) + LPD_GRAPH_HEIGHT -
/*g*/	    ((int) (lpd_acts[i] * ((double) (LPD_GRAPH_HEIGHT-2))));
/*g*/
/*g*/	/* delete anything currently in the relevant part of the display */
/*g*/    if (i <= 17)
/*g*/	pw_rop(long_profile_pw, lpd_x, (3+i*(2+LPD_ROW_HEIGHT)), 13,
/*g*/	  (LPD_ROW_HEIGHT-1), PIX_CLR, 0, 0, 0);
/*g*/    else
/*g*/	/* delete anything currently in the relevant part of the display */
/*g*/	pw_rop(long_profile_pw, lpd_x, (3+(i-18)*(2+LPD_ROW_HEIGHT)), 13,
/*g*/	  (LPD_ROW_HEIGHT-1), PIX_CLR, 0, 0, 0); 
/*g*/
/*g*/	/* draw in a small symbol to represent the current activation of */
/*g*/	/* the system (as long as not just gone back to beginning) */
/*g*/	if ((lpd_x-old_lpd_x[i]) > 0) {
/*g*/      if (i <= 17)
/*g*/	    pw_vector(long_profile_pw, old_lpd_x[i], old_lpd_y[i], lpd_x, lpd_y,
/*g*/		  (PIX_SRC|PIX_COLOR(40)), 0);
/*g*/      else
/*g*/	    pw_vector(long_profile_pw, old_lpd_x[i], old_lpd_y[i], lpd_x, lpd_y,
/*g*/		  (PIX_SRC|PIX_COLOR(40)), 0);
/*g*/    }
/*g*/
/*g*/	old_lpd_y[i] = lpd_y;
/*g*/   old_lpd_x[i] = lpd_x;
/*g*/  }
/*g*/
/*g*/  /* delete any current blobs */
/*g*/  delete_lpd_blobs();
/*g*/
/*g*/  draw_lpd_blob(action);
/*g*/
/*g*/  lpd_position = (lpd_position+4) % (LDP_WIDTH-20);
/*g*/
/*g*/  /* write in the names of each profile */
/*g*/  pw_ttext(long_profile_pw, 5, (2+LPD_ROW_HEIGHT+0*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "sleep");
/*g*/  pw_ttext(long_profile_pw, 5, (2+LPD_ROW_HEIGHT+1*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "rest");
/*g*/  pw_ttext(long_profile_pw, 5, (2+LPD_ROW_HEIGHT+2*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "freeze");
/*g*/  pw_ttext(long_profile_pw, 5, (2+LPD_ROW_HEIGHT+3*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "eat cf");
/*g*/  pw_ttext(long_profile_pw, 5, (2+LPD_ROW_HEIGHT+4*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "eat ff");
/*g*/  pw_ttext(long_profile_pw, 5, (2+LPD_ROW_HEIGHT+5*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "drink");
/*g*/  pw_ttext(long_profile_pw, 5, (2+LPD_ROW_HEIGHT+6*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "clean");
/*g*/  pw_ttext(long_profile_pw, 5, (2+LPD_ROW_HEIGHT+7*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "court");
/*g*/  pw_ttext(long_profile_pw, 5, (2+LPD_ROW_HEIGHT+8*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "mate");
/*g*/  pw_ttext(long_profile_pw, 5, (2+LPD_ROW_HEIGHT+9*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "pounce");
/*g*/  pw_ttext(long_profile_pw, 5, (2+LPD_ROW_HEIGHT+10*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "look around");
/*g*/  pw_ttext(long_profile_pw, 5, (2+LPD_ROW_HEIGHT+11*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "look n");
/*g*/  pw_ttext(long_profile_pw, 5, (2+LPD_ROW_HEIGHT+12*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "look ne");
/*g*/  pw_ttext(long_profile_pw, 5, (2+LPD_ROW_HEIGHT+13*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "look e");
/*g*/  pw_ttext(long_profile_pw, 5, (2+LPD_ROW_HEIGHT+14*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "look se");
/*g*/  pw_ttext(long_profile_pw, 5, (2+LPD_ROW_HEIGHT+15*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "look s");
/*g*/  pw_ttext(long_profile_pw, 5, (2+LPD_ROW_HEIGHT+16*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "look sw");
/*g*/  pw_ttext(long_profile_pw, 5, (2+LPD_ROW_HEIGHT+17*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "look w");
/*g*/  pw_ttext(long_profile_pw, 5+EXTRA, (2+LPD_ROW_HEIGHT+0*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "look nw");

/*g*/  pw_ttext(long_profile_pw, 5+EXTRA, (2+LPD_ROW_HEIGHT+1*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "move n");
/*g*/  pw_ttext(long_profile_pw, 5+EXTRA, (2+LPD_ROW_HEIGHT+2*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "move ne");
/*g*/  pw_ttext(long_profile_pw, 5+EXTRA, (2+LPD_ROW_HEIGHT+3*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "move e");
/*g*/  pw_ttext(long_profile_pw, 5+EXTRA, (2+LPD_ROW_HEIGHT+4*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "move se");
/*g*/  pw_ttext(long_profile_pw, 5+EXTRA, (2+LPD_ROW_HEIGHT+5*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "move s");
/*g*/  pw_ttext(long_profile_pw, 5+EXTRA, (2+LPD_ROW_HEIGHT+6*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "move sw");
/*g*/  pw_ttext(long_profile_pw, 5+EXTRA, (2+LPD_ROW_HEIGHT+7*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "move w");
/*g*/  pw_ttext(long_profile_pw, 5+EXTRA, (2+LPD_ROW_HEIGHT+8*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "move nw");
/*g*/  pw_ttext(long_profile_pw, 5+EXTRA, (2+LPD_ROW_HEIGHT+9*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "move fast n");
/*g*/  pw_ttext(long_profile_pw, 5+EXTRA, (2+LPD_ROW_HEIGHT+10*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "move fast ne");
/*g*/  pw_ttext(long_profile_pw, 5+EXTRA, (2+LPD_ROW_HEIGHT+11*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "move fast e");
/*g*/  pw_ttext(long_profile_pw, 5+EXTRA, (2+LPD_ROW_HEIGHT+12*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "move fast se");
/*g*/  pw_ttext(long_profile_pw, 5+EXTRA, (2+LPD_ROW_HEIGHT+13*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "move fast s");
/*g*/  pw_ttext(long_profile_pw, 5+EXTRA, (2+LPD_ROW_HEIGHT+14*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "move fast sw");
/*g*/  pw_ttext(long_profile_pw, 5+EXTRA, (2+LPD_ROW_HEIGHT+15*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "move fast w");
/*g*/  pw_ttext(long_profile_pw, 5+EXTRA, (2+LPD_ROW_HEIGHT+16*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "move fast nw");
/*g*/
/*g*/  /* draw in horizontal axis for each graph */
/*g*/  for (i = 0; i < NUM_SYSTEMS_FOR_LPD; i++)
/*g*/    if (i <= 17)
/*g*/      pw_vector(long_profile_pw, 3, (2+i*(2+LPD_ROW_HEIGHT)+
/*g*/        LPD_GRAPH_HEIGHT), (LDP_WIDTH-3), (2+i*(2+LPD_ROW_HEIGHT)+
/*g*/        LPD_GRAPH_HEIGHT), (PIX_SRC|PIX_COLOR(40)), 0);
/*g*/    else
/*g*/      pw_vector(long_profile_pw, 3+EXTRA, (2+(i-18)*(2+LPD_ROW_HEIGHT)+
/*g*/        LPD_GRAPH_HEIGHT), (LDP_WIDTH-3+EXTRA), (2+(i-18)*
/*g*/        (2+LPD_ROW_HEIGHT)+LPD_GRAPH_HEIGHT), (PIX_SRC|PIX_COLOR(40)), 0);
/*g*/}
/*g*/
/*g*/
/*g*//* function to draw a little blob when an action relevant to the */ 
/*g*//* system is chosen */
/*g*/void draw_lpd_blob(profile_number)
/*g*/int profile_number;
/*g*/{
/*g*/
/*g*/  int blob_x, blob_y;
/*g*/
/*g*/  /* calculate coordinates of blob (to go just beneath bar for relevant */
/*g*/  /* system, at current time */
/*g*/  blob_x = (3+lpd_position);
/*g*/  if (profile_number > 17)
/*g*/    blob_x += EXTRA;
/*g*/  blob_y = 2+profile_number*(2+LPD_ROW_HEIGHT)+LPD_GRAPH_HEIGHT+1; 
/*g*/  if (profile_number > 17)
/*g*/    blob_y -= 18*(2+LPD_ROW_HEIGHT);
/*g*/
/*g*/  pw_rop(long_profile_pw, blob_x, blob_y, 4, 3,
/*g*/	(PIX_SRC|PIX_COLOR(40)), 0, 0, 0);
/*g*/}
/*g*/
/*g*/
/*g*//* function to delete any existing blobs */
/*g*/void delete_lpd_blobs()
/*g*/{
/*g*/
/*g*/  int i, blob_x, blob_y;
/*g*/
/*g*/  blob_x = (4 + lpd_position);
/*g*/
/*g*/  for (i = 0; i < NUM_SYSTEMS_FOR_LPD; i++) {
/*g*/	blob_y = 2 + i*(2+LPD_ROW_HEIGHT) + LPD_GRAPH_HEIGHT + 1;
/*g*/   if (i > 17) {
/*g*/     blob_x += EXTRA;
/*g*/     blob_y -= 18*(2+LPD_ROW_HEIGHT);
/*g*/   }
/*g*/	pw_rop(long_profile_pw, blob_x, blob_y, 13, 3, PIX_CLR, 0, 0, 0);
/*g*/  }
/*g*/}
/*g*/
/*g*/
/*g*/void profile_change_proc()
/*g*/{
/*g*/
/*g*/  if (window_get(long_profile_frame, WIN_SHOW) == FALSE) {
/*g*/	turn_on(long_profile_frame);
/*g*/	profile_display_on = TRUE;
/*g*/  }
/*g*/  else {
/*g*/	turn_off(long_profile_frame);
/*g*/	profile_display_on = FALSE;
/*g*/  }
/*g*/}


/* function to scan along 'l' elements of a vector 'v' of doubles and return
the value of the largest element */
double vector_max(v, l)
double *v;		/* pointer to vector of doubles */
int l;			/* number of elements to find maximum over */
{

  int i;
  double maximum = -1000000.0;

  for (i = 0; i < l; i++)
	if (v[i] > maximum)
	  maximum = v[i];

  return(maximum);
}


/* function to scan along 'l' elements of a vector 'v' of doubles and return
the value of the smallest or most negative element */
double vector_min(v, l)
double *v;		/* pointer to vector of doubles */
int l;			/* number of elements to find minimum over */
{

  int i;
  double minimum = 1000000.0;

  for (i = 0; i < l; i++)
	if (v[i] < minimum)
	  minimum = v[i];

  return(minimum);
}


/* function to scan along 'l' elements of a vector 'v' of doubles and return
the sum of all the elements */
double vector_sum(v, l)
double *v;		/* pointer to vector of doubles */
int l;			/* number of elements to find sum of */
{

  int i;
  double sum = 0.0;

  for (i = 0; i < l; i++)
	sum += v[i];

  return(sum);
}


/* function to combine the inputs to a node in an arbitrary fashion */
combine_inputs(n, type)
int n;					/* number of node */
int type;				/* type of node (single or multiple) */
{

  int j;
  double a;			/* activation of node */
  double m_a[8];	/* activations of multiple node */

  if (strcmp(nodes[n].name, "get food") == 0) {
	a = is[0] + is[1];
	if (a > 0.95)
	  a = 0.95;
  }
  else if (strcmp(nodes[n].name, "get water") == 0) {
	a = is[0] + is[1];
	if (a > 0.95)
	  a = 0.95;
  }
  else if (strcmp(nodes[n].name, "cool down") == 0) {
	a = is[0] + is[1];
  }
  else if (strcmp(nodes[n].name, "warm up") == 0) {
	a = is[0] + is[1];
  }
  else if (strcmp(nodes[n].name, "avoid p1s") == 0) {
	a = is[0];
  }
  else if (strcmp(nodes[n].name, "avoid p2s") == 0) {
	a = is[0];
  }
  else if (strcmp(nodes[n].name, "avoid dps") == 0) {
	a = is[0];
  }
  else if (strcmp(nodes[n].name, "avoid irrs") == 0) {
	a = is[0];
  }
  else if (strcmp(nodes[n].name, "reproduce") == 0) {
	a = is[0];
  }
  else if (strcmp(nodes[n].name, "sleep in den") == 0) {
	a = is[0] + (is[0]*is[1]);
  }
  else if (strcmp(nodes[n].name, "keep variance low") == 0) {
	a = is[0];
  }
  else if (strcmp(nodes[n].name, "keep clean") == 0) {
	a = is[0] + is[1];
  }
  else if (strcmp(nodes[n].name, "avoid edges") == 0) {
	a = is[0];
  }
  else if (strcmp(nodes[n].name, "scan for predators") == 0) {
	a = is[0] + is[1];
  }
  else if (strcmp(nodes[n].name, "stay close to cover") == 0) {
	a = is[0];
  }
  else if (strcmp(nodes[n].name, "eat food") == 0) {
	a = is[0];
  }
  else if (strcmp(nodes[n].name, "approach p food") == 0) {
	a = is[0] + is[1] + is[2];
  }
  else if (strcmp(nodes[n].name, "approach r food") == 0) {
	a = is[0] + is[1] + is[2];
  }
  else if (strcmp(nodes[n].name, "explore for food") == 0) {
	a = is[0] + is[1] + is[2];
  }
  else if (strcmp(nodes[n].name, "don't use up food") == 0) {
	a = is[0];
  }
  else if (strcmp(nodes[n].name, "drink (il)") == 0) {
	a = is[0];
  }
  else if (strcmp(nodes[n].name, "approach p water") == 0) {
	a = is[0] + is[1] + is[2];
  }
  else if (strcmp(nodes[n].name, "approach r water") == 0) {
	a = is[0] + is[1] + is[2];
  }
  else if (strcmp(nodes[n].name, "explore for water") == 0) {
	a = is[0] + is[1] + is[2];
  }
  else if (strcmp(nodes[n].name, "don't use up water") == 0) {
	a = is[0];
  }
  else if (strcmp(nodes[n].name, "rest in shelter") == 0) {
	a = is[0];
  }
  else if (strcmp(nodes[n].name, "approach p shelter") == 0) {
	a = is[0] + is[1] + is[2];
  }
  else if (strcmp(nodes[n].name, "head back to den for shelter") == 0) {
	a = is[0] + is[1] + is[2];
  }
  else if (strcmp(nodes[n].name, "do strenuous actions") == 0) {
	a = is[0];
  }
  else if (strcmp(nodes[n].name, "rest in shade") == 0) {
	a = is[0];
  }
  else if (strcmp(nodes[n].name, "approach p shade") == 0) {
	a = is[0] + is[1] + is[2];
  }
  else if (strcmp(nodes[n].name, "head back to den for shade") == 0) {
	a = is[0] + is[1] + is[2];
  }
  else if (strcmp(nodes[n].name, "don't do strenuous actions") == 0) {
	a = is[0];
  }
  else if (strcmp(nodes[n].name, "avoid dp square") == 0) {
	a = is[0];
  }
  else if (strcmp(nodes[n].name, "leave current square because of dp") == 0) {
	a = is[0] * is[1];
  }
  else if (strcmp(nodes[n].name, "move away from dp square") == 0) {
	a = is[0];
  }
  else if (strcmp(nodes[n].name, "avoid irr square") == 0) {
	a = is[0];
  }
  else if (strcmp(nodes[n].name, "don't approach dps") == 0) {
	a = is[0];
  }
  else if (strcmp(nodes[n].name, "don't approach irrs") == 0) {
	a = is[0];
  }
  else if (strcmp(nodes[n].name, "leave current square because of irr") == 0) {
	a = is[0] * is[1];
  }
  else if (strcmp(nodes[n].name, "move away from irr square") == 0) {
	a = is[0];
  }
  else if (strcmp(nodes[n].name, "freeze because of p1") == 0) {
	a = is[0];
  }
  else if (strcmp(nodes[n].name, "approach shelter because of p1") == 0) {
	a = (is[0] * is[2]) + is[3];
  }
  else if (strcmp(nodes[n].name, "run away from p1 square") == 0) {
	a = (is[0] * is[1]);
  }
  else if (strcmp(nodes[n].name, "leave current square because of p1") == 0) {
	a = (is[0] * is[1]);
  }
  else if (strcmp(nodes[n].name, "don't approach p1s") == 0) {
	a = is[0];
  }
  else if (strcmp(nodes[n].name, "freeze because of p2") == 0) {
	a = is[0];
  }
  else if (strcmp(nodes[n].name, "approach shelter because of p2") == 0) {
	a = (is[0] * is[2]) + is[3];
  }
  else if (strcmp(nodes[n].name, "run away from p2 square") == 0) {
	a = (is[0] * is[1]);
  }
  else if (strcmp(nodes[n].name, "leave current square because of p2") == 0) {
	a = (is[0] * is[1]);
  }
  else if (strcmp(nodes[n].name, "don't approach p2s") == 0) {
	a = is[0];
  }
  else if (strcmp(nodes[n].name, "mate (il)") == 0) {
	a = is[0];
  }
  else if (strcmp(nodes[n].name, "court (il)") == 0) {
	a = is[0] + is[1] + is[2];
  }
  else if (strcmp(nodes[n].name, "approach p mate") == 0) {
	a = is[0] + is[1] + is[2];
  }
  else if (strcmp(nodes[n].name, "explore for mate") == 0) {
	a = is[0] + is[1] + is[2];
  }
  else if (strcmp(nodes[n].name, "sleep (il)") == 0) {
	a = is[0];
  }
  else if (strcmp(nodes[n].name, "approach p den to sleep") == 0) {
	a = is[0] + is[1] + is[2];
  }
  else if (strcmp(nodes[n].name, "approach r den to sleep") == 0) {
	a = is[0] + is[1] + is[2];
  }
  else if (strcmp(nodes[n].name, "rest (il)") == 0) {
	a = is[0];
  }
  else if (strcmp(nodes[n].name, "approach p den to reduce variance") == 0) {
	a = is[0];
  }
  else if (strcmp(nodes[n].name, "approach r den to reduce variance") == 0) {
	a = is[0];
  }
  else if (strcmp(nodes[n].name, "don't explore because of variance") == 0) {
	a = is[0];
  }
  else if (strcmp(nodes[n].name, "clean (il)") == 0) {
	a = is[0];
  }
  else if (strcmp(nodes[n].name, "leave current square (den) to clean") == 0) {
	a = (is[0]*is[1]) + is[2] + is[3];
  }
  else if (strcmp(nodes[n].name, "avoid edge square") == 0) {
	a = is[0];
  }
  else if (strcmp(nodes[n].name,
	  "leave current square because of edge") == 0) {
	a = is[0];
  }
  else if (strcmp(nodes[n].name, "look around (il)") == 0) {
	a = is[0];
  }
  else if (strcmp(nodes[n].name, "look towards p1") == 0) {
	a = is[0];
  }
  else if (strcmp(nodes[n].name, "look towards p2") == 0) {
	a = is[0];
  }
  else if (strcmp(nodes[n].name, "approach cover") == 0) {
	a = is[0];
  }
  else if (strcmp(nodes[n].name, "approach r den for cover") == 0) {
	a = is[0] + is[1] + is[2];
  }
  else if (strcmp(nodes[n].name, "explore") == 0) {
	a = combine_explore_inputs();
  }
  else if (strcmp(nodes[n].name, "eat cf") == 0) {
	a = is[0] * is[1];
  }
  else if (strcmp(nodes[n].name, "eat ff") == 0) {
	a = is[0] * is[1];
  }
  else if (strcmp(nodes[n].name, "pounce") == 0) {
	a = is[0] * is[1] * is[2];
  }
  else if (strcmp(nodes[n].name, "drink (bl)") == 0) {
	a = is[0] * is[1];
  }
  else if (strcmp(nodes[n].name, "rest (bl)") == 0) {
	a = combine_rest_inputs();
  }
  else if (strcmp(nodes[n].name, "freeze") == 0) {
	a = combine_freeze_inputs();
  }
  else if (strcmp(nodes[n].name, "court (bl)") == 0) {
	a = combine_court_inputs();
  }
  else if (strcmp(nodes[n].name, "mate (bl)") == 0) {
	a = combine_mate_inputs();
  }
  else if (strcmp(nodes[n].name, "sleep (bl)") == 0) {
	a = (is[0]*is[1]);
  }
  else if (strcmp(nodes[n].name, "clean (bl)") == 0) {
	a = (is[0]*is[1]);
  }
  else if (strcmp(nodes[n].name, "look n") == 0) {
	for (j = 0; j < 8; j++)
	  m_a[j] = combine_look_inputs(j);
  }
  else if (strcmp(nodes[n].name, "move n") == 0) {
	for (j = 0; j < 8; j++)
	  m_a[j] = combine_move_inputs(j);
  }
  else if (strcmp(nodes[n].name, "move fast n") == 0) {
	for (j = 0; j < 8; j++)
	  m_a[j] = combine_move_fast_inputs(j);
  }
  else if (strcmp(nodes[n].name, "look around (bl)") == 0) {
	a = is[0];
  }
  else {
	printf("trying to combine inputs for bad node (%s)\n", nodes[n].name);
	exit(1);
  }

  if (type == NODE)
	nodes[n].activation = a;
  else if (type == MULTIPLE_NODE)
	for (j = 0; j < 8; j++)
	  nodes[n+j].activation = m_a[j];
}


/* function to specify how the inputs to the node "explore" are combined */
double combine_explore_inputs()
{

  double sum_weighting, max_weighting;
  double result;

  max_weighting = 1.0;
  sum_weighting = (1.0/(3.0*3.0));

  result = (((max_weighting * vector_max(is, 3)) +
			 (sum_weighting * vector_sum(is, 3))) /
			(max_weighting + sum_weighting)) +
		   is[3];							/* is[3] is negative */

  return(result);
}


/* function to specify how the inputs to the node "rest (bl)" are combined */
double combine_rest_inputs()
{

  double sum_weighting, max_weighting;
  double positive_inputs[3];
  double result;

  /* do multiplications first */
  positive_inputs[0] = (is[0]*is[1]);
  positive_inputs[1] = (is[2]*is[3]);
  positive_inputs[2] = (is[4]*is[5]);

  max_weighting = 1.0;
  sum_weighting = (1.0/(3.0*3.0));

  /* do combination of multiples (no negative input in this case) */
  result = (((max_weighting * vector_max(positive_inputs, 3)) +
			 (sum_weighting * vector_sum(positive_inputs, 3))) /
			(max_weighting + sum_weighting));

  return(result);
}


/* function to specify how the inputs to the node "freeze" are combined */
double combine_freeze_inputs()
{

  double sum_weighting, max_weighting;
  double positive_inputs[2];
  double result;

  /* do multiplications first */
  positive_inputs[0] = (is[0]*is[1]*is[2]);
  positive_inputs[1] = (is[3]*is[4]*is[5]);

  max_weighting = 1.0;
  sum_weighting = (1.0/(2.0*2.0));

  /* do combination of multiples (no negative input in this case) */
  result = (((max_weighting * vector_max(positive_inputs, 2)) +
			 (sum_weighting * vector_sum(positive_inputs, 2))) /
			(max_weighting + sum_weighting));

  return(result);
}


/* function to specify how the inputs to the node "court" are combined */
double combine_court_inputs()
{

  double sum_weighting, max_weighting;
  double positive_inputs[2];
  double negative_inputs[3];
  double positive_result, negative_result;
  double result;

  /* do multiplications first */
  positive_inputs[0] = (is[0]*is[1]*is[2]);
  positive_inputs[1] = is[6];

  negative_inputs[0] = is[3];
  negative_inputs[1] = is[4];
  negative_inputs[2] = is[5];

  max_weighting = 1.0;
  sum_weighting = (1.0/(2.0*2.0));

  positive_result = (((max_weighting * vector_max(positive_inputs, 2)) +
					  (sum_weighting * vector_sum(positive_inputs, 2))) /
					 (max_weighting + sum_weighting));

  sum_weighting = (1.0/(3.0*3.0));

  negative_result = (((max_weighting * vector_min(negative_inputs, 3)) +
					  (sum_weighting * vector_sum(negative_inputs, 3))) /
					 (max_weighting + sum_weighting));

/*  if (negative_result < -0.1)
	printf("inhibition of %.3f to court\n", negative_result);
*/
  result = positive_result + negative_result;

  return(result);
}


/* function to specify how the inputs to the node "mate" are combined */
double combine_mate_inputs()
{

  double sum_weighting, max_weighting;
  double positive_inputs[2];
  double negative_inputs[3];
  double positive_result, negative_result;
  double result;

  /* do multiplications first */
  positive_inputs[0] = (is[0]*is[1]*is[2]);
  positive_inputs[1] = is[6];

  negative_inputs[0] = is[3];
  negative_inputs[1] = is[4];
  negative_inputs[2] = is[5];

  max_weighting = 1.0;
  sum_weighting = (1.0/(2.0*2.0));

  positive_result = (((max_weighting * vector_max(positive_inputs, 2)) +
					  (sum_weighting * vector_sum(positive_inputs, 2))) /
					 (max_weighting + sum_weighting));

  sum_weighting = (1.0/(3.0*3.0));

  negative_result = (((max_weighting * vector_min(negative_inputs, 3)) +
					  (sum_weighting * vector_sum(negative_inputs, 3))) /
					 (max_weighting + sum_weighting));

/*  if (negative_result < -0.1)
	printf("inhibition of %.3f to mate\n", negative_result);
*/
  result = positive_result + negative_result;

  return(result);
}


/* function to specify how the inputs to the multiple node "look" are
combined */
double combine_look_inputs(direction)
int direction;				/* element of array being considered */
{

  double sum_weighting, max_weighting;
  double positive_inputs[2];
  double result;

  /* do multiplications first */
  positive_inputs[0] = (m_is[0][direction]*m_is[1][direction]);
  positive_inputs[1] = (m_is[2][direction]*m_is[3][direction]);

  max_weighting = 1.0;
  sum_weighting = (1.0/(2.0*2.0));

  result = (((max_weighting * vector_max(positive_inputs, 2)) +
			 (sum_weighting * vector_sum(positive_inputs, 2))) /
			(max_weighting + sum_weighting));

  return(result);
}


/* function to specify how the inputs to the multiple node "move" are
combined */
double combine_move_inputs(d)
int d;				/* element of array being considered */
{

  double sum_weighting, max_weighting;
  double positive_inputs[23];
  double negative_inputs[9];
  double positive_result, negative_result;
  double result;
  int i;

  /* do multiplications first */
  positive_inputs[0] = (m_is[0][d]*m_is[1][d]);
  positive_inputs[1] = (m_is[2][d]*m_is[3][d]);
  positive_inputs[2] = (m_is[4][d]*m_is[5][d]);
  positive_inputs[3] = (m_is[6][d]*m_is[7][d]);
  positive_inputs[4] = (m_is[8][d]*m_is[9][d]);
  positive_inputs[5] = (m_is[10][d]*m_is[11][d]);
  positive_inputs[6] = (m_is[12][d]*m_is[13][d]);
  positive_inputs[7] = (m_is[14][d]*m_is[15][d]);
  positive_inputs[8] = (m_is[32][d]*m_is[33][d]);
  positive_inputs[9] = (m_is[34][d]*m_is[35][d]);
  positive_inputs[10] = (m_is[36][d]*m_is[37][d]);
  positive_inputs[11] = (m_is[38][d]*m_is[39][d]);
  positive_inputs[12] = (m_is[40][d]*m_is[41][d]);
  positive_inputs[13] = (m_is[44][d]*m_is[45][d]);
  positive_inputs[14] = (m_is[46][d]*m_is[47][d]);
  positive_inputs[15] = (m_is[48][d]*m_is[49][d]);
  positive_inputs[16] = (m_is[50][d]*m_is[51][d]);
  positive_inputs[17] = (m_is[52][d]*m_is[53][d]);
  positive_inputs[18] = (m_is[54][d]*m_is[55][d]);
  positive_inputs[19] = (m_is[56][d]*m_is[57][d]);
  positive_inputs[20] = (m_is[58][d]*m_is[59][d]);
  positive_inputs[21] = (m_is[60][d]*m_is[61][d]);
  positive_inputs[22] = (m_is[62][d]*m_is[63][d]);

  negative_inputs[0] = (m_is[16][d]*m_is[17][d]);
  negative_inputs[1] = (m_is[18][d]*m_is[19][d]);
  negative_inputs[2] = (m_is[20][d]*m_is[21][d]);
  negative_inputs[3] = (m_is[22][d]*m_is[23][d]);
  negative_inputs[4] = (m_is[24][d]*m_is[25][d]);
  negative_inputs[5] = (m_is[26][d]*m_is[27][d]);
  negative_inputs[6] = (m_is[28][d]*m_is[29][d]);
  negative_inputs[7] = (m_is[30][d]*m_is[31][d]);
  negative_inputs[8] = (m_is[42][d]*m_is[43][d]);

  max_weighting = 1.0;
  sum_weighting = (1.0/(23.0*23.0));

  positive_result = (((max_weighting * vector_max(positive_inputs, 23)) +
					  (sum_weighting * vector_sum(positive_inputs, 23))) /
					 (max_weighting + sum_weighting));

  sum_weighting = (1.0/(9.0*9.0));

  negative_result = (((max_weighting * vector_min(negative_inputs, 9)) +
					  (sum_weighting * vector_sum(negative_inputs, 9))) /
					 (max_weighting + sum_weighting));

  result = positive_result + negative_result;

  return(result);
}


/* function to specify how the inputs to the multiple node "move_fast" are
combined */
double combine_move_fast_inputs(d)
int d;				/* element of array being considered */
{

  double sum_weighting, max_weighting;
  double positive_inputs[5];
  double negative_inputs[10];
  double positive_result, negative_result;
  double result;

  /* do multiplications first */
  positive_inputs[0] = (m_is[0][d]*m_is[1][d]);
  positive_inputs[1] = (m_is[2][d]*m_is[3][d]);
  positive_inputs[2] = (m_is[4][d]*m_is[5][d]);
  positive_inputs[3] = (m_is[6][d]*m_is[7][d]);
  positive_inputs[4] = m_is[25][d];

  negative_inputs[0] = (m_is[8][d]*m_is[9][d]);
  negative_inputs[1] = (m_is[10][d]*m_is[11][d]);
  negative_inputs[2] = (m_is[12][d]*m_is[13][d]);
  negative_inputs[3] = (m_is[14][d]*m_is[15][d]);
  negative_inputs[4] = (m_is[16][d]*m_is[17][d]);
  negative_inputs[5] = (m_is[18][d]*m_is[19][d]);
  negative_inputs[6] = (m_is[20][d]*m_is[21][d]);
  negative_inputs[7] = m_is[22][d];
  negative_inputs[8] = m_is[23][d];
  negative_inputs[9] = m_is[24][d];

  max_weighting = 1.0;
  sum_weighting = (1.0/(5.0*5.0));

  positive_result = (((max_weighting * vector_max(positive_inputs, 5)) +
					  (sum_weighting * vector_sum(positive_inputs, 5))) /
					 (max_weighting + sum_weighting));


  sum_weighting = (1.0/(10.0*10.0));

  negative_result = (((max_weighting * vector_min(negative_inputs, 10)) +
					  (sum_weighting * vector_sum(negative_inputs, 10))) /
					 (max_weighting + sum_weighting));

  result = positive_result + negative_result;

/*  if (negative_result < -0.1)
	printf("inhibition of %.3f to move fast node %d (result = %.3f)\n",
	  negative_result, d, result);
*/
  return(result);
}



/*g*//* function to initialise a display for longitudinal profiles for each */
/*g*//* sub-problem, showing motivational levels and occurrences of relevant */
/*g*//* actions */
/*g*/void set_up_long_profile2()
/*g*/{
/*g*/
/*g*/  int i;
/*g*/
/*g*/  screen_r_7 = pf_open("/usr/lib/fonts/fixedwidthfonts/screen.r.7");
/*g*/
/*g*/  long_profile_frame2 = window_create(NULL, FRAME,
/*g*/    WIN_SHOW, TRUE,
/*g*/    FRAME_LABEL, "Longitudinal Profiles (sub-systems)",
/*g*/    WIN_X, 200,
/*g*/    WIN_Y, 50,
/*g*/    WIN_WIDTH, LDP_WIDTH+10,
/*g*/    WIN_HEIGHT, (13*(LPD_ROW_HEIGHT+2)+2+25),
/*g*/    0);
/*g*/
/*g*/  long_profile_canvas2 = window_create(long_profile_frame2, CANVAS,
/*g*/    WIN_SHOW, TRUE,
/*g*/    CANVAS_AUTO_SHRINK, TRUE,
/*g*/    CANVAS_AUTO_EXPAND, TRUE,
/*g*/    WIN_X, 0,
/*g*/    WIN_Y, 3,
/*g*/    WIN_WIDTH, LDP_WIDTH+6,
/*g*/    WIN_HEIGHT, (13*(LPD_ROW_HEIGHT+2)+2),
/*g*/    0);
/*g*/
/*g*/  long_profile_pw2 = canvas_pixwin(long_profile_canvas2);
/*g*/
/*g*/  /* attach colourmap to pixwin */
/*g*/  pw_setcmsname(long_profile_pw2, "doesntmatter");
/*g*/  pw_putcolormap(long_profile_pw2, 0, CMAP_SIZE, red, green, blue);
/*g*/
/*g*/  /* draw in blank graphs for each system */
/*g*/  for (i = 0; i < 13; i++) {
/*g*/
/*g*/	  pw_vector(long_profile_pw2, 3, (2+i*(2+LPD_ROW_HEIGHT)), (LDP_WIDTH-3),
/*g*/		(2+i*(2+LPD_ROW_HEIGHT)), (PIX_SRC|PIX_COLOR(40)), 0);
/*g*/	  pw_vector(long_profile_pw2, 3, (2+i*(2+LPD_ROW_HEIGHT)), 3, 
/*g*/		(2+LPD_ROW_HEIGHT+i*(2+LPD_ROW_HEIGHT)), (PIX_SRC|PIX_COLOR(40)), 0);
/*g*/	  pw_vector(long_profile_pw2, 3, (2+LPD_ROW_HEIGHT+i*(2+LPD_ROW_HEIGHT)),
/*g*/		(LDP_WIDTH-3), (2+LPD_ROW_HEIGHT+i*(2+LPD_ROW_HEIGHT)),
/*g*/		(PIX_SRC|PIX_COLOR(40)), 0); 
/*g*/	  pw_vector(long_profile_pw2, (LDP_WIDTH-3), (2+i*(2+LPD_ROW_HEIGHT)),
/*g*/		(LDP_WIDTH-3), (2+LPD_ROW_HEIGHT+i*(2+LPD_ROW_HEIGHT)),
/*g*/		(PIX_SRC|PIX_COLOR(40)), 0);
/*g*/	  pw_vector(long_profile_pw2, 3, (2+i*(2+LPD_ROW_HEIGHT)+LPD_GRAPH_HEIGHT),
/*g*/		(LDP_WIDTH-3), (2+i*(2+LPD_ROW_HEIGHT)+LPD_GRAPH_HEIGHT),
/*g*/		(PIX_SRC|PIX_COLOR(40)), 0);
/*g*/  }
/*g*/  /* initialise old_lpd_x and old_lpd_y */
/*g*/  for (i = 0; i < 13; i++) {
/*g*/      old_lpd_x2[i] = 3;
/*g*/      old_lpd_y2[i] = 2 + i*(2+LPD_ROW_HEIGHT) + LPD_GRAPH_HEIGHT;
/*g*/  }
/*g*/
/*g*/  /* initialise number of each node above */
/*g*/  sub_sys_node_num[0] = 15;
/*g*/  sub_sys_node_num[1] = 16;
/*g*/  sub_sys_node_num[2] = 17;
/*g*/  sub_sys_node_num[3] = 20;
/*g*/  sub_sys_node_num[4] = 25;
/*g*/  sub_sys_node_num[5] = 26;
/*g*/  sub_sys_node_num[6] = 29;
/*g*/  sub_sys_node_num[7] = 34;
/*g*/  sub_sys_node_num[8] = 59;
/*g*/  sub_sys_node_num[9] = 56;
/*g*/  sub_sys_node_num[10] = 68;
/*g*/  sub_sys_node_num[11] = 69;
/*g*/  sub_sys_node_num[12] = 70;
/*g*/
/*g*/  turn_on(long_profile_frame2);
/*g*/}
/*g*/
/*g*/
/*g*//* function to update each of the longitudinal profiles */
/*g*/void update_long_profile2()
/*g*/{
/*g*/
/*g*/  int i, lpd_x, lpd_y;
/*g*/  double max_action_activation;
/*g*/
/*g*/  /* for each profile */
/*g*/  for (i = 0; i < 13; i++) {
/*g*/
/*g*/	  lpd_x = 4 + lpd_position;
/*g*/
/*g*/	/* calculate y-position of graph point (dependent on some measure */
/*g*/	/* of activation of the system) */
/*g*/
/*g*/	lpd_acts2[i] = nodes[sub_sys_node_num[i]].activation/2.0;
/*g*/	  lpd_y = 2 + i*(2+LPD_ROW_HEIGHT) + LPD_GRAPH_HEIGHT -
/*g*/	    ((int) (lpd_acts2[i] * ((double) (LPD_GRAPH_HEIGHT-2))));
/*g*/
/*g*/	/* delete anything currently in the relevant part of the display */
/*g*/	pw_rop(long_profile_pw2, lpd_x, (3+i*(2+LPD_ROW_HEIGHT)), 13,
/*g*/	  (LPD_ROW_HEIGHT-1), PIX_CLR, 0, 0, 0);
/*g*/
/*g*/	/* draw in a small symbol to represent the current activation of */
/*g*/	/* the system (as long as not just gone back to beginning) */
/*g*/	if ((lpd_x-old_lpd_x[i]) > 0) {
/*g*/	    pw_vector(long_profile_pw2, old_lpd_x2[i], old_lpd_y2[i], lpd_x, lpd_y,
/*g*/		  (PIX_SRC|PIX_COLOR(40)), 0);
/*g*/    }
/*g*/
/*g*/	old_lpd_y2[i] = lpd_y;
/*g*/   old_lpd_x2[i] = lpd_x;
/*g*/  }
/*g*/
/*g*/  /* write in the names of each profile */
/*g*/  pw_text(long_profile_pw2, 5, (2+LPD_ROW_HEIGHT+0*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "approach p food");
/*g*/  pw_text(long_profile_pw2, 5, (2+LPD_ROW_HEIGHT+1*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "approach r food");
/*g*/  pw_text(long_profile_pw2, 5, (2+LPD_ROW_HEIGHT+2*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "explore for food");
/*g*/  pw_text(long_profile_pw2, 5, (2+LPD_ROW_HEIGHT+3*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "approach p water");
/*g*/  pw_text(long_profile_pw2, 5, (2+LPD_ROW_HEIGHT+4*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "approach p shelter");
/*g*/  pw_text(long_profile_pw2, 5, (2+LPD_ROW_HEIGHT+5*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "head back to den for shelter");
/*g*/  pw_text(long_profile_pw2, 5, (2+LPD_ROW_HEIGHT+6*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "approach p shade");
/*g*/  pw_text(long_profile_pw2, 5, (2+LPD_ROW_HEIGHT+7*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "don't approach dps");
/*g*/  pw_text(long_profile_pw2, 5, (2+LPD_ROW_HEIGHT+8*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "approach r den to reduce variance");
/*g*/  pw_text(long_profile_pw2, 5, (2+LPD_ROW_HEIGHT+9*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "approach r den to sleep");
/*g*/  pw_text(long_profile_pw2, 5, (2+LPD_ROW_HEIGHT+10*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "approach cover");
/*g*/  pw_text(long_profile_pw2, 5, (2+LPD_ROW_HEIGHT+11*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "approach r den for cover");
/*g*/  pw_text(long_profile_pw2, 5, (2+LPD_ROW_HEIGHT+12*(2+LPD_ROW_HEIGHT)-2),
/*g*/	(PIX_SRC|PIX_COLOR(30)), screen_r_7, "explore");
/*g*/}
/*g*/
/*g*/
