;preproc.ss
;Macro preprocessor for SLaTeX -- Scheme version
;(c) Dorai Sitaram, Oct. 1992, Rice University

(define macro$table '())

(define macro$define-transformer
  (lambda (keyword transformer)
    (let ((cell (assq keyword macro$table)))
      (if cell
          (set-cdr! cell transformer)
          (set! macro$table
            (cons (cons keyword transformer) macro$table))))))

(define macro$expand
  (lambda (e)
    (if (not (pair? e)) e
      (let* ((a (car e))
	     (c (assq a macro$table)))
	(cond (c (macro$expand (apply (cdr c) (cdr e))))
	      ((eq? a 'quote) e)
	      ((eq? a 'lambda)
	      ;so as not to trip on (... . z) style arguments
	       (cons a (cons (cadr e)
			 (map macro$expand (cddr e)))))
	      (else (map macro$expand e)))))))

(define gensym
  (let ((n -1))
    (lambda ()
      ;generates an allegedly new symbol;
      ;this is a gross hack since there is no standardized way of
      ;getting uninterned symbols
      (set! n (+ n 1))
      (string->symbol (string-append "%:g" (number->string n) "%")))))

;;some macros

;fluid-let

(macro$define-transformer 'fluid-let
  (lambda (let-pairs . body)
    (let ((x-s (map car let-pairs))
    	  (i-s (map cadr let-pairs))
    	  (old-x-s (map (lambda (p) (gensym)) let-pairs)))
      `(let ,(map (lambda (old-x x) `(,old-x ,x)) old-x-s x-s)
         ,@(map (lambda (x i) `(set! ,x ,i)) x-s i-s)
         (let ((%temp% (begin ,@body)))
    	   ,@(map (lambda (x old-x) `(set! ,x ,old-x)) x-s old-x-s)
    	   %temp%)))))

;defenum

(macro$define-transformer 'defenum
  (lambda z
    (let loop ((z z) (n 0) (r '()))
      (if (null? z) `(begin ,@r)
          (loop (cdr z) (+ n 1) 
	    (cons `(define ,(car z) (integer->char ,n)) r))))))

;defrecord

(macro$define-transformer 'defrecord
  (lambda (name . fields)
    (let loop ((fields fields) (i 0) (r '()))
      (if (null? fields)
	  `(begin (define ,name (lambda () (make-vector ,i)))
		  ,@r)
	  (loop (cdr fields) (+ i 1)
	    (cons `(define ,(car fields) ,i) r))))))

;of

(macro$define-transformer 'of
  (lambda (r i . z)
  (cond ((null? z) `(vector-ref ,r ,i))
	((and (eq? i '/) (= (length z) 1)) 
	 `(string-ref ,r ,(car z)))
	(else `(of (vector-ref ,r ,i) ,@z)))))

;setf

(macro$define-transformer 'setf
  (lambda (l r)
  (if (symbol? l) `(set! ,l ,r)
    (let ((a (car l)))
      `(,(cond ((eq? a 'list-ref) 'list-set!)
	       ((eq? a 'string-ref) 'string-set!)
	       ((eq? a 'vector-ref) 'vector-set!)
	       ((eq? a 'of) 'the-setter-for-of)
	       (else (lerror 'setf)))
	,@(cdr l) ,r)))))

;the-setter-for-of

(macro$define-transformer 'the-setter-for-of
  (lambda (r i j . z)
  (cond ((null? z) `(vector-set! ,r ,i ,j))
	((and (eq? i '/) (= (length z) 1))
	 `(string-set! ,r ,j ,(car z)))
	(else `(the-setter-for-of (vector-ref ,r ,i) ,j ,@z)))))

;extract-if(-not)

(macro$define-transformer 'extract-if
  (lambda (dialects . body)
    (if (memq *dialect* dialects)
        `(begin ,@body) #f)))

(macro$define-transformer 'extract-if-not
  (lambda (dialects . body)
    (if (not (memq *dialect* dialects))
        `(begin ,@body) #f)))

;module, extern

(define *module* 
  ;current module is initially '()
  '())

(define module$mload-needed? #f)

(define module$table 
  ;module$table is an alist.  Each alist entry consists of
  ;a module spec with a sub-alist.  Each sub-alist entry is
  ;a symbol pair
  (list (cons '() '())))

(define module$goto
  (lambda (m)
	;(usr local lib) refers to the module lib in
	;local in usr -- hence the reverse below.  Modules
	;form a tree structure.
    (let* ((m (reverse m))
	   (c (assoc m module$table)))
      (if (not c)
	  (set! module$table
	    (cons (cons m '()) module$table)))
      (set! *module* m))))

(define module$update
  (lambda (e)
    (if (pair? e)
	(let ((a (car e)))
	  (cond ((eq? a 'module)
		 (if (not module$mload-needed?)
			 ;optimization for mload, q.v.
		     (set! module$mload-needed? #t))
		 (module$goto (cdr e)))
		((eq? a 'define)
		 (let ((b (cadr e)))
		   (module$alist-update *module*
		     *module* (if (pair? b) (car b) b))))
		((memq a '(define-macro! define-syntax kludge))
		 (module$alist-update *module* *module*
		   (cadr e)))
		((eq? a 'extend-syntax)
		 (module$alist-update *module* *module*
		   (caadr e)))
		((eq? a 'extern)
		 (let ((ext-module (cadr e)))
		   (for-each
		     (lambda (x)
		       (module$alist-update *module* 
			 ext-module x))
		     (cddr e))))
		((memq a '(begin if))
		 ;add progn for CL
		 (for-each module$update (cdr e)))
		((eq? a 'cond)
		 (for-each
		   (lambda (clause)
		     (for-each module$update clause))
		   (cdr e)))
		(else #f))))))
	  
(define module$alist-update
  (lambda (curr-module ext-module x)
    (let ((curr-alist (assoc curr-module module$table)))
      (if (not (assq x (cdr curr-alist)))
	  (set-cdr! curr-alist
	    (cons (cons x
		    (string->symbol
		      (string-append
			(apply string-append
			  (reverse!
			    (map symbol->string ext-module)))
			(symbol->string x))))
	      (cdr curr-alist)))))))

(define module$translate
  (lambda (e)
    (cond ((pair? e)
	   (let ((a (car e)))
	     (cond ((eq? a 'module)
		    (module$goto (cdr e))
		    #f)
		   ((eq? a 'extern) #f)
		   ((eq? a 'scheme$) (cadr e))
		   (else
		     (cons (module$translate (car e))
		       (module$translate (cdr e)))))))
	  ((symbol? e)
	   (module$symbol-fullname e *module*))
	  (else e))))

(define module$symbol-fullname
  (lambda (x mdl)
    ;unless externed, a module's symbols are visible only
    ;to itself and its submodules
    (let loop ((mdl mdl))
      (let ((c (assq x (cdr (assoc mdl module$table)))))
	(cond (c (cdr c))
	      ((null? mdl) x)
	      (else (loop (cdr mdl))))))))    

(define module$call-on-file
  (lambda (file proc)
    (call-with-input-file file
      (lambda (inp)
	(let loop ()
	  (let ((e (read inp)))
	    (if (not (eof-object? e))
		(begin (proc e) (loop)))))))))
