;preproc.cl
;Macro preprocessor for SLaTeX -- CL version
;(c) Dorai Sitaram, Nov. 1992

; convert lambda, let, ... to scheme-lambda, scheme-let, ...

(defun macro$noclash (e)
  (if (not (pair? e)) e
      (let ((a (car e)))
        (if (eq? a 'quote) e
            (cons
              (cond ((pair? a) (macro$noclash a))
		    ((eq? a 'lambda) 'scheme-lambda)
               	    ((eq? a 'let) 'scheme-let)
               	    ((eq? a 'loop) 'loop-scheme)
               	    ((eq? a 'assoc) 'scheme-assoc)
               	    ((eq? a 'member) 'scheme-member)
               	    ((eq? a 'map) 'scheme-map)
               	    ((eq? a 'make-string) 'scheme-make-string)
               	    ((eq? a 'peek-char) 'scheme-peek-char)
               	    ((eq? a 'read) 'scheme-read)
               	    ((eq? a 'read-char) 'scheme-read-char)
               	    ((eq? a 'setf) 'scheme-setf)
               	    ((eq? a 'string) 'scheme-string)
               	    ((eq? a 'write) 'scheme-write)
               	    (else a))
              (macro$noclash (cdr e)))))))

; macro preprocessor

(defvar macro$table '())

(defun macro$define-transformer (keyword transformer)
  (let ((cell (assq keyword macro$table)))
    (if cell
        (setf (cdr cell) transformer)
        (setq macro$table
          (cons (cons keyword transformer) macro$table)))))

(defun macro$expand (e)
  (if (not (pair? e)) e
      (let* ((a (car e))
             (c (assq a macro$table)))
         (cond (c (macro$expand (apply (cdr c) (cdr e))))
               ((eq? a 'quote) e)
               (else (scheme-map #'macro$expand e))))))

(defun macro$insert-rest (xx)
  ;change the ". z" format of scheme lambda to 
  ;the "&rest z" format of cl lambda
  (let ((yy '()))
    (loop
      (cond ((null? xx) (return))
	    ((symbol? xx) 
	     (setq yy (cons xx (cons '&rest yy))) (return))
	    ((pair? xx)
	     (setq yy (cons (car xx) yy))
	     (setq xx (cdr xx)))
	    (else (error "insert-rest"))))
    (reverse! yy)))

; insert funcalls in Scheme code

;dispatch on lambda, let, cond, function,  eval-when
;lambda applications
;applications

(defun macro$insert-funcalls (e &optional bvs)
  (if (not (pair? e)) e
    (let ((a (car e)))
      (cond ((and (pair? a) (eq? (car a) 'lambda))
             (macro$insert-funcalls-m e bvs))
	    ((or (not (symbol? a)) (memq a bvs))
	     (cons 'funcall (macro$insert-funcalls-m e bvs)))
	    ((eq? a 'quote) e)
	    ((eq? a 'lambda)
	     (let* ((new-bvs (cadr e))
	            (ext-bvs (append new-bvs bvs)))
  	       `(function
  	          (lambda ,new-bvs
 	            ,@(macro$insert-funcalls-m (cddr e) ext-bvs)))))
            ((eq? a 'function)
             (let ((d (macro$insert-funcalls (cadr e) bvs)))
               (if (and (pair? d) (eq? (car d) 'function)) d
                   (list 'function d)))) 	            
	    ((eq? a 'funcall)
	     (macro$insert-funcalls-m e bvs))
            ((eq? a 'cond)
             `(cond ,@(scheme-map
			#'(lambda (c) (macro$insert-funcalls-m c bvs))
                        (cdr e))))
 	     ((eq? a 'let)
              (let* ((new-bvs (scheme-map #'car (cadr e)))
                     (ext-bvs (append new-bvs bvs)))
                 `(let
                    ,(scheme-map #'(lambda (xv)
                                 (let ((x (car xv)) (v (cadr xv)))
                                   `(,x ,(macro$insert-funcalls v bvs))))
                          (cadr e))
                     ,@(macro$insert-funcalls-m (cddr e) ext-bvs))))
	     ((memq a '(eval-when do-all-symbols))
	      `(,a ,(cadr e)
	         ,@(macro$insert-funcalls-m (cddr e) bvs)))
	     (else (macro$insert-funcalls-m e bvs))))))
	        
(defun macro$insert-funcalls-m (ee &optional bvs)
  (scheme-map #'(lambda (e) (macro$insert-funcalls e bvs)) ee))

;macros

(macro$define-transformer 'scheme-lambda 
  #'(lambda (v &rest z)
      `(function 
         (lambda ,(macro$insert-rest v) ,@z))))

(macro$define-transformer 'set!
  #'(lambda (x v)
      `(setq ,x ,v)))

(macro$define-transformer 'begin
  #'(lambda (&rest z)
      `(progn ,@z)))

(macro$define-transformer 'define
  #'(lambda (x v)
      `(begin
	 (kludge ,x)
	 (set! ,x ,v)
         (cond ((symbol? ,x) 'void)
               ;CLtLII's functionp may returns t on symbols
               ;and some pairs
               ((functionp ,x) (setf (symbol-function ',x) ,x))
               (else 'void)))))

(macro$define-transformer 'letrec
  #'(lambda (pp &rest b)
      `(let ,(scheme-map #'(lambda (p) `(,(car p) 'void)) pp)
         ,@(scheme-map #'(lambda (p) `(set! ,(car p) ,(cadr p))) pp)
         ,@b)))

(defun tempsym (&optional d)
  (intern (symbol-name (gensym))))

(macro$define-transformer 'tail-recur
  #'(lambda (n let-pairs &rest b)
      (let* ((x-s (scheme-map #'car let-pairs))
             (y-s (scheme-map #'tempsym x-s))
             (tag (tempsym)))
         `(let ,let-pairs
            (let ((,n (lambda ,y-s
              		  ,@(scheme-map #'(lambda (x y) `(setq ,x ,y))
              		      x-s y-s)
              		   (throw ',tag 'void))))
               (loop
                 (catch ',tag (return (begin ,@b)))))))))

(macro$define-transformer 'recur 
  #'(lambda (name let-pairs &rest body)
    `(letrec ((,name (function (lambda ,(scheme-map #'car let-pairs) 
			,@body))))
       (funcall ,name ,@(scheme-map #'cadr let-pairs)))))
              		      
(macro$define-transformer 'scheme-let
  #'(lambda (a &rest b)
    (cond ((and a (not (symbol? a))) `(let ,a ,@b))
          ((let ((s (symbol-name a)))
             (and (>= (length s) 4)
                  (char-equal (char s 0) #\l)
                  (char-equal (char s 1) #\o)
                  (char-equal (char s 2) #\o)
                  (char-equal (char s 3) #\p)))
            `(tail-recur ,a ,@b))
          (else `(recur ,a ,@b)))))

(macro$define-transformer 'let*
  #'(lambda (let-pairs &rest body)
      (if (null? let-pairs) `(begin ,@body)
          `(let ((,(caar let-pairs) ,(cadar let-pairs)))
             (let* ,(cdr let-pairs) ,@body)))))

(macro$define-transformer 'fluid-let
  #'(lambda (let-pairs &rest body)
      `(let ,let-pairs
         (declare (special ,@(scheme-map #'car let-pairs)))
         ,@body)))

'(macro$define-transformer 'fluid-let
  #'(lambda (let-pairs &rest body)
      (let ((x-s (scheme-map #'car let-pairs))
      	    (i-s (scheme-map #'cadr let-pairs))
      	    (old-x-s (scheme-map #'tempsym let-pairs)))
      	`(let ,(scheme-map #'(lambda (old-x x) `(,old-x ,x)) old-x-s x-s)
      	   ,@(scheme-map #'(lambda (x i) `(set! ,x ,i)) x-s i-s)
      	   (let ((%temp% (begin ,@b)))
      	     ,@(scheme-map #'(lambda (x old-x) `(set! ,x ,old-x)) x-s old-x-s)
      	     %temp%)))))

 (macro$define-transformer 'defenum
  #'(lambda (&rest z)
     (do ((z z (cdr z))
          (n 0 (+ n 1))
          (r '() (cons `(define ,(car z) (integer->char ,n)) r)))
        ((null? z) `(begin ,@r)))))

(macro$define-transformer 'defrecord
  #'(lambda (name &rest fields)
      (do ((fields fields (cdr fields))
           (i 0 (+ i 1))
           (r '() (cons `(defvar ,(car fields) ,i) r)))
         ((null? fields)
          `(begin (define ,name (lambda () (make-vector ,i)))
          	  ,@r)))))

(macro$define-transformer 'of
  #'(lambda (r i &rest z)
      (cond ((null? z) `(vector-ref ,r ,i))
      	    ((and (eq? i '/) (= (length z) 1))
      	     `(string-ref ,r ,(car z)))
      	    (else `(of (vector-ref ,r ,i) ,@z)))))

(macro$define-transformer 'scheme-setf
  #'(lambda (l r)
      (if (symbol? l) `(set! ,l ,r)
          (let ((a (car l)))
            `(,(cond ((eq? a 'list-ref) 'list-set!)
            	     ((eq? a 'string-ref) 'string-set!)
            	     ((eq? a 'vector-ref) 'vector-set!)
            	     ((eq? a 'of) 'the-setter-for-of)
            	     (else (error "setf")))
              ,@(cdr l) ,r)))))

(macro$define-transformer 'the-setter-for-of
  #'(lambda (r i j &rest z)
      (cond ((null? z) `(vector-set! ,r ,i ,j))
      	    ((and (eq? i '/) (= (length z) 1))
      	     `(string-set! ,r ,j ,(car z)))
      	    (else `(the-setter-for-of (vector-ref ,r ,i) ,j ,@z)))))


;extract-if(-not)

(macro$define-transformer 'extract-if
  #'(lambda (dialects &rest body)
    (if (memq *dialect* dialects)
        `(begin ,@body) #f)))

(macro$define-transformer 'extract-if-not
  #'(lambda (dialects &rest body)
    (if (not (memq *dialect* dialects))
        `(begin ,@body) #f)))

;module, extern

(defvar *module* 
  ;current module is initially '()
  '())

(defvar module$mload-needed? #f)

(defvar module$table 
  ;module$table is an alist.  Each alist entry consists of
  ;a module spec with a sub-alist.  Each sub-alist entry is
  ;a symbol pair
  (list (cons '() '())))

(defun module$goto (m)
	;(usr local lib) refers to the module lib in
	;local in usr -- hence the reverse below.  Modules
	;form a tree structure.
  (let* ((m (reverse m))
	 (c (scheme-assoc m module$table)))
    (if (not c)
	(setq module$table
	  (cons (cons m '()) module$table)))
    (setq *module* m)))

(defun module$update (e)
  (if (pair? e)
      (let ((a (car e)))
	(cond ((eq? a 'module)
	       (if (not module$mload-needed?)
			 ;optimization for mload, q.v.
		   (setq module$mload-needed? #t))
	       (module$goto (cdr e)))
	      ((eq? a 'define)
	       (let ((b (cadr e)))
		 (module$alist-update *module*
		   *module* (if (pair? b) (car b) b))))
	      ((memq a '(define-macro! define-syntax kludge))
	       (module$alist-update *module* *module*
		 (cadr e)))
	      ((eq? a 'extend-syntax)
	       (module$alist-update *module* *module*
		 (caadr e)))
	      ((eq? a 'extern)
	       (let ((ext-module (cadr e)))
		 (for-each
		   #'(lambda (x)
		     (module$alist-update *module* 
		       ext-module x))
		   (cddr e))))
	      ((memq a '(begin if progn)) 
	       (for-each #'module$update (cdr e)))
	      ((eq? a 'cond)
	       (for-each
		 #'(lambda (clause)
		   (for-each #'module$update clause))
		 (cdr e)))
	      ((eq? a 'quote) ;kludge
	       (module$update (cadr e)))
	      (else #f)))))
	  
(defun module$alist-update (curr-module ext-module x)
  (let ((curr-alist (scheme-assoc curr-module module$table)))
    (if (not (assq x (cdr curr-alist)))
	(set-cdr! curr-alist
	  (cons (cons x
		  (string->symbol
		    (string-append
		      (apply #'string-append
			(reverse!
			  (scheme-map #'symbol->string ext-module)))
		      (symbol->string x))))
	    (cdr curr-alist))))))

(defun module$translate (e)
  (cond ((pair? e)
	 (let ((a (car e)))
	   (cond ((eq? a 'module)
		  (module$goto (cdr e))
		  #f)
		 ((eq? a 'extern) #f)
		 ((eq? a 'scheme$) (cadr e))
		 (else
		   (cons (module$translate (car e))
		     (module$translate (cdr e)))))))
	((symbol? e)
	 (module$symbol-fullname e *module*))
	(else e))))

(defun module$symbol-fullname (x mdl)
    ;unless externed, a module's symbols are visible only
    ;to itself and its submodules
  (loop
    (let ((c (assq x (cdr (scheme-assoc mdl module$table)))))
      (cond (c (return (cdr c)))
	    ((null? mdl) (return x))
	    (else (setq mdl (cdr mdl)))))))

(defun module$call-on-file (file proc)
  (call-with-input-file file
    #'(lambda (inp)
	(loop
	  (let ((e (scheme-read inp)))
	    (if (not (eof-object? e))
		(funcall proc e)
		(return #t))))))))


