/* Fill prefix and comment commands for Epsilon.

   (c) Copyright 1990 Carl W. Hoffman.  All rights reserved.

   This file may be freely copied, distributed, or modified for non-commercial
   use provided that this copyright notice is not removed.  For further
   information, contact the following address:

   Carl W. Hoffman, 363 Marlborough Street, Boston, MA 02115, U.S.A.
   Internet: CWH@AI.MIT.EDU    CompuServe: 76416,3365    Fax: 617-262-4284

   This code has been tested with Epsilon version 4.13. */

#include <eel.h>

buffer char fill_prefix[60];

buffer short comment_column = 60;
buffer char comment_start[10];
buffer char comment_end[10];

when_loading()
{
  strcpy(fill_prefix, "");
  strcpy(fill_prefix.default, "");
  strcpy(comment_start, "// ");
  strcpy(comment_start.default, "// ");
  strcpy(comment_end, "");
  strcpy(comment_end.default, "");
  }

/* To be compatible with standard implementations of fill prefix, only when
   the fill prefix is empty should forward_paragraph and backward_paragraph be
   used to determine the paragraph to be filled.  If a fill prefix is defined,
   the paragraph is defined to be a contiguous group of lines having the same
   fill prefix.  The forward_paragraph and backward_paragraph commands should
   observe this convention, also. */

#define at_beginning_of_line(pos) \
  (((pos) == 0) || (character((pos)-1) == '\n'))

command fill_paragraph() on reg_tab[ALT('q')]
{
  int end, start=point;
  int prefix_length = strlen(fill_prefix);

  iter = 0;
  forward_paragraph();
  end = point - 1;
  backward_paragraph(); 
  if (!prefix_length) {
    /* Leave leading whitespace intact */
    re_search(1, "[ \t\n]*");
    region_fill(point, end);
    }
  else {

    int prefix_seen = 0;
    int fill_start = point;
    int *fill_end = alloc_spot();
    *fill_end = end;

    /* Remove existing fill prefixes, if any exist. */

    while (search(1, fill_prefix)) {
      if (!(point < *fill_end))
        break;
      if (at_beginning_of_line(matchstart)) {
        if (point < start)
          start -= prefix_length;
        else if (matchstart < start)
          start = matchstart;
        delete(matchstart, point);
        prefix_seen = 1;
        }
      }

    /* Fill the region.  Even though fill_end is a spot,
       it is left at the beginning of the last line of the
       filled region.  Therefore, move it to the end of the
       line before putting the fill prefixes back. */

    if (prefix_seen)
      margin_right -= prefix_length;
    region_fill(fill_start, *fill_end);
    point = *fill_end;
    to_end_line();
    *fill_end = point;

    /* If at least one fill prefix was removed from the region,
       then put them back on every line in the region. */

    if (prefix_seen) {
      margin_right += prefix_length;
      point = fill_start;
      while (point < *fill_end) {
        if (point < start)
          start += prefix_length;
        stuff(fill_prefix);
        nl_forward();
        }
      }
    free_spot(fill_end);
    }

  if (start > size()) start = size();
  point = start;
  }

command set_fill_prefix() on cx_tab['.']
{
  int start=point;
  to_begin_line();
  grab(point, start, fill_prefix);
  say("Fill prefix set to %s", fill_prefix);
  }

command set_comment_column() on cx_tab[';']
{
  comment_column = current_column();
  say("Comment column set to %d.", comment_column);
  }

/* This could be more clever in several ways:
   1. It shouldn't scan to the end of the buffer looking for a comment.
   2. It should ignore whitespace at the end of comment_start when looking
      for a match.
   3. It shouldn't be fooled by the comment characters appearing in
      quoted strings. */

command to_comment() on reg_tab[ALT(';')]
{
  if (strlen(comment_start) > 0) {
    int start = point;
    int line_end;
    to_end_line();
    line_end = point;
    to_begin_line();
    if (!search(1, comment_start) || (point > line_end)) {

      /* Insert a new comment */

      point = line_end;
      to_column(comment_column);
      stuff(comment_start);
      start = point;
      stuff(comment_end);
      point = start;
      }
    else {

      /* Reposition the existing comment */

      int delta;
      if (point > start) start = point;
      delta = line_end - start;
      point = matchstart;
      to_column(comment_column);
      to_end_line();
      point -= delta;
      }
    }
  }

/* If we are in an empty comment, kill it. */

kill_empty_comment()
{
  /* If we're not immediately after the comment start string, give up. */

  int start = point;
  if (!(search(-1, comment_start) && (matchstart == start))) {
    point = start;
    return;
    }

  if (strlen(comment_end) == 0) {

    /* No comment end string is defined.  If the comment start string is at
       the end of the line, kill the comment. */

    if ((character(start) == '\n') || (start == size())) {
      delete(point, start);
      delete_horizontal_space();
      }
    }
  else {

    /* A comment end string is defined.  If we are immediately before it,
       kill the comment. */

    int kill_start = point;
    if (search(1, comment_end) && (matchstart == start) && 
        ((curchar() == '\n') || (point == size()))) {
      delete(kill_start, point);
      delete_horizontal_space();
      }
    }
  }

command next_comment_line() on reg_tab[ALT('n')]
{
  if (strlen(comment_start) == 0)
    down_line();
  else {
    kill_empty_comment();
    nl_forward();
    to_comment();
    }
  }

command previous_comment_line() on reg_tab[ALT('p')]
{
  if (strlen(comment_start) == 0)
    up_line();
  else {
    kill_empty_comment();
    nl_reverse();
    to_begin_line();
    to_comment();
    }
  }

suffix_bat() { bat_mode(); }

command bat_mode ()
{
  margin_right = 79;
  fill_mode = 0;
  strcpy(fill_prefix, "rem ");
  major_mode = "DOS Batch";
  make_mode();
  }

suffix_asp() { asp_mode(); }

command asp_mode()
{
  margin_right = 79;
  fill_mode = 0;
  strcpy(fill_prefix, ";; ");
  strcpy(comment_start, "; ");
  major_mode = "Procomm Plus ASP";
  make_mode();
  }

/* One of c_mode or make_mode must be called at the end of this command.
   Otherwise, the mode line will be incorrect. */

better_c_mode(comment_type)
short comment_type;
{
  margin_right = 79;
  fill_mode = 0;
  strcpy(fill_prefix, "// ");
  if (comment_type) {
    strcpy(comment_start, "/* ");
    strcpy(comment_end, " */");
    }
  c_mode();
  }

suffix_c()   { better_c_mode(0); }
suffix_h()   { better_c_mode(0); }
suffix_e()   { better_c_mode(1); }
suffix_pre() { better_c_mode(0); }
suffix_rc()  { better_c_mode(0); }
suffix_y()   { better_c_mode(0); }
