% Notes on using DEFABS for building abstractions in Scheme.
% Nikhil, September 19, 1988

\documentstyle[11pt]{article}

% HORIZONTAL MARGINS
% Left margin 1 inch (0 + 1)
\setlength{\oddsidemargin}{0in}
% Text width 6.5 inch (so right margin 1 inch).
\setlength{\textwidth}{6.5in}
% ----------------
% VERTICAL MARGINS
% Top margin 0.5 inch (-0.5 + 1)
\setlength{\topmargin}{-0.5in}
% Head height 0.25 inch (where page headers go)
\setlength{\headheight}{0.25in}
% Head separation 0.25 inch (between header and top line of text)
\setlength{\headsep}{0.25in}
% Text height 9 inch (so bottom margin 1 in)
\setlength{\textheight}{9in}
% ----------------
% PARAGRAPH INDENTATION
\setlength{\parindent}{0in}
% SPACE BETWEEN PARAGRAPHS
\setlength{\parskip}{\medskipamount}
% ----------------
% STRUTS
% HORIZONTAL STRUT.  One argument (width).
\newcommand{\hstrut}[1]{\hspace*{#1}}
% VERTICAL STRUT. Two arguments (offset from baseline, height).
\newcommand{\vstrut}[2]{\rule[#1]{0in}{#2}}
% ----------------
% EMPTY BOXES OF VARIOUS WIDTHS, FOR INDENTATION
\newcommand{\hm}{\hspace*{1em}}
\newcommand{\hmm}{\hspace*{2em}}
\newcommand{\hmmm}{\hspace*{3em}}
\newcommand{\hmmmm}{\hspace*{4em}}
% ----------------
% VARIOUS CONVENIENT WIDTHS RELATIVE TO THE TEXT WIDTH, FOR BOXES.
\newlength{\hlessmm}
\setlength{\hlessmm}{\textwidth}
\addtolength{\hlessmm}{-2em}

\newlength{\hlessmmmm}
\setlength{\hlessmmmm}{\textwidth}
\addtolength{\hlessmmmm}{-4em}
% ----------------
% ``TIGHTLIST'' ENVIRONMENT (no para space betwee items, small indent)
\newenvironment{tightlist}%
{\begin{list}{$\bullet$}{%
    \setlength{\topsep}{0in}
    \setlength{\partopsep}{0in}
    \setlength{\itemsep}{0in}
    \setlength{\parsep}{0in}
    \setlength{\leftmargin}{1.5em}
    \setlength{\rightmargin}{0in}
    \setlength{\itemindent}{0in}
}
}%
{\end{list}
}
% ----------------
% CODE FONT (e.g. {\cf x := 0}).
\newcommand{\cf}{\footnotesize\tt}
% ----------------------------------------------------------------
% LISP CODE DISPLAYS.
% Lisp code displays are enclosed between \bid and \eid.
% Most characters are taken verbatim, in typewriter font,
% Except:
%  Commands are still available (beginning with \)
%  Math mode is still available (beginning with $)

\outer\def\blisp{%
  \begin{list}{$\bullet$}{%
    \setlength{\topsep}{0in}
    \setlength{\partopsep}{0in}
    \setlength{\itemsep}{0in}
    \setlength{\parsep}{0in}
    \setlength{\leftmargin}{1.5em}
    \setlength{\rightmargin}{0in}
    \setlength{\itemindent}{0in}
  }\item[]
  \obeyspaces
  \obeylines \footnotesize\tt}

\outer\def\elisp{%
  \end{list}
  }

{\obeyspaces\gdef {\ }}

% ----------------------------------------------------------------
% ILLUSTRATIONS
% This command should specify a NEWT directory for ps files for illustrations.
\def\psfileprefix{/jj/nikhil/ps/}
\def\illustration#1#2{
\vbox to #2{\vfill\special{psfile=\psfileprefix#1.ps hoffset=-72 voffset=-45}}} 

% \illuswidth is used to set up boxes around illustrations.
\newlength{\illuswidth}
\setlength{\illuswidth}{\textwidth}
\addtolength{\illuswidth}{-7pt}
% ----------------------------------------------------------------
% ----------------------------------------------------------------
% HERE BEGINS THE DOCUMENT

\begin{document}

\begin{center}
MASSACHUSETTS INSTITUTE OF TECHNOLOGY \\
Department of Electrical Engineering and Computer Science \\
6.001 Structure and Interpretation of Computer Programs \\
Fall Semester, 1988

\bigskip

{\large\bf Using {\tt defabs} to Automate the Building of Abstractions}

\end{center}

\section{Introduction}

To create a new abstraction, we typically want to define the following things:
\begin{tightlist}

\item A {\em constructor\/} to create new instances of abstract objects.  A
constructor may be a constant, or it may be a function that takes one or more
arguments.

\item A {\em predicate\/} that tests whether a given object belongs to that abstraction.

\item Zero or more {\em selectors\/}. Each selector, given an abstract object,
picks out a component of the object.

\end{tightlist}
In this way, the actual representation of the object is completely hidden
(that's why it's abstract).

For example, in a database of employees, suppose we want to record ``how an
employee travels to work''.  Suppose there are two possibilities:  by bicycle
or by automobile. In the latter case, we also wish to record the license
plate number, represented as a string of characters.  We would like to define
the following things:

\begin{center}
\begin{tabular}{|l|l|p{3in}|}
\hline
{\cf bicycle}  &
Constructor &
A constant (i.e., niladic constructor) representing ``bicycle''. \\
\hline
{\cf (bicycle? x) } &
Predicate &
Tests if an object {\cf x} is the ``bicycle'' object. \\
\hline
\hline
{\cf (auto s) } &
Constructor &
Given a string {\cf s} representing a license plate number,  returns a new
``automobile'' containing it. \\
\hline
{\cf (auto? x)} &
Predicate &
Tests if an object {\cf x} is an ``automobile''. \\
\hline
{\cf (auto-lic-plate-num x)} &
Selector &
Given an ``automobile'' {\cf x}, returns its license plate number (a string). \\
\hline
\end{tabular}
\end{center}

\section{Why Build Abstractions?}

Instead of defining the things in the table above, suppose we used
{\cf nil} to represent ``bicycle'', and a string (the license plate number)
to represent an ``automobile''.  Then, we can use {\cf nil?} to test if
something is a bicycle, {\cf string?} to test if it is an automobile, and the
``automobile'' itself {\em is\/} the license plate number.

Unfortunately, in addition to making the program more obscure (less
readable), these direct, visible encodings also make the program more
difficult to debug and maintain.  For example, suppose we want to change the
representation, e.g., we now want ``bicycles'' and ``automobiles'' to also
include their make, model and color.  We no longer can use {\cf nil?} and
{\cf string?} to test for bicycles and automobiles.  This will probably have
a drastic effect on our program, involving widespread changes.

The key thing to do in defining abstractions is to {\em hide the
representation\/}, i.e., the {\em only\/} way one should manipulate abstract
objects is through a set of ``certified'' procedures specific to that
abstraction--- constructors, predicates and selectors.  That way, in addition
to making things more readable, any future decision to change the
representation implies only a local change to those certified procedures.

\section{Automating it}

Unfortunately, defining constructors, predicates and selectors for each
abstraction is generally tedious and boring, so we often tend to break our
discipline about abstractions, and, for expedience, use direct, visible
encodings.

To solve this problem, we have defined a new procedure called ``{\cf
defabs}'' to automate the process of defining abstractions. 
For example, the above abstractions can be defined using two {\cf
defabs} statements:
\blisp
(defabs 'bicycle)
(defabs '(auto lic-plate-num))
\elisp

In general, there are two forms that {\cf defabs} expects--- a quoted symbol
or a quoted list of symbols.  The former is used for niladic constructors,
i.e., objects with no components, whereas the latter is used for objects that
do have components.

\begin{center}
\begin{tabular}{|l|l|l|}
\hline
Form                & \multicolumn{2}{c|}{Defines} \\
\hline
\hline
{\cf (defabs 'foo)} & Constructor & {\cf foo} \hmm (a constant) \\
                    & Predicate   & {\cf (foo? x)} \\
\hline
{\cf (defabs '(bar component-1 ... component-N)) }
                    & Constructor & {\cf (bar e1 ... eN)} \\
                    & Predicate   & {\cf (bar? x)} \\
                    & Selector    & {\cf (bar-component-1 x)} \\
                    & ...         & ...                       \\
                    & Selector    & {\cf (bar-component-N x)} \\
\hline
\end{tabular}
\end{center}
Notice how the names of the constructors, predicates and selectors are
derived from the names in the argument to {\cf defabs}.

\section{Examples}

In each of the following examples, observe that there is absolutely no
mention of the actual representation of the abstract object.

\subsection{Example 1: Symbolic Arithmetic Expressions}

Suppose we want to represent arithmetic expressions consisting of additions,
multiplications, and negations of variables and numbers.  Here is the
definition of the abstractions:
\blisp
(defabs '(var x))
(defabs '(number x))
(defabs '(add x y))
(defabs '(mult x y))
(defabs '(neg x))
\elisp

Now, let's try some programs that use the abstraction. First,  some example
expressions.  The expression ``$2 \times 3 + (-4) \times 5$'':
\blisp
(define test-exp-1
     (add (mult (number 2)
                (number 3))
          (mult (neg (number 4))
                (number 5))))
\elisp
The expression ``$2x + (-x)y$'':
\blisp
(define test-exp-2
     (add (mult (number 2)
                (var 'x))
          (mult (neg (var 'x))
                (var 'y))))
\elisp

Now, a function to differentiate an expression {\cf e} with respect to a
variable {\cf x}:
\blisp
(define (differentiate e x)
    (cond
      ((var?    e) (if (eq? (var-x e) (var-x x))
                      (number 1)
                      (number 0)))
      ((number? e) (number 0))
      ((add?    e) (add  (differentiate (add-x e) x)
                         (differentiate (add-y e) x)))
      ((mult?   e) (add (mult (mult-x e)
                              (differentiate (mult-y e) x))
                        (mult (differentiate (mult-x e) x)
                              (mult-y e))))
      ((neg?    e) (neg (differentiate (neg-x e) x)))
      (else        (error "differentiate: argument is not an expression"))))
\elisp
Finally, a program that takes an arithmetic expression and produces a new
arithmetic expression in ``reduced form''.  For example, sub-expressions
involving only numbers (no variables) are evaluated, $e \times 0$ is replaced
by $0$, $e \times 1$ is replaced by $e$, $e + 0$ is replaced by $e$, etc.
\blisp
(define (reduce e)
    (cond
      ((var?    e) e)
      ((number? e) e)
      ((add?    e) (reduce-add  (reduce (add-x e))  (reduce (add-y e))))
      ((mult?   e) (reduce-mult (reduce (mult-x e)) (reduce (mult-y e))))
      ((neg?    e) (reduce-neg  (reduce (neg-x e))))
      (else        (error "reduce: argument is not an expression"))))
\elisp

\blisp
(define (is-number e n)
    (and (number? e) (= (number-x e) n)))
\elisp

\blisp
(define (reduce-add e1 e2)
    (cond
      ((and (number? e1) (number? e2)) (number (+ (number-x e1) (number-x e2))))
      ((is-number e1 0) e2)
      ((is-number e2 0) e1)
      (else (add e1 e2))))
\elisp

\blisp
(define (reduce-mult e1 e2)
    (cond
      ((and (number? e1) (number? e2)) (number (* (number-x e1) (number-x e2))))
      ((is-number e1 0) e1)
      ((is-number e1 1) e2)
      ((is-number e2 0) e2)
      ((is-number e2 1) e1)
      (else (mult e1 e2))))
\elisp

\blisp
(define (reduce-neg e)
    (cond
      ((number? e) (number (- (number-x e))))
      (else (neg e))))
\elisp

\subsection{Example 2: Binary Trees}

A binary tree is a recursive structure that is either empty or a node
containing three things:  a value, a left subtree and a right subtree.  The
subtrees are themselves binary trees.  Here is the definition:
\blisp
(defabs 'empty)
(defabs '(node val left right))
\elisp
An example tree:
\blisp
(define test-tree-1
    (node 100
          (node 50
                empty
                (node 75
                      (node 60
                            empty
                            empty)
                      (node 90
                            empty
                            empty)))
          (node 200
                (node 150
                      empty
                      empty)
                empty)))
\elisp
Pictorially, this tree may be depicted as in Figure \ref{defabs-btree}, where
each box depicts a node, and empty subtrees are shown with a line diagonally
across the slot, in the same way that we depict {\cf nil} in list
box-and-pointer diagrams. 

\begin{figure}[htbp]
\fbox{%
  \parbox%
    {\illuswidth}
    {\illustration{defabs-btree}{2.3 true in}
     \caption{%
        \label{defabs-btree}
        Structure of ``{\tt test-tree-1}''
             }
    }}
\end{figure}

A procedure to add up all the numbers in a binary tree:
\blisp
(define (addup-tree t)
    (cond
      ((empty? t) 0)
      ((node?  t) (+ (node-val t)
                     (addup-tree (node-left t))
                     (addup-tree (node-right t))))
      (else       (error "addup-tree: argument is not a tree"))))
\elisp

A procedure to ``reflect'' a tree, i.e. produce a left/right mirror-image:
\blisp
(define (reflect-tree t)
    (cond
      ((empty? t) t)
      ((node?  t) (node (node-val t)
                        (reflect-tree (node-right t))
                        (reflect-tree (node-left  t))))
      (else       (error "reflect-tree: argument is not a tree"))))
\elisp

A {\em binary search tree\/} is a binary tree with the additional
constraint that for any node with value $v$ and left and right subtrees $l$
and $r$, respectively,
\[
v_l < v < v_r \hmmmm {\rm forall} \hm v_l \hm {\rm in} \hm l, \hm v_r \hm
{\rm in} \hm r
\]
i.e. all values in the left subtree are smaller than $v$, which is itself
smaller than all values in the right subtree.  In fact, {\cf test-tree-1} is
organized as a binary search tree.

Binary search trees can be used to sort lists:
\blisp
;;; insert one element into a binary search tree
\null
(define (insert x t)
    (cond
      ((empty? t) (node x empty empty))
      ((node?  t) (cond
                    ((< x (node-val t)) (node (node-val t)
                                              (insert x (node-left t))
                                              (node-right t)))
                    ((> x (node-val t)) (node (node-val t)
                                              (node-left t)
                                              (insert x (node-right t))))
                    (else               t)))))
\null
;;; insert all elements from a list into an binary search tree
\null
(define (insert-list-into-tree l t)
    (if (null? l)
        t
        (insert-list-into-tree (cdr l) (insert (car l) t))))
\null
;;; produce a list of the elements in a tree, using in-order traversal
\null
(define (inorder-into-list t l)
    (cond
      ((empty? t) l)
      ((node?  t) (inorder-into-list (node-left t)
                                     (cons (node-val t)
                                           (inorder-into-list (node-right t)
                                                              l))))))
\null
;;; Finally, the top-level sort function: build a binary search tree from the
;;; elements of the list, then produce a list of the values in the tree in
;;; order.
\null
(define (sort l)
    (inorder-into-list (insert-list-into-tree l empty)
                       '()))
\elisp

\section{Limitation of {\tt defabs}}

{\cf Defabs} only provides abstractions that are ``raw'' structures.
Sometimes you may need to wrap it in another layer to produce structures that
maintain additional properties.  For example, consider an abstraction for
fractions:
\blisp
(defabs '(frac num denom))
\elisp
Then, {\cf frac} is a constructor that takes two numbers representing the
numerator and denominator, and returns a ``fraction'' object.  However,
suppose we want always to keep fractions in reduced form, so that, for
example, given $12$ and $8$, the fraction constructed is $3/2$.  {\cf Frac}
is inadequate, since it blindly constructs a fraction out of any two numbers.
We need a new constructor:
\blisp
(define red-frac (lambda (numer denom)
    (let
        ((g (gcd numer denom)))
      (frac (quotient numer g) (quotient denom g)))))
\elisp
Thus, {\cf red-frac} is a new constructor that reduces the numbers and then
gives them to the ``raw'' constructor, which blindly constructs the fraction.

\section{Looking under the hood}

Note that in all the previous examples, there was absolutely no mention of
the representation of the various abstract objects.  In fact, to use the
abstraction, nothing more is needed.

But, of course, we're always curious about what {\em really\/} happens
underneath.  Here are the details of what the current implementation of {\cf
defabs} actually produces for the binary tree abstraction.  Keep in mind that
this is only one of many possible representations (in particular, this is not
a very efficient representation), and also, you should never rely on this
particular representation (by manipulating it directly).

Empty trees are simply represented by the symbol {\cf 'empty}.
Typing in the form \mbox{\cf (defabs 'empty)} is equivalent to typing in the
following definitions:
\blisp
(define empty  'empty)
(define empty? (lambda (x) (eq? x 'empty)))
\elisp

Nodes are represented by lists that look like this: \mbox{\cf ('node val left right)}:
Typing in the form \mbox{\cf (defabs '(node val left right))} is equivalent to typing
in the following definitions:
\blisp
(define node       (lambda (x0 x1 x2) (list 'node x0 x1 x2)))
\null
(define node?      (lambda (x) (and (pair? x) (eq? (car x) 'node))))
\null
(define node-val   (lambda (x) (list-ref x 1)))
(define node-left  (lambda (x) (list-ref x 2)))
(define node-right (lambda (x) (list-ref x 3)))
\elisp

\section{Gymnastics}

{\cf Defabs\/} happens to implement abstractions internally using {\cf
cons}, but this is not necessary.  For example, here is another
possible version of {\cf defabs}:
\blisp
(defabs '(node val left right))
\elisp
could expand into:
\blisp
(define node
    (lambda (val left right)
        (lambda (message)
           (cond
             ((eq? message 'what-are-you?)      'node)
             ((eq? message 'what-is-your-val?)   val)
             ((eq? message 'what-is-your-left?)  left)
             ((eq? message 'what-is-your-right?) right)))))
\null
(define node?      (lambda (x) (eq? (x 'what-are-you?) 'node)))
(define node-val   (lambda (x) (x 'what-is-your-val?)))
(define node-left  (lambda (x) (x 'what-is-your-left?)))
(define node-right (lambda (x) (x 'what-is-your-right?)))
\elisp

In fact, to invert things a little, we could imagine a language in which {\cf
defabs} was primitive but the traditional list operations were not.  Then, we
could define the list primitives using {\cf defabs} as follows:

\blisp
(defabs 'nil)
(defabs '(cons a b))
\elisp
This defines the constructors {\cf nil} and {\cf cons}.  Scheme's predicates
and selectors use different names from standard ones provided by
{\cf defabs}; we could define the standard Scheme names as follows:
\blisp
(define null? nil?)
\null
(define pair? cons?)
(define car cons-a)
(define cdr cons-b)
\elisp

\vfill

{\tiny RSN/9-19-88}

\end{document}
