;;; Example showing use of ``defabs'' construct:
;;;     Manipulation of symbolic arithmetic expressions
;;; Rishiyur Nikhil, Aug 23, 1988

;;; ----------------------------------------------------------------
;;; Our arithmetic expressions have one of five forms:

(defabs '(var x))
(defabs '(number x))
(defabs '(add x y))
(defabs '(mult x y))
(defabs '(neg x))

;;; ----------------------------------------------------------------
;;; Reducing all numeric sub-expressions to ``normal form''

(define (reduce e)
    (cond
      ((var?    e) e)
      ((number? e) e)
      ((add?    e) (reduce-add  (reduce (add-x e))  (reduce (add-y e))))
      ((mult?   e) (reduce-mult (reduce (mult-x e)) (reduce (mult-y e))))
      ((neg?    e) (reduce-neg  (reduce (neg-x e))))
      (else        (error "reduce: argument is not an expression"))))

(define (is-number e n)
    (and (number? e) (= (number-x e) n)))

(define (reduce-add e1 e2)
    (cond
      ((and (number? e1) (number? e2)) (number (+ (number-x e1) (number-x e2))))
      ((is-number e1 0) e2)
      ((is-number e2 0) e1)
      (else (add e1 e2))))

(define (reduce-mult e1 e2)
    (cond
      ((and (number? e1) (number? e2)) (number (* (number-x e1) (number-x e2))))
      ((is-number e1 0) e1)
      ((is-number e1 1) e2)
      ((is-number e2 0) e2)
      ((is-number e2 1) e1)
      (else (mult e1 e2))))

(define (reduce-neg e)
    (cond
      ((number? e) (number (- (number-x e))))
      (else (neg e))))

;;; ----------------------------------------------------------------
;;; Differentiate an expression E w.r.t. a variable X

(define (differentiate e x)
    (cond
      ((var?    e) (if (eq? (var-x e) (var-x x)) (number 1) (number 0)))
      ((number? e) (number 0))
      ((add?    e) (add  (differentiate (add-x e) x)
                         (differentiate (add-y e) x)))
      ((mult?   e) (add (mult (mult-x e)
                              (differentiate (mult-y e) x))
                        (mult (differentiate (mult-x e) x)
                              (mult-y e))))
      ((neg?    e) (neg (differentiate (neg-x e) x)))
      (else        (error "differentiate: argument is not an expression"))))

;;; ----------------------------------------------------------------
;;; Some test expressions

(define test-exp-1
     (add (mult (number 2)
                (number 3))
          (mult (neg (number 4))
                (number 5))))

(define test-exp-2
     (add (mult (number 2)
                (var 'x))
          (mult (neg (var 'x))
                (var 'y))))
