;;; This is the code file for problem set 8

;;; It contains the metacircular evaluator, as described in section 4.1
;;; of the course notes, with a few minor modifications.

;;; You should just load this file into Scheme without editing it.  The
;;; new procedures that you will need to modify in order to do the
;;; problem set have been copied into a separate file for your
;;; convenience.

;;; SETTING UP THE ENVIRONMENT

;;; We initialize the global environment by snarfing a few primitives
;;; from the underlying SCHEME system, and binding them (to symbols of
;;; the same name).  The actual structure of the environment is
;;; determined by the constructor EXTEND-ENVIRONMENT which is listed
;;; below together with the code that manipulates environments.  If you
;;; want to add more primitives to your evaluator, you can modify the
;;; list PRIMITIVE-NAMES to include more Scheme primitives.

(define primitive-names
  '(+ - * / = < > 1+ -1+ cons car cdr atom? eq? null? not user-print))

(define (setup-environment)
  (define initial-env
    (extend-environment
     primitive-names
     (mapcar (lambda (pname)
               (eval pname user-initial-environment))
             primitive-names)
     nil))
  (define-variable! 'nil nil initial-env)
  (define-variable! 't (not nil) initial-env)
  initial-env)


;;; note that the definition of setup-environment differs from that in the
;;; textbook, in that we use the mapcar above to extract the underlying SCHEME
;;; versions of the primitive procedures

(define the-global-environment nil)

;;; INITIALIZATION AND DRIVER LOOP

;;; The following code initializes the machine and starts the Lisp
;;; system.  You should not call it very often, because it will clobber
;;; the global environment, and you will lose any definitions you have
;;; accumulated.

(define (initialize-lisp)
  (set! the-global-environment (setup-environment))
  (driver-loop))

;;; Here is the actual driver loop.  It reads in an expression, passes
;;; it to the machine to be evaluated in the global environment, and
;;; prints the result

;;; When/If your interaction with the evaluator bombs out in an error,
;;; you should restart it by calling DRIVER-LOOP.  Note that the driver
;;; uses a prompt of "**==>" to help you avoid confusing typing to the
;;; simulator with typing to the underlying SCHEME interpreter.

(define (driver-loop)
  (newline)
  (princ '**==>)
  (user-print (mini-eval (read) the-global-environment))
  (driver-loop))

;;; We use a special PRINT here, which avoids printing the environment
;;; part of a compound procedure, since the latter is a very long (or
;;; even circular) list.

(define (user-print object)
  (cond
   ((compound-procedure? object)
    (print (list 'compound-procedure
                 (procedure-text object))))
   (else (print object))))


;;; THE GUTS OF THE EVALUATOR

(define (mini-eval exp env)
  (cond ((self-evaluating? exp) exp)
        ((quoted? exp) (text-of-quotation exp))
        ((variable? exp) (lookup-variable-value exp env))
        ((definition? exp) (eval-definition exp env))
        ((assignment? exp) (eval-assignment exp env))
        ((lambda? exp) (make-procedure exp env))
        ((conditional? exp) (eval-cond (clauses exp) env))
        ((application? exp)
         (mini-apply (mini-eval (operator exp) env)
                (list-of-values (operands exp) env)))
        (else (error "Unknown expression type --MINI-EVAL"))))


(define (mini-apply procedure arguments)
  (cond ((primitive-procedure? procedure)
         (apply-primitive-procedure procedure arguments))
        ((compound-procedure? procedure)
         (eval-sequence (procedure-body procedure)
                        (extend-environment
                         (parameters procedure)
                         arguments
                         (procedure-environment procedure))))
        (else (error "Unknown procedure type -- MINI-APPLY"))))


(define (list-of-values exps env)
  (cond ((no-operands? exps) '())
        (else (cons (mini-eval (first-operand exps) env)
                    (list-of-values (rest-operands exps)
                                    env)))))
(define (eval-sequence exps env)
  (cond ((last-exp? exps) (mini-eval (first-exp exps) env))
        (else (mini-eval (first-exp exps) env)
              (eval-sequence (rest-exps exps) env))))

(define (eval-cond clist env)
  (cond ((no-clauses? clist) 'nil)
        ((else-clause? (first-clause clist))
         (eval-sequence (actions (first-clause clist))
                        env))
        ((true? (mini-eval (predicate (first-clause clist)) env))
         (eval-sequence (actions (first-clause clist))
                        env))
        (else (eval-cond (rest-clauses clist) env))))



(define (eval-assignment exp env)
  (let ((value (mini-eval (assignment-value exp) env)))
    (set-variable-value! (assignment-variable exp) value env)
    value))

(define (eval-definition exp env)
  (define-variable! (definition-variable exp)
                    (mini-eval (definition-value exp) env)
                    env)
  (definition-variable exp))



;;; Syntax of the language

;;; note that the version in the textbook uses if for some of these definitions.
;;; since we are asking you to modify if, we have changed the definitions below.

(define (self-evaluating? exp)
  (or (number? exp) (eq? exp 'nil) (eq? exp 't)))

(define (quoted? exp)
  (if (not (atom? exp))
      (eq? (car exp) 'quote)))

(define (text-of-quotation exp) (cadr exp))

(define (variable? exp) (symbol? exp))

(define (assignment? exp)
  (if (not (atom? exp))
      (eq? (car exp) 'set!)))

(define (assignment-variable exp) (cadr exp))

(define (assignment-value exp) (caddr exp))

(define (definition? exp)
  (if (not (atom? exp))
      (eq? (car exp) 'define)))


(define (definition-variable exp)
  (cond ((variable? (cadr exp))
         (cadr exp))
        (else
         (caadr exp))))


(define (definition-value exp) 
  (cond ((variable? (cadr exp))
         (caddr exp))           
        (else
         (cons 'lambda
               (cons (cdadr exp)     ;Formal parameters
                     (cddr exp))))))

(define (lambda? exp)
  (if (not (atom? exp))
      (eq? (car exp) 'lambda)))

(define (conditional? exp)
  (if (not (atom? exp))
      (eq? (car exp) 'cond)))

(define (clauses exp) (cdr exp))

(define (no-clauses? clauses) (null? clauses))

(define (first-clause clauses) (car clauses))

(define (rest-clauses clauses) (cdr clauses))

(define (else-clause? clause) (eq? (predicate clause) 'else))

(define (predicate clause) (car clause))

(define (true? x) (not (null? x)))

(define (actions clause) (cdr clause))

(define (last-exp? seq) (null? (cdr seq)))

(define (first-exp seq) (car seq))

(define (rest-exps seq) (cdr seq))

(define (application? exp) (not (atom? exp)))

(define (operator app) (car app))

(define (operands app) (cdr app))

(define (no-operands? args) (null? args))

(define (first-operand args) (car args))

(define (rest-operands args) (cdr args))

(define (last-operand? args)
  (null? (cdr args)))

(define (make-procedure lambda-exp env)
       (list 'procedure lambda-exp env))

(define (compound-procedure? proc)
       (if (not (atom? proc))
           (eq? (car proc) 'procedure)))

(define (procedure-text proc) (cadr proc))

(define (parameters proc) (cadr (cadr proc)))

(define (procedure-body proc) (cddr (cadr proc)))

(define (procedure-environment proc) (caddr proc))


;;; APPLYING PRIMITIVE PROCEDURES

;;; The mechanism for applying primitive procedures is somewhat
;;; different from the one given in the course notes.  We can recognize
;;; primitive procedures (which are all inherited from Scheme) by asking
;;; Scheme if the object we have is a Scheme procedure.

(define (primitive-procedure? p)
  (applicable? p))

;;; To apply a primitive procedure, we ask the underlying Scheme system
;;; to perform the application.  (Of course, an implementation on a
;;; low-level machine would perform the application in some other way.)

(define (apply-primitive-procedure p args)
  (apply p args))


;;; ENVIRONMENTS


;;; Environments are represented as association lists.
;;; Note that this is a simplified version of that given in the textbook
;;; in that we only use one long list of bindings

(define the-empty-environment '())

(define (add-binding-pair var val env)
  (cons (list var val) env))

(define (lookup-variable-value var env)
  (let ((bp (binding-pair var env)))
    (cond ((null? bp) (error "Unbound variable" var))
          (else (cadr bp)))))

(define (set-variable-value! var val env)
  (let ((bp (binding-pair var env)))
    (cond ((null? bp) (error "Unbound variable" var))
          (else (set-car! (cdr bp) val)))))

(define (define-variable! var val env)
  (set-car! env
            (cons (list var val)
                  (car env))))

(define (extend-environment variables values base-env)
  (define (pair-up variables values)
    (cond ((null? variables)
           (cond ((null? values) '())
                 (t
                  (error "Too many arguments supplied"))))
          ((null? values)
           (error "Too few arguments supplied"))
          (else (cons (list (car variables) (car values))
                      (pair-up (cdr variables)
                               (cdr values))))))
  (cons (pair-up variables values) base-env))

(define (binding-pair var env)
  (define (scan pairlist)
    (cond ((null? pairlist)
           (cond ((null? (cdr env)) 'nil)
                 (else (binding-pair var (cdr env)))))
          ((eq? var (caar pairlist)) (car pairlist))
          (else (scan (cdr pairlist)))))
  (scan (car env)))

