;;; Turn the sybase calls into more scheme-like calls.
;;;
;;; Karl O. Pinc
;;;
;;; March 22, 1991
;;;
;;; A set of high-level sybase access procedures.  These procedures, together 
;;; with the following
;;; low-level sybase calls, will comprise the basis of most applications:
;;;
;;; dbcoltype
;;; dbexit
;;;
;;; These procedures call the Chez Scheme "error" procedure when the sybase
;;; calls return #f.  It may be useful to use fluid-let to redefine "error"
;;; during the execution of these calls.  This is espically true as the
;;; standard "error" procedure discards the current continuation, which would
;;; cause sybase to remain initialized.
;;;
;;; The scheme globals sybase-message-handler, sybase-error-handler, and
;;; sybase-message/error-handler may also be rebound to desired procedures.
;;; See the CUSTOM.SS file.

(define call-with-sybase-ready
  ; Returns the result of calling proc, a procedure of one argument, with a
  ; boolean which indicates whether sybase is ready to run.  When a true value
  ; is passed to proc and this procedure is executing,
  ; Sybase database calls may be made.
  (lambda (proc)
    (let ((initialized (dbinit)))
      (if initialized
          (begin
           ;; Install the C error handlers
           (dbmsghandle (get-message-handler))
           (dberrhandle (get-error-handler))))
      (let ((result (proc initialized)))
        (dbexit)
        result))))

(define call-with-dbproc
  ; Return the result of calling proc, a procedure of one argument, with a dbproc.
  ; The dbproc passed will be a connection using the given loginrec
  ; to the named server.  If a connection cannot be established the dbproc
  ; passed will be '().  If the server is "" the default server will be used.
  ;
  ; The dbproc will be closed when the call returns and should not be used
  ; thereafter.
  (lambda (proc loginrec server)
    (let* ((dbproc (dbopen loginrec server))
           (result (proc dbproc)))
      (if (not (dbclose dbproc))
          (error 'call-with-dbproc
                 "attempt to close dbproc '~s' failed"
                 dbproc))
      result)))

(define call-with-loginrec
  ; Return the result of calling proc with one argument, a loginrec.
  ; The loginrec will be setup
  ; with the user, password, host name, and application name supplied.
  ; If a loginrec cannot be established, #f will be passed to proc.
  ; If any of these arguments are the empty string, the system defaults
  ; will be used.
  ;
  ; The loginrec will be deallocated when this procedure returns and should not
  ; be used thereafter.
  (lambda (proc user password host application)
    (let ((loginrec (dblogin)))
      (if loginrec
          (begin
           (if user
               (if (not (dbsetluser loginrec user))
                   (error 'call-with-loginrec
                          "cannot set user to ~s with dbsetluser"
                          user)))
           (if password
               (if (not (dbsetlpwd loginrec password))
                   (error 'call-with-loginrec
                          "cannot set password with dbsetlpwd"))) ; Don't tell 'em the password.
           (if host
               (if (not (dbsetlhost loginrec host))
                   (error 'call-with-loginrec
                          "cannot set host to ~s with dbsetlhost"
                          host)))
           (if application
               (if (not (dbsetlapp loginrec application))
                   (error 'call-with-loginrec
                          "cannot set application to ~s with dbsetlapp"
                          application)))))
      (let ((result (proc loginrec)))
        (mark-closed! loginrec)
        result))))

(define for-each-sql-statement
  (lambda (proc dbproc . sql-fragments)
    ; Adds sql code to the Sybase sql buffer associated with the connection on
    ; dbproc and then runs the batch in the buffer.  Call proc with a boolean after each
    ; statement is executed and before the next statement is executed.  A
    ; non-false value passed to proc indicates the statement successfully
    ; executed, a non-true value indicates the statement did not execute.
    ; The result returned by this procedure is undefined.
    ;
    ; Note that the sql-fragments are strings and there may be zero or more of them.
    ; The sql-fragments are submited to the server as supplied, no whitespace is
    ; inserted between them.  Consequently, as with dbcmd, the calling program
    ; must explictily separate the sql-fragments from one another when necessary,
    ; and, each sql-fragment
    ; supplied may be a part of an complete SQL statement, a complete SQL statement,
    ; or more than one complete SQL statement as deisred.
    ;
    ; A call with zero sql-fragments can be used to run statements already
    ; assembled, for example with dbcmd, in the command buffer.
    (letrec ((run-each
              (lambda ()
                (let ((return-code (dbresults dbproc)))
                  (if (not (eq? return-code 'NO_MORE_RESULTS))
                      (begin
                       (proc return-code)
                       (run-each)))))))
      (apply buffer-sql (cons dbproc sql-fragments))
      (if (not (dbsqlexec dbproc))
          (error 'for-each-sql-statement
                 "cannot execute the batch on dbproc ~s with dbsqlexec"
                 dbproc))
      (run-each))))

(define process-sql-statements
  (lambda (proc dbproc . sql-fragments)
    ; Run a batch of SQL code on connection dbproc in the order of the statements
    ; in the batch. Call proc after each statement is executed with a boolean, 
    ; indicating success or failure. Return a list of the results of each call 
    ; to proc. 
    ;
    ; Just like for-each-sql-statement but returns a list of results.
    (letrec ((run-each
              (lambda ()
                (let ((return-code (dbresults dbproc)))
                  (if (eq? return-code 'NO_MORE_RESULTS)
                      '()
                      (cons (proc return-code)
                            (run-each)))))))
      (apply buffer-sql (cons dbproc sql-fragments))
      (if (not (dbsqlexec dbproc))
          (error 'process-sql-statements
                 "cannot execute the batch on dbproc ~s with dbsqlexec"
                 dbproc))
      (run-each))))

(define buffer-sql
  (lambda (dbproc . sql-fragments)
    ; Add sql-fragments to the sql command buffer for dbprocess dbproc.  This is
    ; just like dbcmd but it calls the Chez Scheme "error" procedure if there
    ; is an error and accecpts more than one piece of sql code.  When more than one
    ; piece of sql code is supplied, it is like multiple calls to dbcmd.
    ;
    ; Returns: undefined.
    (if sql-fragments
        (if (not (dbcmd dbproc 
                        (if (cdr sql-fragments)
                            (apply string-append sql-fragments)
                            (car sql-fragments))))
            (error 'buffer-sql
                   "cannot add sql code ~s to the buffer on dbproc ~s with dbcmd"
                   sql-fragments
                   dbproc)))))

(define for-each-sql-row
  (lambda (proc dbproc)
    ; Call proc as each row is returned, in the order they are returned, by the
    ; SQL statement executed on connection dbproc.
    ; Proc should be a procedure of one argument, the possible values passed
    ; to proc are: #f, the retreival of the row failed; 'REG_ROW,
    ; the row retreived is a regular (not a compute) row; a computeid, the
    ; row is a computed row.  The result returned by this procedure is undefined.
    (letrec ((do-row
              (lambda ()
                (let ((return-code (dbnextrow dbproc)))
                  (if (not (eq? return-code 'NO_MORE_ROWS))
                      (begin
                       (proc return-code)
                       (do-row)))))))
      (do-row))))

(define process-sql-rows
  (lambda (proc dbproc)
    ; Call proc as each row is returned, in the order they are returned, by the
    ; SQL statement executed on connection dbproc.
    ; Proc should be a procedure of one argument, the possible values passed
    ; to proc are: #f, the retreival of the row failed; 'REG_ROW,
    ; the row retreived is a regular (not a compute) row; a computeid, the
    ; row is a computed row.  Return a list of the results of the calls of proc.
    ;
    ; Just like for-each-sql-row but returns a list of results.
    (letrec ((do-row
              (lambda ()
                (let ((return-code (dbnextrow dbproc)))
                  (if (eq? return-code 'NO_MORE_ROWS)
                      '()
                      (cons (proc return-code)
                            (do-row)))))))
      (do-row))))

(define column-types
  (lambda (dbproc)
    ; Return a list of the datatypes of the columns returned by the sql statement
    ; returning rows on the connection identified by dbproc.
    (letrec ((do-column
              (lambda (this-column)
                (if (> this-column last-column)
                    '()
                    (cons (dbcoltype dbproc this-column)
                          (do-column (+ this-column 1))))))
             (last-column (dbnumcols dbproc)))
      (do-column 1))))

(define compute-column-types
  (lambda (dbproc computeid)
    ; Return a list of the datatype of the columns returned in the compute row
    ; identified by computeid in the current SQL statement returning rows
    ; on the connection identified by dbproc.
    (letrec ((do-column
              (lambda (this-column)
                (if (> this-column last-column)
                    '()
                    (cons (dbalttype dbproc computeid)
                          (do-column (+ this-column 1))))))
             (last-column (dbnumalts dbproc computeid)))
      (if (= -1 last-column)
          (error 'compute-column-types
                 "invalid computeid ~s on dbproc ~s passed to dbnumalts"
                 computeid
                 dbproc))
      (do-column 1))))

(define data-columns->list
  ; Return a list of the data columns of the current row returned on dbproc.
  ; The list returned will contain one element for every element of the c-types list,
  ; a list of symbols which define sybase datatypes and determine how the columns
  ; are to be interperted.  The symbols are those used with the column-> and
  ; compute-> procedures.  When there are fewer datatype symbols than columns in the
  ; result row returned from the database, only the leftmost rows, and only as many
  ; rows as symbols in the c-types list, will be returned.
  (lambda (dbproc c-types)
    (let loop ((counter 1) (c-types c-types))
      (if (null? c-types)
          '()
          (cons (column-> dbproc counter (car c-types))
                (loop (+ counter 1) (cdr c-types)))))))

(define compute-columns->list
  ; Return a list of the compute columns of the current row returned on dbproc.
  ; The list returned will contain one element for every element of the c-types list,
  ; a list of symbols which define sybase datatypes and determine how the columns
  ; are to be interperted.  The symbols are those used with the column-> and
  ; compute-> procedures.  The computeid is the value passed to the proc called by
  ; for-each-sql-row or process-sql-rows.
  (lambda (dbproc computeid c-types)
    (let loop ((counter 1) (c-types c-types))
      (if (null? c-types)
          '()
          (cons (compute-> dbproc computeid counter (car c-types))
                (loop (+ counter 1) (cdr c-types)))))))

(define sql-columns->list
  ; Return a list of the columns of the current row returned on dbproc.
  ; rowinfo is the value passed to the proc called by for-each-sql-row or
  ; process-sql-rows.  c-types are the list of sybase datatypes, it will
  ; only be used when the row returned is a regular result row, as opposed to
  ; a compute row.  When the row returned is a regular result row, as with
  ; data-columns->list,   When there are fewer datatype symbols than columns in the
  ; result row returned from the database, only the leftmost rows, and only as many
  ; rows as symbols in the c-types list, will be returned.
  (lambda (dbproc rowinfo c-types)
    (if (computeid? rowinfo)
        (compute-columns->list dbproc 
                               rowinfo 
                               (compute-column-types dbproc rowinfo))
        (data-columns->list dbproc c-types))))

;; The procedures column-> and compute-> correspond with dbdata and dbadata but
;; take an extra one or two required arguments. The first argument
;; is a symbol identifying the type of data being returned. They are the
;; same as the datatypes used on the sql server.  The second argument is
;; required for binary, text, and image data, othewise it should not be
;; present.  It is a positive integer;
;; the number of bytes of data returned.  The symbols and the
;; corresponding returned datatypes are:
;;
;;   'int          integer
;;   'smallint     integer
;;   'tinyint      integer
;;   'float        real
;;   'char         string
;;   'varchar      string
;;   'binary       not implimented
;;                 Binary data should be implimented with it's own abstraction
;;                 and set of manipulation routines.  The
;;                 implimentation should probably be numbers from 0 to
;;                 255 in a vector with calls to C routines to do the
;;                 manipulation, but then again, who knows.  (This would
;;                 avoid problems with non-byte length data.)
;;   'varbinary    not implimented
;;   'text         string
;;   'image        not implimented
;;   'bit          boolean
;;   'money        real
;;   'datetime     not implimented.  Again, another abstraction and set of associated
;;                 manipulation routines are called for.  Probably implimented as an
;;                 analog of the sybase datetime data structure (days and
;;                 milliseconds).
;;   'timestamp    not implimented and probably should not be implimented
;;   'sysname      not implimented and perhaps should not be implimented
;;
;;
;; Unfortunately, Sybase never returns a empty string, it returns a single
;; space instead (!).  Consequently, these routines work the same way.  There
;; just seems to be no way to distinguish between a string with a space in it and
;; an empty string as far as db-library is concerned.

(define column-> '())
(define compute-> '())
  
(let ()
  ;; Take a pointer and a sybase data type and return the data pointed to.
  (define peek.ptr
    (lambda (ptr sybtype length) ; length is ignored when it is implied by type.
      (if (zero? ptr)
          #f
          (case sybtype
            (int (peek.integer-32 ptr))
            (smallint (peek.integer-16 ptr))
            (tinyint (peek.unsigned-8 ptr))
            ((char varchar text)
             (let ((result (make-string length)))
               (vms-memory->string! result ptr length)
               result))
            (float (peek.double-float ptr))
            (bit (peek.boolean ptr))
            (money (let ((high-bits (peek.integer-32 ptr)))
                     (/ ((if (> high-bits 0) 
                             + 
                             -) (* high-bits
                                   4294967296) ; (expt 2 32)
                                (peek.unsigned-32 (+ ptr 4)))
                        1000)))
            (else
             (error 'peek.ptr
                    "unknown sybase data type: ~a"
                    sybtype))))))
  
  (set! column-> 
        (lambda (dbproc column sybtype)
          (let ((length (dbdatlen dbproc column)))
            (case length
              (-1 (error 'column-> 
                         "the ~a column is not in range on dbproc ~a" 
                         column
                         dbproc))
              (0 '()) ; The database is returning null, we use '() instead.
              (else
               (peek.ptr (dbdata dbproc column)
                         sybtype
                         length))))))
  
  (set! compute-> 
        (lambda (dbproc computeid column sybtype)
          (let ((length (dbadlen dbproc computeid column)))
            (case length
              (-1 (error 'compute-> 
                         "the ~a column is not in range on computeid ~a, dbproc ~a"
                         column
                         computeid
                         dbproc))
              (0 '()) ; The database is returning null, we use '() instead.
              (else
               (peek.ptr (dbadata dbproc computeid column)
                         sybtype
                         length)))))))