;;; -*- Emacs-Lisp -*- 
;;;    File:	luther.el  (~bevemyr/KAM/Tracer/luther.el)
;;;    Author:	Johan Bevemyr (Uppsala University, Sweden)
;;;    Created:	Tue Jan 21 17:19:24 1992
;;;    Purpose:	A low level WAM tracer written for GNU emacs.
;;;

(require 'cl)


;; this is a dirty fix, should be fixed another way

(setq max-lisp-eval-depth 2000)


;;
;; This machine is inspired by the Luther WAM emulator written in C. 
;;
;; For an introduction to the WAM see "Warren's Abstract Machine" by 
;; Hassan Ait-Kaci, the MIT Press, 1991.
;;


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Constants
;;

(defvar *wam-xregstart*  0    "start of x registers area")
(defvar *wam-xregsize*   20   "number of x registers")

(defvar *wam-stackstart* 100   "start of stack area")  
(defvar *wam-stacksize*  200  "size of stack area")

(defvar *wam-heapstart*  1000 "start of heap area")
(defvar *wam-heapsize*   200  "size of heap area")

(defvar *wam-trailstart* 2000 "start of trail area")
(defvar *wam-trailsize*  100  "size of trail area")

(defvar *wam-codestart*  3000 "start of code area")
(defvar *wam-codesize*   200  "size of code area")

(defvar *atom-nil*       '(atom . pnil)  "atom nil")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; 
;; Global registers
;;

(defvar *heaptop* *wam-heapstart*  "Pointer to top of heap")
(defvar *heap-uncond* *heaptop*    "Heap backtrackpoint" )
(defvar *frame*   *wam-stackstart* "Pointer to current environment")
(defvar *choice*  *wam-stackstart* "Pointer to current choicepoint")
(defvar *choice0* *wam-stackstart* "Pointer to current choice at last call")
(defvar *pc*      *wam-codestart*  "Current code pointer")
(defvar *next-instr* *pc*          "Pointer to continuation")
(defvar *trail*   *wam-trailstart* "Pointer to top of trail")
(defvar *arity*   0                "Current arity")
(defvar *write-mode* nil           "t if write-mode, nil if read")
(defvar *code*    *wam-codestart*  "Pointer to top of code heap")
(defvar *s*       *heaptop*        "Pointer to structure argument")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Global variables
;;

(defvar *running*        nil  "true if WAM is running")
(defvar *failchoice*     nil  "first choicepoint")

(defvar *database*       nil)
(defvar *predtable*      nil)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Initializing
;;

(defun wam-initialize ()
  "Initalize the wam. Allocates memory and initalizes the database"
  (interactive)
  (wam-init-memory)
  (wam-init-database)
  (wam-init-registers)
  (wam-init-display)
  (wam-init-running))

(defun wam-reset ()
  "Resets the wam but keeps the database."
  (interactive)
  (wam-initialize)
  (wam-load))

(defun wam-init-registers ()
  "Initializes the global registers."
  (setq *heaptop* *wam-heapstart*)
  (setq *heap-uncond* *heaptop*)
  (setq *pc*      *wam-codestart*)
  (setq *next-instr* (1+ *pc*))
  (setq *trail*   *wam-trailstart*)
  (setq *frame*   (- *wam-stackstart* (wam-sizeof-frame)))
  (setq *choice*  (1- *frame*))
  (setq *choice*  (wam-make-choicepoint *trail* *heaptop* *choice* *frame*
					*next-instr* *next-instr* 0))
  (setq *choice0* *choice*)
  (setq *failchoice* *choice*))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Running
;;

(defmacro wam-save-buffer (&rest body)
  "Save the current buffer location."
  (let ((oldwin (make-symbol "oldwin")))
    (` (let (((, oldwin) (get-buffer-window (current-buffer))))
	 (unwind-protect
	     (progn (,@ body))
	     (select-window (, oldwin)))))))

(defun wam-init-running ()
  (setq *running* nil))

(defun wam-start ()
  "Starts the execution by calling the predicate start/0."
  (interactive)
  (wam-save-buffer
    (let ((pc (wam-get-definition '(start . 0) *predtable*)))
      (wam-init-registers)
      (setq *pc* (wam-execute pc))
      (setq *running* t)
      (wam-mark-line *pc*)
      (wam-flush-global-registers))))

(defun wam-cont ()
  "Continues the execution at the next pc."
  (interactive)
  (wam-save-buffer
    (cond ((or (null *pc*)
	       (not *running*))
	   (message "WAM is not running"))
	  (t (setq *pc* (wam-execute *pc*))
	     (wam-mark-line *pc*)
	     (wam-flush-global-registers)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Main engine
;;

(defmacro wam-get-index-1 () '(second instruction))
(defmacro wam-get-index-2 () '(third instruction))
(defmacro wam-get-functor () '(second instruction))
(defmacro wam-get-constant () '(wam-Tagify (second instruction) 'atom))
(defmacro wam-get-table ()   '(second instruction))
(defmacro wam-get-default () '(third instruction))
(defmacro wam-get-label () '(second instruction))

(defun wam-execute (pc)
  "Executes a WAM instruction and returns a new PC"
  (let ((instruction (wam-get-instruction pc)))
    (case (wam-get-opcode instruction)

      ;; INDEXING instructions ******************************

      (switch-on-term
       (wam-setXreg 0 (wam-deref (wam-getXreg 0)))
       (cond ((wam-IsVar (wam-getXreg 0))  (wam-dispatchVar pc instruction))
	     ((wam-IsAtom (wam-getXreg 0)) (wam-dispatchAtom pc instruction))
	     ((wam-IsList (wam-getXreg 0)) (wam-dispatchList pc instruction))
	     ((wam-IsStruct (wam-getXreg 0))
	      (wam-dispatchStruct pc instruction))))
      
      (switch-on-constant
       (wam-dispatch-atom-table pc (wam-getXreg 0) (wam-get-table)
			   (wam-get-default)))
      
      (switch-on-structure
       (wam-dispatch-struct-table pc (wam-getXreg 0) (wam-get-table)
			   (wam-get-default)))

      ;; CHOICE instructions ******************************

      (try
       (setq *choice*
	     (wam-make-choicepoint *trail* *heaptop* *choice* *frame*
				   *next-instr* (1+ pc) *arity*))
       (setq *heap-uncond* *heaptop*)
       (+ pc (wam-get-label)))
      
      (retry
       (wam-set-choice-nextclause *choice* (1+ pc))
       (+ pc (wam-get-label)))
      
      
      (trust
       (setq *choice* (wam-get-choice-lastchoice *choice*))
       (setq *heap-uncond* (wam-get-choice-heap *choice*))
       (+ pc (wam-get-label)))
      
      (try-me-else 
       (setq *choice*
	     (wam-make-choicepoint *trail* *heaptop* *choice* *frame*
				   *next-instr* (+ pc (wam-get-label))
				   *arity*))
       (setq *heap-uncond* *heaptop*)
       (1+ pc))
      
      (retry-me-else 
       (wam-set-choice-nextclause *choice* (+ pc (wam-get-label)))
       (1+ pc))
      
      (trust-me 
       (setq *choice* (wam-get-choice-lastchoice *choice*))
       (setq *heap-uncond* (wam-get-choice-heap *choice*))
       (1+ pc))
      
      ;; CONTROL instructions ******************************

      (choice-x
       (wam-setXreg (wam-get-index-1) (wam-PointerToTerm *choice0*))
       (1+ pc))
      
      (choice-y
       (wam-setYreg (wam-get-index-1) (wam-PointerToTerm *choice0*))
       (1+ pc))
      
      (cut
       (if (> *choice* *choice0*)
	   (progn (setq *choice* *choice0*)
		  (setq *heap-uncond* (wam-get-choice-heap *choice*))))
       (1+ pc))
      
      (cut-x
       (setq *choice* (wam-TagToPointer (wam-getXreg (wam-get-index-1))))
       (setq *heap-uncond* (wam-get-choice-heap *choice*))
       (1+ pc))
      
      (cut-y 
       (setq *choice* (wam-TagToPointer (wam-getYreg (wam-get-index-1))))
       (setq *heap-uncond* (wam-get-choice-heap *choice*))
       (1+ pc))
      
      (builtin         ;; not implemented
       (1+ pc))
      
      (meta-call       ;; not implemented
       (1+ pc))
      
      (meta-execute    ;; not implemented
       (1+ pc))
      
      (allocate
       (let ((newframe (wam-get-local-stacktop)))
	 (wam-set-frame-contenv newframe *frame*)
	 (wam-set-frame-nextinstr newframe *next-instr*)
	 (setq *frame* newframe))
       (1+ pc))
      
      (deallocate 
       (setq *next-instr* (wam-get-frame-nextinstr *frame*))
       (setq *frame* (wam-get-frame-contenv *frame*))
       (1+ pc))
      
      (init-y          ;; not implemented
       (1+ pc))
      
      (call
       (setq *choice0* *choice*)
       (setq *next-instr* (1+ pc))
       (setq *arity* (rest (wam-get-functor)))
       (let ((newpc (wam-get-definition (wam-get-functor) *predtable*)))
	 (cond ((null newpc) (wam-fail))
	       (t newpc))))
      
      (execute
       (setq *choice0* *choice*)
       (setq *arity* (rest (wam-get-functor)))
       (let ((newpc (wam-get-definition (wam-get-functor) *predtable*)))
	 (cond ((null newpc) (wam-fail))
	       (t newpc))))
      
      (proceed
       *next-instr*)
      
      (fail
       (wam-fail))
      
      ;; GET instructions ******************************

      (get-x-variable
       (wam-setXreg (wam-get-index-1)
		    (wam-getXreg (wam-get-index-2)))
       (1+ pc))
      
      (get-y-variable
       (wam-setYreg (wam-get-index-1)
		    (wam-getXreg (wam-get-index-2)))
       (1+ pc))
      
      (get-x-value
       (if (wam-unify (wam-deref (wam-getXreg (wam-get-index-1)))
		      (wam-deref (wam-getXreg (wam-get-index-2))))
	   (1+ pc)
	 (wam-fail)))
      
      (get-y-value 
       (if (wam-unify (wam-deref (wam-getYreg (wam-get-index-1)))
		      (wam-deref (wam-getXreg (wam-get-index-2))))
	   (1+ pc)
	 (wam-fail)))

      (get-constant
       (if (wam-unify (wam-get-constant)
		      (wam-deref (wam-getXreg (wam-get-index-2))))
	   (1+ pc)
	 (wam-fail)))

      (get-nil
       (if (wam-unify *atom-nil*
		      (wam-deref (wam-getXreg (wam-get-index-1))))
	   (1+ pc)
	 (wam-fail)))

      (get-structure
       (let ((Xi (wam-deref (wam-getXreg (wam-get-index-2))))
	     (str (wam-get-functor)))
	 (cond ((wam-IsVar Xi)
		(setq *write-mode* t)
		(wam-bind Xi (wam-make-str str))
		(1+ pc))
	       ((wam-IsStruct Xi)
		(setq *write-mode* nil)
		(if (equal (wam-GetFunctor Xi) str)
		    (progn (setq *s* (1+ (wam-TagToPointer Xi)))
			   (1+ pc))
		  (wam-fail)))
	       (t (wam-fail)))))
		  
      (get-list
       (let ((Xi (wam-deref (wam-getXreg (wam-get-index-1)))))
	 (cond ((wam-IsVar Xi)
		(setq *write-mode* t)
		(wam-bind Xi (wam-make-lst))
		(1+ pc))
	       ((wam-IsList Xi)
		(setq *write-mode* nil)
		(setq *s* (wam-TagToPointer Xi))
		(1+ pc))
	       (t (wam-fail)))))

      ;; PUT instructions ******************************

      (put-x-void
       (wam-LoadHVA (wam-get-index-1))
       (1+ pc))

      (put-y-void
       (wam-LoadSVA (wam-get-index-1))
       (1+ pc))

      (put-x-variable
       (wam-LoadHVA (wam-get-index-1))
       (wam-setXreg (wam-get-index-2) (wam-getXreg (wam-get-index-1)))
       (1+ pc))

      (put-y-variable
       (wam-LoadSVA (wam-get-index-1))
       (wam-setXreg (wam-get-index-2) (wam-getYreg (wam-get-index-1)))
       (1+ pc))

      (put-x-value
       (wam-setXreg (wam-get-index-2) (wam-getXreg (wam-get-index-1)))
       (1+ pc))

      (put-y-value
       (wam-setXreg (wam-get-index-2) (wam-getYreg (wam-get-index-1)))
       (1+ pc))

      (put-x-unsafe-value
       (wam-RefStackUnsafe (wam-get-index-2) (wam-getXreg (wam-get-index-1)))
       (wam-setXreg (wam-get-index-1)
		    (wam-getXreg (wam-get-index-2)))
       (1+ pc))

      (put-y-unsafe-value 
       (wam-RefStackUnsafe (wam-get-index-2) (wam-getYreg (wam-get-index-1)))
       (1+ pc))

      (put-constant
       (wam-setXreg (wam-get-index-2) (wam-get-constant))
       (1+ pc))

      (put-nil
       (wam-setXreg (wam-get-index-1) *atom-nil*)
       (1+ pc))

      (put-structure
       (wam-setXreg (wam-get-index-2) (wam-make-str (wam-get-functor)))
       (setq *write-mode* t)
       (1+ pc))

      (put-list
       (wam-setXreg (wam-get-index-1) (wam-make-lst))
       (setq *write-mode* t)
       (1+ pc))

      ;; SET instructions ******************************

      (set-x-variable
       (wam-LoadHVA (wam-get-index-1))
       (1+ pc))

      (set-y-variable
       (wam-setYreg (wam-get-index-1) (wam-makeHVA))
       (1+ pc))

      (set-x-value
       (wam-PushOnHeap (wam-getXreg (wam-get-index-1)))
       (1+ pc))

      (set-y-value
       (wam-PushOnHeap (wam-getYreg (wam-get-index-1)))
       (1+ pc))
      
      (set-x-local-value
       (wam-WriteLocalValue (wam-getXreg (wam-get-index-1)))
       (1+ pc))
       
      (set-y-local-value
       (wam-WriteLocalValue (wam-getXreg (wam-get-index-1)))
       (1+ pc))
       
      (set-constant
       (wam-PushOnHeap (wam-get-constant))
       (1+ pc))

      (set-nil
       (wam-PushOnHeap *atom-nil*)
       (1+ pc))

      (set-void
       (let ((n (wam-get-index-1)))
	 (while (< 0 n)
	   (wam-CreateHVA)
	   (setq n (1- n))))
       (1+ pc))

      ;; UNIFY instructions ******************************

      (unify-void
       (cond (*write-mode*
	      (let ((n (wam-get-index-1)))
		(while (< 0 n)
		  (wam-CreateHVA)
		  (setq n (1- n))))
	      (1+ pc))
	     (t
	      (setq *s* (+ *s* (wam-get-index-1)))
	      (1+ pc))))

      (unify-x-variable
       (cond (*write-mode*
	      (wam-LoadHVA (wam-get-index-1))
	      (1+ pc))
	     (t
	      (wam-setXreg (wam-get-index-1) (wam-ref *s*))
	      (setq *s* (1+ *s*))
	      (1+ pc))))

      (unify-y-variable
       (cond (*write-mode*
	      (wam-setYreg (wam-get-index-1) (wam-makeHVA))
	      (1+ pc))
	     (t
	      (wam-setYreg (wam-get-index-1) (wam-ref *s*))
	      (setq *s* (1+ *s*))
	      (1+ pc))))

      (unify-y-first-value
       (cond (*write-mode*
	      (wam-setYreg (wam-get-index-1) (wam-makeHVA))
	      (1+ pc))
	     (t
	      (wam-LoadSVA (wam-get-index-1))
	      (wam-bind (wam-getYreg (wam-get-index-1))
			(wam-ref *s*))
	      (setq *s* (1+ *s*))
	      (1+ pc))))


      (unify-x-value
       (cond (*write-mode*
	      (wam-PushOnHeap (wam-getXreg (wam-get-index-1)))
	      (1+ pc))
	     (t
	      (if (wam-unify (wam-deref (wam-ref *s*))
			     (wam-deref (wam-getXreg (wam-get-index-1))))
		  (progn (setq *s* (1+ *s*))
			 (1+ pc))
		(wam-fail)))))

      (unify-y-value
       (cond (*write-mode*
	      (wam-PushOnHeap (wam-getYreg (wam-get-index-1)))
	      (1+ pc))
	     (t
	      (if (wam-unify (wam-deref (wam-ref *s*))
			     (wam-deref (wam-getYreg (wam-get-index-1))))
		  (progn (setq *s* (1+ *s*))
			 (1+ pc))
		(wam-fail)))))

      (unify-x-local-value
       (cond (*write-mode*
	      (wam-WriteLocalValue (wam-getXreg (wam-get-index-1)))
	      (1+ pc))
	     (t
	      (if (wam-unify (wam-deref (wam-ref *s*))
			     (wam-deref (wam-getXreg (wam-get-index-1))))
		  (progn (setq *s* (1+ *s*))
			 (1+ pc))
		(wam-fail)))))


      (unify-y-local-value 
       (cond (*write-mode*
	      (wam-WriteLocalValue (wam-getXreg (wam-get-index-1)))
	      (1+ pc))
	     (t
	      (if (wam-unify (wam-deref (wam-ref *s*))
			     (wam-deref (wam-getYreg (wam-get-index-1))))
		  (progn (setq *s* (1+ *s*))
			 (1+ pc))
		(wam-fail)))))


      (unify-constant
       (cond (*write-mode*
	      (wam-PushOnHeap (wam-get-constant))
	      (1+ pc))
	     (t
	      (if (wam-unify (wam-deref (wam-ref *s*))
			     (wam-get-constant))
		  (progn (setq *s* (1+ *s*))
			 (1+ pc))
		(wam-fail)))))

      (unify-nil 
       (cond (*write-mode*
	      (wam-PushOnHeap *atom-nil*)
	      (1+ pc))
	     (t
	      (if (wam-unify (wam-deref (wam-ref *s*))
			     *atom-nil*)
		  (progn (setq *s* (1+ *s*))
			 (1+ pc))
		(wam-fail)))))

      (unify-structure
       (cond (*write-mode*
	      (wam-PushOnHeap (wam-Tagify (1+ *heaptop*) 'struct))
	      (wam-PushOnHeap (wam-get-functor))
	      (1+ pc))
	     (t
	      (let ((Ds (wam-deref (wam-ref *s*)))
		    (str (wam-get-functor)))
		(cond ((wam-IsVar Ds)
		       (wam-bind Ds (wam-make-str str))
		       (1+ pc))
		      ((wam-IsStruct Ds)
		       (if (equal (wam-GetFunctor Ds) str)
			   (progn (setq *s* (1+ (wam-TagToPointer Ds)))
				  (1+ pc))
			 (wam-fail)))
		      (t (wam-fail)))))))

      (unify-list
       (cond (*write-mode*
	      (wam-PushOnHeap (wam-Tagify (1+ *heaptop*) 'list))
	      (1+ pc))
	     (t
	      (let ((Ds (wam-deref (wam-ref *s*))))
		(cond ((wam-IsVar Ds)
		       (wam-bind Ds (wam-make-lst))
		       (1+ pc))
		      ((wam-IsList Ds)
		       (setq *s* (wam-TagToPointer Ds))
		       (1+ pc))
		      (t (wam-fail)))))))
      (halt
       (setq *running* nil)
       (1+ pc))
      (t
       (setq *running* nil)
       (beep)
       (message (concat "Instruction "
			(prin1-to-string (wam-get-opcode instruction))
			" not implemented"))
       (1+ pc)))))
      
(defun wam-fail ()      
  "Backtracks to the last choicepoint and returns a new pc."
  (cond ((= *choice* *failchoice*)
	 (message "Execution stopped with FAIL")
	 (setq *running* nil)
	 *pc*)
	(t (wam-unwind-trail (wam-get-choice-trailtop *choice*))
	   (setq *heaptop* (wam-get-choice-heap *choice*))
	   (setq *choice0* (wam-get-choice-lastchoice *choice*))
	   (wam-restore-registers *choice*)
	   (setq *heap-uncond* *heaptop*)
	   (setq *next-instr* (wam-get-choice-nextinstr *choice*))
	   (setq *frame* (wam-get-choice-contenv *choice*))
	   (wam-get-choice-nextclause *choice*))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Indexing help predicates
;; 

(defun wam-dispatchVar (pc instruction)
  "Increments PC by the label given in INSTRUCTION."
  (let ((distance (second instruction)))
    (cond ((eql distance 'fail)
	   (wam-fail))
	  (t (+  pc distance)))))

(defun wam-dispatchAtom (pc instruction)
  "Increments PC by the label given in INSTRUCTION."
  (let ((distance (third instruction)))
    (cond ((eql distance 'fail)
	   (wam-fail))
	  (t (+  pc distance)))))

(defun wam-dispatchList (pc instruction)
  "Increments PC by the label given in INSTRUCTION."
  (let ((distance (fourth instruction)))
    (cond ((eql distance 'fail)
	   (wam-fail))
	  (t (+  pc distance)))))

(defun wam-dispatchStruct (pc instruction)
  "Increments PC by the label given in INSTRUCTION."
  (let ((distance (fifth instruction)))
    (cond ((eql distance 'fail)
	   (wam-fail))
	  (t (+  pc distance)))))

(defun wam-dispatch-atom-table (pc term table def)
  "   Increments PC by the label associated with TERM in TABLE,
   or by DEF if TERM is not found in TABLE."
  (let ((intab (wam-get-atom-table-entry term table)))
    (cond ((null intab) 
	   (if (eql def 'fail)
	       (wam-fail)
	     (+ pc def)))
	  (t (+ pc intab)))))

(defun wam-dispatch-struct-table (pc term table def)
  "   Increments PC by the label associated with TERM in TABLE,
   or by DEF if TERM is not found in TABLE."
  (let ((intab (wam-get-struct-table-entry term table)))
    (cond ((null intab) 
	   (if (eql def 'fail)
	       (wam-fail)
	     (+ pc def)))
	  (t (+ pc intab)))))

(defun wam-get-atom-table-entry (term table)
  "   Returns the label associated with TERM in TABLE, or nil
   if no entry is found."
  (cond ((null table) nil)
	((equal (wam-TagToPointer term) (first (first table)))
	 (second (first table)))
	(t (wam-get-atom-table-entry term (rest table)))))

(defun wam-get-struct-table-entry (term table)
  "   Returns the label associated with TERM in TABLE, or nil
   if no entry is found."
  (cond ((null table) nil)
	((equal term (first (first table)))
	 (second (first table)))
	(t (wam-get-struct-table-entry term (rest table)))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Misc.
;;

(defun wam-get-instruction (pc)
  "Returns the instruction found at PC."
  (wam-ref pc))

(defun wam-get-opcode (instruction)
  "Returns the opcode of INSTRUCTION."
  (car instruction))

(defun wam-FrameSize (pc)
  "Returns the size of the current environment."
  (third (wam-read-memory (1- pc))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Choicepoints
;;
;; 0 trail
;; 1 heaptop
;; 2 lastchoice
;; 3 contenv
;; 4 next-instr
;; 5 next-clause
;; 6 arity
;; 7 registers
;; 

(defun wam-sizeof-choice () 
  "Returns the size of a choicepoint, not counting argument registers."
  7)

(defun wam-make-choicepoint (trail heap choice frame next-instr 
			     next-clause arity)
  "   Construct a choicepoint at the top of the stack. Returns a 
   pointer to the new choicepoint."
  (let ((newchoice (wam-get-local-stacktop))
	(stacktop nil)
	(a 0))
    (setq stacktop newchoice)
    (wam-set-memory stacktop trail)
    (wam-set-memory (+ stacktop 1) heap)
    (wam-set-memory (+ stacktop 2) choice)
    (wam-set-memory (+ stacktop 3) frame)
    (wam-set-memory (+ stacktop 4) next-instr)
    (wam-set-memory (+ stacktop 5) next-clause)
    (wam-set-memory (+ stacktop 6) arity)
    (setq stacktop (+ stacktop 7))
    (while (< a arity)
      (wam-set-memory stacktop (wam-getXreg a))
      (setq a (1+ a))
      (setq stacktop (1+ stacktop)))
    newchoice))

(defun wam-get-choice-trailtop (choice)
  "Returns the trailtop field of CHOICE."
  (wam-read-memory choice))

(defun wam-get-choice-heap (choice)
  "Returns the heaptop field of CHOICE."
  (wam-read-memory (+ choice 1)))

(defun wam-get-choice-lastchoice (choice)
  "Returns the lastchoice field of CHOICE."
  (wam-read-memory (+ choice 2)))

(defun wam-get-choice-contenv (choice)
  "Returns the contenv field of CHOICE."
  (wam-read-memory (+ choice 3)))

(defun wam-get-choice-nextinstr (choice)
  "Returns the nextinstr field of CHOICE."
  (wam-read-memory (+ choice 4)))

(defun wam-get-choice-nextclause (choice)
  "Returns the nextclause field of CHOICE."
  (wam-read-memory (+ choice 5)))

(defun wam-get-choice-arity (choice)
  "Returns the arity field of CHOICE."
  (wam-read-memory (+ choice 6)))

(defun wam-get-choice-register (choice reg)
  "Returns register REG of the choicepoint CHOICE."
  (wam-read-memory (+ choice 7 reg)))

(defun wam-set-choice-nextclause (choice nc)
  "Sets the nextclause filed of CHOICE to NC."
  (wam-set-memory (+ choice 5) nc))

(defun wam-restore-registers (choice)
  "Restores the argument registers from CHOICE."
  (wam-restore-registers-two choice (wam-get-choice-arity choice)))

(defun wam-restore-registers-two (choice reg)
  (cond ((= reg 0))
	(t (wam-setXreg (1- reg) (wam-get-choice-register choice (1- reg)))
	   (wam-restore-registers-two choice (1- reg)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Frames
;;
;; 0 contenv
;; 1 next-instr
;; 2 yregs
;;

(defun wam-sizeof-frame () 2)

(defun wam-set-frame-contenv (frame value)
  "Sets the contenv field of FRAME to VALUE."
  (wam-set-memory frame value))

(defun wam-set-frame-nextinstr (frame value)
  "Sets the nextinstr field of FRAME to VALUE."
  (wam-set-memory (1+ frame) value))

(defun wam-setYreg (reg value)
  "Sets Y register REG to VALUE."
  (wam-set-memory (+ *frame* 2 reg) value))

(defun wam-get-frame-contenv (frame)
  "Returns the contenv field of FRAME."
  (wam-read-memory frame))

(defun wam-get-frame-nextinstr (frame)
  "Returns the nextinstr field of FRAME."
  (wam-read-memory (1+ frame)))

(defun wam-getYreg (reg)
  "Returns the Y register REG of the current frame."
  (wam-read-memory (+ *frame* 2 reg)))

(defun wam-address-Yreg (reg)
  "Returns the address of the Y register REG of the current frame."
  (+ *frame* 2 reg))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Terms
;;

(defun wam-TagToPointer (term)
  "Returns the pointer part of a tagged pointer."
  (cdr term))

(defun wam-PointerToTerm (term)
  "Returns a tagged pointer with term as value field"
  (wam-Tagify term 'atom))

(defun wam-Tagify (pointer tag)
  "Puts a tag on a pointer."
  (cons tag pointer))

(defun wam-tagof (term)
  "Returns the tag of an object."
  (car term))

(defun wam-IsBoundVar (term)
  "Returns t if TERM is an unbound variable"
  (and (wam-IsVar term)
       (not (equal term (wam-refvar term)))))

(defun wam-ref (pointer)
  "Returns the contents of the memory cell at POINTER."
  (wam-read-memory pointer))

(defun wam-refvar (term)
  "Returns the contents of the variable TERM."
  (wam-ref (wam-TagToPointer term)))

(defun wam-IsVar (term)
  "Returns true if TERM is a variable."
  (or (eql (wam-tagof term) 'href)
      (eql (wam-tagof term) 'sref)))

(defun wam-IsHeapVar (term)
  "Returns true if TERM is a heap variable."
  (eql (wam-tagof term) 'href))

(defun wam-IsStackVar (term)
  "Returns true if TERM is a stack variable."
  (eql (wam-tagof term) 'sref))

(defun wam-IsAtom (term)
  "Returns true if TERM is an atom."
  (eql (wam-tagof term) 'atom))

(defun wam-IsList (term)
  "Returns true if TERM is a list pointer."
  (eql (wam-tagof term) 'list))

(defun wam-IsStruct (term)
  "Returns true if TERM is a struct pointer."
  (eql (wam-tagof term) 'struct))

(defun wam-getFunctor (term)
  "Returns the functor of the untagged structure pointer TERM."
  (wam-refvar term))

(defun wam-getArity (term)
  "Returns the arity of a structure."
  (cdr (wam-getFunctor term)))

(defun wam-getCar (list)
  "Returns the Car element of the list pointed to by LIST."
  (wam-ref (wam-TagToPointer list)))

(defun wam-getCdr (list)
  "Returns the Cdr element of the list pointed to by LIST."
  (wam-ref (1+ (wam-TagToPointer list))))

(defun wam-GetFunctor (struct)
  "Returns the functor of a structure pointed to by STRUCT."
  (wam-ref (wam-TagToPointer struct)))

(defun wam-GetFunctorArity (func)
  "Returns the arity of the functor FUNC."
  (cdr func))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Unification
;;

(defun wam-deref (term)
  "Returns the result of dereferencing TERM."
  (cond
   ((wam-IsBoundVar term)
    (wam-deref (wam-read-memory (wam-TagToPointer term))))
   (t term)))

(defun wam-unify (Dx Dy)
  "Returns true if DX and DY can be unified, nil otherwise."
  (cond ((equal Dx Dy))
	((wam-IsHeapVar Dx)
	 (wam-bind Dx Dy))
	((wam-IsHeapVar Dy)
	 (wam-bind Dy Dx))
	((wam-IsStackVar Dx)
	 (wam-bind Dx Dy))
	((wam-IsStackVar Dy)
	 (wam-bind Dy Dx))
	((and (wam-IsList Dx) (wam-IsList Dy))
	 (wam-unify-list Dx Dy))
	((and (wam-IsStruct Dx) (wam-IsStruct Dy))
	 (wam-unify-struct Dx Dy))))

(defun wam-bind (var term)
  "Binds the variable VAR to TERM."
  (wam-trail var)
  (wam-set-memory (wam-TagToPointer var) term))

(defun wam-undo (var)
  "Undoes the binding of VAR."
  (wam-set-memory (wam-TagToPointer var) var))

(defun wam-unify-list (list1 list2)
  "Unifies the lists pointed to by LIST1 and LIST2."
  (if (wam-unify (wam-deref (wam-getCar list1))
		 (wam-deref (wam-getCar list2)))
      (wam-unify (wam-deref (wam-getCdr list1))
		 (wam-deref (wam-getCdr list2)))))

(defun wam-unify-struct (struct1 struct2)
  "Unifies the structures pointed to by STRUCT1 and STRUCT2."
  (if (equal (wam-getFunctor struct1) (wam-getFunctor struct2))
      (wam-unify-struct-arg (wam-getArity struct1)
			     (1+ (wam-TagToPointer struct1))
			     (1+ (wam-TagToPointer struct2)))))

(defun wam-unify-struct-arg (arity s1 s2)
  "Unifies the two structures S1 and S2 with arity ARITY."
  (cond ((= arity 0))
	((and (wam-unify (wam-deref (wam-ref s1))
			 (wam-deref (wam-ref s2)))
	      (wam-unify-struct-arg (1- arity) (1+ s1) (1+ s2))))))

(defun wam-trail (var)
  "Trails the variable VAR."
  (cond ((wam-IsHeapVar var)
	 (if (< (wam-TagToPointer var) *heap-uncond*)
	     (wam-push-on-trail var)))
	((wam-IsStackVar var)
	 (if (< (wam-TagToPointer var) *choice*)
	     (wam-push-on-trail var)))))

(defun wam-unwind-trail (trail)
  "Unwinds the trail to TRAIL."
  (while (> *trail* trail)
    (wam-undo (wam-pop-from-trail))))

(defun wam-push-on-trail (var)
  "Push VAR on the trail."
  (wam-set-memory *trail* var)
  (setq *trail* (1+ *trail*)))

(defun wam-pop-from-trail ()
  "Pops an element from the trail."
  (setq *trail* (1- *trail*))
  (wam-read-memory *trail*))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Memory
;;

(defun wam-init-memory ()
  "Initializes the memory."
  (wam-init-memory-buffer "X registers" "*wam-xreg*" *wam-xregstart*
			  *wam-xregsize*)
  (wam-init-memory-buffer "Stack" "*wam-stack*" *wam-stackstart*
			  *wam-stacksize*)
  (wam-init-memory-buffer "Heap" "*wam-heap*" *wam-heapstart* *wam-heapsize*)
  (wam-init-memory-buffer "Trail" "*wam-trail*" *wam-trailstart* 
			  *wam-trailsize*)
  (wam-init-memory-buffer "Code" "*wam-code*" *wam-codestart* *wam-codesize*))

(defun wam-init-memory-buffer (name buffer start size)
  "Initializes the given 'memory'-buffer with memory addresses."
  (let ((curr start))
    (get-buffer-create buffer)
    (set-buffer buffer)
    (erase-buffer)
    (insert name "\n")
    (insert "-------------\n")
    (while (< curr (+ start size))
      (insert (int-to-string curr) " 0\n")
      (setq curr (1+ curr)))
    (goto-char (point-min))))

(defun wam-read-memory (location)
  "Returns the memory contents of LOCATION."
  (wam-read-buffer (wam-location-buffer location) location))

(defun wam-read-buffer (buffer loc)
  (set-buffer buffer)
  (goto-char (point-min))
  (search-forward (concat "\n" (int-to-string loc)) (point-max) t)
  (let* ((start (point))
	 (end (progn (search-forward "\n" (point-max) t) (point))))
    (read (buffer-substring  start end))))

(defun wam-set-memory (location value)
  "Sets the memory contents of LOCATION to VALUE."
  (wam-set-buffer (wam-location-buffer location) location value))

(defun wam-set-buffer (buffer loc value)
  (let ((buf (get-buffer-window buffer)))
    (if buf (select-window buf)))
  (set-buffer buffer)
  (goto-char (point-min))
  (search-forward (concat "\n" (int-to-string loc)) (point-max) t)
  (let* ((start (point))
	 (end (progn (search-forward "\n" (point-max) t) (point))))
    (delete-region start end)
    (insert " " (prin1-to-string value) "\n")
    t))

(defun wam-location-buffer (location)
  "Returns the buffer wich contains LOCATION."
  (cond
   ((< location *wam-stackstart*) "*wam-xreg*")
   ((< location *wam-heapstart*) "*wam-stack*")
   ((< location *wam-trailstart*) "*wam-heap*")
   ((< location *wam-codestart*) "*wam-trail*")
   (t "*wam-code*")))

(defun wam-getXreg (reg)
  "Returns the value of X register REG."
  (wam-read-memory (+ *wam-xregstart* reg)))

(defun wam-setXreg (reg value)
  "Sets the value of X register REG to VALUE."
  (wam-set-memory (+ *wam-xregstart* reg) value))

(defun wam-get-local-stacktop ()
  "Returns the position of the local stack top."
  (if (> *frame* *choice*)
      (+ *frame* (wam-FrameSize *next-instr*) (wam-sizeof-frame))
    (+ *choice* (wam-get-choice-arity *choice*) (wam-sizeof-choice))))

(defun wam-make-str (str)
  "Returns a tagged pointer to a structure with functor STR."
  (let ((arity (wam-GetFunctorArity str))
	(ret (wam-Tagify *heaptop* 'struct)))
    (wam-set-memory *heaptop* str)
    (setq *heaptop* (1+ *heaptop*))
    ret))

(defun wam-make-lst ()
  "Returns a tagge pointer to a new list."
  (wam-Tagify *heaptop* 'list))

(defun wam-LoadHVA (reg)
  "Initializes the temporary register REG to a heap variable."
  (wam-set-memory *heaptop* (wam-Tagify *heaptop* 'href))
  (wam-setXreg reg (wam-Tagify *heaptop* 'href))
  (setq *heaptop* (1+ *heaptop*)))

(defun wam-LoadSVA (reg)
  "Initializes the permanent register REG to be unbound."
  (wam-setYreg reg (wam-Tagify (wam-address-Yreg reg) 'sref)))

(defun wam-CreateHVA ()
  "Puts an initialized variable on the heap."
  (wam-set-memory *heaptop* (wam-Tagify *heaptop* 'href))
  (setq *heaptop* (1+ *heaptop*)))

(defun wam-makeHVA ()
  "Returns a tagged pointer to a new heap variable."
  (let ((ret (wam-Tagify *heaptop* 'href)))
    (wam-set-memory *heaptop* ret)
    (setq *heaptop* (1+ *heaptop*))
    ret))

(defun wam-PushOnHeap (value)
  "Pushes VALUE on the heap."
  (wam-set-memory *heaptop* value)
  (setq *heaptop* (1+ *heaptop*)))

(defun wam-RefStackUnsafe (xreg term)
  "Globalizes TERM if it is a stack variable and store in X register XREG."
  (let ((tmp (wam-deref term)))
    (cond ((and (wam-IsStackVar tmp) (wam-IsUnsafe tmp))
	   (wam-LoadHVA xreg)
	   (wam-set-memory (wam-TagToPointer tmp) (wam-getXreg xreg)))
	  (t (wam-setXreg xreg tmp)))))

(defun wam-IsUnsafe (var)
  "Returns true if VAR is unsafe."
  (> (wam-TagToPointer var) *frame*))

(defun wam-WriteLocalValue (term)
  "Writes TERM on the heap, globalizes stack variables if needed."
  (let ((tmp (wam-deref term)))
    (cond ((wam-IsStackVar tmp)
	   (let ((hva (wam-makeHVA)))
	     (wam-set-memory (wam-TagToPointer tmp) hva)))
	  (t (wam-set-memory *heaptop* tmp)
	     (setq *heaptop* (1+ *heaptop*))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Database
;;

(defmacro predicate (name arity code)
  (progn (setq *database* (cons (cons (cons name arity) code) *database*)) t))

(defun wam-init-database ()
  "Initializes the database."
  (setq *code* (1+ *wam-codestart*))
  (setq *predtable* nil)
  (wam-set-memory *wam-codestart* '(call (dummy . 0) 0)))

(defun wam-reset-database ()
  "Resets the database."
  (interactive)
  (wam-save-buffer
    (wam-initialize)
    (setq *database* nil)))
  
(defun wam-get-definition (name database)
  "   NAME is a functor (name.arity) and DATABASE is an assoc list
   where NAME is associated with a pc. That pc is returned, or nil
   if the predicate is undefined."
  (cond ((null database) nil)
	((equal name (first (first database)))
	 (rest (first database)))
	(t (wam-get-definition name (rest database)))))

(defun wam-load ()
  (interactive)
  (setq *predtable* (wam-load-predicates *database*)))

(defun wam-load-file (filename)
  "Load wam code from the file FILENAME into database"
  (interactive "fWAM file: ")
  (wam-reset-database)
  (load-file filename)
  (setq *predtable* (wam-load-predicates *database*)))

(defun wam-load-predicates (data)
  (cond ((null data) nil)
	(t (let ((entry (cons (first (first data)) *code*)))
	     (wam-insert-predicate (rest (first data)))
	     (cons entry (wam-load-predicates (rest data)))))))

(defun wam-insert-predicate (code)
  (cond ((null code))
	(t
	 (let ((c (wam-fix-code (first code) (rest code))))
	   (cond ((null c)
		  (wam-insert-predicate (rest code)))
		 (t (wam-set-memory *code* c)
		    (setq *code* (1+ *code*))
		    (wam-insert-predicate (rest code))))))))

(defun wam-fix-code (instruction code)
  "Transforms labels into offsets."
  (case (wam-get-opcode instruction)
    (switch-on-term
     (list 'switch-on-term
	   (wam-find-label (second instruction) code)
	   (wam-find-label (third instruction) code)
	   (wam-find-label (fourth instruction) code)
	   (wam-find-label (fifth instruction) code)))
    (switch-on-constant
     (list 'switch-on-constant
	   (wam-fix-table (second instruction) code)
	   (wam-find-label (third instruction) code)))
    (switch-on-structure
     (list 'switch-on-structure
	   (wam-fix-table (second instruction) code)
	   (wam-find-label (third instruction) code)))
    (try (list 'try (wam-find-label (second instruction) code)))
    (retry (list 'retry (wam-find-label (second instruction) code)))
    (trust (list 'trust (wam-find-label (second instruction) code)))
    (try-me-else (list 'try-me-else
		       (wam-find-label (second instruction) code)))
    (retry-me-else (list 'retry-me-else
		       (wam-find-label (second instruction) code)))
    (label nil)
    (t instruction)))

(defun wam-find-label (lab code)
  "Returns the offset to LAB in CODE."
  (cond
   ((eql lab 'fail) 'fail)
   ((null code) 'fail)
   ((eql (wam-get-opcode (first code)) 'label)
    (if (eql (second (first code)) lab)
	1
      (wam-find-label lab (rest code))))
   (t (let ((label (wam-find-label lab (rest code))))
	(if (eql label 'fail)
	    'fail
	  (1+ label))))))

(defun wam-fix-table (table code)
  "Adjusts the labels in the switch table TABLE according to CODE."
  (cond ((null table) nil)
	(t (let ((entry (first table)))
	     (cons (list (first entry) (wam-find-label (second entry) code))
		   (wam-fix-table (rest table) code))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Display stuff
;;

(defun run-wam ()
  "Run WAM tracer."
  (interactive)
  (setq *database* nil)
  (wam-initialize)
  (wam-setup-windows)
  (wam-flush-global-registers)
  (wam-load))

(defun wam-quit ()
  "Delete all buffers associated with WAM."
  (interactive)
  (delete-other-windows)
  (wam-kill-buffers '("*wam-heap*" "*wam-stack*" "*wam-trail*" "*wam-code*"
		      "*wam-global*" "*wam-xreg*")))

(defun wam-kill-buffers (blist)
  "Removes the buffers specified by BLIST."
  (cond
   ((null blist))
   ((null (get-buffer (first blist)))
    (wam-kill-buffers (rest blist)))
   (t (set-buffer (first blist))
      (set-buffer-modified-p nil)
      (kill-buffer (first blist))
      (wam-kill-buffers (rest blist)))))

(defun wam-init-display ()
  "Sets up the display."
  (set-buffer "*wam-code*")
  (setq overlay-arrow-string "====>")
  (or overlay-arrow-position
      (setq overlay-arrow-position (make-marker)))
  (goto-char (point-max))
  (search-backward "\n")
  (goto-char (1+ (point)))
  (set-marker overlay-arrow-position (point) (current-buffer))
  (goto-char (point-min))
  (get-buffer-create "*wam-global*"))

(defun wam-mark-line (pc)
  "Mark the line pointed to by PC with an arrow."
  (let ((buf (get-buffer-window "*wam-code*")))
    (if buf (select-window buf)))
  (set-buffer "*wam-code*")
  (goto-char (point-min))
  (search-forward (concat "\n" (int-to-string pc)) (point-max) t)
  (search-backward "\n")
  (goto-char (1+ (point)))
  (set-marker overlay-arrow-position (point) (current-buffer)))

(defun wam-flush-global-registers ()
  "Display the global registers."
  (set-buffer "*wam-global*")
  (erase-buffer)
  (insert "Global registers\n")
  (insert "-------------------------------------\n")
  (insert "H             " (prin1-to-string *heaptop*) " \t(heap top)\n")
  (insert "heap-uncond   " (prin1-to-string *heap-uncond*) " \t(need trail)\n")
  (insert "S             " (prin1-to-string *s*) " \t(struct)\n")
  (insert "E             " (prin1-to-string *frame*) " \t(environment)\n")
  (insert "B             " (prin1-to-string *choice*) " \t(choicepoint)\n")
  (insert "B0            " (prin1-to-string *choice0*) " \t(cut choicep)\n")
  (insert "P             " (prin1-to-string *pc*) " \t(code pointer)\n")
  (insert "CP            " (prin1-to-string *next-instr*) " \t(cont. point)\n")
  (insert "T             " (prin1-to-string *trail*) " \t(trail top)\n")
  (insert "arity         " (prin1-to-string *arity*) "\n")
  (insert "write-mode    " (prin1-to-string *write-mode*)))

(defun wam-setup-windows ()
  "Configures the windows to display heap, stack, trail, registers, and code."
  (interactive)
  (delete-other-windows)
  (switch-to-buffer "*wam-heap*")
  (wam-mode)
  (goto-char (point-min))
  (split-window-vertically)
  (split-window-horizontally)
  (split-window-horizontally)
  (other-window 1)
  (switch-to-buffer "*wam-stack*")
  (wam-mode)
  (goto-char (point-min))
  (other-window 1)
  (split-window-horizontally)
  (switch-to-buffer "*wam-xreg*")
  (wam-mode)
  (goto-char (point-min))
  (other-window 1)
  (switch-to-buffer "*wam-trail*")
  (wam-mode)
  (goto-char (point-min))
  (other-window 1)
  (split-window-horizontally)
  (switch-to-buffer "*wam-global*")
  (wam-mode)
  (goto-char (point-min))
  (other-window 1)
  (switch-to-buffer "*wam-code*")
  (wam-mode)
  (goto-char (point-min)))


(defvar wam-mode-map nil "local key map used by wam")

(if wam-mode-map
    nil
  (setq wam-mode-map (make-keymap))
  (suppress-keymap wam-mode-map)
  (define-key wam-mode-map " " 'wam-cont)
  (define-key wam-mode-map "n" 'wam-cont)
  (define-key wam-mode-map "r" 'wam-reset)
  (define-key wam-mode-map "d" 'wam-reset-database)
  (define-key wam-mode-map "q" 'wam-quit)
  (define-key wam-mode-map "s" 'wam-start)
  (define-key wam-mode-map "l" 'wam-load-file)
  (define-key wam-mode-map "c" 'wam-setup-windows))

(defun wam-mode ()
  "Major mode for the WAM tracer
  Special commands:
\\{wam-mode-map}
  "
  (interactive)
  (use-local-map wam-mode-map)
  (setq mode-name "WAM")
  (setq major-mode 'wam-mode)
  (run-hooks 'wam-mode-hook))

