%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% Filename:	reader.pl
%%% Author:	olin (Peter Olin) 
%%% Modified:	Apr 25, 1989
%%% Modified:	July 18, 1990 for Andorra by Johan Bevemyr
%%% Version:	1.2 (unfinished)
%%% 
%%% Description:
%%% 	
%%% 
%%%
%%%		Predicate				File
%%%		---------				----
%%% Requires:	compile_predicate/3			parser.pl
%%%		parse_predicate/3			- " -
%%%
%%% Exports:	compilef/2
%%%		process_file/3
%%%		open_files/3
%%%		close_files/3
%%%		fixname/3
%%%		read_clause/2
%%%		compile_all_predicates/4
%%%		read_predicate/3 [DCG]
%%%		last_clause_in_predicate/3 [DCG]
%%%		not_last_clause_in_predicate/4 [DCG]
%%%		in_same_predicate/2
%%%		dlist/1 [DCG]
%%%		num/1
%%%		num/2
%%%		write_header/2
%%%		write_footer/2
%%%		printcode/3
%%%		lispprintcode/2
%%%		prologprintcode/2
%%%		builtin/1
%%%		redefinition_warning/3
%%%		directive_execution/3
%%%		report_illegal_instruction/1
%%%		report_compile_time/3
%%%		report_completion/2
%%%		report_time_lapse/2
%%%		get_runtime/1
%%% Notes:	
%%% Bugs:
%%% Bugfixes:
%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% compilef(Filename,OutputMode)
%%% +Filename	The source-filename without extension.  If Filename is
%%%		'foo' 'foo.tl' is compiled and the resut is written to
%%%		'foo.twam'.
%%%
%%% +OutputMode	One of 'prolog' and 'lisp', determines how the resulting
%%%		code is written to the output-file.  'prolog' gives code
%%%		readable for human eyes, while 'lisp' is the code used
%%%		by the TROLL WAM-emulator.
%%%
%%%
%%%
%%% This is the top predicate called by the user. 
%%% Compilef/2 compiles a file and prints statistics on the screen.
%%% 

compilef(File) :-
	compilef(File,lisp).

compilef(File,OutputMode) :-
	open_files(File,InStream,OutStream),	%Fix filename & open
	process_file(InStream,OutStream,OutputMode), %Compile the file
	close_files(InStream,OutStream).	%Close files



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% process_file(InStream,OutStream,OutputMode)
%%%
%%% +InStream	The stream from which the source file is read.
%%%
%%% +OutStream	The stream to which the compiled code is written.
%%%
%%% +OutputMode	The format of the output.
%%%
%%%
%%%
%%% Writes appropriate header and footer to the object file as well
%%% as the actual code produced by compile_all_predicates.
%%%

process_file(InStream,OutStream,OutputMode) :-
	write_header(OutputMode,OutStream),
	read_clause(InStream,LA), 
	compile_all_predicates(LA,InStream,OutStream,OutputMode),
	write_footer(OutputMode,OutStream).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% open_files(InFile,InStream,OutStream)
%%%
%%% +InFile	Input-file (without extension) supplied by the user.
%%% 
%%% -InStream	System-generated stream to read from.
%%%
%%% -OutStream	System-generated stream to write to.
%%%
%%%
%%%
%%% Uses InFile without extension to create two filenames with extension
%%% '.tl' for the input file and '.twam' for the output file, and opens
%%% these files.
%%%

open_files(InFile,InStream,OutStream) :-
	fixname(InFile,InFile1,OutFile),
	open(InFile1,read,InStream),
	open(OutFile,write,OutStream).
	
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% close_files(InStream,OutStream)
%%%
%%% +InStream	Stream associated with the input file
%%%
%%% +OutStream	Stream associated with the outputfile
%%%
%%%
%%%
%%% Closes the files associated with the given streams.
%%%

close_files(InStream,OutStream) :-
	close(InStream), close(OutStream).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% fixname(File, InFile,OutFile)
%%%
%%% +File	Base filename without extension
%%%
%%% -InFile	'File.tl'
%%%
%%% -OutFile	'File.twam'
%%%
%%%
%%%
%%% Creates the output and input filenames from the specified File.

fixname(File,InFile,OutFile) :-
	name(File,Chars),
	dlist(Chars,IF,".pl"),
	dlist(Chars,OF,".wam"),
	name(InFile,IF),
	name(OutFile,OF).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% read_clause(Stream, Clause) 
%%%
%%% +Stream	Stream to read from.
%%%
%%% -Clause	Next clause in Stream.
%%%
%%%
%%%
%%% reads a clause (Prolog-term) from Stream, and returns the result in Clause.
%%% directives as ':- setglobal(foo,t)' are executed.
%%% Trying to redefine a predefined predicate results in a warning.

read_clause(InStream,Clause) :-
	read(InStream,Term),
	expand_term(Term,MClause),
	(directive_execution(InStream,MClause,Clause) % A directive ':-...'
    ;
	redefinition_warning(InStream,MClause,Clause) % Predefined predicate
    ;
	Clause = MClause).			% A normal clause
	 

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% compile_all_predicates(LA,InStream,OutStream,OutputMode)
%%%
%%% +LA		Lookahead; the last clause (term) read.
%%%
%%% +OutputMode	
%%%
%%% +InStream
%%%
%%% +OutStream
%%%
%%%
%%%
%%% Recursively compiles all predicates in a file.  Needs a lookahead
%%% when called.  Terminates when 'end_of_file' is found.
%%%


compile_all_predicates(end_of_file,_,_,_).

compile_all_predicates(LA,InStream,OutStream,OutputMode) :-
	get_runtime(T1),

	read_predicate(InStream,LA,LA2,Struct,[]),       %READ
	compile_emit(Struct,OutputMode,OutStream),       %COMPILE and EMIT CODE

	get_runtime(T2),
	report_time_lapse(T1,T2),!,
	compile_all_predicates(LA2,InStream,OutStream,OutputMode).

compile_emit(Struct,OutputMode,OutStream) :-
	compile_emit2(Struct,OutputMode,OutStream).
compile_emit(_,_,_).

compile_emit2(Struct,OutputMode,OutStream) :-
	copy_term(Struct,Scopy),
	parse_predicate(Struct, NStruct),	  	        %PARSE
	compile_predicate(OutputMode,OutStream,NStruct,Scopy),  %COMPILE
	!,fail.


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% read_predicate(InStream,LA,LA2) [DCG]
%%%
%%% +InStream	Stream to read from
%%%
%%% +LA		Lookahead (first clause in this predicate)
%%% 
%%% -LA2	New lookahead (first clause in next predicate)
%%%
%%%
%%%
%%% Reads all clauses in a predicate from InStream. If C is a clause
%%% from a new predicate last_clause_in_predicate applies, otherwise
%%% not_last_clause_in_predicate_applies.
%%%

read_predicate(InStream,LA,LA2) -->
	{read_clause(InStream,C)},
	(last_clause_in_predicate(LA,C,LA2)
    ;
	 not_last_clause_in_predicate(InStream,LA,C,LA2)).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% last_clause_in_predicate(LA,Clause,LA2) [DCG]
%%%
%%% +LA		Lookahead
%%%
%%% +Clause	The last clause read
%%%
%%% -LA2		New lookahead if Clause is the last in this predicate.
%%%
%%%
%%%
%%% Generates this clause (LA) if the next (Clause) is in another predicate.
%%% This is the base i a weird recursive predicate consisting of this predicate
%%% and (!) not_last_clause_in_predicate.
%%% I'll make it better later...
%%%

last_clause_in_predicate(LA,C,C) -->
	{ \+in_same_predicate(LA,C)},
	[LA].

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% not_last_clause_in_predicate(InStream,LA,Clause,LA2) [DCG]
%%%
%%% +InStream	Stream to read from
%%%
%%% +LA		Lookahead
%%%
%%% +Clause	The last clause read
%%%
%%% -LA2	New lookahead.
%%%
%%%
%%%
%%% Generates a list of clauses if the predicate consists of more than on
%%% clause.  Recursively calls read_predicate.

not_last_clause_in_predicate(InStream,LA,C,LA2) -->
	{in_same_predicate(C,LA)},		% Not necessary?? %%% 
	[LA],
	read_predicate(InStream,C,LA2).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% in_same_predicate(Clause1,Clause2)
%%%
%%% +Clause1	A clause
%%%
%%% +Clause2	A clause
%%%
%%% 
%%%
%%% True if Clause1 and Clause2 have the same name and arity.
%%%
%%% ***ATTENTION!!! *** in_same_predicate is slimy, perhaps it can 
%%% be improved.
%%% 
%%% Fix1: Cuts inserted to exclude non-wanted cases.
%%% Fix2: Cuts made unnecessary by inserting constraints on the 
%%%       arguments instead.

in_same_predicate(Clause1,Clause2) :-
	clause_head_same(Clause1,Head1),
	clause_head_same(Clause2,Head2),
	functor(Head1,Name,Arity),
	functor(Head2,Name,Arity).

clause_head_same((Head:-_),Head) :- !.
clause_head_same(Head,Head).



%%% MCs dlist

dlist([]) --> !.
dlist([X|Y]) --> [X],dlist(Y).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% Some mixed predicates.
%%%
%%%


directive_execution(InStream,MClause,Clause) :- 
	MClause = (:-Goal),!,
	format("-->:-~w ignored.~n",[Goal]),
	read_clause(InStream,Clause).

directive_execution(_,X,X).


redefinition_warning(InStream,MClause,Clause ) :-
	functor(MClause,Name,Arity),
	builtin(Name/Arity),!,
	format("??> Trying to redefine ~w/~w, you will lose!~n",[Name,Arity]),
	read_clause(InStream,Clause).

redefinition_warning(InStream,MClause,Clause ) :-
	MClause = (Head :- _),
	functor(Head,Name,Arity),
	builtin(Name/Arity),!,
	format("??> Trying to redefine ~w/~w, you will lose!~n",[Name,Arity]),
	read_clause(InStream,Clause).

redefinition_warning(_,X,X).

report_illegal_instruction(I) :- 
	format("???Illegal instruction in num/2: ~w~n~???probably something else failing before.",[I]),
	!,fail.


report_compile_time(PTime,CTime) :-
	Total is PTime + CTime,
	format("Parse:~d+Compile:~d ms=~d ms", [PTime,CTime,Total]).

report_completion(Runtime) :-
	format(">>>Compilation completed!~n~>>>Total runtime:~d ms~n",[Runtime]).

 
report_time_lapse(T1,T2) :- 
	T is T2 - T1,
	format("(~d ms)~n",[T]).

get_runtime(Time) :- 
	statistics(runtime,[Time,_]).




