%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% Filename:	head.pl
%%% Author:	olin (Peter Olin) 
%%% Modified:	May 30, 1989
%%% Modified:   18 July, 1990 for Andorra by Johan Bevemyr
%%% Version:	1.2 (unfinished)
%%% 
%%% Description:
%%% 	This file contains the code for compiling the head of a clause.
%%% 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% head_code(Clause,PV,Flag) [DCG]
%%%
%%% +Clause	The clause
%%%
%%% +PV		Number of permanent variables
%%%
%%% +Flag	switch/no_switch, signals if a switch-instr
%%%             has been generated.
%%%
%%%
%%% generates code for Head beginning at HeadLabel.
%%%

head_code(Clause,Ys,_) --> {head(Clause,[])}, !, allocate(Ys).

head_code(Clause,Ys,switch) --> {head(Clause,Arglist)}, 
	allocate(Ys),
	get_args_after_switch(Arglist).

head_code(Clause,Ys,no_switch) --> {head(Clause,Arglist)}, 
	allocate(Ys),
	get_args(Arglist,0).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% 
%%% allocate(PermVars)
%%%

allocate(Ys) --> [allocate(Ys)].


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% get_args_after_switch(Arglist, ArgNr)
%%%
%%%
%%% Recursively compiles all arguments in a head. Since the first
%%% arguments has been switched on, we compile the first get 
%%% instructions into a get_..._x0 instruction that use that knowledge.
%%%

get_args_after_switch([]) --> !, [].
get_args_after_switch([Arg|Args]) -->
	get_arg_after_switch(Arg),
	get_args(Args,1).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% get_args(Arglist, ArgNr)
%%%
%%%
%%% Recursively compiles all arguments in a head.
%%%

get_args([],_) --> !, [].

get_args([Arg|Args], ArgNr) -->
	get_arg(Arg,ArgNr), {ArgNr1 is ArgNr + 1},
	get_args(Args,ArgNr1).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% get_arg(Argument,ArgumentNumber)
%%%
%%% Depending on Argument the following code is generated:
%%%
%%% # is the argument's position in the head.
%%%
%%%
%%%	nil		get_nil(a_reg(#))
%%%
%%%	atom(C)
%%%	number(C)	get_constant(C,a_reg(#)).
%%%
%%%	struct(F,A)	get_structure(F/Arity,a_reg(#))
%%%			   ... code for the arguments to F ...
%%%			   ... code for structure arguments ...
%%%			Arity = length of A, the list of arguments
%%%
%%%	cons(Car,Cdr)	get_list(a_reg(#))
%%%			   ... code for the elements in the list ...
%%%
%%% Variables occuring only once are ignored
%%%
%%%	var(_,void)	nothing (increase the argument counter)
%%%
%%% The first occurence of a variable is compiled to a get_variable-instruction
%%%
%%%	var(X,first)	get_variable(X,a_reg(#))
%%%
%%% The subsequent occurences of a variable is compiled to a get_value-instr.
%%%
%%%	var(X,nth)	get_value(X,a_reg(#))
%%%	
%%%
%%%  f(g(1),2,f(3))
%%%    compiles to
%%%
%%%  get_structure(f/2,a(#))     f(....)
%%%
%%%   unify_variable(a(7))         g(1)
%%%   unify_constant(2)            2
%%%   unify_variable(a(8))         f(3)
%%%   get_structure(g/1,a(7)       g( )
%%%   unify_constant(1)              1
%%%   get_structure(f/1,a(8))      f( )
%%%   unify_constant(3)              3
%%%
%%% We want the substructures to a struct/list in the argument
%%% to come after the top-level of the argument.
%%%
%%% cons(a,(cons(cons(b,nil),cons(c,nil)))) = [a,[b],c]
%%%
%%% get_list(a(#))	    cons(
%%%  unify_constant(a)            a,
%%%  unify_variable(a(7))           cons(...)
%%%   get_list(a(7))
%%%    unify_variable(a(8))
%%%    unify_variable(a(9))
%%%     get_list(a(8))
%%%      unify_constant(b)
%%%      unify_nil
%%%     get_list(a(9))
%%%      unify_constant(c)
%%%      unify_nil

% - NIL
get_arg(nil,Arg) --> !, [get_nil(Arg)].

% - ATOM
get_arg(atom(C),Arg) --> !, [get_constant(C,Arg)].

% - NUMBER
get_arg(number(C),Arg) --> !, [get_constant(C,Arg)].

% - VOID
get_arg(var(_,void),_) --> !, [].

% - first VAR
get_arg(var(perm(X),first),Arg) --> !, [get_y_variable(perm(X),Arg)].

% - nth VAR
get_arg(var(perm(X),nth),Arg) --> !, [get_y_value(perm(X),Arg)].

% - first VAR
get_arg(var(temp(X),first),Arg) --> !, [get_x_variable(temp(X),Arg)].

% - nth VAR
get_arg(var(temp(X),nth),Arg) --> !, [get_x_value(temp(X),Arg)].

% - STRUCTURE
get_arg(struct(F/1,[Args]),Arg) --> !,
	[get_structure(F/1,Arg)],
	unify_one_arg(Args,Sub), { Sub = none -> Sub1 = [] ; Sub1 = Sub },
	head_substructure_code(Sub1).

% - STRUCTURE
get_arg(struct(F,Args),Arg) --> !,
	[get_structure(F,Arg)],
	unify_struct_args(Args,Sub),
	head_substructure_code(Sub).

% - LIST
get_arg(cons(Car,Cdr),Arg) --> !,
	[get_list(Arg)],
	unify_cons_args(cons(Car,Cdr),Sub),
	head_substructure_code(Sub).

get_arg(FAIL,_) --> !, [failure(FAIL)].


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% get_arg_after_switch(Arglist, ArgNr)
%%%
%%%
%%% As get_arg but compiles the first instructions into an get_..._x0.
%%%


% - NIL
get_arg_after_switch(nil) --> !, [get_nil(0)].

% - ATOM
get_arg_after_switch(atom(C)) --> !, [get_constant(C,0)].

% - NUMBER
get_arg_after_switch(number(C)) --> !, [get_constant(C,0)].

% - VOID
get_arg_after_switch(var(_,void)) --> !, [].

% - first VAR
get_arg_after_switch(var(X,N)) --> !, get_arg(var(X,N),0).

% - STRUCTURE
get_arg_after_switch(struct(F/1,[Args])) --> !,
	[get_structure(F/1,0)],
	unify_one_arg(Args,Sub), { Sub = none -> Sub1 = [] ; Sub1 = Sub },
	head_substructure_code(Sub1).

% - STRUCTURE
get_arg_after_switch(struct(F,Args)) --> !,
	[get_structure(F,0)],
	unify_struct_args(Args,Sub),
	head_substructure_code(Sub).

% - LIST
get_arg_after_switch(cons(Car,Cdr)) --> !,
	[get_list(0)],
	unify_cons_args(cons(Car,Cdr),Sub),
	head_substructure_code(Sub).

get_arg_after_switch(FAIL) --> !, [failure(FAIL)].

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% head_substructure_code(Substructures)
%%% 

head_substructure_code([]) --> [].
head_substructure_code([sub(Reg,StructOrList)|Subs]) -->
	head_unify_sub_arg(Reg,StructOrList),
	head_substructure_code(Subs).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% head_unify_sub_arg(Arg,Substructure)
%%%
%%% 

% - STRUCT
head_unify_sub_arg(Reg,struct(F/1,[Args])) --> !,
	[get_structure(F/1,Reg)],
	unify_one_arg(Args,Sub), { Sub = none -> Sub1 = [] ; Sub1 = Sub },
	head_substructure_code(Sub1).

head_unify_sub_arg(Reg,struct(F,Args)) --> !,
	[get_structure(F,Reg)],
	unify_struct_args(Args,Sub),
	head_substructure_code(Sub).

% - LIST
head_unify_sub_arg(Reg,cons(Car,Cdr)) --> !,
	[get_list(Reg)],
	unify_cons_args(cons(Car,Cdr),Sub),
	head_substructure_code(Sub).




%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% unify_arg(Arg,SubCode)
%%%
%%% +Arg		Argument to compile
%%%
%%% -SubCode	'none' if no SubCode should be generated.  Otherwise
%%% 		A struct sub(Reg,Arg) to pass to substructure_code.
%%%
%%%
%%%
%%% These predicates are common to head and body unification.
%%%

% - NIL
unify_arg(nil,none) --> !,
	[unify_nil].

% - ATOM
unify_arg(atom(C),none) --> !,
	[unify_constant(C)].

% - NUMBER
unify_arg(number(C),none) --> !,
	[unify_constant(C)].

% - STRUCT
unify_arg(struct(F,A),sub(temp(Reg),struct(F,A))) --> !,
	[unify_x_variable(temp(Reg))].


% - LIST 
unify_arg(cons(Car,Cdr),sub(temp(Reg),cons(Car,Cdr))) --> !,
	[unify_x_variable(temp(Reg))].

%%% unify_void, this one needs to do some lookahead to find out how many
%%%             voids there are in a row.

% - VOID
unify_arg(var(_,void),none) -->!,
	[unify_void].

% - first VAR
unify_arg(var(perm(X),first),none) -->!,
	[unify_y_variable(perm(X))].

% - nth VAR
unify_arg(var(perm(X),nth),none) -->!,
	[unify_y_value(perm(X))].

% - first VAR
unify_arg(var(temp(X),first),none) -->!,
	[unify_x_variable(temp(X))].

% - nth VAR
unify_arg(var(temp(X),nth),none) -->!,
	[unify_x_value(temp(X))].

unify_one_arg(struct(F/1,[A]), Sub) --> !,
	[unify_structure(F/1)],
	unify_one_arg(A,Sub).

unify_one_arg(Arg,Sub) --> unify_arg(Arg,Sub).


concatenate(none,none,[]) :- !.
concatenate(X,none,[X]) :- !.
concatenate(none,X,[X]) :- !.
concatenate(X,Y,[X,Y]).

concatenate2(none,X,X) :- !.
concatenate2(X,Y,Z) :- append([X],Y,Z).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% unify_struct_args(Args) [DCG]
%%%
%%% +Args	Arguments to a structure.
%%%
%%%
%%%
%%% Compiles the structure args "in-place" with one instruction, 
%%% simple arguments such as constants and variables are compiled
%%% to one single instruction.  Structures and lists are compiled
%%% to a "placeholding" unify_variable.  The code for
%%% structures and lists is placed after the code for all other arguments.
%%% Each structure/list code sequence refers to the "placeholder".
%%% 

unify_struct_args(Args,Sub) -->
	toplevel_code(Args,Sub).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% unify_cons_args(Args) [DCG]
%%%
%%% +Args	Arguments to a cons.
%%%
%%%
%%%
%%% Compiles the cons args "in-place" with one instruction, 
%%% simple elements such as constants and variables are compiled
%%% to one single instruction.  Structures and conses are compiled
%%% to a "placeholding" unify_variable.  The code for
%%% conses and structures is placed after the code for all other arguments.
%%% Each structure's/cons' code sequence refers to the "placeholder".
%%% 

unify_cons_args(cons(Car,Cdr),Sub) --> { Cdr = cons(_,_),! },
	unify_arg(Car,CarSub), 
	[unify_list],
	unify_cons_args(Cdr,CdrSub),
	{CdrSub = [] -> concatenate(CarSub,none,Sub) 
	; concatenate2(CarSub,CdrSub,Sub)}.

unify_cons_args(cons(Car,Cdr),Sub) --> 
	unify_arg(Car,CarSub), 
	unify_arg(Cdr,CdrSub),
	{concatenate(CarSub,CdrSub,Sub)}.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% toplevel_code(Args,SubArgs) [DCG]
%%%
%%% +Args	The structure arguments to compile
%%%
%%% -SubArgs	The structure arguments + placeholder for those arguments
%%%		that need subcode after the top-level code.  Subsequently
%%%		used by substructure_code.
%%%
%%%
%%% Generates top-level code for a list of structure arguments, as well as
%%% returns those arguments that need subcode.
%%%
%%% Example: toplevel_code([f(a),2],[sub(temp(X),f(a))]) --> 
%%%         [unify_variable(temp(X))
%%%          unify_constant(2)]


toplevel_code([],[]) --> [].

toplevel_code([Arg|Args],[Sub|Subs]) -->	%Subcode
	unify_arg(Arg,Sub), {Sub \== none},
	toplevel_code(Args,Subs).

toplevel_code([Arg|Args],Subs) -->		%No subcode
	unify_arg(Arg,none),			% {Sub = none},  ??????
	toplevel_code(Args,Subs).



