%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%%	File:		sequence.pl
%%%	Author:		Peter Olin
%%%	Date:		June 5, 1989
%%%	Modified:	Februari 2, 1993 by Katrin Boberg
%%%	Purpouse:	To annotate each occurence of a variable with
%%%			information about *where* it occurs (void,first,nth).
%%%			
%%%
%%% HISTORY
%%% Author	Date		Description
%%% PO		5/6-89		Extracted from parser.pl

	
	
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% sequence_head(Args,Found,Found1)
%%%
%%% +Args
%%% +Found
%%% -Found1
%%%
%%% Annotates all variables occuring in the head with either first, nth or void.
%%% Returns a new list of found variables (Found1).

sequence_head(Arglist,Found,Found1) :-
	sequence_arglist(Arglist,Found,Found1).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% sequence_calls
%%%
%%%
%%% Annotates all variables occuring in the body with either void, first or nth.

sequence_calls([],Found,Found).
sequence_calls([Call|Calls],Found,Found2) :-
	sequence_call(Call,Found,Found1),
	sequence_calls(Calls,Found1,Found2).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% sequence_call(Call,Found,Found1)
%%%
%%% Annotates all variables occuring in a call with either void, first or nth.
%%% 
%%%

sequence_call(call(_,Arglist),Found,Found1) :-
	sequence_arglist(Arglist,Found,Found1).
sequence_call(builtin(_,Arglist,_),Found,Found1) :-
	sequence_arglist(Arglist,Found,Found1).
sequence_call(unify(X,Y),Found,Found1) :-
	sequence_arglist([X,Y],Found,Found1).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% sequence_arglist
%%%
%%% 
%%% Recurisvely annotates variables occuring in the terms in a list of terms.

sequence_arglist([],Found,Found).
sequence_arglist([A|As],Found,Found1) :-
	sequence_term(A,Found,Found0),
	sequence_arglist(As,Found0,Found1).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% sequence_term(Term,Table,Table1)
%%%
%%% The first occurence of a variable is set to 'void', and its
%%% existence is noted in the Table.  If this variable is found again, 
%%% backtracking sets the variable to 'first'.  Further occurences are set
%%% to 'nth'. This predicate also links the safenessvariables. If 
%%% the first argument in two var/3 structures the second argument 
%%% has to be equal too.
%%%

sequence_term(nil,T,T).
sequence_term(atom(_),T,T).
sequence_term(number(_),T,T).
sequence_term(cons(Car,Cdr),T,T2):-
	sequence_term(Car,T,T1),
	sequence_term(Cdr,T1,T2).
sequence_term(struct(_,Arglist),T,T1) :-
	sequence_struct_args(Arglist,T,T1).
sequence_term(var(X,S,Loc),T,T1) :-		%1st occurence
	\+ known(X,_,_,T),
	note(X,S,Loc,T,T1).
sequence_term(var(X,S,Loc),T,T) :-		%nth occurence
	known(X,S,first,T),
	Loc = nth.


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% sequence_struct_args(Args,T,T1)
%%%
%%% Helper to sequence_term.
%%%

sequence_struct_args([],T,T).
sequence_struct_args([A|As],T,T2) :-
	sequence_term(A,T,T1),
	sequence_struct_args(As,T1,T2).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% known(Var,Loc,Table)
%%%
%%%
%%% True if X has been seen and is set to Loc.

known(X,XS,Loc,[(T,TS,Loc)|_]) :- X == T, Loc = first, XS = TS.
known(X,S,Loc,[_|Ts]) :- known(X,S,Loc,Ts).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% note
%%%
%%% Record X as Loc.

note(X,S,Loc,Y,[(X,S,Loc)|Y]).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% sequence_void
%%%
%%% 

sequence_voids([]).
sequence_voids([(_,_,void)|Ts]) :- !, sequence_voids(Ts).
sequence_voids([_|Ts]) :- !, sequence_voids(Ts).

