%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% Filename:	parser.pl
%%% Author:	olin (Peter Olin) 
%%% Modified:	June 5, 1989
%%% Modified:	July 18, 1990 for Andorra by Johan Bevemyr
%%% Version:	1.2 (unfinished)
%%% 
%%% Description:
%%%	This file contains the code for parsing a file.  
%%%	Parse_predicate/2 takes a list of clauses forming a predicate
%%%	as its argument and returns a parse-tree for the predicate. 
%%% 	
%%% 
%%%
%%%		Predicate				File
%%%		---------				----
%%% Requires:   printcode/3                             print.pl
%%% Exports:	
%%% Notes:	Major rewritings are necessary. The parse-structure
%%%		should contain information about variables' permanency,
%%%		number of permanent variables etc.
%%% Bugs:	-
%%% Bugfixes:	-
%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
	
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% parse_predicate(Clauses,Structure)
%%%
%%% +Body	A list of clauses forming a predicate.
%%%
%%% -Structure	A structure (see below) forming the parsed predicate
%%%
%%%
%%%
%%% Each predicate is parsed to a structure containing its name, 
%%% arity, and clauses. The structure constructed by the parser 
%%% looks like:   P is the number of permanent variables,
%%%
%%%	predicate( Name / Arity , 
%%%                 [ clause(P, _ , hb( head( ... ),body( ... ))),
%%%                   clause(P, _ , hb( head( ... ),body( ... ))),
%%%                   ...
%%%                   clause(P, _ , hb( head( ... ),body( ... ))),
%%%		   ]
%%%               )
%%%
%%% parse_predicate/2 extracts the Name and Arity from the list of clauses
%%% and parses each clause separately.
%%%

parse_predicate(Clauses,predicate(Name/Arity,StructGroup)) :- !,
	extract_name_and_arity(Clauses,Name,Arity),
	mapcar(Clauses,parse_clause,StructGroup),
	analyze_variables(StructGroup).

parse_predicate(_,parsing_failed).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% extract_name_and_arity(Clauses, Name, Arity)
%%%
%%% +Clauses	A list of clauses forming a predicate
%%%
%%% -Name	The name of the predicate
%%%
%%% -Arity	The arity of the predicate
%%%
%%%
%%%
%%% The name and arity are extracted by looking at the first clause
%%% in the list.  If the main functor is :-/2 we look at the left argument
%%% (the head), otherwise it is a clause without body and we look at the head
%%% immediately.
%%%

extract_name_and_arity([(Head :- _)|_],Name,Arity) :- !, %Clause with body
	functor(Head,Name,Arity).

extract_name_and_arity([HeadOnly|_],Name,Arity) :- !, %Clause with no body
        functor(HeadOnly,Name,Arity).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% parse_clause(Clause,Structure)
%%%
%%% +Clause	A clause
%%%
%%% -Structure	The parsed clause
%%%
%%%
%%%
%%% A clause is parsed to a Structure hb/2 with one
%%% argument for head and body respectively.
%%% It looks like:
%%%
%%% 		hb( head( ... ),
%%%		    body( ... ))
%%%

parse_clause((Head :- Body),Clause) :- !,
	parse_head(Head,HeadStruct),		
	flatten_conjunction(Body,BodyList),
	parse_body(BodyList,BodyStruct),
	head(Clause,HeadStruct),
	body(Clause,BodyStruct).

parse_clause(Head, Clause) :-
	\+(functor(Head,(:-),2)),		%Clause with no body
	parse_head(Head,HeadStruct),
	body(Clause,[]),
	head(Clause,HeadStruct).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% parse_head(Head,HeadStruct)
%%%
%%% +Head	The head of a clause
%%%
%%% +HeadStruct	The parsed head
%%%
%%%
%%%
%%% The head of a clause is parsed to a structure head/1, where the single 
%%% argument is a list of the head's parsed arguments.
%%%
%%%		head([A1,...,An])
%%%
%%% where A1,...,,An are the parsed arguments of the clause's head.
%%%

parse_head(Head,HeadStruct) :-
	Head =.. [_|HeadArgs],
	parse_head_args(HeadArgs,HeadStruct).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% parse_head_args
%%%
%%% +Args	The arguments to the head
%%%
%%% +ArgsStruct	The parsed arguments
%%%
%%%

parse_head_args(Args,ArgsStruct) :-
	mapcar(Args,parse_term,ArgsStruct).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% parse_body(Body, BodyStruct)
%%%
%%% +Body	The body of a clause
%%%
%%% -BodyStruct	The parsed body
%%%
%%%
%%% The body is parsed to a list of parsed subgoals.
%%% BodyStruct will look like:
%%%
%%%		[call( _ , Name / Arity, [A1,...,An]),
%%%		 call( _ , Name / Arity, [A1,...,An]),
%%%		.
%%%		.
%%%		.
%%%		 call( _ , Name / Arity, [A1,...,An])]
%%%

parse_body(BodyList,BodyStruct) :-
	mapcar(BodyList,parse_call,BodyStruct).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% parse_call(Call, CallStruct)
%%%
%%% +Call	The call. This can in fact be uninstantiated, but
%%%		if it is, it is treated as if it were a 'call(X)'
%%%
%%% -CallStruct
%%%
%%%
%%%
%%% There are 3 cases:
%%% 1) The call is a variable, in which case it is treated as if
%%%    it were a 'call(X)'
%%%
%%% 2) The call is a conditonal goal.  Parse each side of the '->' separately.
%%%    It is an error if the right side have more than on all.
%%%
%%% 3) An ordinary call.  Parse its arguments.
%%%

parse_call(VarCall,call(call/1,[VarCall])) :-	%Var call
	var(VarCall),!,
	format("parse_call: uninstantiated call slipped thru flatten_conjunction",[]).


parse_call('='(X,Y),unify(A1,A2)) :- !,
	mapcar([X,Y],parse_term,[A1,A2]).
parse_call(Builtin,builtin(Name/Arity,ArgStruct,_)) :-
	functor(Builtin,Name,Arity),
	builtin(Name/Arity),!,
	Builtin =.. [_|Args],
	mapcar(Args,parse_term,ArgStruct).
parse_call(Call,call(Name/Arity,ArgStruct)) :- %Ordinary call
	Call =.. [_|Args],
	functor(Call,Name,Arity),
	mapcar(Args,parse_term,ArgStruct).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% parse_term(Term,TermStruct)
%%%
%%% +Term	A constant, list, structure or variable
%%%
%%% -TermStruct	The parsed term
%%%
%%%
%%% 
%%% Parse a TROLL-term to a more informative structure
%%%
%%% A variable X   is parsed to	var(X,_,_)
%%% []		   - " -	nil
%%% An atom a	   - " -	atom(a)
%%% A number n	   - " -	number(n)
%%% A list [A1,...] - " -	cons(A1,...) (recursively)
%%% A struct. f(..) - " -	struct(...)  (recursively)
%%%


parse_term(X,var(X,_,_)) :- var(X),!.
parse_term([],nil) :- !.
parse_term(X,atom(X)) :- atom(X),!.
parse_term(X,number(X)) :- number(X),!.
parse_term([H|T],cons(Car,Cdr)) :- !, parse_term(H,Car),parse_term(T,Cdr).
parse_term(X,struct(F/Arity,A)) :-
	functor(X,_,Arity),
	X =.. [F|A1],!,
	mapcar(A1,parse_term,A).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% flatten_conjunction(Conjunction,List)
%%%
%%% +Conjuction	Conjuction (A,(B,(C,(D,...))))
%%%
%%% +List	Flat list [A,B,C,D,...]
%%%
%%%

flatten_conjunction(Conjunction,List) :-
	flatten_conjunction(Conjunction,List,[]).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% flatten_conjunction(Conjunction) [DCG]
%%%
%%% +Conjunction	Conjunction (A,(B,(C,...)))
%%%
%%%
%%% 
%%% I think this one does a little bit too much, test it thoroughly.
%%% 

%%% flatten_conjunction(true) --> !.

flatten_conjunction(X) --> {var(X)},[call(X)].
flatten_conjunction((P,Q)) --> {var(P)},[call(P)],flatten_conjunction(Q).
flatten_conjunction((P,Q)) --> !,flatten_conjunction(P),flatten_conjunction(Q).
flatten_conjunction(G) --> [G].



    
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% compile_predicate(OutputMode,OutStream,Struct)
%%%
%%% +OutputMode	'lisp' or 'prolog' to produce different output.
%%%
%%% +OutStream	Stream to print object code to
%%%
%%% +Struct	Parse-tree to compile
%%%
%%%
%%%
%%% If the parsing failed, Struct indicates this, and the failure is reported.
%%% Otherwise compile the predicate, if this fails the third clause reports 
%%% to the user.
%%%

compile_predicate(_,_,parsing_failed,_) :-
	format(">??Failed to parse ~n",[]).
	
compile_predicate(OutputMode,OutStream,predicate(NameArity,Struct),Predicate) :-
%	Struct = predicate(NameArity,_),
	format(">>> ~w ",[NameArity]),
	generate_code(predicate(NameArity,Struct),Code,[]),
	printcode(OutputMode,OutStream,[predicate(NameArity,Code)],Predicate),
	format("compiled ",[]).

compile_predicate(_,_, Struct,_) :-
	format(">??Failed to compile ~w~n",[Struct]).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% generate_code(Struct) [DCG]
%%%
%%% +Struct	The parse tree
%%%
%%%
%%% Generates the object code from the parse tree.
%%%
%%% Zero-arity predicates are compiled according to the first clause.
%%% General clauses are compiled according to the second clause.

generate_code(predicate(_,[Clause])) --> !,
	compile_clause(Clause,no_switch).

generate_code(predicate(_/0,Clauses)) --> !,
	clause_instructions(_,Clauses).

generate_code(predicate(_,Clauses)) -->
 	try_instructions(Clauses).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% analyze_variables(Clauses)
%%%
%%% +-Clauses	The clause to analyze.
%%%
%%%
%%%
%%% The analyzation can be split into a number of parts:
%%% 1) Determine the variables permanent/temporary-status,
%%%    This characteristic is common to all occurences of one variable
%%%    and should therefore be distributed by simply instantiating the 
%%%    variable to something suitable, like temp(X) or perm(X).
%%%    The most natural way to do this seems to be to first make all
%%%    variables occuring in the body permanent and after that make
%%%    all others temporary.
%%%
%%% 2) Annotate each occurence with either 'first' or 'nth' so the code-
%%%    generator doesn't have to find out which instruction to generate.
%%%
%%% 3) Determine which variables that should be 'dangerous'
%%%
%%% 4) More?
%%% 		clause(_, _ , hb( head( ... ),body( ... ))),

analyze_variables([]).
analyze_variables([X|Xs]) :-
	analyze_clause(X),
	analyze_variables(Xs).

%%temp% analyze_clause(Clause) :-
%%temp% 	permvars(Clause,NrPermVars),
%%temp% 	head(Clause,Head),
%%temp% 	body(Clause,Body),
%%temp% 	body_lifespan(Body,0,NrPermVars),
%%temp% 	head_lifespan(Head),
%%temp% 	sequence_head(Head,[],Found),
%%temp% 	sequence_calls(Body,Found,Found2),
%%temp% 	sequence_voids(Found2).

analyze_clause(Clause) :-
	permvars(Clause,NrPermVars),
	head(Clause,Head),
	body(Clause,Body),
	sequence_head(Head,[],Found),
	sequence_calls(Body,Found,Found2),
	sequence_voids(Found2),
	body_lifespan(Body,0,NrPermVars),
	head_lifespan(Head).

