%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%%	File:		clauses.pl
%%%	Author:		Peter Olin
%%%	Date:		June 6, 1989
%%%	Modified:	July 18, 1990 for Andorra by Johan Bevemyr
%%%	Purpouse:	To generate try_me_else, retry_me_else, 
%%%			trust_me_else_fail and switch_on_term instructions.
%%%
%%%
%%% HISTORY
%%% Author	Date		Description
%%% PO		6/6-89		Changed variable names.
%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% clause_instructions(Labels,Struct) [DCG]
%%%
%%% +Labels	A structure of labels.
%%%
%%% +Struct	The parsed predicate
%%%
%%%
%%% 
%%% Generates the code for the clauses, including the try-instructions.
%%%

clause_instructions(LABELS,SrcStruct) --> one_clause(LABELS,SrcStruct).
clause_instructions(LABELS,SrcStruct) --> two_clauses(LABELS,SrcStruct).
clause_instructions(LABELS,SrcStruct) --> several_clauses(LABELS,SrcStruct).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% one_clause(Labels,Struct) [DCG]
%%%
%%%
%%% +Labels	Used to extract the addres to the first clause.
%%%
%%% +Struct	The parsed predicate
%%%
%%%
%%%
%%%  If the predicate (Struct) consists of only one clause, don't generate 
%%%  any try-instructions.
%%%
%%% The generated code will look like:
%%%
%%%	Start: <Code for first and only clause>
%%%

one_clause(labels(Lv,_,_,_),[Clause]) -->
	{label(Clause,Lv)},
	[try_me],
	compile_clause(Clause,no_switch).
	

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% two_clauses(Labels,Struct) [DCG]
%%%
%%% +Labels	Used to extract the addres to the first clause.
%%%
%%% +Struct	The parsed predicate
%%%
%%%
%%%
%%% If the predicate consists of two clauses, generate the following code:
%%%
%%%	Start: try_me_else(right,Next)
%%%	       <Code for first clause>
%%%	Next:  trust_me_else_fail
%%%	       <Code for second clause>
%%%


two_clauses(labels(_,_,_,_),[C1,C2]) --> !,
	[try_me_else(L)],
	compile_clause(C1,no_switch),
	[label(L),trust_me],
	compile_clause(C2,no_switch).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% several_clauses(Labels,Struct)
%%%
%%% +Labels	Used to extract the addres to the first clause.
%%%
%%% +Struct	The parsed predicate
%%%
%%%
%%% Three or more clauses. Generate a try_me_else for
%%% the first, then one  retry_me_else for each clause except for
%%% the last one, which should be accompanied by a trust_me_else_fail.
%%%
%%% So the code will look like:
%%%
%%%	Start: try_me_else(right,Next1)
%%%	       <Code for 1st clause>
%%%	Next1:  retry_me_else(Next2)
%%%	       <Code for 2nd clause>
%%%	Next2: retry_me_else(Next3)
%%%		...
%%%	Last:  trust_me_else_fail
%%%	       <Code for last clause>

several_clauses(labels(_,_,_,_),[C|Clauses]) -->
	[try_me_else(L1)],
	compile_clause(C,no_switch),
	several_clauses_aux(L1,L2,Clauses,LastClause),
	[label(L2),trust_me],
	compile_clause(LastClause,no_switch).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% several_clauses_aux(Start, Last, Clauses, LastClause)
%%%
%%% +Start	Address to retry-instruction
%%%
%%% -Last	Address to trust_instruction
%%%
%%% +Clauses	
%%%
%%% -LastClause	The last clause in the predicate
%%%
%%%
%%% Terminates and returns the last clause if there are only two clauses 
%%% left.
%%%

several_clauses_aux(L1,L2,[C,LastClause],LastClause) --> !,
	[label(L1),retry_me_else(L2)],
	compile_clause(C,no_switch).

several_clauses_aux(L1,L3,[C|RestClauses],LastClause)-->
	[label(L1),retry_me_else(L2)],
	compile_clause(C,no_switch),
	several_clauses_aux(L2,L3,RestClauses,LastClause).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% compile_clause(Clause)
%%%
%%% +Clause	The parsed clause
%%%
%%%
%%%
%%% compiles the Head and Body of a clause placing the code
%%% at Label.
%%%

compile_clause(Clause,Switch_flag) -->
	{compile_clause1(Clause,Arity,PV,Switch_flag,Code,[])},
	allocate_registers1(Code,regcount(Arity,0),PV).

compile_clause1(Clause,Arity,PV,Switch_flag) -->
   {clause_arity(Clause,Arity),
    body(Clause,Body)
    },
   head_code(Clause,PV,Switch_flag),
   body_code(Body),
   {write('.'),flush_output(user_output)}.


clause_arity(Clause,Arity) :- head(Clause,Head), length(Head,Arity).
