/*    File:	 engine.c  (~bevemyr/KAM/Emulator/engine.c)
 *    Author:	 Johan Bevemyr
 *    Created:	 Tue Mar 24 14:52:13 1992
 *    Purpose:   
 */ 

#include "include.h"

#define Get_OpCode(PC)     ((opcode) *PC++)
#define Get_Reg(PC)        ((int) *PC++)
#define Get_Const(PC)      ((TAGGED) *PC++)
#define Get_Functor(PC)    ((TAGGED) *PC++)
#define Get_Definition(PC) ((definition *) *PC++)
#define Get_Index(PC)      ((int) *PC++)
#define Get_Label(PC)      ((code *) *PC++)
#define Get_UseArgs(PC)    ((long *) PC)

/* We cache the heap top pointer in a register using the following defs */

#define H                  htop
#define LoadH              htop = heap_current
#define SaveH              heap_current = htop

#define Dispatch(PC,O)     (((code *) (((int) PC) + ((int) *(PC+O)))) + O)

BOOL unify(x,y,heap_uncond)
     register TAGGED x,y, *heap_uncond;
{
  Deref(x,x);  Deref(y,y);
  
  if(x != y)
    {
      if(IsVAR(x))
	{
	  if(IsVAR(y) && x < y)
	    {
	      Bind(y,x);
	    }
	  else
	    {
	      Bind(x,y);
	    }
	}
      else if(IsVAR(y)) 
	{
	  Bind(y,x);
	}
      else if(IsSTR(x) && IsSTR(y) &&
	      (GetFunctor(x) == GetFunctor(y)))
	{
	  register int i;
	  register TAGGED *xa, *ya;
	  
	  i = GetArity(x);
	  xa = GetArg(x,0);
	  ya = GetArg(y,0);
	  
	  while(i--)
	    {
	      if(unify(*xa,*ya, heap_uncond) == FALSE)
		return FALSE;
	      xa++; ya++;
	    }
	}
      else
	return FALSE;
      return TRUE;
    }
  else
    return TRUE;
}

BOOL engine(def)
    register definition *def;
{
  register code *pc;
  TAGGED aregs[255];
  register TAGGED *areg = aregs;
  register choicepoint *choice;
  register TAGGED *s;
  register TAGGED *heap_uncond;
  register TAGGED *htop;
  register int arity;

  /**********************************************************************
   *  execute initial definition
   */

  switch(def->enter_instruction) {
  case ENTER_EMULATED:
    arity = ArityOf(def->name);
    pc = def->entry_code.incoreinfo;
    break;
    
  case ENTER_C:
  case ENTER_UNDEFINED:
    printf("predicate %s/%d undefined\n",
	   GetString(FunctorToAtom(def->name)),
	   ArityOf(def->name));
    return FALSE;
  }

  LoadH;

  /**********************************************************************
   *
   *  This is the main decoding loop. Why not try threaded code instead?
   *  It is much faster than this!
   *  
   */

 read_start:

  switch(Get_OpCode(pc)) {
  case SWITCH_ON_TERM:
  switch_on_term_read:
    {
      Deref(X(0),X(0));
      switch(TagOf(X(0))) {
      case VAR:
	pc = Dispatch(pc,0);
	break;
      case CON:
	pc = Dispatch(pc,1);
	break;
      case STR:
	pc = Dispatch(pc,2);
	break;
      default:
	printf("switch_on_term -- no such tag %d",TagOf(X(0)));
	return FALSE;
      }
      goto read_start;
    }

  case SWITCH_ON_CONSTANT:
  switch_on_constant_read:
    {
      register int i;
      register TAGGED c;
      
      i = Get_Reg(pc);

      do {
	c = Get_Const(pc);
	if(X(0) != c) 
	  pc += 1;
	else 
	  break;
      } while (--i);
      pc = Dispatch(pc,0);
      goto read_start;
    }

  case SWITCH_ON_STRUCTURE:
  switch_on_structure_read:
    {
      register int i;
      register TAGGED c, f;
      
      i = Get_Reg(pc);
      f = GetFunctor(X(0));

      do {
	c = Get_Functor(pc);
	if(f != c) 
	  pc += 1;
	else 
	  break;
      } while (--i);
      pc = Dispatch(pc,0);
      goto read_start;
    }

  case TRY:
  try_read:
    {
      Make_CP(stack_current, trail_current, H, choice, pc+1, arity);

      heap_uncond = H;

      pc = Dispatch(pc,0);

      goto read_start;
    }

  case RETRY:
  retry_read:
    {
      choice->alternative = pc+1;
      pc = Dispatch(pc,0);
      goto read_start;
    }

  case TRUST:
  trust_read:
    {
      choice = choice->last_choice;
      heap_uncond = choice->heap_top;
      pc = Dispatch(pc,0);
      goto read_start;
    }

  case GET_STRUCTURE:
  get_structure_read:
    {
      register int i;
      register TAGGED f, Xi;

      f = Get_Functor(pc);
      i = Get_Reg(pc);

      Deref(Xi,X(i));
      
      if(IsVAR(Xi)) {
	register TAGGED new;
	Make_STR(H,new,f);
	Bind(Xi,new);
	goto write_start;
      } else if(IsSTR(Xi) && (GetFunctor(Xi) == f)) {
	s = GetArg(Xi,0);
	goto read_start;
      } else
	goto fail;
    }
      
  case GET_CONSTANT:
  get_constant_read:
    {
      register int i;
      register TAGGED c, Xi;
      
      c = Get_Const(pc);
      i = Get_Reg(pc);

      Deref(Xi,X(i));
      
      if (Xi == c) {
	goto read_start;
      } else if (IsVAR(Xi)) {
	Bind(Xi,c);
	goto read_start;
      } else
	goto fail;
    }

  case GET_VARIABLE:
  get_variable_read:
    {
      register int i,n;

      n = Get_Reg(pc);
      i = Get_Reg(pc);

      X(n) = X(i);

      goto read_start;
    }

  case GET_VALUE:
  get_value_read:
    {
      register int i,n;
      
      n = Get_Reg(pc);
      i = Get_Reg(pc);

      Unify(X(n),X(i));
      
      goto read_start;
    }

  case PUT_STRUCTURE:
  put_structure_read:
    {
      register int i;
      register TAGGED f;

      f = Get_Functor(pc);
      i = Get_Reg(pc);
      
      Make_STR(H, X(i), f);
      
      goto write_start;
    }

  case PUT_DEFINITION:
  put_definition_read:
    {
      register int i;
      register TAGGED f;

      f = (TAGGED) Get_Definition(pc);
      i = Get_Reg(pc);
      
      Make_STR(H, X(i), PointerToTerm(f));
      
      goto write_start;
    }

  case PUT_CONSTANT:
  put_constant_read:
    {
      register int i;
      register TAGGED c;

      c = Get_Const(pc);
      i = Get_Reg(pc);

      X(i) = c;

      goto read_start;
    }

  case PUT_VARIABLE:
  put_variable_read:
    {
      register int i,n;
      
      n = Get_Reg(pc);
      i = Get_Reg(pc);
      
      LoadHVA(H,X(n));
      X(i) = X(n);
     
      goto read_start;
    }

  case PUT_VALUE:
  put_value_read:
    {
      register int i,n;
      
      n = Get_Reg(pc);
      i = Get_Reg(pc);
      
      X(i) = X(n);
      
      goto read_start;
    }

  case PUT_VOID:
  put_void_read:
    {
      register int i;
      i = Get_Reg(pc);
      LoadHVA(H,X(i));
      goto read_start;
    }
    
  case UNIFY_VOID:
    {
      register int n;

      n = Get_Reg(pc);

      s += n;

      goto read_start;
    }

  case UNIFY_VARIABLE:
    {
      register int n;

      n = Get_Reg(pc);

      X(n) = *(s++);

      goto read_start;
    }

  case UNIFY_VALUE:
    {
      register int n;

      n = Get_Reg(pc);
      
      Unify(*(s++),X(n));

      goto read_start;
    }

  case UNIFY_CONSTANT:
    {
      register TAGGED c, Si;

      c = Get_Const(pc);

      Deref(Si,*(s++));

      if(Si == c) {
	goto read_start;
      } else if (IsVAR(Si)) {
	Bind(Si,c);
	goto read_start;
      } else
	goto fail;
    }

  case EXECUTE:
  execute_read:
    {
      def = Get_Definition(pc);
      
    execute_enter:

      switch(def->enter_instruction) {
      case ENTER_EMULATED:
	arity = ArityOf(def->name);
	pc = def->entry_code.incoreinfo;
	goto read_start;

      case ENTER_C:
	SaveH;
	switch((def->entry_code.cinfo)(areg)) {
	case FALSE:
	  goto fail;
	case TRUE:
	  { 
	    register TAGGED goal;
	    register int i;

	    LoadH;
	  
	    Deref(goal,X(ArityOf(def->name)-1));

	    def = (definition *) TermToPointer(GetFunctor(goal));
	    
	    i = arity = ArityOf(def->name);
	    
	    while(i--) {
	      X(i) = *GetArg(goal,i);
	    }
	    goto execute_enter;
	  }
	}

      case ENTER_UNDEFINED:
	printf("predicate %s/%d undefined\n",
	       GetString(FunctorToAtom(def->name)),
	       ArityOf(def->name));
	goto fail;
      }
    }

  case META_EXECUTE:
  meta_execute_read:
    {
      register int i, n;
      register TAGGED name, goal;

      i = Get_Reg(pc); /* pointer to goal */
      n = Get_Reg(pc); /* pointer to cont */

      Deref(goal,X(i));
      
      if(IsSTR(goal)) {
	name = GetFunctor(goal);
	arity = ArityOf(goal);
	i = arity;

	X(i) = X(n); /* move continuation pointer */

	while(i--) {
	  X(i) = *GetArg(goal,i);
	}
      } else if (IsCON(goal)) {
	name = MakeFunctor(goal,0);
	arity = 0;
      } else
	goto fail;

      def = get_definition(name);

      goto execute_enter;
    }

  case DEMO_EXECUTE:
  demo_execute_read:
    {
      register int i;
      register TAGGED goal;

      i = Get_Reg(pc);

      Deref(goal,X(i));

      def = (definition *) TermToPointer(GetFunctor(goal));

      arity = ArityOf(def->name);

      i = arity;

      while(i--) {
	X(i) = *GetArg(goal,i);
      }
      
      goto execute_enter;
    }

  case INLINE:
  inline_read:
    {
      register int fnk;
      
      fnk = Get_Index(pc);
      
      pc += 1;

      SaveH;

      if ((GetInlineFnk(fnk))(Get_UseArgs(pc), areg) == FALSE) {
	pc = Dispatch(pc,-1);
      } else {
	pc += GetInlineArity(fnk);
      }

      LoadH;

      goto read_start;
    }

  case BUILTIN:
  builtin_read:
    {
      register int fnk;
      
      fnk = Get_Index(pc);
      
      SaveH;

      if ((GetInlineFnk(fnk))(Get_UseArgs(pc), areg) == FALSE) goto fail;

      LoadH;

      pc += GetInlineArity(fnk);

      goto read_start;
    }

  case HALT:
  halt_read:
    return TRUE;
  }

 /**********************************************************************
  *
  * Write mode versions of the instructions
  *
  */

 write_start:

  switch(Get_OpCode(pc)) {
  case SWITCH_ON_TERM:
    goto switch_on_term_read;
  case SWITCH_ON_CONSTANT:
    goto switch_on_constant_read;
  case SWITCH_ON_STRUCTURE:
    goto switch_on_structure_read;

  case TRY:
    goto try_read;
  case RETRY:
    goto retry_read;
  case TRUST:
    goto trust_read;

  case GET_STRUCTURE:
    goto get_structure_read;
  case GET_CONSTANT:
    goto get_constant_read;
  case GET_VARIABLE:
    goto get_variable_read;
  case GET_VALUE:
    goto get_value_read;

  case PUT_STRUCTURE:
    goto put_structure_read;
  case PUT_DEFINITION:
    goto put_definition_read;
  case PUT_CONSTANT:
    goto put_constant_read;
  case PUT_VARIABLE:
    goto put_variable_read;
  case PUT_VALUE:
    goto put_value_read;
  case PUT_VOID:
    goto put_void_read;

  case UNIFY_VOID:
    {
      register int i;

      i = Get_Reg(pc);
      
      do {
	CreateHVA(H);
      } while(--i);

      goto write_start;
    }

  case UNIFY_VARIABLE:
    {
      register int n;

      n = Get_Reg(pc);
      
      LoadHVA(H,X(n));
      
      goto write_start;
    }

  case UNIFY_VALUE:
    {
      register int n;
      
      n = Get_Reg(pc);
      
      PushOnHeap(H,X(n));
      
      goto write_start;
    }

  case UNIFY_CONSTANT:
    {
      register TAGGED c;

      c = Get_Const(pc);
      
      PushOnHeap(H,c);

      goto write_start;
    }

  case EXECUTE:
    goto execute_read;
  case META_EXECUTE:
    goto meta_execute_read;
  case DEMO_EXECUTE:
    goto demo_execute_read;
  case INLINE:
    goto inline_read;
  case BUILTIN:
    goto builtin_read;
  case HALT:
    goto halt_read;
  }

/**********************************************************************
 * 
 * Restore old state from choicepoint
 *
 */

 fail:
  {
    register int i;

    UnwindTrail(choice->trail_top);

    i = choice->arity;

    stack_current = (TAGGED *) (((long) choice) + CHOICESIZE + i);

    while(i--) {
      X(i) = choice->areg[i];
    }

    H = heap_uncond = choice->heap_top;
    pc = choice->alternative;
    
    goto read_start;
  }
  
}

