%   File   : WRITE.PL
%   Author : Richard A. O'Keefe
%   Updated: 22 October 1984
%   Purpose: Portable definition of write/1 and friends.

/*  Adapted from shared code written by the same author; all changes */
/*  Copyright (C) 1988, Swedish Institute of Computer Science. */

%:- public
%	write_canonical/1, write_canonical/2,
%	display/1,
%	print/1, print/2,
%	write/1, write/2,
%	writeq/1, writeq/2,
%	listing/0,
%	listing/1,
%	portray_clause/1, portray_clause/2.
%

/*  WARNING!
    This file was written to assist portability and to help people
    get a decent set of output routines off the ground fast.  It is
    not particularly efficient.  Information about atom names and
    properties should be precomputed and fetched as directly as
    possible, and strings should not be created as lists!

    The four output routines differ in the following respects:
    [a] write_canonical doesn't use operator information or handle {X} or
	[H|T] specially.  The others do.
    [b] print calls portray/1 to give the user a chance to do
	something different.  The others don't.
    [c] writeq puts quotes around atoms that can't be read back.
	The others don't.
    Since they have such a lot in common, we just pass around
    arguments (SynStyle, LexStyle) saying what to do.

    In a Prolog which supports strings;
	write(<string>) should just write the text of the string, this so
	that write("Beware bandersnatch") can be used.  The other output
	commands should quote the string.

    listing(Preds) is supposed to write the predicates out so that they
    can be read back in exactly as they are now, provided the operator
    declarations haven't changed.  So it has to use writeq.  $VAR(X)
    will write the atom X without quotes, this so that you can write
    out a clause in a readable way by binding each input variable to
    its name.
*/

% Priority 999 is o.k. if printed e.g. as elements of a list. /MC

writeq_quick(Term) :- var(Term), '$write'(Term).
writeq_quick(Term) :- atomic(Term), '$write'(Term).
writeq_quick(Term) :- generic(Term), '$write'(Term).

write_quick(Term) :- var(Term), '$display'(Term).
write_quick(Term) :- atomic(Term), '$display'(Term).
write_quick(Term) :- generic(Term), '$display'(Term).

display_quick(Term) :- var(Term), '$display'(user_output, Term).
display_quick(Term) :- atomic(Term), '$display'(user_output, Term).
display_quick(Term) :- generic(Term), '$display'(user_output, Term).

write_canonical(Stream, Term) :-
	'$output'(Curr, Stream), !,
        write_canonical(Term), 
	'$output'(_, Curr).
write_canonical(Stream, Term) :-
	illarg(Stream, stream, _, write_canonical(Stream,Term), 1).

write_canonical(Term) :-
	writeq_quick(Term), !.
write_canonical(Term) :- 
	write_out(Term, noop, quote, 1200, 0, 0, '(', 2'100, _).

display(Term) :-
	display_quick(Term), !.
display(Term) :-
	'$output'(Curr, user),
	write_out(Term, noop, noquote, 1200, 0, 0, '(', 2'100, _), 
	'$output'(_, Curr).

print(Stream, Term, Limit) :-
	'$output'(Curr, Stream),
	write_out(Term, print(Limit), quote, 1200, 0, 0, '(', 2'100, _),
	'$output'(_, Curr).

print(Stream, Term) :-
	'$output'(Curr, Stream), !,
	print(Term), 
	'$output'(_, Curr).
print(Stream, Term) :-
	illarg(Stream, stream, _, print(Stream,Term), 1).

print(Term) :-
	write_out(Term, print(1000000), noquote, 1200, 0, 0, '(', 2'100, _).

write(Stream, Term) :-
	'$output'(Curr, Stream), !,
	write(Term), 
	'$output'(_, Curr).
write(Stream, Term) :-
	illarg(Stream, stream, _, write(Stream,Term), 1).

write(Term) :-
	write_quick(Term), !.
write(Term) :-
	write_out(Term, op, noquote, 1200, 0, 0, '(', 2'100, _).


writeq(Stream, Term) :- 
	'$output'(Curr, Stream), !,
	writeq(Term), 
	'$output'(_, Curr).
writeq(Stream, Term) :-
	illarg(Stream, stream, _, writeq(Stream,Term), 1).

writeq(Term) :-
	writeq_quick(Term), !.
writeq(Term) :-
	write_out(Term, op, quote, 1200, 0, 0, '(', 2'100, _).


%   maybe_paren(P, Prio, Char, Ci, Co)
%   writes a parenthesis if the context demands it.
%   Context = 2'000 for alpha
%   Context = 2'001 for quote
%   Context = 2'010 for other
%   Context = 2'100 for punct

maybe_paren(P, Prio, Lpar, '(', _, 2'100) :-
	P > Prio, !,
	'$display'(Lpar).
maybe_paren(_, _, Lpar, Lpar, C, C).

maybe_paren(P, Prio, _, 2'100) :-
	P > Prio, !,
	'$display'(')').
maybe_paren(_, _, C, C).



%   maybe_space(LeftContext, TypeOfToken)
%   generates spaces as needed to ensure that two successive
%   tokens won't run into each other.

maybe_space(Ci, Co) :-
	(   Ci\/Co<2'100, Ci#Co<2'010 -> '$put'(0' )
        ;   true
        ).

/*
sticky_contexts(alpha, alpha).
sticky_contexts(quote, quote).
sticky_contexts(other, other).
sticky_contexts(alpha, quote).
sticky_contexts(quote, alpha).
*/

%   write_out(Term, SynStyle, LexStyle, Prio, PrePrio, Depth, Lpar, Ci, Co)
%   writes out a Term in given SynStyle, LexStyle
%   at nesting depth Depth
%   in a context of priority Priority (that is, expressions with
%   greater priority must be parenthesized), 
%   and prefix operators =< PrePrio must be parenthesized,
%   where the last token to be
%   written was of type Ci, and reports that the last token it wrote
%   was of type Co.

write_out(Term, _, _, _, _, _, _, Ci, 2'000) :-
	var(Term), !,
	maybe_space(Ci, 2'000),
	'$write'(Term).
write_out('$VAR'(N), SynStyle, LexStyle, _, _, Depth, _, Ci, Co) :- !,
	Depth1 is Depth+1,
	write_VAR(N, SynStyle, LexStyle, Depth1, Ci, Co).
write_out(_, print(Limit), _, _, _, Depth, _, Ci, 2'010) :-
	Depth > Limit, !,
	maybe_space(Ci, 2'010),
	'$display'(...).
write_out(Term, print(_), _, _, _, _, _, _, 2'000) :-
	'USERCALL'(current_predicate(_,portray(_))),
	(   \+'USERCALL'(portray(Term)) ->
	    fail		 % portray might bind variables
        ;   true
        ), !.
write_out(Atom, _, LexStyle, _, PrePrio, _, Lpar, _, 2'100) :-
	atom(Atom),
	current_prefixop(Atom, _, P, _),
	P =< PrePrio, !,
	'$display'(Lpar),
	write_atom(LexStyle, Atom, 2'100, _),
	'$put'(0')).
write_out(Atom, _, LexStyle, _, _, _, _, Ci, Co) :-
	atom(Atom), !,
	write_atom(LexStyle, Atom, Ci, Co).
write_out(N, _, _, _, _, _, _, Ci, 2'000) :-
	number(N), !,
	(   N < 0 -> maybe_space(Ci, 2'010)
	;   maybe_space(Ci, 2'000)
	),
	'$write'(N).
write_out(Term, noop, LexStyle, _, _, Depth, _, Ci, 2'100) :-
	functor(Term, Atom, Arity), !,
	write_atom(LexStyle, Atom, Ci, _),
	Depth1 is Depth+1,
	write_args(0, Arity, Term, noop, LexStyle, Depth1).
write_out({Term}, SynStyle, LexStyle, _, _, Depth, _, _, 2'100) :- !,
	'$put'(0'{),
	Depth1 is Depth+1,
	write_out(Term, SynStyle, LexStyle, 1200, 0, Depth1, '(', 2'100, _),
	'$put'(0'}).
write_out([Head|Tail], SynStyle, LexStyle, _, _, Depth, _, _, 2'100) :- !,
	'$put'(0'[),
	Depth1 is Depth+1,
	write_out(Head, SynStyle, LexStyle, 999, 0, Depth1, '(', 2'100, _),
	write_tail(Tail, SynStyle, LexStyle, Depth1).
write_out((A,B), SynStyle, LexStyle, Prio, _, Depth, Lpar, Ci, Co) :- !,
	%  This clause stops writeq quoting commas.
	Depth1 is Depth+1,
	maybe_paren(1000, Prio, Lpar, Lpar1, Ci, C1),
	write_out(A, SynStyle, LexStyle, 999, 0, Depth1, Lpar1, C1, _),
	'$put'(0',),
	write_out(B, SynStyle, LexStyle, 1000, 1000, Depth1, '(', 2'100, C2),
	maybe_paren(1000, Prio, C2, Co).
write_out(Term, SynStyle, LexStyle, Prio, PrePrio, Depth, Lpar, Ci, Co) :-
	functor(Term, F, N), !,
	Depth1 is Depth+1,
	write_out(N, F, Term, SynStyle, LexStyle, Prio, PrePrio, Depth1, Lpar, Ci, Co).
write_out(Term, _, _, _, _, _, _, _, _) :-
	'$write'(Term).

write_out(1, F, Term, SynStyle, LexStyle, Prio, _, Depth, Lpar, Ci, Co) :-
        current_postfixop(F, _, P, O), !,
	(current_infixop(F, _, _, _, _) -> eq(O1,1200); eq(O1,O)),
	maybe_paren(O1, Prio, Lpar, Lpar1, Ci, C1),
	arg(1, Term, A),
	write_out(A, SynStyle, LexStyle, P, 1200, Depth, Lpar1, C1, C2),
	write_atom(LexStyle, F, C2, C3),
	maybe_paren(O1, Prio, C3, Co).
write_out(1, F, Term, SynStyle, LexStyle, Prio, PrePrio, Depth, Lpar, Ci, Co) :-
        F \== -,
        current_prefixop(F, _, O, P), !,
	(eq(PrePrio,1200) -> O1 is P+1; eq(O1,O)),% for "fy X yf" etc. cases
	maybe_paren(O1, Prio, Lpar, _, Ci, C1),
	write_atom(LexStyle, F, C1, C2),
	arg(1, Term, A),
	write_out(A, SynStyle, LexStyle, P, P, Depth, ' (', C2, C3),
	maybe_paren(O1, Prio, C3, Co).
write_out(2, F, Term, SynStyle, LexStyle, Prio, PrePrio, Depth, Lpar, Ci, Co) :-
        current_infixop(F, _, P, O, Q), !,
	(eq(PrePrio,1200) -> O1 is Q+1; eq(O1,O)), %for "U xfy X yf" etc. cases
	maybe_paren(O1, Prio, Lpar, Lpar1, Ci, C1),
	arg(1, Term, A),
	write_out(A, SynStyle, LexStyle, P, 1200, Depth, Lpar1, C1, C2),
	write_atom(LexStyle, F, C2, C3),
	arg(2, Term, B),
	write_out(B, SynStyle, LexStyle, Q, Q, Depth, '(', C3, C4),
	maybe_paren(O1, Prio, C4, Co).
write_out(N, F, Term, SynStyle, LexStyle, _, _, Depth, _, Ci, 2'100) :-
	write_atom(LexStyle, F, Ci, _),
	write_args(0, N, Term, SynStyle, LexStyle, Depth).

write_VAR(N, SynStyle, _, _, Ci, 2'000) :-
	integer(N), N >= 0,
	SynStyle \== noop, !,
	maybe_space(Ci, 2'000),
	Letter is N mod 26 + 0'A,
	'$put'(Letter),
	(   N>=26 ->
	    Rest is N//26, '$write'(Rest)
	;   true
        ).
write_VAR(String, SynStyle, _, _, Ci, Co) :-
	nonvar(String),
	(   name(Atom, String) -> true
	;   eq(Atom = String)
	),
	atom(Atom),
	SynStyle \== noop, !,
	'$atom_mode'(Atom, Co),
	maybe_space(Ci, Co),
	'$display'(Atom).
write_VAR(X, SynStyle, LexStyle, Depth, Ci, 2'100) :-
	write_atom(LexStyle, '$VAR', Ci, _),
	write_args(0, 1, '$VAR'(X), SynStyle, LexStyle, Depth).

write_atom(noquote, Atom, Ci, Co) :-
	'$atom_mode'(Atom, Co),
	maybe_space(Ci, Co),
        '$display'(Atom).
write_atom(quote, Atom, Ci, Co) :-
	'$atom_mode'(Atom, Co),
	maybe_space(Ci, Co),
        '$write'(Atom).


%   write_args(DoneSoFar, Arity, Term, SynStyle, LexStyle, Depth)
%   writes the remaining arguments of a Term with Arity arguments
%   all told in SynStyle, LexStyle, given that DoneSoFar have already been written.

write_args(N, N, _, _, _, _) :- !,
	'$put'(0')).
write_args(I, _, _, print(Limit), _, Depth) :-
	Depth > Limit, !,
	write_args(I, Depth),
	'$display'(...),
	'$put'(0')).
write_args(I, N, Term, SynStyle, LexStyle, Depth) :-
	write_args(I, Depth),
	J is I+1,
	arg(J, Term, A),
	write_out(A, SynStyle, LexStyle, 999, 0, Depth, '(', 2'100, _),
	Depth1 is Depth+1,
	write_args(J, N, Term, SynStyle, LexStyle, Depth1).

write_args(0, _) :- !, '$put'(0'().
write_args(I, I) :- !, '$display'(', ').
write_args(_, _) :- '$put'(0',).



%   write_tail(Tail, SynStyle, LexStyle, Depth)
%   writes the tail of a list of a given SynStyle, LexStyle, Depth.

write_tail(Var, _, _, _) :-			%  |var]
	var(Var), !,
	'$put'(0'|),
	'$write'(Var),
	'$put'(0']).
write_tail([], _, _, _) :- !,			%  ]
	'$put'(0']).
write_tail(_, print(Limit), _, Depth) :-
	Depth > Limit, !,
	'$put'(0',),
	'$display'(...),
	'$put'(0']).
write_tail([Head|Tail], SynStyle, LexStyle, Depth) :- !, %  ,Head tail
	'$put'(0',),
	write_out(Head, SynStyle, LexStyle, 999, 0, Depth, '(', 2'100, _),
	Depth1 is Depth+1,
	write_tail(Tail, SynStyle, LexStyle, Depth1).
write_tail(Other, SynStyle, LexStyle, Depth) :-	%  |junk]
	'$put'(0'|),
	write_out(Other, SynStyle, LexStyle, 999, 0, Depth, '(', 2'100, _),
	'$put'(0']).


/*  The listing/0 and listing/1 commands are based on the Dec-10
    commands, but the format they generate is based on the "pp" command.
    The idea of portray_clause/1 came from PDP-11 Prolog.

    BUG: the arguments of goals are not separated by comma-space but by
    just comma.  This should be fixed, but I haven't the time right not.
    Run the output through COMMA.EM if you really care.
    (Now fixed by Mats C).

    An irritating fact is that we can't guess reliably which clauses
    were grammar rules, so we can't print them out in grammar rule form.

    We need a proper pretty-printer that takes the line width into
    acount, but it really isn't all that feasible in Dec-10 Prolog.
    Perhaps we could use some ideas from NIL?
*/

%listing :-
%   '$typein_module'(Module, Module),
%   '$current_predicate'(_, Pred, Module),
%   listing1(Pred, Module),
%   fail.
%listing.
%
%%listing(Arg) :- parse_functor_spec(Arg, X, M, listing1(X, M), listing(X)), !.
%listing(Arg0) :- 
%	get_module_v(Arg0, Arg, M),
%	(eq(Arg , N/A) -> true; eq(Arg , N)),
%	current_module(M),
%	'$current_predicate'(N, H, M),
%	functor(H, N, A),
%	listing1(H, M),
%	fail.
%listing(_).
%
%listing1(Pred, Module) :-
%	'$current_clauses'(Pred, Root, Module),
%	'$first_instance'(Root, _),
%	'$nl',
%	'$current_instance'(Head0, Body, Root, _),
%	(eq(Module , user) -> eq(Head, Head0)
%	; eq(Head ,Module:Head0)),
%	prettyvars((Head:-Body)),
%	portray_clause1((Head:-Body)).
%
%prettyvars(Term) :-
%	prettyvars(Term, Vars0, []),
%	keysort(Vars0, Vars),
%	set_singleton_vars(Vars, 0).
%
%prettyvars(Var) -->
%	{var(Var)}, !, [Var-[]].
%prettyvars([X|Xs]) --> !,
%	prettyvars(X),
%	prettyvars(Xs).
%prettyvars(X) -->
%	{functor(X, _, A)},
%	prettyvars(0, A, X).
%
%prettyvars(A, A, _) --> !.
%prettyvars(A0, A, X) -->
%	{A1 is A0+1},
%	{arg(A1, X, X1)},
%	prettyvars(X1),
%	prettyvars(A1, A, X).
%
%set_singleton_vars([], _).
%set_singleton_vars([X,Y|Xs], N0) :-
%	X==Y, !,
%	eq(X,'$VAR'(N0)-[]),
%	N is N0+1,
%	set_singleton_vars(Xs, X, N).
%set_singleton_vars(['$VAR'('_')-[]|Xs], N0) :-
%	set_singleton_vars(Xs, N0).
%
%set_singleton_vars([X|Xs], Y, N0) :-
%	X==Y, !,
%	set_singleton_vars(Xs, Y, N0).
%set_singleton_vars(Xs, _, N0) :-
%	set_singleton_vars(Xs, N0).
%
%
%% This must be careful not to bind any variables in Clause.
%portray_clause(Clause) :-
%	prettyvars(Clause),
%	portray_clause1(Clause),
%	fail.
%portray_clause(_).
%
%portray_clause(Stream, Clause) :-
%	'$output'(S, Stream),
%	portray_clause(Clause),
%	'$output'(_, S).
%
%portray_clause1(:-(Command)) :-
%	functor(Command, Key, 1),
%	current_op(_, fx, Key), !,
%	arg(1, Command, Body),
%	'list clauses'(Body, :-(Key), 8, Co),
%	write_fullstop(Co).
%portray_clause1((Pred:-Body)) :- !,
%	write_head(Pred, Ci),
%	(   eq(Body,true) -> write_fullstop(Ci)
%	;   'list clauses'(Body, 0, 8, Co),
%	    write_fullstop(Co)
%        ).
%portray_clause1((Pred-->Body)) :- !,
%	write_head(Pred, _),
%	'list clauses'(Body, 2, 8, Co),
%	write_fullstop(Co).
%portray_clause1(Module:Clause) :- !,
%	portray_clause2(Clause, Module).
%portray_clause1(Pred) :-
%	write_head(Pred, Ci),
%	write_fullstop(Ci).
%
%portray_clause2(Module:Clause, _) :- !, portray_clause2(Clause, Module).
%portray_clause2((Pred:-Body), Module) :- !,
%	portray_clause1((Module:Pred:-Body)).
%portray_clause2((Pred-->Body), Module) :- !,
%	portray_clause1((Module:Pred-->Body)).
%portray_clause2(Pred, Module) :-
%	write_head(Module:Pred, Ci),
%	write_fullstop(Ci).
%
%write_head(_:M:Head, Ci) :- !, write_head(M:Head, Ci).
%write_head(user:Head, Ci) :- !,
%	write_out(Head, op, quote, 1199, 1200, -1, '(', 2'100, Ci).  % writeq
%write_head(Head, Ci) :-
%	write_out(Head, op, quote, 1199, 1200, -1, '(', 2'100, Ci).  % writeq
%
%write_fullstop(Ci) :-
%	maybe_space(Ci, 2'010),
%	'$put'(0'.), '$nl'.
%
%
%'list clauses'((A,B), L, D, Co) :- !,
%	'list clauses'(A, L, D, _),
%	'list clauses'(B, 1, D, Co).
%'list clauses'((A;B), L, D, 2'100) :- !,
%	'list magic'(L, D),
%	'list disj'(A, 3, D),
%	'list disj'(B, D).
%'list clauses'((A->B), L, D, 2'100) :- !,
%	'list magic'(L, D),
%	E is D+4,
%	'list clauses'(A, 3, E, _),
%	'list clauses'(B, 5, E, _),
%	'$nl', '$tab'(D),
%	'$put'(0')).
%'list clauses'(!, 0, _, 2'100) :- !,
%	'$display'(' :- !').
%'list clauses'(!, 1, _, 2'100) :- !,
%	'$display'(', !').
%'list clauses'(!, 2, _, 2'100) :- !,
%	'$display'(' --> !').
%'list clauses'(Goal, L, D, Co) :-
%	'list magic'(L, D),
%	write_out(Goal, op, quote, 999, 0, -1, '(', 2'100, Co). % writeq
%
%
%'list magic'(0, D) :-
%	'$display'(' :-'),
%	'$nl', '$tab'(D).
%'list magic'(1, D) :-
%	'$put'(0',),
%	'$nl', '$tab'(D).
%'list magic'(2, D) :-
%	'$display'(' -->'),
%	'$nl', '$tab'(D).
%'list magic'(3, _) :-
%	'$display'('(   ').
%'list magic'(4, _) :-
%	'$display'(';   ').
%'list magic'(5, D) :-
%	'$display'(' ->'),
%	'$nl', '$tab'(D).
%'list magic'(:-(Key), D) :-
%	'$display'(':- '),
%	'$write'(Key),
%	'$nl', '$tab'(D).
%
%'list disj'((A;B), D) :- !,
%	'list disj'(A, 4, D),
%	'list disj'(B, D).
%'list disj'(Conj, D) :-
%	'list disj'(Conj, 4, D),
%	'$put'(0')).
%
%'list disj'((A->B), L, D) :- !,
%	E is D+4,
%	'list clauses'(A, L, E, _),
%	'list clauses'(B, 5, E, _),
%	'$nl', '$tab'(D).
%'list disj'(A, L, D) :-
%	E is D+4,
%	'list clauses'(A, L, E, _),
%	'$nl', '$tab'(D).

