% (c) Jonas Barklund, Uppsala University 1988.

/* #D
   read(?Answer)
   read(+Stream, ?Answer)
   read_with_dictionary(?Answer, ?Dictionary)
   read_with_dictionary(+Stream, ?Answer, ?Dictionary)
	reads a term from the current input stream (or Stream) and 
	unifies it with Answer.  Dictionary is a list of Atom=Variable pairs.
#D */

read(Answer) :-
   read_with_dictionary(Answer,_).

%

read(Stream,Answer) :-
   with_default_input(Stream, read_with_dictionary(Answer,_)).

%

read_with_dictionary(Stream,Answer,Dictionary) :-
   with_default_input(Stream, read_with_dictionary(Answer,Dictionary)).

%

read_with_dictionary(Answer,Variables) :-
   repeat,
   read_tokens(Tokens, Variables),
 ( maximal_term(Tokens,1200,T,S), no_garbage_at_end(S) ;
   syntax_error(Tokens) ),
 ( recorded('*SYNTAX-ERROR*',_,R) -> erase(R) ; true ), !,
   eq(Answer, T).

eq(X,X).
%

maximal_term(S0,M,T,S) :-
   minimal_term(S0,L,P,S1),
 ( P =< M -> max_term(S1,M,L,0,T,S) ;
   syntax_error(['Prefix',operator,with,precedence,P,in,context,M],S1) ).

:- mode max_term(+,+,+,+,-,?).

max_term(S1,M,F,N,T,S) :-
   atom(F), current_prefixop(F,_,P,Rp),
%  P =< M,
   maximal_term(S1,Rp,A1,S2), % !, removed as a test /Monika 890413
   functor(U,F,1), arg(1,U,A1),
   max_term(S2,M,U,P,T,S).

max_term([atom(F)|S2],M,A1,N,T,S) :-
   current_infixop(F,_,Lp,P,Rp),
 ( P =< M, N =< Lp ->
    ( maximal_term(S2,Rp,A2,S3) ->
         functor(U,F,2), arg(1,U,A1), arg(2,U,A2), max_term(S3,M,U,P,T,S) ;
      syntax_error(['Expression',expected],S2) ) ;
   syntax_error(['Infix',operator,with,precedence,P,in,context,M],S2) ).

max_term([atom(F)|S2],M,A1,N,T,S) :-
   current_postfixop(F,_,Lp,P),
 ( P =< M, N =< Lp -> !, functor(U,F,1), arg(1,U,A1), max_term(S2,M,U,P,T,S) ;
   syntax_error(['Postfix',operator,with,precedence,P,in,context,M],S2) ).

max_term([','|S2],M,A1,N,T,S) :-
   1000 =< M, N =< 999, maximal_term(S2,1000,A2,S3), !,
   eq(U,(A1,A2)), max_term(S3,M,U,1000,T,S).

max_term([X|S2],_,_,_,_,_) :-
   cannot_follow_expr(X,Msg,[cannot,follow,an,expression]),
   syntax_error(Msg,S2).

max_term(S,M,T,N,T,S).

:- mode cannot_follow_expr(+,-,+).

cannot_follow_expr(atom(_)) --> ['An',atom].

cannot_follow_expr(var(_,_)) --> ['A',variable].

cannot_follow_expr(atomic(_)) --> ['An',atomic,term].

cannot_follow_expr(' (') --> ['An',opening,bracket].

cannot_follow_expr('(') --> ['An',opening,bracket].

cannot_follow_expr('[') --> ['An',opening,bracket].

cannot_follow_expr('{') --> ['An',opening,bracket].

:- mode minimal_term(+,-,-,?).

minimal_term([var(Variable,_)|S1],T,P,S) :- !,
   eq(S1,S), eq(T,Variable), eq(P,0).

minimal_term([atom(-),number(Num)|S2],T,P,S) :- 
   number(Num), !,
   eq(S2,S), T is -Num, eq(P,0).

minimal_term([atom('$ARRAY'),'('|S2],T,P,S) :- !,
   maximal_term(S2,999,A,S3), compound_args(S3,L,S), eq(P,0),
   eq(T,'$$ARRAY'(nil,nil,U)), U =.. [array,A|L].

minimal_term([atom(Functor),'('|S2],T,P,S) :- !,
   maximal_term(S2,999,A,S3), compound_args(S3,L,S), eq(P,0),
   T =.. [Functor,A|L].

minimal_term([atom(Atom)|S1],T,P,S) :- 
   current_prefixop(Atom,_,P,_), !,
   eq(S1,S), eq(T,Atom).

minimal_term([atom(Atom)|S1],T,P,S) :- !,
   eq(S1,S), eq(T,Atom), eq(P,0).

minimal_term([number(Atom)|S1],T,P,S) :- !,
   eq(S1,S), eq(T,Atom), eq(P,0).

%minimal_term([atomic(Atomic)|S1],T,P,S) :- !,
%   eq(S1,S), eq(T,Atomic), eq(P,0).
%
%minimal_term([ubox(Ubox)|S1],T,P,S) :- !,
%   eq(S1,S), eq(T,Ubox), eq(P,0).

minimal_term([string(String)|S1],T,P,S) :- !,
   eq(S1,S), eq(T,String), eq(P,0).

minimal_term(['[',']'|S2],T,P,S) :- !,
   eq(S2,S), eq(T,[]), eq(P,0).

minimal_term(['['|S1],T,P,S) :- !,
   maximal_term(S1,999,L,S2), eq(T,[L|R]), eq(P,0), rest_of_list(S2,R,S).

minimal_term([' ('|S1],T,P,S) :- !,
   maximal_term(S1,1200,T,S2), eq(P,0),
   expect(')',S2,S).

minimal_term(['('|S1],T,P,S) :- !,
   maximal_term(S1,1200,T,S2), eq(P,0),
   expect(')',S2,S).

minimal_term(['{','}'|S2],T,P,S) :- !,
   eq(S2,S), eq(T,{}), eq(P,0).

minimal_term(['{'|S1],T,P,S) :- !,
   maximal_term(S1,1200,X,S2), eq(P,0), eq(T,{X}),
   expect('}',S2,S).

/* Database */
minimal_term(['|','|'|S2],T,P,S) :- !,
   eq(S2,S), eq(T,{}), eq(P,0).

minimal_term(['|'|S1],T,P,S) :- !,
   maximal_term(S1,1200,X,S2), eq(P,0), 
%    dbtrans_1(X, X1),
%    T = (prolog::dbtrans_2(X1, Vars)),
   eq(T,dbcall(X)),
   expect('|',S2,S).
/* End database */

minimal_term([end_of_file|S1],T,P,S) :- !,
   eq(S1,S), eq(T,end_of_file), eq(P,0).

minimal_term([Token|S1],_,_,_) :-
   syntax_error([Token,cannot,start,a,term],S1).

%

compound_args([','|S1],L0,S) :-!,
   maximal_term(S1,999,A,S2), eq(L0,[A|L]), compound_args(S2,L,S).

compound_args([')'|S1],L,S) :- !,
   eq(S1,S), eq(L,[]).

compound_args(S,_,_) :-
   syntax_error(['","',or,'")"',expected],S).

:- mode rest_of_list(+,-,?).

rest_of_list([','|S1],L0,S) :- !,
   maximal_term(S1,999,A,S2), eq(L0,[A|L]), rest_of_list(S2,L,S).

rest_of_list(['|'|S1],L,S) :- !,
   maximal_term(S1,999,L,S2),
   expect(']',S2,S).

rest_of_list([']'|S1],L,S) :- !,
   eq(S1,S), eq(L,[]).

rest_of_list(S,_,_) :-
   syntax_error(['","',',','"|"',',',or,'"]"',expected],S).

:- mode expect(+,+,?).

expect(Token,[Token|S1],S) :- !, eq(S1,S).

expect(Token,S,_) :- syntax_error([Token,expected],S).

:- mode no_garbage_at_end(+).

no_garbage_at_end([]).
no_garbage_at_end([.]).
no_garbage_at_end([A|S1]) :- syntax_error(['Garbage',after,term],[A|S1]).

%

syntax_error(Msg,S) :- recorded('*SYNTAX-ERROR*',_,_), !, fail.

syntax_error(Msg,S) :-
   length(S,K), recorda('*SYNTAX-ERROR*',(Msg,K),_), fail.

syntax_error(S) :-
   recorded('*SYNTAX-ERROR*',(M,K),R), erase(R),
   length(S,L), W is L-K, splitl(W,S,Sb,Sa),
%   telling(Out), tell(user),
   write_tokens(['%'|M]), nl,
   write_tokens(['%'|Sb]), write('<-- HERE'), nl, write_tokens(['%'|Sa]), nl,
%   tell(Out),
   fail.



:- mode splitl(+,+,-,-).

splitl(0,L,La,Lb) :- !, eq(La,[]), eq(Lb,L).

splitl(K,[A|L],La,Lb) :- eq(La,[A|Laa]), KK is K-1, splitl(KK,L,Laa,Lb).

:- mode write_tokens(+).

write_tokens([M|Msg]) :- write_token(M), put(32), write_tokens(Msg).

write_tokens([]).

:- mode write_token(+).

write_token(var(_,N)) :- !, write_tokens_string(N).

write_token(atom(A)) :- !, '$display'(A).

write_token(atomic(A)) :- !, '$display'(A).

write_token(number(A)) :- !, '$display'(A).

write_token(string(S)) :- !, format(user_output, """~s""", [S]).

write_token(T) :- '$display'(T).

read_byte(A) :- get0(A).

write_tokens_string([]).
write_tokens_string([T|Ts]) :-
	put(T),
	write_tokens_string(Ts).
