% poormansformat.pl
% Jonas Barklund 1991.
% Simple implementation of a subset of Quintus Prolog's format.
% Inspired by a more complete (and slow) implementation by Monika Danielsson.

format(Control, Args) :- 
	current_output(Stream), 
	format(Stream, Control, Args).

%

format(Stream, Control, Args) :-
	stringify(Control, Ctrl),
	format1(Ctrl, Stream, Args), !.

format(Stream, Control, Args) :- 
	format_error("Format failed!", []).

%

stringify("", Ctrl) :- 
	!, Ctrl = "".

stringify(Control, Ctrl) :- 
	atom(Control), 
	!, name(Control, Ctrl).

stringify(Control, Control).

%

format1([], _Stream, []) :- 
	!.

format1([], Stream, [_|_]) :- 
	format_error("Format did not consume all arguments", []).

format1([0'~|C], Stream, Args) :- 
	!, form(C, Stream, Args).

format1([A|C], Stream, Args) :- 
	put(Stream, A), 
	format1(C, Stream, Args).

%

form([], _Stream, _Args) :- 
	format_error("Format string ended prematurely", []).

form([A|C], Stream, Args) :-
	frm(A, Stream, Args, C).

%

frm(0'w, _Stream, [], _C) :- 
	!, too_few_args.

frm(0'w, Stream, [T|Args], C) :- 
	write(Stream, T), 
	format1(C, Stream, Args).

frm(0'd, _Stream, [], _C) :- 
	!, too_few_args.

frm(0'd, Stream, [T|Args], C) :- 
	write(Stream, T), 
	format1(C, Stream, Args).

frm(0'q, _Stream, [], _C) :- 
	!, too_few_args.

frm(0'q, Stream, [T|Args], C) :- 
	writeq(Stream, T), 
	format1(C, Stream, Args).

frm(0'p, _Stream, [], _C) :- 
	!, too_few_args.

frm(0'p, Stream, [T|Args], C) :- 
	print(Stream, T), 
	format1(C, Stream, Args).

frm(0'a, Stream, Args, C) :- 
	frm(0'w, Stream, Args, C).

frm(0's, Stream, [], C) :- 
	!, too_few_args.

frm(0's, Stream, [S|Args], C) :- 
	wstring(Stream, S), 
	format1(C, Stream, Args).

frm(0'c, _Stream, [], _C) :- 
	!, too_few_args.

frm(0'c, Stream, [N|Args], C) :- 
	put(Stream, N), 
	format1(C, Stream, Args).

frm(0'n, Stream, Args, C) :- 
	nl(Stream), 
	format1(C, Stream, Args).

frm(0'N, Stream, Args, C) :- 
	line_position(Stream, 0), 
	!, format1(C, Stream, Args).

frm(0'N, Stream, Args, C) :- 
	frm(0'n, Stream, Args, C).

%

too_few_args :- 
	format_error("Format received a too short argument list", []).

%

format_error(Cntrl,Args) :- 
	call(error(Cntrl,Args)). % [PM] don't ask

% For printing a string.

wstring(Stream, String) :-
  wstr(String, Stream), 
  fail. % don't let it instantiate nonvar tail etc.

wstring(_Stream, _String).

%

wstr([], _Stream).
wstr([A|X], Stream) :- 
	put(Stream, A), 
	wstr(X, Stream).

%

error(X,Y) :- 
	write('Error in format: '),
	write_error(X), write(Y).

%

write_error([]).
write_error([H|T]) :-
	put(H),
	write_error(T).

%
