% -*- Prolog -*- 
%    File:	builtin.pl  (~bevemyr/Luther/Library/builtin.pl)
%    Author:	Johan Bevemyr
%    Created:	Fri Jun 14 11:45:10 1991
%    Purpose:   Definition of builtin predicates in Luther Prolog.
 

publify :-
	public([statistics/2, nl/1, close/1, open/3, load/1, qload/1,
	        end_of_file/0, get0/2, get/2, with_default_input/2,
		'.'/2, consult/1, repeat/0, length/2, numbervars/3,
		assert/1, assert/2, assertz/1, assertz/2, asserta/1,
		asserta/2, clause/2, clause/3, retrace/1, recorda/3,
		recordz/3, recorded/3, erase/1, reconsult/1,retract/1]),
	public(['='/2, compare/3, functor/3, arg/3, '=='/2, '\=='/2,
	        '@<'/2, '@>'/2, '@=<'/2, '@>='/2, '=:='/2, '=\='/2,
		'<'/2, '>'/2, '=<'/2, '>='/2, atom/1, atomic/1, generic/1,
		integer/1, float/1, number/1, nonvar/1, var/1, '=..'/2,
		elt/3, size/2, par_reduce_plus/2, par_reduce_times/2]),
	public(['SYSCALL'/1,'USERCALL'/1]),
	public([write_canonical/1, write_canonical/2,
	        display/1,
		print/1, print/2,
		write/1, write/2,
		writeq/1, writeq/2,
		listing/0,
		listing/1,
		portray_clause/1, portray_clause/2, predicate_property/2,
		numbervars/3,list_all/0]),
        public([read/1,read/2,read_with_dictionary/2,read_with_dictionary/3,
	        var_call/3, with_default_output/2, call/1,'\+'/1,!/1]),
	public([','/2, ';'/2, '->'/2, '^'/2]),
	public(['mode_error'/4, format/2, format/3, error/2]),
	public([setof/3,bagof/3,findall/3,findall/4]),
	public([(not)/1]),
	public([keysort/2, merge/3, merge/5,msort/2,sort/2,sort/4]),
	public(['prolog_flag'/3, 'prolog_flag'/2]).


%

start :- 
	load_runtime_library,
	publify,
	'$set_module'(_,user),
	version, 
	toploop,
	halt.
%

load_runtime_library :-
	quiet_load('##libdir##/format'),
	quiet_load('##libdir##/read'),
	quiet_load('##libdir##/rdtok'),
	quiet_load('##libdir##/write'),
	quiet_load('##libdir##/expand_term'),
	quiet_load('##libdir##/current_op'),
	quiet_load('##libdir##/setof'),
	quiet_load('##libdir##/not'),
	quiet_load('##libdir##/sorts'),
	quiet_load('##libdir##/interpret3').

%

toploop :- repeat, query.

%

query :-
	'$choice'(C),
	init_trace(C),
	'$prompt'(Old,'| ?- '),
	read_with_dictionary(Query,Vars),
	'$prompt'(_,Old),
	'$topchoice'(Ch),
	trace_body(Query,Ch),
	write_vars(Vars),
	get_more(Vars),
	nl,
	write('yes'),nl,
	!,fail.

query :- nl, write('no'), nl, fail.

%

init_trace(C) :- 	
	'$trace', !,
	write('{trace}'),nl,
	'$save_choice'(C), 
	'$set_inc_trace_level'(0).

init_trace(C) :-
	'$save_choice'(C), 
	'$set_inc_trace_level'(0).

%

get_more([]) :- !.

get_more(_) :-
	'$prompt'(Old2,' ? '),
	get0(More),
	skip_rest(More),
	'$prompt'(_,Old2),
	More \== 59.

%

skip_rest(10) :- !.

skip_rest(X) :- '$skip'(10).

%

write_vars([]) :- !.

write_vars([[Name,Value|_]|Vars]) :-
	nl,
	write_name(Name),
	write(' = '),
	write(Value),
	write_vars_c(Vars),
	write_vars(Vars).

%

write_vars_c([]) :- !.

write_vars_c(_) :- write(',').

%

write_name([]).

write_name([C|Cs]) :- put(C), write_name(Cs).

%

statistics(runtime, X) :- '$statistics_runtime'(X).

statistics(gctime, X) :- '$statistics_gctime'(X).

statistics(gcnr, X) :- '$statistics_gcnr'(X).

statistics(gcbytes, X) :- '$statistics_gcbytes'(X).

statistics(walltime, X) :- '$statistics_walltime'(X).

statistics(parallel, X) :- '$statistics_parallel_runtime'(X).

statistics(memory, X) :- '$statistics_memory'(X).

statistics(global, X) :- '$statistics_global'(X).

statistics(local, X) :- '$statistics_local'(X).

statistics(trail, X) :- '$statistics_trail'(X).

statistics(code, X) :- '$statistics_code'(X).

statistics(atom, X) :- '$statistics_atom'(X).

%

prolog_flag(Flag,New) :- prolog_flag(Flag,_,New).

%

prolog_flag(gc_verbose, Old, New) :- prolog_flag_gc_verbose(Old, New).

prolog_flag(load_verbose, Old, New) :- prolog_flag_load_verbose(Old, New).

%


open(Filename, Mode, Stream) :- 
	valid_mode(Mode,M),!,
	'$fopen'(Filename, M, Stream).

open(File,Mode,Stream) :- 
	write(open(File,Mode,Stream)),
	write(' -- illegal mode spec'), 
	nl, fail.

%

load([]) :- !.

load([B|C]):- !, load(B), load(C).

load(E):-
	fixname(E,G,".wam"),
	write('{loading '),
	write(G),write('...}'),
	nl,
	statistics(code,[H|I]),statistics(atom,[J|K]),statistics(runtime,L),
	'$load'(G),
%	update_load_time(G),
	statistics(code,[M|N]),statistics(atom,[O|P]),
	statistics(runtime,[Q,R]),
	S is M+O-H-J,
	write('{'),write(G),write(' loaded, '),write(R),write(' msec '),
	write(S),write(' bytes}'),nl.

%

quiet_load([]) :- !.

quiet_load([B|C]):- !, quiet_load(B), quiet_load(C).

quiet_load(E):-
	fixname(E,G,".wam"),
	'$load'(G).

%

qload([]) :- !.

qload([B|C]):- !, qload(B), qload(C).

qload(E):-
	fixname(E,G,".ql"),
	write('{qloading '),
	write(G),write('...}'),
	nl,
	statistics(code,[H|I]),statistics(atom,[J|K]),statistics(runtime,L),
	'$qload'(G),
%	update_load_time(G),
	statistics(code,[M|N]),statistics(atom,[O|P]),
	statistics(runtime,[Q,R]),
	S is M+O-H-J,
	write('{'),write(G),write(' qloaded, '),write(R),write(' msec '),
	write(S),write(' bytes}'),nl.
% 

end_of_file :- halt.

%


fixname(user,user,_) :- !.
fixname(user_input, user_input, _) :- !.
fixname(N1,N2,Suffix) :-
	name(N1,Chars),
	dlist_name(Chars,F1,Suffix),
	name(N2,F1),!.

%

dlist_name([],Z,Z) :- !.

dlist_name(Z,Z,Z)  :- !.

dlist_name([X|Xs],[X|Ys],Z) :- dlist_name(Xs,Ys,Z).

%

valid_mode(read,r).

valid_mode(write,w).

valid_mode(append,a).

%

'$output'(Index1,Index2) :-
	'current_output'(Index1),
	'$set_output'(Index2).

%

with_default_input(Index1,G) :-
	'current_input'(Index2),
	'$set_input'(Index1),
	(  call(G) -> '$set_input'(Index2) 
	; '$set_input'(Index2), fail).

%

with_default_output(Index1,G) :-
	'current_output'(Index2),
	'$set_output'(Index1),
	(  call(G) -> '$set_output'(Index2) 
	; '$set_output'(Index2), fail). 

%

[F|Fs] :- consult([F|Fs]).

%

reconsult(X) :- consult(X).

%

consult([]) :- !.

consult([F|Fs]) :-
	!,
	consult2(F),
	consult(Fs).

consult(F) :- consult2(F).

%

%
% An  alternative to using $fdopen is to define consulting
% predicates that doesn't expect a stream as argument. They
% would use read/1 instead of read/2. 
%

consult2(F) :-
	fixname(F,File,".pl"),
	open(File,read,S),
	write('{consulting '),write(File),write('...}'), nl,
	statistics(code,[C1|_]),statistics(atom,[A1|_]),statistics(runtime,_),
	read(S,Term),
	expand_term(Term,ExTerm),   % consult_first expects an expanded term
	consult_first(Term,S),
	close(S),
%	update_load_time(File),
	statistics(runtime,[_,T]),statistics(code,[C2|_]),
	statistics(atom,[A2|_]),
	write('{'),write(File),	write(' consulted, '),write(T),
	write(' msec '),
	M is C2 + A2 - C1 - A1,
	write(M),write(' bytes}'),nl.

%

consult_first(end_of_file,_) :- !.

consult_first((:- Directive), S) :-
	!,
	(
	    call(Directive) -> true
	; 
	    (
		write('{Warning: '), write(Directive), 
		write(' - goal failed}'), nl
	    )
	),
	read(S,Term),
	expand_term(Term,ExTerm),
	consult_first(ExTerm,S).

consult_first(Clause,S) :-
	assert_delete_other(Clause),
	clause_functor(Clause,Func),
	read(S,Term),
	expand_term(Term,ExTerm),
	consult_rest(ExTerm,Func,S).

%

consult_rest(end_of_file,_,_) :- !.

consult_rest((:- Directive), Func, S) :-
	!,
	(
	    call(Directive) -> true
	; 
	    (
		write('{Warning: '), write(Directive), 
		write(' - goal failed}'), nl
	    )
	),
	read(S,Term),
	expand_term(Term,ExTerm),
	consult_rest(ExTerm,Func,S).

consult_rest(Clause,Func,S) :-
	clause_functor(Clause,Func),
	!,
	assert(Clause),
	read(S,Term),
	expand_term(Term,ExTerm),
	consult_rest(ExTerm,Func,S).

consult_rest(Clause,_,S) :-
	assert_delete_other(Clause),
	read(S,Term),
	expand_term(Term,ExTerm),
	clause_functor(Clause,Func2),
	consult_rest(ExTerm,Func2,S).

%

clause_functor((Head :- _), F/A) :- !,functor(Head,F,A).

clause_functor(Head, F/A) :- functor(Head,F,A).

%

update_load_time(File) :-
	'$file_mod_time'(File,Hi,Lo),
	(    recorded(load_time(File),time(_,_),Ref) 
	->   erase(Ref)
	;    true
	),
	recordz(load_time(File),time(Hi,Lo),_).

%

repeat.

repeat :- repeat.

%

number_chars(N,C) :- number_chars(N,C,10).

%

length(X,Y) :- var(Y), !, length_var(X,0,Y).

length(X,Y) :- length_num(X,0,Y).

%

length_var([],N,N).

length_var([_|T],N0,N) :- N1 is N0 + 1, length_var(T,N1,N).

%

length_num([],N,N) :- !.

length_num([_|T],N0,N) :- N > N0, N1 is N0 + 1, length_num(T,N1,N).

%

numbervars(X, I, N) :- 
	integer(I),
	numbervars1(X,I,N).

%

numbervars1('$VAR'(N0), N0, N) :- !, N is N0 + 1.

numbervars1(Atm, N, N) :- atomic(Atm),!.

numbervars1(Str, N0,N) :-
	functor(Str,_,Arity),
	numbervars1(Str, 0, Arity, N0, N).

%

numbervars1(Str, A, A, N, N) :- !.

numbervars1(Str, A1, A, N0, N) :-
	A2 is A1 + 1,
	arg(A2, Str, T),
	numbervars1(T, N0, N1),
	numbervars1(Str, A2, A, N1, N).

%

assert(C) :- assertz(C,_).

assert(C,R) :- assertz(C,R).

%

assertz(C) :- assertz(C,_).

assertz((Head :- Body),Ref) :-
	!,var(Ref),
	eq(Ref,'$ref'(P,R)),
	'$assertz'(Head,Body,P,R).

assertz(Head,Ref) :-
	var(Ref),
	eq(Ref,'$ref'(P,R)),
	'$assertz'(Head,true,P,R).

%

asserta(C) :- asserta(C,_).

asserta((Head :- Body),Ref) :-
	!,var(Ref),
	eq(Ref,'$ref'(P,R)),
	'$asserta'(Head,Body,P,R).

asserta(Head,Ref) :-
	var(Ref),
	eq(Ref,'$ref'(P,R)),
	'$asserta'(Head,true,P,R).

%

assert_delete_other(C) :- assert_delete_other(C,_).

assert_delete_other((Head :- Body),Ref) :-
	!,var(Ref),
	eq(Ref,'$ref'(P,R)),
	'$assert_delete_other'(Head,Body,P,R).

assert_delete_other(Head,Ref) :-
	var(Ref),
	eq(Ref,'$ref'(P,R)),
	'$assert_delete_other'(Head,true,P,R).

%

predicate_property(Name,Prop) :-
	current_predicate(_,Name),
	'$predicate_property'(Name,Prop).

% 

clause(H,B) :- '$clause'(H,B,0,0).

clause(Head,Body,Ref) :- '$clause'(Head, Body, Ref, 0,0).

%

retract((Head :- Body)) :-
	!, clause(Head,Body,Ref),
	erase(Ref).

retract(Head) :-
	clause(Head,true,Ref),
	erase(Ref).

%

recorda(Key, Term, Ref) :-
	asserta('$record slot'(Key, Term), Ref).

recordz(Key, Term, Ref) :-
	assertz('$record slot'(Key, Term), Ref).

recorded(Key, Term, Ref) :-
	clause('$record slot'(Key, Term), true ,Ref).

erase('$ref'(P,R)) :- '$erase'(P,R).

current_key(Key, Term) :-
	clause('$record slot'(Key, Term), true).
%

X = Y :- X = Y.

% inlineable predicates

'SYSCALL'(G) :-
	'$set_module'(Old,prolog),
	call(G),
	'$set_module'(_,Old).

'USERCALL'(G) :-
	'$set_module'(Old,user),
	call(G),
	'$set_module'(_,Old).

%

public([]).

public([N/A|T]) :- '$public'(N,A), public(T).

%%

abort :-
	'$display'('{ Execution aborted }'),nl,
	'$get_saved_choice'(C),
	'$cut'(C),
	fail.

%%

list_all :-
	predicate_property(Pred,Prop),
	functor(Pred,F,A),
	write('	'), write(F/A), write('   	( '), 
	write(Prop), write(' )'), nl,
	fail.
list_all.

%%

listing :-
	predicate_property(Pred,interpreted),
	functor(Pred,F,A),
	listing(F/A),
	fail.

listing.

%

listing(F/A) :-
	nonvar(F), nonvar(A),
	functor(H,F,A),
	clause(H,B),
	listing_write(H,B),
	fail.

listing(_).

%

listing_write(H,B) :-
	B == true,
	!,write(H),
	write('.'),nl.

listing_write(H,B) :-
	write(H),
	write(' :-'),
	nl,
	listing_write_body(B).

%

listing_write_body(X) :-
	var(X),!,
	write('        '),
	write(C), write('.'), nl.

listing_write_body(','(C,Cs)) :-
	!,write('        '),
	write(C),write(','),nl,
	listing_write_body(Cs).

listing_write_body(C) :-
	write('        '),
	write(C), write('.'), nl.

%%%%%
%
% Inline/builtin calls
%

compare(X,Y,Z) :- compare(X,Y,Z).

functor(X,Y,Z) :- functor(X,Y,Z).

arg(X,Y,Z) :- arg(X,Y,Z).

size(X,Y,Z) :- size(X,Y,Z).

elt(X,Y,Z) :- elt(X,Y,Z).

X = Y :- X = Y.

X == Y :- X == Y.

X \== Y :- X \== Y.

X @< Y :- X @< Y.

X @> Y :- X @> Y.

X @=< Y :- X @=< Y.

X @>= Y :- X @>= Y.

X =:= Y :- X =:= Y.

X =\= Y :- X =\= Y.

X < Y :- X < Y.

X > Y :- X > Y.

X =< Y :- X =< Y.

X >= Y :- X >= Y.

atom(X) :- atom(X).

atomic(X) :- atomic(X).

generic(X) :- generic(X).

integer(X) :- integer(X).

float(X) :- float(X).

number(X) :- number(X).

nonvar(X) :- nonvar(X).

var(X) :- var(X).

X =.. Y :- X =.. Y.

par_reduce_plus(X,Y) :- par_reduce_plus(X,Y).

par_reduce_times(X,Y) :- par_reduce_times(X,Y).


%

% Signal error when predicate is called with known wrong mode.

mode_error(X,Type,P,N) :-
	format(user_error,'{ Mode violation in ~q: ~q must have type ~q }~n',
	       [P/N,X,Type]),
	fail.
