/*    File:	 unify.c 
 *    Author:	 Johan Bevemyr
 *    Created:	 Sat May 25 20:14:53 1991
 */ 

#include "include.h"
#include "unify.h"
#include "engine.h"

/* We use a table to determine what action to take when
   we try to unify two terms.

    0: Fail unification
    1: Bind younger variable to older
    2: Bind SVA to HVA
    3: Bind HVA to constant or structure
    4: Bind SVA to HVA
    5: Bind younger variable to older
    6: Bind HVA to constant or structure
    7: Unify numbers
    8: Unify floats
    9: Unify lists
   10: Unify args of two structures 
   11: Unify SVA to constant or structure
   12: Unify SVA to constant or structure
   13: Unify a generic object with anything but a variable
   14: Unify a generic object with anything but a variable or another g obj.

         H  C  S  L  N  F  A  L  S  G
         V  V  V  C  U  L  T  S  T  E
         A  A  A  K  M  T  M  T  R  N
   HVA { 1, 3, 2, 0, 3, 3, 3, 3, 3, 3 }
   CVA { 6,13, 2, 0,13,13,13,13,13,13 }
   SVA { 4,14, 5, 0, 4, 4, 4, 4, 4, 4 }
   LCK { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 }
   NUM { 6,14, 2, 0, 7, 0, 0, 0, 0,12 }
   FLT { 6,14, 2, 0, 0, 8, 0, 0, 0,12 }
   ATM { 6,14, 2, 0, 0, 0, 0, 0, 0,12 }
   LST { 6,14, 2, 0, 0, 0, 0, 9, 0,12 }
   STR { 6,14, 2, 0, 0, 0, 0, 0,10,12 }
   GEN { 6,14, 2, 0,11,11,11,11,11,11 }
              
 */


#ifdef LOCKING
#define L(X,Y) X,Y
#else
#define L(X,Y) Y
#endif

#ifdef CONSTR
#define C(X,Y) X,Y
#else
#define C(X,Y) Y
#endif

#ifdef UNBOUND
#define U(X,Y) X,Y
#else
#define U(X,Y) X
#endif

int unifytable[MAX_TAG_LO][MAX_TAG_LO] = {
  { 1, C(3,2), L(0,3), 3, 3, 3, 3, U(3,0)},
#ifdef CONSTR
  { 6,C(13,2), L(0,13),13,13,13,13,U(13,0)},
#endif
  { 4,C(14,5), L(0,4), 4, 4, 4, 4, U(4,0)},
#ifdef LOCKING
  { 0, C(0,0), L(0,0), 0, 0, 0, 0, U(0,0)},
#endif
  { 6,C(14,2), L(0,7), 0, 0, 0, 0,U(12,0)},
  { 6,C(14,2), L(0,0), 8, 0, 0, 0,U(12,0)},
  { 6,C(14,2), L(0,0), 0, 0, 0, 0,U(12,0)},
  { 6,C(14,2), L(0,0), 0, 0, 9, 0,U(12,0)},
  { 6,C(14,2), L(0,0), 0, 0, 0,10,U(12,0)},
  { 6,C(14,2), L(0,11),11,11,11,11,U(11,0)}
#ifdef UNBOUND
, { 0,C( 0,0), L(0,0), 0, 0, 0, 0, 0, 0}
#endif 
};

static BOOL unify_structure(s1, s2, arity,w)
    TAGGED *s1, *s2;
    int arity;
    worker *w;
{
    TAGGED t1, t2;

    while(arity--) {
	DerefNLL(t1, Ref(s1)); s1 += VARSIZE;
	DerefNLL(t2, Ref(s2)); s2 += VARSIZE;
	if(!unify(t1,t2,w)) return FALSE;
    }
    return TRUE;
}

BOOL unify(x,y,w)
    TAGGED x, y;
    worker *w;
{
#ifdef UNBOUND
    register TAGGED x_time, y_time;
#endif
 start:
    if(x!=y) {
	switch(unifytable[LowTagOf(x)][LowTagOf(y)]) {
	case 0: /* unification failed */
	    return FALSE;

	case 1: /* HVA - HVA */
#ifdef UNBOUND
	    x_time = *RemoveTag(x,HVA);
	    y_time = *RemoveTag(y,HVA);

	    if(!(IsUVA(x_time) && IsUVA(y_time))) unify_deref(x,y,w);
	    
	    if(GetHVATime(x_time) > GetHVATime(y_time)) {
#else  /* UNBOUND */
	    if(GetHVATime(x) > GetHVATime(y)) {
#endif /* UNBOUND */
#ifdef LOCKING
		register TAGGED xv;
		Grab_Lock(xv,x);
		if(x != xv) {
		    Drop_Lock(x,xv);
		    DerefNLL(x,x);
		    goto start;
		} else 
#endif /* LOCKING */
		  Bind_Unsafe_HVA(x,y,{return FALSE;});

#ifdef UNBOUND
	    } else if(GetHVATime(x_time) < GetHVATime(y_time)) {
#else
	    } else if(GetHVATime(x) < GetHVATime(y)) {
#endif /* UNBOUND */
#ifdef LOCKING
		register TAGGED yv;
		Grab_Lock(yv,y);
		if(y != yv) {
		    Drop_Lock(y,yv);
		    DerefNLL(y,y);
		    goto start;
		} else 
#endif /* LOCKING */
		  Bind_Unsafe_HVA(y,x,{return FALSE;});
	    } else if(x > y) {
#ifdef LOCKING
		register TAGGED xv;
		Grab_Lock(xv,x);
		if(x != xv) {
		    Drop_Lock(x,xv);
		    DerefNLL(x,x);
		    goto start;
		} else 
#endif /* LOCKING */
		    Bind_Unsafe_HVA(x,y,{return FALSE;});
	    } else {
#ifdef LOCKING
		register TAGGED yv;
		Grab_Lock(yv,y);
		if(y != yv) {
		    Drop_Lock(y,yv);
		    DerefNLL(y,y);
		    goto start;
		} else 
#endif /* LOCKING */
		    Bind_Unsafe_HVA(y,x,{return FALSE;});
	    }		
	    break;

	case 2: /* HVA \/ constant - SVA */
	    Bind_SVA(y,x);
	    break;

	case 3: /* HVA - Constant */
	    {
#ifdef LOCKING
	      register TAGGED xv;
	      Grab_Lock(xv,x);
	      if(x != xv) {
		Drop_Lock(x,xv);
		DerefNLL(x,x);
		goto start;
	      } else 
#endif /* LOCKING */
		Bind_Unsafe_HVA(x,y,{return FALSE;});
	      break;
	    }

	case 4: /* SVA - HVA \/ constant */
	    Bind_SVA(x,y);
	    break;

	case 5: /* SVA - SVA */
	    if(GetSVATime(x) > GetSVATime(y)) {
		Bind_SVA(x,y);
	    } else {
		Bind_SVA(y,x);
	    }
	    break;

	case 6: /* Constant - HVA */
	    {
#ifdef LOCKING
	      register TAGGED yv;
	      Grab_Lock(yv,y);
	      if(y != yv) {
		Drop_Lock(y,yv);
		DerefNLL(y,y);
		goto start;
	      } else 
#endif /* LOCKING */
		Bind_Unsafe_HVA(y,x,{return FALSE;});
	      break;
	    }

	case 7: /* NUM - NUM */
	    return FALSE;

	case 8: /* FLT - FLT */
	    if(GetFloat(x) == GetFloat(y))
		break;
	    else
		return FALSE;

	case 9: /* LST - LST */
	    return unify_structure(GetCar(x),GetCar(y), 2, w);

	case 10: /* STR - STR */
	    { 
	      register UNTAGGED xu, yu;

	      xu = (UNTAGGED) RemoveTag(x,STR);
	      yu = (UNTAGGED) RemoveTag(y,STR);
	    
	      if(Struct(xu)->functor == Struct(yu)->functor)
		{
		  return unify_structure(GetUntArg(xu,0), GetUntArg(yu,0),
					 StructArity(xu), w);
		}
	      else
		return FALSE;
	    
	    }

	case 11: /* GEN - CON */
	    return GetMethod(unify,x)(x,y,w);

	case 12: /* GEN - CON */
	    return GetMethod(unify,y)(y,x,w);

#ifdef CONSTR
	case 13: /* CVA - constant \/ CVA */
	    {
	      register TAGGED xv;
	      Grab_Lock(xv,x);
	      if(x != xv)
		{
		  Drop_Lock(x,xv);
		  DerefNLL(x,x);
		  goto start;
		}
	      else
		{
		  Bind_CVA(x,y);
		}
	      break;
	    }

	case 14: /* constant - CVA */
	    {
	      register TAGGED yv;
	      Grab_Lock(yv,y);
	      if(y != yv)
		{
		  Drop_Lock(y,yv);
		  DerefNLL(y,y);
		  goto start;
		}
	      else
		{
		  Bind_CVA(y,x);
		}
	      break;
	    }

#endif /* CONSTR */
	default:
	    Error("unify - Out of range in unify");
	    return TRUE;
	}
	return TRUE;
    }
    return TRUE;
}

BOOL unify_deref(x,y,w)
    TAGGED x, y;
    worker *w;
{
    DerefNLL(x,x);
    DerefNLL(y,y);
    return unify(x,y,w);
}
