/*    File:	 term.h  
 *    Author:	 Johan Bevemyr
 *    Created:	 Sat May 25 18:34:03 1991
 */ 

/**********************************************************************
 *           tagged objects
 */

typedef u32 TAGGED;        /* 32 bit */
typedef u32 UNTAGGED;      /* 32 bit */

typedef enum {
    HVA = 0, 
#ifdef CONSTR
    CVA,
#endif /* CONSTR */    
    SVA,     
#ifdef LOCKING
    LCK,
#endif /* LOCKING */    
    NUM,     
    FLT,     
    ATM,     
    LST,     
    STR,     
    GEN,
#ifdef UNBOUND 
    UVA,
#endif /* UNBOUND */
    MAX_TAG,
} tagvalue;

typedef enum {
    HVA_LO = 0,
#ifdef CONSTR
    CVA_LO,
#endif /* CONSTR */
    SVA_LO,
#ifdef LOCKING
    LCK_LO,
#endif /* LOCKING */
    NUM_LO,
    FLT_LO,
    ATM_LO,
    LST_LO,
    STR_LO,
    GEN_LO,
#ifdef UNBOUND
    UVA_LO,
#endif /* UNBOUND */
    MAX_TAG_LO
} lowtagvalue;

#ifdef LOWTAGS

#define FUNCSIZE         2
#define VARSIZE          2
#define SVASIZE          2
#define ARSIZE           2

#else /* LOWTAGS */

#define FUNCSIZE         1 
#define VARSIZE          1
#define SVASIZE          1
#define ARSIZE           1

#endif /* LOWTAGS */

#if (LOCKING | CONSTR | UNBOUND)

#define TAGBITS          4

#else

#define TAGBITS          3

#endif /* LOCKING */

#define WORDSIZE         32
#define GCBITS	         2

#define TOPBIT           (0x1L<<WORDSIZE-1)

#ifdef LOWTAGS /* ------------------------------ */

#define GCOFFSET         (WORDSIZE-GCBITS)
#define TAGOFFSET        0
#define TAGMASK          0x7                            /* 0...00111 */

#define LowTagOf(T)      ((lowtagvalue)(((u32)(T)) & TAGMASK))
#define TagToPointer(T)  ((TAGGED *) OffsetBase((((s32) (T))|TAGMASK)-TAGMASK))
#define Tagify(P,T)      ((TAGGED) (BaseOffset((u32) (P)) | ((u32) (T))))
#define RemoveTag(P,T)   ((TAGGED *) OffsetBase((((s32) (P))-(T))))

#else /* LOWTAGS  ------------------------------ */

#define GCOFFSET 0
#define TAGOFFSET        (WORDSIZE-TAGBITS)
#define TAGMASK          ((u32) (((s32) TOPBIT) >> (TAGBITS-1)))

#define LowTagOf(T)      ((lowtagvalue)(((u32)(T))>>TAGOFFSET))
#define TagToPointer(T)  ((TAGGED *) OffsetBase(((u32) POINTMASK)&((u32) (T))))
#define Tagify(P,T)      ((TAGGED) (BaseOffset((u32) (P)) |\
				    (((u32) (T)) << TAGOFFSET)))
#define RemoveTag(P,T)   TagToPointer(P)

#endif /* LOWTAGS  ------------------------------ */

#define POINTMASK        (~(TAGMASK | GCMASK))
#define TagOf(T)         ((tagvalue) LowTagOf(T))
#define SetTag(P,T)      Tagify(((u32) (P) & ~TAGMASK), T)
#define PointerToTerm(T) Tagify(T,NUM)
#define TermToPointer(T) RemoveTag(T,NUM)

#ifdef MallocBase

#define OffsetBase(T)    (((u32) (T)) + ((u32) MallocBase))
#define BaseOffset(T)    (((u32) (T)) & POINTMASK)

#else /* MallocBase */

#define OffsetBase(T)    (T)
#define BaseOffset(T)    (T)

#endif /* MallocBase */

#define IsHVA(V)         (TagOf(V) == HVA)
#define IsCVA(V)         (TagOf(V) == CVA)
#define IsSVA(V)         (TagOf(V) == SVA)
#define IsLCK(V)         (TagOf(V) == LCK)
#define IsNUM(V)         (TagOf(V) == NUM)
#define IsFLT(V)         (TagOf(V) == FLT)
#define IsATM(V)         (TagOf(V) == ATM)
#define IsLST(V)         (TagOf(V) == LST)
#define IsSTR(V)         (TagOf(V) == STR)
#define IsGEN(V)         (TagOf(V) == GEN)
#define IsUVA(V)         (TagOf(V) == UVA)
                         
#define IsNonVar(V)      (TagOf(V) > SVA)
#define IsNonCon(V)      (TagOf(V) < NUM)
#define IsHVAorCVA(V)    (TagOf(V) < SVA)

#define HVA_CASE(X)      case HVA: X
#define SVA_CASE(X)      case SVA: X
#define NUM_CASE(X)      case NUM: X
#define FLT_CASE(X)      case FLT: X
#define ATM_CASE(X)      case ATM: X
#define LST_CASE(X)      case LST: X
#define STR_CASE(X)      case STR: X
#define GEN_CASE(X)      case GEN: X

#ifdef CONSTR

#define CVA_CASE(X)      case CVA: X

#else  /* CONSTR */

#define CVA_CASE(X)  

#endif /* CONSTR */

#define Switch_Tag(X,H,C,S,N,F,A,L,T,G,B) \
{                                         \
  switch(LowTagOf(X))                     \
    {                                     \
      HVA_CASE(H);                        \
      CVA_CASE(C);                        \
      SVA_CASE(S);                        \
      NUM_CASE(N);                        \
      FLT_CASE(F);                        \
      ATM_CASE(A);                        \
      LST_CASE(L);                        \
      STR_CASE(T);                        \
      GEN_CASE(G);                        \
    default:                              \
      B;                                  \
    }                                     \
}

#ifdef LOCKING           

#define IsVar(V)         (TagOf(V) < LCK)

#else /* LOCKING */

#define IsVar(V)         (TagOf(V) < NUM)

#endif /* LOCKING */     
                         
#define IsComplex(A)     (IsLST(A) || IsSTR(A))
#define IsCompound(A)    IsComplex(A)
#define IsNumber(A)      (IsNUM(A) || IsFLT(A))
#define IsHeapTerm(V)    ((TagOf(V) < SVA) ||\
		          (TagOf(V) > ATM))

#define IsHeapBox(V)     ((TagOf(V) == FLT) && IsDynBox(*TagToPointer(V)))
#define IsHeapBoxM(V)    ((TagOf(V) == FLT) && IsMarked(*TagToPointer(V)))
                         
#define Var(V)           ((TAGGED *) (V))
#define Atom(T)          ((atom) (T))
#define Choice(T)        ((choicepoint *) (T))
#define Generic(G)       ((generic) (G))
                         
#define GetCar(L)        RemoveTag(L,LST)
#define GetCdr(L)        (RemoveTag(L,LST)+VARSIZE)
                         
#ifdef TIMESTAMP         
                         
#undef VARSIZE
#define VARSIZE          2

#define TIMEUNIT         1
#define GetHVATime(T)    ((s32) *(((TAGGED *) OffsetBase(T))+1))
#define GetCVATime(T)    ((s32) *(RemoveTag(T,SVA)+1))

#ifdef SHORT_SVA
#define GetSVATime(T)    RemoveTag(T,SVA)
#else
#undef SVASIZE
#define SVASIZE          2
#define GetSVATime(T)    ((s32) *(RemoveTag(T,SVA)+1))
#endif /* SHORT_SVA */

#else  /* TIMESTAMP */

#ifdef UNBOUND
#define TIMEUNIT         4
#define GetHVATime(T)    ((s32) (BaseOffset(RemoveTag(T,UVA))))
#else  /* UNBOUND */
#define GetHVATime(T)    ((TAGGED *) OffsetBase(T))
#endif /* UNBOUND */
#define GetSVATime(T)    RemoveTag(T,SVA)

#endif /* TIMESTAMP */

#ifdef UNBOUND
#define Ref(X)           ((TAGGED) BaseOffset(X))
#else
#define Ref(X)           *(X)
#endif

/**********************************************************************
 * Constrained variables
 */

#define Wake_CVA(V)

#define IsFirstLeft(F,W)                                                \
{                                                                       \
  register int if_i;                                                    \
  register s32 if_lev = w->level[w->pid-1];                             \
  F = TRUE;                                                             \
  for(if_i = 0 ; if_i < w->global->active_workers ; if_i++)             \
    if(w->level[if_i] < if_lev) F = FALSE;                              \
}

#define IsFirstRight(F,W)                                               \
{                                                                       \
  register int if_i;                                                    \
  register s32 if_lev = w->level[w->pid-1];                             \
  F = TRUE;                                                             \
  for(if_i = 0 ; if_i < w->global->active_workers ; if_i++)             \
    if(w->level[if_i] > if_lev) F = FALSE;                              \
}


/**********************************************************************
 * Atoms
 */

typedef struct atom_bucket {
    TAGGED a;
    struct atom_bucket *next;
} atom_bucket;

struct atom {
    TAGGED mode;
    char *pname;
};
typedef struct atom *atom;

#define GetAtomMode(A)    get_mode((atom)RemoveTag(A,ATM),w)
#define GetString(A,W)    get_string((atom)RemoveTag(A,ATM),W)

#define AtomHasSpecial(T) (GetAtomMode(T) == Make_Integer(1))
#define AtomHasSquote(T)  (GetAtomMode(T) == Make_Integer(1))

#define OffsetAtom(A)     Atom(((s32)(A))+((s32) w->global->atomoffset))
#define DeOffsetAtom(A)   Atom(((s32)(A))-((s32) w->global->atomoffset))


/**********************************************************************
 *           functors
 */

#define ARITYOFFSET  20
#define BOXBITS 1

                     /* ARITYMASK = 0000 1111 11 000000 00000000 00000000 */
#ifdef LOWTAGS
#define ARITYMASK    ((u32) (((s32)TOPBIT)>>(WORDSIZE-ARITYOFFSET-1)))
#define BOXMASK      (((s32) TOPBIT) >> (BOXBITS-1))
#else
#define BOXMASK      ((((s32) TAGMASK) >> BOXBITS)^TAGMASK)
#define ARITYMASK    ((u32) ((((s32)TOPBIT)>>(WORDSIZE-ARITYOFFSET-1))^   \
			     (TAGMASK | BOXMASK)))
#endif    

#define ArityOf(F)   ((int) (((F) & ARITYMASK) >> ARITYOFFSET))
#define Arityfy(F,A) (((u32)(F)) | (((u32)(A)) << ARITYOFFSET))

#define FunctorToAtom(F) ((TAGGED) (((u32) (F)) & ~ARITYMASK))

/**********************************************************************
 *           structures
 */

struct structure {
    TAGGED functor;
#if (FUNCSIZE==2)
    TAGGED dummy;
#endif
    TAGGED arg[ANY];
};

typedef struct structure *structure;

#define Struct(T)       ((structure) (T))

#define StructArity(S)  ArityOf(((structure) (S))->functor)
#define StructString(S) GetString(FunctorToAtom(GetFunctor(S)),w)

#define GetArg(S,I)     GetUntArg(RemoveTag(S,STR),I)
#define GetUntArg(S,I)  (&(((structure) (S))->arg[(I)*VARSIZE]))
#define GetFunctor(S)   (Struct(RemoveTag(S,STR))->functor)
#define GetArity(S)     (ArityOf(GetFunctor(S)))
#define GetSTRatom(S)   FunctorToAtom(GetFunctor(S))

#define GetNthHead(V,I) (RemoveTag(V,LST)+2*I*VARSIZE)
#define GetNthTail(V,I) (RemoveTag(V,LST)+2*I*VARSIZE+VARSIZE)

/**********************************************************************
 *           integers
 */

#ifdef LOWTAGS

#define TOPMASK         (~GCMASK)
#define GetNumber(N)    ((int) (((int) ((N) << GCBITS)) >> (TAGBITS+GCBITS)))
#define Make_Integer(N) Tagify(ClearTop(((u32) (N)) << TAGBITS), NUM)

#else /* LOWTAGS */

#define TOPMASK         (~TAGMASK)
#define GetNumber(N)    ((int) (((int) ((N) << TAGBITS)) >> (TAGBITS+GCBITS)))
#define Make_Integer(N) Tagify(ClearTop(((u32) (N)) << GCBITS), NUM)

#endif /* LOWTAGS */

#define Number(T)       ((integer) (T))

#define ClearTop(N)     (((u32) (N)) & TOPMASK)

#define Make_Atomspace_Integer(N) Make_Integer(N)

/**********************************************************************
 *           floating point numbers
 */

#define FLOATSIZE 4

#define GetFloat(N)      get_float(N)

/**********************************************************************
 *           boxed objects
 */

#define BOXDYNBITS 1

#ifdef LOWTAGS
#define BOXDYNMASK         (((u32) 0x1L) << GCBITS)
#define GetBoxSize(T)      ((((u32) (T)) << (GCBITS + BOXBITS)) >> \
			    (TAGBITS + BOXDYNBITS + GCBITS + BOXBITS))
#define MakeBoxSize(T)     Tagify((((u32) (T)) << (TAGBITS + BOXDYNBITS)),ATM)
#else
#define BOXDYNMASK         (((u32) 0x1L) << TAGBITS)
#define GetBoxSize(T)      ((((u32) (T)) << (TAGBITS + BOXBITS)) >> \
			    (GCBITS + BOXDYNBITS + TAGBITS + BOXBITS))
#define MakeBoxSize(T)     Tagify((((u32) (T)) << (GCBITS + BOXDYNBITS)),ATM)
#endif

#define IsBox(T)     	   (IsATM(T) && (((u32) (T)) & BOXMASK))

#define IsDynBox(T)        (!IsStatBox(T))
#define IsStatBox(T)       (((u32) (T)) & BOXDYNMASK)
#define MakeDynBoxSize(T)  (MakeBoxSize(T) | BOXMASK)
#define MakeStatBoxSize(T) (MakeBoxSize(T) | BOXDYNMASK | BOXMASK)

/**********************************************************************
 *           type declarations
 */

typedef s32 code;
typedef s32 indx;

typedef TAGGED integer;

struct list {
    TAGGED car;
    TAGGED cdr;
};

typedef struct list *list;

/**********************************************************************
 *   worker support
 */

#include "worker.h"

/**********************************************************************
 *
 * Generic objects
 */

#define GetGenTag(G)   (Generic(RemoveTag(G,GEN))->method)
#define GetMethod(M,G) (Generic(RemoveTag(G,GEN))->method->M)


struct generic {
    struct method *method;
    TAGGED data[ANY];
};

typedef struct generic *generic;

typedef struct method {
    int         (*size) PROTO((TAGGED));
    BOOL	(*unify) PROTO((TAGGED,TAGGED,worker *));
    void        (*print) PROTO((FILE *,TAGGED,worker *));
    SIZE        (*compare) PROTO((TAGGED,TAGGED,worker *));
    void        (*undo) PROTO((TAGGED));
    BOOL        (*gc) PROTO((TAGGED));
    TAGGED      (*deref) PROTO((TAGGED));
    TAGGED      (*copy) PROTO((TAGGED,TAGGED *,worker *));
} method;


/**********************************************************************
 *
 * Functions defined in term.c 
 */

extern TAGGED make_atom PROTO((char *,worker *));
extern TAGGED make_float PROTO((worker *, double));
extern double get_float PROTO((TAGGED));
extern TAGGED make_atomspace_float PROTO((double, worker *));
extern TAGGED make_string_list PROTO((worker *, char *));
