/*    File:	 storage.c  
 *    Author:	 Johan Bevemyr
 *    Created:	 Sat May 25 14:56:19 1991
 *    Purpose:   Memory management routines for Luther.
 */ 

#include "include.h"

/* We have a number of different memory areas.

   WAM 
   
	Global Heap                 
	Local Stack
	Trail Stack

   Runtime system

	Atom Heap
	Code Area
	Backpatch Heap

 */

long total_size = 0;
long heap_size, local_size, atom_size, patch_size;
long code_size, trail_size, temp_size, margin_size;
long read_size;
long globalsize;
char *globalstart;

#ifdef PARALLEL
long worker_size = 0;
long worker_heap_size, worker_local_size, worker_trail_size;
#endif /* PARALLEL */

static worker *init_worker_global_storage(globalstart,n)
    char *globalstart;
    int n;
{
    worker *w;
    int i;
    globalvar *glob;

#ifdef PARALLEL
    s32 *reclevel;
#endif

    /* Allocate workers */

    w = (worker *) globalstart;
    globalstart += sizeof(worker) * n;

    /* Global variables (atom and code heap etc.) */
    
    glob = (globalvar *) globalstart;
    globalstart += sizeof(globalvar);

    	      /* Number of workers */
    glob->active_workers = (s32) (n - 1);

              /* Atom heap */
    glob->atom_start = glob->atom_current = (heap *) globalstart;
    globalstart += atom_size;
    glob->atom_end = (heap *) globalstart;
    glob->atomoffset = (heap *) BaseOffset(glob->atom_start);

             /* Patch Heap */
    glob->patch_start = glob->patch_current = (heap *) globalstart;
    globalstart += patch_size;
    glob->patch_end = (heap *) globalstart;

             /* Code Heap */
    glob->code_start = glob->code_current = (code *) globalstart;
    globalstart += code_size;
    glob->code_end = (code *) globalstart;

#ifdef PARALLEL
             /* Global registers */
    glob->global_regs = (TAGGED *) globalstart;
    globalstart += temp_size;
    AlignDouble(globalstart);
    glob->reduction_results = (double *) globalstart;
    globalstart += sizeof(double) * (n-1);
    glob->collect = (TAGGED *) globalstart;
    globalstart += sizeof(TAGGED) * (n-1);
#endif /* PARALLEL */

#ifndef LOCALPRED
    glob->predtable = (definition **) globalstart;
    globalstart += sizeof(definition *) * PREDHASHLEN;
#endif

    glob->atomtable = (atom_bucket **) globalstart;
    globalstart += sizeof(atom_bucket *) * ATOMHASHLEN;

#ifdef PARALLEL
             /* Iteration levels */
    reclevel = (s32 *) globalstart;
    globalstart += sizeof(u32)*(n-1);
#endif /* PARALLEL */

    for(i = 0 ; i < n ; i++) {
#ifdef PARALLEL
	w[i].level = reclevel;
#endif /* PARALLEL */

	w[i].pid = i;
	w[i].global = glob;
	AlignDouble(globalstart);
	w[i].stats = (statistics *) globalstart;
	globalstart += sizeof(statistics);
    }


    /* Data Heap */

#ifdef COPY_GC
    w[0].heap_start = w[0].heap_top = (TAGGED *) globalstart;
    globalstart += heap_size/2;
    w[0].heap_end = (TAGGED *) globalstart;
    w[0].heap_margin = (TAGGED *) ((u32) globalstart-(u32) margin_size);

    AlignDouble(globalstart);
    w[0].heap_copy_start = w[0].heap_copy_top = (TAGGED *) globalstart;
    globalstart += heap_size/2;
    w[0].heap_copy_end = (TAGGED *) globalstart;
    w[0].heap_copy_margin = (TAGGED *) ((u32) globalstart-(u32) margin_size);
#else /* COPY_GC */
    w[0].heap_start = w[0].heap_top = (TAGGED *) globalstart;
    globalstart += heap_size;
    w[0].heap_end = (TAGGED *) globalstart;
    w[0].heap_margin = (TAGGED *) ((u32) globalstart-(u32) margin_size);
#endif /* COPY_GC */

#ifdef PARALLEL
    for(i = 1 ; i < n ; i++) {
#ifdef COPY_GC
	w[i].heap_start = w[i].heap_top = (TAGGED *) globalstart;
	globalstart += worker_heap_size/2;
	w[i].heap_end = (TAGGED *) globalstart;
	w[i].heap_margin = (TAGGED *) ((u32) globalstart-(u32) margin_size);

	AlignDouble(globalstart);
	w[i].heap_copy_start = w[i].heap_copy_top = (TAGGED *) globalstart;
	globalstart += worker_heap_size/2;
	w[i].heap_copy_end = (TAGGED *) globalstart;
	w[i].heap_copy_margin = (TAGGED *) ((u32) globalstart-
					    (u32) margin_size);
#else /* COPY_GC */
	w[i].heap_start = w[i].heap_top = (TAGGED *) globalstart;
	globalstart += worker_heap_size;
	w[i].heap_end = (TAGGED *) globalstart;
	w[i].heap_margin = (TAGGED *) ((u32) globalstart-(u32) margin_size);
#endif /* COPY_GC */
    }
#endif

    return w;
}

void reset_heap(w)
    worker *w;
{
    w->heap_top = w->heap_start;
}

void init_localstack(size,w)
    long size;
    worker *w;
{
    w->stack_start = (TAGGED *) Malloc(size);

    if(w->stack_start == NULL) 
	FatalError("couldn't allocate local stack");
    w->stack_end = (TAGGED *) (((char *) w->stack_start) + size - 512);
}

void init_trailstack(size,w)
    long size;
    worker *w;
{
  w->trail_top = w->trail_start = (TAGGED *) Malloc(size);

  if(w->trail_top == NULL)
    FatalError("couldn't allocate trail stack");
  w->trail_end = (TAGGED *) (((char *) w->trail_start) + size);
}

void reset_backpatchheap(w)
    worker *w;
{
    w->global->patch_current = w->global->patch_start;
}

heap *atom_alloc(size,w)
    unsigned long size;
    worker *w;
{
    register heap *ret;

    ret = w->global->atom_current;
    w->global->atom_current += size;
    if(w->global->atom_current > w->global->atom_end)
	FatalError("atomheap out of memory");

    return ret;
}

heap *aligned_atom_alloc(size,w)
    unsigned long size;
    worker *w;
{
    register heap *ret;

#ifdef LOWTAGS
    AlignDouble(w->global->atom_current);
#endif /* LOWTAGS */

    ret = w->global->atom_current;

    w->global->atom_current += size;
    if(w->global->atom_current > w->global->atom_end)
	FatalError("atomheap out of memory");

    return ret;
}

#ifdef ALIGN_DOUBLE
heap *float_atom_alloc(size,w)
    unsigned long size;
    worker *w;
{
    register heap *ret;
    register int align;

    align = (((unsigned long) w->global->atom_current) % sizeof(double));

    if(align == 0) {
	ret = w->global->atom_current;
	w->global->atom_current += size;
    } else {
        w->global->atom_current =
	  (heap *) (((unsigned long) w->global->atom_current) +
		    sizeof(double) - align);
	ret = w->global->atom_current;
	w->global->atom_current += size;
    }
    if(w->global->atom_current > w->global->atom_end)
	FatalError("atomheap out of memory");

    return ret;
}
#endif /* ALIGN_DOUBLE */

heap *patch_alloc(size,w)
    unsigned long size;
    worker *w;
{
    register heap *ret;

    ret = w->global->patch_current;
    w->global->patch_current += size;
    if(w->global->patch_current > w->global->patch_end)
	FatalError("patchheap out of memory");

    return ret;
}

void init_temporary_registers(size,w)
    long size;
    worker *w;
{
#ifdef PARALLEL
    if(w->pid == 0) /* the root process */
      {
	w->regs = w->global->global_regs;
#ifdef CACHE_G
	w->g_regs = w->regs;
#endif
      }
    else
#endif
      {
	w->regs = (TAGGED *) Malloc(size);

	if(w->regs == NULL)
	  FatalError("couldn't allocate temporary registers");
#ifdef CACHE_G
	w->g_regs = (TAGGED *) Malloc(size);
	if(w->g_regs == NULL)
	  FatalError("couldn't allocate global registers cache");
#endif /* CACHE_G */
      }

    return;
}

void init_read_stack(size,w)
    long size;
    worker *w;
{
    w->rdstack = (TAGGED *) Malloc(size);

    if(w->rdstack == NULL)
	FatalError("couldn't allocate head unification stack");

    return;
}

worker *init_global_memory(n)
    int n;
{
    float total_parts; 

#ifdef PARALLEL
    float total_worker_parts;
    
    if(worker_size == 0) 
	GETENV(worker_size,"LUTHER_WORKER_MEMORY",LUTHER_WORKER_MEMORY_DEF);
#endif

    if(total_size == 0)
	GETENV(total_size,"LUTHER_TOTAL_MEMORY",LUTHER_TOTAL_MEMORY_DEF);
    
    GETENV(margin_size,"LUTHER_HEAP_MARGIN",LUTHER_HEAP_MARGIN_DEF);
    GETENV(patch_size,"LUTHER_BACKPATCHHEAP_SIZE",
	   LUTHER_BACKPATCHHEAP_SIZE_DEF);
    GETENV(temp_size,"LUTHER_TEMP_SIZE",LUTHER_TEMP_SIZE_DEF);
    GETENV(read_size,"LUTHER_READ_SIZE",LUTHER_READ_SIZE_DEF);

    total_parts = LUTHER_ATOMHEAP_PART + LUTHER_CODESPACE_PART +
	          LUTHER_HEAP_PART + LUTHER_TRAILSTACK_PART +
	          LUTHER_LOCAL_PART;
#ifdef PARALLEL
    total_worker_parts = LUTHER_HEAP_PART + LUTHER_TRAILSTACK_PART +
                         LUTHER_LOCAL_PART;

    worker_heap_size = (long) (LUTHER_HEAP_PART * worker_size / 
			       total_worker_parts);
    worker_local_size = (long)(LUTHER_LOCAL_PART * worker_size /
			       total_worker_parts);
    worker_trail_size = (long)(LUTHER_TRAILSTACK_PART * worker_size /
			       total_worker_parts);

    AlignTagged(worker_heap_size);
    AlignTagged(worker_local_size);
    AlignTagged(worker_trail_size);
#endif /* PARALLEL */
    

    atom_size = (long)(LUTHER_ATOMHEAP_PART * total_size / total_parts);
    code_size = (long)(LUTHER_CODESPACE_PART * total_size / total_parts);
    heap_size = (long)(LUTHER_HEAP_PART * total_size / total_parts);
    trail_size = (long)(LUTHER_TRAILSTACK_PART * total_size / total_parts);
    local_size = (long)(LUTHER_LOCAL_PART * total_size / total_parts);

    AlignTagged(atom_size);
    AlignTagged(code_size);
    AlignTagged(heap_size);
    AlignTagged(trail_size);
    AlignTagged(local_size);

    globalsize = sizeof(globalvar) + temp_size + atom_size + patch_size +
                 code_size + heap_size + 
		 sizeof(statistics) + sizeof(double) + sizeof(worker) +
#ifdef PARALLEL
		 (n-1) *
		 (
		   sizeof(statistics) + sizeof(double) +
		   sizeof(s32) + sizeof(worker) +
		   sizeof(double) + sizeof(TAGGED) + worker_heap_size  
		 ) + 
#endif /* PARALLEL */
#ifndef LOCALPRED
		 sizeof(definition *)*PREDHASHLEN +
#endif		     
		 sizeof(atom_bucket *)*ATOMHASHLEN +
		 1024;                                /* spill buffer */


#ifdef SHARED

    globalstart = mmap_global(globalsize);

#else /* SHARED */

    if ((globalstart = (char *) Malloc(globalsize)) == NULL) {
	FatalError("can't allocate global memory");
    }

#endif /* SHARED */

    return init_worker_global_storage(globalstart, n);
}

void init_local_memory(w)
    worker *w;
{
#ifdef PARALLEL
  if(w->pid != 0) {
    init_localstack(worker_local_size,w);
    init_trailstack(worker_trail_size,w);
  } else 
#endif /* PARALLEL */
  {
    init_localstack(local_size,w);
    init_trailstack(trail_size,w);
  }

  init_temporary_registers(temp_size,w);
  init_read_stack(read_size,w);
}

#ifdef ALIGN_MALLOC

char *aligned_malloc(size)
     s32 size;
{
  register char *res;
  register long align;
  
  res = malloc(size+4);
  
  if ((align = ((long) res) % sizeof(s32)) != 0) {
    res += sizeof(s32) - align;
  }
  
  return res;
}

#endif /* ALIGN_MALLOC */
