/*    File:	 inout.c 
 *    Author:	 Johan Bevemyr
 *    Created:	 Thu Jun  6 14:11:17 1991
 */ 

#include "include.h"
#include "engine.h"
#include "unify.h"
#include "display.h"
#include "expand_file_name.h"
#include "ctype.h"

#ifdef THINK_C
#include <unix.h>
#endif /* THINK_C */

static struct stream_node streams[MAXSTREAMS];

FILE *currin;
FILE *currout;
FILE *currerr;
BOOL currtty_io;

int currin_nl_last = 1;

static TAGGED currinx;
static TAGGED curroutx;
static int first_user_stream;

char char_table[256];

/* $prompt(Old,New) -- used by rdtok.pl */

BOOL luther_prompt(Arg)
    Argdecl;
{
    TAGGED New,Old;

    DerefNLL(Old,Xw(0));
    DerefNLL(New,Xw(1));

    if(IsVar(New)) return unify(Old,prompt,w);

    if(IsATM(New)) {
	if(unify(Old,prompt,w)) {
	    prompt = New;
	    return TRUE;
	}
    }
    return FALSE;
}

/* tty I/O */

/* ttyflush/0 */
BOOL luther_ttyflush(Arg)
    Argdecl;
{
    fflush(stdout);
    return TRUE;
}

/* ttynl/0 */
BOOL luther_ttynl(Arg)
    Argdecl;
{
    putc('\n',stdout);
    return TRUE;
}

/* ttyget0/1 */
BOOL luther_ttyget0(Arg)
    Argdecl;
{
    register TAGGED X0;
    register int i;

    DerefNLL(X0,Xw(0));
    i = getc(stdin);
    return unify(X0,Make_Integer(i),w);
}

/* ttyget/1 */
BOOL luther_ttyget(Arg)
    Argdecl;
{
    register TAGGED X0;
    register int i;

    DerefNLL(X0,Xw(0));
    do { i = getc(stdin);
     } while( !isprint(i) && i != EOF);
    return unify(X0,Make_Integer(i),w);
}

BOOL luther_ttyskip(Arg)
    Argdecl;
{
    register TAGGED Ch;
    register int i,skipchar;

    DerefNLL(Ch,Xw(0));

    if(!IsNUM(Ch)) {
	Error("$skip - illegal 1:st argument");
	return FALSE;
    }

    skipchar = GetNumber(Ch);
    
    do { i = getc(stdin);
    } while( i != skipchar && i != EOF);
    
    return TRUE;
}

/* ttyput/1 */
BOOL luther_ttyput(Arg)
    Argdecl;
{
    register TAGGED X0;

    DerefNLL(X0,Xw(0));

    if(IsNUM(X0)) {
	putc((char) GetNumber(X0),stdout);
    } else {
	luther_error(E_PUT_ARG,X0,w);
	return FALSE;
    }
    return TRUE;
}

/* $ttygetch0(Char,Type) -- used by interpret.pl */

BOOL luther_ttygetch0(Arg)
    Argdecl;
{
    register TAGGED Ch,Ty;
    register int i;
    
    DerefNLL(Ch,Xw(0));
    DerefNLL(Ty,Xw(1));

    if(!(IsNUM(Ch) || IsVar(Ch))) {
	Error("$ttygetch0 - illegal 1:st argument");
	return FALSE;
    }

    if(!(IsNUM(Ty) || IsVar(Ty))) {
	Error("$ttygetch0 - illegal 2:nd argument");
	return FALSE;
    }

    if((currtty_io) && (currin_nl_last == 1)) {
	currin_nl_last = 0;
    }

    i = getc(stdin);

    if((currtty_io) && (i == '\n')) currin_nl_last = 1;
    
    if(unify(Ch,Make_Integer(i),w) &&
       unify(Ty,Make_Integer(GetCharCode(i)),w))
	return TRUE;
    else
	return FALSE;
}


/* I/O on currin/currout */

/* $display/1 */
BOOL luther_display(Arg)
    Argdecl;
{
  display_term(currout,Xw(0),w);
  return TRUE;
}

/* write/1 */
BOOL luther_write(Arg)
    Argdecl;
{
  write_term(currout,Xw(0),w);
  return TRUE;
}

/* flush/0 */
BOOL luther_flush(Arg)
    Argdecl;
{
  fflush(currout);
  return TRUE;
}

/* nl/0 */
BOOL luther_nl(Arg)
    Argdecl;
{
  PL_Print1(currout,"\n");
  return TRUE;
}

/* $getch(Char,Type) -- used by rdtok.pl */

BOOL luther_getch(Arg)
    Argdecl;
{
    register TAGGED Ch,Ty;
    register int i;
    
    DerefNLL(Ch,Xw(0));
    DerefNLL(Ty,Xw(1));

    if(!(IsNUM(Ch) || IsVar(Ch))) {
	Error("$getch - illegal 1:st argument");
	return FALSE;
    }

    if(!(IsNUM(Ty) || IsVar(Ty))) {
	Error("$getch - illegal 2:nd argument");
	return FALSE;
    }

    do {
	if((currtty_io) && (currin_nl_last == 1)) {
	  display_term(currout,prompt,w);
	  fflush(currout);
	  currin_nl_last = 0;
	}
	i = getc(currin);
	if((currtty_io) && (i == '\n')) currin_nl_last = 1;
     } while( !isprint(i) && i != EOF);

    if((i == EOF) && currtty_io) currin_nl_last = 1;

    if(unify(Ch,Make_Integer(i),w) &&
       unify(Ty,Make_Integer(GetCharCode(i)),w))
	return TRUE;
    else
	return FALSE;
}

/* $getch0(Char,Type) -- used by rdtok.pl */

BOOL luther_getch0(Arg)
    Argdecl;
{
    register TAGGED Ch,Ty;
    register int i;
    
    DerefNLL(Ch,Xw(0));
    DerefNLL(Ty,Xw(1));

    if(!(IsNUM(Ch) || IsVar(Ch))) {
	Error("$getch0 - illegal 1:st argument");
	return FALSE;
    }

    if(!(IsNUM(Ty) || IsVar(Ty))) {
	Error("$getch0 - illegal 2:nd argument");
	return FALSE;
    }

    if((currtty_io) && (currin_nl_last == 1)) {
      display_term(currout,prompt,w);
      fflush(currout);
      currin_nl_last = 0;
    }

    i = getc(currin);
    if((currtty_io) && ((i == '\n') || (i == EOF))) currin_nl_last = 1;
    
    if(unify(Ch,Make_Integer(i),w) &&
       unify(Ty,Make_Integer(GetCharCode(i)),w))
	return TRUE;
    else
	return FALSE;
}

/* $skip(Char) -- used by tdtok.pl */

BOOL luther_skip(Arg)
    Argdecl;
{
    register TAGGED Ch;
    register int i,skipchar;

    DerefNLL(Ch,Xw(0));

    if(!IsNUM(Ch)) {
	Error("$skip - illegal 1:st argument");
	return FALSE;
    }

    skipchar = GetNumber(Ch);
    
    do {
	if((currtty_io) && (currin_nl_last == 1)) {
	  display_term(currout,prompt,w);
	  fflush(currout);
	  currin_nl_last = 0;
	}
	i = getc(currin);
	if((currtty_io) && (i == '\n')) currin_nl_last = 1;
    } while( i != skipchar && i != EOF);

    if((i == EOF) && currtty_io) currin_nl_last = 1;
    
    return TRUE;
}


/* get0/1 */
BOOL luther_get0(Arg)
    Argdecl;
{
    register TAGGED X0;
    register int i;

    DerefNLL(X0,Xw(0));

    if((currtty_io) && (currin_nl_last == 1)) {
      display_term(currout,prompt,w);
      fflush(currout);
      currin_nl_last = 0;
    }
    i = getc(currin);
    if((currtty_io) && ((i == '\n') || (i == EOF))) currin_nl_last = 1;

    return unify(X0,Make_Integer(i),w);
}

/* get/1 */
BOOL luther_get(Arg)
    Argdecl;
{
    register TAGGED X0;
    register int i;

    DerefNLL(X0,Xw(0));

    do {
	if((currtty_io) && (currin_nl_last == 1)) {
	  display_term(currout,prompt,w);
	  fflush(currout);
	  currin_nl_last = 0;
	}
	i = getc(currin);
	if((currtty_io) && (i == '\n')) currin_nl_last = 1;
    } while( !isprint(i) && i != EOF);

    if((i == EOF) && currtty_io) currin_nl_last = 1;

    return unify(X0,Make_Integer(i),w);
}

/* put/1 */
BOOL luther_put(Arg)
    Argdecl;
{
    register TAGGED X0;

    DerefNLL(X0,Xw(0));
    if(IsNUM(X0)) {
	putc((char) GetNumber(X0), currout);
    } else {
	luther_error(E_PUT_ARG,X0,w);
    }
    return TRUE;
}

/***********************************************************************
 *
 * Given a stream as an argument                                        
 *
 */

/* $write/2 */
BOOL luther_write2(Arg)
    Argdecl;
{
  register TAGGED Stream;
  register FILE *strm;
  
  DerefNLL(Stream,Xw(0));
  GetOutStream(Stream,strm);

  write_term(strm, Xw(1),w);
  return TRUE;
}

/* $display/2 */
BOOL luther_display2(Arg)
    Argdecl;
{
    register TAGGED Stream;
    register FILE *strm;
    
    DerefNLL(Stream,Xw(0));
    GetOutStream(Stream,strm);

    display_term(strm, Xw(1), w);
    return TRUE;
}


/* flush/1 */
BOOL luther_flush_stream(Arg)
    Argdecl;
{
    register TAGGED Stream;
    register FILE *strm;
    
    DerefNLL(Stream,Xw(0));
    GetOutStream(Stream,strm);

    fflush(strm);
    return TRUE;
}

/* $nl/1 */
BOOL luther_nl_stream(Arg)
    Argdecl;
{
    register TAGGED Stream;
    register FILE *strm;
    
    DerefNLL(Stream,Xw(0));
    GetOutStream(Stream,strm);
    
    putc('\n',strm);
    return TRUE;
}

/* get0/2 */
BOOL luther_get0_stream(Arg)
    Argdecl;
{
    register TAGGED Stream, Ch;
    register FILE *strm;
    register int i;
    
    DerefNLL(Stream,Xw(0));
    GetInStream(Stream,strm);

    DerefNLL(Ch,Xw(1));
    i = getc(strm);
    return unify(Ch, Make_Integer(i),w);
}

/* get/2 */
BOOL luther_get_stream(Arg)
    Argdecl;
{
    register TAGGED Stream, Ch;
    register FILE *strm;
    register int i;
    
    DerefNLL(Stream,Xw(0));
    GetInStream(Stream,strm);

    DerefNLL(Ch,Xw(1));

    do { i = getc(strm);
     } while( !isprint(i) && i != EOF);
    return unify(Ch,Make_Integer(i),w);
}

/* put/2 */
BOOL luther_put_stream(Arg)
    Argdecl;
{
    register TAGGED Stream, Ch;
    register FILE *strm;
    
    DerefNLL(Stream,Xw(0));
    GetOutStream(Stream,strm);

    DerefNLL(Ch,Xw(1));

    if(IsNUM(Ch))
      {
	putc((char) GetNumber(Ch), strm);
      }
    else
      {
	luther_error(E_PUT_ARG,Ch,w);
      }
    return TRUE;
}

/***********************************************************************
 *
 * Primitives for manipulating streams 
 *
 */
    
/* currnet_input/1 */
BOOL luther_current_input(Arg)
    Argdecl;
{
    register TAGGED X0;

    DerefNLL(X0,Xw(0));

    return unify(X0, currinx, w);
}

/* current_output/1 */
BOOL luther_current_output(Arg)
    Argdecl;
{
    register TAGGED X0;

    DerefNLL(X0,Xw(0));

    return unify(X0, curroutx, w);
}

/* set_input/1 */
BOOL luther_set_input(Arg)
    Argdecl;
{
    register TAGGED X0;
    register int index;

    DerefNLL(X0,Xw(0));

    GetStreamInIndex(X0,index);

    currin = streams[index].file;
    currtty_io = streams[index].tty_io;
    currinx = X0;

    return TRUE;
}

/* set_output/1 */
BOOL luther_set_output(Arg)
    Argdecl;
{
    register TAGGED X0;
    register int index;

    DerefNLL(X0,Xw(0));

    GetStreamOutIndex(X0,index);

    currout = streams[index].file;
    curroutx = X0;

    return TRUE;
}

/* streams/1 unifies the argument with a list containing the
   index of all opened streams */

BOOL luther_streams(Arg)
    Argdecl;
{
    register TAGGED X0;
    TAGGED lst, rest;
    int i;

    extern TAGGED atom_nil;

    DerefNLL(X0, Xw(0));
    rest = atom_nil;

    for(i = first_user_stream ; i < MAXSTREAMS ; i++) {
	if(streams[i].file != NULL) {
	    lst = Tagify(w->heap_top,LST);
	    PushOnHeap(w->heap_top,Make_Integer(i));
	    PushOnHeap(w->heap_top,rest);
	    rest = lst;
	}
    }
    return unify(X0, rest,w);
}

/* primitives for getting information about a stream */

BOOL luther_stream_name(Arg)
    Argdecl;
{
  register TAGGED X0, X1;
  register int index;
  
  DerefNLL(X0,Xw(0));
  DerefNLL(X1,Xw(1));
  
  GetStreamIndex(X0,index);
  
  if(streams[index].name != (TAGGED) NULL)
    {
      return unify(X1, streams[index].name,w);
    }
  else
    return FALSE;
}

BOOL luther_stream_mode(Arg)
    Argdecl;
{
  register TAGGED X0,X1;
  register int index;
  
  DerefNLL(X0,Xw(0));
  DerefNLL(X1,Xw(1));
  
  GetStreamIndex(X0,index);
  
  if(streams[index].mode != (TAGGED) NULL)
    {
      return unify(X1,Make_Integer(streams[index].mode),w);
    }
  else
    return FALSE;
}

BOOL luther_stream_file(Arg)
    Argdecl;
{
    register TAGGED X0,X1;
    register int index;

    DerefNLL(X0,Xw(0));
    DerefNLL(X1,Xw(1));

    GetStreamIndex(X0,index);

    if(streams[index].file != NULL)
      {
	return unify(X1,PointerToTerm(streams[index].file),w);
      }
    else
      return FALSE;
}

#ifndef THINK_C
BOOL luther_stream_code(Arg)
    Argdecl;
{
    register TAGGED X0,X1;
    register int index;

    DerefNLL(X0,Xw(0));
    DerefNLL(X1,Xw(1));

    GetStreamIndex(X0,index);

    if(streams[index].file != NULL)
      {
	return unify(X1,PointerToTerm(fileno(streams[index].file)),w);
      }
    else
      return FALSE;
}

BOOL luther_file_code(Arg)
    Argdecl;
{
    register TAGGED X0,X1;
    register int index;

    DerefNLL(X0,Xw(0));
    DerefNLL(X1,Xw(1));

    GetStreamIndex(X0,index);

    if(streams[index].file != NULL)
      {
	return unify(X1,Make_Integer(fileno(streams[index].file)),w);
      }
    else
      return FALSE;
}
#endif
    
BOOL luther_code_file(Arg)
    Argdecl;
{
    register TAGGED X0,X1;

    DerefNLL(X0,Xw(0));
    DerefNLL(X1,Xw(1));

    return unify(X1,X0,w);
}
    
/* $fopen/3 opens a file, first argument is a file name, second argument is
   "r"/"w"/"a", the third argument is unified with a file index.
   */

BOOL luther_fopen(Arg)
    Argdecl;
{
  register TAGGED X0, X1, X2;
  
  DerefNLL(X0,Xw(0));
  DerefNLL(X1,Xw(1));
  DerefNLL(X2,Xw(2));
  
  if (IsATM(X1))
    {
      if (IsATM(X0))
	{
	  if (X0 == atom_user || X0 == atom_user_input ||
	      X0 == atom_user_output)
	    {
	      register TAGGED strref;
	      register FILE *fp;
	      register int i, fd;
	      
	      for(i = 0; streams[i].file != NULL && i < MAXSTREAMS ; i++);
	      
	      if(i == MAXSTREAMS)
		{
		  luther_error(E_NR_FILES,X0,w);
		  return FALSE;
		}
	      
	      if ( X1 == atom_r )
		{
		  fd = fileno(streams[USER_IN].file);
		}
	      else
		{
		  fd = fileno(streams[USER_OUT].file);
		}
	      
	      fp = (FILE *) fdopen(fd,GetString(X1,w));
	      
	      if(fp == NULL)
		{
		  luther_error(E_OPEN_FILE,X0,w);
		  return FALSE;
		}
	      
	      streams[i].file = fp;
	      streams[i].name = X0;
	      streams[i].mode = X1;
	      streams[i].tty_io = TRUE;
	      
	      Make_STR(w->heap_top,strref,functor_d_stream);
	      PushOnHeap(w->heap_top,Make_Integer(i));
	      return unify(X2,strref,w);
	    }
	  else
	    {
	      register TAGGED strref;
	      register FILE *fp;
	      register int i;
	      char pathBuf[MAXPATHLEN+1];
	      
	      for(i = 0; streams[i].file != NULL && i < MAXSTREAMS ; i++);
	      
	      if(i == MAXSTREAMS)
		{
		  luther_error(E_NR_FILES,X0,w);
		  return FALSE;
		}
	      
	      if(!expand_file_name(GetString(X0,w),pathBuf))
		return FALSE;
	      
	      fp = fopen(pathBuf, GetString(X1,w));
	      
	      if(fp == NULL)
		{
		  luther_error(E_OPEN_FILE,X0,w);
		  return FALSE;
		}
	      
	      streams[i].file = fp;
	      streams[i].name = X0;
	      streams[i].mode = X1;
	      streams[i].tty_io = FALSE;
	      
	      Make_STR(w->heap_top,strref,functor_d_stream);
	      PushOnHeap(w->heap_top,Make_Integer(i));
	      return unify(X2,strref,w);
	    }
	}
      else if (IsStream(X0))
	{
	  TAGGED StreamIndex;
	  
	  DerefNLL(StreamIndex, Ref(GetArg(X0,0)));
	  
	  if (IsNUM(StreamIndex))
	    {
	      return (unify(X1,streams[GetNumber(StreamIndex)].mode,w) &&
		      unify(X0,X2,w));
	    }
	}
    }

  Error("fopen - illegal arguments");
  return FALSE;
}

/* $fdopen/3 opens a file, first argument is a descriptor, second argument is
   "r"/"w"/"a", the last argument is unified with a file index
   */

BOOL luther_fdopen(Arg)
    Argdecl;
{
    register TAGGED X0, X1, X2, strref;
    int i;
    FILE *fp;

    DerefNLL(X0, Xw(0));
    DerefNLL(X1, Xw(1));
    DerefNLL(X2, Xw(2));

    if(!(IsNUM(X0) && IsATM(X1))) {
	Error("fdopen - illegal arguments");
	return FALSE;
    }

    for(i = 0; streams[i].file != NULL && i < MAXSTREAMS; i++);

    if(i == MAXSTREAMS) {
	luther_error(E_NR_FILES, X0,w);
	return FALSE;
    }

    fp = (FILE *) fdopen(GetNumber(X0), GetString(X1,w));

    if(fp == (TAGGED) NULL) {
	luther_error(E_OPEN_FILE,X0,w);
	return FALSE;
    }

    streams[i].file = fp;
    streams[i].name = (TAGGED) NULL;
    streams[i].mode = X1;
    streams[i].tty_io = FALSE;

    Make_STR(w->heap_top,strref,functor_d_stream);
    PushOnHeap(w->heap_top,Make_Integer(i));

    return unify(X2,strref,w);
}

/* rewind/1 */


BOOL luther_rewind(Arg)
    Argdecl;
{
  register TAGGED X0;
  register int index;

  DerefNLL(X0,Xw(0));

  GetStreamIndex(X0,index);

  if (streams[index].file != NULL) {
    rewind(streams[index].file);
  }
  return TRUE;
}

/*  $close/1 closes a file given a file index
    */

BOOL luther_close(Arg)
    Argdecl;
{
  register TAGGED X0;
  register int index;

  DerefNLL(X0,Xw(0));

  GetStreamIndex(X0,index);

  if((MAXSTREAMS > index) && (index >= first_user_stream))
    {
      if (streams[index].tty_io == FALSE) 
	fclose(streams[index].file);
      streams[index].file = NULL;
      streams[index].name = (TAGGED) NULL;
      streams[index].mode = (TAGGED) NULL;
      streams[index].tty_io = (TAGGED) NULL;
    }
  return TRUE;

}

void initialize_streams()
{
  /* streams initialisation */
  
  streams[USER_IN].file = stdin;
  streams[USER_IN].name = atom_user_input;
  streams[USER_IN].mode = atom_r;
  streams[USER_IN].tty_io = TRUE;
  
  streams[USER_OUT].file = stdout;
  streams[USER_OUT].name = atom_user_output;
  streams[USER_OUT].mode = atom_w;
  streams[USER_OUT].tty_io = TRUE;
  
  streams[USER_ERR].file = stderr;
  streams[USER_ERR].name = atom_user_error;
  streams[USER_ERR].mode = atom_w;
  streams[USER_ERR].tty_io = TRUE;
  
  first_user_stream = 3;
  
  currin  = stdin;
  currout = stdout;
  currerr = stderr;

  currtty_io = TRUE;
  
  currinx =  atom_user_input;
  curroutx = atom_user_output;
  
  return;
}
