/*    File:	 inline.c 
 *    Author:	 Johan Bevemyr
 *    Created:	 Wed Jun  5 15:49:52 1991
 */ 

#include "include.h"
#include "inline.h"
#include "engine.h"
#include "initial.h"
#include "unify.h"
#include "array.h"
#include <math.h>

static TAGGED functor_plus, functor_minus, functor_times, functor_div;
static TAGGED functor_u_minus;
static TAGGED functor_intdiv, functor_mod, functor_integer;
static TAGGED functor_float, functor_bin_or, functor_bin_and, functor_bin_xor;
static TAGGED functor_exp, functor_log, functor_sin, functor_cos, functor_tan;
static TAGGED functor_asin, functor_acos, functor_atan, functor_rshift;
static TAGGED functor_lshift, functor_aref, functor_abs, functor_pow;
static TAGGED functor_floor, functor_ceil;

inline_entry inline_table[INLINE_TABLE_SIZE] = {
  {"$compare",        3, luther_compare,		I_PRED, 0},
  {"$functor",        3, luther_functor,		I_PRED, 0},
  {"$arg",            3, luther_arg,			I_PRED, 0},
  {"$eq_univ",        2, luther_eq_univ,		I_PRED, 0},
  {"$ineq_univ",      2, luther_ineq_univ,		I_PRED, 0},
  {"$lt_univ",        2, luther_lt_univ,		I_PRED, 0},
  {"$gt_univ",        2, luther_gt_univ,		I_PRED, 0},
  {"$le_univ",        2, luther_le_univ,		I_PRED, 0},
  {"$ge_univ",        2, luther_ge_univ,		I_PRED, 0},
  {"$eq",             2, luther_math_eq,		I_PRED, 0},
  {"$ineq",           2, luther_math_ineq,		I_PRED, 0},
  {"$lt",             2, luther_lt,			I_PRED, 0},
  {"$gt",             2, luther_gt,			I_PRED, 0},
  {"$le",             2, luther_le,			I_PRED, 0},
  {"$ge",             2, luther_ge,			I_PRED, 0},
  {"$atom",           1, luther_atom,			I_PRED, 0},
  {"$atomic",         1, luther_atomic,			I_PRED, 0},
  {"$integer",        1, luther_integer,		I_PRED, 0},
  {"$float",          1, luther_float,			I_PRED, 0},
  {"$number",         1, luther_number,			I_PRED, 0},
  {"$nonvar",         1, luther_nonvar,			I_PRED, 0},
  {"$var",            1, luther_var,			I_PRED, 0},
  {"$plus",           3, luther_plus,			I_FUNC, 1},
  {"$plus_1",         2, luther_plus_1,			I_FUNC, 1},
  {"$minus",          3, luther_minus,			I_FUNC, 1},
  {"$minus_1",        2, luther_minus_1,		I_FUNC, 1},
  {"$times",          3, luther_times,			I_FUNC, 1},
  {"$div",            3, luther_div,			I_FUNC, 1},
  {"$intdiv",         3, luther_intdiv,			I_FUNC, 1},
  {"$mod",            3, luther_mod,			I_FUNC, 1},
  {"$tointeger",      2, luther_tointeger,		I_FUNC, 1},
  {"$tofloat",        2, luther_tofloat,		I_FUNC, 1},
  {"$eval_math",      2, luther_eval_math,		I_FUNC, 1},
  {"$univ",           2, luther_univ,			I_PRED, 0},
  {"$b_or",           3, luther_bin_or,			I_FUNC, 1},
  {"$b_and",          3, luther_bin_and,		I_FUNC, 1},
  {"$b_xor",          3, luther_bin_xor,		I_FUNC, 1},
  {"$un_minus",       2, luther_unary_minus,		I_FUNC, 1},
  {"$exp",            2, luther_exp,			I_FUNC, 1},
  {"$log",            2, luther_log,			I_FUNC, 1},
  {"$sin",            2, luther_sin,			I_FUNC, 1},
  {"$cos",            2, luther_cos,			I_FUNC, 1},
  {"$tan",            2, luther_tan,			I_FUNC, 1},
  {"$asin",           2, luther_asin,			I_FUNC, 1},
  {"$atan",           2, luther_atan,			I_FUNC, 1},
  {"$acos",           2, luther_acos,			I_FUNC, 1},
  {"$lshift",         3, luther_lshift,			I_FUNC, 1},
  {"$rshift",         3, luther_rshift,			I_FUNC, 1},
  {"$array_size",     2, luther_array_in_size,		I_PRED, 0},
  {"$array_elt",      3, luther_array_in_elt,		I_PRED, 0},
  {"$unify",          2, luther_inline_unify,		I_PRED, 0},
  {"$generic",        1, luther_generic,		I_PRED, 0},
  {"$aref",           3, luther_aref,			I_FUNC, 1},
  {"$array_ref",      3, luther_array_in_ref,		I_FUNC, 1},
  {"$reduce_vector_+",3, luther_reduce_vector_plus,	I_FUNC, 1},
  {"$abs",            2, luther_abs,			I_FUNC, 1},
  {"$reduce_vector_*",3, luther_reduce_vector_times,	I_FUNC, 1},
  {"$par_reduce_+",   2, luther_par_reduce_plus,	I_PRED, 0},
  {"$par_reduce_*",   2, luther_par_reduce_times,	I_PRED, 0},
  {"$pow",            3, luther_pow,                    I_FUNC, 1},
  {"$active",         1, luther_active,                 I_FUNC, 1},
  {"$save",           1, luther_save,                   I_PRED, 0},
  {"$collect_plus",   1, luther_collect_plus,           I_PRED, 0},
  {"$collect_times",  1, luther_collect_times,          I_PRED, 0},
  {"$array_size_var", 2, luther_array_var_size,		I_FUNC, 1},
  {"$array_elt_var",  3, luther_array_var_elt,		I_FUNC, 1},
  {"$elt",  	      3, luther_elt,		        I_FUNC, 3},
  {"$floor",  	      2, luther_floor,		        I_FUNC, 1},
  {"$ceil",  	      2, luther_ceil,		        I_FUNC, 1}
};

/* compare/3 */
/* The order is as follows:

   Variables, ordered by age

   Integers, in numeric order

   Floats, in numeric order

   Atoms, in alphabetical order

   Compound terms, ordered first by arity, then by name of the
   principal functor, then by the arguments (in left-to-right order).
   Remember that a list is a structure with fuctor '.'.
   
   0 = Less
   1 = Greater

         H  C  S  L  N  F  A  L  S  G
         V  V  V  C  U  L  T  S  T  E
         A  A  A  K  M  T  M  T  R  N
   HVA { 2,13, 0, 1, 0, 0, 0, 0, 0, 0 }
   CVA {13,13, 0, 1, 0, 0, 0, 0, 0, 0 }
   SVA { 1, 1, 3, 1, 0, 0, 0, 0, 0, 0 }
   LCK { 0, 0, 0, 1, 0, 0, 0, 0, 0, 0 }
   NUM { 1, 1, 1, 1, 4, 0, 0, 0, 0,12 }
   FLT { 1, 1, 1, 1, 1, 5, 0, 0, 0,12 }
   ATM { 1, 1, 1, 1, 1, 1, 6, 0, 0,12 }
   LST { 1, 1, 1, 1, 1, 1, 1, 7, 8,12 }
   STR { 1, 1, 1, 1, 1, 1, 1, 9,10,12 }
   GEN { 1, 1, 1, 1,11,11,11,11,11,11 }
 */           

#ifdef LOCKING 
#define L(X,Y) X,Y
#else
#define L(X,Y) Y
#endif

#ifdef CONSTR
#define C(X,Y) X,Y
#else
#define C(X,Y) Y
#endif

#ifdef UNBOUND
#define U(X,Y) X,Y
#else
#define U(X,Y) X
#endif
  
static int compare_table[MAX_TAG_LO][MAX_TAG_LO] = {
    { 2,C(13,0), L(1,0), 0, 0, 0, 0, U(0,0)},
#ifdef CONSTR
    {13,C(13,0), L(1,0), 0, 0, 0, 0, U(0,0)},
#endif
    { 1, C(1,3), L(1,0), 0, 0, 0, 0, U(0,0)},
#ifdef LOCKING
    { 0, C(0,0), L(1,0), 0, 0, 0, 0, U(0,0)},
#endif
    { 1, C(1,1), L(1,4), 0, 0, 0, 0,U(12,0)},
    { 1, C(1,1), L(1,1), 5, 0, 0, 0,U(12,0)},
    { 1, C(1,1), L(1,1), 1, 6, 0, 0,U(12,0)},
    { 1, C(1,1), L(1,1), 1, 1, 7, 8,U(12,0)},
    { 1, C(1,1), L(1,1), 1, 1, 9,10,U(12,0)}, 
    { 1, C(1,1), L(1,11),11,11,11,11,U(11,0) }
#ifdef UNBOUND
,   { 0, C(0,0), L(0,0),  0, 0, 0, 0, 0, 0}
#endif
};

           
SIZE compare_struct(term1,term2,arity,w)
    register TAGGED *term1, *term2;
    register int arity;
    worker *w;
{
    register TAGGED dt1, dt2;

    register SIZE rem;
    while(arity--) {
	DerefNLL(dt1,Ref(term1));
	DerefNLL(dt2,Ref(term2));
	rem = compare_term(dt1,dt2,w);
	if(rem != EQUAL) return rem;
	term1 += VARSIZE; term2 += VARSIZE;
    }
    return EQUAL;
}
	

static SIZE compare_term(term1, term2,w)
    TAGGED term1, term2;
    worker *w;
{
    if(term1 == term2) return EQUAL;
    switch(compare_table[LowTagOf(term1)][LowTagOf(term2)]) {
    case 0:
	return LESS;
    case 1:
	return GREATER;
    case 2:
	if(term1 < term2) return LESS;
	return GREATER;
    case 3:
	if(term1 < term2) return LESS;
	return GREATER;
    case 4:
	if(GetNumber(term1) < GetNumber(term2))
	    return LESS;
	if(GetNumber(term1) > GetNumber(term2))
	    return GREATER;
	return COM_ERROR;
    case 5:
	if(GetFloat(term1) < GetFloat(term2))
	    return LESS;
	if(GetFloat(term1) > GetFloat(term2))
	    return GREATER;
	return EQUAL;
    case 6: {
	int res = strcmp(GetString(term1,w),GetString(term2,w));
	if(res < 0) return LESS;
	if(res > 0) return GREATER;
	return COM_ERROR;
    }
    case 7:
	return compare_struct(RemoveTag(term1,STR),RemoveTag(term2,STR),2,w);
    case 8: {
	int res;
	int arity = StructArity(RemoveTag(term2,STR));

	if(arity < 2) return GREATER;
	if(arity > 2) return LESS;
	res = strcmp(".",StructString(term2));
	if(res < 0) return LESS;
	if(res > 0) return GREATER;
	Error("compare_term - structure looks like list but isn't!");
	return COM_ERROR;
    }
    case 9: {
	int res;
	int arity = StructArity(RemoveTag(term1,STR));

	if(arity < 2) return LESS;
	if(arity > 2) return GREATER;
	res = strcmp(".",StructString(term2));
	if(res < 0) return GREATER;
	if(res > 0) return LESS;
	Error("compare_term - structure looks like list but isn't!");
	return COM_ERROR;
    }
    case 10: {
	int a1, a2, res;
	a1 = StructArity(RemoveTag(term1,STR));
	a2 = StructArity(RemoveTag(term2,STR));

	if(a1 < a2) return GREATER;
	if(a1 > a2) return GREATER;
	if(GetFunctor(term1) == GetFunctor(term2))
	    return compare_struct(GetArg(term1,0),
				  GetArg(term2,0), a1,w);
	res = strcmp(StructString(term1),StructString(term2));
	if(res < 0) return LESS;
	if(res > 0) return GREATER;
	Error("compare_term - same arity same name but term1 != term2");
	return COM_ERROR;
    }
    case 11: 
	return GetMethod(compare,term1)(term1,term2,w);
    case 12 : {
	SIZE res = GetMethod(compare,term2)(term2,term1,w);
	if(res == LESS) return GREATER;
	if(res == GREATER) return LESS;
	return res;
    }
	
#ifdef CONSTR
    case 13:
	if(RemoveTag(term1,CVA) < RemoveTag(term2,CVA)) return LESS;
	return GREATER;
#endif /* LOCKING */

    default:
	Error("compare_term - compare index table broken")
	return COM_ERROR;
    }
}

static BOOL luther_compare(InArg)
    InArgdecl;
{
    TAGGED operator, term1, term2;
    int result;

    DerefNLL(term1,    Xw(regs[1]));
    DerefNLL(term2,    Xw(regs[2]));

    operator = Xw(regs[0]);
    result = compare_term(term1,term2,w);
    DerefLockSwitch(operator,
		    {
		      switch(result) {
		      case LESS:
			Bind_Unsafe(operator, atom_less, { return FALSE; });
			return TRUE;
		      case EQUAL:
			Bind_Unsafe(operator, atom_equal,{ return FALSE; });
			return TRUE;
		      case GREATER:
			Bind_Unsafe(operator, atom_greater,{return FALSE;});
			return TRUE;
		      case COM_ERROR:
			Drop_P_Lock(operator, operator);
			return FALSE;
		      }
		    },
		    {
		      switch(result) {
		      case LESS:
			if(operator == atom_less) return TRUE;
			break;
		      case EQUAL:
			if(operator == atom_equal) return TRUE;
			break;
		      case GREATER:
			if(operator == atom_greater) return TRUE;
			break;
		      case COM_ERROR:
			return FALSE;
		      }
		      return FALSE;
		    });
    return FALSE;
}

static BOOL luther_eq_univ(InArg)
    InArgdecl;
{
    TAGGED X0, X1;

    DerefNLL(X0,Xw(regs[0])); /* Term1 */
    DerefNLL(X1,Xw(regs[1])); /* Term2 */

    if(compare_term(X0,X1,w) == EQUAL) return TRUE;

    return FALSE;
}

static BOOL luther_ineq_univ(InArg)
    InArgdecl;
{
    TAGGED X0, X1;

    DerefNLL(X0,Xw(regs[0])); /* Term1 */
    DerefNLL(X1,Xw(regs[1])); /* Term2 */

    if(compare_term(X0,X1,w) != EQUAL) return TRUE;

    return FALSE;
}

/* @< */
static BOOL luther_lt_univ(InArg)
    InArgdecl;
{
    TAGGED X0, X1;

    DerefNLL(X0,Xw(regs[0])); /* Term1 */
    DerefNLL(X1,Xw(regs[1])); /* Term2 */

    if(compare_term(X0,X1,w) == LESS) return TRUE;

    return FALSE;
}

/* @> */

static BOOL luther_gt_univ(InArg)
    InArgdecl;
{
    TAGGED X0, X1;

    DerefNLL(X0,Xw(regs[0])); /* Term1 */
    DerefNLL(X1,Xw(regs[1])); /* Term2 */

    if(compare_term(X0,X1,w) == GREATER) return TRUE;

    return FALSE;
}

/* @=< */

static BOOL luther_le_univ(InArg)
    InArgdecl;
{
    TAGGED X0, X1;

    DerefNLL(X0,Xw(regs[0])); /* Term1 */
    DerefNLL(X1,Xw(regs[1])); /* Term2 */

    if(compare_term(X0,X1,w) != GREATER) return TRUE;

    return FALSE;
}

/* @>= */

static BOOL luther_ge_univ(InArg)
    InArgdecl;
{
    TAGGED X0, X1;

    DerefNLL(X0,Xw(regs[0])); /* Term1 */
    DerefNLL(X1,Xw(regs[1])); /* Term2 */

    if(compare_term(X0,X1,w) != LESS) return TRUE;

    return FALSE;
}

/* $univ(Term,List) (=..) */

static BOOL luther_univ(InArg)
    InArgdecl;
{
    TAGGED Term,List,Functor,Car,Cdr,Str;
    int arity;

    DerefNLL(Term,Xw(regs[0]));
    DerefNLL(List,Xw(regs[1]));

    switch(LowTagOf(Term)) {
    case HVA_LO:
#ifdef CONSTR
    case CVA_LO:
#endif /* LOCKING */
    case SVA_LO:
	if(!IsLST(List)) goto barf;

	/* Check arguments */
	DerefNLL(Cdr,Ref(GetCdr(List)));
	for(arity = 0 ; IsLST(Cdr) ; arity++)
	    DerefNLL(Cdr,Ref(GetCdr(Cdr)));
	if(Cdr != atom_nil) goto barf;

	DerefNLL(Functor, Ref(GetCar(List)));
	if((arity == 0)) {
	    if(IsATM(Functor) || IsNumber(Functor)) {
		return unify(Term,Functor,w);
	    } else
		return FALSE;
	} else if (!IsATM(Functor)) goto barf;
	

	/* check if list */
	if((Functor == atom_list) && (arity == 2)) {
	    Make_LST(w->heap_top,Str);
	    DerefNLL(Cdr,Ref(GetCdr(List)));
	    DerefNLL(Car,Ref(GetCar(Cdr)));
	    DerefNLL(Cdr,Ref(GetCdr(Cdr)));
	    PushOnHeap(w->heap_top,Car);
	    DerefNLL(Car,Ref(GetCar(Cdr)));
	    PushOnHeap(w->heap_top,Car);
	    
	    return unify(Term,Str,w);
	} else {
	    /* Make heap term */

	    Make_STR(w->heap_top,Str,StoreFunctor(Functor,arity));
	    DerefNLL(Cdr,Ref(GetCdr(List)));

	    while(arity--) {
		DerefNLL(Car,Ref(GetCar(Cdr)));
		PushOnHeap(w->heap_top,Car);
		DerefNLL(Cdr,Ref(GetCdr(Cdr)));
	    }

	    return unify(Term,Str,w);
	}
	break;
	
    case NUM_LO:
    case FLT_LO:
    case ATM_LO:
	Make_LST(w->heap_top,Str);
	PushOnHeap(w->heap_top,Term);
	PushOnHeap(w->heap_top,atom_nil);
	return unify(Str,List,w);
	break;
	
    case LST_LO:
	Make_LST(w->heap_top,Str);
	PushOnHeap(w->heap_top, atom_list);
	PushOnHeap(w->heap_top, Tagify(w->heap_top+VARSIZE,LST));
	PushOnHeap(w->heap_top, Ref(GetCar(Term)));
	PushOnHeap(w->heap_top, Tagify(w->heap_top+VARSIZE,LST));
	PushOnHeap(w->heap_top, Ref(GetCdr(Term)));

	return unify(Str,List,w);
	break;
	
    case STR_LO:
	{
	    TAGGED *strarg;
	    
	    strarg = GetArg(Term,0);
	    arity  = GetArity(Term);
	    
	    Make_LST(w->heap_top,Str);
	    PushOnHeap(w->heap_top,GetSTRatom(Term));
	    
	    while(arity--) {
		PushOnHeap(w->heap_top,Tagify(w->heap_top+VARSIZE,LST));
		PushOnHeap(w->heap_top,Ref(strarg));
		strarg += VARSIZE;
	    }
	    PushOnHeap(w->heap_top,atom_nil);
	    
	    return unify(Str,List,w);
	}
	break;

    default:
	goto barf;
    }

 barf:
    Error("=../2: illegal arguments");
    return FALSE;

}

static BOOL luther_functor(InArg)
    InArgdecl;
{
    TAGGED X0, X1, X2;

    DerefNLL(X0,Xw(regs[0])); /* Term */
    DerefNLL(X1,Xw(regs[1])); /* Name */
    DerefNLL(X2,Xw(regs[2])); /* Arity */

    switch(TagOf(X0)) {
    case HVA:
#ifdef CONSTR
    case CVA:
#endif /* LOCKING */	
    case SVA:
    {
	TAGGED f;
	int j;

	if((IsATM(X1) || IsNUM(X1)) && IsNUM(X2)) {
	    if(GetNumber(X2) == 0) {
		return unify(X0,X1,w);
	    }
	    if((X1 == atom_list) && (GetNumber(X2) == 2)) {
		f =  Tagify(w->heap_top,LST);
		CreateHVA(w->heap_top,w);
		CreateHVA(w->heap_top,w);
		return unify(X0,f,w);
	    }
	    if(IsATM(X1)) {
		f = Tagify(w->heap_top,STR);
		j = GetNumber(X2);
		PushOnHeapF(w->heap_top,StoreFunctor(X1,j));
		while((j--) != 0) {
		    CreateHVA(w->heap_top,w);
		}
		return unify(X0,f,w);
	    }
	} else 
	    return FALSE;
    }
    case NUM:
    case FLT:
    case ATM:
	if(unify(X1,X0,w))
	    return unify(X2,Make_Integer(0),w);
	else
	    return FALSE;

    case LST:
	if(unify(X1,atom_list,w))
	    return unify(X2,Make_Integer(2),w);
	else
	    return FALSE;

    case STR: {
	UNTAGGED x0 = (UNTAGGED) RemoveTag(X0,STR);
	if(unify(FunctorToAtom(Struct(x0)->functor),X1,w))
	    return unify(Make_Integer(StructArity(x0)),X2,w);
	else
	    return FALSE;
    }
    case GEN:
        return FALSE;
    default:
	Error("functor - no such term type");
	return FALSE;
    }			       
}

static BOOL luther_arg(InArg)
    InArgdecl;
{
    TAGGED X0, X1, X2, Tmp;

    DerefNLL(X0,Xw(regs[0])); /* ArgNo */
    DerefNLL(X1,Xw(regs[1])); /* Term */
    DerefNLL(X2,Xw(regs[2])); /* Arg */

    if((!IsNUM(X0)) || (!IsCompound(X1)))
	return FALSE;

    if(IsLST(X1)) {
	if(GetNumber(X0) == 1) {
	    DerefNLL(Tmp,Ref(GetCar(X1)));
	    return unify(Tmp,X2,w);
	}
	if(GetNumber(X0) == 2) {
	    DerefNLL(Tmp,Ref(GetCdr(X1)));
	    return unify(Tmp,X2,w);
	}
	return FALSE;
    }

    if((GetNumber(X0) > StructArity(RemoveTag(X1,STR))) ||
       (GetNumber(X0) <= 0))
	return FALSE;
    DerefNLL(Tmp,Ref(GetArg(X1,(GetNumber(X0)-1))));
    return unify(Tmp,X2,w);
}

static BOOL luther_elt(InArg)
    InArgdecl;
{
    TAGGED X0, X1, Tmp;

    DerefNLL(X0,Xw(regs[0])); /* ArgNo */
    DerefNLL(X1,Xw(regs[1])); /* Term */

    if((!IsNUM(X0)) || (!IsCompound(X1)))
	return FALSE;

    if(IsLST(X1)) {
	if(GetNumber(X0) == 1) {
	    DerefNLL(Tmp,Ref(GetCar(X1)));
	    Xw(regs[2]) = Tmp;
	    return TRUE;
	}
	if(GetNumber(X0) == 2) {
	    DerefNLL(Tmp,Ref(GetCdr(X1)));
	    Xw(regs[2]) = Tmp;
	    return TRUE;
	}
	return FALSE;
    }

    if((GetNumber(X0) > StructArity(RemoveTag(X1,STR))) ||
       (GetNumber(X0) <= 0))
	return FALSE;
    DerefNLL(Tmp,Ref(GetArg(X1,(GetNumber(X0)-1))));
    Xw(regs[2]) = Tmp;
    return TRUE;
}

static BOOL luther_inline_unify(InArg)
    InArgdecl;
{
    TAGGED Xd,Yd;

    DerefNLL(Xd,Xw(regs[0]));
    DerefNLL(Yd,Xw(regs[1]));

    return unify(Xd,Yd,w);
}

static BOOL luther_array_in_elt(InArg)
    InArgdecl;
{
    TAGGED X0, X1, X2, Tmp;
    register int i;

    DerefNLL(X0,Xw(regs[0])); /* ArgNo */
    DerefNLL(X1,Xw(regs[1])); /* Array */
    DerefNLL(X2,Xw(regs[2])); /* Arg */

    if((!IsNUM(X0)) || (!IsArray(X1)))
	return FALSE;
    
    i = GetNumber(X0);

    if ((i < 0) || (i >= GetNumber(GetArraySize(X1)))) 
      return FALSE;

    DerefNLL(Tmp,Ref(&GetArrayArg(X1,i)));
    return unify(Tmp,X2,w);
}

static BOOL luther_array_var_elt(InArg)
    InArgdecl;
{
    TAGGED X0, X1, X2, Tmp;
    register int i;

    DerefNLL(X0,Xw(regs[0])); /* Array */
    DerefNLL(X1,Xw(regs[1])); /* ArgNo */
    DerefNLL(X2,Xw(regs[2])); /* Arg */

    if((!IsNUM(X1)) || (!IsArray(X0)))
	return FALSE;
    
    i = GetNumber(X1);

    if ((i < 0) || (i >= GetNumber(GetArraySize(X0)))) 
      return FALSE;

    GetArrayArg(X0,i) = X2;

    return TRUE;
}

static BOOL luther_array_in_ref(InArg)
    InArgdecl;
{
    TAGGED X0, X1, X2, Tmp;
    register int i;

    DerefNLL(X1,Xw(regs[1])); /* Array */
    DerefNLL(X2,Xw(regs[2])); /* ArgNo */

    if((!IsNUM(X2)) || (!IsArray(X1)))
	return FALSE;
    
    i = GetNumber(X2);

    if ((i < 0) || (i >= GetNumber(GetArraySize(X1)))) 
      return FALSE;

    Xw(regs[0]) = Ref(&GetArrayArg(X1,i));
    return TRUE;
}

static BOOL luther_array_in_size(InArg)
    InArgdecl;
{
  TAGGED Array, Size;
  
  DerefNLL(Array,Xw(regs[0])); 
  DerefNLL(Size,Xw(regs[1])); 
  
  if(IsVar(Array))
    {
      if(IsNUM(Size) && (GetNumber(Size) > 0))
	{
	  register generic new;
	  register int i;
	  register TAGGED *ht;
	  
	  new = Generic(w->heap_top);
	  new->method = &array_method_table;
	  new->data[0] = Size;
	  
	  w->heap_top += ARRAY_BASE_SIZE;
	  
	  i = GetNumber(Size);
	  ht = w->heap_top;
	  while(i--) {
	    CreateHVA(ht,w);
	  }
	  w->heap_top = ht;
	  return unify(Array,Tagify(new,GEN),w);
	}
      else
	{
	  return FALSE;
	}
    }
  else if (IsArray(Array))
    {
      return unify(Size,GetArraySize(Array),w);
    }
  else
    {
      return FALSE;
    }
}    

static BOOL luther_array_var_size(InArg)
    InArgdecl;
{
  TAGGED Array, Size;
  
  DerefNLL(Size,Xw(regs[1])); 
  
  if(IsNUM(Size) && (GetNumber(Size) > 0))
    {
      register generic new;
      register int i;
      register TAGGED *ht;
      
      new = Generic(w->heap_top);
      new->method = &array_method_table;
      new->data[0] = Size;
      
      i = GetNumber(Size);

      w->heap_top += ARRAY_BASE_SIZE + i * VARSIZE;
      Xw(regs[0]) = Tagify(new,GEN);
      return TRUE;
    }
  else
    {
      return FALSE;
    }
}    

static BOOL luther_reduce_vector_plus(InArg)
    InArgdecl;
{
  register TAGGED Vector, Size, *V, Num;
  register s32 vectorsize, sum;
  
  DerefNLL(Vector,Xw(regs[1]));
  DerefNLL(Size,Xw(regs[2]));
  
  if(!(IsLST(Vector) && IsNUM(Size))) return FALSE;
  
  V = RemoveTag(Vector,LST);
  
  sum = 0;
  vectorsize = GetNumber(Size);

  while(vectorsize)
    {
      DerefNLL(Num,Ref(V));

      if(!IsNUM(Num)) return FALSE;

      sum += GetNumber(Num);

      V += 2*VARSIZE;
      vectorsize--;
    }
  
  Xw(regs[0]) = Make_Integer(sum);
  return TRUE;
}

static BOOL luther_reduce_vector_times(InArg)
    InArgdecl;
{
  register TAGGED Vector, Size, *V, Num;
  register s32 vectorsize, prod;
  
  DerefNLL(Vector,Xw(regs[1]));
  DerefNLL(Size,Xw(regs[2]));
  
  if(!(IsLST(Vector) && IsNUM(Size))) return FALSE;
  
  V = RemoveTag(Vector,LST);
  
  prod = 1;
  vectorsize = GetNumber(Size);

  while(vectorsize)
    {
      DerefNLL(Num,Ref(V));

      if(!IsNUM(Num)) return FALSE;

      prod *= GetNumber(Num);

      V += 2*VARSIZE;
      vectorsize--;
    }
  
  Xw(regs[0]) = Make_Integer(prod);
  return TRUE;
}

static BOOL luther_par_reduce_plus(InArg)
    InArgdecl;
{
#ifdef PARALLEL
    TAGGED Result, Array;
    double sum;
    register s32 i;

    DerefNLL(Result,Xw(regs[0])); 
    DerefNLL(Array,Xw(regs[1]));

    if(!IsArray(Array)) return FALSE;

    w->global->global_fail = FALSE;

    w->global->parallel_start.type = W_REDUCE_PLUS;
    w->global->parallel_start.code = (code *) Array;

    ActivateWorkers(w);

    if(w->global->global_fail) return FALSE;

    for(sum = 0.0, i = 0 ; i < w->global->active_workers ; i++)
	sum += w->global->reduction_results[i];

    if(sum == floor(sum))
      {
	return unify(Result, Make_Integer(((int) sum)), w);
      }
    else
      {
	return unify(Result, make_float(w,sum), w);
      }
#else
    return FALSE;
#endif
}

static BOOL luther_active(InArg)
    InArgdecl;
{
#ifdef PARALLEL
  Xw(regs[0]) = Make_Integer(w->global->active_workers);
  return TRUE;
#else
  return FALSE;
#endif /* PARALLEL */
}

static BOOL luther_save(InArg)
    InArgdecl;
{
#ifdef PARALLEL
  register TAGGED saved;
  DerefNLL(saved,Xw(regs[0]));
  w->global->collect[w->pid-1] = saved;
  return TRUE;
#else
  return FALSE;
#endif /* PARALLEL */
}

static BOOL luther_collect_plus(InArg)
    InArgdecl;
{
#ifdef PARALLEL
  register TAGGED saved, Result;
  register double sum;
  register s32 i;

  for(sum = 0.0, i = 0 ; i < w->global->active_workers ; i++) 
    {
      saved = w->global->collect[i];
      if(IsNUM(saved))
	{
	  sum += (double) GetNumber(saved);
	}
      else if (IsFLT(saved))
	{
	  sum += GetFloat(saved);
	}
      else return FALSE;
    }

  DerefNLL(Result, Xw(regs[0]));
  
  if(sum == floor(sum))
    {
      return unify(Result, Make_Integer(((int) sum)), w);
    }
  else
    {
      return unify(Result, make_float(w,sum), w);
    }
#else
  return FALSE;
#endif /* PARALLEL */
}

static BOOL luther_collect_times(InArg)
    InArgdecl;
{
#ifdef PARALLEL
  register TAGGED saved, Result;
  register double sum;
  register s32 i;

  for(sum = 1.0, i = 0 ; i < w->global->active_workers ; i++) 
    {
      saved = w->global->collect[i];
      if(IsNUM(saved))
	{
	  sum *= (double) GetNumber(saved);
	}
      else if (IsFLT(saved))
	{
	  sum *= GetFloat(saved);
	}
      else return FALSE;
    }

  DerefNLL(Result, Xw(regs[0]));
  
  if(sum == floor(sum))
    {
      return unify(Result, Make_Integer(((int) sum)), w);
    }
  else
    {
      return unify(Result, make_float(w,sum), w);
    }
#else
  return FALSE;
#endif /* PARALLEL */
}


static BOOL luther_par_reduce_times(InArg)
    InArgdecl;
{
#ifdef PARALLEL
    TAGGED Result, Array;
    double sum;
    register s32 i;

    DerefNLL(Result,Xw(regs[0])); 
    DerefNLL(Array,Xw(regs[1]));

    if(!IsArray(Array)) return FALSE;

    w->global->global_fail = FALSE;

    w->global->parallel_start.type = W_REDUCE_TIMES;
    w->global->parallel_start.code = (code *) Array;

    ActivateWorkers(w);

    if(w->global->global_fail) return FALSE;

    for(sum = 1.0, i = 0 ; i < w->global->active_workers ; i++)
	sum *= w->global->reduction_results[i];

    if(sum == floor(sum))
      {
	return unify(Result, Make_Integer(((int) sum)), w);
      }
    else
      {
	return unify(Result, make_float(w,sum), w);
      }
#else
    return FALSE;
#endif
}

#define MathComp(Op) {                                          \
    TAGGED X0, X1;                                              \
                                                                \
    DerefNLL(X0,Xw(regs[0])); /* Expr1 */                       \
    DerefNLL(X1,Xw(regs[1])); /* Expr2 */                       \
                                                                \
 start:                                                         \
                                                                \
    if(IsFLT(X0)) {                                             \
	if(IsFLT(X1))                                           \
	    if(GetFloat(X0) Op GetFloat(X1))                    \
		return TRUE;                                    \
	    else                                                \
		return FALSE;                                   \
	if(IsNUM(X1))                                           \
	    if(GetFloat(X0) Op GetNumber(X1))                   \
		return TRUE;                                    \
	    else                                                \
		return FALSE;                                   \
	goto eval_second;                                       \
    }                                                           \
    if(IsNUM(X0)) {                                             \
	if(IsFLT(X1))                                           \
	    if(GetNumber(X0) Op GetFloat(X1))                   \
		return TRUE;                                    \
	    else                                                \
		return FALSE;                                   \
	if(IsNUM(X1))                                           \
	    if(GetNumber(X0) Op GetNumber(X1))                  \
		return TRUE;                                    \
	    else                                                \
		return FALSE;                                   \
	goto eval_second;                                       \
    }                                                           \
                                                                \
    if(IsSTR(X0)) {                                             \
	TAGGED a[2], *save;                                     \
	s32    r[2];                                            \
	save = w->regs;                                         \
	w->regs = a;                                            \
	a[1] = X0; r[0] = 0; r[1] = 1;                          \
	if(luther_eval_math(w,r)) {                             \
	    X0 = a[0];                                          \
	    w->regs = save;                                     \
	    goto start;                                         \
	} else {                                                \
            w->regs = save;                                     \
	    return FALSE;                                       \
	}                                                       \
    }                                                           \
    goto barf1;                                                 \
                                                                \
 eval_second:                                                   \
    if(IsSTR(X1)) {                                             \
	TAGGED a[2], *save;                                     \
	s32    r[2];                                            \
	save = w->regs;                                         \
        w->regs = a;                                            \
	a[1] = X1; r[0] = 0; r[1] = 1;                          \
	if(luther_eval_math(w,r)) {                             \
	    X1 = a[0];                                          \
            w->regs = save;                                     \
	    goto start;                                         \
	} else {                                                \
	    w->regs = save;                                     \
	    return FALSE;                                       \
	}                                                       \
    }                                                           \
    goto barf1;                                                 \
                                                                \
 barf1:                                                         \
    luther_error(E_ILLEGAL_AR_EX,0,w);                          \
    return FALSE;                                               \
}   

/* $eq */
static BOOL luther_math_eq(InArg)
    InArgdecl;
{
    MathComp(==);
}

/* $ineq */
static BOOL luther_math_ineq(InArg)
    InArgdecl;
{
    MathComp(!=);
}

static BOOL luther_lt(InArg)
    InArgdecl;
{
    MathComp(<);
}

/* > */

static BOOL luther_gt(InArg)
    InArgdecl;
{
    MathComp(>);
}

/* <= */

static BOOL luther_le(InArg)
    InArgdecl;
{
    MathComp(<=);
}

/* >= */

static BOOL luther_ge(InArg)
    InArgdecl;
{
    MathComp(>=);
}



/* $atom/1 */

static BOOL luther_atom(InArg)
    InArgdecl;
{
    TAGGED X0; /* X */

    DerefNLL(X0,Xw(regs[0]));

    if(IsATM(X0))
	return TRUE;
    else
	return FALSE;
}

/* $atomic/1 */

static BOOL luther_atomic(InArg)
    InArgdecl;
{
    TAGGED X0;

    DerefNLL(X0,Xw(regs[0]));

    if(IsATM(X0) || IsNumber(X0))
	return TRUE;
    else
	return FALSE;
}

/* $generic/1 */

static BOOL luther_generic(InArg)
    InArgdecl;
{
    TAGGED X0;

    DerefNLL(X0,Xw(regs[0]));

    if(IsGEN(X0))
	return TRUE;
    else
	return FALSE;
}

/* $integer/1 */

static BOOL luther_integer(InArg)
    InArgdecl;
{
    TAGGED X0;

    DerefNLL(X0,Xw(regs[0])); /* X */

    if(IsNUM(X0))
	return TRUE;
    else
	return FALSE;
}

/* $float/1 */

static BOOL luther_float(InArg)
    InArgdecl;
{
    TAGGED X0;

    DerefNLL(X0,Xw(regs[0])); /* X */

    if(IsFLT(X0))
	return TRUE;
    else
	return FALSE;
}

/* $number/1 */

static BOOL luther_number(InArg)
    InArgdecl;
{
    TAGGED X0;

    DerefNLL(X0,Xw(regs[0])); /* X */

    if(IsNumber(X0))
	return TRUE;
    else
	return FALSE;
}

/* $nonvar/1 */

static BOOL luther_nonvar(InArg)
    InArgdecl;
{
    TAGGED X0;

    DerefNLL(X0,Xw(regs[0])); /* X */

    if(!IsVar(X0))
	return TRUE;
    else
	return FALSE;
}

/* $var/1 */

static BOOL luther_var(InArg)
    InArgdecl;
{
    TAGGED X0;

    DerefNLL(X0,Xw(regs[0])); /* X */

    if(IsVar(X0))
	return TRUE;
    else
	return FALSE;
}

#define MathOp(Op) {\
    register TAGGED X1,X2;\
\
    DerefNLL(X1,Xw(regs[1])); /* Expr1 */\
    DerefNLL(X2,Xw(regs[2])); /* Expr2 */\
\
 start:\
\
    if(IsNUM(X1)) {\
	if(IsNUM(X2)) {\
	    Xw(regs[0]) = Make_Integer(GetNumber(X1) Op GetNumber(X2));\
	    return TRUE;\
	}\
	if(IsFLT(X2)) {\
	    Xw(regs[0]) = make_float(w,GetNumber(X1) Op GetFloat(X2));\
	    return TRUE;\
	}\
	goto eval_second;\
    }\
\
    if(IsFLT(X1)) {\
	if(IsFLT(X2)) {\
	    Xw(regs[0]) = make_float(w,GetFloat(X1) Op GetFloat(X2));\
	    return TRUE;\
	}\
	if(IsNUM(X2)) {\
	    Xw(regs[0]) = make_float(w,GetFloat(X1) Op GetNumber(X2));\
	    return TRUE;\
	}\
	goto eval_second;\
    }\
\
 eval_first:\
\
    if(IsSTR(X1)) {\
	if(luther_eval_math(w,regs)) {\
	    X1 = Xw(regs[0]);\
	    goto start;\
	} else\
	    return FALSE;\
    }\
    goto error;\
\
    eval_second:\
\
    if(IsSTR(X2)) {\
	regs[1] = regs[2];\
	if(luther_eval_math(w,regs)) {\
	    X2 = Xw(regs[0]);\
	    goto start;\
	} else\
	    return FALSE;\
    }\
\
 error:\
    \
    luther_error(E_ILLEGAL_AR_EX,0,w);\
    return FALSE;\
}

#define MathBinOp(Op) {\
    register TAGGED X1,X2;\
\
    DerefNLL(X1,Xw(regs[1])); /* Expr1 */\
    DerefNLL(X2,Xw(regs[2])); /* Expr2 */\
\
 start:\
\
    if(IsNUM(X1)) {\
	if(IsNUM(X2)) {\
	    Xw(regs[0]) = Make_Integer(GetNumber(X1) Op GetNumber(X2));\
	    return TRUE;\
	}\
	if(IsFLT(X2)) {\
	    Xw(regs[0]) = Make_Integer(GetNumber(X1) Op ((int) GetFloat(X2)));\
	    return TRUE;\
	}\
	goto eval_second;\
    }\
\
    if(IsFLT(X1)) {\
	if(IsFLT(X2)) {\
	    Xw(regs[0]) = Make_Integer(((int) GetFloat(X1)) Op\
				      ((int) GetFloat(X2)));\
	    return TRUE;\
	}\
	if(IsNUM(X2)) {\
	    Xw(regs[0]) = Make_Integer(((int) GetFloat(X1)) Op\
				      ((int) GetNumber(X2)));\
	    return TRUE;\
	}\
	goto eval_second;\
    }\
\
 eval_first:\
\
    if(IsSTR(X1)) {\
	if(luther_eval_math(w,regs)) {\
	    X1 = Xw(regs[0]);\
	    goto start;\
	} else\
	    return FALSE;\
    }\
    goto error;\
\
    eval_second:\
\
    if(IsSTR(X2)) {\
	regs[1] = regs[2];\
	if(luther_eval_math(w,regs)) {\
	    X2 = Xw(regs[0]);\
	    goto start;\
	} else\
	    return FALSE;\
    }\
\
 error:\
    \
    luther_error(E_ILLEGAL_AR_EX,0,w);\
    return FALSE;\
}

#define MathOp1(Op) {\
    register TAGGED X1;\
\
    DerefNLL(X1,Xw(regs[1])); /* Expr */\
\
    start:\
\
    if(IsNUM(X1)) {\
	Xw(regs[0]) = Make_Integer(GetNumber(X1) Op 1);\
	return TRUE;\
    }\
    if(IsFLT(X1)) {\
	Xw(regs[0]) = make_float(w,GetFloat(X1) Op 1.0);\
	return TRUE;\
    }\
    if(IsSTR(X1)) {\
      if(luther_eval_math(w,regs)) {\
	X1 = Xw(regs[0]);\
	goto start;\
      } else\
	return FALSE;\
    }\
    luther_error(E_ILLEGAL_AR_EX,0,w);\
    return FALSE;\
}

/* $plus/3 */

static BOOL luther_plus(InArg)
    InArgdecl;
{
    MathOp(+);
}

static BOOL luther_minus(InArg)
    InArgdecl;
{
    MathOp(-);
}

static BOOL luther_times(InArg)
    InArgdecl;
{
    MathOp(*);
}

static BOOL luther_bin_or(InArg)
    InArgdecl;
{
    MathBinOp(|);
}

static BOOL luther_bin_and(InArg)
    InArgdecl;
{
    MathBinOp(&);
}

static BOOL luther_bin_xor(InArg)
    InArgdecl;
{
    MathBinOp(^);
}


static BOOL luther_plus_1(InArg)
    InArgdecl;
{
    MathOp1(+);
}
static BOOL luther_minus_1(InArg)
    InArgdecl;
{
    MathOp1(-);
}

static BOOL luther_div(InArg)
    InArgdecl;
{
    register TAGGED X1,X2;

    DerefNLL(X1,Xw(regs[1])); /* Expr1 */
    DerefNLL(X2,Xw(regs[2])); /* Expr2 */

    start:

    if(IsNUM(X1)) {
	if(IsNUM(X2)) {
	    if(GetNumber(X2) == 0) {
		goto error_zero;
	    }
	    Xw(regs[0]) = make_float(w,((double) GetNumber(X1) / GetNumber(X2)));
	    return TRUE;
	}
	if(IsFLT(X2)) {
	    if(GetFloat(X2) == 0.0) {
		goto error_zero;
	    }
	    Xw(regs[0]) = make_float(w,GetNumber(X1) / GetFloat(X2));
	    return TRUE;
	}
	goto eval_second;
    }
    if(IsFLT(X1)) {
        if(IsFLT(X2)) {
	    if(GetFloat(X2) == 0.0) {
		goto error_zero;
	    }
	    Xw(regs[0]) = make_float(w,GetFloat(X1) / GetFloat(X2));
	    return TRUE;
	}
	if(IsNUM(X2)) {
	    if(GetNumber(X2) == 0) {
		goto error_zero;
	    }
	    Xw(regs[0]) = make_float(w,GetFloat(X1) / GetNumber(X2));
	    return TRUE;
	}
	goto eval_second;
    }

 eval_first:

    if(IsSTR(X1)) {
      if(luther_eval_math(w,regs)) {
	X1 = Xw(regs[0]);
	goto start;
      } else
	return FALSE;
    }
    goto error;
    
 eval_second:

    if(IsSTR(X2)) {
	regs[1] = regs[2];
	if(luther_eval_math(w,regs)) {
	    X2 = Xw(regs[0]);
	    goto start;
	} else
	    return FALSE;
    }

 error:

    luther_error(E_ILLEGAL_AR_EX,0,w);
    return FALSE;

 error_zero:

    luther_error(E_DIV_ZERO,0,w);
    return FALSE;
}

static BOOL luther_intdiv(InArg)
    InArgdecl;
{
    register TAGGED X1,X2;
    
    DerefNLL(X1,Xw(regs[1])); /* Expr1 */
    DerefNLL(X2,Xw(regs[2])); /* Expr2 */
    
 start:
    
    if(IsNUM(X1)) {
	if(IsNUM(X2)) {
	    if(GetNumber(X2) == 0) {
		goto error_zero;
	    }
	    Xw(regs[0]) = Make_Integer(((s32) (GetNumber(X1) / GetNumber(X2))));
	    return TRUE;
	}
	if(IsFLT(X2)) {
	    if(((s32) GetFloat(X2)) == 0 ) {
		goto error_zero;
	    }
	    Xw(regs[0]) = Make_Integer(((s32) (GetNumber(X1) /
					      ((s32) GetFloat(X2)))));
	    return TRUE;
	}
	goto eval_second;
    }

    if(IsFLT(X1)) {
        if(IsFLT(X2)) {
	    if(((s32) GetFloat(X2)) == 0 ) {
		goto error_zero;
	    }
	    Xw(regs[0]) = Make_Integer(((s32) (GetFloat(X1) /
					      ((s32) GetFloat(X2)))));
	    return TRUE;
	}
	if(IsNUM(X2)) {
	    if(GetNumber(X2) == 0) {
		goto error_zero;
	    }
	    Xw(regs[0]) = Make_Integer(((s32) (GetFloat(X1) / GetNumber(X2))));
	    return TRUE;
	}
	goto eval_second;
    }

 eval_first:
    if(IsSTR(X1)) {
	if(luther_eval_math(w,regs)) {
	    X1 = Xw(regs[0]);
	    goto start;
	} else
	    return FALSE;
    }
    goto error;
    
 eval_second:
    if(IsSTR(X2)) {
	regs[1] = regs[2];
	if(luther_eval_math(w,regs)) {
	    X2 = Xw(regs[0]);
	    goto start;
	} else
	    return FALSE;
    }
    
 error:
    luther_error(E_ILLEGAL_AR_EX,0,w);
    return FALSE;

 error_zero:
    luther_error(E_DIV_ZERO,0,w);
    return FALSE;
}


static BOOL luther_mod(InArg)
    InArgdecl;
{
    register TAGGED X1,X2;
    
    DerefNLL(X1,Xw(regs[1])); /* Expr1 */
    DerefNLL(X2,Xw(regs[2])); /* Expr2 */
         
 start:
    
    if(IsNUM(X1)) {
	if(IsNUM(X2)) {
	    Xw(regs[0]) = Make_Integer(((s32) (GetNumber(X1) % GetNumber(X2))));
	    return TRUE;
	}
	if(IsFLT(X2)) {
	    Xw(regs[0]) = Make_Integer(((s32) (GetNumber(X1) %
					      ((s32) GetFloat(X2)))));
	    return TRUE;
	}
	goto eval_second;
    }
    if(IsFLT(X1)) {
        if(IsFLT(X2)) {
	    Xw(regs[0]) = Make_Integer(((s32) (((s32) GetFloat(X1)) %
					      ((s32) GetFloat(X2)))));
	    return TRUE;
	}
	if(IsNUM(X2)) {
	    Xw(regs[0]) = Make_Integer(((s32) (((s32) GetFloat(X1)) %
					      GetNumber(X2))));
	    return TRUE;
	}
	goto eval_second;
    }
    
 eval_first:
    if(IsSTR(X1)) {
	if(luther_eval_math(w,regs)) {
	    X1 = Xw(regs[0]);
	    goto start;
	} else
	    return FALSE;
    }
    goto error;
    
 eval_second:
    if(IsSTR(X2)) {
	regs[1] = regs[2];
	if(luther_eval_math(w,regs)) {
	    X2 = Xw(regs[0]);
	    goto start;
	} else
	    return FALSE;
    }
    
 error:
    luther_error(E_ILLEGAL_AR_EX,0,w);
    return FALSE;
}

static BOOL luther_rshift(InArg)
    InArgdecl;
{
    register TAGGED X1,X2;
    
    DerefNLL(X1,Xw(regs[1])); /* Expr1 */
    DerefNLL(X2,Xw(regs[2])); /* Expr2 */
         
 start:
    
    if(IsNUM(X1)) {
	if(IsNUM(X2)) {
	    Xw(regs[0]) = Make_Integer(((s32) (GetNumber(X1) >> GetNumber(X2))));
	    return TRUE;
	}
	if(IsFLT(X2)) {
	    Xw(regs[0]) = Make_Integer(((s32) (GetNumber(X1) >>
					      ((s32) GetFloat(X2)))));
	    return TRUE;
	}
	goto eval_second;
    }
    if(IsFLT(X1)) {
        if(IsFLT(X2)) {
	    Xw(regs[0]) = Make_Integer(((s32) (((s32) GetFloat(X1)) >>
					      ((s32) GetFloat(X2)))));
	    return TRUE;
	}
	if(IsNUM(X2)) {
	    Xw(regs[0]) = Make_Integer(((s32) (((s32) GetFloat(X1)) >>
					      GetNumber(X2))));
	    return TRUE;
	}
	goto eval_second;
    }
    
 eval_first:
    if(IsSTR(X1)) {
	if(luther_eval_math(w,regs)) {
	    X1 = Xw(regs[0]);
	    goto start;
	} else
	    return FALSE;
    }
    goto error;
    
 eval_second:
    if(IsSTR(X2)) {
	regs[1] = regs[2];
	if(luther_eval_math(w,regs)) {
	    X2 = Xw(regs[0]);
	    goto start;
	} else
	    return FALSE;
    }
    
 error:
    luther_error(E_ILLEGAL_AR_EX,0,w);
    return FALSE;
}

static BOOL luther_lshift(InArg)
    InArgdecl;
{
    register TAGGED X1,X2;
    
    DerefNLL(X1,Xw(regs[1])); /* Expr1 */
    DerefNLL(X2,Xw(regs[2])); /* Expr2 */
         
 start:
    
    if(IsNUM(X1)) {
	if(IsNUM(X2)) {
	    Xw(regs[0]) = Make_Integer(((s32) (GetNumber(X1) << GetNumber(X2))));
	    return TRUE;
	}
	if(IsFLT(X2)) {
	    Xw(regs[0]) = Make_Integer(((s32) (GetNumber(X1) <<
					      ((s32) GetFloat(X2)))));
	    return TRUE;
	}
	goto eval_second;
    }
    if(IsFLT(X1)) {
        if(IsFLT(X2)) {
	    Xw(regs[0]) = Make_Integer(((s32) (((s32) GetFloat(X1)) <<
					      ((s32) GetFloat(X2)))));
	    return TRUE;
	}
	if(IsNUM(X2)) {
	    Xw(regs[0]) = Make_Integer(((s32) (((s32) GetFloat(X1)) <<
					      GetNumber(X2))));
	    return TRUE;
	}
	goto eval_second;
    }
    
 eval_first:
    if(IsSTR(X1)) {
	if(luther_eval_math(w,regs)) {
	    X1 = Xw(regs[0]);
	    goto start;
	} else
	    return FALSE;
    }
    goto error;
    
 eval_second:
    if(IsSTR(X2)) {
	regs[1] = regs[2];
	if(luther_eval_math(w,regs)) {
	    X2 = Xw(regs[0]);
	    goto start;
	} else
	    return FALSE;
    }
    
 error:
    luther_error(E_ILLEGAL_AR_EX,0,w);
    return FALSE;
}

static BOOL luther_pow(InArg)
    InArgdecl;
{
    register TAGGED X1,X2;
    
    DerefNLL(X1,Xw(regs[1])); /* Expr1 */
    DerefNLL(X2,Xw(regs[2])); /* Expr2 */
         
 start:
    
    if(IsNUM(X1)) {
	if(IsNUM(X2)) {
	    Xw(regs[0]) = Make_Integer(((s32)(pow((double) GetNumber(X1),
						 (double) GetNumber(X2))
					      + 0.5)));
	    return TRUE;
	}
	if(IsFLT(X2)) {
	    Xw(regs[0]) = make_float(w,pow((double) GetNumber(X1),
					 GetFloat(X2)));
	    return TRUE;
	}
	goto eval_second;
    }
    if(IsFLT(X1)) {
        if(IsFLT(X2)) {
	    Xw(regs[0]) = make_float(w,pow(GetFloat(X1),GetFloat(X2)));
	    return TRUE;
	}
	if(IsNUM(X2)) {
	    Xw(regs[0]) = make_float(w,pow(GetFloat(X1),
					   (double)GetNumber(X2)));
	    return TRUE;
	}
	goto eval_second;
    }
    
 eval_first:
    if(IsSTR(X1)) {
	if(luther_eval_math(w,regs)) {
	    X1 = Xw(regs[0]);
	    goto start;
	} else
	    return FALSE;
    }
    goto error;
    
 eval_second:
    if(IsSTR(X2)) {
	regs[1] = regs[2];
	if(luther_eval_math(w,regs)) {
	    X2 = Xw(regs[0]);
	    goto start;
	} else
	    return FALSE;
    }
    
 error:
    luther_error(E_ILLEGAL_AR_EX,0,w);
    return FALSE;
}

static BOOL luther_unary_minus(InArg)
    InArgdecl;
{
    TAGGED X1;

    DerefNLL(X1,Xw(regs[1])); /* Expr */

    start:

    if(IsFLT(X1)) {
	Xw(regs[0]) = make_float(w,GetFloat(X1)*-1.0);
	return TRUE;
    }
    if(IsNUM(X1)) {
	Xw(regs[0]) = Make_Integer(GetNumber(X1) * -1);
	return TRUE;
    }
    if(IsSTR(X1)) {
      if(luther_eval_math(w,regs)) {
	X1 = Xw(regs[0]);
	goto start;
      } else
	return FALSE;
    }

    luther_error(E_ILLEGAL_AR_EX,0,w);
    return FALSE;
}

static BOOL luther_tointeger(InArg)
    InArgdecl;
{
    TAGGED X1;

    DerefNLL(X1,Xw(regs[1])); /* Expr */

    start:

    if(IsFLT(X1)) {
	Xw(regs[0]) = Make_Integer((s32) GetFloat(X1));
	return TRUE;
    }
    if(IsNUM(X1)) {
	Xw(regs[0]) = X1;
	return TRUE;
    }
    if(IsSTR(X1)) {
      if(luther_eval_math(w,regs)) {
	X1 = Xw(regs[0]);
	goto start;
      } else
	return FALSE;
    }

    luther_error(E_ILLEGAL_AR_EX,0,w);
    return FALSE;
}

static BOOL luther_floor(InArg)
    InArgdecl;
{
    TAGGED X1;

    DerefNLL(X1,Xw(regs[1])); /* Expr */

    start:

    if(IsFLT(X1)) {
	Xw(regs[0]) = Make_Integer((s32) floor(GetFloat(X1)));
	return TRUE;
    }
    if(IsNUM(X1)) {
	Xw(regs[0]) = X1;
	return TRUE;
    }
    if(IsSTR(X1)) {
      if(luther_eval_math(w,regs)) {
	X1 = Xw(regs[0]);
	goto start;
      } else
	return FALSE;
    }

    luther_error(E_ILLEGAL_AR_EX,0,w);
    return FALSE;
}

static BOOL luther_ceil(InArg)
    InArgdecl;
{
    TAGGED X1;

    DerefNLL(X1,Xw(regs[1])); /* Expr */

    start:

    if(IsFLT(X1)) {
	Xw(regs[0]) = Make_Integer((s32) ceil(GetFloat(X1)));
	return TRUE;
    }
    if(IsNUM(X1)) {
	Xw(regs[0]) = X1;
	return TRUE;
    }
    if(IsSTR(X1)) {
      if(luther_eval_math(w,regs)) {
	X1 = Xw(regs[0]);
	goto start;
      } else
	return FALSE;
    }

    luther_error(E_ILLEGAL_AR_EX,0,w);
    return FALSE;
}

static BOOL luther_tofloat(InArg)
    InArgdecl;
{
    TAGGED X1;

    DerefNLL(X1,Xw(regs[1])); /* Expr */
    
    start:

    if(IsNUM(X1)) {
      Xw(regs[0]) = make_float(w,(double) GetNumber(X1));
      return TRUE;
    }
    if(IsFLT(X1)) {
	Xw(regs[0]) = X1;
	return TRUE;
    }
    if(IsSTR(X1)) {
      if(luther_eval_math(w,regs)) {
	X1 = Xw(regs[0]);
	goto start;
      } else
	return FALSE;
    }

    luther_error(E_ILLEGAL_AR_EX,0,w);
    return FALSE;
}

#define EvalBinary(Fun) {\
    TAGGED *save;\
    save = w->regs;\
    w->regs = newarg;\
    newarg[1] = Ref(GetArg(X1,0));\
    newarg[2] = Ref(GetArg(X1,1));\
    newregs[0] = 0; newregs[1] = 1; newregs[2] = 2;\
    if(Fun(w,newregs)) {\
        w->regs = save;\
	Xw(regs[0]) = newarg[0];\
	return TRUE;\
    } else {\
        w->regs = save;\
    }\
    return FALSE;\
}

#define EvalUnary(Fun) {\
    TAGGED *save;\
    save = w->regs;\
    w->regs = newarg;\
    newarg[1] = Ref(GetArg(X1,0));\
    newregs[0] = 0; newregs[1] = 1;\
    if(Fun(w,newregs)) {\
        w->regs = save;\
	Xw(regs[0]) = newarg[0];\
	return TRUE;\
    } else {\
        w->regs = save;\
    }\
    return FALSE;\
}

#define MathUnOp(Op) {\
    register TAGGED X1;\
\
    DerefNLL(X1,Xw(regs[1])); /* Expr */\
\
    start:\
\
    if(IsNUM(X1)) {\
	Xw(regs[0]) = make_float(w,Op((double) GetNumber(X1)));\
	return TRUE;\
    }\
    if(IsFLT(X1)) {\
	Xw(regs[0]) = make_float(w,Op((double) GetFloat(X1)));\
	return TRUE;\
    }\
    if(IsSTR(X1)) {\
      if(luther_eval_math(w,regs)) {\
	X1 = Xw(regs[0]);\
	goto start;\
      } else\
	return FALSE;\
    }\
    luther_error(E_ILLEGAL_AR_EX,0,w);\
    return FALSE;\
}

static BOOL luther_exp(InArg)
    InArgdecl;
{
  MathUnOp(exp);
}

static BOOL luther_log(InArg)
    InArgdecl;
{
  MathUnOp(log);
}

static BOOL luther_sin(InArg)
    InArgdecl;
{
  MathUnOp(sin);
}

static BOOL luther_cos(InArg)
    InArgdecl;
{
  MathUnOp(cos);
}

static BOOL luther_tan(InArg)
    InArgdecl;
{
  MathUnOp(tan);
}

static BOOL luther_asin(InArg)
    InArgdecl;
{
  MathUnOp(asin);
}

static BOOL luther_atan(InArg)
    InArgdecl;
{
  MathUnOp(atan);
}

static BOOL luther_acos(InArg)
    InArgdecl;
{
  MathUnOp(acos);
}

#define MathUnOp_Int_Flt(IOp,FOp) {\
    register TAGGED X1;\
\
    DerefNLL(X1,Xw(regs[1])); /* Expr */\
\
    start:\
\
    if(IsNUM(X1)) {\
	Xw(regs[0]) = Make_Integer(IOp(GetNumber(X1)));\
	return TRUE;\
    }\
    if(IsFLT(X1)) {\
	Xw(regs[0]) = make_float(w,FOp((double) GetFloat(X1)));\
	return TRUE;\
    }\
    if(IsSTR(X1)) {\
      if(luther_eval_math(w,regs)) {\
	X1 = Xw(regs[0]);\
	goto start;\
      } else\
	return FALSE;\
    }\
    luther_error(E_ILLEGAL_AR_EX,0,w);\
    return FALSE;\
}

static BOOL luther_abs(InArg)
    InArgdecl;
{
  MathUnOp_Int_Flt(abs,fabs);
}

static BOOL luther_aref(InArg)
    InArgdecl;
{
    TAGGED X1,X2,tmp;

    DerefNLL(X1,Xw(regs[1])); /* Array */
    DerefNLL(X2,Xw(regs[2])); /* Index */
    
    if(!IsArray(X1)) goto barf;

  start:

    if(IsNUM(X2)) {
      register int i = GetNumber(X2);
      if((i < 0) || (i >= GetNumber(GetArraySize(X1)))) return FALSE;
      DerefNLL(tmp,Ref(&GetArrayArg(X1,i)));
      if(IsNUM(tmp)) {
	Xw(regs[0]) = tmp;
	return TRUE;
      } else {
	Xw(regs[1]) = tmp;
	return luther_eval_math(w,regs);
      }
    }
    if(IsSTR(X2)) {
      Xw(regs[1]) = X2;
      if(luther_eval_math(w,regs)) {
	X2 = Xw(regs[0]);
	goto start;
      } else
	return FALSE;
    }

  barf:
    luther_error(E_ILLEGAL_AR_EX,0,w);
    return FALSE;
}


static BOOL luther_eval_math(InArg)
    InArgdecl;
{
    TAGGED X1, newarg[3], Operator;
    s32 newregs[3];

    DerefNLL(X1,Xw(regs[1])); /* Expr */

    if(IsSTR(X1)) {
	Operator = GetFunctor(X1);

	if(Operator == functor_plus)    EvalBinary(luther_plus);
	if(Operator == functor_minus)   EvalBinary(luther_minus);
	if(Operator == functor_times)   EvalBinary(luther_times);
	if(Operator == functor_div)     EvalBinary(luther_div);
	if(Operator == functor_intdiv)  EvalBinary(luther_intdiv);
	if(Operator == functor_mod)     EvalBinary(luther_mod);
	if(Operator == functor_rshift)  EvalBinary(luther_rshift);
	if(Operator == functor_lshift)  EvalBinary(luther_lshift);

	if(Operator == functor_exp)     EvalUnary(luther_exp);
	if(Operator == functor_log)     EvalUnary(luther_log);
	if(Operator == functor_sin)     EvalUnary(luther_sin);
	if(Operator == functor_cos)     EvalUnary(luther_cos);
	if(Operator == functor_tan)     EvalUnary(luther_tan);
	if(Operator == functor_asin)    EvalUnary(luther_asin);
	if(Operator == functor_acos)    EvalUnary(luther_acos);
	if(Operator == functor_atan)    EvalUnary(luther_atan);
	if(Operator == functor_abs)     EvalUnary(luther_abs);

	if(Operator == functor_bin_or)  EvalBinary(luther_bin_or);
	if(Operator == functor_bin_and) EvalBinary(luther_bin_and);
	if(Operator == functor_bin_xor) EvalBinary(luther_bin_xor);
	if(Operator == functor_pow)     EvalBinary(luther_pow);

	if(Operator == functor_integer) EvalUnary(luther_tointeger);
	if(Operator == functor_floor)   EvalUnary(luther_floor);
	if(Operator == functor_ceil)    EvalUnary(luther_ceil);
	if(Operator == functor_float)   EvalUnary(luther_tofloat);
	if(Operator == functor_u_minus) EvalUnary(luther_unary_minus);

	if(Operator == functor_aref)    EvalBinary(luther_aref);
    }
    if(IsNumber(X1)) {
	Xw(regs[0]) = X1;
	return TRUE;
    }

    luther_error(E_ILLEGAL_AR_EX,0,w);
    return FALSE;
}

/* is/2 */

BOOL luther_is(InArg)
    InArgdecl;
{
    TAGGED newargs[2], *save, Res;
    s32 newregs[2];

    newargs[1] = Xw(1);
    
    newregs[0] = 0;
    newregs[1] = 1;

    save = w->regs;
    w->regs = newargs;

    if(luther_eval_math(w, newregs)) {
	w->regs = save;
	DerefNLL(Res,Xw(0));
	return unify(newargs[0], Res ,w);
    }

    w->regs = save;
    
    return FALSE;
}

void init_inline()
{
    functor_plus    = StoreFunctor(atom_table_tagged[ATOM_PLUS],2);
    functor_minus   = StoreFunctor(atom_table_tagged[ATOM_MINUS],2);
    functor_u_minus = StoreFunctor(atom_table_tagged[ATOM_MINUS],1);
    functor_times   = StoreFunctor(atom_table_tagged[ATOM_TIMES],2);
    functor_div     = StoreFunctor(atom_table_tagged[ATOM_DIV],2);
    functor_intdiv  = StoreFunctor(atom_table_tagged[ATOM_INTDIV],2);
    functor_mod     = StoreFunctor(atom_table_tagged[ATOM_MOD],2);
    functor_rshift  = StoreFunctor(atom_table_tagged[ATOM_RSHIFT],2);
    functor_lshift  = StoreFunctor(atom_table_tagged[ATOM_LSHIFT],2);
    functor_integer = StoreFunctor(atom_table_tagged[ATOM_INTEGER],1);
    functor_floor   = StoreFunctor(atom_table_tagged[ATOM_FLOOR],1);
    functor_ceil    = StoreFunctor(atom_table_tagged[ATOM_CEIL],1);
    functor_float   = StoreFunctor(atom_table_tagged[ATOM_FLOAT],1);
    functor_abs     = StoreFunctor(atom_table_tagged[ATOM_ABS],1);

    functor_exp     = StoreFunctor(atom_table_tagged[ATOM_EXP],1);
    functor_log     = StoreFunctor(atom_table_tagged[ATOM_LOG],1);
    functor_sin     = StoreFunctor(atom_table_tagged[ATOM_SIN],1);
    functor_cos     = StoreFunctor(atom_table_tagged[ATOM_COS],1);
    functor_tan     = StoreFunctor(atom_table_tagged[ATOM_TAN],1);
    functor_asin    = StoreFunctor(atom_table_tagged[ATOM_ASIN],1);
    functor_acos    = StoreFunctor(atom_table_tagged[ATOM_ACOS],1);
    functor_atan    = StoreFunctor(atom_table_tagged[ATOM_ATAN],1);

    functor_bin_or  = StoreFunctor(atom_table_tagged[ATOM_B_OR],2);
    functor_bin_and = StoreFunctor(atom_table_tagged[ATOM_B_AND],2);
    functor_bin_xor = StoreFunctor(atom_table_tagged[ATOM_B_XOR],2);

    functor_aref    = StoreFunctor(atom_table_tagged[ATOM_AREF],2);
    functor_pow     = StoreFunctor(atom_table_tagged[ATOM_EXP],2);
}





