/*    File:	 gc.c  
 *    Author:	 Johan Bevemyr
 *    Created:	 Mon Oct 21 13:34:44 1991
 *    Purpose:   To garbage collect Luther Prologs heap. This garbage
 *               collector is an implementation of the algorithm described
 *               in "Carbage Collection for Prolog Based on WAM" by
 *               K. Appleby, M. Carlsson, S. Haridi and D. Sahlin,
 *               Communications of the AMC, June 1988, Volume 31, Number 6.
 * 
 *               The algorithm used is of the Mark and Sweep kind.
 */ 

/* 
	Addition for boxed objects:
	Objects such as bit-strings and floating point numbers cannot
	be marked using bits of the value field. Such objects are 
	represented as follows:

        ---------------------------------------------------------
        | ATM | Box Flag |  Object Size   | Stat Flag | GC-bits |     
        ---------------------------------------------------------
        |                   Object                              |
        ---------------------------------------------------------
        | ATM | Box Flag |  Object Size   | Stat Flag | GC-bits |     
        ---------------------------------------------------------

	When marking only the first word of the box is marked.

	The main issues concerning floating point objects are:

	1. There is no space left for gc-bits in the float.
	2. The float has to be double aligned
	3. There must be a way of differentiate between permanent
	   (floats refered to from the code area) and temporary floats
	   (floats created during execution).
	
 */
	

#include "include.h"
#include "engine.h"
#include "unify.h"
#include "think.h"

#ifndef COPY_GC

static void marking_phase PROTO((worker *, int, int));
static void mark_registers PROTO((worker *, int));
static void mark_environments PROTO((environment *, worker *, int));
static void mark_choicepoints PROTO((worker *));
static void mark_variable PROTO((TAGGED *));
static void update_choicepoints PROTO((worker *));
static void compact_trail PROTO((worker *));
static void compacting_phase PROTO((worker *, int, int));
static void updateReallocChain PROTO((TAGGED *, TAGGED *));
static void compact_heap PROTO((worker *));
static void push_registers PROTO((worker *, int));
static void pop_registers PROTO((worker *, int));
static void sweep_trail PROTO((worker *));
static void sweep_environments PROTO((environment *, worker *, int));
static void sweep_choicepoints PROTO((worker *));

int total_marked;
int trailcells_deleted;

void garbage_collect(w,arity,envsize)
    worker *w;
    int arity, envsize;
{
  char *heap_gc_start;
  int gc_start_time;
    
#ifdef PARALLEL
  FatalError("GC needed but not supported (yet)");
#endif /* PARALLEL */

  gc_start_time = usertime();
  heap_gc_start = (char *) w->heap_top;
  total_marked = 0;
  trailcells_deleted = 0;

  if(w->global->flags.gc_verbose == TRUE)
    {
      PL_Print1(currout,"{running GC}\n");
    }

  marking_phase(w,arity,envsize);
  compacting_phase(w,arity,envsize);

  w->stats->heap_gc_nr++;
  w->stats->heap_gc_bytes += (int) (heap_gc_start - (char *) w->heap_top);
  w->stats->heap_gc_time += usertime() - gc_start_time;

  if(w->global->flags.gc_verbose == TRUE)
    {
      PL_Print2(currout,"{GC reclaimed %d bytes}\n", ((int) heap_gc_start) -
		((int) w->heap_top));
    }

  if(w->heap_top > w->heap_margin) {
      FatalError("Prolog out of memory");
  }
}

/**********************************************************************
 * Marking                                                            *
 **********************************************************************/

static void marking_phase(w, arity, envsize)
    worker *w;
    int arity, envsize;
{
    mark_registers(w,arity);
    mark_environments(w->frame,w,envsize);
    mark_choicepoints(w);
    update_choicepoints(w);
    compact_trail(w);
}

static void mark_registers(w, arity)
    worker *w;
    int arity;
{
  register TAGGED *areg = w->regs;
  
  while(arity--)
    {
      if(IsHeapTerm(X(arity)) || IsHeapBox(X(arity)))
	{
	  *w->trail_top = X(arity);
	  mark_variable(w->trail_top);
	}
    }
}

static void mark_environments(frame,w,size)
    register environment *frame;
    worker *w;
    int size;
{
    while(frame != NULL) {
	while(size--) {
	    switch(LowTagOf(Yf(size))) {
	    case HVA_LO:
#ifdef CONSTR 
	    case CVA_LO:
#endif /* CONSTR */
	      mark_variable(&(Yf(size)));
	      break;
	    case SVA_LO:
	      break;
	    case NUM_LO:
	      break;
	    case FLT_LO:
	      mark_variable(&(Yf(size)));
	      break;
	    case ATM_LO:
	      break;
	    case LST_LO:
	    case STR_LO:
	    case GEN_LO:
	      mark_variable(&(Yf(size)));
	      break;
	    default:
	      Error("mark_environment -- LowTagOf out of range");
	    }
	}
	size = FrameSize(frame->next_instr);
	frame = frame->cont_env;
    }
}


static void mark_choicepoints(w)
    register worker *w;
{
  register int i;
  register TAGGED *areg, *t = w->trail_top-1;
  register choicepoint *choice = w->choice;
  
  trailcells_deleted = 0;
  
  while (choice != NULL)
    {
      while (t >= choice->trail_top)
	{
	  if (!IsMarked(*TagToPointer(*t)))
	    {
	      if(IsValueTrail(t)) {  /* Value trail entry */
		Reset(*t);
		*t-- = 0;
		*t = 0;
		trailcells_deleted += 2;
	      } else if (IsHVA(*t)) {
		Reset(*t);
		*t = 0;
		trailcells_deleted++;
	      }
	    }
	  else
	    if (IsValueTrail(t)) {   /* Value trail entry */
	      t--;
	      mark_variable(t);
	    }
	  t--;
	}
      mark_environments(choice->cont_env,w,FrameSize(choice->next_instr));
      i = choice->arity;
      areg = choice->areg;
      while(i--)
	{
	  if(IsHeapTerm(X(i)) || IsHeapBox(X(i)))
	    {
	      mark_variable(&(X(i)));
	    }
	}
      choice = choice->last_choice;
    }
}

static void mark_variable(start)
    TAGGED *start;
{
  register TAGGED *current, *next;

  current = start;
  next = TagToPointer(*current);

  MarkF(*current);

  goto first_forward;

 forward:

  if (IsMarked(*current)) goto backward;

  total_marked++;

 first_forward:
  Mark(*current);

  switch(LowTagOf(*current)) {

  case HVA_LO:
#ifdef CONSTR
  case CVA_LO:
#endif /* CONSTR */
    if ( IsForM(*next) ) goto backward;
    Reverse(current,next);
    goto forward;

  case SVA_LO:
    FatalError("mark_variable -- Stack variable in heap!");
    break;
  case NUM_LO:
    goto backward;
  case FLT_LO:
    { 
      if(IsDynBox(*next))
	/* mark box */
	if(!IsMarked(*next))
	  {
	    Mark(*next);
	    total_marked += GetBoxSize(*next);
	  }
      goto backward;
    }
  case ATM_LO:
    goto backward;

  case LST_LO:
    {
      if ((IsMarkedF(*(next+1))) ||
	  (IsMarked(*next) &&
	   IsMarked(*(next+1)))) goto backward;

      next++;
      MarkF(*(next));
      
      Reverse(current, next);
      
      goto forward;
    }
    
  case STR_LO:
    { 
      register int arity;

      if (IsMarked(*next) ||
	  IsMarkedF(*(next+1))) goto backward;

      /* To take care of the functor */
      Mark(*next);
      total_marked++;

      arity = StructArity(next);

      if(arity == 0) {
	Error("mark_variable: Structure arity is zero");
	return;
      }

      while(arity--) {
	next++;
	MarkF(*(next));
      }

      Reverse(current, next);

      goto forward;
    }
    
  case GEN_LO:
    GetMethod(gc,*current)(*current);
    goto backward;

  default:
    Error("mark_variable -- LowTagOf out of range");
  }

 backward:

  if (!IsMarkedF(*current)) {

    Undo(current,next);
    goto backward;

  } else {

    UnMarkF(*current);

    if (current == start) return;

    Advance(current, next);

    goto forward;
  }
}
  
static void update_choicepoints(w)
    worker *w;
{
  register choicepoint *choice = w->choice;
  TAGGED *t = w->trail_top-1;
  
  while(choice != NULL)
    {
      while(t >= choice->trail_top)
	{
	  if(*t == 0)
	    trailcells_deleted--;
	  t--;
	}
      choice->trail_top -= trailcells_deleted;
      choice = choice->last_choice;
    }
}

static void compact_trail(w)
    worker *w;
{
  register TAGGED *dest, *current;

  dest = w->trail_start;

  for (current = w->trail_start ;
       current != w->trail_top ;
       current++)
      if (*current != 0)
	{
	  *dest = *current;
	  dest++;
	}
  w->trail_top = dest;
}

/**********************************************************************     
 * The Compaction Phase                                               *
 **********************************************************************/

static void compacting_phase(w, arity, envsize)
    worker *w;
    int arity, envsize;
{
  push_registers(w,arity);
  sweep_trail(w);
  sweep_environments(w->frame,w,envsize);
  sweep_choicepoints(w);
  compact_heap(w);
  pop_registers(w,arity);
}

static void intoReallocChain(J,C)
    TAGGED *J,*C;
{
    *(C) = PutValueFirst(*(J),*(C));
    *(J) = PutValueFirst((UL(C) | GC_F_MASK),*(J));
}

static void updateReallocChain(current, dest)
     TAGGED *current, *dest;
{
  register TAGGED *j;

  while(IsMarkedF(*current))
    {
      j = TagToPointer(*current);
      *current = PutValueFirst(*j,*current);
      *j = PutValue(dest,*j);
      UnMarkF(*j);
    }
}

static void compact_heap(w)
    worker *w;
{
  register TAGGED *dest, *current;

#ifdef PARALLEL_GC
  /* When garbage collecting several workers we have to 
     calculate 'total_marked' separately for each heap. 
     This is due to that a variable chain might span over 
     a number of heaps. The mark_variable procedure might
     therefore have marked variables in other heaps */

  /* calculate 'total_marked' */
  for(current = w->heap_start, 
      total_marked = 0         ; current < w->heap_top ; )
    {
      if (IsMarked(*current))
	{
	  if(IsBox(*current))
	    {
	      register int size = GetBoxSize(*current);
	      total_marked += size;
	      current += size;
	    }
	  else
	    {
	      total_marked++;
	      current++;
	    }
	}
      else if (IsBox(*current))
	{
	  current += GetBoxSize(*current);
	}
      else
	{
	  current++;
	}
    }

  w->gc_info.total_marked = total_marked;

#endif /* PARALLEL_GC */

  dest = w->heap_start + total_marked - 1;

  /* the upward phase */

  for (current = w->heap_top-1  ;
       current >= w->heap_start ; )
    {
      if (IsMarked(*current))
	{
	  updateReallocChain(current,dest);
	  if (IsHeapTerm(*current) || IsHeapBoxM(*current))
	    {
	      if (TagToPointer(*current) < current)
		{
		  intoReallocChain(TagToPointer(*current), current);
		}
	      else
		if (current == TagToPointer(*current))
		  *current = PutValue(dest, *current);
	    }
	  current--;
	  dest--;
	}
      else if (IsBox(*current)) /* Boxed object */
	{
	  register s32 size = GetBoxSize(*current);
	  current -= size;
	  if(IsMarked(*(current+1)))
	    {
	      dest -= size;
	      updateReallocChain(current+1,dest+1);
	    }
	}
      else
	current--;
    }

  /* the downward phase */

#if 1  
  if(dest+1 != w->heap_start) {
      Error("gc: dest != heap_start");
  }
#endif 

  dest = w->heap_start;

  for (current = w->heap_start ;
       current < w->heap_top ; 
       )
    {
      if (IsMarked(*current))
	{
	  updateReallocChain(current,dest);
	  if ((IsHeapTerm(*current) || IsHeapBoxM(*current)) &&
	      TagToPointer(*current) > current)
	    {
	      intoReallocChain(TagToPointer(*current), dest);
	      *dest = SetTag(*dest,TagOf(*current));
	      UnMark(*dest);
	      dest++;
	      current++;
	    }
	  else if(IsBox(*current))
	    {
	      register s32 i, size = GetBoxSize(*current);
	      for(i = 0; i < size ; i++)
		{
		  dest[i] = current[i];
		}
	      UnMark(*dest);
	      dest += size;
	      current += size;
	    }
	  else
	    {
	      *dest = *current;
	      UnMarkF(*dest);
	      UnMark(*dest);
	      dest++;
	      current++;
	    }
	}
      else if (IsBox(*current)) /* Boxed object */
	{
	  current += GetBoxSize(*current);
	}
      else
	{
	  current++;
	}
    }
  w->heap_top = dest;
}

static void push_registers(w,arity)
    worker *w;
    int arity;
{
  register TAGGED *areg = w->regs;
    
  while(arity--) {
    PushOnTrail(w->trail_top,X(arity));
  }
}

static void pop_registers(w,arity)
    worker *w;
    int arity;
{
  register TAGGED *areg = w->regs;
  register int i;

  for(i = 0 ; i != arity ; i++)
    PopFromTrail(w->trail_top,X(i));
}

static void sweep_trail(w)
    worker *w;
{
  register TAGGED *current;

  for (current = w->trail_top-1 ;
       current >= w->trail_start  ;
       current--)
    {
      if (IsHeapTerm(*current) || IsHeapBoxM(*current))
	{
	  intoReallocChain(TagToPointer(*current),current);
	}
    }
}
	  
static void sweep_environments(frame, w, size)
    register environment *frame;
    worker *w;
    register int size;
{
  while(frame != NULL) {
    while(size--) {
      if (IsHeapTerm(Yf(size)))
	{
	  if (!IsMarked(Yf(size)))
	    return;
	  else {
	    UnMark(Yf(size));
	    intoReallocChain(TagToPointer(Yf(size)),&(Yf(size)));
	  }
	}
      else if (IsHeapBoxM(Yf(size)))
	{
	  UnMark(Yf(size));
	  intoReallocChain(TagToPointer(Yf(size)),&(Yf(size)));
	}
    }
    size = FrameSize(frame->next_instr);
    frame = frame->cont_env;
  }
}

static void sweep_choicepoints(w)
    worker *w;
{
  register choicepoint *choice = w->choice;
  int arity;

  while(choice != NULL)
    {
      sweep_environments(choice->cont_env,w,FrameSize(choice->next_instr));
      arity = choice->arity;

      while(arity--)
	{
	  UnMark(Xc(arity));
	  if (IsHeapTerm(Xc(arity)) || IsHeapBoxM(Xc(arity)))
	    {
	      intoReallocChain(TagToPointer(Xc(arity)),&(Xc(arity)));
	    }
	}

      if(!IsMarked(*(choice->global_top)))
	{
	  *(choice->global_top) = atom_nil;
	  Mark(*(choice->global_top));
	  if(w->heap_top == choice->global_top)
	    {
	      w->heap_top++;
	    }
	  total_marked++;
	}

      intoReallocChain(choice->global_top,((TAGGED *) &(choice->global_top)));
      choice = choice->last_choice;
    }
}


      
#endif /* COPY_GC */
