/*    File:	 new_engine.c  (~bevemyr/Luther2/SharedEmulator/new_engine.c)
 *    Author:	 Johan Bevemyr
 *    Created:	 Wed Apr 14 15:42:25 1993
 *    Purpose:   Cleanup of the emulator code.
 */ 

#include "include.h"
#include "engine.h"
#include "unify.h"
#include "debug.h"
#include "inline.h"
#include "labelsort.h"
#include "initial.h"
#include "array.h"
#include "parallel.h"

#ifdef NEW_READ_NOC

#define Maybe_Get_Index_1(K,S,I)         {K = S ; Get_Index_I(1,I);}
#define Maybe_Get_Index_2(K,I)           {K = NULL;Get_Index_I(2,I);}

#else /* NEW_READ_NOC */

#define Maybe_Get_Index_1(K,S,I)         {K = Get_Index_I(1,I);}
#define Maybe_Get_Index_2(K,I)           {K = Get_Index_I(2,I);}

#endif /* NEW_READ_NOC */

/* Globally accessible WAM variables */

definition    *interpret_goal;          
definition    *interpret_goal_trace;    
definition    *abort_code;    

/* Table of instruction addresses for threaded code compilation */

#ifdef THREADED_CODE /* ---------------------------------------- */

c_address *lab_table;

#endif /* THREADED_CODE ---------------------------------------- */

code start_pred[] = { 0, HALT, HALT, HALT, HALT, HALT, HALT }; 

/**********************************************************************
 * WAM Emulator 
 *
 * If the worker pointer (w) is NULL then the label table is initialized
 * for threaded code. 
 *
 * If the definition pointer (def) is NULL then the wam is called in 
 * a parallel context by a parallel worker and should await commands
 * from the sequential worker.
 * 
 **********************************************************************/

void wam(def,w)
     definition *def;
     register worker *w;
{
  register code *pc;

  register TAGGED instruction,
                  *s,
                  *areg, 
                  *yreg, 
                  *hp;

  choicepoint *choice0,
              *failchoice;  

  int k;
  TAGGED *cont;

#include "engine_threaded.c"

  /* Initialize worker */

  init_wam(w);

  /* Initial choicepoint. If we ever backtrack to this choicepoint
   * we know that then entire program has failed. This is used to
   * make a graceful exit.
   */

  failchoice = w->choice;

  /* Cache variables in registers. This will hopefully speed 
   * things up a bit. Macros are used to hide this. 
   */

  areg = w->regs;
  s = w->s;
  pc = w->pc;
  hp = w->heap_top;

  /* If this wam is used as a co-worker then it should wait
   * for work.
   */

  if(def == NULL) 
    {
#ifdef PARALLEL
      goto wait_first_time;
#else
      goto done;
#endif /* PARALLEL */
    }

  /* We start the wam by running the definition the wam was
   * called with.
   */

 start:
  
  start_pred[3] = (TAGGED) def;
  pc = &(start_pred[3]);
  goto execute;

  /* The engire program failed, if co-worker report failure and
   * wait for more work. If sequential worker, return. 
   */

 global_fail:
  
  if(w->pid != 0)
    {
      Report_Global_Fail;
      goto done;
    }
  else
    {
      FatalError("Global Failure");
    }

  /* No more code to execute. The continuation and a 
   * HALT instruction has been executed.
   */

 done:

#ifdef PARALLEL /* ---------------------------------------- */

  StoreHeap;

 wait_for_work:
  w->stats->current_user_time = (int) usertime();
  DropSemaphore(SEMA_DONE,1,w);

 wait_first_time:
  GrabActivateSem(w->pid-1);

  /* 
   * First, check if this worker is active. 
   */

  if(w->pid > w->global->active_workers) 
    {
      /* This worker is not active, report done */
      goto wait_for_work;
    }

  /* Decode command from sequential worker */

    switch(w->global->parallel_start.type)
      {
      case W_EXECUTE_CODE:
	LoadHeap;
	pc = w->global->parallel_start.code;
	/* 
	 * Save current heap top on trail for backtracking
	 */
	PushOnTrail(w->trail_top,Tagify(H,NUM));
	reinitialize_stack(w);
	Execute_Read_Instr;
	
      case W_RESET:
	reinitialize(w);
	failchoice = w->choice;
	goto wait_for_work;
	
      case W_BACKTRACK:
	Worker_Unwind_Trail;      
	goto wait_for_work;
	
      case W_DEBUG:
	if(((s32) w->global->parallel_start.code) == w->pid) {
	  debugflag = TRUE;
	  debugmode = D_CREEP;
	}
	goto wait_for_work;
	
      case W_REDUCE_PLUS:
	w->global->reduction_results[w->pid-1] = reduce_vector_plus(w);
	goto wait_for_work;
	
      case W_REDUCE_TIMES:
	w->global->reduction_results[w->pid-1] = reduce_vector_times(w);
	goto wait_for_work;
	
      default:
	Error("No such command");
	goto wait_for_work;
      }

#endif /* PARALLEL ---------------------------------------- */

  /***********************************************************************
   * Main emulator loop. Instruction is decoded then switched on.
   */

 instructions:

  DisplayInstr("read");
  
  instruction = Get_Code(pc);

  switch(Get_Op(instruction)) 
    {
      /************************
       * DUMMY_INSTRUCTION
       ************************/
    case DUMMY_INSTRUCTION:
    dummy_instruction:
      Execute_Read_Instr;
      break;

      /************************
       * SWITCH_ON_TERM
       ************************/
    case SWITCH_ON_TERM:
    switch_on_term:
      {
	DerefNLL(X(0),X(0));

	switch(LowTagOf(X(0)))
	  {
	  case HVA_LO:
#ifdef CONSTR
	  case CVA_LO:
#endif /* CONSTR */
	  case SVA_LO:
	    Dispatch(pc,Var_Label);
	  case NUM_LO:
	  case FLT_LO:
	    Dispatch(pc,NUM_Label);
	  case ATM_LO:
	    Dispatch(pc,ATM_Label);
	  case LST_LO:
	    Dispatch(pc,LST_Label);
	  case STR_LO:
	    Dispatch(pc,STR_Label);
	  case GEN_LO:
	    /* This could be extended so that switch_on_term has a special
	       field for generic objects, and an introduction of switch_on_
	       generic could also be done. For now, we use the same as for
	       variables as they are the most general. */
	    Dispatch(pc,Var_Label);
	  default:
	    PL_Print2(currout,"engine: switch_on_term - no such termtype %lu",
		      LowTagOf(X(0)));
	    luther_exit(0);
	  }
      }
    
      /************************
       * SWITCH_ON_CONSTANT
       ************************/
    case SWITCH_ON_CONSTANT:
    switch_on_constant:
      { register indx i;
	
	i = Get_Index(pc);
	
	if(i < 5) { /* linear search */
	  register TAGGED c;
	  do {
	    c = Get_Tagged(pc);
	    if(X(0) != c)
	      Inc_Label(pc);
	    else 
	      break;
	  } while (--i);
	} else { /* binary search */
	  register labelt *table = (labelt *) pc;
	  register int x, l, r;
	  l = 0; r = i-1;
	  do {
	    x = (l + r) / 2;
	    if (X(0) < table[x].constant)
	      r = x - 1;
	    else if (X(0) > table[x].constant)
	      l = x + 1;
	    else {
	      pc = ((code *) &(table[x])) + 1;
	      goto sw_done;
	    }
	  } while (r >= l);
	  /* default */
	  pc = ((code *) &(table[i]));
	}
	
      sw_done:
	Dispatch(pc,0);
      }
      
      /************************
       * SWITCH_ON_STRUCTURE
       ************************/
    case SWITCH_ON_STRUCTURE:
    switch_on_structure: 
      {
	register indx i;
	register TAGGED func;
	
	i = Get_Index(pc);
	func = GetFunctor(X(0));
	
	if(i < 5) {
	  register TAGGED str;
	  do {
	    str = Get_Tagged(pc);
	    if(func != str) 
	      Inc_Label(pc);
	    else
	      break;
	  } while (--i);
	} else {
	  register labelt *table = (labelt *) pc;
	  register int x, l, r;
	  l = 0; r = i-1;
	  do {
	    x = (l + r) / 2;
	    if (func < table[x].constant)
	      r = x - 1;
	    else if (func > table[x].constant)
	      l = x + 1;
	    else {
	      pc = ((code *) &(table[x])) + 1;
	      goto sw_done_s;
	    }
	  } while (r >= l);
	  /* default */
	  pc = ((code *) &(table[i]));
	}
      sw_done_s:
	Dispatch(pc,0);
      }
      /************************
       * TRY
       ************************/
    case TRY: 
    try: 
      {
	register choicepoint *newchoice;
	register int i;
	register int arity;
	
	arity = Get_Index_I(1,instruction);

#if defined(TIMESTAMP) || defined(UNBOUND)
	w->time += TIMEUNIT;
#endif
	
	newchoice = (choicepoint *) Get_Local_Stack_Top;

	if(newchoice > (choicepoint *) w->stack_end) {
	  FatalError("Local stack overflow");
	}

	newchoice->trail_top = w->trail_top;
	newchoice->global_top = H;
	newchoice->last_choice = w->choice;
	newchoice->cont_env = w->frame;
	newchoice->next_instr = w->next_instr;
	newchoice->next_clause = pc+1;
	newchoice->arity = arity;
	for(i=0; i!=arity ; i++)
	  newchoice->areg[ARSIZE*i] = X(i);
#if defined(TIMESTAMP) || defined(UNBOUND)
	newchoice->timestamp = w->time;
	w->uncond = w->time;
#else
	w->uncond = H;
#endif /* TIMESTAMP */
	w->choice = newchoice;
	pc = DispatchLabel(pc,0);
	Execute_Read_Instr;
      }
      
      /************************
       * RETRY
       ************************/
    case RETRY:
    retry: 
      {
	w->choice->next_clause = pc+1;
	pc = DispatchLabel(pc,0);
	Execute_Read_Instr;
      }
      
      /************************
       * TRUST
       ************************/
    case TRUST:
    trust: 
      {
	w->choice = w->choice->last_choice;
#if defined(TIMESTAMP) || defined(UNBOUND)
	w->uncond = w->choice->timestamp;
#else
	w->uncond = w->choice->global_top;
#endif /* TIMESTAMP */
	
	pc = DispatchLabel(pc,0);
	Execute_Read_Instr;
      }
      
      /************************
       * TRY_ME_ELSE
       ************************/
    case TRY_ME_ELSE:
    try_me_else: 
      {
	register choicepoint *newchoice;
	register int i;
	register int arity;
	Pre_FetchTop(2);
	
	arity = Get_Index_I(1,instruction);

#if defined(TIMESTAMP) || defined(UNBOUND)
	w->time += TIMEUNIT;
#endif
	
	newchoice = (choicepoint *) Get_Local_Stack_Top;

	if(newchoice > (choicepoint *) w->stack_end) {
	  FatalError("Local stack overflow");
	}

	newchoice->trail_top = w->trail_top;
	newchoice->global_top = H;
	newchoice->last_choice = w->choice;
	newchoice->cont_env = w->frame;
	newchoice->next_instr = w->next_instr;
	newchoice->next_clause = DispatchLabel(pc,0);
	pc++;
	newchoice->arity = arity;
	
	for(i=0; i!=arity ; i++)
	  newchoice->areg[ARSIZE*i] = X(i);
#if defined(TIMESTAMP) || defined(UNBOUND)
	newchoice->timestamp = w->time;
	w->uncond = w->time;
#else
	w->uncond = H;
#endif /* TIMESTAMP */
	
	w->choice = newchoice;
	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * RETRY_ME_ELSE
       ************************/
    case RETRY_ME_ELSE:
    retry_me_else: 
      {
	Pre_FetchTop(1);
	w->choice->next_clause = DispatchLabel(pc,0);
	pc++;
	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * TRUST_ME
       ************************/
    case TRUST_ME:
    trust_me:
      {
	Pre_FetchTop(0);
	w->choice = w->choice->last_choice;
#if defined(TIMESTAMP) || defined(UNBOUND)
	w->uncond = w->choice->timestamp;
#else
	w->uncond = w->choice->global_top;
#endif
	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * CHOICE_X
       ************************/
    case CHOICE_X:
    choice_x: 
      {
	register indx n;
	Pre_FetchTop(1);
	
	n = Get_Index_I(1,instruction);
	
	X(n) = PointerToTerm(choice0);
	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * CHOICE_Y
       ************************/
    case CHOICE_Y:
    choice_y: 
      {
	register indx n;
	Pre_FetchTop(1);
	
	n = Get_Index_I(1,instruction);
	
	Y(n) = PointerToTerm(choice0);
	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * CUT
       ************************/
    case CUT:
    cut: 
      {
	Pre_FetchTop(0);
	if(w->choice > choice0) {
	  w->choice = choice0;
#if defined(TIMESTAMP) || defined(UNBOUND)
	  w->uncond = w->choice->timestamp;
#else
	  w->uncond = w->choice->global_top;
#endif
	  TidyTrail;
	}
	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * CUT_X
       ************************/
    case CUT_X:
    cut_x:
      {
	register indx n;
	Pre_FetchTop(1);
	
	n = Get_Index_I(1,instruction);
	
	w->choice = (choicepoint *) TermToPointer(X(n));
#if defined(TIMESTAMP) || defined(UNBOUND)
	w->uncond = w->choice->timestamp;
#else
	w->uncond = w->choice->global_top;
#endif
	TidyTrail;
	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * CUT_Y
       ************************/
    case CUT_Y:
    cut_y: 
      {
	register indx i;
	Pre_FetchTop(1);
	
	i = Get_Index_I(1,instruction);
	
	w->choice = (choicepoint *) TermToPointer(Y(i));
#if defined(TIMESTAMP) || defined(UNBOUND)
	w->uncond = w->choice->timestamp;
#else
	w->uncond = w->choice->global_top;
#endif
	TidyTrail;                    /* no - op */
	Pre_Execute_Read_Instr(1);
      }
      
      
      /************************
       * INLINE
       ************************/
    case INLINE:
    in_line: 
      {
	register int fnk;
	
	BuiltinCallStatistics;
	
	fnk = Get_Index_I(1,instruction);
	
	StoreHeap;
	
	Inc_Label(pc);
	if ((GetInlineFnk(fnk))(w,(s32 *)Get_UseArgs(pc)) == FALSE) {
	  pc = DispatchLabel(pc,-1);
	} else {
	  pc += GetInlineArity(fnk);
	}
	
	LoadHeap;
	
	Execute_Read_Instr;
      }
      
      /************************
       * BUILTIN
       ************************/
    case BUILTIN:
    builtin: 
      {
	register int fnk;
	
	BuiltinCallStatistics;
	
	fnk = Get_Index_I(1,instruction);
	
	StoreHeap;
	
	if ((GetInlineFnk(fnk))(w,(s32 *)Get_UseArgs(pc)) == FALSE) goto fail;
	
	LoadHeap;
	
	pc += GetInlineArity(fnk);
	Execute_Read_Instr;
      }
      
      /************************
       * META_CALL
       ************************/
    case META_CALL:
    meta_call:
      Get_Index_I(1,instruction);
      w->next_instr = pc+1;

      /************************
       * META_EXECUTE
       ************************/
    case META_EXECUTE:
    meta_execute:
      {
	register TAGGED goal;
	register definition *def;
	register indx i;
	Pre_FetchInit;
	
	i = Get_Index_I_M(1,instruction);

	DerefNLL(goal,X(i));
	
	/* Get definition */
	
	if (IsSTR(goal))
	  {
	    def = get_definition(GetFunctor(goal),w);
	    i = GetArity(goal);
	  }
	else if (IsATM(goal))
	  {
	    def = get_definition(StoreFunctor(goal,0),w);
	    i = 0;
	  }
	else if (IsLST(goal))
	  {
	    def = get_definition(functor_list,w);
	    i = 2;
	  }
	else if (IsNumber(goal))
	  {
	    goto fail;
	  }
	else
	  {
	    luther_error(E_ILLEGAL_GOAL, goal,w);
	    goto fail;
	  }
	
	switch (def->enter_instruction) {
	case ENTER_INTERPRETED:
	  choice0 = w->choice;
	  X(0) = goal;
	  if(w->lut_trace == 1) {
	    pc = interpret_goal_trace->entry_code.incoreinfo;
	    Pre_Fetch;
	  } else {
	    pc = interpret_goal->entry_code.incoreinfo;
	    Pre_Fetch;
	  }
	  break;
	  
	case ENTER_SPY:
	  choice0 = w->choice;
	  X(0) = goal;
	  pc = interpret_goal_trace->entry_code.incoreinfo;
	  Pre_Fetch;
	  break;
	  
	case ENTER_EMULATED:
	  /* Copy arguments from structure */
	  if(IsLST(goal)) {
	    X(0) = Ref(GetCar(goal));
	    X(1) = Ref(GetCdr(goal));
	  } else {
	    while(i--) {
	      X(i) = Ref(GetArg(goal,i));
	    }
	  }
	  
	  /* save the program counter in the continuation */
	  choice0 = w->choice;
	  pc = def->entry_code.incoreinfo;
	  Pre_Fetch;
	  break;
	  
	case ENTER_C:
	  /* Copy arguments from structure */
	  while(i--) {
	    X(i) = Ref(GetArg(goal,i));
	  }
	  
	  StoreHeap;
	  switch((def->entry_code.cinfo)(w)) {
	  case FALSE:
	    goto fail;
	  case TRUE:
	    pc = w->next_instr;
	    Pre_Fetch;
	    break;
	  }
	  LoadHeap;
	  break;
	  
	case ENTER_UNDEFINED:
	  luther_error(E_PRED_NOT_DEF, (TAGGED) def,w);
	  goto fail;
	}
	
	Pre_Execute_Read_Instr(0);
      }
      
#ifdef EXTENDED_CALL
      /************************
       * VAR_CALL
       ************************/
    case VAR_CALL:
    var_call:
      Get_Index_I(1,instruction);
      w->next_instr = pc+1;
      
      /************************
       * VAR_EXECUTE
       ************************/
    case VAR_EXECUTE:
    var_execute:
      {
	register TAGGED goal, name, vdef;
	register definition *def;
	register indx i;
	
	DerefNLL(name,X(0));
	DerefNLL(goal,X(1));
	
	if(IsVar(name))
	  goto fail;
	
	vdef = X(2);
	DerefLockSwitch(vdef,
			{
			  /* Get definition */
			  if (IsSTR(goal))
			    {
			      i = GetArity(goal);
			      def = get_definition(StoreFunctor(name,i),w);
			    }
			  else if (IsATM(goal))
			    {
			      def = get_definition(StoreFunctor(name,0),w);
			      i = 0;
			    }
			  else if (IsLST(goal))
			    {
			      def = get_definition(functor_list,w);
			      i = 2;
			    }
			  else if (IsNumber(goal))
			    {
			      goto fail;
			    }
			  else
			    {
			      luther_error(E_ILLEGAL_GOAL, goal,w);
			      goto fail;
			    }
			  Bind_Unsafe(vdef,PointerToTerm(def),{ goto fail; });
			}, 
			{
			  def = (definition *) TermToPointer(vdef);
			  i = ArityOf(def->name);
			  if (i != GetArity(goal))
			    goto fail;
			});
	
	switch (def->enter_instruction) {
	case ENTER_INTERPRETED:
	  {
	    register int arity;
	    /* make goal structure on heap */
	    
	    Make_STR(H,X(0),StoreFunctor(name,i));
	    for(arity = 0 ; arity != i ; arity++)
	      PushOnHeap(H,Ref(GetArg(goal,arity)));
	    choice0 = w->choice;
	    if(w->lut_trace == 1) {
	      pc = interpret_goal_trace->entry_code.incoreinfo;
	    } else {
	      pc = interpret_goal->entry_code.incoreinfo;
	    }
	    break;
	  }
	  
	case ENTER_SPY:
	  choice0 = w->choice;
	  pc = interpret_goal_trace->entry_code.incoreinfo;
	  break;
	  
	case ENTER_EMULATED:
	  /* Copy arguments from structure */
	  
	  if(IsLST(goal)) {
	    X(0) = Ref(GetCar(goal));
	    X(1) = Ref(GetCdr(goal));
	  } else {
	    while(i--) {
	      X(i) = Ref(GetArg(goal,i));
	    }
	  }
	  
	  /* save the program counter in the continuation */
	  choice0 = w->choice;
	  pc = def->entry_code.incoreinfo;
	  break;
	  
	case ENTER_C:
	  /* Copy arguments from structure */
	  while(i--) {
	    X(i) = Ref(GetArg(goal,i));
	  }
	  
	  StoreHeap;
	  switch((def->entry_code.cinfo)(w)) {
	  case FALSE:
	    goto fail;
	  case TRUE:
	    pc = w->next_instr;
	    break;
	  }
	  LoadHeap;
	  break;
	  
	case ENTER_UNDEFINED:
	  luther_error(E_PRED_NOT_DEF, (TAGGED) def,w);
	  goto fail;
	}
	
	Execute_Read_Instr;
      }
      break;
#endif /* EXTENDED_CALL */
      
      /************************
       * REQUIRE
       ************************/
    case REQUIRE:
    require:
      Inc_Index(pc);

      GC_If_Needed_Require(w,0);

      Execute_Read_Instr;
      
      
      /************************
       * REQUIRE_USING
       ************************/
    case REQUIRE_USING:
    require_using:
      { register s32 nr_live_x;

	Inc_Index(pc);

	nr_live_x = Get_Index_I(1,instruction) + 1;

	GC_If_Needed_Require(w,nr_live_x);
      
	Execute_Read_Instr;
      }
      
      /************************
       * ALLOCATE
       ************************/
    case ALLOCATE:
    allocate: 
      {
	register environment *newframe;
	Pre_FetchTop(0);
	
	newframe = (environment *) Get_Local_Stack_Top;

	if(newframe > (environment *) w->stack_end) {
	  FatalError("Local stack overflow");
	}
	
	newframe->cont_env = w->frame;
	newframe->next_instr = w->next_instr;
	SetY(newframe->yreg);
	w->frame = newframe;
	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * ALLOCATE2
       ************************/
    case ALLOCATE2:
    allocate2:
#ifdef BOUNDED_Q
      {
	register environment *newframe;
	
	newframe = (environment *) Get_Local_Stack_Top;

	if(newframe > (environment *) w->stack_end) {
	  FatalError("Local stack overflow");
	}
	
	newframe->cont_env = w->frame->cont_env;
	newframe->next_instr = w->frame->next_instr;
	SetY(newframe->yreg);
	w->frame = newframe;
	Execute_Read_Instr;
      }
#else
      {
	/* Not implemented just yet */
	FatalError("Instructions not implemented.");
	break;
      }
#endif /* BOUNDED_Q */
      
      /************************
       * DEALLOCATE
       ************************/
    case DEALLOCATE:
    deallocate:
      {
	Pre_FetchTop(0);
	w->next_instr = w->frame->next_instr;
	w->frame = w->frame->cont_env;
	SetY(w->frame->yreg);
	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * INIT
       ************************/
    case INIT:
    init:
      {
	register int i;
	register indx n;
	
	i = Get_Index_I(1,instruction);
	
	while(i--) {
	  n = Get_Index(pc);
	  LoadSVA(Y(n),w);
	}
	
	Execute_Read_Instr;
      }
      
      /************************
       * CALL
       ************************/
    case CALL:
    call: 

      Get_Index_I(1,instruction);
      w->next_instr = pc+1;

      /************************
       * EXECUTE
       ************************/
    case EXECUTE:
    execute: 
      {
	register definition *def;
	Pre_FetchInit;

	
	CallStatistics;
	
	def = Get_Definition(pc);

	/* check event flag (gc, trace, wake suspended, etc) */
	
	GC_If_Needed_Execute(w,def);

	/* call definition */
	
	switch (def->enter_instruction) {
	case ENTER_INTERPRETED:
	  { register TAGGED goal;
	    register int i;
	    register int arity;
	    
	    /* Make goal structure on heap */
	    Make_STR(H,goal,def->name);
	    
	    for(i = 0, arity = ArityOf(def->name) ; i != arity ; i++) {
	      if(IsSVA(X(i))) {
		WriteLocalValue(H,X(i));
	      } else {
		PushOnHeap(H,X(i));
	      }
	    }
	    
	    X(0) = goal;
	    
	    choice0 = w->choice;
	    if(w->lut_trace == 1) {
	      pc = interpret_goal_trace->entry_code.incoreinfo;
	    } else {
	      pc = interpret_goal->entry_code.incoreinfo;
	    }
	    Pre_Fetch;
	    break;
	  }	  
	case ENTER_SPY:
	  { register TAGGED goal;
	    register int i;
	    register int arity;
	    
	    /* Make goal structure on heap */
	    Make_STR(H,goal,def->name);
	    
	    for(i = 0, arity = ArityOf(def->name) ; i != arity ; i++)
	      PushOnHeap(H,X(i));
	    
	    X(0) = goal;
	    
	    choice0 = w->choice;
	    pc = interpret_goal_trace->entry_code.incoreinfo;
	    Pre_Fetch;
	    break;
	  }	  
	case ENTER_EMULATED:
	  pc = def->entry_code.incoreinfo;
	  Pre_Fetch;
	  choice0 = w->choice;
	  break;
	case ENTER_C:
	  StoreHeap;
	  switch((def->entry_code.cinfo)(w)) {
	  case FALSE:
	    goto fail;
	  case TRUE:
	    pc = w->next_instr;
	    Pre_Fetch;
	    break;
	  }
	  LoadHeap;
	  break;
	case ENTER_UNDEFINED:
	  luther_error(E_PRED_NOT_DEF, (TAGGED) def,w);
	  goto fail;
	}

#include "event.c"

	Pre_Execute_Read_Instr(0);
      }
      
      /************************
       * PROCEED
       ************************/
    case PROCEED:
    proceed:
      {
	Pre_FetchInit;

	pc = w->next_instr;
	Pre_Fetch;

	Pre_Execute_Read_Instr(0);
      }
      
      /************************
       * FAIL
       ************************/
    case FAIL:
    fail:
      {
	register int i;
	Pre_FetchInit;
	
	FailStatistics;
	
	if(w->choice<=failchoice) 
	  goto global_fail;              /* The entire program failed */
	
	Unwind_Trail(w->choice->trail_top);
#ifndef DISABLE_HEAP_RECLAIM
	H = w->choice->global_top;
#endif /* DISABLE_HEAP_RECLAIM */
	choice0 = w->choice->last_choice;
	
	i = w->choice->arity;
	while(i) {
	  i--;
	  X(i) = w->choice->areg[ARSIZE*i];
	}
	
#if defined(TIMESTAMP) || defined(UNBOUND)
	w->uncond = w->time = w->choice->timestamp;
#else
	w->uncond = H;
#endif /* TIMESTAMP */
	pc = w->choice->next_clause;
	Pre_Fetch;
	w->next_instr = w->choice->next_instr;
	w->frame = w->choice->cont_env;
	SetY(w->frame->yreg);
	
	DisplayFail("read");
	
	Pre_Execute_Read_Instr(0);
      }
      
      /************************
       * GET_X_VARIABLE
       ************************/
    case GET_X_VARIABLE:
    get_x_variable: 
      {
	register indx i,n;
	Pre_FetchTop(2);
	
	n = Get_Index_I_M(1,instruction);
	i = Get_Index_I(2,instruction);
	
	X(n) = X(i);

	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * GET_Y_VARIABLE
       ************************/
    case GET_Y_VARIABLE:
    get_y_variable: 
      {
	register indx i,n;
	Pre_FetchTop(2);
	
	n = Get_Index_I_M(1,instruction);
	i = Get_Index_I(2,instruction);
	
	Y(n) = X(i);
	Pre_Execute_Read_Instr(1);
      }
      /************************
       * GET_Y_FIRST_VALUE
       ************************/
    case GET_Y_FIRST_VALUE:
    get_y_first_value: 
      {
	register indx i,n;
	Pre_FetchTop(2);
	
	n = Get_Index_I_M(1,instruction);
	i = Get_Index_I(2,instruction);
	
	Y(n) = X(i);

	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * GET_X_VALUE
       ************************/
    case GET_X_VALUE:
    get_x_value: 
      {
	register indx i,n;
	Pre_FetchTop(2);
	
	n = Get_Index_I_M(1,instruction);
	i = Get_Index_I(2,instruction);
	
	Unify(X(n),X(i));
	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * GET_Y_VALUE
       ************************/
    case GET_Y_VALUE:
    get_y_value:
      {
	register indx i,n;
	Pre_FetchTop(2);
	
	n = Get_Index_I_M(1,instruction); 
	i = Get_Index_I(2,instruction);
	
	Unify(Y(n),X(i));

	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * GET_CONSTANT
       ************************/
    case GET_CONSTANT:
    get_constant:
      {
	register TAGGED c, Xi;
	register indx i;
	Pre_FetchTop(2);
	
	c = Get_Tagged(pc); 
	i = Get_Index_I(1,instruction);
	
	/* Unify1(X(i),c); unfolwing yields */
	Xi = X(i);
	DerefLockSwitch(Xi,
			{
			  Bind_Unsafe(Xi,c,{goto fail;});
			},
			{
			  if (Xi != c)
			    goto fail;
			});
	
	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * GET_NIL
       ************************/
    case GET_NIL:
    get_nil:
      {
	register indx i;
	register TAGGED Xi;
	Pre_FetchTop(1);
	
	i = Get_Index_I(1,instruction);
	
	Xi = X(i);
	DerefLockSwitch(Xi,
			{
			  Bind_Unsafe(Xi,atom_nil,{goto fail;});
			},
			{
			  if (Xi != atom_nil) 
			    goto fail;
			});
	
	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * GET_STRUCTURE
       ************************/
    case GET_STRUCTURE:
    get_structure:
      {
	register TAGGED Xi,new, f;
	Pre_FetchTop(2);
	
	f = Get_Functor(pc);
	
	Xi = X(Get_Index_I(1,instruction));
	DerefLockSwitch(Xi,
			{
			  Make_STR_A(H,s,new,f);
			  Bind_Unsafe(Xi,new,
				      {
					Error("get_structure - unsafe");
					goto fail;
				      });
			  Pre_Execute_Write_Instr(1);
			},
			{
			  if(IsSTR(Xi)) {
			    if(GetFunctor(Xi) == f) {
			      s = GetArg(Xi,0);
			      Pre_Execute_Read_Instr(1);
			    } else
			      goto fail;
			  } else
			    goto fail;
			});
      }
      
      /************************
       * GET_LIST
       ************************/
    case GET_LIST:
    get_list:
      {
	register TAGGED Xi;
	register indx i;      
	Pre_FetchTop(1);
	
	i = Get_Index_I(1,instruction);
	
	Xi = X(i);
	DerefLockSwitch(Xi,
			{
			  register TAGGED l;
			  
			  Make_LST_A(H,s,l);
			  Bind_Unsafe(Xi,l,
				      {
					Error("get_list - unsafe");
					goto fail;
				      });
			  Pre_Execute_Write_Instr(1);
			},
			{
			  if(IsLST(Xi)) {
			    s = RemoveTag(Xi,LST);
			    Pre_Execute_Read_Instr(1);
			  } else
			    goto fail;
			});
      }
      
      /************************
       * GET_CONSTANT_X0
       ************************/
    case GET_CONSTANT_X0:
    get_constant_x0:
      {
	register TAGGED c,X0;
	Pre_FetchTop(1);
	
	c = Get_Tagged(pc);
	X0 = X(0);      
	DerefLockSwitch(X0,
			{
			  Bind_Unsafe(X0,c,{goto fail;});
			},
			{
			  if (X0 != c)
			    goto fail;
			});
	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * GET_NIL_X0
       ************************/
    case GET_NIL_X0:
    get_nil_x0:
      {
	register TAGGED X0;
	Pre_FetchTop(0);
	
	X0 = X(0);
	DerefLockSwitch(X0,
			{
			  Bind_Unsafe(X0,atom_nil,{goto fail;});
			},
			{
			  if (X0 != atom_nil) 
			    goto fail;
			});
	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * GET_STRUCTURE_X0
       ************************/
    case GET_STRUCTURE_X0:
    get_structure_x0:
      {
	register TAGGED f,X0;
	Pre_FetchTop(1);
	
	f = Get_Functor(pc);
	
	X0 = X(0);
	DerefLockSwitch(X0,
			{
			  register TAGGED new;

			  Make_STR_A(H,s,new,f);
			  Bind_Unsafe(X0,new,
				      {
					Error("get_structure_x0 - unsafe");
					goto fail;
				      });
			  Pre_Execute_Write_Instr(1);
			},
			{
			  if(GetFunctor(X0) != f)
			    goto fail;
			  s = GetArg(X0,0);
			  Pre_Execute_Read_Instr(1);
			});
      }
      
      /************************
       * GET_LIST_X0
       ************************/
    case GET_LIST_X0:
    get_list_x0:
      {
	register TAGGED X0;
	Pre_FetchTop(0);
	
	X0 = X(0);
	DerefLockSwitch(X0,
			{
			  register TAGGED l;
			  
			  Make_LST_A(H,s,l);
			  Bind_Unsafe(X0,l,
				      {
					Error("get_list_x0 - unsafe");
					goto fail;
				      });
			  Pre_Execute_Write_Instr(1);
			},
			{
			  if (IsLST(X0)) {
			    s = RemoveTag(X0,LST);
			    Pre_Execute_Read_Instr(1);
			  } else
			    goto fail;
			});
      }
      
      /************************
       * PUT_X_VOID
       ************************/
    case PUT_X_VOID:
    put_x_void:
      {
	register indx i;
	Pre_FetchTop(1);
	
	i = Get_Index_I(1,instruction);
	
	LoadHVA(H,X(i),w);
	
	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * PUT_Y_VOID
       ************************/
    case PUT_Y_VOID:
    put_y_void:
      {
	register indx i;      
	Pre_FetchTop(1);
	
	i = Get_Index_I(1,instruction);
	
	LoadSVA(Y(i),w);
	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * PUT_X_VARIABLE
       ************************/
    case PUT_X_VARIABLE:
    put_x_variable:
      {
	register indx i,n;      
	Pre_FetchTop(2);
	
	n = Get_Index_I_M(1,instruction);
	i = Get_Index_I(2,instruction);
	
	LoadHVA(H,X(n),w);
	X(i) = X(n);
	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * PUT_Y_VARIABLE
       ************************/
    case PUT_Y_VARIABLE:
    put_y_variable: 
      {
	register indx i,n;      
	Pre_FetchTop(2);
	
	n = Get_Index_I_M(1,instruction);
	i = Get_Index_I(2,instruction);
	
	LoadSVA(Y(n),w);
	X(i) = Y(n);
	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * PUT_X_VALUE
       ************************/
    case PUT_X_VALUE:
    put_x_value:
      {
	register indx i,n;      
	Pre_FetchTop(2);
	
	n = Get_Index_I_M(1,instruction);
	i = Get_Index_I(2,instruction);
	
	X(i) = X(n);
	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * PUT_Y_VALUE
       ************************/
    case PUT_Y_VALUE:
    put_y_value:
      {
	register indx i,n;      
	Pre_FetchTop(2);
	
	n = Get_Index_I_M(1,instruction);
	i = Get_Index_I(2,instruction);
	
	X(i) = Y(n);
	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * PUT_X_UNSAFE_VALUE
       ************************/
    case PUT_X_UNSAFE_VALUE:
    put_x_unsafe_value:
      {
	register indx i,n;      
	Pre_FetchTop(2);
	
	n = Get_Index_I_M(1,instruction);
	i = Get_Index_I(2,instruction);
	
	RefStackUnsafe(H,X(i),X(n));
	X(n) = X(i);
	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * PUT_Y_UNSAFE_VALUE
       ************************/
    case PUT_Y_UNSAFE_VALUE:
    put_y_unsafe_value:
      {
	register indx i,n;
	Pre_FetchTop(2);
	
	n = Get_Index_I_M(1,instruction);
	i = Get_Index_I(2,instruction);
	
	RefStackUnsafe(H,X(i),Y(n));
	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * PUT_CONSTANT
       ************************/
    case PUT_CONSTANT:
    put_constant: 
      {
	register indx i;
	register TAGGED c;
	Pre_FetchTop(2);
	
	c = Get_Tagged(pc);
	i = Get_Index_I(1,instruction);
	
	X(i) = c;
        Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * PUT_NIL
       ************************/
    case PUT_NIL:
    put_nil:
      {
	register indx i;
	Pre_FetchTop(1);
	i = Get_Index_I(1,instruction);
	
	X(i) = atom_nil;
	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * PUT_STRUCTURE
       ************************/
    case PUT_STRUCTURE:
    put_structure:
      {
	register TAGGED f;
	register indx i;
	Pre_FetchTop(2);
	
	f = Get_Functor(pc);
	i = Get_Index_I(1,instruction);
	
	Make_STR_A(H,s,X(i),f);

	Pre_Execute_Write_Instr(1);
      }
      
      /************************
       * PUT_LIST
       ************************/
    case PUT_LIST:
    put_list:
      {
	register indx n;
	Pre_FetchTop(1);

	n = Get_Index_I(1,instruction);
	
	Make_LST_A(H,s,X(n));
	Pre_Execute_Write_Instr(1);
      }    
      
      /************************
       * UNIFY_VOID
       ************************/
    case UNIFY_VOID:
    unify_void:
      WriteModeDispatch(unify_void_write);
      {
	Pre_FetchTop(1);

	s += Get_Index_I(1,instruction) * VARSIZE;
	
	Pre_Execute_Read_Instr(1);
      }    
      
      /************************
       * UNIFY_X_VARIABLE
       ************************/
    case UNIFY_X_VARIABLE: 
    unify_x_variable:
      WriteModeDispatch(unify_x_variable_write);
      {
	register indx n;
	Pre_FetchTop(1);

	n = Get_Index_I(1,instruction);
	
	X(n) = Ref(s);
	s += VARSIZE;
	Pre_Execute_Read_Instr(1);
      }    
      
      /************************
       * UNIFY_Y_VARIABLE
       ************************/
    case UNIFY_Y_VARIABLE:
    unify_y_variable:
      WriteModeDispatch(unify_y_variable_write);
      {
	register indx i;
	Pre_FetchTop(1);

	i = Get_Index_I(1,instruction);
	
/*
#ifdef TRAIL_ALL
  PushOnTrail(w->trail_top, Y(i));
#else
  if ((TAGGED *)yreg < (TAGGED *)w->choice) {
  PushOnTrail(w->trail_top, Y(i));
  }
#endif
  */
	Y(i) = Ref(s);
	s += VARSIZE;
	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * UNIFY_Y_FIRST_VALUE
       ************************/
    case UNIFY_Y_FIRST_VALUE:
    unify_y_first_value:
      WriteModeDispatch(unify_y_first_value_write);
      {
	register indx n;
	Pre_FetchTop(1);

	n = Get_Index_I(1,instruction);
	
	Bind_SVA(Tagify(&(Y(n)),SVA),Ref(s));
	s += VARSIZE;

	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * UNIFY_X_VALUE
       ************************/
    case UNIFY_X_VALUE:
    unify_x_value:
      WriteModeDispatch(unify_x_value_write);
      {
	register indx n;
	Pre_FetchTop(1);

	n = Get_Index_I(1,instruction);
	
	Unify(Ref(s),X(n));
	s += VARSIZE;

	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * UNIFY_Y_VALUE
       ************************/
    case UNIFY_Y_VALUE:
    unify_y_value:
      WriteModeDispatch(unify_y_value_write);
      {
	register indx n;
	Pre_FetchTop(1);

	n = Get_Index_I(1,instruction);
	
	Unify(Ref(s),Y(n));
	s += VARSIZE;
	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * UNIFY_X_LOCAL_VALUE
       ************************/
    case UNIFY_X_LOCAL_VALUE:
    unify_x_local_value:
      WriteModeDispatch(unify_x_local_value_write);
      {
	register indx n;
	Pre_FetchTop(1);

	n = Get_Index_I(1,instruction);
	
	Unify(X(n),Ref(s));
	s += VARSIZE;

	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * UNIFY_Y_LOCAL_VALUE
       ************************/
    case UNIFY_Y_LOCAL_VALUE:
    unify_y_local_value:
      WriteModeDispatch(unify_y_local_value_write);
      {
	register indx n;
	Pre_FetchTop(1);

	n = Get_Index_I(1,instruction);
	
	Unify(Y(n),Ref(s));
	s += VARSIZE;

	Pre_Execute_Read_Instr(1);
      }    
      
      /************************
       * UNIFY_CONSTANT
       ************************/
    case UNIFY_CONSTANT:
    unify_constant:
      WriteModeDispatch(unify_constant_write);
      { 
	register TAGGED c, Si;
	Pre_FetchTop(1);
	
	c = Get_Tagged(pc);
	
	Si = Ref(s);
	s += VARSIZE;
	DerefLockSwitchHVA(Si,
			   {
			     Bind_Unsafe_Heap(Si,c,{goto fail;});
			   },
			   { 
			     if (Si != c) 
			       goto fail;
			   });
	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * UNIFY_NIL
       ************************/
    case UNIFY_NIL:
    unify_nil:
      WriteModeDispatch(unify_nil_write);
      {
	register TAGGED Si;
	Pre_FetchTop(0);
	
	Si = Ref(s);
	s += VARSIZE;
	DerefLockSwitchHVA(Si,
			   {
			     Bind_Unsafe_Heap(Si,atom_nil,{goto fail;});
			   },
			   {
			     if (Si != atom_nil)
			       goto fail;
			   });
	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * UNIFY_STRUCTURE
       ************************/
    case UNIFY_STRUCTURE:
    unify_structure:
      WriteModeDispatch(unify_structure_write);
      {
	register TAGGED f,Ds,str;
	Pre_FetchTop(1);
	
	f = Get_Functor(pc);
	
	Ds = Ref(s);
	DerefLockSwitchHVA(Ds,
			   {
			     Make_STR_A(H,s,str,f);
			     Bind_Unsafe(Ds,str,
					 {
					   Error("unify_structure - unsafe");
					   goto fail;
					 });
			     Pre_Execute_Write_Instr(1);
			   },
			   {
			     if(IsSTR(Ds)) {
			       if(GetFunctor(Ds) == f) {
				 s = GetArg(Ds,0);
				 Pre_Execute_Read_Instr(1);
			       }
			     }
			   });
	goto fail;
      }
      
      /************************
       * UNIFY_LIST
       ************************/
    case UNIFY_LIST:
    unify_list:
      WriteModeDispatch(unify_list_write);
      { 
	register TAGGED Ds, lst;
	Pre_FetchTop(0);
	
	Ds = Ref(s);
	DerefLockSwitchHVA(Ds,
			   {
			     Make_LST_A(H,s,lst);
			     Bind_Unsafe(Ds,lst,
					 {
					   Error("unify_list - unsafe");
					   goto fail;
					 });
			     Pre_Execute_Write_Instr(1);
			   },
			   {
			     if(IsLST(Ds)) {
			       s = GetCar(Ds);
			       Pre_Execute_Read_Instr(1);
			     } else
			       goto fail;
			   });
      }
      
#ifdef JUMP_CALL
      /************************
       * CJUMP
       ************************/
    case CJUMP:
    cjump:
      {
	CallStatistics;
	Get_Index_I_M(1,instruction);
	w->next_instr = pc+1;
	choice0 = w->choice;
	pc = Get_PC_No_Inc(pc);
	Execute_Read_Instr;
      }

      /************************
       * EJUMP
       ************************/
    case EJUMP:
    ejump:
      {
	CallStatistics;
	pc = Get_PC_No_Inc(pc);
	choice0 = w->choice;
	Execute_Read_Instr;
      }
#endif /* JUMP_CALL */

#ifdef NEW_READ
      /************************
       * READ_LIST_TOP
       ************************/
    case READ_LIST_TOP:
    read_list_top:
      {
	register indx i;
	register TAGGED Xi, lst;
	Pre_FetchTop(2);
	
	InitCont; /* initialize read stack */
	
	i = Get_Index_I_M(1,instruction);
	
	Xi = X(i);
#ifdef PARALLEL
	DerefLockSwitch(Xi,
			{
 			  Maybe_Get_Index_2(k,instruction);
			  Make_LST_S(H,s,lst);
			  PushCont(Xi);
			  PushCont(lst);
			  Pre_Execute_Write_Instr(1);
			},
			{
			  Get_Index_I(2,instruction);
			  if (IsLST(Xi)) {
			    s = GetCar(Xi);
			    Pre_Execute_Read_Instr(1);
			  } else
			    goto fail;
			});
#else
	DerefLockSwitch(Xi,
			{
 			  Maybe_Get_Index_2(k,instruction);
			  Make_LST_S(H,s,lst);
			  Bind_Unsafe(Xi,lst,
				      {
					Error("read_list_top - unsafe");
					goto fail;
				      });
			  Pre_Execute_Write_Instr(1);
			},
			{
			  Get_Index_I(2,instruction);
			  if (IsLST(Xi)) {
			    s = GetCar(Xi);
			    Pre_Execute_Read_Instr(1);
			  } else
			    goto fail;
			});
#endif /* PARALLEL */
      }
      
      /************************
       * READ_STRUCT_TOP
       ************************/
    case READ_STRUCT_TOP:
    read_struct_top:
      {
	  register TAGGED Xi, str, f;
	  indx i;
	  Pre_FetchTop(3);
	  
	  InitCont;
	  
	  i = Get_Index_I_M(1,instruction);
	  f = Get_Functor(pc);
	  
	  Xi = X(i);
#ifdef PARALLEL
	  DerefLockSwitch(Xi,
			  {
			    Maybe_Get_Index_2(k,instruction);
			    Make_STR_Alloc(H,s,str,f);
			    PushCont(Xi);
			    PushCont(str);
			    Pre_Execute_Write_Instr(1);
			  },
			  {
			    Get_Index_I(2,instruction);
			    if (IsSTR(Xi)) {
			      if(GetFunctor(Xi) == f) {
				s = GetArg(Xi,0);
				Pre_Execute_Read_Instr(1);
			      } else
				goto fail;
			    } else
			      goto fail;
			  });
#else
  	  DerefLockSwitch(Xi,
			  {
			    Maybe_Get_Index_2(k,instruction);
			    Make_STR_Alloc(H,s,str,f);
			    Bind_Unsafe(Xi,str,
					{
					  Error("read_struct_top - unsafe");
					  goto fail;
					});
			    Pre_Execute_Write_Instr(1);
			  },
			  {
			    Get_Index_I(2,instruction);
			    if (IsSTR(Xi)) {
			      if(GetFunctor(Xi) == f) {
				s = GetArg(Xi,0);
				Pre_Execute_Read_Instr(1);
			      } else
				goto fail;
			    } else
			      goto fail;
			  });
#endif /* PARALLEL */
	}
      
      /************************
       * READ_LIST
       ************************/
    case READ_LIST:
      read_list:
      WriteModeDispatch(read_list_write);
      {
	register TAGGED Ds, lst;
	Pre_FetchTop(1);
	
	Ds = Ref(s);
	s += VARSIZE;
#ifdef PARALLEL
	DerefLockSwitchHVA(Ds,
			   {
			     Maybe_Get_Index_1(k,s,instruction);
			     PushCont(s);
			     Make_LST_S(H,s,lst);
			     PushCont(Ds);
			     PushCont(lst);
			     Pre_Execute_Write_Instr(1);
			   },
			   {
			     Get_Index_I(1,instruction);
			     if (IsLST(Ds)) {
			       PushCont(s);
			       s = GetCar(Ds);
			       Pre_Execute_Read_Instr(1);
			     } else
			       goto fail;
			   });
#else
	DerefLockSwitchHVA(Ds,
			   {
			     Maybe_Get_Index_1(k,s,instruction);
			     PushCont(s);
			     Make_LST_S(H,s,lst);
			     Bind_Unsafe(Ds,lst,
					 {
					   Error("read_list - unsafe");
					   goto fail;
					 });
			     Pre_Execute_Write_Instr(1);
			   },
			   {
			     Get_Index_I(1,instruction);
			     if (IsLST(Ds)) {
			       PushCont(s);
			       s = GetCar(Ds);
			       Pre_Execute_Read_Instr(1);
			     } else
			       goto fail;
			   });
#endif /* PARALLEL */
	
      }
      
      /************************
       * READ_STRUCT
       ************************/
    case READ_STRUCT:
      read_struct:
      WriteModeDispatch(read_struct_write);
      {
	register TAGGED f, Ds, str;
	Pre_FetchTop(2);
	
	f = Get_Functor(pc);
	
	Ds = Ref(s);
	s += VARSIZE;
#ifdef PARALLEL
	DerefLockSwitchHVA(Ds,
			   {
			     Maybe_Get_Index_1(k,s,instruction);
			     PushCont(s);
			     Make_STR_Alloc(H,s,str,f);
			     PushCont(Ds);
			     PushCont(str);
			     Pre_Execute_Write_Instr(1);
			   },
			   {
			     Get_Index_I(1,instruction);
			     if (IsSTR(Ds)) {
			       PushCont(s);
			       if(GetFunctor(Ds) == f) {
				 s = GetArg(Ds,0);
				 Pre_Execute_Read_Instr(1);
			       }
			     }
			   });
#else
	DerefLockSwitchHVA(Ds,
			   {
			     Maybe_Get_Index_1(k,s,instruction);
			     PushCont(s);
			     Make_STR_Alloc(H,s,str,f);
			     Bind_Unsafe(Ds,str,
					 {
					   Error("read_struct - unsafe");
					   goto fail;
					 });
			     Pre_Execute_Write_Instr(1);
			   },
			   {
			     Get_Index_I(1,instruction);
			     if (IsSTR(Ds)) {
			       PushCont(s);
			       if(GetFunctor(Ds) == f) {
				 s = GetArg(Ds,0);
				 Pre_Execute_Read_Instr(1);
			       }
			     }
			   });
#endif /* PARALLEL */	
	goto fail;
      }
      
      /************************
       * READ_LIST_TAIL
       ************************/
    case READ_LIST_TAIL:
      read_list_tail:
      WriteModeDispatch(read_list_tail_write);
      {
	register TAGGED Ds, lst;
	Pre_FetchTop(1);
	
	Ds = Ref(s);
#ifdef PARALLEL
	DerefLockSwitchHVA(Ds,
			   {
			     Maybe_Get_Index_1(k,TopCont,instruction);
			     Make_LST_S(H,s,lst);
			     PushCont(Ds);
			     PushCont(lst);
			     Pre_Execute_Write_Instr(1);
			   },
			   {
			     Get_Index_I(1,instruction);
			     if (IsLST(Ds)) {
			       s = GetCar(Ds);
			       Pre_Execute_Read_Instr(1);
			     } else
			       goto fail;
			   });
#else
	DerefLockSwitchHVA(Ds,
			   {
			     Maybe_Get_Index_1(k,TopCont,instruction);
			     Make_LST_S(H,s,lst);
			     Bind_Unsafe(Ds,lst,
					 {
					   Error("read_list_tail - unsafe");
					   goto fail;
					 });
			     Pre_Execute_Write_Instr(1);
			   },
			   {
			     Get_Index_I(1,instruction);
			     if (IsLST(Ds)) {
			       s = GetCar(Ds);
			       Pre_Execute_Read_Instr(1);
			     } else
			       goto fail;
			   });
#endif /* PARALLEL */	
      }
      
      /************************
       * READ_STRUCT_TAIL
       ************************/
    case READ_STRUCT_TAIL:
      read_struct_tail:
      WriteModeDispatch(read_struct_tail_write);
      {
	register TAGGED f, Ds, str;
	Pre_FetchTop(2);
	
	f = Get_Functor(pc);
	
	Ds = Ref(s);
#ifdef PARALLEL
	DerefLockSwitchHVA(Ds,
			   {
			     Maybe_Get_Index_1(k,TopCont,instruction);
			     Make_STR_Alloc(H,s,str,f);
			     PushCont(Ds);
			     PushCont(str);
			     Pre_Execute_Write_Instr(1);
			   },
			   {
			     Get_Index_I(1,instruction);
			     if (IsSTR(Ds)) {
			       if(GetFunctor(Ds) == f) {
				 s = GetArg(Ds,0);
				 Pre_Execute_Read_Instr(1);
			       }
			     }
			   });
#else	
	DerefLockSwitchHVA(Ds,
			   {
			     Maybe_Get_Index_1(k,TopCont,instruction);
			     Make_STR_Alloc(H,s,str,f);
			     Bind_Unsafe(Ds,str,
					 {
					   Error("read_struct_tail - unsafe");
					   goto fail;
					 });
			     Pre_Execute_Write_Instr(1);
			   },
			   {
			     Get_Index_I(1,instruction);
			     if (IsSTR(Ds)) {
			       if(GetFunctor(Ds) == f) {
				 s = GetArg(Ds,0);
				 Pre_Execute_Read_Instr(1);
			       }
			     }
			   });
#endif /* PARALLEL */
	goto fail;
      }
      
      /************************
       * UNIFY_CONSTANT_UP
       ************************/
    case UNIFY_CONSTANT_UP:
      unify_constant_up:
      WriteModeDispatch(unify_constant_up_write);
      {
	register TAGGED c, Si;
	Pre_FetchTop(1);
	
	c = Get_Tagged(pc);
	
	Si = Ref(s);
	DerefLockSwitchHVA(Si,
			   {
			     Bind_Unsafe_Heap(Si,c,{goto fail;});
			   },
			   {
			     if (Si != c)
			       goto fail;
			   });
	
	s = PopCont;
	
	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * UNIFY_X_VARIABLE_UP
       ************************/
    case UNIFY_X_VARIABLE_UP:
      unify_x_variable_up:
      WriteModeDispatch(unify_x_variable_up_write);
      {
	register indx n;
	Pre_FetchTop(1);

	n = Get_Index_I(1,instruction);
	
	X(n) = *(s);
	s = PopCont;
	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * UNIFY_Y_VARIABLE_UP
       ************************/
    case UNIFY_Y_VARIABLE_UP:
      unify_y_variable_up:
      WriteModeDispatch(unify_y_variable_up_write);
      {
	register indx i;
	Pre_FetchTop(1);

	i = Get_Index_I(1,instruction);
	
/*
#ifdef TRAIL_ALL
  PushOnTrail(w->trail_top, Y(i));
#else
  if ((TAGGED *)yreg < (TAGGED *)w->choice) {
  PushOnTrail(w->trail_top, Y(i));
  }
#endif 
  */
	
	Y(i) = *(s);
	s = PopCont;

	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * UNIFY_X_VALUE_UP
       ************************/
    case UNIFY_X_VALUE_UP:
      unify_x_value_up:
      WriteModeDispatch(unify_x_value_up_write);
      {
	register indx n;
	Pre_FetchTop(1);

	n = Get_Index_I(1,instruction);
	
	UnifyPop(*(s),X(n));
/*      s = PopCont; */
	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * UNIFY_Y_VALUE_UP
       ************************/
    case UNIFY_Y_VALUE_UP:
      unify_y_value_up:
      WriteModeDispatch(unify_y_value_up_write);
      {
	register indx n;
	Pre_FetchTop(1);

	n = Get_Index_I(1,instruction);
	
	UnifyPop(*(s),Y(n));
/*	s = PopCont; */
	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * UNIFY_X_LOCAL_VALUE_UP
       ************************/
    case UNIFY_X_LOCAL_VALUE_UP:
      unify_x_local_value_up:
      WriteModeDispatch(unify_x_local_value_up_write);
      {
	register indx n;
	Pre_FetchTop(1);

	n = Get_Index_I(1,instruction);
	
	Unify(X(n),Ref(s));

	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * UNIFY_Y_LOCAL_VALUE_UP
       ************************/
    case UNIFY_Y_LOCAL_VALUE_UP:
      unify_y_local_value_up:
      WriteModeDispatch(unify_y_local_value_up_write);
      {
	register indx n;
	Pre_FetchTop(1);

	n = Get_Index_I(1,instruction);
	
	Unify(Y(n),Ref(s));

	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * UNIFY_VOID_UP
       ************************/
    case UNIFY_VOID_UP:  
      unify_void_up:
      WriteModeDispatch(unify_void_up_write);
      {
	Pre_FetchTop(0);

	s = PopCont;
	
	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * UNIFY_NIL_UP
       ************************/
    case UNIFY_NIL_UP:
      unify_nil_up:
      WriteModeDispatch(unify_nil_up_write);
      {
	register TAGGED Si;
	Pre_FetchTop(0);
	
	Si = Ref(s);
	DerefLockSwitchHVA(Si,
			   {
			     Bind_Unsafe_Heap(Si,atom_nil,{goto fail;});
			   },
			   {
			     if (Si != atom_nil)
			       goto fail;
			   });
	s = PopCont;
	
	Pre_Execute_Read_Instr(1);
      }
      
#ifdef PARALLEL
      /************************
       * NEW_UNLOCK
       ************************/
    case NEW_UNLOCK:
      new_unlock:
      WriteModeDispatch(new_unlock_write);
      Execute_Read_Instr;
#endif /* PARALLEL */
      
#endif /* NEW_READ */
      
#ifdef NEW_WRITE
      /************************
       * WRITE_LIST_TOP
       ************************/
    case WRITE_LIST_TOP:
      write_list_top:
      goto write_list_top_write;
      
      /************************
       * WRITE_STRUCT_TOP
       ************************/
    case WRITE_STRUCT_TOP:
      write_struct_top:
      goto write_struct_top_write;
      
      /************************
       * PUSH_LIST
       ************************/
    case PUSH_LIST:
      push_list:
      WriteModeDispatch(push_list_write);
      Error("push_write encountered in read mode");
      Get_Index(pc);
      Execute_Read_Instr;
      
      /************************
       * PUSH_STRUCT
       ************************/
    case PUSH_STRUCT:
      push_struct:
      WriteModeDispatch(push_struct_write);
      Error("push_struct encountered in read mode");
      Get_Index(pc);
      Execute_Read_Instr;
      
      /************************
       * PUSH_STRUCT_FUNC
       ************************/
    case PUSH_STRUCT_FUNC:
      push_struct_func:
      WriteModeDispatch(push_struct_func_write);
      Error("push_struct_func encountered in read mode");
      Get_Functor(pc); /* Functor */
      Get_Index(pc);   /* offset  */
      Execute_Read_Instr;
      
      /************************
       * PUSH_FUNCTOR
       ************************/
    case PUSH_FUNCTOR:
      push_functor:
      goto push_functor_write;
      
      /************************
       * PUSH_VOID
       ************************/
    case PUSH_VOID:
      push_void:
      WriteModeDispatch(push_void_write);
      Error("push_void encountered in read mode");
      Execute_Read_Instr;
      
#endif /* NEW_WRITE */
      
#ifdef BOUNDED_Q
      /************************
       * ZEROP
       ************************/
    case ZEROP:
    zerop:
      {
	register indx i;
	register int n;
	Pre_FetchTop(2);
	
	i = Get_Index_I(1,instruction);
	
	DerefNLL(X(i),X(i));
	
	n = GetNumber(X(i));
	
	if(n < 0) goto fail;
	
	if(n == 0) {
	  pc = DispatchLabel(pc,0);
	  Execute_Read_Instr;
	} else {
	  Inc_Label(pc);
	  Pre_Execute_Read_Instr(1);
	}
      }
      
      /************************
       * LISTP
       ************************/
    case LISTP:
    listp:
      {
	register indx i;
	Pre_FetchTop(2);
	
	i = Get_Index_I(1,instruction);
	
	DerefNLL(X(i),X(i));
	
	if(!IsLST(X(i))) {
	  pc = DispatchLabel(pc,0);
	  Execute_Read_Instr;
	} else {
	  Inc_Label(pc);
	  Pre_Execute_Read_Instr(1);
	}
      }
      
      /************************
       * DETERMINISTIC
       ************************/
    case DETERMINISTIC:
    deterministic:
      
      {
	Pre_FetchTop(1);

	if(((void *) w->frame) >= ((void *) w->choice)) {
	  pc = DispatchLabel(pc,0);
	  Execute_Read_Instr;
	} else {
	  Inc_Label(pc);
	  Pre_Execute_Read_Instr(1);
	}
      }
      
      /************************
       * ALLOCATE_STAR
       ************************/
    case ALLOCATE_STAR:
    allocate_star: 
      {
	register environment *newframe;
	Pre_FetchTop(1);
	
	newframe = (environment *) Get_Stack_Top(Get_Index_I(1,instruction));
	
	newframe->cont_env = w->frame;
	newframe->next_instr = w->next_instr;
	SetY(newframe->yreg);
	w->frame = newframe;
	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * REPEAT
       ************************/
    case REPEAT:
    repeat:
      /* this instruction should copy the environment if the clause is
	 undeterministic this far, when done it should jump to the label
	 */
      {
	register int size;
	Pre_FetchInit;

	size = FrameSize(w->next_instr);

	pc = DispatchLabel(pc,0);
	Pre_Fetch;

	if(((void *) w->frame) < ((void *) w->choice)) {
	  register environment *newframe;
	  
	  newframe = (environment *) Get_Local_Stack_Top;

	  if(newframe > (environment *) w->stack_end) {
	    FatalError("Local stack overflow");
	  }

	  newframe->cont_env = w->frame->cont_env;
	  newframe->next_instr = w->frame->next_instr;
	  
	  size--;
	  while(size--) {
	    newframe->yreg[size] = X(size) = Y(size);
	  }
	  
	  SetY(newframe->yreg);
	  w->frame = newframe;
	} else {
	  size--;
	  while(size--) {
	    X(size) = Y(size);
	  }
	}

	Pre_Execute_Read_Instr(0);
      } 
      
      /************************
       * ITERATE_INT
       ************************/
    case ITERATE_INT:
    iterate_int:
      {
	register indx m,n;
	
	m = Get_Index_I_M(1,instruction);
	n = Get_Index_I(2,instruction);
	
	X(m) = Make_Integer(GetNumber(X(m))+1);
	
	if(GetNumber(X(m)) < GetNumber(X(n))) {
	  pc = DispatchLabel(pc,0);
	} else {
	  Inc_Label(pc);
	}
	Execute_Read_Instr;
      }
      /************************
       * ITERATE_INT_STAR
       ************************/
    case ITERATE_INT_STAR:
    iterate_int_star:
      {
	register indx m,n;
	
	m = Get_Index_I_M(1,instruction);
	n = Get_Index_I(2,instruction);
	
	X(m) = Make_Integer(GetNumber(X(m))+1);
	
	if(GetNumber(X(m)) >= GetNumber(X(n))) {
	  pc = DispatchLabel(pc,0);
	} else {
	  Inc_Label(pc);
	}
	Execute_Read_Instr;
      }
      
      /************************
       * ITERATE_LIST
       ************************/
    case ITERATE_LIST:
    iterate_list:
      {
	register indx i;
	register TAGGED Xi;
	
	i = Get_Index_I(1,instruction);
	DerefNLL(Xi,X(i));
	if(IsLST(Xi)) {
	  pc = DispatchLabel(pc,0);
	} else {
	  Inc_Label(pc);
	}
	Execute_Read_Instr;
      }
      
      /************************
       * ITERATE_LIST_STAR
       ************************/
    case ITERATE_LIST_STAR:
    iterate_list_star:
      {
	register indx i;
	register TAGGED Xi;
	
	i = Get_Index_I(1,instruction);
	DerefNLL(Xi,X(i));
	if(!IsLST(Xi)) {
	  pc = DispatchLabel(pc,0);
	} else {
	  Inc_Label(pc);
	}
	Execute_Read_Instr;
      }
      
      /************************
       * ENSURE_LIST_TRY
       ************************/
    case ENSURE_LIST_TRY:
    ensure_list_try:
      {
	register indx i;
	register TAGGED Xi;
	
	i = Get_Index_I_M(1,instruction);
	DerefNLL(Xi,X(i));
	
	if(IsVar(Xi)) {                    /* create choicepoint */
	  register choicepoint *newchoice;
	  
#if defined(TIMESTAMP) || defined(UNBOUND)
	  w->time += TIMEUNIT;
#endif	  
	  newchoice = (choicepoint *)
	    Get_Stack_Top(Get_Index_I(2,instruction));
	  
	  newchoice->trail_top = w->trail_top;
	  newchoice->global_top = H;
	  newchoice->last_choice = w->choice;
	  newchoice->cont_env = w->frame;
	  newchoice->next_instr = w->next_instr;
	  newchoice->next_clause = pc+1;
	  newchoice->arity = 1;
	  newchoice->areg[0] = Xi;
#if defined(TIMESTAMP) || defined(UNBOUND)
	  newchoice->timestamp = w->time;
	  w->uncond = w->time;
#else
	  w->uncond = H;
#endif
	  w->choice = newchoice;
	  AlwaysTrail(Xi);
	  AlwaysBind(Xi,atom_nil);
	}
	
	X(0) = atom_nil;
	pc = DispatchLabel(pc,0);
	Execute_Read_Instr;
      }
      
      /************************
       * ENSURE_LIST_TRUST
       ************************/
    case ENSURE_LIST_TRUST:
    ensure_list_trust:
      {
	register TAGGED lst;
	
	X(0) = w->choice->areg[0];
	w->choice = w->choice->last_choice;
#if defined(TIMESTAMP) || defined(UNBOUND)
	w->uncond = w->choice->timestamp;
#else
	w->uncond = w->choice->global_top;
#endif /* TIMESTAMP */
	
	Make_LST(H,lst);
	CreateHVA(H,w);
	CreateHVA(H,w);
	
	Bind(X(0),lst);
	
	X(0) = lst;
	
	Execute_Read_Instr;
      }
      
      
#endif /* BOUNDED_Q */

#ifdef PARALLEL_BQ
      /************************
       * SPAWN_LEFT_BQ
       ************************/
    case SPAWN_LEFT_BQ:
    spawn_left_bq:
      {
	register indx step, i, g;
	register s32 level;
	Pre_FetchTop(3);

	if(w->global->global_fail) goto done;

	step = Get_Index_I_M(1,instruction);
	i = Get_Index_I_M(2,instruction);
	g = Get_Index_I_M(3,instruction);
      
	if(w->global->scheduling == STATIC)
	  {
	    level = w->level[w->pid-1] + step*w->global->active_workers;
	  }
	else
	  {
	    GrabLevel(level,w) 
	    w->global->sched_level = level + step;
	  }
	    

	w->level[w->pid-1] = level;
	X(i) = Make_Integer(level);

	if (level >= GetNumber(G(g))) {
	    pc = DispatchLabel(pc,0);
	    Execute_Read_Instr;
    	}

	Inc_Label(pc);

	Pre_Execute_Read_Instr(1);
      }
#endif /* PARALLEL_BQ */

#ifdef JUMP_CODE
      /************************
       * JUMP
       ************************/
    case JUMP:
    jump:
      {
	pc = DispatchLabel(pc,0);
	Execute_Read_Instr;
      }
#endif /* BOUNDED_Q || REFORM */

#ifdef REFORM
      /************************
       * BUILD_REC_POSLIST
       ************************/
    case BUILD_REC_POSLIST:
    build_rec_poslist:
      {
	register indx i, n, v, t;
	Pre_FetchTop(4);

	i = Get_Index_I_M(1,instruction); /* recursion list */
	n = Get_Index_I_M(2,instruction); /* length of recursion list */
	v = Get_Index_I_M(3,instruction); /* vectorized recursion list */
	instruction = Get_Instruction(pc);
	t = Get_Index_I(1,instruction); /* last tail of recursion list */

	/* Traverse recursion list and build vector */

	Make_LST(H,X(v));

	/* build vector */
	{
	  register TAGGED List;
	  register s32 len;

	  DerefNLL(List, X(i));

	  PushOnHeap(H,Ref(GetCar(List)));
	  DerefNLL(List,Ref(GetCdr(List)));
	  
	  for(len = 1 ; IsLST(List) ; len++) {
	    PushOnHeap(H,Tagify(H+VARSIZE,LST));
	    PushOnHeap(H,Ref(GetCar(List)));
	    DerefNLL(List,Ref(GetCdr(List)));
	  }

	  PushOnHeap(H,List);

	  /* Last tail */
	  X(t) = List;
	  
	  /* Vector length */
	  X(n) = Make_Integer(len);
	}

	Pre_Execute_Read_Instr(1);
      }
	    
	

      /************************
       * BUILD_POSLIST
       ************************/
    case BUILD_POSLIST:
    build_poslist:
      {
	register indx i, n, v, t;
	Pre_FetchTop(4);

	i = Get_Index_I_M(1,instruction); /* initial list */
	n = Get_Index_I_M(2,instruction); /* length */
	v = Get_Index_I_M(3,instruction); /* vector */
	instruction = Get_Instruction(pc);
	t = Get_Index_I(1,instruction);   /* last tail */

	/* Traverse recursion list and build vector */

	Make_LST(H,X(v));

	/* build vector */
	{
	  register TAGGED List;
	  register s32 len, vectorsize;

	  vectorsize = GetNumber(X(n));

	  DerefNLL(List, X(i));

	  len = 0;

	  for( ; IsLST(List) && len < (vectorsize - 1) ; len++)
	    {
	      PushOnHeap(H,Ref(GetCar(List)));
	      PushOnHeap(H,Tagify(H+VARSIZE,LST));
	      DerefNLL(List,Ref(GetCdr(List)));
	    }

	  if(IsLST(List))
	    {
	      PushOnHeap(H,Ref(GetCar(List)));
	      X(t) = Ref(GetCdr(List));
	      PushOnHeap(H,X(t));
	    }
	  else
	    {
	      if(!unify(List,Tagify(H,LST),w)) goto fail;
	      for( ; len < (vectorsize - 1) ; len++)
		{
		  CreateHVA(H,w);
		  PushOnHeap(H,Tagify(H+VARSIZE,LST));
		}
	      CreateHVA(H,w);
	      LoadHVA(H,X(t),w);
	    }
	}

	Pre_Execute_Read_Instr(1);
      }

      /************************
       * BUILD_POSLIST_VALUE
       ************************/
    case BUILD_POSLIST_VALUE:
    build_poslist_value:
      {
	register indx i, n, v, t;
	Pre_FetchTop(4);

	i = Get_Index_I_M(1,instruction); /* list */
	n = Get_Index_I_M(2,instruction); /* length */
	v = Get_Index_I_M(3,instruction); /* vector */
	instruction = Get_Instruction(pc);
	t = Get_Index_I(1,instruction);   /* last tail */

	/* Match list with vector */

	{
	  TAGGED VectorList, List;
	  s32 vectorsize;
	  
	  DerefNLL(List,X(i));
	  DerefNLL(VectorList,X(v));
	  
	  vectorsize = GetNumber(X(n));
	  
	  while(vectorsize) {
	    if(IsLST(List)) {
	      Unify(Ref(GetCar(List)),Ref(GetCar(VectorList)));
	      DerefNLL(List,Ref(GetCdr(List)));
	      VectorList = Ref(GetCdr(VectorList));
	    } else 
	      break;
	    vectorsize--;
	  }
	  
	  if(vectorsize) {
	    register TAGGED Copy;
	    
	    Make_LST(H,Copy);
	    PushOnHeap(H,Ref(GetCar(VectorList)));
	    vectorsize -= 1;
	    
	    while(vectorsize--) {
	      PushOnHeap(H,Tagify(H+VARSIZE,LST));
	      VectorList = Ref(GetCdr(VectorList));
	      PushOnHeap(H,Ref(GetCar(VectorList)));
	    }
	    
	    LoadHVA(H,X(t),w);
	    if(!unify(List,Copy,w)) goto fail;
	    
	  } else {
	    X(t) = List;
	  }
	}

	Pre_Execute_Read_Instr(1);
      }

      /************************
       * BUILD_NEGLIST
       ************************/
    case BUILD_NEGLIST:
    build_neglist:
      {
	register indx i, n, v, t;
	register s32 vectorsize;
	register TAGGED *htop;
	Pre_FetchTop(4);

	i = Get_Index_I_M(1,instruction); /* last tail element */
	n = Get_Index_I_M(2,instruction); /* length of vector */
	v = Get_Index_I_M(3,instruction); /* head of vector */
	instruction = Get_Instruction(pc);
	t = Get_Index_I(1,instruction);   /* head of list */

	vectorsize = GetNumber(X(n)) - 1;

	X(v) = Tagify(H,LST);

	CreateHVA(H,w);
	PushOnHeap(H,X(i));

	while(vectorsize--)
	  {
	    CreateHVA(H,w);
	    PushOnHeap(H,Tagify(H-3*VARSIZE,LST));
	  }
	    
	X(t) = Tagify(H-2*VARSIZE,LST);
	
	Pre_Execute_Read_Instr(1);
      }

      /************************
       * BUILD_NEGLIST_VALUE
       ************************/
    case BUILD_NEGLIST_VALUE:
    build_neglist_value:
      {
	register indx m, n, v, t, i;
	register s32 vectorsize;
	register TAGGED *htop, *Vector;
	Pre_FetchTop(5);
	
	i = Get_Index_I_M(1,instruction); /* tail element */
	n = Get_Index_I_M(2,instruction); /* length       */
	m = Get_Index_I_M(3,instruction); /* match vector */
	instruction = Get_Instruction(pc);
	v = Get_Index_I_M(1,instruction); /* resulting vector */
	t = Get_Index_I(2,instruction);   /* start of list */


	vectorsize = GetNumber(X(n)) - 1;
	Vector = RemoveTag(X(m),LST);

	X(v) = Tagify(H,LST);

	PushOnHeap(H,Ref(Vector));
	Vector += 2*VARSIZE;
	PushOnHeap(H,X(i));

	while(vectorsize--)
	  {
	    PushOnHeap(H,Ref(Vector));
	    Vector += 2*VARSIZE;
	    PushOnHeap(H,Tagify(H-3*VARSIZE,LST));
	  }
	    
	X(t) = Tagify(H-2*VARSIZE,LST);
	
	Pre_Execute_Read_Instr(1);
      }

      /************************
       * BUILD_VARIABLES
       ************************/
    case BUILD_VARIABLES:
    build_variables:
      {
	register indx i, n, v, t;
	register s32 vectorsize;
	register int x;
	Pre_FetchTop(4);

	i = Get_Index_I_M(1,instruction); /* first variable */
	n = Get_Index_I_M(2,instruction); /* length of recursion list */
	v = Get_Index_I_M(3,instruction); /* variable vector */
	instruction = Get_Instruction(pc);
	t = Get_Index_I(1,instruction);   /* head of last tail */

	vectorsize = GetNumber(X(n));
	
	Make_LST(H,X(v));

	{
	  register TAGGED varvar;
	  DerefNLL(varvar,X(i));
	  
	  if(IsSVA(varvar))
	    {
	      register TAGGED newvar;
	      LoadHVA(H,newvar,w);
	      Bind_SVA(varvar,newvar);
	    }
	  else
	    PushOnHeap(H,varvar);
	}

	for(x = 1 ; x < vectorsize ; x++) {
	  PushOnHeap(H,Tagify(H+VARSIZE,LST));
	  CreateHVA(H,w);
	}

	PushOnHeap(H,Tagify(H+VARSIZE,LST));
	LoadHVA(H,X(t),w);
	CreateHVA(H,w);

	Pre_Execute_Read_Instr(1);
      }

      
      /************************
       * PUT_NTH_HEAD
       ************************/
    case PUT_NTH_HEAD:
    put_nth_head:
      {
	register indx v, l, i;
	register s32 o;
	Pre_FetchTop(4);

	v = Get_Index_I_M(1,instruction);
	l = Get_Index_I_M(2,instruction);
	o = Get_Index(pc);
	i = Get_Index_I_M(3,instruction);
	
	X(i) = Ref(GetNthHead(G(v),(GetNumber(X(l)) + o)));

	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * PUT_NTH_TAIL
       ************************/
    case PUT_NTH_TAIL:
    put_nth_tail:
      {
	register indx v, l, i;
	register s32 o;
	Pre_FetchTop(4);

	v = Get_Index_I_M(1,instruction);
	l = Get_Index_I_M(2,instruction);
	o = Get_Index(pc);
	i = Get_Index_I_M(3,instruction);
	
	X(i) = Ref(GetNthTail(G(v),(GetNumber(X(l)) + o)));

	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * PUT_GLOBAL_ARG
       ************************/
    case PUT_GLOBAL_ARG:
    put_global_arg:
      {
	register indx i, n;
	Pre_FetchTop(2);

	i = Get_Index_I_M(1,instruction);
	n = Get_Index_I(2,instruction);

	X(n) = G(i);

	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * UNIFY_NTH_HEAD
       ************************/
    case UNIFY_NTH_HEAD:
    unify_nth_head:
      WriteModeDispatch(unify_nth_head_write);
      {
	register indx v, l;
	register s32 o;
	Pre_FetchTop(3);

	v = Get_Index_I_M(1,instruction);
	l = Get_Index_I(2,instruction);
	o = Get_Index(pc);
	
	Unify(Ref(s),Ref(GetNthHead(G(v),(GetNumber(X(l))+ o))));
	s += VARSIZE;

	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * UNIFY_NTH_TAIL
       ************************/
    case UNIFY_NTH_TAIL:
    unify_nth_tail:
      WriteModeDispatch(unify_nth_tail_write);
      {
	register indx v, l;
	register s32 o;
	Pre_FetchTop(3);

	v = Get_Index_I_M(1,instruction);
	l = Get_Index_I(2,instruction);
	o = Get_Index(pc);
	
	Unify(Ref(s),Ref(GetNthTail(G(v),(GetNumber(X(l))+o))));
	s += VARSIZE;

	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * UNIFY_GLOBAL_ARG
       ************************/
    case UNIFY_GLOBAL_ARG:
    unify_global_arg:
      WriteModeDispatch(unify_global_arg_write);
      {
	register indx i;
	Pre_FetchTop(1);

	i = Get_Index_I(1,instruction);

	Unify(Ref(s), G(i));
	s += VARSIZE;
	
	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * START_RIGHT_BODY
       ************************/
    case START_RIGHT_BODY:
    start_right_body:
      {
	register int i;
	register TAGGED Num;
	register indx n;

	n = Get_Index_I(1,instruction);

	DerefNLL(Num, X(n));
	if(!IsNUM(Num)) goto fail;
	
	if(w->global->scheduling == DYNAMIC)
	  {
	    w->global->sched_level = GetNumber(Num);
	  }

	/* Reset level count, this could be done in parallel but
         * would require synchronization.
	 */

	for(i=0 ;
	    i < w->global->active_workers ;
	    w->level[i++] = GetNumber(Num));

	goto start_body;
      }

      /************************
       * START_LEFT_BODY
       ************************/
    case START_LEFT_BODY:
    start_left_body:
      {
	register int i;
      
	if(w->global->scheduling == DYNAMIC)
	  {
	    w->global->sched_level = 0;
	  }

	/* Reset level count, this could be done in parallel but
         * would require synchronization.
	 */

	for(i=0 ; i < w->global->active_workers ; w->level[i++] = 0);
      }

    start_body:
      {
	Pre_FetchTop(1);

	w->global->parallel_start.type = W_EXECUTE_CODE;
	w->global->parallel_start.code = DispatchLabel(pc,0);
	w->global->global_fail = FALSE;

	/* Activate worker backtracking, mainly to restore their
	 * heaps.
	 */
	PushOnTrail(w->trail_top,Tagify(NULL,STR));

	ActivateWorkers(w);

	if(w->global->global_fail) goto fail;

#if defined(TIMESTAMP) || defined(UNBOUND)
	{
	  register int i;
	  i = w->global->active_workers;
	  while(i--) {
	    if(w[i].time > w->time) {
	      w->time = w[i].time;
	    }
	  }
	}
#endif	
	
	Inc_Label(pc);
	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * INITIALIZE_RIGHT
       ************************/
    case INITIALIZE_RIGHT:
    initialize_right:
      {
	Pre_FetchTop(2);
#ifdef CACHE_G	
	{ register int i;
	  for(i = 0 ; i < 10 ; i++) {
	    G(i) = w->global->global_regs[i];
	  }
	}
#endif CACHE_G
	if(w->global->scheduling == STATIC)  
	  {
	    register indx step, g;
	    
	    step = Get_Index_I_M(1,instruction);
	    g = Get_Index_I(2,instruction);
	    
	    w->level[w->pid-1] = GetNumber(G(g)) - 1 - step*(w->pid-1) +
	                         step * w->global->active_workers;
	    
	  }
	else
	  {
	    Get_Index_I_M(1,instruction);
	    Get_Index_I(2,instruction);
	  }
	w->direction = RIGHT_BODY;
	Pre_Execute_Read_Instr(1);
      }

      /************************
       * INITIALIZE_LEFT
       ************************/
    case INITIALIZE_LEFT:
    initialize_left:
      {
	Pre_FetchTop(1);
#ifdef CACHE_G	
	{ register int i;
	  for(i = 0 ; i < 10 ; i++) {
	    G(i) = w->global->global_regs[i];
	  }
	}
#endif CACHE_G
	if(w->global->scheduling == STATIC)  
	  {
	    register indx step;
	    
	    step = Get_Index_I_M(1,instruction);
	    
	    w->level[w->pid-1] =step * (w->pid-1) -
	      step * w->global->active_workers;
	  }
	else
	  {
	    Get_Index_I_M(1,instruction);
	  }
	w->direction = LEFT_BODY;
	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * SPAWN_RIGHT
       ************************/
    case SPAWN_RIGHT:
    spawn_right:
      {
	register indx step, i;
	register s32 level;
	Pre_FetchTop(2);

	if(w->global->global_fail) goto done;

	step = Get_Index_I_M(1,instruction);
	i = Get_Index_I_M(2,instruction);

	if(w->global->scheduling == STATIC)
	  {
	    level = w->level[w->pid-1] - step*w->global->active_workers;
	  }
	else
	  {
	    GrabLevel(level,w); 
	    w->global->sched_level = level - step;
	  }
	
	w->level[w->pid-1] = level;
	X(i) = Make_Integer(level);

	if (level < 0) goto done;

	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * SPAWN_LEFT
       ************************/
    case SPAWN_LEFT:
    spawn_left:
      {
	register indx step, i, g;
	register s32 level;
	Pre_FetchTop(3);

	if(w->global->global_fail) goto done;

	step = Get_Index_I_M(1,instruction);
	i = Get_Index_I_M(2,instruction);
	g = Get_Index_I_M(3,instruction);
      
	if(w->global->scheduling == STATIC)
	  {
	    level = w->level[w->pid-1] + step*w->global->active_workers;
	  }
	else
	  {
	    GrabLevel(level,w) 
	    w->global->sched_level = level + step;
	  }

	w->level[w->pid-1] = level;
	X(i) = Make_Integer(level);
	    
	if (level >= GetNumber(G(g))) goto done;

	Pre_Execute_Read_Instr(1);
      }

      /************************
       * AWAIT_LEFTMOST
       ************************/
    case AWAIT_LEFTMOST:
    await_leftmost:
#ifdef ACTIVE_CONSUMER
      WriteModeDispatch(await_leftmost_write);
      {
	register BOOL isfirst = FALSE;
	Pre_FetchTop(0);

	while(!isfirst) {
	  if(w->direction == LEFT_BODY)
	    {
	      IsFirstLeft(isfirst,w);
	    }
	  else
	    {
	      IsFirstRight(isfirst,w);
	    }
	  AwaitCountStat(w);
	  if(w->global->global_fail) goto done;
	}

	Pre_Execute_Read_Instr(1);
      }
#endif /* ACTIVE_CONSUMER */

      /************************
       * AWAIT_NONVAR
       ************************/
    case AWAIT_NONVAR:
    await_nonvar:
#ifdef ACTIVE_CONSUMER
      {
	register indx i;
	register TAGGED Xi;
	Pre_FetchTop(1);

	i = Get_Index_I(1,instruction);
	
	DerefNLL(Xi,X(i));
	
	if(IsHVA(Xi))
	  {
	    /* Wait until first or bound */
	    register BOOL isfirst;

	    AwaitStat(w);

	    while(TRUE)
	      {
		if(w->direction == LEFT_BODY)
		  {
		    IsFirstLeft(isfirst,w);
		  }
		else
		  {
		    IsFirstRight(isfirst,w);
		  }
		DerefNLL(Xi,Xi);
		if(!IsVar(Xi) || isfirst)
		  {
		    X(i) = Xi;
		    Execute_Read_Instr;
		  }
		if(w->global->global_fail) goto done;
		AwaitCountStat(w);
	      }
	  }

	X(i) = Xi;
	
	Pre_Execute_Read_Instr(1);
      }
#endif /* ACTIVE_CONSUMER */

      /************************
       * AWAIT_STRICTLY_NONVAR
       ************************/
    case AWAIT_STRICTLY_NONVAR:
    await_strictly_nonvar:
#ifdef ACTIVE_CONSUMER
      {
	register indx i;
	register TAGGED Xi;
	Pre_FetchTop(1);

	i = Get_Index_I(1,instruction);
	
	DerefNLL(Xi,X(i));
	
	if(IsVar(Xi))
	  {
	    /* Wait until bound or first,
	     * if first genereate runtime error
	     */
	    register BOOL isfirst;

	    AwaitStat(w);

	    while(TRUE)
	      {
		if(w->direction == LEFT_BODY)
		  {
		    IsFirstLeft(isfirst,w);
		  }
		else
		  {
		    IsFirstRight(isfirst,w);
		  }
		DerefNLL(Xi,Xi);
		if(!IsVar(Xi))
		  {
		    X(i) = Xi;
		    Pre_Execute_Read_Instr(1);
		  }
		else if(isfirst)
		  {
		    if(IsSVA(Xi)) {
		      w->global->global_fail = TRUE;
		      Error("await_strictly_nonvar: stack variable");
		      goto done;
		    }
		    w->global->global_fail = TRUE;
		    Error("await_strictly_nonvar: level first, term is variable");
		    goto done;
		  }
		if(w->global->global_fail) goto done;
		AwaitCountStat(w);
	      }
	  }
	else
	  {
	    X(i) = Xi;
	  }

	Pre_Execute_Read_Instr(1);
      }
#endif /* ACTIVE_CONSUMER */

      /************************
       * AWAIT_VARIABLE
       ************************/
    case AWAIT_VARIABLE:
    await_variable:
#ifdef ACTIVE_CONSUMER
      {
	register indx i;
	register TAGGED Xi;
	Pre_FetchTop(1);

	i = Get_Index_I(1,instruction);
	
	DerefNLL(Xi,X(i));
	
	if(IsVar(Xi))
	  {
	    register BOOL isfirst;
	    /* Wait until first, fail if bound */
	    if(IsSVA(Xi)) {
	      Execute_Read_Instr;
	    }

	    AwaitStat(w);
	    while(TRUE)
	      {
		if(w->direction == LEFT_BODY)
		  {
		    IsFirstLeft(isfirst,w);
		  }
		else
		  {
		    IsFirstRight(isfirst,w);
		  }
		DerefNLL(Xi,Xi);
		if(isfirst) 
		  if(IsVar(Xi)) {
		    Pre_Execute_Read_Instr(1);
		  }
		  else
		    goto fail;
		if(!IsVar(Xi)) goto fail;
		if(w->global->global_fail) goto done;
		AwaitCountStat(w);
	      }
	  }
	else
	  goto fail;

	Pre_Execute_Read_Instr(1);
      }

#endif /* ACTIVE_CONSUMER */
      /************************
       * AWAIT_NONVAR_UNIFY
       ************************/
    case AWAIT_NONVAR_UNIFY:
    await_nonvar_unify:
#ifdef ACTIVE_CONSUMER
      WriteModeDispatch(await_nonvar_unify_write);
      {
	register TAGGED Xi;
	Pre_FetchTop(0);

	DerefNLL(Xi,Ref(s));
	
	if(IsHVA(Xi))
	  {
	    /* Wait until first or bound */
	    register BOOL isfirst;

	    AwaitStat(w);

	    while(TRUE)
	      {
		if(w->direction == LEFT_BODY)
		  {
		    IsFirstLeft(isfirst,w);
		  }
		else
		  {
		    IsFirstRight(isfirst,w);
		  }
		DerefNLL(Xi,Xi);
		if(!IsVar(Xi) || isfirst)
		  {
		    Execute_Read_Instr;
		  }
		if(w->global->global_fail) goto done;
		AwaitCountStat(w);
	      }
	  }

	Pre_Execute_Read_Instr(1);
      }
#endif /* ACTIVE_CONSUMER */

      /************************
       * AWAIT_STRICTLY_NONVAR_UNIFY
       ************************/
    case AWAIT_STRICTLY_NONVAR_UNIFY:
    await_strictly_nonvar_unify:
#ifdef ACTIVE_CONSUMER
      WriteModeDispatch(await_strictly_nonvar_unify_write);
      {
	register TAGGED Xi;
	Pre_FetchTop(0);

	DerefNLL(Xi,Ref(s));
	
	if(IsVar(Xi))
	  {
	    /* Wait until bound or first,
	     * if first genereate runtime error
	     */
	    register BOOL isfirst;

	    AwaitStat(w);

	    while(TRUE)
	      {
		if(w->direction == LEFT_BODY)
		  {
		    IsFirstLeft(isfirst,w);
		  }
		else
		  {
		    IsFirstRight(isfirst,w);
		  }
		DerefNLL(Xi,Xi);
		if(!IsVar(Xi))
		  {
		    Pre_Execute_Read_Instr(1);
		  }
		else if(isfirst)
		  {
		    if(IsSVA(Xi)) {
		      w->global->global_fail = TRUE;
		      Error("await_strictly_nonvar: stack variable");
		      goto done;
		    }
		    w->global->global_fail = TRUE;
		    Error("await_strictly_nonvar: level first, term is variable");
		    goto done;
		  }
		if(w->global->global_fail) goto done;
		AwaitCountStat(w);
	      }
	  }

	Pre_Execute_Read_Instr(1);
      }
#endif /* ACTIVE_CONSUMER */

      /************************
       * AWAIT_VARIABLE_UNIFY
       ************************/
    case AWAIT_VARIABLE_UNIFY:
    await_variable_unify:
#ifdef ACTIVE_CONSUMER
      WriteModeDispatch(await_variable_unify_write);
      {
	register TAGGED Xi;
	Pre_FetchTop(0);

	DerefNLL(Xi,Ref(s));
	
	if(IsVar(Xi))
	  {
	    register BOOL isfirst;
	    /* Wait until first, fail if bound */
	    if(IsSVA(Xi)) {
	      Execute_Read_Instr;
	    }

	    AwaitStat(w);
	    while(TRUE)
	      {
		if(w->direction == LEFT_BODY)
		  {
		    IsFirstLeft(isfirst,w);
		  }
		else
		  {
		    IsFirstRight(isfirst,w);
		  }
		DerefNLL(Xi,Xi);
		if(isfirst) 
		  if(IsVar(Xi)) {
		    Pre_Execute_Read_Instr(1);
		  }
		  else
		    goto fail;
		if(!IsVar(Xi)) goto fail;
		if(w->global->global_fail) goto done;
		AwaitCountStat(w);
	      }
	  }
	else
	  goto fail;

	Pre_Execute_Read_Instr(1);
      }

#endif /* ACTIVE_CONSUMER */
      /************************
       * PAR_BUILTIN
       ************************/
    case PAR_BUILTIN:
    par_builtin:
      {
	register int fnk, nr;
	s32 *waitlist;
	
	BuiltinCallStatistics;
	
	fnk = Get_Index_I_M(1,instruction);
	nr = Get_Index_I(2,instruction);

	waitlist = (s32 *) Get_UseArgs(pc);
	pc += nr;

	while(nr--)
	  {
	    register TAGGED Xi;
	    DerefNLL(Xi,X(*waitlist++));
	    
	    if(IsHVA(Xi))
	      {
		/* Wait until first or bound */
		register BOOL isfirst;
		
		AwaitStat(w);

		while(TRUE)
		  {
		    if(w->direction == LEFT_BODY)
		      {
			IsFirstLeft(isfirst,w);
		      }
		    else
		      {
			IsFirstRight(isfirst,w);
		      }
		    DerefNLL(Xi,Xi);

		    if(isfirst || !IsVar(Xi)) break;

		    if(w->global->global_fail) goto done;
		  }
	      }
	  }
	
	StoreHeap;
	
	if ((GetInlineFnk(fnk))(w,(s32 *)Get_UseArgs(pc)) == FALSE) goto fail;
	
	LoadHeap;
	
	pc += GetInlineArity(fnk);
	Execute_Read_Instr;
      }

      /************************
       * PAR_INLINE
       ************************/
    case PAR_INLINE:
    par_inline: 
      {
	register int fnk, nr;
	s32 *waitlist;
	code *faillabel;
	
	BuiltinCallStatistics;
	
	fnk = Get_Index_I_M(1,instruction);
	nr = Get_Index_I(2,instruction);

	faillabel = DispatchLabel(pc,0);
	Inc_Label(pc);

	waitlist = (s32 *) Get_UseArgs(pc);
	pc += nr;

	while(nr--)
	  {
	    register TAGGED Xi;
	    DerefNLL(Xi,X(*waitlist++));
	    
	    if(IsHVA(Xi))
	      {
		/* Wait until first or bound */
		register BOOL isfirst;
		
		while(TRUE)
		  {
		    if(w->direction == LEFT_BODY)
		      {
			IsFirstLeft(isfirst,w);
		      }
		    else
		      {
			IsFirstRight(isfirst,w);
		      }

		    DerefNLL(Xi,Xi);

		    if(isfirst || !IsVar(Xi)) break;

		    if(w->global->global_fail) goto done;
		  }
	      }
	  }
	
	StoreHeap;
	
	if ((GetInlineFnk(fnk))(w,(s32 *)Get_UseArgs(pc)) == FALSE) {
	  pc = faillabel;
	} else {
	  pc += GetInlineArity(fnk);
	}
	
	LoadHeap;
	
	Execute_Read_Instr;
      }

      /************************
       * LOCK_AND_GET_LIST
       ************************/
    case LOCK_AND_GET_LIST:
    lock_and_get_list:
      {
	register TAGGED Xi;
	register indx i,n;      
	Pre_FetchTop(2);
	
	i = Get_Index_I_M(1,instruction);
	n = Get_Index_I(2,instruction);
	
	Xi = X(i);
	DerefLockSwitch(Xi,
			{
			  register TAGGED l;
			  
			  Make_LST_A(H,s,X(n));
#ifdef LOCKING
			  X(i) = Xi;
#else
			  Drop_P_Lock(Xi,Xi);
#endif
			  Pre_Execute_Write_Instr(1);
			},
			{
			  if(IsLST(Xi)) {
			    X(i) = Xi;
			    X(n) = Xi;
			    s = RemoveTag(Xi,LST);
			    Pre_Execute_Read_Instr(1);
			  } else
			    goto fail;
			});
      }

      /************************
       * LOCK_AND_GET_STRUCTURE
       ************************/
    case LOCK_AND_GET_STRUCTURE:
    lock_and_get_structure:
      {
	register TAGGED Xi,new, f;
	register indx i,n;
	Pre_FetchTop(3);

	f = Get_Functor(pc);
	i = Get_Index_I_M(1,instruction);
	n = Get_Index_I(2,instruction);
	
	
	Xi = X(i);
	DerefLockSwitch(Xi,
			{
			  Make_STR_A(H,s,X(n),f);
#ifdef LOCKING
			  X(i) = Xi;
#else  /* LOCKING */
			  Drop_P_Lock(Xi,Xi);
#endif /* LOCKING */
			  Pre_Execute_Write_Instr(1);
			},
			{
			  if(IsSTR(Xi)) {
			    if(GetFunctor(Xi) == f) {
			      X(i) = Xi;
			      X(n) = Xi;
			      s = GetArg(Xi,0);
			      Pre_Execute_Read_Instr(1);
			    } else
			      goto fail;
			  } else
			    goto fail;
			});
      }

      /************************
       * UNLOCK
       ************************/
    case UNLOCK:
    unlock:
      {
	register indx i, n;
	Pre_FetchTop(2);

	i = Get_Index_I_M(1,instruction);
	n = Get_Index_I(2,instruction);

#ifdef LOCKING
	if (X(i) != X(n)) {
	  Bind_HVA(X(i),X(n));
	}
#else  /* LOCKING */
	if (X(i) != X(n)) {
	  Unify1(X(i),X(n));
	}
#endif /* LOCKING */
	
	Pre_Execute_Read_Instr(1);
      }
#endif /* REFORM */

      /************************
       * HALT
       ************************/
    case HALT:
    halt:
      goto done;
      
      /************************
       * NOOP
       ************************/
    case NOOP:
    noop:
      Execute_Read_Instr;
      
      /************************
       * END_OF_PRED
       ************************/
    case END_OF_PRED:
    end_of_pred:
      Execute_Read_Instr;
      
    default: 
      PL_Print2(stderr,"default in instruction switch, %d does not exist\n",
		Get_Instr(instruction));
      luther_exit(0); 
    }
  
  
  /*********************
   * write instructions
   *********************/
  
 write_instructions:
  
  DisplayInstr("write");
  
  instruction = Get_Code(pc);
  
  switch(Get_Instr(instruction))
    {
	/************************
	 * DUMMY_INSTRUCTION
	 ************************/
    case DUMMY_INSTRUCTION:
      Execute_Write_Instr;
      
      /************************
       * SWITCH_ON_TERM
       ************************/
    case SWITCH_ON_TERM:
      goto switch_on_term;
      
      /************************
       * SWITCH_ON_CONSTANT
       ************************/
    case SWITCH_ON_CONSTANT:
      goto switch_on_constant;
      
      /************************
       * SWITCH_ON_STRUCTURE
       ************************/
    case SWITCH_ON_STRUCTURE:
      goto switch_on_structure;
      
      /************************
       * TRY
       ************************/
    case TRY:
      goto try;
      
      /************************
       * RETRY
       ************************/
    case RETRY:
      goto retry;
      
      /************************
       * TRUST
       ************************/
    case TRUST:
      goto trust;
      
      /************************
       * TRY_ME_ELSE
       ************************/
    case TRY_ME_ELSE:
      goto try_me_else;
      
      /************************
       * RETRY_ME_ELSE
       ************************/
    case RETRY_ME_ELSE:
      goto retry_me_else;
      
      /************************
       * TRUST_ME
       ************************/
    case TRUST_ME:
      goto trust_me;
      
      /************************
       * CHOICE_X
       ************************/
    case CHOICE_X:
      goto choice_x;
      
      /************************
       * CHOICE_Y
       ************************/
    case CHOICE_Y:
      goto choice_y;
      
      /************************
       * CUT
       ************************/
    case CUT:
      goto cut;
      
      /************************
       * CUT_X
       ************************/
    case CUT_X:
      goto cut_x;
      
      /************************
       * CUT_Y
       ************************/
    case CUT_Y:
      goto cut_y;
      
      /************************
       * INLINE
       ************************/
    case INLINE:
      goto in_line;
      /************************
       * BUILTIN
       ************************/
    case BUILTIN:
      goto builtin;
      
      /************************
       * META_CALL
       ************************/
    case META_CALL:
      goto meta_call;
      
      /************************
       * META_EXECUTE
       ************************/
    case META_EXECUTE:
      goto meta_execute;
      
#ifdef EXTENDED_CALL
      /************************
       * VAR_CALL
       ************************/
    case VAR_CALL:
      goto var_call;
      
      /************************
       * VAR_EXECUTE
       ************************/
    case VAR_EXECUTE:
      goto var_execute;
#endif /* EXTENDED_CALL */
      
      /************************
       * REQUIRE
       ************************/
    case REQUIRE:
      /************************
       * REQUIRE_USING
       ************************/
    case REQUIRE_USING:
      FatalError("require encountered in write mode");
      break;
      
      /************************
       * ALLOCATE
       ************************/
    case ALLOCATE:
      goto allocate;
      
      /************************
       * ALLOCATE2
       ************************/
    case ALLOCATE2:
      goto allocate2;
      
      /************************
       * DEALLOCATE
       ************************/
    case DEALLOCATE:
      goto deallocate;
      
      /************************
       * INIT
       ************************/
    case INIT:
      goto init;
      
      /************************
       * CALL
       ************************/
    case CALL:
      goto call;
      
      /************************
       * EXECUTE
       ************************/
    case EXECUTE:
      goto execute;
      
      /************************
       * PROCEED
       ************************/
    case PROCEED:
      goto proceed;
      
      /************************
       * FAIL
       ************************/
    case FAIL:
      goto fail;
      
      /************************
       * GET_X_VARIABLE
       ************************/
    case GET_X_VARIABLE:
      goto get_x_variable;
      
      /************************
       * GET_Y_VARIABLE
       ************************/
    case GET_Y_VARIABLE:
      goto get_y_variable;
      
      /************************
       * GET_Y_FIRST_VALUE
       ************************/
    case GET_Y_FIRST_VALUE:
      goto get_y_first_value;
      
      /************************
       * GET_X_VALUE
       ************************/
    case GET_X_VALUE:
      goto get_x_value;
      
      /************************
       * GET_Y_VALUE
       ************************/
    case GET_Y_VALUE:
      goto get_y_value;
      
      /************************
       * GET_CONSTANT
       ************************/
    case GET_CONSTANT:
      goto get_constant;
      
      /************************
       * GET_NIL
       ************************/
    case GET_NIL:
      goto get_nil;
      
      /************************
       * GET_STRUCTURE
       ************************/
    case GET_STRUCTURE:
      goto get_structure;
      
      /************************
       * GET_LIST
       ************************/
    case GET_LIST:
      goto get_list;
      
      /************************
       * GET_CONSTANT_X0
       ************************/
    case GET_CONSTANT_X0:
      goto get_constant_x0;
      
      /************************
       * GET_NIL_X0
       ************************/
    case GET_NIL_X0:
      goto get_nil_x0;
      
      /************************
       * GET_STRUCTURE_X0
       ************************/
    case GET_STRUCTURE_X0:
      goto get_structure_x0;
      
      /************************
       * GET_LIST_X0
       ************************/
    case GET_LIST_X0:
      goto get_list_x0;
      
      /************************
       * PUT_X_VOID
       ************************/
    case PUT_X_VOID:
      goto put_x_void;
      
      /************************
       * PUT_Y_VOID
       ************************/
    case PUT_Y_VOID:
      goto put_y_void;
      
      /************************
       * PUT_X_VARIABLE
       ************************/
    case PUT_X_VARIABLE:
      goto put_x_variable;
      
      /************************
       * PUT_Y_VARIABLE
       ************************/
    case PUT_Y_VARIABLE:
      goto put_y_variable;
      
      /************************
       * PUT_X_VALUE
       ************************/
    case PUT_X_VALUE:
      goto put_x_value;
      
      /************************
       * PUT_Y_VALUE
       ************************/
    case PUT_Y_VALUE:
      goto put_y_value;
      
      /************************
       * PUT_X_UNSAFE_VALUE
       ************************/
    case PUT_X_UNSAFE_VALUE:
      goto put_x_unsafe_value;
      
      /************************
       * PUT_Y_UNSAFE_VALUE
       ************************/
    case PUT_Y_UNSAFE_VALUE:
      goto put_y_unsafe_value;
      
      /************************
       * PUT_CONSTANT
       ************************/
    case PUT_CONSTANT:
      goto put_constant;
      
      /************************
       * PUT_NIL
       ************************/
    case PUT_NIL:
      goto put_nil;
      
      /************************
       * PUT_STRUCTURE
       ************************/
    case PUT_STRUCTURE:
      goto put_structure;
      
      /************************
       * PUT_LIST
       ************************/
    case PUT_LIST:
      goto put_list;
      
      /************************
       * UNIFY_VOID
       ************************/
    case UNIFY_VOID: 
    unify_void_write:
      {
	register indx n;
	Pre_FetchTop(1);
	
	n = Get_Index_I(1,instruction); 
	
	do {
#ifdef NEW_READ
	  CreateHVA(s,w);
#else 
	  CreateHVA(H,w);
#endif 
	} while(--n);
	
	Pre_Execute_Write_Instr(1);
      }
      
      /************************
       * UNIFY_X_VARIABLE
       ************************/
    case UNIFY_X_VARIABLE:
    unify_x_variable_write:
      {
	register indx n;
	Pre_FetchTop(1);

	n = Get_Index_I(1,instruction);
#ifdef NEW_READ
	LoadHVA(s,X(n),w);
#else
	LoadHVA(H,X(n),w);
#endif      
	Pre_Execute_Write_Instr(1);
      }
      
      /************************
       * UNIFY_Y_VARIABLE
       ************************/
    case UNIFY_Y_VARIABLE:
    unify_y_variable_write:
      {
	register indx n;
	Pre_FetchTop(1);

	n = Get_Index_I(1,instruction);
#ifdef NEW_READ
	LoadHVA(s,Y(n),w);
#else
	LoadHVA(H,Y(n),w);
#endif      
	Pre_Execute_Write_Instr(1);
      }
      
      /************************
       * UNIFY_Y_FIRST_VALUE
       ************************/
    case UNIFY_Y_FIRST_VALUE:
    unify_y_first_value_write:
      {
	register indx n;
	Pre_FetchTop(1);

	n = Get_Index_I(1,instruction);
#ifdef NEW_READ
	LoadHVA(s,Y(n),w);
#else
	LoadHVA(H,Y(n),w);
#endif
	Pre_Execute_Write_Instr(1);
      }
      
      /************************
       * UNIFY_X_VALUE
       ************************/
    case UNIFY_X_VALUE:
    unify_x_value_write:
      {
	register indx n;
	Pre_FetchTop(1);

	n = Get_Index_I(1,instruction);
#ifdef NEW_READ
	PushOnHeap(s,X(n));
#else
	PushOnHeap(H,X(n));
#endif
	Pre_Execute_Write_Instr(1);
      }
      
      /************************
       * UNIFY_Y_VALUE
       ************************/
    case UNIFY_Y_VALUE:
    unify_y_value_write:
      {
	register indx n;
	Pre_FetchTop(1);

	n = Get_Index_I(1,instruction);

#ifdef NEW_READ
	PushOnHeap(s,Y(n));
#else
	PushOnHeap(H,Y(n));
#endif

	Pre_Execute_Write_Instr(1);
      }
      
      /************************
       * UNIFY_X_LOCAL_VALUE
       ************************/
    case UNIFY_X_LOCAL_VALUE:
    unify_x_local_value_write:
      {
	register indx n;
	Pre_FetchTop(1);
	
	n = Get_Index_I(1,instruction);
#ifdef NEW_READ
	WriteLocalValue(s,X(n));
#else
	WriteLocalValue(H,X(n));
#endif
	Pre_Execute_Write_Instr(1);
      }
      
      /************************
       * UNIFY_Y_LOCAL_VALUE
       ************************/
    case UNIFY_Y_LOCAL_VALUE:
    unify_y_local_value_write:
      {
	register indx n;
	Pre_FetchTop(1);
	
	n = Get_Index_I(1,instruction);
	
#ifdef NEW_READ
	WriteLocalValue(s,Y(n));
#else
	WriteLocalValue(H,Y(n));
#endif
	
	Pre_Execute_Write_Instr(1);
      }
      
      /************************
       * UNIFY_CONSTANT
       ************************/
    case UNIFY_CONSTANT:
    unify_constant_write:
      { 
	register TAGGED c;
	Pre_FetchTop(1);
	
	c = Get_Tagged(pc);
	
#ifdef NEW_READ
	PushOnHeap(s,c);
#else
	PushOnHeap(H,c);
#endif
	Pre_Execute_Write_Instr(1);
      }
      
      /************************
       * UNIFY_NIL
       ************************/
    case UNIFY_NIL:
    unify_nil_write:
      {
	Pre_FetchTop(0);
#ifdef NEW_READ
	PushOnHeap(s,atom_nil);
#else
	PushOnHeap(H,atom_nil);
#endif      
	Pre_Execute_Write_Instr(1);
      }
      
      /************************
       * UNIFY_STRUCTURE
       ************************/
    case UNIFY_STRUCTURE:
    unify_structure_write:
      { 
	register TAGGED f, *tmp;
	Pre_FetchTop(1);
	
	f = Get_Functor(pc);
	
#ifdef NEW_READ
	Make_STR_Alloc(H,s,*s,f);
#else
	tmp = H + VARSIZE;
	PushOnHeap(H,Tagify(tmp,STR));
	PushOnHeapF(H,f);
#endif
      Pre_Execute_Write_Instr(1);
      }
      
      /************************
       * UNIFY_LIST
       ************************/
    case UNIFY_LIST:
    unify_list_write:
      { 
	register TAGGED *tmp;
	Pre_FetchTop(0);
	
#ifdef NEW_READ
	Make_LST_S(H,s,*s);
#else      
	tmp = H + VARSIZE;
	PushOnHeap(H,Tagify(tmp,LST));
#endif
	Pre_Execute_Write_Instr(1);
      }

#ifdef JUMP_CALL
      /************************
       * CJUMP
       ************************/
    case CJUMP:
      goto cjump;

      /************************
       * EJUMP
       ************************/
    case EJUMP:
      goto ejump;

#endif /* JUMP_CALL */      

#ifdef NEW_READ
      
      /************************
       * READ_LIST_TOP
       ************************/
    case READ_LIST_TOP:
      goto read_list_top;
      
      /************************
       * READ_STRUCT_TOP
       ************************/
    case READ_STRUCT_TOP:
      goto read_struct_top;
      
      /************************
       * READ_LIST
       ************************/
    case READ_LIST:
    read_list_write:
      {
	Pre_FetchTop(1);

	Get_Index_I(1,instruction);
	PushCont(s+VARSIZE);
	Make_LST_S(H,s,*s);

	Pre_Execute_Write_Instr(1);
      }
      
      /************************
       * READ_STRUCT
       ************************/
    case READ_STRUCT:
    read_struct_write:
      {
	register TAGGED f;
	Pre_FetchTop(2);
	  
	f = Get_Functor(pc);
	Get_Index_I(1,instruction);
	    
	PushCont(s+VARSIZE);
	    
	Make_STR_Alloc(H,s,*s,f);
	    
	Pre_Execute_Write_Instr(1);
      }
      
      /************************
       * READ_LIST_TAIL
       ************************/
    case READ_LIST_TAIL:
    read_list_tail_write:
      {
	Pre_FetchTop(1);

	Get_Index_I(1,instruction);
#ifdef PARALLEL	  
	if(k <= 0) {
	  register TAGGED x,y,lst;
	  x = (TAGGED) PopCont;
	  y = (TAGGED) PopCont;
	  Bind_Unsafe(y,x,{goto fail;});
	  InitHVA(s,x,w);
	  Make_LST_S(H,s,lst);
	  PushCont(x);
	  PushCont(lst);
	} else
#endif
	{
	  Make_LST_S(H,s,*s);
	}
	Pre_Execute_Write_Instr(1);
      }
      
      /************************
       * READ_STRUCT_TAIL
       ************************/
    case READ_STRUCT_TAIL:
    read_struct_tail_write:
      {
	register TAGGED f;
	Pre_FetchTop(2);
	  
	f = Get_Functor(pc);
	Get_Index_I(1,instruction);
	    
#ifdef PARALLEL
	if(k <= 0) {
	    register TAGGED x,y,str;
	    x = (TAGGED) PopCont;
	    y = (TAGGED) PopCont;
	    Bind_Unsafe(y,x,{goto fail;});
	    InitHVA(s,x,w);
	    Make_STR_Alloc(H,s,str,f);
	    PushCont(x);
	    PushCont(str);
	} else 
#endif /* PARALELL */
	  {
	    Make_STR_Alloc(H,s,*s,f);
	  }
	Pre_Execute_Write_Instr(1);
      }
      
      /************************
       * UNIFY_CONSTANT_UP
       ************************/
    case UNIFY_CONSTANT_UP:
    unify_constant_up_write:
      { 
	register TAGGED c;
	
	c = Get_Tagged(pc);
	
	*(s) = c;
	
	DispatchUp(k);
      }
      
      /************************
       * UNIFY_X_VARIABLE_UP
       ************************/
    case UNIFY_X_VARIABLE_UP:
    unify_x_variable_up_write:
      {
	register indx n;
	
	n = Get_Index_I(1,instruction);
	
	LoadHVA(s,X(n),w);
	
	DispatchUp(k);
      }
      
      /************************
       * UNIFY_Y_VARIABLE_UP
       ************************/
    case UNIFY_Y_VARIABLE_UP:
    unify_y_variable_up_write:
      {
	register indx n;
	
	n = Get_Index_I(1,instruction);
	
	LoadHVA(s,Y(n),w);
	
	DispatchUp(k);
      }
      /************************
       * UNIFY_X_VALUE_UP
       ************************/
    case UNIFY_X_VALUE_UP:
    unify_x_value_up_write:
      {
	register indx n;
	
	n = Get_Index_I(1,instruction);
	
	*(s) = X(n);
	
	DispatchUp(k);
      }
      /************************
       * UNIFY_Y_VALUE_UP
       ************************/
    case UNIFY_Y_VALUE_UP:
    unify_y_value_up_write:
      {
	register indx n;
	
	n = Get_Index_I(1,instruction);
	
	*(s) = Y(n);
	
	DispatchUp(k);
      }
      /************************
       * UNIFY_X_LOCAL_VALUE_UP
       ************************/
    case UNIFY_X_LOCAL_VALUE_UP:
    unify_x_local_value_up_write:
      {
	register indx n;
	
	n = Get_Index_I(1,instruction);
	
	WriteLocalValue(s,X(n));
	
	DispatchUp(k);
      }
      /************************
       * UNIFY_Y_LOCAL_VALUE_UP
       ************************/
    case UNIFY_Y_LOCAL_VALUE_UP:
    unify_y_local_value_up_write:
      {
	register indx n;
	
	n = Get_Index_I(1,instruction);
	
	WriteLocalValue(s,Y(n));
	
	DispatchUp(k);
      }
      /************************
       * UNIFY_VOID_UP
       ************************/
    case UNIFY_VOID_UP:  
    unify_void_up_write:
      {
	register indx n;
	
	n = Get_Index_I(1,instruction); 
	
	do {CreateHVA(s,w);} while(--n);
	
	DispatchUp(k);
      }
      
      /************************
       * UNIFY_NIL_UP
       ************************/
    case UNIFY_NIL_UP:
    unify_nil_up_write:
      { 
	*(s) = atom_nil;
	
	DispatchUp(k);
      }
      
#ifdef PARALLEL
      /************************
       * NEW_UNLOCK
       ************************/
    case NEW_UNLOCK:
    new_unlock_write:
      {
	  register TAGGED x,y;
	  Pre_FetchTop(0);

	  x = (TAGGED) PopCont;
	  y = (TAGGED) PopCont;
	  Bind_Unsafe(y,x,{goto fail;});

	  Pre_Execute_Write_Instr(1);
      }
#endif /* PARALLEL */      
      
#endif /* NEW_READ */   
      
#ifdef NEW_WRITE
      /************************
       * WRITE_LIST_TOP
       ************************/
    case WRITE_LIST_TOP:
    write_list_top_write:
      {
	register indx n;
	Pre_FetchTop(1);
	
	InitCont;
	
	n = Get_Index_I(1,instruction);
#ifdef NEW_READ_NOC
	k = NULL;
#else
	k = MAX_WRITE_K;
#endif
	
	Make_LST_S(H,s,X(n));
	
	Pre_Execute_Write_Instr(1);
      }
      
      /************************
       * WRITE_STRUCT_TOP
       ************************/
    case WRITE_STRUCT_TOP:
    write_struct_top_write:
      {
	  register TAGGED f;
	  register indx i;
	  Pre_FetchTop(2);
	  
	  InitCont;
	  
	  f = Get_Functor(pc);
	  i = Get_Index_I(1,instruction);
#ifdef NEW_READ_NOC
	  k = NULL;
#else
	  k = MAX_WRITE_K;
#endif
	    
	  Make_STR_Alloc(H,s,X(i),f);
	    
	  Pre_Execute_Write_Instr(1);
	}
      
      /************************
       * PUSH_LIST
       ************************/
    case PUSH_LIST:
    push_list_write:
      {
	register indx n;
	Pre_FetchTop(1);
	
	n = Get_Index(pc);
	
#ifdef NEW_READ
	*(s) = Tagify(s+n*VARSIZE,LST);
	s += VARSIZE;
	H += VARSIZE;
#else
	*(H) = Tagify(H+n*VARSIZE,LST);
	H += VARSIZE;
#endif
	Pre_Execute_Write_Instr(1);
      }
      /************************
       * PUSH_STRUCT
       ************************/
    case PUSH_STRUCT:
    push_struct_write:
      {
	  register indx n;
	  Pre_FetchTop(2);
	  
	  n = Get_Index(pc);
	    
#ifdef NEW_READ
	    *(s) = Tagify(s+n*VARSIZE,STR);
	    s += VARSIZE;
	    H += Get_Index_I(1,instruction)*VARSIZE + FUNCSIZE;
#else
	    *(H) = Tagify(H+n*VARSIZE,STR);
	    H += VARSIZE;
	    Get_Index_I(1,instruction);
#endif
	    Pre_Execute_Write_Instr(1);
	  }
      
      /************************
       * PUSH_STRUCT_FUNC
       ************************/
    case PUSH_STRUCT_FUNC:
    push_struct_func_write:
      {
	  register TAGGED f, *ns;
	  register indx n;
	  Pre_FetchTop(2);
	  
	  f = Get_Functor(pc);
	    n = Get_Index(pc);
	    
	    ns = s+n*VARSIZE;
	    
	    *s = Tagify(ns,STR);
	    s += VARSIZE;
	    *(ns) = f;
	    H += ArityOf(f)*VARSIZE+FUNCSIZE;
	    
	    Pre_Execute_Write_Instr(1);
	  }
      
      /************************
       * PUSH_FUNCTOR
       ************************/
    case PUSH_FUNCTOR:
    push_functor_write:
      {
	Pre_FetchTop(1);

	*s = Get_Functor(pc);
	s += FUNCSIZE;
	
	Pre_Execute_Write_Instr(1);
      }
      
      /************************
       * PUSH_VOID
       ************************/
    case PUSH_VOID:
    push_void_write:
      {
	Pre_FetchTop(1);

	s += VARSIZE;

	Pre_Execute_Write_Instr(1);
      }
      
#endif /* NEW_WRITE */
      
#ifdef BOUNDED_Q
      /************************
       * ZEROP
       ************************/
    case ZEROP:
      goto zerop;
      
      /************************
       * LISTP
       ************************/
    case LISTP:
      goto listp;
      
      /************************
       * DETERMINISTIC
       ************************/
    case DETERMINISTIC:
      goto deterministic;
      
      /************************
       * ALLOCATE_STAR
       ************************/
    case ALLOCATE_STAR:
      goto allocate_star;
      
      /************************
       * REPEAT
       ************************/
    case REPEAT:
      goto repeat;
      
      /************************
       * ITERATE_INT
       ************************/
    case ITERATE_INT:
      goto iterate_int;
      
      /************************
       * ITERATE_INT_STAR
       ************************/
    case ITERATE_INT_STAR:
      goto iterate_int_star;
      
      /************************
       * ITERATE_LIST
       ************************/
    case ITERATE_LIST:
      goto iterate_list;
      
      /************************
       * ITERATE_LIST_STAR
       ************************/
    case ITERATE_LIST_STAR:
      goto iterate_list_star;
      
      /************************
       * ENSURE_LIST_TRY
       ************************/
    case ENSURE_LIST_TRY:
      goto ensure_list_try;
      
      /************************
       * ENSURE_LIST_TRUST
       ************************/
    case ENSURE_LIST_TRUST:
      goto ensure_list_trust;
      
#endif /* BOUNDED_Q */

#ifdef PARALLEL_BQ
      /************************
       * SPAWN_LEFT_BQ
       ************************/
    case SPAWN_LEFT_BQ:
      goto spawn_left_bq;
#endif /* PARALLEL_BQ */

#ifdef JUMP_CODE
      /************************
       * JUMP
       ************************/
    case JUMP:
      goto jump;

#endif /* BOUNDED_Q || REFORM */
      
#ifdef REFORM
      /************************
       * BUILD_REC_POSLIST
       ************************/
    case BUILD_REC_POSLIST:
      goto build_rec_poslist;
      
      /************************
       * BUILD_POSLIST
       ************************/
    case BUILD_POSLIST:
      goto build_poslist;
      
      /************************
       * BUILD_POSLIST_VALUE
       ************************/
    case BUILD_POSLIST_VALUE:
      goto build_poslist_value;
      
      /************************
       * BUILD_NEGLIST
       ************************/
    case BUILD_NEGLIST:
      goto build_neglist;
      
      /************************
       * BUILD_NEGLIST_VALUE
       ************************/
    case BUILD_NEGLIST_VALUE:
      goto build_neglist_value;
      
      /************************
       * BUILD_VARIABLES
       ************************/
    case BUILD_VARIABLES:
      goto build_variables;
      
      /************************
       * PUT_NTH_HEAD
       ************************/
    case PUT_NTH_HEAD:
      goto put_nth_head;
      
      /************************
       * PUT_NTH_TAIL
       ************************/
    case PUT_NTH_TAIL:
      goto put_nth_tail;
      
      /************************
       * PUT_GLOBAL_ARG
       ************************/
    case PUT_GLOBAL_ARG:
      goto put_global_arg;
      
      /************************
       * UNIFY_NTH_HEAD
       ************************/
    case UNIFY_NTH_HEAD:
    unify_nth_head_write:
      {
	register indx v, l;
	register s32 o;
	Pre_FetchTop(3);

	v = Get_Index_I_M(1,instruction);
	l = Get_Index_I(2,instruction);
	o = Get_Index(pc);
	
#ifdef NEW_READ
	PushOnHeap(s,Ref(GetNthHead(G(v),(GetNumber(X(l)) + o))));
#else	
	PushOnHeap(H,Ref(GetNthHead(G(v),(GetNumber(X(l)) + o))));
#endif /* NEW_READ */

	Pre_Execute_Write_Instr(1);
      }
      
      /************************
       * UNIFY_NTH_TAIL
       ************************/
    case UNIFY_NTH_TAIL:
    unify_nth_tail_write:
      {
	register indx v, l;
	register s32 o;
	Pre_FetchTop(3);

	v = Get_Index_I_M(1,instruction);
	l = Get_Index_I(2,instruction);
	o = Get_Index(pc);
	
#ifdef NEW_READ
	PushOnHeap(s,Ref(GetNthTail(G(v),(GetNumber(X(l)) + o))));
#else
	PushOnHeap(H,Ref(GetNthTail(G(v),(GetNumber(X(l)) + o))));
#endif /* NEW_READ */

	Pre_Execute_Write_Instr(1);
      }
      
      /************************
       * UNIFY_GLOBAL_ARG
       ************************/
    case UNIFY_GLOBAL_ARG:
    unify_global_arg_write:
      {
	register indx i;
	Pre_FetchTop(1);

	i = Get_Index_I(1,instruction);

#ifdef NEW_READ
	PushOnHeap(s,G(i));
#else
	PushOnHeap(H,G(i));
#endif
	
	Pre_Execute_Read_Instr(1);
      }
      
      /************************
       * START_RIGHT_BODY
       ************************/
    case START_RIGHT_BODY:
      goto start_right_body;
      
      /************************
       * START_LEFT_BODY
       ************************/
    case START_LEFT_BODY:
      goto start_left_body;
      
      /************************
       * INITIALIZE_RIGHT
       ************************/
    case INITIALIZE_RIGHT:
      goto initialize_right;
      
      /************************
       * INITIALIZE_LEFT
       ************************/
    case INITIALIZE_LEFT:
      goto initialize_left;
      
      /************************
       * SPAWN_RIGHT
       ************************/
    case SPAWN_RIGHT:
      goto spawn_right;
      
      /************************
       * SPAWN_LEFT
       ************************/
    case SPAWN_LEFT:
      goto spawn_left;

      /************************
       * AWAIT_LEFTMOST
       ************************/
    case AWAIT_LEFTMOST:
    await_leftmost_write:
#ifdef ACTIVE_CONSUMER
      {
	register BOOL isfirst = FALSE;
	Pre_FetchTop(0);

	while(!isfirst) {
	  if(w->direction == LEFT_BODY)
	    {
	      IsFirstLeft(isfirst,w);
	    }
	  else
	    {
	      IsFirstRight(isfirst,w);
	    }
	  AwaitCountStat(w);
	  if(w->global->global_fail) goto done;
	}

	Pre_Execute_Write_Instr(1);
      }
#endif /* ACTIVE_CONSUMER */

      /************************
       * AWAIT_NONVAR
       ************************/
    case AWAIT_NONVAR:
      goto await_nonvar;
      
      /************************
       * AWAIT_STRICTLY_NONVAR
       ************************/
    case AWAIT_STRICTLY_NONVAR:
      goto await_strictly_nonvar;
      
      /************************
       * AWAIT_VARIABLE
       ************************/
    case AWAIT_VARIABLE:
    await_variable_write:
      goto await_variable;

      /************************
       * AWAIT_NONVAR_UNIFY
       ************************/
    case AWAIT_NONVAR_UNIFY:
    await_nonvar_unify_write:
      goto await_leftmost_write;

      /************************
       * AWAIT_STRICTLY_NONVAR_UNIFY
       ************************/
    case AWAIT_STRICTLY_NONVAR_UNIFY:
    await_strictly_nonvar_unify_write:
      w->global->global_fail = TRUE;
      Error("await_strictly_nonvar: level first, term is variable");
      goto done;

      /************************
       * AWAIT_VARIABLE_UNIFY
       ************************/
    case AWAIT_VARIABLE_UNIFY:
    await_variable_unify_write:
      {
	Pre_FetchTop(0);
	Pre_Execute_Write_Instr(1);
      }

      /************************
       * PAR_BUILTIN
       ************************/
    case PAR_BUILTIN:
      goto par_builtin;

      /************************
       * PAR_INLINE
       ************************/
    case PAR_INLINE:
      goto par_inline;

      /************************
       * LOCK_AND_GET_LIST
       ************************/
    case LOCK_AND_GET_LIST:
      goto lock_and_get_list;

      /************************
       * LOCK_AND_GET_STRUCTURE
       ************************/
    case LOCK_AND_GET_STRUCTURE:
      goto lock_and_get_structure;

      /************************
       * UNLOCK
       ************************/
    case UNLOCK:
      goto unlock;
#endif /* REFORM */
      
      /************************
       * HALT
       ************************/
    case HALT:
      goto halt;
      
      /************************
       * NOOP
       ************************/
    case NOOP:
      Execute_Write_Instr;

      /************************
       * END_OF_PRED
       ************************/

    case END_OF_PRED:
      Execute_Write_Instr;
      
    default:
      PL_Print2(stderr,
		"default in write instructions switch, %d does not exist",
		Get_Instr(instruction));
      luther_exit(0);
    }
  
  luther_exit(0);
}
