/*    File:	 debug.c 
 *    Author:	 Johan Bevemyr
 *    Created:	 Wed Jun  5 12:14:48 1991
 */ 

#include "include.h"

int debugflag;
int debugmode;
int debugfail;

#include "display.h"
#include "display_code.h"
#include "debug.h"
#include "engine.h"
#include "inline.h"
#include "signal.h"

static int spypoints[MAXSPYPOINTS];
static code *breakpoints[MAXBREAKPOINTS];
static definition *predspypoints[MAXSPYPOINTS];

static int spyc;
static int breakc;
static int predspyc;
static code *skipbreak;
static BOOL display_args;
static BOOL display_envs;
static BOOL display_fail;
static BOOL display_prolog;

#ifdef DEBUG

#define Putc(C,S) putc(C,S)

static int in_break(pc)
     code *pc;
{
  int i;
  for(i = 0 ; i != breakc ; i++)
    {
      if(pc == breakpoints[i]) return TRUE;
    }
  return FALSE;
}

static int in_spy(inst)
  code inst;
{
  int i;
  for(i = 0 ; i != spyc ; i++)
    {
      if(inst == spypoints[i]) return TRUE;
    }
  return FALSE;
}

static int in_predspy(inst, pc)
  code inst;
code *pc;
{
  int i;
  if(inst == CALL || inst == EXECUTE)
    {
      definition *p;
      pc ++;
      p = Get_Definition(pc);
      for(i = 0 ; i != predspyc ; i++)
	{
	  if(p == predspypoints[i]) return TRUE;
	}
    }
  return FALSE;
}

static void add_spy(inst)
  code inst;
{
  int i;
  for(i = 0 ; i != spyc ; i++)
    {
      if(inst == spypoints[i]) return;
    }
  if(spyc >= MAXSPYPOINTS)
    {
      PL_Print1(currerr,"To many spy points\n");
      return;
    }
  spypoints[spyc++] = inst;
}

static void add_predspy(pred)
  definition *pred;
{
  int i;
  for(i = 0 ; i != spyc ; i++)
    {
      if(pred == predspypoints[i]) return;
    }
  if(predspyc >= MAXSPYPOINTS)
    {
      PL_Print1(currerr,"To many predicate spy points\n");
      return;
    }
  predspypoints[predspyc++] = pred;
}

static void add_break(pc)
  code *pc;
{
  int i;
  for(i = 0 ; i != breakc ; i++)
    {
      if(pc == breakpoints[i]) return;
    }
  if(breakc >= MAXBREAKPOINTS)
    {
      PL_Print1(currerr,"To many break points\n");
      return;
    }
  breakpoints[breakc++] = pc;
}

static void delete_spy(inst)
  code inst;
{
  int i,j;
  for(i = 0 ; i != spyc ; i++)
    {
      if(inst == spypoints[i]) 
	{
	  for(j = i ; j < spyc-1 ; j++)
	    spypoints[j] = spypoints[j+1];
	  spyc--;
	  return;
	}
    }
}

static void delete_predspy(pred)
  definition *pred;
{
  int i,j;
  for(i = 0 ; i != predspyc ; i++) 
    {
      if(pred == predspypoints[i]) 
	{
	  for(j = i ; j < predspyc-1 ; j++)
	    predspypoints[j] = predspypoints[j+1];
	  predspyc--;
	  return;
	}
    }
}

static void delete_break(pc)
  code *pc;
{
  int i,j;
  for(i = 0 ; i != breakc ; i++) 
    {
      if(pc == breakpoints[i]) 
	{
	  for(j = i ; j < breakc-1 ; j++)
	    breakpoints[j] = breakpoints[j+1];
	  breakc--;
	  return;
	}
    }
}

void init_debugger(deb)
  int deb;
{
#ifdef DEBUG
  debugflag = deb;

  debugmode = D_CREEP;
  skipbreak = 0;

  spyc = 0;
  breakc = 0;
  predspyc = 0;
  display_args = TRUE;
  display_envs = TRUE;
  display_fail = TRUE;
  display_prolog = FALSE;
#endif /* DEBUG */
}

int newarity;			/* the arity of the predicate to be called */

static code *display_current_state(pc,e,b,xreg,mode,w)
  code *pc;
environment *e;
choicepoint *b;
TAGGED *xreg;
char *mode;
worker *w;
{
  code *next;
  int op;

  if(display_envs)
    {
      PL_Print2(currerr,"pc: %10ld  \n", (unsigned long) pc);
      PL_Print2(currerr,"e:  %10lx hex\n", (unsigned long) e);
      PL_Print2(currerr,"b:  %10lx hex\n", (unsigned long) b);
      PL_Print2(currerr,"mode: %s\n", mode);
    }

  next = display_code_inc(pc,w);

  if(display_prolog)
    {
      PL_Print1(currerr,".");
    }

  /* Display arguments to CALL and EXECUTE */
  InstToOp(op,Get_Op(*pc));

  if(((op == CALL) || (op == EXECUTE)) && display_args) 
    {
      definition *def;
      int i;
	
      Get_Code(pc);

      if(op == CALL) Get_Index_I(1,0);

      def = Get_Definition(pc);
      newarity= ArityOf(def->name);

      fputs(" (",currerr);
      for(i = 0 ; i != newarity ; i++) 
	{
	  display_term(currerr,xreg[i],w);
	  if(i != newarity - 1) putc(44,currerr);
	}
      fputs(")",currerr);
    }
#ifdef JUMP_CALL
  if((op == CJUMP) && display_args) 
    {
      definition *def;
      int i;
	
      Get_Code(pc);
      Get_Index_I(1,0);
      def = get_def_code(Get_PC_No_Inc(pc),w);

      PL_Print2(currerr," %s",GetString(FunctorToAtom(def->name),w));

      newarity = ArityOf(def->name);

      fputs(" (",currerr);
      for(i = 0 ; i != newarity ; i++) 
	{
	  display_term(currerr,xreg[i],w);
	  if(i != newarity - 1) putc(44,currerr);
	}
      fputs(")",currerr);
    }

  if((op == EJUMP) && display_args) 
    {
      definition *def;
      int i;

      Get_Code(pc);
	
      def = get_def_code(Get_PC_No_Inc(pc),w);

      PL_Print2(currerr," %s",GetString(FunctorToAtom(def->name),w));

      newarity = ArityOf(def->name);

      fputs(" (",currerr);
      for(i = 0 ; i != newarity ; i++) 
	{
	  display_term(currerr,xreg[i],w);
	  if(i != newarity - 1) putc(44,currerr);
	}
      fputs(")",currerr);
    }
#endif /* JUMP_CALL */

  if(((op == META_CALL) || (op == META_EXECUTE)) && display_args) 
    {
      int i;
      Get_Code(pc);
      if(op == META_CALL) Get_Index_I(1,*pc);
      i = Get_Index_I_M(1,*pc);
      fputs(" (",currerr);
      display_term(currerr,xreg[i],w);
      fputs(")",currerr);
    }
  if((op == BUILTIN) && display_args) 
    {
      int i,fnk;
      code instruction;
	
      instruction = Get_Code(pc);

      fnk = Get_Index_I(1,instruction);
      newarity = GetInlineArity(fnk);

      fputs(" (",currerr);
      for(i = 0 ; i < newarity ; i++)
	{
	  if((GetInlineType(fnk) == I_FUNC) &&
	     (GetInlineRetarg(fnk) == i+1))
	    {
	      fputs("<ret.arg.>",currerr);
	      Get_Index(pc);
	    }
	  else
	    display_term(currerr,xreg[Get_Index(pc)],w);
	  if(i != newarity - 1) putc(44,currerr);
	}
      fputs(")",currerr);
    }
  if((op == INLINE) && display_args) 
    {
      int i,fnk;
      code instruction;
	
      instruction = Get_Code(pc);

      fnk = Get_Index_I(1,instruction);
      newarity = GetInlineArity(fnk);

      Inc_Label(pc);

      fputs(" (",currerr);
      for(i = 0 ; i < newarity ; i++)
	{
	  if((GetInlineType(fnk) == I_FUNC) &&
	     (GetInlineRetarg(fnk) == i+1))
	    {
	      fputs("<ret.arg.>",currerr);
	      Get_Index(pc);
	    }
	  else
	    display_term(currerr,xreg[Get_Index(pc)],w);
	  if(i != newarity - 1) putc(44,currerr);
	}
      fputs(")",currerr);
    }
  return next;
}

int debug(pc,w,mode)
  code 	*pc;
worker      *w;
char 	*mode;
{
  char 	debug_str[255];
  int		i,j, op;
  code        *nextpc;

  InstToOp(op,Get_Op(*pc));

  switch(debugmode) 
    {
    case D_LEAP:
      if(in_break(pc) || in_spy(op) || in_predspy(op,pc))
	break;
      if(pc == skipbreak) 
	{
	  skipbreak = NULL;
	  break;
	}
      return TRUE;
    case D_CREEP:
      break;
    case D_TRACE:
      if(in_break(pc) || in_spy(op) || in_predspy(op,pc))
	break;
      (void) display_current_state(pc,w->frame,w->choice,w->regs,mode,w);
      putc('\n',currerr);
      return TRUE;
    case D_CALL_TRACE:
      if(in_break(pc) || in_spy(op) || in_predspy(op,pc))
	break;
      if((op == META_CALL) || (op == META_EXECUTE)) 
	{
	  int i;

	  display_code(pc,w);

	  if(display_args)
	    {
	      i = Get_Index_I_M(1,*(pc));
		
	      display_term(currerr,w->regs[i],w);
	    }
	  putc('\n',currerr);
	  return TRUE;
	}
      if((op == CALL) || (op == EXECUTE)) 
	{
	  definition *def;
	  int i;
	    
	  display_code(pc,w);

	  if(display_args)
	    {
	      Get_Code(pc);
		
	      if(op == CALL) Get_Index_I(1,0);

	      def = Get_Definition(pc);
	      newarity= ArityOf(def->name);
		
	      fputs(" (",currerr);
	      for(i = 0 ; i != newarity ; i++) 
		{
		  display_term(currerr,w->regs[i],w);
		  if(i != newarity - 1) putc(44,currerr);
		}
	      fputs(")",currerr);
	    }
	  PL_Print1(currerr,"\n");
	}
      return TRUE;
    case D_CALL_LEAP:
      if(in_break(pc) || in_spy(op))
	break;
      if(pc == skipbreak) 
	{
	  skipbreak = NULL;
	  break;
	}
      if((op == META_CALL) || (op == META_EXECUTE)) 
	{
	  int i;

	  display_code(pc,w);

	  i = Get_Index_I_M(1,*(pc));

	  if(display_args)
	    {
	      display_term(currerr,w->regs[i],w);
	    }

	  putc('\n',currerr);
	  return TRUE;
	}
      if((op == CALL) || (op == EXECUTE)) 
	{
	  definition *def;
	  int i;
	    
	  display_code(pc,w);

	  if(display_args)
	    {
	      Get_Code(pc);

	      if(op == CALL) Get_Index_I(1,0);
		
	      def = Get_Definition(pc);
	      newarity= ArityOf(def->name);
		
	      fputs(" (",currerr);
	      for(i = 0 ; i != newarity ; i++) 
		{
		  display_term(currerr,w->regs[i],w);
		  if(i != newarity - 1) putc(44,currerr);
		}
	      fputs(")",currerr);
	    }
	  PL_Print1(currerr,"\n");
	}
      return TRUE;
    default:
      PL_Print1(currerr,"No such debug mode\n");
      return TRUE;
    }

  nextpc = display_current_state(pc,w->frame,w->choice,w->regs,mode,w);
    
 debugstart:
    
  PL_Print1(currerr,"\n> ");
  readstring(debug_str);
    
  switch(debug_str[0]) 
    {
    case 'b':
      PL_Print2(currerr,"breaking on: %ld ",(unsigned long) pc);
      display_code(pc,w);
      add_break(pc);
      goto debugstart;
    case 'e':
      if(display_envs) 
	{
	  PL_Print1(currerr,"will not show environment info\n");
	  display_envs = FALSE;
	}
      else 
	{
	  PL_Print1(currerr,"will show environment info\n");
	  display_envs = TRUE;
	}
      goto debugstart;
	
    case 'f':
      if(debugfail) 
	{
	  PL_Print1(currerr,"will not break at fail\n");
	  debugfail = FALSE;
	}
      else 
	{
	  PL_Print1(currerr,"will break at fail\n");
	  debugfail = TRUE;
	}
      goto debugstart;
    case 'r':
      PL_Print2(currerr,"remove break on: %ld ", (unsigned long) pc);
      display_code(pc,w);
      delete_break(pc);
      goto debugstart;
    case 's':
      if((op == EXECUTE) || (op == PROCEED) ||
	 (op == META_CALL) || (op == META_EXECUTE)) 
	{
	  skipbreak = w->next_instr;
	  debugmode = D_LEAP;	    
	}
      else 
	{
	  skipbreak = nextpc;
	  debugmode = D_LEAP;
	}
      return TRUE;
    case 'q':
    case 'a':
      return FALSE;
    case '+':
      switch(debug_str[1])
	{
	case '+':
	  {
	    char tmp[50];
	    register int i;

	    sscanf(&(debug_str[2])," %s",tmp);

	    for(i=1 ; (i < END_OF_PRED) &&
		strcmp(tmp,GetOpString(i)) != 0 ;
		i++);

	    if (i != END_OF_PRED) 
	      {
		PL_Print2(currerr,"spying on %s",GetOpString(i));
		add_spy(i);
	      }
	    else 
	      {
		PL_Print2(currerr,"can't find %s",tmp);
	      }
	    break;
	  }
	case 'p':
	  {
	    char tmp[50];
	    int tmp_arity;
	    register int i;
	    definition *pred;

	    sscanf(&(debug_str[2])," %s %d",tmp, &tmp_arity);

	    pred = get_definition(StoreFunctor(store_atom(tmp,w),tmp_arity),
				  w);

	    if(pred->enter_instruction == ENTER_UNDEFINED)
	      {
		PL_Print1(currerr,"Predicate not defined");
	      }
	    else
	      {
		PL_Print3(currerr,"spying on %s/%d",tmp, tmp_arity);
		add_predspy(pred);
	      }
	    break;
	  }
	default:
	  PL_Print2(currerr,"spying on %s",GetOpString(op));
	  add_spy(op);
	}
      goto debugstart;
    case '-':
      switch(debug_str[1])
	{
	case '-':
	  {
	    char tmp[50];
	    register int i;
	    sscanf(&(debug_str[2])," %s",tmp);
	    for(i=1 ; (i < END_OF_PRED) &&
		strcmp(tmp,GetOpString(i)) != 0 ;
		i++);
	    if (i != END_OF_PRED) 
	      {
		PL_Print2(currerr,"remove spy on %s",GetOpString(i));
		delete_spy(i);
	      }
	    else 
	      {
		PL_Print2(currerr,"can't find %s",tmp);
	      }
	  }
	  break;
	case 'p':
	  {
	    char tmp[50];
	    int tmp_arity;
	    register int i;
	    definition *pred;

	    sscanf(&(debug_str[2])," %s %d",tmp, &tmp_arity);

	    pred = get_definition(StoreFunctor(store_atom(tmp,w),tmp_arity),
				  w);

	    if(pred->enter_instruction == ENTER_UNDEFINED)
	      {
		PL_Print1(currerr,"Predicate not defined");
	      }
	    else
	      {
		PL_Print3(currerr,"no spying on %s/%d",tmp, tmp_arity);
		delete_predspy(pred);
	      }
	  }
	  break;
	default:
	  PL_Print2(currerr,"no spy on %s",GetOpString(op));
	  delete_spy(op);
	}
      goto debugstart;
    case 'l':
      switch(debug_str[1]) 
	{
	case 'c':
	  debugmode = D_CALL_LEAP;
	  return TRUE;
	default:
	  debugmode = D_LEAP;
	  return TRUE;
	}
      break;
    case 'd':
      switch(debug_str[1]) 
	{
	case 't':
	  if(display_args)
	    {
	      PL_Print1(currerr,"DISPLAY ARGS is turned off.\n");
	      display_args = FALSE;
	      break;
	    }
	  else
	    {
	      PL_Print1(currerr,"DISPLAY ARGS is turned on.\n");
	      display_args = TRUE;
	      break;
	    }
	case 'b':
	  PL_Print1(currerr,"Current break points are:\n");
	  for(i = 0 ; i != breakc ; i++) 
	    {
	      PL_Print2(currerr,"pc: %ld\t",(unsigned long)breakpoints[i]);
	      display_code(breakpoints[i],w);
	      putc('\n',currerr);
	    }
	  break;
	case 'f':
	  if(display_fail)
	    {
	      PL_Print1(currerr,"will not show fail info\n");
	      display_fail = FALSE;
	    }
	  else 
	    {
	      PL_Print1(currerr,"will show fail info\n");
	      display_fail = TRUE;
	    }
	  goto debugstart;

	case 'p':
	  display_fail = FALSE;
	  display_args = FALSE;
	  display_envs = FALSE;
	  display_prolog = TRUE;
	  goto debugstart;

	case 's':
	  PL_Print1(currerr,"Current spy points are:\n");
	  for(i = 0 ; i != spyc ; i++) 
	    {
	      PL_Print2(currerr,"Spying on %s",
			instruction_table[spypoints[i]]);
	      putc('\n',currerr);
	    }
	  break;
	case 'a':
	  i = atoi(&(debug_str[2]));
	  PL_Print2(currerr,"a[%d] = ",i);
	  if(w->choice->areg[ARSIZE*i] == 0)
	    PL_Print1(currerr,"NULL");
	  else
	    display_term(currerr,w->choice->areg[ARSIZE*i],w);
	  putc('\n',currerr);
	  break;
#ifdef PARALLEL
	case 'g':
	  i = atoi(&(debug_str[2]));
	  PL_Print2(currerr,"g[%d] = ",i);
	  if(w->global->global_regs[i] == 0)
	    PL_Print1(currerr,"NULL");
	  else
	    display_term(currerr,w->global->global_regs[i],w);
	  putc('\n',currerr);
	  break;	    
#endif /* PARALLEL */
	case 'x':
	  i = atoi(&(debug_str[2]));
	  PL_Print2(currerr,"x[%d] = ",i);
	  if(w->regs[i] == 0)
	    PL_Print1(currerr,"NULL");
	  else
	    display_term(currerr,w->regs[i],w);
	  putc('\n',currerr);
	  break;
	case 'y':
	  i = atoi(&(debug_str[2]));
	  PL_Print2(currerr,"y[%d] = ",i);
	  if(w->frame->yreg[i] == 0)
	    PL_Print1(currerr,"NULL");
	  else
	    display_term(currerr,w->frame->yreg[i],w);
	  putc('\n',currerr);
	  break;
	default:
	  PL_Print1(currerr,"can't print that\n");
	}
      goto debugstart;
    case 'g':
      newarity = atoi(&(debug_str[1]));
      PL_Print1(currerr,"Garbage collecting\n");
      garbage_collect(w,newarity,FrameSize(w->next_instr));
      goto debugstart;
      break;
    case 'h':
      PL_Print2(currerr,"Current heap top = %ld\n",w->heap_top);
      goto debugstart;
      break;
    case 'n':
      PL_Print1(currerr,"Debugger turned off\n");
      debugflag = FALSE;
      return TRUE;
      break;
    case 'p':
      switch(debug_str[1]) 
	{
	case 'y': 
	  {
	    int ysize = FrameSize(w->next_instr);
	    PL_Print1(currerr,"The y register contains: \n");
	    for(i = 0; i != ysize ; i++) 
	      {
		PL_Print2(currerr,"y[%d] = ",i);
		if(w->frame->yreg[i] != 0)
		  display_term(currerr,w->frame->yreg[i],w);
		else
		  PL_Print1(currerr,"NULL");
		putc('\n',currerr);
	      }
	    break;
	  }
	case 'a': 
	  {
	    int arity = w->choice->arity;
	    PL_Print1(currerr,"The a register contains: \n");
	    for(i = 0; i != arity ; i++) 
	      {
		PL_Print2(currerr,"a[%d] = ",i);
		if(w->choice->areg[ARSIZE*i] != 0)
		  display_term(currerr,w->choice->areg[ARSIZE*i],w);
		else
		  PL_Print1(currerr,"NULL");
		putc('\n',currerr);
	      }
	    break;
	  }
	case 'x':
	  j = atoi(&(debug_str[2])) + 1;
	  PL_Print1(currerr,"The x register contains: \n");
	  for(i = 0; i != j ; i++) 
	    {
	      PL_Print2(currerr,"x[%d] = ",i);
	      if(w->regs[i] != 0)
		display_term(currerr,w->regs[i],w);
	      else
		PL_Print1(currerr,"NULL");
	      putc('\n',currerr);
	    }
	  break;
	default:
	  PL_Print1(currerr,"Can't print that\n");
	  break;
	}
      goto debugstart;
    case 't':
      switch(debug_str[1]) 
	{
	case 'c':
	  debugmode = D_CALL_TRACE;
	  break;
	default:
	  debugmode = D_TRACE;
	  break;
	}
      break;
    case '?':
      PL_Print1(currerr,"+           -- spy on instruction\n");
      PL_Print1(currerr,"++ <P>      -- spy on instruction P\n");
      PL_Print1(currerr,"+p <P> <A>  -- spy on predicate P/A\n");
      PL_Print1(currerr,"-           -- remove spy on instruction\n");
      PL_Print1(currerr,"-- <P>      -- remove spy on instruction P\n");
      PL_Print1(currerr,"-p <P> <A>  -- remove spy on predicate P/A\n");
      PL_Print1(currerr,"a           -- abort\n");
      PL_Print1(currerr,"b           -- breakpoint on this instruction\n");
      PL_Print1(currerr,"c           -- creep\n");
      PL_Print1(currerr,"e           -- toggle display env info\n");
      PL_Print1(currerr,"db          -- display break points\n");
      PL_Print1(currerr,"ds          -- display spy points\n");
      PL_Print1(currerr,"dt          -- toggle display arguments\n");
      PL_Print1(currerr,"da <Nr>     -- display a register nr Nr in choicepoint \n");
      PL_Print1(currerr,"df          -- toggle display fail\n");
      PL_Print1(currerr,"dx <Nr>     -- display x register nr Nr\n");
      PL_Print1(currerr,"dy <Nr>     -- display y register nr Nr\n");
#ifdef PARALLEL
      PL_Print1(currerr,"dg <Nr>     -- display global x register nr Nr\n");
#endif /* PARALLEL */
      PL_Print1(currerr,"dp          -- display as prolog terms\n");
      PL_Print1(currerr,"f           -- toggle break at fail\n");
      PL_Print1(currerr,"g           -- force garbage collection\n");
      PL_Print1(currerr,"l           -- leap\n");
      PL_Print1(currerr,"n           -- turn off debugger\n");
      PL_Print1(currerr,"py          -- print y register\n");
      PL_Print1(currerr,"pa          -- print a register in choicepoint\n");
      PL_Print1(currerr,"px <Nr>     -- print x register to reg.nr Nr\n");
      PL_Print1(currerr,"q           -- quit"); putc('\n',currerr);
      PL_Print1(currerr,"r           -- remove breakpoint at this instruction\n");
      PL_Print1(currerr,"s           -- skip (break, leap)\n");
      PL_Print1(currerr,"t           -- trace (leap and print debug info.)\n"); 
      PL_Print1(currerr,"tc          -- trace (leap and print calls)\n"); 
	
      goto debugstart;
    case 'c':
      debugmode = D_CREEP;
      return TRUE;
    case '\n':
    case '\000':
      debugmode = D_CREEP;
      break;
    default:
      PL_Print1(currerr,"unknown command\n");
      goto debugstart;
    }
  return TRUE;
}


int debug_fail(pc,w, mode)
  code *pc;
worker *w;
char *mode;
{
  switch(debugmode) 
    {
    case D_LEAP:
      break;
    case D_CREEP:
    case D_TRACE:
    case D_CALL_TRACE:
    case D_CALL_LEAP:
      if(display_fail)
	{
	  PL_Print1(currerr, "------------ FAIL -----------\n");
	  break;
	}
    }

  if(debugfail) 
    {
      if(debugmode == D_LEAP) 
	{
	  if(display_fail)
	    {
	      PL_Print1(currerr, "------------ FAIL -----------\n");
	    }
	}
      debugmode = D_CREEP;
      return debug(pc, w, mode);
    }
  else 
    {
      return TRUE;
    }
}

void debug_prolog_terms()
{
  debugmode = D_TRACE;
  display_fail = FALSE;
  display_args = FALSE;
  display_envs = FALSE;
  display_prolog = TRUE;
  debugflag = TRUE;
}

#endif


