/*    File:	 database.c  
 *    Author:	 Johan Bevemyr
 *    Created:	 Sat May 25 19:12:08 1991
 *    Purpose:   To keep track of atoms and predicates.
 */ 

/* 
   There are two hash tables: atom and predicate.
 */ 

#include "include.h"
#include "unify.h"
#include "display_code.h"
#include "engine.h"

code *lstart = 0;

TAGGED current_module;

#ifdef LOCALPRED
definition *predtable[PREDHASHLEN];
#else
definition **predtable;
#endif

definition *make_undefined(name, next,w)
    TAGGED name;
    definition *next;
    worker *w;
{
    definition *def;

    def = (definition *) atom_alloc(sizeof(definition),w);
    def->enter_instruction = ENTER_UNDEFINED;
    def->name = name;
    def->next = next;
    def->module = current_module;

#ifdef JUMP_CALL
    def->users = NULL;
#endif /* JUMP_CALL */

    return def;
}

in_switch *make_empty_index_table(w)
    worker *w;
{
    in_switch *res = (in_switch *) atom_alloc(sizeof(in_switch),w);

    res->var = NULL;
    res->last = NULL;

    return res;
}

static void reset_index_table(tab)
    in_switch *tab;
{
    retry_node *tn;

    for(tn = tab->var ; tn != NULL ; tn = tn->next) {
	tn->clause = NULL;
    }
    tab->last = NULL;

    return;
}
	
retry_node *make_retry_node(c,next,w)
    code *c;
    retry_node *next;
    worker *w;
{
    register retry_node *res = (retry_node *) atom_alloc(sizeof(retry_node),w);

    res->next = next;
    res->clause = c;

    return res;
}

/* This function is not used */

void reset_dynamic_predicate(tab, n)
    in_switch *tab;
    retry_node *n;
{
    register retry_node *n1, *tlast;

    if((tab->var == NULL) || (n->clause == NULL)) return;

    tlast = tab->last;
    
    n->clause = NULL;

    if( n == tab->var ) {
	if(tlast == n) {
	    tab->last = NULL;
	}
	return;
    }
	
    n1 = tab->var;

 start:
    if (n1->next == n) {
	if(tlast == n) {
	    tab->last = n1;
	} else {
	    n1->next = n->next;
	    n->next = tlast->next;
	    tlast->next = n;
	}
	return;
    } else {
	n1 = n1->next;
	goto start;
    }
}

void remove_dynamic_clause(tab, n)
    in_switch *tab;
    retry_node *n;
{
    register retry_node *n1, *tlast;

    if((tab->var == NULL) || (n->clause == NULL)) return;

    n->clause = NULL;

    if( n == tab->var ) {
	if(tab->last == n)
	  {
	    tab->last = NULL;
	  }
	else
	  {
	    tab->var = tab->var->next;
	  }
	return;
    }
	
    n1 = tab->var;
    tlast = tab->last;
    
 start:
    if (n1->next == n)
      {
	if(tlast == n)
	  {
	    tab->last = n1;
	    return;
	  }
	n1->next = n->next;
	n->next = tlast->next;
	tlast->next = n;
	return;
      }
    else
      {
	n1 = n1->next;
	goto start;
      }
}

retry_node *add_dynamic_first(tab, c, w)
    in_switch *tab;
    code *c;
    worker *w;
{
    register retry_node *n1;

    if(tab->var == NULL) {
	tab->last = tab->var = make_retry_node(c,NULL,w);
	return tab->var;
    }

    if(tab->last == NULL) {
	tab->var->clause = c;
	tab->last = tab->var;
    } else if(tab->last->next == NULL) {
	tab->var = make_retry_node(c,tab->var,w);
    } else {
	n1 = tab->last->next;
	tab->last->next = n1->next;
	n1->clause = c;
	n1->next = tab->var;
	tab->var = n1;
    }
    return tab->var;
}

retry_node *add_dynamic_last(tab, c, w)
    in_switch *tab;
    code *c;
    worker *w;
{
    register retry_node *n1;

    if(tab->var == NULL) {
	tab->last = tab->var = make_retry_node(c,NULL,w);
	return tab->last;
    }

    n1 = tab->last;

    if(n1 == NULL) {
	tab->var->clause = c;
	tab->last = tab->var;
    } else if(n1->next == NULL) {
	n1->next = make_retry_node(c,NULL,w);
	tab->last = n1->next;
    } else {
	n1->next->clause = c;
	tab->last = n1->next;
    }
    return tab->last;
}

/*
int hash_functor(f)
    TAGGED f;
{
    register int res;
    res = (ArityOf(f) + (((unsigned long) f) >> 2)) % PREDHASHLEN;
    printf("hashvalue of %s/%d = %d\n",GetString(FunctorToAtom(f),w),
	   ArityOf(f), res);
    return res;
}
*/

#define HashFunctor(F) (ArityOf(F)+(((unsigned long) F) >> 2)) % PREDHASHLEN

void init_database(w)
    worker *w;
{
    int i;

#ifndef LOCALPRED
    predtable = w->global->predtable;
#endif    
    for(i=0 ; i != PREDHASHLEN ; i++)
	predtable[i] = NULL;
}

/*
definition *get_definition(name)
    TAGGED name;
{
    register definition *bucket;
    register int hv;

    hv = HashFunctor(name);

    for(bucket = predtable[hv];
	bucket != NULL;
	bucket = bucket->next)
      {
	if((bucket->name == name) &&
	   ((bucket->module == module_public) ||
	    (bucket->module == current_module)))
	  return bucket;
      }
    
    predtable[hv] = make_undefined(name,predtable[hv]);
    return predtable[hv];
}
*/

definition *get_definition(name,w)
    TAGGED name;
    worker *w;
{
    register definition *pl, **bucket;

    bucket = &(predtable[HashFunctor(name)]);

    for(pl = *bucket; pl != NULL; pl = pl->next) {
	if(pl->name == name) {
	    if((pl->module == current_module) ||
	       (pl->module == module_public))
		return pl;
	}
    }

    pl = make_undefined(name, *bucket, w);
    *bucket = pl;

    return pl;
}

#ifdef JUMP_CALL
void add_user(def, p, w)
     definition *def;
     code *p;
     worker *w;
{
  calluse *cu = (calluse *) atom_alloc(sizeof(calluse),w);

  cu->where = p;
  cu->next = def->users;
  def->users = cu;

  return;
}

void patch_callers(def)
     definition *def;
{
  calluse *cu = def->users;
  code *new = def->entry_code.incoreinfo; 

  while(cu != NULL) {
#ifdef THREADED_CODE
    if(*(cu->where) == (code)lab_table[(int)CALL]) {
      *(cu->where) = (code) lab_table[(int)CJUMP];
      *(cu->where+2) = (code) new;
    } else {
      *(cu->where) = (code) lab_table[(int)EJUMP];
      *(cu->where+1) = (code) new;
    }
#else /* THREADED_CODE */
    if(Get_Op(*(cu->where)) == CALL) {
      *(cu->where) = (*(cu->where) & (~0xffL)) | CJUMP;
      *(cu->where+1) = (code) new;
    } else {
      *(cu->where) = EJUMP;
      *(cu->where+1) = (code) new;
    }
#endif /* THREADED_CODE */    
    cu = cu->next;
  }
  return;
}

void unpatch_callers(def)
     definition *def;
{
  calluse *cu = def->users;

  while(cu != NULL) {
#ifdef THREADED_CODE
    if(*(cu->where) == (code)lab_table[(int)CJUMP]) {
      *(cu->where) = (code) lab_table[(int)CALL];
      *(cu->where+2) = (code) def;
    } else {
      *(cu->where) = (code) lab_table[(int)EXECUTE];
      *(cu->where+1) = (code) def;
    }
#else /* THREADED_CODE */
    if(Get_Op(*(cu->where)) == CJUMP) {
      *(cu->where) = (*(cu->where) & (~0xffL)) | CALL;
      *(cu->where+1) = (code) def;
    } else {
      *(cu->where) = EXECUTE;
      *(cu->where+1) = (code) def;
    }
#endif /* THREADED_CODE */    
    cu = cu->next;
  }
  return;
}

void repatch_callers(def)
     definition *def;
{
  calluse *cu = def->users;
  code *new = def->entry_code.incoreinfo; 

  while(cu != NULL) {
#ifdef THREADED_CODE
    if(*(cu->where) == (code)lab_table[(int)CJUMP]) {
      *(cu->where+2) = (code) new;
    } else {
      *(cu->where+1) = (code) new;
    }
#else /* THREADED_CODE */
    *(cu->where+1) = (code) new;
#endif /* THREADED_CODE */    
    cu = cu->next;
  }
  return;
}

definition *get_ex_definition(name,w)
    TAGGED name;
    worker *w;
{
    definition *def;

    def = get_definition(name, w);

    add_user(def,w->global->code_current-1,w);

    if(def->enter_instruction == ENTER_EMULATED)
      {
#ifdef THREADED_CODE
	*(w->global->code_current-1) = (code) lab_table[(int)EJUMP];
#else
	*(w->global->code_current-1) = EJUMP;
#endif
	return (definition *) def->entry_code.incoreinfo;
      }
    else return def;
}

definition *get_c_definition(name,w,inst)
    TAGGED name;
    worker *w;
    s32 *inst;
{
    definition *def;

    def = get_definition(name, w);

#ifdef THREADED_CODE
    add_user(def,w->global->code_current-1,w);
#else
    add_user(def,w->global->code_current,w);
#endif /* THREADED_CODE */
    
    if(def->enter_instruction == ENTER_EMULATED)
      {
#ifdef THREADED_CODE
	*(w->global->code_current-1) = (code) lab_table[CJUMP];
	return (definition *) def->entry_code.incoreinfo;
#else
	*inst = (*inst & (~0xffL)) | CJUMP;
	return (definition *) def->entry_code.incoreinfo;
#endif /* THREADED_CODE */
      }
    else return def;
}

/* Given a code pointer this function finds the definition it is
 * associated with 
 */
definition *get_def_code(c,w)
     code *c;
     worker *w;
{
    int i;
    definition *pl;
    code *pc;

    for(i = 0 ; i != PREDHASHLEN ; i++) {

	if(predtable[i] != NULL) {
	    pl = predtable[i];

	    while(pl != NULL) {

		switch(pl->enter_instruction) {

		case ENTER_INTERPRETED:
		case ENTER_SPY:
		    break;
		case ENTER_EMULATED:
		    if(pl->entry_code.incoreinfo == c)
		      return pl;
		case ENTER_C:
		case ENTER_UNDEFINED:
		    break;

		default:
		    PL_Print2(currerr,"get_def_code -- predicate type %d\n",
			      pl->enter_instruction);
		    break;
		}
		pl = pl->next;
	    }
	}
    }
    PL_Print1(currerr,"get_def_code -- can't find predicate def\n");
    return NULL;
}

#else /* JUMP_CALL */

definition *get_ex_definition(name,w)
    TAGGED name;
    worker *w;
{
    return get_definition(name, w);
}

definition *get_c_definition(name,w,i)
    TAGGED name;
    worker *w;
    s32 *i;
{
    return get_definition(name, w);
}
#endif /* JUMP_CALL */

definition *make_public(name,w)
    TAGGED name;
    worker *w;
{
    register definition **c, *n, **bucket, *ret;

    c = bucket = &(predtable[HashFunctor(name)]);
    n = *c;
    ret = NULL;

    while (n != NULL) {
      if(n->name == name) {
	if (n->module == current_module)
	  {
	    n->module = module_public;
	    ret = n;
	    c = &(n->next);
	    n = *c;
	  }
	else if (n->module != module_public)
	  {
	    *c = n->next;
	    n = *c;
	  }
	else
	  {
	    ret = n;
	    c = &(n->next);
	    n = *c;
	  }
      } else {
	  c = &(n->next);
	  n = *c;
      }
    }
	
    if (ret == NULL) {
      ret = make_undefined(name, *bucket, w);
      ret->module = module_public;
      *bucket = ret;
    }

    return ret;
}    

/**/

void store_emulated_predicate(name,c,w)
    TAGGED name;
    code *c;
    worker *w;
{
    definition *def = get_definition(name,w);

    def->entry_code.incoreinfo = c;
#ifdef JUMP_CALL
    if(def->enter_instruction == ENTER_EMULATED) {
      repatch_callers(def);
    } else {
      patch_callers(def);
    }
#endif /* JUMP_CALL */
    def->enter_instruction = ENTER_EMULATED;
}

void store_c_predicate(name,func,module,w)
    TAGGED name;
    BOOL (*func)();
    TAGGED module;
    worker *w;
{
    definition *def = get_definition(name,w);

#ifdef JUMP_CALL
    if(def->enter_instruction == ENTER_EMULATED) {
      unpatch_callers(def);
    }
#endif /* JUMP_CALL */

    def->enter_instruction = ENTER_C;
    def->entry_code.cinfo = func;
    def->module = module;
}

void store_dynamic_predicate(name,c,p,r,w) /* delete all old definitions */
    TAGGED name,p,r;
    code *c;
    worker *w;
{
    definition *def = get_definition(name,w);

#ifdef JUMP_CALL
    if(def->enter_instruction == ENTER_EMULATED) {
      unpatch_callers(def);
    }
#endif /* JUMP_CALL */

    if((def->enter_instruction == ENTER_INTERPRETED) ||
       (def->enter_instruction == ENTER_SPY)) {
	/* clear old index table */
	reset_index_table(def->entry_code.indexinfo);
    } else {
	def->enter_instruction = ENTER_INTERPRETED;
	def->entry_code.indexinfo = make_empty_index_table(w);
    }

    Bind(p,PointerToTerm(def));

    /* enter new predicate */
    Bind(r,PointerToTerm(add_dynamic_first(def->entry_code.indexinfo, c, w)));

    return;
}    

void add_first_dynamic_predicate(name,c,p,r,w) 
    TAGGED name, p, r;
    code *c;
    worker *w;
{
    definition *def = get_definition(name,w);

    if((def->enter_instruction != ENTER_INTERPRETED) &&
       (def->enter_instruction != ENTER_SPY)) {
	def->enter_instruction = ENTER_INTERPRETED;
	def->entry_code.indexinfo = make_empty_index_table(w);
    }

    Bind(p,PointerToTerm(def->entry_code.indexinfo));

    /* enter new predicate */
    Bind(r,PointerToTerm(add_dynamic_first(def->entry_code.indexinfo, c, w)));

    return;
}    

void add_last_dynamic_predicate(name,c,p,r,w) 
    TAGGED name, p, r;
    code *c;
    worker *w;
{
    definition *def = get_definition(name,w);

    if((def->enter_instruction != ENTER_INTERPRETED) &&
       (def->enter_instruction != ENTER_SPY)) {
	def->enter_instruction = ENTER_INTERPRETED;
	def->entry_code.indexinfo = make_empty_index_table(w);
    }

    Bind(p,PointerToTerm(def->entry_code.indexinfo));

    /* enter new predicate */
    Bind(r,PointerToTerm(add_dynamic_last(def->entry_code.indexinfo, c, w)));

    return;
}    

/*
TAGGED store_functor(a,arity)
    TAGGED a;
    int arity;
{
    return Arityfy(a,arity);
}
*/
	
void listing(w)
    worker *w;
{
    int i;
    definition *pl;
    code *pc;

    for(i = 0 ; i != PREDHASHLEN ; i++) {

	if(predtable[i] != NULL) {
	    pl = predtable[i];

	    while(pl != NULL) {

		PL_Print3(currerr,"predicate(%s/%d\n",
			  GetString(FunctorToAtom(pl->name),w),
			  ArityOf(pl->name));

		switch(pl->enter_instruction) {

		case ENTER_INTERPRETED:
		case ENTER_SPY:
		    break;
		case ENTER_EMULATED:
		    pc = pl->entry_code.incoreinfo;
		    lstart = pc; 
/*		    lstart = NULL;  */
		    while(*pc != END_OF_PRED) {
			PL_Print2(currerr,"%10lu ",((s32) pc)-((s32) lstart));
			pc = display_code_inc(pc,w);
			fputs("\n",stderr);
		    }
		    PL_Print1(currerr,")\n");
		    break;

		case ENTER_C:
		case ENTER_UNDEFINED:
		    PL_Print1(currerr,"----- predicate defined in C -----\n)\n");
		    break;

		default:
		    PL_Print2(currerr,"listing -- no such predicate type %d\n",
			    pl->enter_instruction);
		    break;
		}
		pl = pl->next;
	    }
	}
    }
    lstart = 0;
}

void emulated_listing(w)
    worker *w;
{
    int i;
    definition *pl;
    code *pc;

    for(i = 0 ; i != PREDHASHLEN ; i++) {

	if(predtable[i] != NULL) {
	    pl = predtable[i];

	    while(pl != NULL) {

		switch(pl->enter_instruction) {

		case ENTER_INTERPRETED:
		case ENTER_SPY:
		    break;
		case ENTER_EMULATED:
		    PL_Print3(currerr,"predicate(%s/%d\n",
			      GetString(FunctorToAtom(pl->name),w),
			      ArityOf(pl->name));

		    pc = pl->entry_code.incoreinfo;
		    lstart = pc; 
/*		    lstart = NULL; */
		    while(*pc != END_OF_PRED) {
			PL_Print2(currerr,"%10lu ",((s32) pc)-((s32) lstart));
			pc = display_code_inc(pc,w);
			fputs("\n",stderr);
		    }
		    PL_Print1(currerr,")\n");
		    break;

		case ENTER_C:
		case ENTER_UNDEFINED:
		    break;

		default:
		    PL_Print2(currerr,"listing -- no such predicate type %d\n",
			      pl->enter_instruction);
		    break;
		}
		pl = pl->next;
	    }
	}
    }
    lstart = 0;
}
