/*    File:	 copy-gc-struct.c  
 *    Author:	 Johan Bevemyr
 *    Created:	 Mon Oct 21 13:34:44 1991
 *    Purpose:   To implement a first version of a copying garbage-collector.
 */ 

/*
   The idea is to use the avaliable timestamps for implementing 
   a copying garbage collector. The main difficulties involved in
   this are:

   1. The order of the objects in the Prolog heap is used for
      
       a) deciding whether to trail a variable binding or not 
       b) deciding in which direction a binding should go
       c) ordering variables when compared

   2. The heap is devided into heap segments that can be deallocated
      on backtracking. This is not always possible when using a 
      copying gc, only heap segments that has not been copied can be
      reclaimed in this fassion. Note that for deterministic programs
      there is no difference between mark-and-sweep and copying gc
      since no backtracking occur.
      
   The algorithm:
   
   Phase 1.
     All arguments to reachable structures (and lists) are marked. This
     is done in order to not duplicate objects during the copying phase.

   Phase 2.
     All reachable objects are copied in a depth first order. A copied
     term is marked with a forward tag and its value field is made to
     point out its new location. 

   Phase 3. 
     The choicepoints are updated and the trail compacted.

 */	


#include "include.h"
#include "engine.h"
#include "unify.h"
#include "think.h"

#ifdef COPY_GC

#define SwapTAGGED(T1,T2)                                                \
{                                                                        \
  register TAGGED *st_tmp = (T1);                                        \
  T1 = T2;                                                               \
  T2 = st_tmp;                                                           \
}

typedef struct copy_rec {
  TAGGED *new_top,*start,*end;
} copy_rec;


#define IsCopied(T)        IsMarked(T)
#define IsInNew(T,S)       (((T)>=(S)->start)&&((T)<(S)->end))
#define GetNewLocation(T)  ((TAGGED *) (RemoveMark(T)))
#define CopyMark(T)        PutMark(T)

#define IsOldStruct(T)     (IsMarkedF(*T) || IsMarkedF(*(T+1)))

#define InNewArea(P,CP)    ((P > (CP)->start) && (P < (CP)->end))

static void copy PROTO((worker *, copy_rec *, int, int));
static void copy_registers PROTO((worker *, copy_rec *, int));
static void copy_environments PROTO((environment *, worker *,copy_rec *, int));
static void copy_choicepoints PROTO((worker *, copy_rec *));
static TAGGED copy_variable PROTO((TAGGED, copy_rec *));
static TAGGED copy_term PROTO((TAGGED, copy_rec *));
static void update_choicepoints PROTO((worker *, copy_rec *));
static void mark PROTO((worker *, int, int));
static void mark_registers PROTO((worker *, int));
static void mark_environments PROTO((environment *,worker *,int));
static void mark_choicepoints PROTO((worker *));
static void mark_variable PROTO((TAGGED *));

static int trailcells_deleted;

void garbage_collect(w,arity,envsize)
    worker *w;
    int arity, envsize;
{
  int gc_start_time = usertime();
  u32 initial_size = ((u32) w->heap_top) - ((u32) w->heap_start);
  u32 final_size;
  copy_rec cs;
  copy_rec *copy_state = &cs;
  
  if(w->global->flags.gc_verbose == TRUE)
    {
      PL_Print1(currout,"{running GC}\n");
    }

  copy_state->start = copy_state->new_top = w->heap_copy_start;
  copy_state->end = w->heap_copy_end;

  /* ****************************** */

  mark(w,arity,envsize);
  copy(w,copy_state,arity,envsize);
  update_choicepoints(w, copy_state);

  /* ****************************** */

  w->heap_top = copy_state->new_top;

  final_size = ((u32) copy_state->new_top) - ((u32) w->heap_copy_start);

  SwapTAGGED(w->heap_start, w->heap_copy_start);
  SwapTAGGED(w->heap_end, w->heap_copy_end);
  SwapTAGGED(w->heap_margin, w->heap_copy_margin);
  w->heap_copy_top = w->heap_copy_start;

#ifndef UNBOUND
  w->uncond = w->heap_top;
#endif
  
  w->stats->heap_gc_nr++;
  w->stats->heap_gc_bytes += initial_size - final_size;
  w->stats->heap_gc_time += usertime() - gc_start_time;

  if(w->global->flags.gc_verbose == TRUE)
    {
      PL_Print2(currout,"{GC reclaimed %d bytes}\n", ((s32) initial_size) - 
		((s32) final_size));
    }

  if(w->heap_top > w->heap_margin) {
      FatalError("Prolog out of memory");
  }
}

/**********************************************************************
 * Marking
 **********************************************************************/

static void mark(w, arity, envsize)
    worker *w;
    int arity, envsize;
{
    mark_registers(w,arity);
    mark_environments(w->frame,w,envsize);
    mark_choicepoints(w);
}

static void mark_registers(w, arity)
    worker *w;
    int arity;
{
  register TAGGED *areg = w->regs;
  
  while(arity--)
    {
      if(IsHeapTerm(X(arity)) || IsHeapBox(X(arity)))
	{
	  *w->trail_top = X(arity);
	  mark_variable(w->trail_top);
	}
    }
}

static void mark_environments(frame,w,size)
    register environment *frame;
    worker *w;
    int size;
{
    while(frame != NULL) {
	while(size--) {
	    switch(LowTagOf(Yf(size))) {
	    case HVA_LO:
#ifdef CONSTR 
	    case CVA_LO:
#endif /* CONSTR */
		mark_variable(&(Yf(size)));
		break;
	    case SVA_LO:
		break;
	    case NUM_LO:
		break;
	    case FLT_LO:
		mark_variable(&(Yf(size)));
		break;
	    case ATM_LO:
		break;
	    case LST_LO:
	    case STR_LO:
	    case GEN_LO:
		mark_variable(&(Yf(size)));
		break;
	    default:
		Error("mark_environment -- LowTagOf out of range");
	    }
	}
	size = FrameSize(frame->next_instr);
	frame = frame->cont_env;
    }
}


static void mark_choicepoints(w)
    register worker *w;
{
  register int i;
  register TAGGED *areg, *t = w->trail_top-1;
  register choicepoint *choice = w->choice;
  
  trailcells_deleted = 0;
  
  while (choice != NULL)
    {
      while (t >= choice->trail_top)
	{
	  if (IsValueTrail(t)) {   /* Value trail entry */
	    t--;
	    mark_variable(t);
	  }
	  t--;
	}
      mark_environments(choice->cont_env,w,FrameSize(choice->next_instr));
      i = choice->arity;
      areg = choice->areg;
      while(i--)
	{
	  if(IsHeapTerm(X(i)) || IsHeapBox(X(i)))
	    {
	      mark_variable(&(X(i)));
	    }
	}
      choice = choice->last_choice;
    }
}

static void mark_variable(start)
    TAGGED *start;
{
  register TAGGED *current, *next;

  current = start;
  next = TagToPointer(*current);

  MarkF(*current);

  goto first_forward;

 forward:

  if (IsMarked(*current)) goto backward;

 first_forward:

  switch(LowTagOf(*current)) {

  case HVA_LO:
#ifdef CONSTR
  case CVA_LO:
#endif /* CONSTR */
    if ( IsForM(*next) ) goto backward;

    Mark(*current);

    Reverse(current,next);
    goto forward;

  case SVA_LO:
    FatalError("mark_variable -- Stack variable in heap!");
    break;
  case NUM_LO:
  case FLT_LO:
  case ATM_LO:
    goto backward_const;

  case LST_LO:
    {
      if (IsMarkedF(*(next+1))) goto backward;

      next++;
      MarkF(*(next));
      
      Mark(*current);
      Reverse(current, next);
      
      goto forward;
    }
    
  case STR_LO:
    { 
      register int arity;

      if (IsMarkedF(*(next+1))) goto backward;

      arity = StructArity(next);

      if(arity == 0) {
	Error("mark_variable: Structure arity is zero");
	return;
      }

      while(arity--) {
	next++;
	MarkF(*(next));
      }

      Mark(*current);
      Reverse(current, next);

      goto forward;
    }
    
  case GEN_LO:
    GetMethod(gc,*current)(*current);
    goto backward;

  default:
    Error("mark_variable -- LowTagOf out of range");
  }

 backward:

  UnMark(*current);

 backward_const:

  if (!IsMarkedF(*current))
    {
      Undo(current,next);
      goto backward;
    }
  else
    {
      if (current == start)
	{
	  UnMarkF(*current);
	  return;
	}
      Advance(current, next);

      goto forward;
    }
}

/**********************************************************************
 * Copying
 **********************************************************************/

static void copy(w, copy_state, arity, envsize)
    worker *w;
    copy_rec *copy_state;
    int arity, envsize;
{
    copy_registers(w,copy_state,arity);
    copy_environments(w->frame,w,copy_state,envsize);
    copy_choicepoints(w,copy_state);
}

static void copy_registers(w, copy_state, arity)
     worker *w;
     copy_rec *copy_state;
     int arity;
{
  register TAGGED *areg = w->regs;
  
  while(arity--)
    {
      if(IsHeapTerm(X(arity)) || IsHeapBox(X(arity)))
	{
	  X(arity) = copy_variable(X(arity),copy_state);
	}
    }
}

static void copy_environments(frame,w,copy_state,size)
     register environment *frame;
     worker *w;
     copy_rec *copy_state;
     int size;
{
    while(frame != NULL) {
	while(size--) {
	    switch(LowTagOf(Yf(size))) {
	    case HVA_LO:
#ifdef CONSTR 
	    case CVA_LO:
#endif /* CONSTR */
		if (!IsInNew(TagToPointer(Yf(size)),copy_state))
		  {
		    Yf(size) = copy_variable(Yf(size), copy_state);
		  }
		break;
	    case SVA_LO:
		break;
	    case NUM_LO:
		break;
	    case FLT_LO:
		if (!IsInNew(TagToPointer(Yf(size)),copy_state))
		  {
		    Yf(size) = copy_variable(Yf(size), copy_state);
		  }
		break;
	    case ATM_LO:
		break;
	    case LST_LO:
	    case STR_LO:
	    case GEN_LO:
		if (!IsInNew(TagToPointer(Yf(size)),copy_state))
		  {
		    Yf(size) = copy_variable(Yf(size), copy_state);
		  }
		break;
	    default:
		Error("copy_environment -- LowTagOf out of range");
	    }
	}
	size = FrameSize(frame->next_instr);
	frame = frame->cont_env;
    }
}


static void copy_choicepoints(w, copy_state)
     register worker *w;
     copy_rec *copy_state;
{
  register int i;
  register TAGGED *areg, *t = w->trail_top-1;
  register choicepoint *choice = w->choice;
  register TAGGED *old;
  
  trailcells_deleted = 0;
  
  while (choice != NULL)
    {
      while (t >= choice->trail_top)
	{
	  if(IsSVA(*t))
	    {
	      t -= 1;
	    }
	  else
	    {
	      old = TagToPointer(*t);
	      if (IsCopied(*old))
		{
		  register u32 tag = TagOf(*t);
		  *t = Tagify(GetNewLocation(*old), tag);
		  if(IsValueTrail(t))
		    {
		      t -= 1;
		      *t = copy_variable(*t, copy_state);
		      t -= 1;
		    }
		  else
#ifdef UNBOUND
		    {
		      t -= 2;
		    }
#else  /* UNBOUND */
		  {
		    t -= 1;
		  }
#endif /* UNBOUND */
		}
	      else
		{
		  if(IsValueTrail(t)) {	/* Value trail entry */
		    Reset(*t);
		    *t-- = 0;
		    *t = 0;
		    trailcells_deleted += 2;
		  } else if (IsHVA(*t)) {
#ifdef UNBOUND
		    *((TAGGED *) OffsetBase(*t)) = *(t-1);
		    *t = 0;
		    t -= 1;
		    *t = 0;
		    trailcells_deleted += 2;
#else  /* UNBOUND */
		    Reset(*t);
		    *t = 0;
		    trailcells_deleted++;
#endif /* UNBOUND */
		  }
		  t -= 1;
		}
	    }
	}
      copy_environments(choice->cont_env,w,copy_state,
			FrameSize(choice->next_instr));
      i = choice->arity;
      areg = choice->areg;
      while(i--)
	{
	  if(IsHeapTerm(X(i)) || IsHeapBox(X(i)))
	    {
	      X(i) = copy_variable(X(i), copy_state);
	    }
	}
      choice = choice->last_choice;
    }
}

static TAGGED copy_variable(start, copy_state)
     TAGGED start;
     copy_rec *copy_state;
{
  register u32 tag;
  register TAGGED *old;

  tag = TagOf(start);
  old = TagToPointer(start);

  if(IsCopied(*old))
    {
      return Tagify(GetNewLocation(*old), tag);
    }
  else
    {
      return copy_term(start, copy_state);
    }
}

static TAGGED copy_term(start, copy_state)
     TAGGED start;
     copy_rec *copy_state;
{
  register TAGGED *new, *current, *old, ret;


  new = copy_state->new_top;
  current = new;
  old = TagToPointer(start);

  if(IsFLT(start))
    {
      register s32 i, size = GetBoxSize(*old);

      for(i=0 ; i < size ; i++)	/* copy term          */
	{       
	  new[i] = old[i];
	}

      old[0] = CopyMark(&(new[0])); /* update oldspace     */

      ret = Tagify(new,FLT);

      new += size;
      copy_state->new_top = new;

      return ret;
    }
  else if (IsOldStruct(old))            
    {
      register TAGGED *retadd;
      register int i;

      /* find head of struct */
      for(retadd = new ; IsMarkedF(*old) ; old--, retadd++);

      /* calculate new location of object */
      ret = Tagify(retadd,TagOf(start));

      /* copy structure */
      new[0] = old[0];
      old[0] = CopyMark(&(new[0]));

      for(i = 1 ; IsMarkedF(old[i]) ; i++)
	{
	  new[i] = RemoveMarkF(old[i]);
	  old[i] = CopyMark(&(new[i]));
	}
      
      new += i;
    }
  else
    {
      if(InNewArea(old,copy_state)) return start;

      new[0] = old[0];                  /* copy term         */
      old[0] = CopyMark(new);           /* update oldspace   */
      ret = Tagify(new,TagOf(start));
      new += 1; 
    }

 start:

  if (current == new) 
    {
      copy_state->new_top = new;
      return ret; /* done */
    }

  if(IsCopied(*current))
    {
      *current = *GetNewLocation(*current);
    }
  else
    {
      switch(TagOf(*current))
	{
	case HVA:
	  old = TagToPointer(*current);
	  if(IsCopied(*old))
	    {
	      *current = Tagify(GetNewLocation(*old), HVA);
	    }
	  else
	    {
	      if(IsOldStruct(old))
		{
		  register TAGGED *newloc;
		  register int i;
		  
		  /* find head of struct */
		  for(newloc = new ; IsMarkedF(*old) ; old--, newloc++);

		  /* calculate new location of object */
		  *current = Tagify(newloc,HVA);

		  /* copy structure */
		  new[0] = old[0];
		  old[0] = CopyMark(&(new[0]));

		  for(i = 1 ; IsMarkedF(old[i]) ; i++)
		    {
		      new[i] = RemoveMarkF(old[i]);
		      old[i] = CopyMark(&(new[i]));
		    }
      
		  new += i;
		}
	      else
		{
		  *current = Tagify(new,HVA);
		  new[0] = old[0];
		  old[0] = CopyMark(new);
		  new += 1;
		}
	    }
	  break;

	case SVA:
	  Error("gc_copy_term: stack variable in heap");
	  break;

	case NUM:
	  break;

	case FLT: /* box */
	  old = TagToPointer(*current);
	  if(IsCopied(*old))
	    {
	      *current = Tagify(GetNewLocation(*old), FLT);
	    }
	  else
	    {
	      if(IsDynBox(*TagToPointer(*current)))
		{
		  register s32 i, size = GetBoxSize(*old);
		  
		  *current = Tagify(new, FLT);
		  for(i=0 ; i < size ; i++)           /* copy term */
		    {
		      new[i] = old[i];
		    }
		  old[0] = CopyMark(&(new[0])); /* update oldspace */
		  new += size;
		}
	    }
	  break;
	      
	case ATM:
	  if (IsBox(*current))
	    {
	      current += GetBoxSize(*current) - 1;
	    }
	  break;

	case LST:
	  old = TagToPointer(*current);
	  if(IsCopied(*old))
	    {
	      *current = Tagify(GetNewLocation(*old), LST);
	    }
	  else
	    {
	      *current = Tagify(new,LST);
	      new[0] = old[0];	            /* copy term           */
	      old[0] = CopyMark(&(new[0])); /* update oldspace     */
	      new[1] = RemoveMarkF(old[1]); /* copy term           */
	      old[1] = CopyMark(&(new[1])); /* update oldspace     */
	      new += 2;
	    }
	  break;

	case STR:
	  old = TagToPointer(*current);
	  if(IsCopied(*old))
	    {
	      *current = Tagify(GetNewLocation(*old), STR);
	    }
	  else
	    {
	      register s32 i, size = ArityOf(GetFunctor(*current));
	
	      *current = Tagify(new,STR);
	      new[0] = old[0];              /* copy term         */
	      old[0] = CopyMark(&(new[0])); /* update oldspace   */
	      for(i=1 ; i < size+1 ; i++)
		{
		  new[i] = RemoveMarkF(old[i]); /* copy term         */
		  old[i] = CopyMark(&(new[i])); /* update oldspace   */
		}
	      new += size + 1;
	    }
	  break;

#ifdef UNBOUND
	case UVA:
	  break;
#endif /* UNBOUND */

	default:
	  Error("gc_copy_term: no such term type");
	}
    }
  current++;
  goto start;
}
 
static void update_choicepoints(w, copy_state)
     worker *w;
     copy_rec *copy_state;
{
  register choicepoint *choice = w->choice;
  TAGGED *t = w->trail_top-1;
  TAGGED *new = copy_state->new_top;
  
  while(choice != NULL)
    {
      while(t >= choice->trail_top)
	{
	  if(*t == 0)
	    trailcells_deleted--;
	  t--;
	}
      choice->trail_top -= trailcells_deleted;
      choice->global_top = new;
      choice = choice->last_choice;
    }
  /* compact trail */

  {
    register TAGGED *dest, *current;

    dest = w->trail_start;

    for (current = w->trail_start ;
	 current != w->trail_top ;
	 current++)
      if (*current != 0)
	{
	  *dest = *current;
	  dest++;
	}
    w->trail_top = dest;
  }
}

#endif /* COPY_GC */
