/*    File:	 builtin.c 
 *    Author:	 Johan Bevemyr
 *    Created:	 Thu Jun  6 13:49:58 1991
 */ 

#include <ctype.h>
#include "include.h"
#include "engine.h"
#include "unify.h"
#include "initial.h"
#include "debug.h"
#include "assert.h"
#include "builtin.h"
#include "think.h"
#include "version.h"

static int unique_name_index = 0;
static code *luther_clause_code, *luther_retry_choice_code;
static code *luther_retry_curr_pred, *luther_clause_code_noref;

static TAGGED trace_level;

BOOL luther_fail(Arg)
     Argdecl;
{
  return FALSE;
}

BOOL luther_true(Arg)
     Argdecl;
{
  return TRUE;
}

BOOL luther_name(Arg)
     Argdecl;
{
  char temp[MAXATOMLEN];
  int i,intflag,floatflag,ch;
  TAGGED dummy, lst, atm;
  TAGGED X0, X1;

  intflag = 1;
  floatflag = 0;

  DerefNLL(X0,Xw(0));
  DerefNLL(X1,Xw(1));

  if(IsLST(X1))
    {
      lst = X1;

      DerefNLL(dummy,Ref(GetCar(lst)));

      if(!IsNUM(dummy)) return FALSE;

      ch = GetNumber(dummy);

      if(!isdigit(ch)) {
	if(ch != '-' && ch != '+')
	  intflag = 0;
      }

      temp[0] = (char) ch;

      DerefNLL(lst,Ref(GetCdr(lst)));

      i = 1;

      while(IsLST(lst)) {
	DerefNLL(dummy,Ref(GetCar(lst)));

	if(!IsNUM(dummy)) return FALSE;
	ch = GetNumber(dummy);
	if(intflag)
	  {
	    if(!isdigit(ch))
	      {
		if(ch == '.' && !floatflag)
		  {
		    floatflag = 1;
		  }
		else 
		  intflag = 0;
	      }
	  }
	temp[i++] = (char) ch;
	if( i == (MAXATOMLEN-1) )
	  {
	    Error("name - atom exceeds max atom length");
	    return FALSE;
	  }
	DerefNLL(lst,Ref(GetCdr(lst)));
      }

      if(lst != atom_nil) return FALSE;
      temp[i] = '\0';
      if(intflag)
	{
	  if(floatflag)
	    {
	      float tmpflt;
	      sscanf(temp,"%f",&tmpflt);
	      atm = make_float(w,(double) tmpflt);
	    }
	  else
	    {
	      if(i == 1 && (temp[0] == '-' || temp[0] == '+'))
		{
		  atm = store_atom(temp,w);
		}
	      else 
		atm = Make_Integer(atoi(temp));
	    }
	} 
      else 
	atm = store_atom(temp,w);
      return unify(X0,atm,w);
    }
  else
    {
      if(IsATM(X0))
	{
	  lst = make_string_list(w,GetString(X0,w));
	  return unify(X1,lst,w);
	}
      else if (IsNUM(X0))
	{
	  sprintf(temp,"%d",GetNumber(X0));
	  lst = make_string_list(w,temp);
	  return unify(X1,lst,w);
	}
      else if (IsFLT(X0))
	{
	  sprintf(temp,"%f",GetFloat(X0));
	  lst = make_string_list(w,temp);
	  return unify(X1,lst,w);
	}
      else
	{
	  return FALSE;
	}
    }
}


#define IsDigit(N,R) (R < 11 ? (isdigit(N) && ((N - '0') < radix))\
			     : (isdigit(N) ||\
				((N >= 'a') && (N < ('a'+radix-10)))))
#define CharOf(Num)  (Num < 10 ? Num + '0' : Num - 10 + 'a')
#define NumOf(Char)  (((int) Char) <= '9' ? ((int) Char) - '0' \
			 		  : ((int) Char) - 'a' + 10)

/* number_chars(Number,Chars,Radix) -- used by rdtok.pl */

BOOL luther_number_chars(Arg)
     Argdecl;
{
    char temp[MAXATOMLEN];
    register radix;
    register TAGGED Num, Chrs, Radix;
    
    DerefNLL(Num,Xw(0));
    DerefNLL(Chrs,Xw(1));
    DerefNLL(Radix,Xw(2));
    
    if(!IsNUM(Radix)) {
	Error("number_chars - illegal radix");
	return FALSE;
    } else {
	radix = GetNumber(Radix);
    }
    
    if(IsLST(Chrs)) {
	register TAGGED chr, atm;
	register int i = 0;
	register int floatflag = 0;
	register int ch;
	int negflag = 0;

	while(IsLST(Chrs)) {
	  DerefNLL(chr,Ref(GetCar(Chrs)));
	  
	  if(!IsNUM(chr)) return FALSE;
	  
	  ch = GetNumber(chr);
	  
	  if(!IsDigit(ch,radix)) {
	      if((i == 0) && (ch == '-')) {
		  negflag = 1;
	      } else {
		  if(!((i == 0) && (ch == '+'))) 
		      return FALSE;
	      }
	  } else {
	      if (ch == '.') {
		  if(i == 0) return FALSE;   /* numbers can't start with . */
		  if(floatflag)  return FALSE;
		  floatflag = 1;
	      }
	      temp[i++] = (char) ch;
	      if( i == (MAXATOMLEN-1) ) {
		  Error("name - atom exceeds max atom length");
		  return FALSE;
	      }
	  }
	  DerefNLL(Chrs,Ref(GetCdr(Chrs)));
	}

	if(Chrs != atom_nil) return FALSE;

	temp[i] = '\0';

	if(radix == 10) {
	    if(floatflag) {
		float tmpflt;
		sscanf(temp,"%f",&tmpflt);
		atm = make_float(w,(double) (negflag == 1 ? 
					     tmpflt*-1.0 : tmpflt));
	    } else {
		atm = Make_Integer((negflag==1 ? atoi(temp)*-1 : atoi(temp)));
	    }
	    return unify(Num,atm,w);
	} else {
	    register int sum = 0;
	    char *tmp = &(temp[0]);
	    
	    if(floatflag) return FALSE;
	    
	    for(i-- ; i >= 0 ; i--) {
		sum *= radix;
		sum += NumOf(*tmp);
		tmp++;
	    }
	    if(negflag == 1) sum *= -1;
	    
	    return unify(Num,Make_Integer(sum),w);
	}
    } else {
	register TAGGED newlst;

	if(radix == 10) {
	    if(IsNUM(Num)) {
		sprintf(temp,"%d",GetNumber(Num));
		newlst = make_string_list(w,temp);
		return unify(Chrs,newlst,w);
	    }
	    if (IsFLT(Num)) {
		sprintf(temp,"%f",GetFloat(Num));
		newlst = make_string_list(w,temp);
		return unify(Chrs,newlst,w);
	    } 
	    return FALSE;
	} else {
	    register int sum, i, t;
	    register char *a, *b, ch;
	    int negflag = 0;

	    if(!IsNUM(Num)) return FALSE;

	    sum = GetNumber(Num);
	    if(sum < 0) {
		sum *= -1;
		negflag = 1;
	    }
	    i = 0;

	    while(sum) {
		t = sum % radix;
		temp[i++] = CharOf(t);
		sum = sum / radix;
	    }
	    if(negflag == 1) temp[i++] = '-';

	    if(i == 0) temp[i++] = '0';

	    temp[i] = '\0';

	    if(i != 1) {
		a = &(temp[((int)(((float) i)/2.0-0.5))]);
		b = &(temp[(i/2)]);

		while(a != temp) {
		    ch = *a;
		    *a-- = *b;
		    *b++ = ch;
		};
		ch = *a;
		*a = *b;
		*b = ch;
	    }
	    newlst = make_string_list(w,temp);
	    return unify(Chrs,newlst,w);
	}
    }
}

/* atom_chars/2 */

BOOL luther_atom_chars(Arg)
     Argdecl;
{
    return luther_name(w);
}

/* halt/0 */

BOOL luther_halt(Arg)
     Argdecl;
{
  luther_exit(0);
  return FALSE;
}

/* version/0 */

BOOL luther_version(Arg)
     Argdecl;
{
#ifdef PARALLEL
    PL_Print3(currout,"Parallel Luther Prolog version %s, %s.\n",
	      LUTHER_VERSION, LUTHER_COMPILE_DATE);
    PL_Print1(currout,"Copyright (C) 1991 Uppsala University.\n");
    if (w->global->active_workers == 1)
      {
	PL_Print1(currout,"{ Started with 1 active worker }\n");
      }
    else
      {
	PL_Print2(currout,"{ Started with %d active workers }\n",
		  w->global->active_workers);
      }
#else
    PL_Print3(currout,"Luther Prolog version %s, %s.\n",
	      LUTHER_VERSION, LUTHER_COMPILE_DATE);
    PL_Print1(currout,"Copyright (C) 1991 Uppsala University.\n");
#endif /* PARALLEL */
    return TRUE;
}

/* debug/0 */

BOOL luther_debug(Arg)
     Argdecl;
{
    PL_Print1(currout,"{The debugger will first creep -- showing everyting}\n");

#ifdef DEBUG
    debugflag = TRUE;
    debugmode = D_CREEP;
#endif /* DEBUG */
    return TRUE;
}

BOOL luther_prolog_debug(Arg)
     Argdecl;
{
#ifdef DEBUG
  debug_prolog_terms();
#endif /* DEBUG */
  return TRUE;
}

/* wamdebug/1 */
BOOL luther_par_debug(Arg)
     Argdecl;
{
#ifdef PARALLEL
#ifdef DEBUG
    register TAGGED Nr;

    DerefNLL(Nr,Xw(0));

    if(!IsNUM(Nr)) return FALSE;
    if((GetNumber(Nr) < 1 ) ||
       (GetNumber(Nr) > w->global->active_workers))
      {
	Error("worker not active");
	return FALSE;
      }

    w->global->parallel_start.type = W_DEBUG;
    w->global->parallel_start.code = (code *) GetNumber(Nr);

    ActivateWorkers(w);
    
    PL_Print1(currout,"{The debugger will first creep -- ");
    PL_Print1(currout,"showing everyting}\n");
#endif /* DEBUG */
#endif /* PARALLEL */
    return TRUE;
}

/* nodebug/0 */

BOOL luther_nodebug(Arg)
     Argdecl;
{
#ifdef DEBUG
    debugflag = FALSE;
#endif
    return TRUE;
}
    
/* $topchoice(X) */

BOOL luther_topchoice(Arg)
     Argdecl;
{
    TAGGED X0;

    DerefNLL(X0,Xw(0));

    return unify(X0, PointerToTerm(w->choice),w);
}

/* unique_name(?Prefix, ?Suffix, -Name) */

BOOL luther_unique_name(Arg)
     Argdecl;
{
    TAGGED X0,X1,X2, New;
    char newatom[MAXATOMLEN];
    char number[50];

    int number_len;

    DerefNLL(X1,Xw(1));                   /* suffix */

    X0 = Xw(0);                           /* prefix */
    DerefLockSwitch(X0,
		    {
		      Bind_Unsafe(X0,default_prefix,
			          {
				    Error("unique_name - default_prefix");
				    return FALSE;
				  });
		      X0 = default_prefix;
		    },
		    {
		      if(IsATM(X0))
			default_prefix = X0;
		      else
			return FALSE;
		    });

    X2 = Xw(2);                           /* new name */
    DerefLockSwitch(X2,
		    {
		      if(IsATM(X1) || IsVar(X1)) {
		      start:
			
			sprintf(number,"%d",unique_name_index++);
			
			number_len = strlen(number);
			
			strncpy(newatom,GetString(X0,w),
				MAXATOMLEN-number_len-1);
			newatom[MAXATOMLEN-number_len] = 0;
			strcat(newatom,number);
			
			if(IsATM(X1)) strncat(newatom,GetString(X1,w),
					      MAXATOMLEN-1);
			
			if(!(New = atom_exist(newatom,w))) {
			  goto start;
			}
			
			Bind_Unsafe(X2,New, { return FALSE; });
			return TRUE;
		      } else {
			Drop_P_Lock(X2,X2);
			return FALSE;
		      }
		    }
		    ,
		    {
		      return FALSE;
		    }
		    ;);
    return FALSE;
}


/* setarg(+ArgNo, +CompoundTerm, ?NewArg) */
BOOL luther_setarg(Arg)
     Argdecl;
{
    TAGGED X0, X1, X2, *New;
    

    DerefNLL(X0,Xw(0));    /* ArgNo    */
    DerefNLL(X1,Xw(1));    /* Compound */
    DerefNLL(X2,Xw(2));    /* NewArg   */

    if(!(IsNUM(X0)))
	goto barf1;

    if(!(IsCompound(X1)))
	goto barf2;

    if(IsLST(X1)) {
	if(GetNumber(X0) == 1) {
	    New = GetCar(X1);
	    goto zap;
	}
	if(GetNumber(X0) == 2) {
	    New = GetCdr(X1);
	    goto zap;
	}
	goto barf1;
    }

    /* X1 must be a structure then */

    if(GetNumber(X0) >= GetArity(X1))
	goto barf1;

    New = GetArg(X1,GetNumber(X0));
        
 zap:

    /* We can't have stack variables on the heap */

    if(IsSVA(X2)) {
	register TAGGED tmp;
	LoadHVA(w->heap_top,tmp,w);
	Bind_SVA(X2,tmp);
	X2 = tmp;
    }

    ValueTrail(New, *New);
    *New = X2;

    return TRUE;

 barf1:

    Error("setarg - incorrect 1st argument");
    return FALSE;
    
 barf2:

    Error("setarg - incorrect 2st argument");
    return FALSE;
}

BOOL luther_assert_delete_other(Arg)
     Argdecl;
{
    TAGGED Head, Body, Name, P, R;
    code *c;

    DerefNLL(Head,Xw(0));
    DerefNLL(Body,Xw(1));
    DerefNLL(P,Xw(2));
    DerefNLL(R,Xw(3));

    if(IsATM(Head)) {
	Name = StoreFunctor(Head,0);
    } else if (IsSTR(Head)) {
	Name = GetFunctor(Head);
    } else {
	Error("assert: malformed clause");
	return FALSE;
    }

    c = compile_clause(Head,Body,w);

    if(c == NULL) {
	Error("can't assert clause");
	return FALSE;
    }

    store_dynamic_predicate(Name, c, P, R,w);

    return TRUE;
}

BOOL luther_asserta(Arg)
     Argdecl;
{
    TAGGED Head, Body, Name, P, R;
    code *c;

    DerefNLL(Head,Xw(0));
    DerefNLL(Body,Xw(1));
    DerefNLL(P,Xw(2));
    DerefNLL(R,Xw(3));

    if(IsATM(Head)) {
	Name = StoreFunctor(Head,0);
    } else if (IsSTR(Head)) {
	Name = GetFunctor(Head);
    } else {
	Error("assert: malformed clause");
	return FALSE;
    }

    c = compile_clause(Head,Body,w);

    if(c == NULL) {
	Error("can't assert clause");
	return FALSE;
    }

    add_first_dynamic_predicate(Name, c, P, R,w);

    return TRUE;
}

BOOL luther_assertz(Arg)
     Argdecl;
{
    TAGGED Head, Body, Name, P, R;
    code *c;

    DerefNLL(Head,Xw(0));
    DerefNLL(Body,Xw(1));
    DerefNLL(P,Xw(2));
    DerefNLL(R,Xw(3));

    if(IsATM(Head)) {
	Name = StoreFunctor(Head,0);
    } else if (IsSTR(Head)) {
	Name = GetFunctor(Head);
    } else {
	Error("assert: malformed clause");
	return FALSE;
    }

    c = compile_clause(Head,Body,w);

    if(c == NULL) {
	Error("can't assert clause");
	return FALSE;
    }

    add_last_dynamic_predicate(Name, c, P, R,w);

    return TRUE;
}


/* clause(Head,Body,Ref,iRef1,iRef2) */
BOOL luther_clause(Arg)
     Argdecl;
{
    TAGGED Head, Name, Ref;
    definition *def;
    retry_node *current_clause;
    choicepoint *newchoice = NULL;

    DerefNLL(Head,Xw(0));

    Ref = Xw(2);

    /* The Body (Xw(1)) is just passed on to 'match_term' */


    if(GetNumber(Xw(3)) != 0)
      {  
	def = (definition *) TermToPointer(Xw(3));
	current_clause = ((retry_node *) TermToPointer(Xw(4)))->next;
	goto retry;
      }
    else if (IsSTR(Head))
      {
	  Name = GetFunctor(Head);
      }
    else if(IsATM(Head))
      {
	  Name = StoreFunctor(Head,0);
      }
    else
      return FALSE;

    def = get_definition(Name,w);

    if((def->enter_instruction != ENTER_INTERPRETED) &&
       (def->enter_instruction != ENTER_SPY)) {
      if(def->enter_instruction == ENTER_UNDEFINED) {
	return FALSE;
      }
      luther_error(E_CLAUSE_NOT_DYNAMIC, Head,w);
      return FALSE;
    }

    current_clause = def->entry_code.indexinfo->var;

    if(IsEmptyDynPred(current_clause)) return FALSE;

  try:  /* we set the necessary parts of a choicepoint */

    DerefNLL(Ref,Ref);          /* reference passed to user  */

#if defined(TIMESTAMP) || defined(UNBOUND)
    w->time += TIMEUNIT;
#endif

    newchoice = (choicepoint *) Get_Local_Stack_Top;

    if(newchoice > (choicepoint *) w->stack_end) {
      FatalError("Local stack overflow");
    }
    
    newchoice->last_choice = w->choice;
    newchoice->global_top = w->heap_top;
    newchoice->trail_top = w->trail_top;
    newchoice->areg[0] = Head;
    newchoice->areg[ARSIZE*1] = Xw(1);
#if defined(TIMESTAMP) || defined(UNBOUND)
    newchoice->timestamp = w->time;
    w->uncond = w->time;
#else
    w->uncond = w->heap_top;
#endif /* TIMESTAMP | UNBOUND */

    w->choice = newchoice;

  retry: /* Try until one matches or we run out of clauses. */

    if(IsLastClause(current_clause)) {
      goto trust;
    } else {
      if(match_term(w,current_clause->clause) == TRUE)
	{
	  register TAGGED NewRef;

	  if(newchoice != NULL) {
	    newchoice->next_instr = w->next_instr;
	    newchoice->next_clause = luther_clause_code;
	    newchoice->cont_env = w->frame;
	    newchoice->arity = 5;
	    newchoice->areg[ARSIZE*2] = Ref;
	    newchoice->areg[ARSIZE*3] = PointerToTerm(def);
	    newchoice->areg[ARSIZE*4] = PointerToTerm(current_clause);
	  } else {
	      w->choice->areg[ARSIZE*3] = PointerToTerm(def);
	      w->choice->areg[ARSIZE*4] = PointerToTerm(current_clause);
	  }

	  /* '$ref'(def,current_clause) */
	  Make_STR(w->heap_top,NewRef,functor_d_ref); 
	  PushOnHeap(w->heap_top,PointerToTerm(def));
	  PushOnHeap(w->heap_top,PointerToTerm(current_clause));
	  
	  return unify(Ref,NewRef,w);
	}
      else
	{
	  Unwind_Trail(w->choice->trail_top);
	  w->heap_top = w->choice->global_top;
	  Xw(0) = w->choice->areg[0]; Xw(1) = w->choice->areg[ARSIZE*1];
	  current_clause = current_clause->next;
	  goto retry;
	}
    }

  trust:
    /* deallocate choicepoint */

    w->choice = w->choice->last_choice;
#if defined(TIMESTAMP) || defined(UNBOUND)
    w->uncond = w->choice->timestamp;
#else
    w->uncond = w->choice->global_top;
#endif /* TIMESTAMP */
    
    if(match_term(w,current_clause->clause) == TRUE) {
      register TAGGED NewRef;

      /* '$ref'(def,current_clause) */
      Make_STR(w->heap_top,NewRef,functor_d_ref);   
      PushOnHeap(w->heap_top,PointerToTerm(def));
      PushOnHeap(w->heap_top,PointerToTerm(current_clause));

      return unify(Ref,NewRef,w);
    } else
      return FALSE;
}

/* clause(Head,Body,iRef1,iRef2) */
BOOL luther_clause_noref(Arg)
     Argdecl;
{
    TAGGED Head, Name;
    definition *def;
    retry_node *current_clause;
    choicepoint *newchoice = NULL;

    DerefNLL(Head,Xw(0));

    /* The Body (Xw(1)) is just passed on to 'match_term' */


    if(GetNumber(Xw(2)) != 0)
      {  
	def = (definition *) TermToPointer(Xw(2));
	current_clause = ((retry_node *) TermToPointer(Xw(3)))->next;
	goto retry;
      }
    else if (IsSTR(Head))
      {
	  Name = GetFunctor(Head);
      }
    else if(IsATM(Head))
      {
	  Name = StoreFunctor(Head,0);
      }
    else
      return FALSE;

    def = get_definition(Name,w);

    if((def->enter_instruction != ENTER_INTERPRETED) &&
       (def->enter_instruction != ENTER_SPY)) {
      if(def->enter_instruction == ENTER_UNDEFINED) {
	return FALSE;
      }
      luther_error(E_CLAUSE_NOT_DYNAMIC, Head,w);
      return FALSE;
    }

    current_clause = def->entry_code.indexinfo->var;

    if(IsEmptyDynPred(current_clause)) return FALSE;

  try:  /* we set the necessary parts of a choicepoint */

#if defined(TIMESTAMP) || defined(UNBOUND)
    w->time += TIMEUNIT;
#endif

    newchoice = (choicepoint *) Get_Local_Stack_Top;

    if(newchoice > (choicepoint *) w->stack_end) {
      FatalError("Local stack overflow");
    }

    newchoice->last_choice = w->choice;
    newchoice->global_top = w->heap_top;
    newchoice->trail_top = w->trail_top;
    newchoice->areg[0] = Head;
    newchoice->areg[ARSIZE*1] = Xw(1);
#if defined(TIMESTAMP) || defined(UNBOUND)
    newchoice->timestamp = w->time;
    w->uncond = w->time;
#else
    w->uncond = w->heap_top;
#endif /* TIMESTAMP */
    w->choice = newchoice;

  retry: /* Try until one matches or we run out of clauses. */

    if(IsLastClause(current_clause)) {
      goto trust;
    } else {
      if(match_term(w,current_clause->clause) == TRUE)
	{
	  if(newchoice != NULL) {
	    newchoice->next_instr = w->next_instr;
	    newchoice->next_clause = luther_clause_code_noref;
	    newchoice->cont_env = w->frame;
	    newchoice->arity = 4;
	    newchoice->areg[ARSIZE*2] = PointerToTerm(def);
	    newchoice->areg[ARSIZE*3] = PointerToTerm(current_clause);
	  } else {
	      w->choice->areg[ARSIZE*2] = PointerToTerm(def);
	      w->choice->areg[ARSIZE*3] = PointerToTerm(current_clause);
	  }

	  return TRUE;
	}
      else
	{
	  Unwind_Trail(w->choice->trail_top);
	  w->heap_top = w->choice->global_top;
	  Xw(0) = w->choice->areg[0]; Xw(1) = w->choice->areg[ARSIZE*1];
	  current_clause = current_clause->next;
	  goto retry;
	}
    }

  trust:
    /* deallocate choicepoint */

    w->choice = w->choice->last_choice;
#if defined(TIMESTAMP) || defined(UNBOUND)
    w->uncond = w->choice->timestamp;
#else
    w->uncond = w->choice->global_top;
#endif    
    
    if(match_term(w,current_clause->clause) == TRUE) {
      return TRUE;
    } else
      return FALSE;
}

/* '$current_predicate'(Name,Head) */

BOOL luther_current_predicate(Arg)
     Argdecl;
{
    TAGGED Name, Head, str, name;
    definition *def;
    int arity;

    DerefNLL(Name,Xw(0));
    DerefNLL(Head,Xw(1));

    if(IsATM(Head)) {
	def = get_definition(StoreFunctor(Head,0),w);

	if(def->enter_instruction == ENTER_UNDEFINED) 
	    return FALSE;

	return unify(Name,Head,w);
    } else if(IsSTR(Head)) {
	def = get_definition(GetFunctor(Head),w);

	if(def->enter_instruction == ENTER_UNDEFINED) 
	    return FALSE;

	return unify(Name,GetSTRatom(Head),w);
    }

    if(IsVar(Head)) {
      if(IsATM(Name)) {
	/* then we have to search the entire predicate database
	 * to find one with that name
	 */
	int i;
	
	for(i = 0 ; i < PREDHASHLEN ; i++) {
	  if((def = predtable[i]) != NULL) {
	    do { if((FunctorToAtom(def->name) == Name) &&
		    (def->enter_instruction != ENTER_UNDEFINED) &&
		    ((def->module == current_module) ||
		     (def->module == module_public)))
		   goto found_first;
	       } while((def=def->next) != NULL);
	  }
	}
	return FALSE;

      found_first:
	name = def->name;
	
	/* Then we have to see if we have to build a choicepoint or not.
	  */
	
	while((def=def->next) != NULL) {
	  if((FunctorToAtom(def->name) == Name) &&
	     (def->enter_instruction != ENTER_UNDEFINED) &&
	     ((def->module==current_module) || (def->module==module_public)))
	    goto found_second;
	}
	for(i++ ; i < PREDHASHLEN ; i++) {
	  if((def = predtable[i]) != NULL) {
	    do { if((FunctorToAtom(def->name) == Name) &&
		    (def->enter_instruction != ENTER_UNDEFINED) &&
		    ((def->module==current_module) ||
		     (def->module==module_public)))
		   goto found_second;
	       } while((def=def->next) != NULL);
	  }
	}
	/* no choicepoint needed */
	
	arity = ArityOf(name);
	
	if(arity == 0)
	  str = FunctorToAtom(name);
	else {
	  Make_STR(w->heap_top,str,name);
	  while(arity--) CreateHVA(w->heap_top,w);
	}
	
	return unify(Head,str,w);
	
      found_second:
	/* There are more alternatives, we have to build a choicepoint
	 */
	{
	  choicepoint *nc;
	  
#if defined(TIMESTAMP) || defined(UNBOUND)
	  w->time += TIMEUNIT;
#endif
	  
	  nc = (choicepoint *) Get_Local_Stack_Top;

	  if(nc > (choicepoint *) w->stack_end) {
	    FatalError("Local stack overflow");
	  }

	  nc->last_choice = w->choice;
	  nc->global_top = w->heap_top;
	  nc->trail_top = w->trail_top;
	  nc->next_instr = w->next_instr;
	  nc->next_clause = luther_retry_curr_pred;
	  nc->cont_env = w->frame;
	  nc->arity = 4;
	  nc->areg[ARSIZE*0] = Name;
	  nc->areg[ARSIZE*1] = Head;
	  nc->areg[ARSIZE*2] = Make_Integer(i);
	  nc->areg[ARSIZE*3] = PointerToTerm(def);
#if defined(TIMESTAMP) || defined(UNBOUND)
	  nc->timestamp = w->time;
	  w->uncond = w->time;
#else
	  w->uncond = w->heap_top;
#endif /* TIMESTAMP */
	  
	  w->choice = nc;

	}
	
	arity = ArityOf(name);
	if(arity == 0)
	  str = FunctorToAtom(name);
	else {
	  Make_STR(w->heap_top,str,name);
	  while(arity--) CreateHVA(w->heap_top,w);
	}

	return unify(Head,str,w);
	
      } else if(IsVar(Name)) {
	int i;

	for(i = 0 ; i < PREDHASHLEN ; i++) {
	  if((def = predtable[i]) != NULL) {
	    do { if((def->enter_instruction != ENTER_UNDEFINED) &&
		    ((def->module == current_module) ||
		     (def->module == module_public)))
		   goto found_third;
	       } while((def=def->next) != NULL);
	  }
	}
	return FALSE;

      found_third:
	name = def->name;
	
	/* Then we have to see if we have to build a choicepoint
	  * or not.
	    */   
	while((def = def->next) != NULL) {
	  if((def->enter_instruction != ENTER_UNDEFINED) &&
	     ((def->module == current_module) ||
	      (def->module == module_public)))
	    goto found_fourth;
	} 
	
	for(i++ ; i < PREDHASHLEN ; i++) {
	  if((def = predtable[i]) != NULL) {
	    do { if((def->enter_instruction != ENTER_UNDEFINED) &&
		    ((def->module == current_module) ||
		     (def->module == module_public)))
		   goto found_fourth;
	       } while((def=def->next) != NULL);
	  }
	}
	/* no choicepoint needed */

	arity = ArityOf(name);
	if(arity == 0)
	  str = FunctorToAtom(name);
	else {
	  Make_STR(w->heap_top,str,name);
	  while(arity--) CreateHVA(w->heap_top,w);
	}

	return (unify(Head,str,w) && unify(Name,FunctorToAtom(name),w));

      found_fourth:
	{
	  choicepoint *nc;

#if defined(TIMESTAMP) || defined(UNBOUND)
	  w->time += TIMEUNIT;
#endif
	  
	  nc = (choicepoint *) Get_Local_Stack_Top;

	  if(nc > (choicepoint *) w->stack_end) {
	    FatalError("Local stack overflow");
	  }

	  nc->last_choice = w->choice;
	  nc->global_top = w->heap_top;
	  nc->trail_top = w->trail_top;
	  nc->next_instr = w->next_instr;
	  nc->next_clause = luther_retry_curr_pred;
	  nc->cont_env = w->frame;
	  nc->arity = 4;
	  nc->areg[ARSIZE*0] = Name;
	  nc->areg[ARSIZE*1] = Head;
	  nc->areg[ARSIZE*2] = Make_Integer(i);
	  nc->areg[ARSIZE*3] = PointerToTerm(def);
#if defined(TIMESTAMP) || defined(UNBOUND)
	  nc->timestamp = w->time;
	  w->uncond = w->time;
#else
	  w->uncond = w->heap_top;
#endif
	  w->choice = nc;
	}

	arity = ArityOf(name);
	if(arity == 0)
	  str = FunctorToAtom(name);
	else {
	  Make_STR(w->heap_top,str,name);
	  while(arity--) CreateHVA(w->heap_top,w);
	}

	return (unify(Head,str,w) && unify(Name,FunctorToAtom(name),w));
      }
    }
    
    return FALSE;
}

BOOL luther_curr_pred(Arg)
     Argdecl;
{
  TAGGED Name, Head, str, atm;
  int i, arity;
  definition *def;

  DerefNLL(Name,Xw(0));
  DerefNLL(Head,Xw(1));
  
  i = GetNumber(Xw(2));
  def = (definition *) TermToPointer(Xw(3));

  if(IsVar(Name)) {
    atm = FunctorToAtom(def->name);
    arity = ArityOf(def->name);
    if(arity == 0) {
      str = atm;
    } else {
      Make_STR(w->heap_top,str,def->name);
      while(arity--) CreateHVA(w->heap_top,w);
    }

    /* If there are any more, then we should let this choicepoint remain,
     * otherwise remove it.
     */
    while((def = def->next) != NULL) {
      if((def->enter_instruction != ENTER_UNDEFINED) &&
	 ((def->module == current_module) ||
	  (def->module == module_public)))
	goto found_first;
    } 
    for(i++ ; i < PREDHASHLEN ; i++) {
      if((def = predtable[i]) != NULL) {
	do { if((def->enter_instruction != ENTER_UNDEFINED) &&
		((def->module == current_module) ||
		 (def->module == module_public)))
	       goto found_first;
	   } while((def = def->next) != NULL);
      }
    }
    /* no choicepoint needed */

    w->choice = w->choice->last_choice;
#if defined(TIMESTAMP) || defined(UNBOUND)
    w->uncond = w->choice->timestamp;
#else
    w->uncond = w->choice->global_top;
#endif /* TIMESTAMP */

    return (unify(Name,atm,w) && unify(Head,str,w));

  found_first:

    w->choice->areg[ARSIZE*2] = Make_Integer(i);
    w->choice->areg[ARSIZE*3] = PointerToTerm(def);

    
    return (unify(Name,atm,w) && unify(Head,str,w));
  } else {
    arity = ArityOf(def->name);
    if(arity == 0)
      str = Name;
    else {
      Make_STR(w->heap_top,str,def->name);
      while(arity--) CreateHVA(w->heap_top,w);
    }
    /* If there are any more, then we should let this choicepoint remain,
     * otherwise remove it.
     */
    while((def = def->next) != NULL) {
      if((FunctorToAtom(def->name) == Name) &&
	 (def->enter_instruction != ENTER_UNDEFINED) &&
	 ((def->module == current_module) ||
	  (def->module == module_public)))
	goto found_second;
    }
    for(i++ ; i < PREDHASHLEN ; i++) {
      if((def = predtable[i]) != NULL) {
	do { if((FunctorToAtom(def->name) == Name) &&
		(def->enter_instruction != ENTER_UNDEFINED) &&
		((def->module == current_module) ||
		 (def->module == module_public)))
	       goto found_second;
	   } while((def = def->next) != NULL);
      }
    }
    /* no choicepoint needed */

    w->choice = w->choice->last_choice;
#if defined(TIMESTAMP) || defined(UNBOUND)
    w->uncond = w->choice->timestamp;
#else
    w->uncond = w->choice->global_top;
#endif /* TIMESTAMP | UNBOUND */

    return unify(Head,str,w);

  found_second:


    w->choice->areg[ARSIZE*2] = Make_Integer(i);
    w->choice->areg[ARSIZE*3] = PointerToTerm(def);
    
    return unify(Head,str,w);
  }
}

/* '$file_mod_time'(+File,?Hi,?Lo) */
BOOL luther_file_mod_time(Arg)
     Argdecl;
{
  TAGGED File, Lo, Hi;
  long t;
  u32 lo, hi;

  DerefNLL(File,Xw(0));
  DerefNLL(Hi,Xw(1));
  DerefNLL(Lo,Xw(2));
       
  if(!IsATM(File)) return FALSE;

  if((File == atom_user) ||
     (File == atom_user_input) ||
     (File == atom_user_output))
      return TRUE;

  t = file_mod_time(GetString(File,w));

  lo = t & 0xffffL;
  hi = (t >> 16) & 0xffffL;
  
  return unify(Make_Integer(lo),Lo,w) && unify(Make_Integer(hi),Hi,w);
}
  

/* '$predicate_property'(Head,Property) */

BOOL luther_predicate_property(Arg)
     Argdecl;
{
  TAGGED Head, Property, Name;
  definition *def;
  
  DerefNLL(Head,Xw(0));
  DerefNLL(Property,Xw(1));
       
  switch(TagOf(Head)) {
  case HVA:
#ifdef CONSTR  
  case CVA:
#endif
  case SVA:
#ifdef LOCKING
  case LCK:
#endif
  case NUM:
  case FLT:
    return FALSE;
  case ATM:
    Name = StoreFunctor(Head,0);
    break;
  case LST:
    Name = StoreFunctor(atom_table_tagged[ATOM_LIST],2);
    break;
  case STR:
    Name = GetFunctor(Head);
    break;
  case GEN:
    return FALSE;
  }    
  
  def = get_definition(Name,w);

  switch(def->enter_instruction) {
  case ENTER_INTERPRETED:
  case ENTER_SPY:
    return unify(atom_table_tagged[ATOM_INTERPRETED],Property,w);
  case ENTER_EMULATED:
    return unify(atom_table_tagged[ATOM_COMPILED],Property,w);
  case ENTER_C:
    return unify(atom_table_tagged[ATOM_BUILT_IN],Property,w);
  case ENTER_UNDEFINED:
    return FALSE;
  }
}

BOOL luther_erase(Arg)
     Argdecl;
{
    TAGGED R, P;
    definition *def;

    DerefNLL(P,Xw(0));
    DerefNLL(R,Xw(1));

    def = (definition *) TermToPointer(P);

    if((def->enter_instruction != ENTER_INTERPRETED) &&
       (def->enter_instruction != ENTER_SPY)) {
      Error("erase - clause not dynamic");
      return FALSE;
    }

    remove_dynamic_clause(def->entry_code.indexinfo,
			    (retry_node *) TermToPointer(R));

    return TRUE;
}

/* used by meta interpreter */

/* '$retry_choice'(N) */

BOOL luther_retry_choice(Arg)
     Argdecl;
{
    TAGGED Level;
    choicepoint *nc;

#if defined(TIMESTAMP) || defined(UNBOUND)
    w->time += TIMEUNIT;
#endif
    
    DerefNLL(Level,Xw(0));

    nc = (choicepoint *) Get_Local_Stack_Top;

    if(nc > (choicepoint *) w->stack_end) {
      FatalError("Local stack overflow");
    }
 
    nc->last_choice = w->choice;
    nc->global_top = w->heap_top;
    nc->trail_top = w->trail_top;
    nc->next_instr = w->next_instr;
    nc->next_clause = luther_retry_choice_code;
    nc->cont_env = w->frame;
    nc->arity = 1;
    nc->areg[0] = Level;
#if defined(TIMESTAMP) || defined(UNBOUND)
    nc->timestamp = w->time;
    w->uncond = w->time;
#else
    w->uncond = w->heap_top;
#endif
    w->choice = nc;

    return TRUE;
}

BOOL luther_retry_cut(Arg)
     Argdecl;
{
    TAGGED Level;
    choicepoint *c = w->choice;

    DerefNLL(Level,Xw(0));
    
    while(c != NULL) {
	if(c->next_clause == luther_retry_choice_code) {
	    if(c->areg[0] == Level) {
		w->choice = c;
		return TRUE;
	    } else if(GetNumber(c->areg[0]) < GetNumber(Level)) {
		return FALSE;
	    }
	}
	c = c->last_choice;
    }

    return FALSE;
}

BOOL luther_set_spy(Arg)
     Argdecl;
{
    TAGGED Goal, Name;
    definition *def;

    DerefNLL(Goal,Xw(0));

    if(IsSTR(Goal)) {
	Name = GetFunctor(Goal);
	def = get_definition(Name,w);
    } else if(IsATM(Goal)) {
	Name = StoreFunctor(Goal,0);
	def = get_definition(Name,w);
    } else
	return FALSE;

    if((def->enter_instruction != ENTER_INTERPRETED) &&
       (def->enter_instruction != ENTER_SPY)){
	Error("predicate not dynamic");
	return TRUE;
    }

    def->enter_instruction = ENTER_SPY;

    return TRUE;
}

BOOL luther_remove_spy(Arg)
     Argdecl;
{
    TAGGED Goal, Name;
    definition *def;

    DerefNLL(Goal,Xw(0));

    if(IsSTR(Goal)) {
	Name = GetFunctor(Goal);
	def = get_definition(Name,w);
    } else if(IsATM(Goal)) {
	Name = StoreFunctor(Goal,0);
	def = get_definition(Name,w);
    } else
	return FALSE;

    if(def->enter_instruction != ENTER_SPY) {
	Error("no spypoint on predicate");
	return TRUE;
    }

    def->enter_instruction = ENTER_INTERPRETED;

    return TRUE;
}

BOOL luther_d_trace(Arg)
     Argdecl;
{
    if(w->lut_trace == 0)
	return FALSE;
    else
	return TRUE;
}

BOOL luther_trace(Arg)
     Argdecl;
{
    w->lut_trace = 1;
    return TRUE;
}

BOOL luther_notrace(Arg)
     Argdecl;
{
    w->lut_trace = 0;
    return TRUE;
}

BOOL luther_get_trace_level(Arg)
     Argdecl;
{
    TAGGED Level;

    DerefNLL(Level,Xw(0));

    return unify(Level, trace_level,w);
}

BOOL luther_set_inc_trace_level(Arg)
     Argdecl;
{
    TAGGED Level;

    DerefNLL(Level,Xw(0));

    trace_level = Make_Integer(GetNumber(Level)+1);

    return TRUE;
}

BOOL luther_set_dec_trace_level(Arg)
     Argdecl;
{
    TAGGED Level;

    DerefNLL(Level,Xw(0));

    trace_level = Level;

    return TRUE;
}

static TAGGED savedchoice;

BOOL luther_save_choice(Arg)
     Argdecl;
{
    TAGGED C;

    DerefNLL(C,Xw(0));

    savedchoice = C;

    return TRUE;
}

BOOL luther_get_saved_choice(Arg)
     Argdecl;
{
    TAGGED C;

    DerefNLL(C,Xw(0));

    return unify(C,savedchoice,w);
}

BOOL luther_set_module(Arg)
     Argdecl;
{
    TAGGED Old,Module;

    DerefNLL(Old,Xw(0));
    DerefNLL(Module,Xw(1));

    if( (Module != module_prolog) && (Module != module_user) )  
    {
	Error("set_module - illegal module specification");
	return FALSE;
    }

    if(unify(current_module,Old,w)) {
	current_module = Module;
	return TRUE;
    } else
	return FALSE;
}

BOOL luther_public(Arg)
     Argdecl;
{
    TAGGED Pred,Arity;

    DerefNLL(Pred,Xw(0));
    DerefNLL(Arity,Xw(1));

    (void) make_public(StoreFunctor(Pred,GetNumber(Arity)),w);

    return TRUE;
}

/* collect_garbage */

BOOL luther_collect_garbage(Arg)
    Argdecl;
{
    garbage_collect(w, 0, FrameSize(w->next_instr));
    return TRUE;
}

/* reduce(Op, Array, Sum) */
BOOL luther_reduce(Arg)
     Argdecl;
{
  register TAGGED Operator, Array, Sum, *a, tmp;
  register double summa, arity, floatflag;

  DerefNLL(Operator,Xw(0));
  DerefNLL(Array,Xw(1));
  DerefNLL(Sum,Xw(2));

  if(!IsATM(Operator))
    return FALSE;

  if(!IsSTR(Array))
    return FALSE;

  arity = GetArity(Array);
  a = GetArg(Array,0);

  if (Operator == atom_table_tagged[ATOM_PLUS])
    {
      for(summa = 0.0, floatflag = FALSE ; arity-- ; a += VARSIZE) {
	DerefNLL(tmp, Ref(a));
	switch(TagOf(tmp)) {
	case NUM:
	  summa += GetNumber(tmp);
	  break;
	case FLT:
	  summa += GetFloat(tmp);
	  floatflag = TRUE;
	  break;
	default:
	  return FALSE;
	}
      }
    }
  else if (Operator == atom_table_tagged[ATOM_TIMES])
    {
      for(summa = 1.0, floatflag = FALSE ; arity-- ; a += VARSIZE) {
	DerefNLL(tmp,Ref(a));
	switch(TagOf(tmp)) {
	case NUM:
	  summa *= GetNumber(tmp);
	  break;
	case FLT:
	  summa *= GetFloat(tmp);
	  floatflag = TRUE;
	  break;
	default:
	  return FALSE;
	}
      }
    }
    
  if(floatflag)
    return unify(make_float(w,summa),Sum,w);
  else
    return unify(Make_Integer((int) summa),Sum,w);
}

/* copy_term(Term,Copy) */
BOOL luther_copy_term(Arg)
    Argdecl;
{
    register TAGGED Term,Copy,New,*S;

    DerefNLL(Term,Xw(0));
    DerefNLL(Copy,Xw(1));

    switch(TagOf(Term)) {
    case HVA:
#ifdef CONSTR
    case CVA:
#endif
    case SVA:
	LoadHVA(w->heap_top,New,w);
	break;
    case NUM:
    case FLT:
    case ATM:
	New = Term;
	break;
    case LST:
	{
	  TAGGED *trailsave,*start = w->heap_top;

	  trailsave = w->trail_top;

	  Make_LST_S(w->heap_top,S,New);

	  luther_copy_args(2,GetCar(Term),S,start,w);

	  Unwind_Trail(trailsave);

	  break;
	}
    case STR:
	{
	  TAGGED *trailsave,*start = w->heap_top;

	  trailsave = w->trail_top;

	  Make_STR_Alloc(w->heap_top,S,New,GetFunctor(Term));

	  luther_copy_args(GetArity(Term),GetArg(Term,0),S,start,w);

	  Unwind_Trail(trailsave);

	  break;
	}
    case GEN:
	{
	  TAGGED *trailsave,*start = w->heap_top;

	  trailsave = w->trail_top;

	  New = GetMethod(copy,Term)(Term,start,w);

	  Unwind_Trail(trailsave);
	}
    }
    return unify(Copy,New,w);
}

void luther_copy_args(Size,Term,S,Start,w)
    register int Size;
    register TAGGED *Term, *S, *Start;
    worker *w;
{
    register TAGGED t;
    while(Size--) {
	DerefNLL(t,Ref(Term));
	switch(TagOf(t)) {
	case HVA:
	  /* This must be changed in the parallell version */

	    if(((TAGGED *) t) > Start) {
		*S = t;
		S += VARSIZE;
	    } else {
		CreateHVA(S,w);
		AlwaysTrail(t);
		AlwaysBind(t,*(S-1));
	    }
	    break;
#ifdef CONSTR
	case CVA:
#endif
	case SVA:
	    CreateHVA(S,w);
	    AlwaysTrail(t);
	    AlwaysBind(t,*(S-1));
	    break;
	case NUM:
	case FLT:
	case ATM:
	    *S = t;
	    S += VARSIZE;
	    break;
	case LST:
	    { TAGGED *tmps;
	      Make_LST_S(w->heap_top,tmps,*S);
	      S += VARSIZE;
	      luther_copy_args(2,GetCar(t),tmps,Start,w);
	    }
	    break;
	case STR:
	    { TAGGED *tmps;
	      Make_STR_Alloc(w->heap_top,tmps,*S,GetFunctor(t));
	      S += VARSIZE;
	      luther_copy_args(GetArity(t),GetArg(t,0),tmps,Start,w);
	    }
	    break;
	case GEN:
	    *S = GetMethod(copy,t)(t,Start,w);
	    S += VARSIZE;
	    break;
	}
	Term += VARSIZE;
    }
}

BOOL luther_atom_mode(Arg)
    Argdecl;
{
    register TAGGED Term, Mode;

    DerefNLL(Term,Xw(0));
    DerefNLL(Mode,Xw(1));

    if(IsATM(Term))
	return unify(Mode,GetAtomMode(Term),w);
    else
	return FALSE;
}

BOOL luther_nr_workers(Arg)
     Argdecl;
{
    register TAGGED Workers, Orig;
    register int nr;

    DerefNLL(Workers,Xw(0));
    DerefNLL(Orig,Xw(1));

    if(!IsNUM(Workers)) return FALSE;

    nr = GetNumber(Workers);

    if((nr < 0) || (nr > orig_nr_of_workers)) return FALSE;

    if(w->pid != 0) return FALSE;
    
    w->global->active_workers = nr;

    return unify(Orig,Make_Integer(orig_nr_of_workers),w);
}

BOOL luther_scheduling(Arg)
     Argdecl;
{
  register TAGGED Method;
  DerefNLL(Method,Xw(0));

#ifdef PARALLEL
  if(!IsATM(Method)) return FALSE;

  if(Method = atom_table_tagged[ATOM_DYNAMIC]) {
    w->global->scheduling = DYNAMIC;
    return TRUE;
  } else if(Method = atom_table_tagged[ATOM_STATIC]) {
    w->global->scheduling = STATIC;
    return TRUE;
  } else
#endif
    return FALSE;
}

void initialize_flags(w)
     worker *w;
{
  w->global->flags.load_verbose = FALSE;
  w->global->flags.gc_verbose = FALSE;
  return;
}

/* prolog_flag_gc(?Oldvalue, +Newvalue) */
BOOL luther_prolog_flag_gc_verbose(Arg)
     Argdecl;
{
  TAGGED Old, New;
  BOOL oldval;

  DerefNLL(New, Xw(1));

  if(!IsATM(New)) return FALSE;

  oldval = w->global->flags.gc_verbose;

  if (New == atom_table_tagged[ATOM_ON])
    {
      w->global->flags.gc_verbose=TRUE;
    }
  else if (New == atom_table_tagged[ATOM_OFF])
    {
      w->global->flags.gc_verbose=FALSE;
    }

  DerefNLL(Old,Xw(0));

  if(oldval)
    {
      return unify(Old,atom_table_tagged[ATOM_ON], w);
    }
  else
    {
      return unify(Old,atom_table_tagged[ATOM_OFF], w);
    }
}

/* prolog_flag_load_verbose(?Oldvalue, +Newvalue) */
BOOL luther_prolog_flag_load_verbose(Arg)
     Argdecl;
{
  TAGGED Old, New;
  BOOL oldval;

  DerefNLL(New, Xw(1));

  if(!IsATM(New)) return FALSE;

  oldval = w->global->flags.load_verbose;

  if (New == atom_table_tagged[ATOM_ON])
    {
      w->global->flags.load_verbose=TRUE;
    }
  else if (New == atom_table_tagged[ATOM_OFF])
    {
      w->global->flags.load_verbose=FALSE;
    }

  DerefNLL(Old,Xw(0));

  if(oldval)
    {
      return unify(Old,atom_table_tagged[ATOM_ON], w);
    }
  else
    {
      return unify(Old,atom_table_tagged[ATOM_OFF], w);
    }
}

#ifdef NEED_RANDOM
int random()
{
  return rand();
}

void srandom(seed)
     int seed;
{
  srand(seed);
  return;
}
#endif /* NEED_RANDOM */

BOOL luther_random(Arg)
     Argdecl;
{
  register TAGGED Range;
  register TAGGED Random;
  
  DerefNLL(Range,Xw(0));
  DerefNLL(Random,Xw(1));

  if(IsVar(Range))
    {
      return unify(Random,Make_Integer(random()),w);
    }
  else if (IsNUM(Range))
    {
      return unify(Random,Make_Integer((random() % GetNumber(Range))+1),w);
    }
    
}

BOOL luther_srandom(Arg)
     Argdecl;
{
  register TAGGED Seed;
  
  DerefNLL(Seed,Xw(0));

  if(!IsNUM(Seed)) return FALSE;

  srandom(GetNumber(Seed));

  return TRUE;
}

static TAGGED generate_var_chain(length,w)
     worker *w;
     int length;
{
  TAGGED Start;

  Start = Tagify(w->heap_top,HVA);

  length -= 1;
  while(length--) {
    PushOnHeap(w->heap_top,Tagify(w->heap_top+VARSIZE,HVA));
  }
  CreateHVA(w->heap_top,w);
    
  return Start;
}


BOOL luther_test_deref(Arg)
     Argdecl;
{
  register TAGGED Count, Length, Time, TestVar, Dvar;
  register int i;
  register long starttime, endtime;

  DerefNLL(Count,Xw(0));
  DerefNLL(Length,Xw(1));
  DerefNLL(Time,Xw(2));

  if(!(IsNUM(Count) && IsNUM(Length))) return FALSE;

  i = GetNumber(Count);

  TestVar = generate_var_chain(GetNumber(Length),w);
  
  starttime = usertime();
  while(i--)
    {
      DerefNLL(Dvar,TestVar); 
    }
  endtime = usertime();

  return unify(Time,Make_Integer(((int) (endtime - starttime))),w);
}

BOOL luther_test_deref_bind(Arg)
     Argdecl;
{
  register TAGGED Count, Length, Time, TestVar, Dvar;
  register int i;
  register long starttime, endtime;

  DerefNLL(Count,Xw(0));
  DerefNLL(Length,Xw(1));
  DerefNLL(Time,Xw(2));

  if(!(IsNUM(Count) && IsNUM(Length))) return FALSE;

  i = GetNumber(Count);

  TestVar = generate_var_chain(GetNumber(Length),w);
  
  starttime = usertime();
  while(i--)
    {
      Dvar = TestVar;
#ifdef UNBOUND
      DerefLockSwitch(Dvar,
		      {
			register TAGGED foo;
			Swap_Bind(Dvar,Dvar,Dvar,foo,{},{Error("error");});
		      },
		      {
			SetHVA(Dvar, Dvar);
		      });
#else  /* UNBOUND */
      DerefLockSwitch(Dvar,
		      {
			Swap_Bind(Dvar,Dvar,Dvar,{},{Error("error");});
		      },
		      {
			SetHVA(Dvar, Dvar);
		      });
#endif /* UNBOUND */
    }
  endtime = usertime();

  return unify(Time,Make_Integer(((int) (endtime - starttime))),w);
}

BOOL luther_trailtest(Arg)
     Argdecl;
{
  register TAGGED Count, Time, *s,Var;
  register int i;
  register long starttime,endtime;
#if defined(TIMESTAMP) || defined(UNBOUND)
  s32 uncond;
#else
  TAGGED *uncond;
#endif

  DerefNLL(Count,Xw(0));
  DerefNLL(Time,Xw(1));

  if(!IsNUM(Count)) return FALSE;

  i = GetNumber(Count);
  LoadHVA(w->heap_top,Var,w);
#if defined(TIMESTAMP) || defined(UNBOUND)
  uncond = w->uncond;
  w->uncond = w->time+TIMEUNIT;
#else
  w->uncond = w->heap_top+1;
#endif
  starttime = usertime();
  while(i--)
    {
#ifdef UNBOUND
      Trail_HVA(Var,*((TAGGED *) Var));
      w->trail_top -= 2;
#else
      Trail_HVA(Var);
      w->trail_top--;
#endif      
    }
  endtime = usertime();
  w->uncond = uncond;
  return unify(Time, Make_Integer((s32) (endtime - starttime)),w);
}

BOOL luther_trailtest2(Arg)
     Argdecl;
{
  register TAGGED Count, Count2,Time, *s,Var;
  register int i,j;
  register long starttime,endtime;
  register TAGGED *trailtop;

  DerefNLL(Count,Xw(0));
  DerefNLL(Count2,Xw(1));
  DerefNLL(Time,Xw(2));

  if(!IsNUM(Count)) return FALSE;
  if(!IsNUM(Count2)) return FALSE;

  j = GetNumber(Count2);
  LoadHVA(w->heap_top,Var,w);
  trailtop = w->trail_top;
  starttime = usertime();
  while(j--)
    {
      i = GetNumber(Count);
      while(i--)
	{
#ifdef UNBOUND
	  Trail_HVA(Var,*((TAGGED *) Var));
#else
	  Trail_HVA(Var);
#endif	  
	}
      Unwind_Trail(trailtop);
    }
  endtime = usertime();
  return unify(Time, Make_Integer((s32) (endtime - starttime)),w);
}


BOOL luther_inittest(Arg)
     Argdecl;
{
  register TAGGED Count, Time, *s;
  register int i;
  register long starttime,endtime;

  DerefNLL(Count,Xw(0));
  DerefNLL(Time,Xw(1));

  if(!IsNUM(Count)) return FALSE;

  i = GetNumber(Count);
  s = w->heap_top;
  starttime = usertime();
  while(i--)
    {
      InitHVA(s,*s,w);
    }

  endtime = usertime();

  return unify(Time, Make_Integer((s32) (endtime - starttime)),w);
}

#define Store_Tagged(T) *(w->global->code_current)++ = (T)

#define Init_Backtrackable(P,N,A) {\
  P = w->global->code_current;\
  Store_Tagged(EXECUTE);\
  Store_Tagged((TAGGED)\
	       get_definition(StoreFunctor(atom_table_tagged[N],A),w));}
				      

void init_backtrackable_c(w)
    worker *w;
{
  Init_Backtrackable(luther_clause_code, ATOM_D_CLAUSE, 5);
  Init_Backtrackable(luther_clause_code_noref, ATOM_D_CLAUSE, 4);
  Init_Backtrackable(luther_retry_choice_code, ATOM_TRUE, 0);
  Init_Backtrackable(luther_retry_curr_pred, ATOM_D_CURR_PRED, 4);
}

