/*    File:	 array.c  (~bevemyr/Luther2/WorkerEmulator/array.c)
 *    Author:	 Johan Bevemyr
 *    Created:	 Tue Apr  7 14:54:22 1992
 *    Purpose:   Code for arrays
 */ 

#include "include.h"
#include "engine.h"
#include "unify.h"
#include "array.h"
#include "inline.h"
#include "display.h"
#include "builtin.h"

static int luther_array_size(t)
    TAGGED t;
{
    return GetNumber(GetArraySize(t))*VARSIZE + ARRAY_BASE_SIZE;
}

static BOOL luther_array_unify(t,y,w)
    TAGGED t,y;
    worker *w;
{
    register int i;
    register TAGGED *a1, *a2, da1, da2;

    if(!IsArray(y)) return FALSE;

    i = GetNumber(GetArraySize(t));

    a1 = &(GetArrayArg(t,0));
    a2 = &(GetArrayArg(y,0));

    while(i--) {
	DerefNLL(da1,Ref(a1));
	DerefNLL(da2,Ref(a2));
	if(unify(da1,da2,w) == FALSE) return FALSE;
	a1 += VARSIZE;
	a2 += VARSIZE;
    }

    return TRUE;
}

static void luther_array_print(stream, t, w)
    FILE *stream;
    TAGGED t;
    worker *w;
{
  register int i;
  register TAGGED *a;

  PL_Print1(stream,"#<");

  i = GetNumber(GetArraySize(t)) - 1;

  if(i < 0) {
    PL_Print1(stream,">");
    return;
  }

  a = &(GetArrayArg(t,0));

  while(i--) {
    display_term(stream,Ref(a),w);
    PL_Print1(stream,",");
    a += VARSIZE;
  }

  display_term(stream,Ref(a),w);
  PL_Print1(stream,">");
  return;
}

static SIZE luther_array_compare(t,y,w)
    TAGGED t,y;
    worker *w;
{
    if(!IsGEN(y)) return GREATER;

    if(!IsArray(y)) {
	if(GetGenTag(t) < GetGenTag(y))
	    return LESS;
	else 
	    return GREATER;
    }

    /* If both are arrays we have to compare each element in turn
       until we have resolved the case */

    return compare_struct(&(GetArrayArg(t,0)),&(GetArrayArg(y,0)),
			  GetNumber(GetArraySize(t)),w);
}

static void luther_array_undo(t)
    TAGGED t;
{
}

static BOOL luther_array_gc(t)
    TAGGED t;
{
    return TRUE;
}

static TAGGED luther_array_deref(t)
    register TAGGED t;
{
    return t;
}

static TAGGED luther_array_copy(t,start,w)
    register TAGGED t;
    register TAGGED *start;
    register worker *w;
{
    register TAGGED New;

    New = Tagify(w->heap_top,GEN);
    PushOnHeap(w->heap_top,((TAGGED) &array_method_table));
    PushOnHeap(w->heap_top,GetArraySize(t));
    w->heap_top += GetNumber(GetArraySize(t))*VARSIZE;
    
    luther_copy_args(GetNumber(GetArraySize(t)),&(GetArrayArg(t,0)),
		     &(GetArrayArg(New,0)), start, w);
    
    return New;
}

method array_method_table = {
    luther_array_size,
    luther_array_unify,
    luther_array_print,
    luther_array_compare,
    luther_array_undo,
    luther_array_gc,
    luther_array_deref,
    luther_array_copy
    };

/* The layout of an array is:

   Word   Value

   1.     pointer to method table
   2.     size of array
   3.     element 1
   4.     element 2
   n+2.   element n

 */


/* elt(ArgNr, Array, Element) */
BOOL luther_array_elt(Arg)
    Argdecl;
{
    register TAGGED ArgNr, Array, Element;
    register int i;

    DerefNLL(ArgNr,Xw(0));
    DerefNLL(Array,Xw(1));

    if(!IsNUM(ArgNr)) return FALSE;
    if(!IsArray(Array)) return FALSE;

    i = GetNumber(ArgNr);

    if(i < 0) return FALSE;

    if(GetNumber(GetArraySize(Array)) <= i) return FALSE;

    DerefNLL(Element,Xw(2));

    return unify(Element,Ref(&GetArrayArg(Array,i)),w);
}

/* setref(ArgNr, Array, Element) */
BOOL luther_array_setref(Arg)
    Argdecl;
{
    register TAGGED ArgNr, Array, Element;
    register int i;

    DerefNLL(ArgNr,Xw(0));
    DerefNLL(Array,Xw(1));

    if(!IsNUM(ArgNr)) return FALSE;
    if(!IsArray(Array)) return FALSE;

    i = GetNumber(ArgNr);

    if(i < 0) return FALSE;

    if(GetNumber(GetArraySize(Array)) <= i) return FALSE;

    DerefNLL(Element,Xw(2));
    
#if (defined(TRAIL_ALL) | defined(UNBOUND))
    ValueTrail(BaseOffset(&(GetArrayArg(Array,i))),GetArrayArg(Array,i));
#else
    if(GetHVATime(BaseOffset(&(GetArrayArg(Array,i)))) < w->uncond)
	ValueTrail(&(GetArrayArg(Array,i)),GetArrayArg(Array,i));
#endif

    GetArrayArg(Array,i) = Element;

    return TRUE;
}

/* size(Term,Size) */
BOOL luther_array_size_2(Arg)
    Argdecl;
{
    TAGGED Array, Size;

    DerefNLL(Array,Xw(0)); 
    DerefNLL(Size,Xw(1)); 

    if(IsVar(Array)) {
      if(IsNUM(Size) && (GetNumber(Size) > 0)) {
	register generic new;
	register int i;

	new = Generic(w->heap_top);
	new->method = &array_method_table;
	new->data[0] = Size;

	w->heap_top += ARRAY_BASE_SIZE;

	i = GetNumber(Size);
	while(i--) {
	  CreateHVA(w->heap_top,w);
	}
	return unify(Array,Tagify(new,GEN),w);
      } else {
	return FALSE;
      }
    } else if (IsArray(Array)) {
      return unify(Size,GetArraySize(Array),w);
    } else {
      return FALSE;
    }
}

/* array_reduce(Op, Array, Sum) */
BOOL luther_array_reduce(Arg)
     Argdecl;
{
  register TAGGED Operator, Array, Sum, *a, tmp;
  register double summa, arity, floatflag;

  DerefNLL(Operator,Xw(0));
  DerefNLL(Array,Xw(1));
  DerefNLL(Sum,Xw(2));

  if(!IsATM(Operator))
    return FALSE;

  if(!IsArray(Array))
    return FALSE;

  arity = GetNumber(GetArraySize(Array));
  a = &(GetArrayArg(Array,0));

  if (Operator == atom_table_tagged[ATOM_PLUS])
    {
      for(summa = 0.0, floatflag = FALSE ; arity-- ; a+=VARSIZE) {
	DerefNLL(tmp, Ref(a));
	switch(TagOf(tmp)) {
	case NUM:
	  summa += GetNumber(tmp);
	  break;
	case FLT:
	  summa += GetFloat(tmp);
	  floatflag = TRUE;
	  break;
	default:
	  return FALSE;
	}
      }
    }
  else if (Operator == atom_table_tagged[ATOM_TIMES])
    {
      for(summa = 1.0, floatflag = FALSE ; arity-- ; a+=VARSIZE) {
	DerefNLL(tmp,Ref(a));
	switch(TagOf(tmp)) {
	case NUM:
	  summa *= GetNumber(tmp);
	  break;
	case FLT:
	  summa *= GetFloat(tmp);
	  floatflag = TRUE;
	  break;
	default:
	  return FALSE;
	}
      }
    }
    
  if(floatflag)
    return unify(make_float(w,summa),Sum,w);
  else
    return unify(Make_Integer((int) summa),Sum,w);
}


