;;;-*- Mode: Lisp; Syntax: Common-Lisp; Base: 10; Package: eval -*-
#|
Copyright (C) ISST - Fraunhofer Institute for Software Engineering and Systems 
Engineering - Berlin 1994


-----------------------------------------------------------------------------------
TITLE: An interpreter working on LZS-representation of Lisp-programs
-----------------------------------------------------------------------------------
File:    eval.em
Version: 2.0 (last modification on Tue Mar 15 15:18:55 1994)
State:   proposed

DESCRIPTION:
The function EVAL evaluates an expression given in LZS-form. The function CALL
applies an LZS-function to some arguments given also as LZS-objects. The
interpreter is first used during macro expansion, but can also be used by
compile-time evaluation of constant expressions and during activation of the
MOP. The installation of interpreter functions for LZS-functions which body
should not be interpreted is done by %annotate with the option
"interpreter". The value given to this option must be one of the symbols in
$system-function-table, which provides some functions of the compilation
environment to the interpreter.

DOCUMENTATION:

NOTES:
1. Up to now not all special forms are supported.
2. $system-function-table should contain later all "basic" EuLisp functions. (What
basic means?)
3. Global lexical variables are supported only if their initial value is a
simple literal (symbol, character, string or number or lists/vectors of them) or
if the value is set by a macro expansion before using it.

REQUIRES:

PROBLEMS:
1. What must be done to support MOP-activation during compile time?
2. No module is initialized during macro expansion because the macro language is
only a subset of EuLisp and it is not clear when a module should be initialized
during compile time. Another problem is that there is no distinction between
top-level forms needed during syntax expansion and forms needed during runtime
only. 

AUTHOR:
Ingo Mohr

CONTACT: 
Ingo Mohr

HISTORY: 
Log for /export/home/saturn/ukriegel/Eu2C/Apply/eval.em[2.0]:
  The LZS-interpreter used by syntax expansion.
[1.1] Fri Mar 19 13:32:56 1993 imohr@isst proposed
  set interpreter with 3 Args
[1.2] Fri May  7 15:02:58 1993 imohr@isst saved
  + interpreter function null; some error -> warn
[1.3] Fri May  7 17:11:30 1993 imohr@isst proposed
  + interpreter function char-code
[1.4] Wed Aug 25 15:37:08 1993 ukriegel@isst published
  [Wed Aug 25 10:41:16 1993] Intention for change:
  add + for fixed-precision-integer
  done
[1.5] Mon Dec  6 15:48:30 1993 imohr@isst proposed
  [Mon Dec  6 11:06:52 1993] Intention for change:
  improve error signalling
  not done, but inserted handling of structured literals
[1.6] Thu Jan 13 15:49:45 1994 imohr@isst saved
  [Wed Jan 12 15:13:45 1994] Intention for change:
  global variables
  improved error signalling
[1.7] Fri Jan 21 17:09:10 1994 wheick@isst saved
  more function in the $system-function-table like 0.99
[1.8] Wed Jan 26 09:18:20 1994 wheick@isst proposed
  [Fri Jan 14 08:13:00 1994] Intention for change:
  --- no intent expressed ---keywords
[1.9] Fri Jan 28 11:35:49 1994 imohr@isst saved
  get-value changed analog
[1.10] Fri Jan 28 14:30:40 1994 imohr@isst saved
  
[1.11] Wed Feb  2 09:11:24 1994 imohr@isst saved
  [Fri Jan 28 11:32:41 1994] Intention for change:
  set-value must work for var-ref instead of local-static
[1.12] Wed Feb  2 13:58:24 1994 imohr@isst published
  [Wed Feb  2 13:44:10 1994] Intention for change:
  error messages for non interpreted special forms
[1.13] Mon Feb 28 10:47:08 1994 imohr@isst saved
  late initform evaluation, required by let*
[1.14] Mon Mar 21 09:11:16 1994 wheick@isst proposed
  [Tue Mar 15 15:13:43 1994] Intention for change:
  as-upper-case => as-uppercase, as-lower-case => as-lowercase
  done
[2.0] Mon Mar 21 09:11:16 1994 wheick@isst proposed
  [Tue Mar 15 15:13:43 1994] Intention for change:
  as-upper-case => as-uppercase, as-lower-case => as-lowercase
  done

-----------------------------------------------------------------------------------
|#

#module eval

(import (eulisp1 
         lzs
         eval-basic
         el2lzs-error
         (only (find-module) el2lzs-main)
         (only (mapcar mapc assoc rest first append char-code) 
           common-lisp))

 syntax (eulisp1
         eval-basic
         (only (push) common-lisp))

 export (eval set-interpreter call)
 export (initialize-module)

 )

;;; -----------------------------------------------------------------------------------
;;; eval
;;; -----------------------------------------------------------------------------------

(defgeneric eval (lzsobj))

;;; the default case: 
;;; return the object after transformation to a syntactic object
(defmethod eval ((obj <object>))
  (make-syntactic obj))

;;; -----------------------------------------------------------------------------------
;;; function application
;;; -----------------------------------------------------------------------------------

;;; Because of the optimization during transformation from EuLisp to LZS only
;;; <var-ref>, <fun> or <named-const> can appear as operator

(defmethod eval ((obj <app>))
  (call (eval (?function obj)) (mapcar #'eval (?arg-list obj))))

(defvar eval-fun nil)
(defvar eval-args nil)

(defun call (fn args)
  (dynamic-let ((eval-fun fn)
                (eval-args args))
     (call-eval-fun fn args)))

(defgeneric call-eval-fun (fn args))

(defmethod call-eval-fun (obj args)
  (eval-error-expecting-function (make-syntactic obj)))

(defmethod call-eval-fun ((fun <simple-fun>) args)
  (let ((interpreter (?interpreter fun))
        (body (?body fun)))
    (cond (interpreter
           (apply interpreter args))
          ((null (eq body ^unknown))
           (with-new-values (?var-list (?params fun))
             (?rest (?params fun))
             args
             nil ; the arguments are already evaluated
             (eval body)))
          (t (eval-error-cannot-interpret-function fun)))))

(defmethod call-eval-fun ((fun <fun>) args)
  (let ((interpreter (?interpreter fun)))
    (if interpreter
      (apply interpreter args)
      (eval-error-cannot-interpret-function fun))))

;;; -----------------------------------------------------------------------------------
;;; $system-function-table
;;; -----------------------------------------------------------------------------------
;;; Is used by %annotate-handler 'set-interpreter' to install interpreter functions


(defconstant $system-function-table
  (declare-system-functions

;--- special non-EuLisp-functions needed to implement the basics of EuLisp
   append                               ; needed by quasiquote
   char-code                            ; needed to expand character literals

;--- character
   characterp
   as-lowercase
   as-uppercase

;--- collection
   accumulate
   accumulate1
   anyp
   collectionp
   concatenate
   do
   element
   emptyp
   fill
   map
   member
   reverse
   sequencep
   size

;--- comparision
   eq
   eql
   equal
   =
   <
   max
   min
   
;--- conversion
;   convert
;   converter

;--- copy
   shallow-copy
   deep-copy

;--- double-float
   double-float-p

;--- elementary-functions
;   acos
;   asin
;   atan
;   atan2
;   cos
;   sin
;   tan
;   cosh
;   sinh
;   tanh
;   exp
;   log
;   log10
;   pow
;   sqrt
;--- float, number
   floatp
   ceiling
   floor
   round
   truncate
;--- fpi
   fixed-precision-integer-p
;   ;--- formatted-io
;   scan
   format
;--- integer
   integerp
   evenp
   oddp

   ;--- list
   null
   consp
   atom
   cons
   car
   cdr
   list

   ;--- number
   numberp
   +                                    ; needed for :fixed-precision-integer :big
   - * /
   %
   gcd lcm abs
   zerop
   negate
   signum
   positivep
   negativep

;--- stream
   streamp
;   character-stream-p
;   standard-input-stream
;   standard-output-stream
;   standard-error-stream
;   open
   close
;   flush
;   stream-position
;   end-of-stream-p
;   input
;   uninput
;   output
;   read-line
;   prin
   print
;   write
;   newline

;--- string
   stringp
   ;as-lowercase
   ;as-uppercase
   binary<

;--- symbol
   symbolp
   gensym
   make-symbol
   symbol-name
   symbol-exists-p

;--- table
   tablep
   clear-table

;--- vector
   vectorp

   ))

(defun set-interpreter (lzs-fun annotate-key id)
  (let ((entry (assoc id $system-function-table)))
    (if entry
      (setf (?interpreter lzs-fun)
            (cdr entry))
      (eval-error-undefined-interpreter id))))

;;; -----------------------------------------------------------------------------------
;;; named constants and variable references
;;; -----------------------------------------------------------------------------------

(defmethod eval ((obj <named-const>))
  (make-syntactic (?value obj)))

(defmethod eval ((obj <var-ref>))
  (get-value (?var obj)))

(defmethod eval ((obj <setq-form>))
  (set-value (?location obj) (eval (?form obj))))

(defgeneric get-value (var))

(defmethod get-value ((var <named-const>))
 (?value var))

(defmethod get-value ((var-ref <var-ref>))
  (get-value (?var var-ref)))

(defmethod get-value ((var <local-static>))
 (cdr (assoc var (dynamic variable-environment))))

(defmethod get-value ((var <global-static>))
  (let ((entry (assoc var *global-variables*)))
    (cond (entry (cdr entry))
          ((eq (?initial-value var) ^unknown)
           (eval-error-variable-without-value var)
           nil)
          (t (?initial-value var)))))

(defgeneric set-value (var value))

(defmethod set-value ((var-ref <var-ref>) value)
  (set-value (?var var-ref) value))

(defmethod set-value ((var <local-static>) value)
 (setf (cdr (assoc var (dynamic variable-environment)))
       value))

(defmethod set-value ((var <global-static>) value)
  (let ((entry (assoc var *global-variables*)))
    (if entry 
      (setf (cdr entry) value)
      (setq *global-variables* 
              (cons (cons var value) *global-variables*)))
    value))

;;; -----------------------------------------------------------------------------------
;;; special forms
;;; -----------------------------------------------------------------------------------

(defmethod eval ((obj <progn-form>))
  (eval-progn (?form-list obj)))

(defun eval-progn (forms)
  (let ((value nil))
    (mapc (lambda (form)
            (setq value (eval form)))
          forms)
    value))

(defmethod eval ((obj <if-form>))
  (if (eval (?pred obj))
    (eval (?then obj))
    (eval (?else obj))))

(defmethod eval ((obj <switch-form>))
  (eval-error-special-form-not-implemented ^switch)
  nil)

(defmethod eval ((obj <labeled-form>))
  (eval-error-special-form-not-implemented ^labeled-form)
  nil)

(defmethod eval ((obj <let*-form>))
  (with-new-values (?var-list obj) () (?init-list obj) #'eval
    (eval (?body obj))))

(defmethod eval ((obj <labels-form>))
  (eval-error-special-form-not-implemented ^labels)
  nil)

(defmethod eval ((obj <let/cc-form>))
  (eval-error-special-form-not-implemented ^let/cc)
  nil)

(defmethod eval ((obj <tagbody-form>))
  (eval-error-special-form-not-implemented ^tagbody)
  nil)

(defmethod eval ((obj <tagged-form>))
  (eval-error-special-form-not-implemented ^tagged-form)
  nil)

(defmethod eval ((obj <mv-lambda>))
  (eval-error-special-form-not-implemented ^mv-lambda)
  nil)

(defmethod eval ((obj <get-slot-value>))
  (eval-error-special-form-not-implemented ^%select)
  nil)

(defmethod eval ((obj <set-slot-value>))
  (eval-error-special-form-not-implemented ^%setf-select)
  nil)

;;; -----------------------------------------------------------------------------------
;;; transforming literals to syntactic constructs
;;; -----------------------------------------------------------------------------------
;;; make-syntactic transforms LZS-literals to objects of the compile time
;;; environment such that they can act as syntaxtic constructs. For example in
;;; the case of structured-literals of the LZS, which must be transformed to
;;; "normal" literals

(defgeneric make-syntactic (obj))

(defmethod make-syntactic ((obj <object>))
  obj)

(defmethod make-syntactic ((obj <sym>))
  ; storing the retransformed value in ?syntactic fastens evaluation of symbols
  (?identifier obj))

(defmethod make-syntactic ((obj <structured-literal>))
  ; storing the retransformed value in ?syntactic guarantees the eq-equality of
  ; lists and vectors for eval
  ; also eval works a bit faster by retransforming only once
  ; the original value (available before transformation to the LZS) may be
  ; stored in ?syntactic, but doing this only here avoids unnecessary references
  ; to original structures if they are not used during macro expansion and it
  ; makes eval more independent from other parts of the compiler
  (or (?syntactic obj)
      (setf (?syntactic obj)
            (make-structured-literal-syntactic (?value obj)))))

(defgeneric make-structured-literal-syntactic (obj)
  ; this function is in some sense the counterpart of lzslit (el2lzs-rules)
  )

(defmethod make-structured-literal-syntactic (obj) 
  ; the default case
  (make-syntactic obj))

(defmethod make-structured-literal-syntactic ((obj <string>))
  obj)

(defmethod make-structured-literal-syntactic ((obj <vector>))
  (map #'make-syntactic obj))

(defmethod make-structured-literal-syntactic ((obj <pair>))
  (make-list-syntactic obj))

(defun make-list-syntactic (l)
  (cond ((null l) nil)
        ((atom l) (make-syntactic l))
        (t (cons (make-list-syntactic (car l))
                 (make-list-syntactic (cdr l))))))

;;; -----------------------------------------------------------------------------------
;;; module initialization
;;; -----------------------------------------------------------------------------------

(defun initialize-module (name)
  (call (?toplevel-forms (find-module name))
        nil))

#module-end
