/*****************************************************************************\
*                                                                             *
*				alloc - functions			      *
*									      *
\*****************************************************************************/

#include <stdio.h>
#include "xalloc_conf.h"
#include "xalloc_misc.h"

extern void * (*hold_threads)();	/* to be called when deeper in xa */
extern void * (*cont_threads)();

void * (*lock_heap)() = NULL;
void * (*unlock_heap)() = NULL;

#ifdef NOTHREADS
#define LOCK_HEAP()
#define UNLOCK_HEAP()
#else

#ifdef HOLD_ALL
#define LOCK_HEAP()	if(hold_threads) hold_threads()
#define UNLOCK_HEAP()	if(cont_threads) cont_threads()
#else
#define LOCK_HEAP()	if(lock_heap) lock_heap()
#define UNLOCK_HEAP()	if(unlock_heap) unlock_heap()
#endif

#endif

long	*
build_free_list(c, pt, cd)	/* a function to be used in the next macro:
				 * builds an and returns pointer to the
				 * free list on card c */
register card *c;
register long *pt;
register CardDscr cd;
{
	register long	stp = get_size_d(cd);
	register long	*en = c->userspace;
	register long	*fsl = get_free_object(cd);
	pt -= stp;
	while(pt >= en){
		*pt = (long)fsl;
		fsl = pt;
		pt -= stp;
	}
	return(pt + stp);
}

/* some large macros to prepare cards for usage, used in alloc-fcns. */
/* increase space for carddscr cd; offset o is 1 on MTSS, 0 on STSS */

#define	inc_ss_card_space(cd,o){				\
	register card *c;					\
	LOCK_HEAP();						\
	if((c = (card *)new_card()) == (card *)NULL){		\
	    force_garbage_collection();				\
	    if(rv = get_free_object(cd)){			\
		UNLOCK_HEAP();					\
		goto skip;					\
	    } else {						\
		if((c = (card *)new_card()) == (card *)NULL){	\
		    if(!inc_heap())				\
			error("xalloc: Out of storage");	\
		    c = (card *)new_card();			\
		}						\
	    }							\
	}							\
	register_card(c);					\
	into_card_list(cd, c);					\
	put_current_threadnb(c);				\
	UNLOCK_HEAP();						\
	put_tdscr(c, get_type_d(cd));				\
	put_size(c, get_size_d(cd));				\
	put_real_size(c, (get_size_d(cd)-o) * sizeof(long));	\
	put_cardnum(c,1);					\
	store_free_object(cd, build_free_list(c, c+1, cd));	\
	set_mask(c);						\
	clear_marks(c);}

#ifdef	USE_LARGE
#define	install_large_ss_card(cd,c,o){				\
	register int n = basiccards(get_size_d(cd));		\
	LOCK_HEAP();						\
	get_large_card(c,n);					\
	register_card(c);					\
	into_card_list(cd,c);					\
	put_current_threadnb(c);				\
	UNLOCK_HEAP();						\
	put_tdscr(c, get_type_d(cd));				\
	put_size(c, get_size_d(cd));				\
	put_real_size(c, (get_size_d(cd)-o) * sizeof(long));	\
	put_cardnum(c,n);					\
	set_default_mask(c);					\
	large_unmark(c);}

#define	get_large_card(c,n){					\
	if((c = (card *)new_large_card(n)) == (card *)NULL){	\
	    force_garbage_collection();				\
	    if((c = (card *)new_large_card(n)) == (card *)NULL){\
		inc_heap_size(max(n,hincr));			\
		if((c = (card *)new_large_card(n)) == (card *)NULL)	\
			error("get_large_card: Out of storage");\
	    }							\
	}}
#endif	/* defined USE_LARGE */

#ifdef	USE_STMS
#define	inc_ms_card_space(cd,c,n){				\
	register_card(c);					\
	into_card_list(cd, c);					\
	put_tdscr(c, get_type_d(cd));				\
	put_size(c, 0);						\
	put_real_size(c, 0);					\
	put_cardnum(c,n);					\
	put_current_threadnb(c);				\
	((vobject *)(c->userspace))->vlength = USERSPACE;	\
	((vobject *)(c->userspace))->vnext = (vobject *)get_free_object(cd);\
	store_free_object(cd, c->userspace);			\
	set_default_mask(c);					\
	clear_marks(c);}
#endif	/* defined USE_STMS */

#ifdef	USE_STSS

void	*
xalloc_stss(cd)		/* returns pointer to free object of carddscr cd */
CardDscr	cd;
{			/* This fctn. is very fast, because in most cases
			 * only those instructions marked with an asterisk
			 * at the and of the line will be processed
			 * (, if you've decided not to use our security tests).
			 */
 
	register long	*rv;
	
#ifdef	SECURITY_FIRST
	if(!is_cdscr(cd))
		error1("xalloc_stss: unknown carddescriptor %d\nBye",cd);
	if(ms_card_d(cd))
		error1("xalloc_stss: stms-carddescriptor %d\nBye",cd);
	if(mt_card_d(cd))
		error1("xalloc_stss: mtss-carddescriptor %d\nBye",cd);
#endif

	if(!(rv = get_free_object(cd))){	/* empty free list */	/* * */
#ifdef	USE_LARGE
		if(get_size_d(cd) > USERSPACE){	/* large object? */
			card	*c;
			install_large_ss_card(cd,c,0);
			
			return((void *)(c->userspace));
		} else
#endif
	  	{
			inc_ss_card_space(cd,0);
			rv = get_free_object(cd);
		}
	}
skip:	store_free_object(cd, (long *)*rv); /* next into free list */	/* * */
	
	return((void *) rv);						/* * */
}

#endif

#ifdef	USE_MTSS

void	*
xalloc_mtss(cd, td)	/* returns pointer to free object of carddscr cd,
			 * prepared with typedscr td */

CardDscr	cd;
TypeDscr	td;
{
			/* nearly the same as the above fctn., but puts the
			 * typedescriptor into the word ahead the return value
			 */

	register long	*rv;
	

#ifdef	SECURITY_FIRST
	if(!is_cdscr(cd))
		error1("xalloc_mtss: unknown carddescriptor %d\nBye",cd);
	if(!is_tdscr(td))
		error1("xalloc_mtss: unknown typedescriptor %d\nBye",cd);
	if(ms_card_d(cd))
		error1("xalloc_mtss: stms-carddescriptor %d\nBye",cd);
	if(!mt_card_d(cd))
		error1("xalloc_mtss: stss-carddescriptor %d\nBye",cd);
#endif

	if(!(rv = get_free_object(cd))){	/* empty free list */
#ifdef	USE_LARGE
		if(get_size_d(cd) > USERSPACE){	/* large object? */
			card	*c;
			install_large_ss_card(cd,c,1);
			c->userspace[0] = (long)td;
			
			return((void *)(&(c->userspace[1])));
		} else
#endif
	  	{
			inc_ss_card_space(cd,1);
			rv = get_free_object(cd);	/* successful */
		}
	}
skip:	store_free_object(cd, (long *)*rv); /* next into free list */
	*rv = (long)td;			/* store type in first word */
	
	return((void *) ++rv);		/* return second word */
}

#endif

#ifdef	USE_STMS

void	*
xalloc_stms(cd, i)	/* returns pointer to free object of carddscr cd,
			 * with length i bytes */

register CardDscr	cd;
register long		i;
{
			/* this is really something else, because any found
			 * object may be to short. This causes the loop in this
			 * function and makes it in general much slower
			 * than the both above.
			 * if def. fast_stms we prefer gc to long search times
			 */

	register vobject	*space, *last;
	register long	diff;
	register long	sz = byte2word(i) + 1;
	register card	*c;
	char	gc_just_done = 0;
	

#ifdef	SECURITY_FIRST
	if(!is_cdscr(cd))
		error1("xalloc_stms: unknown carddescriptor %d\nBye",cd);
	if(mt_card_d(cd))
		error1("xalloc_stms: mtss-carddescriptor %d\nBye",cd);
	if(!ms_card_d(cd))
		error1("xalloc_stms: stss-carddescriptor %d\nBye",cd);
#ifndef	USE_LARGE
    	if(sz > USERSPACE)
		error("xalloc_stms: size to large");
#endif
#endif	/* security first */
#ifdef	USE_LARGE
    if(sz <= USERSPACE){
#endif
	while(1){	/* start of trying to get a suitable space */
	    last = (vobject *)get_freept_addr(cd);
	    while((space = last->vnext) && ((space->vlength - sz) < 0)){
#ifdef FAST_STMS
		last->vnext = space->vnext;	/* remv. space from free list */
#else
		last = space;
#endif
	    }
	    if(space)	break;	/* suitable space found */
	    LOCK_HEAP();
	    if((c = (card *)new_card()) == (card *)NULL){
		if(!gc_just_done){		/* don't collect twice */
		    force_garbage_collection();	/* call to trace.c */
		    gc_just_done = 1;	/* prevent from further collection */
		    UNLOCK_HEAP();
		    continue;		/* don't inc_ms_card_space */
		} else {
		    if(!inc_heap())		/* call to heap.c */
			error("xalloc_stms: Out of storage");
		    c = (card *)new_card();	/* call to heap.c */
		}
	    }
	    inc_ms_card_space(cd,c,1);		/* look above for def of this */
	    UNLOCK_HEAP();
	}
	/* found a suitable space */
	if((diff = space->vlength - sz) < 2){
				/* no or only one basic element left:
				 * connect parts of free list */
		last->vnext = space->vnext;
				/* set mark bit to mark begin of object */
		mark(CARD_ADDR(space),(long *)space);
				/* put object size into first word */
		((long *)space)[0] = space->vlength;
				/* it may be essential to set last word to zero,
				 * because we might have increased the size */
		((long *)space)[sz - 1] = 0;
				/* return second word */
		
		return((void *) &(((long *)space)[1]));	
	} else {
		space->vlength = diff;	/* leave some bytes there */
				/* set mark bit to mark begin of object */
		mark(CARD_ADDR(space),(long *)space + diff);
				/* put object size into first word */
		((long *)space)[diff] = sz;
				/* return second word */
		
		return((void *) &(((long *)space)[diff + 1]));
	}

#ifdef	USE_LARGE
    }   	/* end if no large object */ 
      else { 	/* handling large objects another way than on ss_cards:
		 * object will be put on the end of one card, 
		 * the part before this will be used by normal objects
		 */

	long	noc = basiccards(sz);
	LOCK_HEAP();
	get_large_card(c, noc);
		/* the following macro will do something wrong this time:
		 * freelist will contain only the userspace of the first card,
		 * but we're lucky to know this ...
		 */
	inc_ms_card_space(cd,c,noc);
		/* we (ab)use noc (to hold that part of the objectsize, which
		 * will stay at the first card) because it is no longer used
		 * and we want to avoid taking another variable
		 */
	UNLOCK_HEAP();
	noc = sz - (noc - 1) * byte2word(CARDSIZE);
		/* furthermore we must assure, that the object itself starts on 
		 * the first basic card; that means it has at min. 2 words on
		 * it: the size/mark-word and one which is the begin of object
		 * to be pointed to from roots or other objects.
		 */
	noc = max(noc,2);
		/* the next process is nearly the same as above, noc taking the
		 * role of sz. (fortunely we decided to fill pages from the end)
		 */
	last = (vobject *)get_freept_addr(cd);
	space = last->vnext;
	if((diff = space->vlength - noc) < 2){	/* no or only one elem left */
		last->vnext = space->vnext;	/* connect free list */
		mark(CARD_ADDR(space),(long *)space);	/* set mark bit */
		*(long *)space = sz;		/* length into first word */
		
		return((void *) ((long *)space + 1));	/* return second word */
	} else {	
		space->vlength = diff;		/* leave some bytes there */
		mark(CARD_ADDR(space),(long *)space + diff); /* set mark bit */
		*((long *)space + diff) = sz;	/* length into first word */
		
		return((void *) ((long *)space + diff + 1)); /* ret 2nd. word */
	}
    }
#endif	/* use large */
}
#endif	/* use stms */

void	*
xalloc(cd, par)
register CardDscr	cd;
register long		par; /* holds size or type-descriptor */
{
#ifdef	USE_MTSS
	if(mt_card_d(cd))
		return(xalloc_mtss(cd, (TypeDscr)par));
#endif
#ifdef	USE_STMS
	if(ms_card_d(cd))
		return(xalloc_stms(cd, par));
#endif
#ifdef	USE_STSS
	return(xalloc_stss(cd));
#endif
}
