;;;-*- Mode: Lisp; Syntax: Common-Lisp; Base: 10; Package: eval -*-
#|
-----------------------------------------------------------------------------------
TITLE: An interpreter for the LZS
-----------------------------------------------------------------------------------
File:    eval.em
Version: 1.5 (last modification on Mon Dec  6 13:23:00 1993)
State:   proposed

DESCRIPTION:
The function EVAL evaluates an expression given in LZS-form. The function CALL
applies an LZS-function to some arguments given also as LZS-objects. The
interpreter is first used during macro expansion, but can also be used by
compile-time evaluation of constant expressions and during activation of the
MOP. The installation of interpretation functions for LZS-functions which body
should not interpreted is done by %describe-function of TAIL with the option
"interpreter". The value given to this option can be one of the symbols in
$system-function-table, which provides some functions of the compilation
environment to the interpreter.

DOCUMENTATION:

NOTES:
Up to now not all special forms are supported.
$system-function-table should contain later all "basic" EuLisp functions. (What
basic means?)

REQUIRES:

PROBLEMS:
- What must be done to support MOP-activation during compile time?

AUTHOR:
Ingo Mohr

CONTACT: 
Ingo Mohr

HISTORY: 
Log for /export/home/saturn/apply-kunde/Lisp/Apply/eval.em[1.0]
	Fri Feb 26 17:01:11 1993 imohr@isst proposed $
 The LZS-interpreter used by syntax expansion.
 
eval.em[1.1] Fri Mar 19 13:32:56 1993 imohr@isst proposed $
 set interpreter with 3 Args
 
eval.em[1.2] Fri May  7 15:02:58 1993 imohr@isst save $
 + interpreter function null; some error -> warn
 
eval.em[1.3] Fri May  7 17:11:30 1993 imohr@isst proposed $
 + interpreter function char-code
 
eval.em[1.4] Wed Aug 25 15:37:08 1993 ukriegel@isst published $
 [Wed Aug 25 10:41:16 1993] Intention for change:
 add + for fixed-precision-integer
 done
 
eval.em[1.5] Mon Dec  6 15:48:30 1993 imohr@isst proposed $
 [Mon Dec  6 11:06:52 1993] Intention for change:
 improve error signalling
 not done, but inserted handling of structured literals
 

-----------------------------------------------------------------------------------
|#

#module eval

(import (level-1-eulisp 
         lzs
         eval-basic
         (only (find-module) el2lzs-main)
         (only (error warn mapcar mapc assoc rest first append char-code) 
           common-lisp))

 syntax (level-1-eulisp
         eval-basic
         (only (push) common-lisp))

 export (eval set-interpreter call)
 export (initialize-module)

 )

;;; -----------------------------------------------------------------------------------
;;; eval
;;; -----------------------------------------------------------------------------------

;;; the default case: 
;;; return the object after transformation to a syntactic object
(defmethod evaluate ((obj <object>))
  (make-syntactic obj))

;;; -----------------------------------------------------------------------------------
;;; function application
;;; -----------------------------------------------------------------------------------

;;; Because of the optimization during transformation from EuLisp to LZS only
;;; <var-ref>, <fun> or <named-const> can appear as operator

(defmethod evaluate ((obj <app>))
  (call (evaluate (?function obj)) (?arg-list obj)))

(defgeneric call (fn args))

(defmethod call ((fun <simple-fun>) args)
  (let ((interpreter (?interpreter fun))
        (body (?body fun)))
    (cond (interpreter
           (apply interpreter (mapcar #'evaluate args)))
          (body 
           (with-new-values (?var-list (?params fun))
             (?rest (?params fun))
             args
             (evaluate body)))
          (t (error "can't interpret function ~A" fun)))))

;;; -----------------------------------------------------------------------------------
;;; $system-function-table
;;; -----------------------------------------------------------------------------------
;;; Is used by %describe-function to install interpreter functions


(defconstant $system-function-table
  (declare-system-functions
   car cdr
   list null
   cons append ; needed by Backquote
   eq
   char-code ; needed for expanding character literals
   +         ; needed for :fixed-precision-integer :big
   ))
#|
(defconstant $system-function-table
  (declare-system-functions
   ;--- needed by quasiquote
   append
   ;--- needed for expanding character literals
   char-code
;   ;--- character
;   characterp
;   as-lowercase
;   as-uppercase
;   ;--- collection
;   accumulate
;   accumulate1
;   anyp
;   collectionp
;   concatenate
;   do
;   element
;   emptyp
;   fill
;   map
;   member
;   reverse
;   sequencep
;   size
   ;--- comparision
   eq
;   eql
;   equal
   =
   <
   max
   min
;   ;--- conversion
;   convert
;   converter
;   ;--- copy
;   shallow-copy
;   deep-copy
;   ;--- double-float
;   double-float-p
;   ;--- elementary-functions
;   acos
;   asin
;   atan
;   atan2
;   cos
;   sin
;   tan
;   cosh
;   sinh
;   tanh
;   exp
;   log
;   log10
;   pow
;   sqrt
;   ;--- float
;   floatp
;   ceiling
;   floor
;   round
;   truncate
;   ;--- fpi
;   fixed-precision-integer-p
;   ;--- formatted-io
;   scan
;   format
;   ;--- integer
;   integerp
;   evenp
;   oddp
   ;--- list
   null
   consp
   atom
   cons
   car
   cdr
   list
   ;--- number
;   numberp
   + - * / 
;   %
;   gcd lcm abs
;   zerop
;   negate
;   signum
;   positivep
;   negativep
;   ;--- stream
;   streamp
;   character-stream-p
;   standard-input-stream
;   standard-output-stream
;   standard-error-stream
;   open
;   close
;   flush
;   stream-position
;   end-of-stream-p
;   input
;   uninput
;   output
;   read-line
;   prin
;   print
;   write
;   newline
;   ;--- string
;   stringp
;   ;--- symbol
;   symbolp
;   gensym
;   symbol-name
;   symbol-exists-p
;   ;--- table
;   tablep
;   clear-table
;   ;--- vector
;   vectorp
   ))
|#

(defun set-interpreter (lzs-fun annotate-key id)
  (let ((entry (assoc id $system-function-table)))
    (if entry
      (setf (?interpreter lzs-fun)
            (cdr entry))
      (warn "undefined interpreter ~A for function ~A of module ~A" 
             id (?identifier lzs-fun) (?module-id lzs-fun)))))

;;; -----------------------------------------------------------------------------------
;;; named constants and variable references
;;; -----------------------------------------------------------------------------------

(defmethod evaluate ((obj <named-const>))
  (make-syntactic (?value obj)))

(defmethod evaluate ((obj <var-ref>))
  (get-value (?var obj)))

;;; -----------------------------------------------------------------------------------
;;; special forms
;;; -----------------------------------------------------------------------------------

(defmethod evaluate ((obj <setq-form>))
  (set-value (?location obj) (evaluate (?form obj))))

(defmethod evaluate ((obj <progn-form>))
  (eval-progn (?form-list obj)))

(defun eval-progn (forms)
  (let ((value nil))
    (mapc (lambda (form)
            (setq value (evaluate form)))
          forms)
    value))

(defmethod evaluate ((obj <if-form>))
  (if (evaluate (?pred obj))
    (evaluate (?then obj))
    (evaluate (?else obj))))

(defmethod evaluate ((obj <switch-form>))
)

(defmethod evaluate ((obj <labeled-form>))
)

(defmethod evaluate ((obj <let*-form>))
  (with-new-values (?var-list obj) () (?init-list obj)
    (evaluate (?body obj))))

(defmethod evaluate ((obj <labels-form>))
)

(defmethod evaluate ((obj <let/cc-form>))
)

(defmethod evaluate ((obj <tagbody-form>))
)

(defmethod evaluate ((obj <tagged-form>))
)

(defmethod evaluate ((obj <mv-lambda>))
)

;;; -----------------------------------------------------------------------------------
;;; transforming literals to syntactic constructs
;;; -----------------------------------------------------------------------------------
;;; make-syntactic transforms LZS-literals to objects of the compile time
;;; environment such that they can act as syntaxtic constructs. For exxample in
;;; the case of structured-literals of the LZS, which must be transformed to
;;; "normal" literals

(defgeneric make-syntactic (obj))

(defmethod make-syntactic ((obj <object>))
  obj)

(defmethod make-syntactic ((obj <sym>))
  (make-symbol (?name obj)))

(defmethod make-syntactic ((obj <structured-literal>))
  (make-structured-literal-syntactic (?value obj)))

(defgeneric make-structured-literal-syntactic (obj)
  ; this function is in some sense the counterpart of lzslit (el2lzs-rules)
  )

(defmethod make-structured-literal-syntactic (obj) 
  ; the default case
  (make-syntactic obj))

(defmethod make-structured-literal-syntactic ((obj <string>))
  obj)

(defmethod make-structured-literal-syntactic ((obj <vector>))
  (map #'make-syntactic obj))

(defmethod make-structured-literal-syntactic ((obj <pair>))
  (make-list-syntactic obj))

(defun make-list-syntactic (l)
  (cond ((null l) nil)
        ((atom l) (make-syntactic l))
        (t (cons (make-list-syntactic (car l))
                 (make-list-syntactic (cdr l))))))

;;; -----------------------------------------------------------------------------------
;;; module initialization
;;; -----------------------------------------------------------------------------------

(defun initialize-module (name)
  (call (?toplevel-forms (find-module name))
        nil))

#module-end
