;;;-*- Mode: Lisp; Syntax: Common-Lisp; Base: 10; Package: ti-lattice -*-
#|
-------------------------------------------------------------------------------
TITLE: Lattice and Lattice Types Used for Type Inference
-------------------------------------------------------------------------------
File:    ti-lattice.em
Version: 1.36 (last modification on Tue Oct 12 15:28:50 1993)
State:   proposed

DESCRIPTION:
Implementation of an complementary lattice with intersection (meet), union 
(join) and complement of lattice types. Lattice types are used in atomic type
expressions (-> ti-exprs).

DOCUMENTATION:

NOTES:

REQUIRES:

PROBLEMS:

AUTHOR:
a.kind

CONTACT: 
a.kind (andreas.kind@isst.fhg.de)

HISTORY: 
Log for /tmp_mnt/net/saturn/apply/Lisp/Apply/ti-lattice.em[1.0]
	Tue Apr  6 13:22:52 1993 akind@isst save $
 
ti-lattice.em[1.1] Tue Apr  6 13:54:51 1993 akind@isst save $
 [Tue Apr  6 13:49:29 1993] Intention for change:
 
ti-lattice.em[1.2] Tue Apr  6 13:56:27 1993 akind@isst save $
 [Tue Apr  6 13:56:15 1993] Intention for change:
 
ti-lattice.em[1.3] Tue Apr  6 15:42:14 1993 akind@isst save $
 [Tue Apr  6 15:41:58 1993] Intention for change:
 
ti-lattice.em[1.4] Tue Apr  6 16:13:54 1993 akind@isst save $
 
ti-lattice.em[1.5] Tue Apr  6 16:26:50 1993 akind@isst save $
 
ti-lattice.em[1.6] Tue Apr  6 16:30:12 1993 akind@isst save $
 
ti-lattice.em[1.7] Tue Apr  6 16:32:27 1993 akind@isst save $
 
ti-lattice.em[1.8] Tue Apr  6 16:41:15 1993 akind@isst save $
 
ti-lattice.em[1.9] Tue Apr  6 17:19:56 1993 akind@isst save $
 
ti-lattice.em[1.10] Wed Apr  7 09:17:12 1993 akind@isst proposed $
 
ti-lattice.em[1.11] Wed Apr  7 11:10:08 1993 akind@isst save $
 
ti-lattice.em[1.12] Wed Apr  7 11:45:39 1993 akind@isst proposed $
 
ti-lattice.em[1.13] Wed Apr  7 14:59:01 1993 akind@isst proposed $
 
ti-lattice.em[1.14] Wed Apr  7 15:26:49 1993 akind@isst proposed $
 
ti-lattice.em[1.15] Tue Apr 13 18:39:15 1993 akind@isst proposed $
 
ti-lattice.em[1.16] Wed Apr 14 09:58:44 1993 akind@isst proposed $
 
ti-lattice.em[1.17] Wed Apr 14 11:46:21 1993 akind@isst proposed $
 
ti-lattice.em[1.18] Wed Apr 14 13:41:11 1993 akind@isst proposed $
 
ti-lattice.em[1.19] Wed Apr 14 20:11:24 1993 akind@isst proposed $
 
ti-lattice.em[1.20] Fri Apr 16 18:18:07 1993 akind@isst proposed $
 
ti-lattice.em[1.21] Mon Apr 19 18:04:32 1993 akind@isst proposed $
 
ti-lattice.em[1.22] Wed Apr 21 16:34:13 1993 akind@isst save $
 
ti-lattice.em[1.23] Wed Apr 21 16:44:37 1993 akind@isst proposed $
 
ti-lattice.em[1.24] Wed May  5 11:25:32 1993 akind@isst proposed $
 
ti-lattice.em[1.25] Tue May 18 17:23:23 1993 akind@isst proposed $
 
ti-lattice.em[1.26] Mon May 24 12:08:34 1993 akind@isst proposed $
 
ti-lattice.em[1.27] Mon May 24 13:43:08 1993 akind@isst proposed $
 
ti-lattice.em[1.28] Mon May 24 14:26:33 1993 akind@isst proposed $
 
ti-lattice.em[1.29] Thu Jun  3 11:16:22 1993 akind@isst proposed $
 
ti-lattice.em[1.30] Tue Aug  3 11:47:07 1993 akind@isst proposed $
 
ti-lattice.em[1.31] Fri Aug 27 17:29:54 1993 akind@isst published $
 
ti-lattice.em[1.32] Tue Sep 14 12:40:35 1993 akind@isst save $
 
ti-lattice.em[1.33] Thu Sep 23 14:33:15 1993 akind@isst proposed $
 [Tue Sep 14 15:17:15 1993] Intention for change:
 
ti-lattice.em[1.34] Fri Oct  1 14:41:24 1993 akind@isst save $
 [Thu Sep 23 15:44:19 1993] Intention for change:
 
ti-lattice.em[1.35] Mon Oct 11 10:30:29 1993 akind@isst proposed $
 
ti-lattice.em[1.36] Tue Oct 12 17:15:12 1993 akind@isst proposed $
 

-------------------------------------------------------------------------------
|#

#module-name ti-lattice
#module-import (lzs lzs-modules lzs-mop tail-module expand-literal ti ti-codes
		    (only (make-hash-table gethash maphash remhash clrhash
                           delete) common-lisp))
#module-syntax-import (ti)
#module-syntax-definitions
#module-header-end 

;;; ---------------------------------------------------------------------------
;;; EXPORT
;;; ---------------------------------------------------------------------------

(EXPORT <lattice-type> ?name ?code ?class ?supertypes ?subtypes ?strategic
	               ?compound ?write-access-stamp ?atomic-expr
	<lattice> ?latest-write-access-stamp
	*the-lattice* *the-lattice-table* initialize-lattice
	*top* *bottom* *top-code* *bottom-code* *%void* *%void-code*
	*%false* *%false-code* *%object* *%object-code*
	*%integer* *%integer-code* *%function* *%function-code*
	*<null>* *<null>-code* *%class* *%class-code*
	*<functon>* *<function>-code*
	get-lattice-type get-lattice-code
	get-defined-lattice-type get-lattice-type-class
	def-sys-lattice-type def-strategic-lattice-type
	add-lattice-type ~compute-lattice-type compute-normalized-lattice-type
	join-lattice-types meet-lattice-types complement-lattice-type-code
	meet-lattice-types-p lattice-subtype-p
	top-lattice-type-p bottom-lattice-type-p
	eq-lattice-type copy-lattice-type
	expand-lattice-type-values reset-lattice-type-values
	find-lattice-type-for-literal
	set-write-access-stamp new-write-access-stamp
	convert-to-super-non-compound-type)

;;; ---------------------------------------------------------------------------
;;; TYPE LATTICE CLASSES
;;; ---------------------------------------------------------------------------

;; Structure of types in the type lattice.
(DEFSTANDARDCLASS <lattice-type> ()
  (name	:accessor :initarg :initform nil)
  (class :accessor :initarg :initform nil)
  (code	:accessor :initarg)
  (atomic-expr :accessor :initarg :initform nil)
  (subtypes :accessor :initarg :initform ())
  (supertypes :accessor :initarg :initform ())
  (strategic :accessor :initarg :initform nil)
  (compound :accessor :initarg :initform nil)
  (write-access-stamp :accessor :initarg :initform nil))

;; Structure of the lattice used for type inference.
(DEFSTANDARDCLASS <lattice> ()
  (top :accessor :initarg :initform nil)
  (bottom :accessor :initarg :initform nil)
  (table :accessor :initform (make-hash-table))
  (latest-write-access-stamp :accessor :initform 0))

;; The global type inference lattice.
(DEFLOCAL *the-lattice* (make <lattice>))

;; The hash table with all of the global type lattice with all lattice types.
(DEFLOCAL *the-lattice-table* (?table *the-lattice*))

;; Here come some ofter used lattice types with their codes; they have to be
;; updated every time a new lattice type is added to *the-lattice*.
(DEFLOCAL *top* ())
(DEFLOCAL *top-code* ())
(DEFLOCAL *bottom* ())
(DEFLOCAL *bottom-code* ())
(DEFLOCAL *%void* ())
(DEFLOCAL *%void-code* ())
(DEFLOCAL *%false* ())
(DEFLOCAL *%false-code* ())
(DEFLOCAL *%object* ())
(DEFLOCAL *%object-code* ())
(DEFLOCAL *%function* ())
(DEFLOCAL *%function-code* ())
(DEFLOCAL *%integer* ())
(DEFLOCAL *%integer-code* ())
(DEFLOCAL *%class* ())
(DEFLOCAL *%class-code* ())
(DEFLOCAL *<function>* ())
(DEFLOCAL *<function>-code* ())
(DEFLOCAL *<null>* ())
(DEFLOCAL *<null>-code* ())

;;; Initializing the lattice; called when loading this file and in
;;; apply-compiler for reset.
(DEFUN initialize-lattice ()
  (ti-format t "~%Initializing the lattice ...")
  (reset-used-codes)
  (setf (?latest-write-access-stamp *the-lattice*) 0)
  (setq *top* (make <lattice-type> :name ^top :code (next-code) :strategic #t))
  (setq *bottom* (make <lattice-type> :name ^bottom :code 0 :strategic #t))
  (setf (?top *the-lattice*) *top*)
  (setf (?bottom *the-lattice*) *bottom*)
  (clrhash *the-lattice-table*)
  (setf (gethash (?name *top*) *the-lattice-table*) *top*)
  (setf (gethash (?name *bottom*) *the-lattice-table*) *bottom*)
  (setf (?subtypes *top*) (list *bottom*))
  (setf (?supertypes *bottom*) (list *top*))
  (ti-format t " done."))

(DEFUN the-lattice () *the-lattice*)

;;; ---------------------------------------------------------------------------
;;; FILLING THE LATTICE
;;; ---------------------------------------------------------------------------

;;; Read in the external describtion of a lattice.
(DEFUN def-sys-lattice-type (type-def)
  (let* ((name (car type-def))
	 (supers (car (cdr type-def)))
	 (subs (car (cdr (cdr type-def))))
	 (class (find-lexical name $tail-module)) ;lookup for TAIL class
	 (new-lattice-type (make <lattice-type> :name name :class class)))
    (add-lattice-type-between new-lattice-type supers subs)
    (if class (setf (?lattice-type class) new-lattice-type))))

;;; Read in the external describtion of a lattice.
(DEFUN def-strategic-lattice-type (type-def)
  (let* ((name (car type-def))
	 (supers (car (cdr type-def)))
	 (subs (car (cdr (cdr type-def))))
	 (class (find-lexical name $tail-module)) ;lookup for TAIL class
	 (new-lattice-type (make <lattice-type>
				 :name name
				 :class class
				 :strategic #t)))
    (add-lattice-type-between new-lattice-type supers subs)
    (if class (setf (?lattice-type class) new-lattice-type))))

;;; Add a lattice type at a specified position into a lattice.
(DEFGENERIC add-lattice-type-between
    (new-lattice-type supertype-names subtype-names))

(DEFMETHOD add-lattice-type-between
    ((new-lattice-type <lattice-type>)
     supertype-names
     subtype-names)
  (ti-format t "~%Add lattice type ~A." (?name new-lattice-type))
  (setf (?code new-lattice-type) (next-code))
  (let ((supers (get-lattice-types supertype-names))
	(subs (get-lattice-types subtype-names)))
    (remove-lattice-links-between supers subs)
    (add-lattice-links-between supers (list new-lattice-type))
    (add-lattice-links-between (list new-lattice-type) subs)
    (update-supertypes new-lattice-type supers)
    (update-lattice-constants))
  (setf (gethash (?name new-lattice-type) *the-lattice-table*)
    new-lattice-type))

;;; Adding strategic lattice types to the lattice; the last argument if
;;; not nil is a list with a flag if type is compound and some literal values,
;;; which are later expanded.
(DEFUN add-lattice-type (name supers subs compound&values)
  (let* ((is-compound (if compound&values (car compound&values) nil))
	 (values (if compound&values (cdr compound&values) ()))
	 (new-lattice-type (make <lattice-type>
				:name name
				:strategic t
				:compound is-compound)))
    (add-lattice-type-values new-lattice-type values)
    (add-lattice-type-between new-lattice-type supers subs)))

;;; Adding lattice type values to *tmp-lattice-type-values*, which are later
;;; be expanded to literals and collected in the table *lattice-type-values*.
(DEFLOCAL *tmp-lattice-type-values* ())
(DEFLOCAL *lattice-type-values* (make-hash-table))

(DEFUN reset-lattice-type-values ()
  (setq *tmp-lattice-type-values* ())
  (setq *lattice-type-values* (make-hash-table)))

(DEFUN add-lattice-type-values (lattice-type values)
  (if values
      (let ((new-entry (cons (car values) lattice-type)))
	(setq *tmp-lattice-type-values*
	  (cons new-entry *tmp-lattice-type-values*))
	(add-lattice-type-values lattice-type (cdr values)))))

(DEFUN expand-lattice-type-values ()
  (mapc #'expand-lattice-type-value *tmp-lattice-type-values*)
  (setq *tmp-lattice-type-values* ()))

(DEFUN expand-lattice-type-value (entry)
  (let ((literal (expand-literal (car entry)))
	(lattice-type (cdr entry)))
    (setf (gethash literal *lattice-type-values*) lattice-type)))

(DEFUN find-lattice-type-for-literal (literal)
  (gethash literal *lattice-type-values*))
	
;;; Adding non-strategic lattice types to the lattice.
;;; Corresponding generic function is defined in lzs-mop.
(DEFMETHOD ~compute-lattice-type ((class <standard-class-def>)
				  supers lattice-supers)
  (let ((new-lattice-type (make <lattice-type>
				:name (?identifier class)
				:class class))
	(subs (list ^bottom)))
    (add-lattice-type-between new-lattice-type 
			      (append supers lattice-supers)
			      subs)))

(DEFMETHOD ~compute-lattice-type ((class <tail-class-def>)
				  supers lattice-supers)
  (if (and (null supers) (null lattice-supers))
      (call-next-method class nil (list ^%struct))
    (call-next-method)))

;;; ---------------------------------------------------------------------------
;;; ACCESSING LATTICE TYPES
;;; ---------------------------------------------------------------------------

;;; Return a list of lattice types corresponding to a list of type name.
(DEFUN get-lattice-types (type-names)
  (if (null type-names)
      (list *top*)
    (mapcar #'get-lattice-type type-names)))

(DEFGENERIC get-lattice-type (obj))

(DEFMETHOD get-lattice-type (obj)
  (let ((lattice-type (gethash obj *the-lattice-table*)))
    (cond (lattice-type
	   lattice-type)
	  (t (ti-format t "~%Warning: no lattice type ~S; using top." obj)
	     (ti-error "error")
	     *top*))))

(DEFMETHOD get-lattice-type ((obj <lattice-type>))
  obj)

(DEFMETHOD get-lattice-type ((obj <class-def>))
  (?lattice-type obj))

;;; Answer the code of a named lattice-type.
(DEFUN get-lattice-code (name)
  (?code (get-lattice-type name)))

;;; Answer the strategic lattice type identified by obj.
(DEFUN get-defined-lattice-type (obj)
  (let ((lattice-type (gethash obj *the-lattice-table*)))
    (if (and lattice-type (?strategic lattice-type))
	lattice-type
      nil)))

;;; ---------------------------------------------------------------------------
;;; Remove all subtype-/supertype-links between the elements of two type-lists.
(DEFUN remove-lattice-links-between (lattice-supertypes lattice-subtypes)
  (dolist (subtype lattice-subtypes)
    (dolist (lattice-type lattice-supertypes)
     (remove-lattice-subtype lattice-type subtype)))
  (dolist (supertype lattice-supertypes)
    (dolist (lattice-type lattice-subtypes)
     (remove-lattice-supertype lattice-type supertype))))

(DEFUN remove-lattice-subtype (lattice-type subtype)
  (delete subtype (?subtypes lattice-type)))

(DEFUN remove-lattice-supertype (lattice-type supertype)
  (delete supertype (?supertypes lattice-type)))

;;; ---------------------------------------------------------------------------
;;; Add subtype-/supertype-links between the elements of two type-lists.
(DEFUN add-lattice-links-between (lattice-supertypes lattice-subtypes)
  (dolist (subtype lattice-subtypes)
    (dolist (lattice-type lattice-supertypes)
     (add-lattice-subtype lattice-type subtype)))
  (dolist (supertype lattice-supertypes)
    (dolist (lattice-type lattice-subtypes)
     (add-lattice-supertype lattice-type supertype))))

(DEFUN add-lattice-subtype (lattice-type subtype)
  (setf (?subtypes lattice-type)
    (cons subtype (?subtypes lattice-type))))

(DEFUN add-lattice-supertype (lattice-type supertype)
  (setf (?supertypes lattice-type)
    (cons supertype (?supertypes lattice-type))))

;;; Tell all supertypes of a new subtype.
(DEFUN update-supertypes (new-lattice-type lattice-supertypes)
  (dolist (type lattice-supertypes)
    (setf (?code type)
      (join-codes (?code type) (?code new-lattice-type)))
    (let ((expr (?atomic-expr type)))
      (if expr
	  (setf (?code expr) (?code type))))
    (update-supertypes new-lattice-type (?supertypes type))))

(DEFUN update-lattice-constants ()
  (setq *top* (?top *the-lattice*))
  (setq *top-code* (?code *top*))
  (setq *bottom* (?bottom *the-lattice*))
  (setq *bottom-code* (?code *bottom*))
  (let ((lattice-type (gethash ^%function *the-lattice-table*)))
    (cond (lattice-type
	   (setq *%function* lattice-type)
	   (setq *%function-code* (?code lattice-type)))))
  (let ((lattice-type (gethash ^%integer *the-lattice-table*)))
    (cond (lattice-type
	   (setq *%integer* lattice-type)
	   (setq *%integer-code* (?code lattice-type)))))
  (let ((lattice-type (gethash ^%void *the-lattice-table*)))
    (cond (lattice-type
	   (setq *%void* lattice-type)
	   (setq *%void-code* (?code lattice-type)))))
  (let ((lattice-type (gethash ^%false *the-lattice-table*)))
    (cond (lattice-type
	   (setq *%false* lattice-type)
	   (setq *%false-code* (?code lattice-type)))))
  (let ((lattice-type (gethash ^%object *the-lattice-table*)))
    (cond (lattice-type
	   (setq *%object* lattice-type)
	   (setq *%object-code* (?code lattice-type)))))
  (let ((lattice-type (gethash ^%class *the-lattice-table*)))
    (cond (lattice-type
	   (setq *%class* lattice-type)
	   (setq *%class-code* (?code lattice-type)))))
  (let ((lattice-type (gethash ^<function> *the-lattice-table*)))
    (cond (lattice-type
	   (setq *<function>* lattice-type)
	   (setq *<function>-code* (?code lattice-type)))))
  (let ((lattice-type (gethash ^<null> *the-lattice-table*)))
    (cond (lattice-type
	   (setq *<null>* lattice-type)
	   (setq *<null>-code* (?code lattice-type))))))

(DEFGENERIC get-lattice-type-class (lattice-type))

(DEFMETHOD get-lattice-type-class ((lattice-type <lattice-type>))
  (let ((class (?class lattice-type)))
    (if class class
      (let ((classes-of-supers
	     (mapcar #'get-lattice-type-class (?supertypes lattice-type))))
	(if classes-of-supers (car classes-of-supers)
	  nil)))))

;;; ---------------------------------------------------------------------------
;;; MEET/JOIN/COMPLEMENT OF LATTICE TYPES
;;; ---------------------------------------------------------------------------

;;; Answer a normalized intersection of two lattice types.
(DEFGENERIC meet-lattice-types (lattice-type1 lattice-type2))

(DEFMETHOD meet-lattice-types
    ((lattice-type1 <lattice-type>)
     (lattice-type2 <lattice-type>))
  (let ((code1 (?code lattice-type1))
	(code2 (?code lattice-type2)))
    (cond ((subcode-p code1 code2) lattice-type1)
	  ((subcode-p code2 code1) lattice-type2)
	  (t (ti-format t "~%Warning: can't meet lattice types")
	     (ti-error)
	     *top*))))

;;; Answer a normalized union of two lattice types.
(DEFGENERIC join-lattice-types (lattice-type1 lattice-type2))

(DEFMETHOD join-lattice-types
    ((lattice-type1 <lattice-type>)
     (lattice-type2 <lattice-type>))
  (let ((code1 (?code lattice-type1))
	(code2 (?code lattice-type2)))
    (cond ((subcode-p code1 code2) lattice-type2)
	  ((subcode-p code2 code1) lattice-type1)
	  (t
	   (let ((joins1
		  (mapcar (lambda (supertype)
			    (join-lattice-types supertype lattice-type2))
			  (?supertypes lattice-type1)))
		 (joins2
		  (mapcar (lambda (supertype)
			    (join-lattice-types supertype lattice-type1))
			  (?supertypes lattice-type2))))
	     (if joins1
		 (car joins1)
	       (if joins2
		   (car joins2)
		 (progn
		   (ti-format t "~%Warning: can't join lattice types")
		   (ti-error)
		   *top*))))))))
 
;;; Answer complement code of a lattice type.
(DEFGENERIC complement-lattice-type-code (lattice-type))

(DEFMETHOD complement-lattice-type-code
    ((lattice-type <lattice-type>))
  (complement-code (?code lattice-type)))
 
;;; Answer whether all subtypes of first arg meet with a given lattice-type.
(DEFGENERIC meet-all-subtypes-p (supertype lattice-type))

(DEFMETHOD meet-all-subtypes-p ((supertype <lattice-type>)
				(lattice-type <lattice-type>))
  (null (member-with-args (lambda (subtype)
			    (null (meet-lattice-types-p subtype lattice-type)))
			  (?subtypes supertype))))
 
;;; Answer the normalized complement of a lattice type.
(DEFGENERIC normalized-complement-lattice-type (lattice-type))

(DEFMETHOD normalized-complement-lattice-type ((lattice-type <lattice-type>))
  *top*)

(DEFUN compute-normalized-lattice-type (name)
  (if (consp name)
      (let ((op-symbol (car name))
	    (arg-def (cdr name)))
	(cond ((eq ^not op-symbol)
	       (normalized-complement-lattice-type
		(compute-normalized-lattice-type (car arg-def))))
	      ((eq ^and op-symbol)
	       (meet-lattice-types
		(compute-normalized-lattice-type (car arg-def))
		(compute-normalized-lattice-type (car (cdr arg-def)))))
	      ((eq ^or op-symbol)
	       (join-lattice-types
		(compute-normalized-lattice-type (car arg-def))
		(compute-normalized-lattice-type (car (cdr arg-def)))))
	      (t
	       (ti-format t "~%Warning: no correct atomic type definition")
	       *top*)))
    name))

;;; ---------------------------------------------------------------------------
;;; TESTING
;;; ---------------------------------------------------------------------------

;;; Answer whether a lattice type denotes the bottom type.
(DEFGENERIC bottom-lattice-type-p (lattice-type))

(DEFMETHOD bottom-lattice-type-p ((lattice-type <lattice-type>))
  (eq-code-p (?code lattice-type) *bottom-code*))

;;; Answer whether a lattice type denotes the top type.
(DEFGENERIC top-lattice-type-p (lattice-type))

(DEFMETHOD top-lattice-type-p ((lattice-type <lattice-type>))
  (eq-code-p (?code lattice-type) *top-code*))

;;; Answer whether the intersection of two lattice types is not *bottom-type*.
(DEFGENERIC meet-lattice-types-p (lattice-type1 lattice-type2))
  
(DEFMETHOD meet-lattice-types-p
    ((lattice-type1 <lattice-type>)
     (lattice-type2 <lattice-type>))
  (meet-codes-p (?code lattice-type1)(?code lattice-type2)))

;;; Answer whether one lattice type is a subtype of another.
(DEFGENERIC lattice-subtype-p (lattice-type1 lattice-type2))
  
(DEFMETHOD lattice-subtype-p
    ((lattice-type1 <lattice-type>)
     (lattice-type2 <lattice-type>))
  (subcode-p (?code lattice-type1) (?code lattice-type2)))

;;; Answer whether two lattice types are equal.
(DEFGENERIC eq-lattice-type (lattice-type1 lattice-type2))
  
(DEFMETHOD eq-lattice-type
    ((lattice-type1 <lattice-type>)
     (lattice-type2 <lattice-type>))
  (eq-code-p (?code lattice-type1) (?code lattice-type2)))

;;; Answer whether the class corresponding to a lattice type is exported.
(DEFGENERIC exported-p (lattice-type))
  
(DEFMETHOD exported-p
    ((lattice-type <lattice-type>))
  #t
;  (let ((application-class (?class lattice-type)))
;    (if (eq nil application-class)
;	#t
;      (?exported application-class)))
  )

;;; ---------------------------------------------------------------------------
;;; WRITE ACCESSES OF COMPOUND TYPES 
;;; ---------------------------------------------------------------------------

(DEFUN new-write-access-stamp ()
  (let ((new-stamp (+ (?latest-write-access-stamp *the-lattice*) 1)))
    (setf (?latest-write-access-stamp *the-lattice*) new-stamp)
    new-stamp))

(DEFGENERIC set-write-access-stamp (lattice-type))

(DEFMETHOD set-write-access-stamp ((lattice-type <lattice-type>))
  (let ((new-stamp (new-write-access-stamp)))
    (setf (?write-access-stamp lattice-type) new-stamp)
    (set-subtype-write-access-stamps lattice-type new-stamp)))

(DEFGENERIC set-subtype-write-access-stamps (lattice-type stamp))

(DEFMETHOD set-subtype-write-access-stamps ((lattice-type <lattice-type>)
					   (new-stamp <spint>))
  (dolist (subtype (?subtypes lattice-type))
    (setf (?write-access-stamp subtype) new-stamp)
    (set-subtype-write-access-stamps subtype new-stamp)))

(DEFGENERIC copy-lattice-type (lattice-type))

(DEFMETHOD copy-lattice-type ((lattice-type <lattice-type>))
  (make <lattice-type>
	:name (?name lattice-type)
	:class (?class lattice-type)
	:code (?code lattice-type)
	:supertypes (?supertypes lattice-type)
	:subtypes (?subtypes lattice-type)
	:write-access-stamp (?write-access-stamp lattice-type)
	:strategic (?strategic lattice-type)
	:compound (?compound lattice-type)))

;; Get the next super lattice type that is not compound and copy all slot
;; values to the given lattice type.
(DEFGENERIC convert-to-super-non-compound-type (lattice-type))

(DEFMETHOD convert-to-super-non-compound-type ((lattice-type <lattice-type>))
  (let ((non-compound-type (find-super-non-compound-type lattice-type)))
    (ti-format t "~%convert ~A to ~A"
	       (?name lattice-type)
	       (?name non-compound-type))
    non-compound-type))

(DEFGENERIC find-super-non-compound-type (lattice-type))

(DEFMETHOD find-super-non-compound-type ((lattice-type <lattice-type>))
  (let* ((supers (?supertypes lattice-type))
	 (type (member-with-args (lambda (x)
				   (null (?compound x))) supers)))
    (if type (car type)
      (find-super-non-compound-type (car supers)))))

#module-end