;;;-*- Mode: Lisp; Syntax: Common-Lisp; Base: 10; Package: eval -*-
#|
-----------------------------------------------------------------------------------
TITLE: An interpreter for the LZS
-----------------------------------------------------------------------------------
File:    eval.em
Version: 1.4 (last modification on Wed Aug 25 11:00:15 1993)
State:   published

DESCRIPTION:
The function EVAL evaluates an expression given in LZS-form. The function CALL
applies an LZS-function to some arguments given also as LZS-objects. The
interpreter is first used during macro expansion, but can also be used by
compile-time evaluation of constant expressions and during activation of the
MOP. The installation of interpretation functions for LZS-functions which body
should not interpreted is done by %describe-function of TAIL with the option
"interpreter". The value given to this option can be one of the symbols in
$system-function-table, which provides some functions of the compilation
environment to the interpreter.

DOCUMENTATION:

NOTES:
Up to now not all special forms are supported.
$system-function-table should contain later all "basic" EuLisp functions. (What
basic means?)

REQUIRES:

PROBLEMS:
- What must be done to support MOP-activation during compile time?

AUTHOR:
Ingo Mohr

CONTACT: 
Ingo Mohr

HISTORY: 
Log for /export/home/saturn/imohr/Lisp/Apply/eval.em[1.0]
	Fri Feb 26 17:01:11 1993 imohr@isst proposed $
 The LZS-interpreter used by syntax expansion.
 
eval.em[1.1] Fri Mar 19 13:32:56 1993 imohr@isst proposed $
 set interpreter with 3 Args
 
eval.em[1.2] Fri May  7 15:02:58 1993 imohr@isst save $
 + interpreter function null; some error -> warn
 
eval.em[1.3] Fri May  7 17:11:30 1993 imohr@isst proposed $
 + interpreter function char-code
 
eval.em[1.4] Wed Aug 25 15:37:08 1993 ukriegel@isst published $
 [Wed Aug 25 10:41:16 1993] Intention for change:
 add + for fixed-precision-integer
 done
 

-----------------------------------------------------------------------------------
|#

#module-name eval
#module-import
(level-1-eulisp lzs
 (only (find-module) el2lzs-main)
 (only (error warn mapcar mapc assoc rest first append char-code) common-lisp))
#module-syntax-import 
(level-1-eulisp
 (only (push) common-lisp))
#module-syntax-definitions

;;; The forms in body are executed with variables in vars bound to the
;;; corresponding values of the LZS-forms in values (computed by eval) and the
;;; variable in rst (unless it is () ) bound to the list of values not used for
;;; the other variables. The macro can be used for handling function
;;; applications and let*-forms of the LZS.

(defmacro with-new-values (vars rst values . body)
  `(dynamic-let ((variable-environment (dynamic variable-environment)))
                (initialize-variables ,vars ,rst ,values)
                ,@body))

(defmacro declare-system-functions functions
  `(list ,@(mapcar (lambda (fun)
                     `(cons ',(make-eulisp-symbol fun)
                            #',fun))
                   functions)))

#module-header-end

(export eval set-interpreter call)

(defgeneric eval (lzsobj))


;;; -----------------------------------------------------------------------------------
;;; the default case: return the object after transformation to a syntactic object
;;; -----------------------------------------------------------------------------------

(defmethod eval ((obj <object>))
  (make-syntactic obj))

;;; -----------------------------------------------------------------------------------
;;; function application
;;; -----------------------------------------------------------------------------------

;;; Because of the optimization during transformation from EuLisp to LZS only
;;; <var-ref>, <fun> or <named-const> can appear as operator

(defmethod eval ((obj <app>))
  (call (eval (?function obj)) (?arg-list obj)))

(defgeneric call (fn args))

(defmethod call ((fun <simple-fun>) args)
  (let ((interpreter (?interpreter fun))
        (body (?body fun)))
    (cond (interpreter
           (apply interpreter (mapcar #'eval args)))
          (body 
           (with-new-values (?var-list (?params fun))
             (?rest (?params fun))
             args
             (eval body)))
          (t (error "can't interpret function ~A" fun)))))

;;; -----------------------------------------------------------------------------------
;;; $system-function-table
;;; -----------------------------------------------------------------------------------
;;; Is used by %describe-function to install interpreter functions


(defconstant $system-function-table
  (declare-system-functions
   car cdr
   list null
   cons append ; needed by Backquote
   eq
   char-code ; needed for expanding character literals
   +         ; needed for :fixed-precision-integer :big
   ))

(defun set-interpreter (lzs-fun annotate-key id)
  (let ((entry (assoc id $system-function-table)))
    (if entry
      (setf (?interpreter lzs-fun)
            (cdr entry))
      (warn "undefined interpreter ~A for function ~A of module ~A" 
             id (?identifier lzs-fun) (?module-id lzs-fun)))))

;;; -----------------------------------------------------------------------------------
;;; initialization, setting and retrieving values of variables
;;; -----------------------------------------------------------------------------------

(defvar variable-environment () )

(defun initialize-variables (vars rst values)
(cond ((null vars) (initialize-rest rst values))
      ((null values) (error "too few arguments"))
      (t (push (cons (first vars) (eval (first values)))
               (dynamic variable-environment))
         (initialize-variables (rest vars) rst (rest values)))))

(defun initialize-rest (rst values)
  (cond ((and (null rst) values)
         (error "too many arguments"))
        ((null rst))
        (t (push (cons rst (mapcar #'eval values))
                 (dynamic variable-environment)))))

(defgeneric get-value (var))

(defmethod get-value ((var <static>))
 (cdr (assoc var (dynamic variable-environment))))

(defmethod get-value ((var <named-const>))
 (?eval-value var))

(defgeneric set-value (var-obj obj))

(defmethod set-value ((var <static>) obj)
 (setf (cdr (assoc var (dynamic variable-environment)))
       obj))

(defmethod set-value ((var <named-const>) obj)
 (setf (?eval-value var) obj))

;;; -----------------------------------------------------------------------------------
;;; named constants and variable references
;;; -----------------------------------------------------------------------------------

(defmethod eval ((obj <named-const>))
  (make-syntactic (?value obj)))

(defmethod eval ((obj <var-ref>))
  (get-value (?var obj)))

;;; -----------------------------------------------------------------------------------
;;; special forms
;;; -----------------------------------------------------------------------------------

(defmethod eval ((obj <setq-form>))
  (set-value (?location obj) (eval (?form obj))))

(defmethod eval ((obj <progn-form>))
  (eval-progn (?form-list obj)))

(defun eval-progn (forms)
  (let ((value nil))
    (mapc (lambda (form)
            (setq value (eval form)))
          forms)
    value))

(defmethod eval ((obj <if-form>))
  (if (eval (?pred obj))
    (eval (?then obj))
    (eval (?else obj))))

(defmethod eval ((obj <switch-form>))
)

(defmethod eval ((obj <labeled-form>))
)

(defmethod eval ((obj <let*-form>))
  (with-new-values (?var-list obj) () (?init-list obj)
    (eval (?body obj))))

(defmethod eval ((obj <labels-form>))
)

(defmethod eval ((obj <let/cc-form>))
)

(defmethod eval ((obj <tagbody-form>))
)

(defmethod eval ((obj <tagged-form>))
)

(defmethod eval ((obj <mv-lambda>))
)

;;; -----------------------------------------------------------------------------------
;;; transforming literals to syntactic constructs
;;; -----------------------------------------------------------------------------------
;;; make-syntactic transforms LZS-literals to objects of the compile time
;;; environment such that they can act as syntaxtic constructs. For exxample in
;;; the case of structured-literals of the LZS, which must be transformed to
;;; "normal" literals

(defgeneric make-syntactic (obj))

(defmethod make-syntactic ((obj <object>))
  obj)

(defmethod make-syntactic ((obj <sym>))
  (make-symbol (?name obj)))

(defmethod make-syntactic ((obj <structured-literal>))
  (warn "structured literals are not yet handled: ~A = ~A" obj (?value obj)))

;;; -----------------------------------------------------------------------------------
;;; module initialization
;;; -----------------------------------------------------------------------------------
(export initialize-module)

(defun initialize-module (name)
  (call (?toplevel-forms (find-module name))
        nil))

#module-end
