/*

   syntax.c

   Copyright, 1993, Brent Benson.  All Rights Reserved.
   0.4 Revisions Copyright 1994, Joseph N. Wilson.  All Rights Reserved.
   
   Permission to use, copy, and modify this software and its
   documentation is hereby granted only under the following terms and
   conditions.  Both the above copyright notice and this permission
   notice must appear in all copies of the software, derivative works
   or modified version, and both notices must appear in supporting
   documentation.  Users of this software agree to the terms and
   conditions set forth in this notice.

*/

#include "syntax.h"
#include "eval.h"
#include "list.h"
#include "class.h"
#include "function.h"
#include "symbol.h"
#include "values.h"
#include "apply.h"
#include "boolean.h"
#include "number.h"
#include "misc.h"
#include "error.h"
#include "string.h"
#include <string.h>

extern void signal_handler_init ();

/* data structures */

struct syntax_entry
{
  Object sym;
  syntax_fun fun;
  struct syntax_entry *next;
};
#define SYNTAX_TABLE_SIZE 1024
struct syntax_entry *syntax_table[SYNTAX_TABLE_SIZE];

/* local variables and functions */

void install_syntax_entry (char *name, syntax_fun fun);
void bind_variables(Object init_list, int top_level);
void add_variable_binding( Object var, Object val, int top_level);

/* functions emobodying evaluation rules for forms */

static Object and_eval (Object form);
static Object begin_eval (Object form);
static Object bind_eval (Object form);
static Object bind_exit_eval (Object form);
static Object bind_methods_eval (Object form);
static Object case_eval (Object form);
static Object cond_eval (Object form);
static Object define_eval (Object form);
static Object define_class_eval (Object form);
static Object define_generic_function_eval (Object form);
static Object define_method_eval (Object form);
static Object define_test_eval (Object form);
static Object dotimes_eval (Object form);
static Object for_eval (Object form);
static Object for_each_eval (Object form);
static Object if_eval (Object form);
static Object method_eval (Object form);
static Object or_eval (Object form);
static Object quasiquote_eval (Object form);
static Object quote_eval (Object form);
static Object select_eval (Object form);
static Object set_eval (Object form);
static Object unless_eval (Object form);
static Object until_eval (Object form);
static Object unwind_protect_eval (Object form);
static Object when_eval (Object form);
static Object while_eval (Object form);


static Object process_test_result (Object name, Object options,
				   Object doc_string, Object result);
static Object record_failure (Object name, Object doc_string, Object result);
static Object record_success (Object name, Object doc_string, Object result);
static Object record_disabled (Object name, Object doc_string);


static char *syntax_operators[] =
{
  "and",
  "begin",
  "bind",
  "bind-exit",
  "bind-methods",
  "case",
  "cond",
  "define",
  "define-variable",
  "define-class",
  "define-generic-function",
  "define-method",
  "define-test",
  "dotimes",
  "for",
  "for-each",
  "if",
  "method",
  "or",
  "quasiquote",
  "quote",
  "select",
  "set!",
  "unless",
  "until",
  "unwind-protect",
  "when",
  "while",
};

static syntax_fun syntax_functions[] =
{
  and_eval,
  begin_eval,
  bind_eval,
  bind_exit_eval,
  bind_methods_eval,
  case_eval,
  cond_eval,
  define_eval,
  define_eval,
  define_class_eval,
  define_generic_function_eval,
  define_method_eval,
  define_test_eval,
  dotimes_eval,
  for_eval,
  for_each_eval,
  if_eval,
  method_eval,
  or_eval,
  quasiquote_eval,
  quote_eval,
  select_eval,
  set_eval,
  unless_eval,
  until_eval,
  unwind_protect_eval,
  when_eval,
  while_eval,
};

void
init_syntax_table (void)
{
  int numops, i;
  Object symbol;

  numops = sizeof (syntax_operators) / sizeof (char *);
  for ( i=0 ; i < numops ; ++i )
    {
      install_syntax_entry (syntax_operators[i],
			    syntax_functions[i]);
    }
}

syntax_fun 
syntax_function (Object sym)
{
  struct syntax_entry *entry;
  int h;

  h = ((int)sym) % SYNTAX_TABLE_SIZE;
  entry = syntax_table[h];
  while ( entry )
    {
      if (entry->sym == sym)
	{
	  return (entry->fun);
	}
      entry = entry->next;
    }
  return (NULL);
}

void 
install_syntax_entry (char *name, syntax_fun fun)
{
  struct syntax_entry *entry;
  Object sym;
  int h;

  sym = make_symbol (name);
  h = ((int)sym) % SYNTAX_TABLE_SIZE;
  entry = (struct syntax_entry *) 
    checking_malloc (sizeof (struct syntax_entry));
  entry->sym = sym;
  entry->fun = fun;
  entry->next = syntax_table[h];
  syntax_table[h] = entry;
}

/* functions that perform the special evaluation
   rules for syntax forms. */

static Object
and_eval (Object form)
{
  Object clauses, ret;
  int i;

  clauses = CDR (form);
  while (! NULLP (clauses)) {
      ret = eval (CAR (clauses));
      if (VALUESP (ret)) {
	  if (PAIRP (CDR (clauses))) {
	      ret = FIRSTVAL (ret);
	  } else {
	      return ret;
	  }
      }
      if (ret == false_object) {
	  return (false_object);
      }
      clauses = CDR (clauses);
  }
  return (ret);
}

static Object 
begin_eval (Object form)
{
  Object res;

  form = CDR (form);
  res = false_object;
  while (! NULLP (form)) {
      res = eval (CAR (form));
      form = CDR (form);
  }
  return (res);
}

static Object
bind_eval (Object form)
{
  Object bindings, body, binding, var, val, res;
  Object first, last, new, type;
  int value_count, i;

  if (NULLP (CDR (form))) {
      error ("malformed bind form", form, NULL);
  }
  bindings = SECOND (form);
  body = CDR (CDR (form));

  push_scope ();
  while (! NULLP (bindings)) {
      bind_variables( CAR (bindings), 0);
      bindings = CDR (bindings);
  }
  
  /* evaluate forms in body */
  while (! NULLP (body)) {
      res = eval (CAR (body));
      body = CDR (body);
  }
  pop_scope ();
  return (res);

}

static Object 
bind_exit_eval (Object form)
{
  Object exit_obj, sym, body, ret, val, sec;
  jmp_buf buf;

  if (NULLP (CDR (form)))
    {
      error ("malformed bind-exit form", form, NULL);
    }
  sec = SECOND (form);
  if (! PAIRP (sec))
    {
      error ("bind-exit: second argument must be a list containing a symbol", sec, NULL);
    }
  sym = CAR (sec);
  body = CDR (CDR (form));
  if (! SYMBOLP (sym))
    {
      error ("bind-exit: bad exit procedure name", sym, NULL);
    }
  exit_obj = make_exit (sym);
  ret = (Object) setjmp (*EXITRET(exit_obj));
  push_scope ();
  add_binding (sym, exit_obj);
  if (! ret )
    {
      ret = false_object;
      while (! NULLP (body))
	{
	  ret = eval (CAR (body));
	  body = CDR (body);
	}
      pop_scope ();
      return (ret);
    }
  else
    {
      pop_scope ();
      return (ret);
    }
}

static Object 
bind_methods_eval (Object form)
{
  Object specs, body, spec, ret;
  Object name, params, method_body, method;

  if (NULLP (CDR (form)))
    {
      error ("bind-methods: bad form", form, NULL);
    }
  specs = SECOND (form);
  body = CDR (CDR (form));
  
  push_scope ();
  /* first bind method names to dummy values */
  while (! NULLP (specs))
    {
      spec = CAR (specs);
      name = FIRST (spec);
      add_binding (name, false_object);
      specs = CDR (specs);
    }

  /* now, actually make the methods */
  specs = SECOND (form);
  while (! NULLP (specs))
    {
      spec = CAR (specs);
      name = FIRST (spec);
      params = SECOND (spec);
      method_body = CDR (CDR (spec));
      method = make_method (name, params, method_body, the_env, 0);
      modify_value (name, method);
      specs = CDR (specs);
    }

  /* evaluate the body forms */
  while (! NULLP (body))
    {
      ret = eval (CAR (body));
      body = CDR (body);
    }
  pop_scope ();
  return (ret);
}

static Object
case_eval (Object form)
{
  Object target_form, branches, branch;
  Object match_list, consequents, ret;

  if (NULLP (CDR (form)))
    {
      error ("malformed case", form, NULL);
    }
  target_form = eval (CAR (CDR (form)));

  if (NULLP (CDR (CDR (form))))
    {
      error ("malformed case", form, NULL);
    }
  branches = CDR (CDR (form));
  while (! NULLP (branches))
    {
      branch = CAR (branches);
      if (! PAIRP (branch))
	{
	  error ("case: malformed branch", branch, NULL);
	}
      match_list = CAR (branch);
      if ((match_list == true_object) || (match_list == else_keyword))
	{
	  consequents = CDR (branch);
	  ret = false_object;
	  while (! NULLP (consequents))
	    {
	      ret = eval (CAR (consequents));
	      consequents = CDR (consequents);
	    }
	  return (ret);
	}
      if (! PAIRP (match_list))
	{
	  error ("select: malformed test expression", match_list, NULL);
	}
      while (! NULLP (match_list))
	{
	  if (id_p (CAR (match_list), target_form, make_empty_list())
	      != false_object)
	    {
	      consequents = CDR (branch);
	      ret = false_object;
	      while (! NULLP (consequents))
		{
		  ret = eval (CAR (consequents));
		  consequents = CDR (consequents);
		}
	      return (ret);
	    }
	  match_list = CDR (match_list);
	}
      branches = CDR (branches);
    }
  error ("case: no matching clause", target_form, NULL);
}

static Object
cond_eval (Object form)
{
  Object clauses, clause, test, ret;

  clauses = CDR (form);
  while (! NULLP (clauses))
    {
      clause = CAR (clauses);
      test = CAR (clause);
      ret = eval (test);
      if (VALUESP (ret)) {
	  ret = FIRSTVAL(ret);
      }
      if (ret != false_object)
	{
	  clause = CDR (clause);
	  
	  while (! NULLP (clause))
	    {
	      ret = eval (CAR (clause));
	      clause = CDR (clause);
	    }
	  return (ret);
	}
      clauses = CDR (clauses);
    }
  return (false_object);
}

static Object 
define_eval (Object form)
{
    Object sym, val;

    if (NULLP (CDR (form)) || NULLP (CDR (CDR (form)))) {
	error ("DEFINE form requires at least two args: (define {<var>} <init>)", form, NULL);
    } else {
	bind_variables( CDR (form), 1);
    }
    return unspecified_object;
}


void bind_variables(Object init_list, int top_level)
{
    Object variable, variables, init, val;
    Object first, last, new;
    int i, value_count;

    if (!PAIRP (init_list) || NULLP (CDR (init_list))) {
	error ("Initializer list requires at least two elements",
	       init_list, NULL);
    }
    variables = init = init_list;
    while (! NULLP (CDR (init))) {
	init = CDR (init);
    }
    val = eval (CAR (init));
    if (VALUESP (val)) {
	value_count = 0;
	while (variables != init) {
	    variable = CAR (variables);
	    if (variable == rest_symbol) {
		variable = SECOND (variables);
		last = NULL;
		first = make_empty_list();
		/* bind rest values */
		for ( i=value_count ; i < VALUESNUM (val) ; ++i ) {
		    new = cons (VALUESELS(val)[i], make_empty_list());
		    if ( last ) {
			CDR (last) = new;
		    } else {
			first = new;
		    }
		    last = new;
		}
		if (top_level) {
		    add_top_level_binding (variable, first);
		} else {
		    add_binding (variable, first);
		}
		/* check for no variables after #rest */
		if (CDR (CDR (variables)) != init) {
		    error("Badly placed #rest specifier", init_list, NULL);
		}
		/* finished with bindings */
		break;
	    } else {
		/* check for not enough inits */
		if (value_count < VALUESNUM (val)) {
		    new = VALUESELS(val)[value_count];
		} else {
		    new = false_object;
		}
		add_variable_binding (variable, new, top_level);
		value_count++;
	    }
	    variables = CDR (variables);
	}
    } else {
	add_variable_binding( CAR (variables), val, top_level);
	for ( variables = CDR (variables);
	      variables != init;
	      variables = CDR (variables)){
	    add_variable_binding( CAR (variables), false_object, top_level);
	}
    }
}

void
add_variable_binding( Object var, Object val, int top_level)
{
    Object type;
    
    if (PAIRP (var)) {
	if (!PAIRP (CDR (var))) {
	    error ("badly formed variable", var, NULL);
	}
	type = eval (SECOND (var));
	if (! instance (type, type_class)) {
	    error ("badly formed variable", var, NULL);
	}
    } else {
	type = object_class;
    }
    if ( ! instance (val, type)) {
	error ("init value does not satisfy type constraint", val, type, NULL);
    }
    if (top_level) {
	add_top_level_binding (var, val);
    } else {
	add_binding(var, val);
    }
}

static Object 
define_class_eval (Object form)
{
  Object name, supers, slots, class, obj;

  if (NULLP (CDR (form)))
    {
      error ("malfored define-class", form, NULL);
    }
  name = SECOND (form);
  if (NULLP (CDR (CDR (form))))
    {
      error ("malformed define-class", form, NULL);
    }
  /*
   * Must introduce binding for the class before eval'ing the slot definitions.
   */
  obj = allocate_object (sizeof (struct class));
  CLASSTYPE (obj) = Class;
  CLASSNAME (obj) = name;
  add_top_level_binding (name, obj);
  supers = map (eval, THIRD (form));
  slots = slot_descriptor_list (CDR (CDR (CDR (form))), 1);
  class = make_class (obj, supers, slots, NULL);
  return (unspecified_object);
}

static Object 
define_generic_function_eval (Object form)
{
  Object name, params, gf;
  
  if (NULLP (CDR (form)))
    {
      error ("define-generic-function: missing name", form, NULL);
    }
  name = SECOND (form);
  if (NULLP (CDR (CDR (form))))
    {
      error ("define-generic-function: missing parameters", form, NULL);
    }
  params = THIRD (form);

  gf = make_generic_function (name, params, make_empty_list());
  add_top_level_binding (name, gf);
  return (unspecified_object);
}

static Object 
define_method_eval (Object form)
{
  Object name, params, body, method, gf;

  if (NULLP (CDR (form)))
    {
      error ("define-method: missing name", form, NULL);
    }
  name = SECOND (form);
  if (! SYMBOLP (name))
    {
      error ("define-method: first argument must be a symbol", name, NULL);
    }
  if (NULLP (CDR (CDR (form))))
    {
      error ("define-method: missing parameter list", form, NULL);
    }
  params = THIRD (form);
  if (! LISTP (params))
    {
      error ("define-method: second argument must be a parameter list", params, NULL);
    }
  body = CDR (CDR (CDR (form)));
  method = make_method (name, params, body, the_env, 1);
  return (unspecified_object);
}

static Object 
dotimes_eval (Object form)
{
  Object clause, var, intval, resform, body, res;
  int i;

  if (NULLP (CDR (form)))
    {
      error ("malformed dotimes expression", form, NULL);
    }
  clause = CAR (CDR (form));
  if (! PAIRP (clause))
    {
      error ("second arg to dotimes must be a list", clause, NULL);
    }
  var = CAR (clause);
  if (! SYMBOLP (var))
    {
      error ("dotimes: first value in spec clause must be a symbol", var, NULL);
    }
  if (NULLP (CDR (clause)))
    {
      error ("dotimes: must specifiy an upper bound", form, NULL);
    }
  intval = eval (CAR (CDR (clause)));
  if (! INTEGERP (intval))
    {
      error ("dotimes: upper bound must an integer", intval, NULL);
    }
  if (! NULLP (CDR (CDR (clause))))
    {
      resform = CAR (CDR (CDR (clause)));
    }
  else
    {
      resform = NULL;
    }

  push_scope ();
  add_binding (var, false_object);
  for ( i=0 ; i < INTVAL (intval) ; ++i )
    {
      change_binding (var, make_integer (i));
      body = CDR (CDR (form));
      while (! NULLP (body))
	{
	  res = eval (CAR (body));
	  body = CDR (body);
	}
    }
  if (resform)
    {
      res = eval (resform);
    }
  else
    {
      res = false_object;
    }
  pop_scope ();
  return (res);
}

/*

  (for ((var-form init-form step-form)
        ...)
       (test-form return-form)
    form1 form2 ...)

*/
static Object 
for_eval (Object form)
{
  Object var_forms, var_form, test_form, return_forms;
  Object var, vars, inits, body, ret, new_vals;

  if (NULLP (CDR (form)))
    {
      error ("malformed FOR", form, NULL);
    }
  if (NULLP (CDR (CDR (form))))
    {
      error ("malformed FOR", form, NULL);
    }
  test_form = FIRST (THIRD (form));
  return_forms = CDR (THIRD (form));

  var_forms = SECOND (form);
  vars = map (car, var_forms);
  inits = map (second, var_forms);
  inits = map (eval, inits);

  push_scope ();
  add_bindings (vars, inits);

  /* now loop until exit */
  while (eval (test_form) == false_object)
    {
      body = CDR (CDR (CDR (form)));
      while (! NULLP (body))
	{
	  eval (CAR (body));
	  body = CDR (body);
	}

      var_forms = SECOND (form);
      vars = map (car, var_forms);
      new_vals = map (third, var_forms);
      new_vals = map (eval, new_vals);

      while (! NULLP (vars))
	{
	  modify_value (CAR (vars), CAR (new_vals));
	  vars = CDR (vars);
	  new_vals = CDR (new_vals);
	}
    }
  if (NULLP (return_forms))
    {
      ret = false_object;
    }
  else
    {
      while (! NULLP (return_forms))
	{
	  ret = eval (CAR (return_forms));
	  return_forms = CDR (return_forms);
	}
    }
  pop_scope ();
  return (ret);
}


/*
   The iteration is terminated if any collection is exhausted 
   (in which case #f is returned) or if the end-test evaluates 
   to #t (in which case the result forms are evaluated and the
   value of the last is returned).
*/
static Object 
for_each_eval (Object form)
{
  Object test_form, return_forms, var_forms;
  Object vars, collections, states, vals, body, ret, temp_vars;
  Object init_state_fun, next_state_fun, cur_el_fun;

  init_state_fun = symbol_value (initial_state_sym );
  if (! init_state_fun)
    {
      error ("for-each: no initial-state function defined", NULL);
    }
  next_state_fun = symbol_value (next_state_sym);
  if (! next_state_fun)
    {
      error ("for-each: no next-state function defined", NULL);
    }
  cur_el_fun = symbol_value (current_element_sym);
  if (! cur_el_fun)
    {
      error ("for-each: no current-element function defined", NULL);
    }

  if (NULLP (CDR (form)))
    {
      error ("malformed FOR-EACH", form, NULL);
    }
  if (NULLP (CDR (CDR (form))))
    {
      error ("malformed FOR-EACH", form, NULL);
    }
  test_form = FIRST (THIRD (form));
  return_forms = CDR (THIRD (form));

  var_forms = SECOND (form);
  vars = map (car, var_forms);
  collections = map (second, var_forms);
  collections = map (eval, collections);
  states = list_map1 (init_state_fun, collections);

  if (member (false_object, states))
    {
      return (false_object);
    }
  vals = list_map2 (cur_el_fun, collections, states);
  push_scope ();
  add_bindings (vars, vals);

  while (eval (test_form) == false_object)
    {
      body = CDR (CDR (CDR (form)));
      while (! NULLP (body))
	{
	  eval (CAR (body));
	  body = CDR (body);
	}
      states = list_map2 (next_state_fun, collections, states);
      if (member (false_object, states))
	{
	  pop_scope ();
	  return (false_object);
	}
      vals = list_map2 (cur_el_fun, collections, states);

      /* modify bindings */
      temp_vars = vars;
      while (! NULLP (temp_vars))
	{
	  modify_value (CAR (temp_vars), CAR (vals));
	  temp_vars = CDR (temp_vars);
	  vals = CDR (vals);
	}
    }

  if (NULLP (return_forms))
    {
      return (false_object);
    }
  else
    {
      while (! NULLP (return_forms))
	{
	  ret = eval (CAR (return_forms));
	  return_forms = CDR (return_forms);
	}
    }
  pop_scope ();
  return (ret);
}

static Object 
if_eval (Object form)
{
  Object testval, thenform, elseform;

  if (NULLP (CDR (form)))
    {
      error ("malformed if expression", form, NULL);
    }
  testval = SECOND (form);
  if (NULLP (CDR (CDR (form))))
    {
      error ("malformed if expression", form, NULL);
    }
  thenform = THIRD (form);
  if (NULLP (CDR (CDR (CDR (form)))))
    {
      error ("if expression must have else clause", form, NULL);
    }
  elseform = FOURTH (form);
  if (! NULLP (CDR (CDR (CDR (CDR (form))))))
    {
      error ("if: too many arguments", NULL);
    }
  
  if (eval(testval) == false_object)
    {
      return (eval (elseform));
    }
  else
    {
      return (eval (thenform));
    }
}

static Object 
method_eval (Object form)
{
  Object params, body, method;

  if (NULLP (CDR (form)))
    {
      error ("method: missing parameters", form, NULL);
    }
  params = SECOND (form);
  body = CDR (CDR (form));
  method = make_method (NULL, params, body, the_env, 0);
  return (method);
}

static Object
or_eval (Object form)
{
  Object clauses, ret;

  clauses = CDR (form);
  while (! NULLP (clauses)) {
      ret = eval (CAR (clauses));
      if (VALUESP (ret)) {
	  if (PAIRP (CDR (clauses))) {
	      ret = FIRSTVAL (ret);
	  } else {
	      return (ret);
	  }
      }
      if (ret != false_object) {
	  return (ret);
      }
      clauses = CDR (clauses);
  }
  return (false_object);
}

static Object qq_help (Object skel);

static Object 
quasiquote_eval (Object form)
{
  return qq_help (SECOND (form));
}

static Object
qq_help(Object skel)
{
    Object head, tmp, tail;

    if (NULLP (skel) || SYMBOLP (skel) || !PAIRP (skel)) {
	return skel;
    } else {
	head = skel;
	tail = CDR (skel);
	if (CAR (head) == unquote_symbol) {
	    if (!NULLP (tail)) {
		if (!NULLP (CDR (tail))) {
		    error("Too many arguments to unquote", NULL);
		}
		return eval (CAR (tail));
	    } else {
		error ("missing argument to unquote", NULL);
	    }
	} else if (PAIRP (CAR (head))
		   && CAR (CAR (head)) == unquote_splicing_symbol) {

	    if (!NULLP (CDR (CAR (head)))) {
		tmp = eval (CAR (CDR (CAR (head))));
		CAR (head) = CAR (tmp);
		CDR (head) = CDR (tmp);
		tmp = head;
		while (!NULLP (CDR (tmp))){
		    tmp = CDR (tmp);
		}
		CDR (tmp) = qq_help (tail);
		return head;
	    } else {
		error ("missing argument to unquote_splicing", NULL);
	    }
	} else {
	    return cons (qq_help (CAR (head)), qq_help ( tail));
	}
    }
}

static Object 
quote_eval (Object form)
{
  return (SECOND (form));
}

static Object 
select_eval (Object form)
{
  Object target_form, test, branches, branch;
  Object match_list, consequents, ret;

  if (NULLP (CDR (form)))
    {
      error ("malformed select", form, NULL);
    }
  target_form = eval (CAR (CDR (form)));

  if (NULLP (CDR (CDR (form))))
    {
      error ("malformed select", form, NULL);
    }
  test = eval (CAR (CDR (CDR (form))));

  if (NULLP (CDR (CDR (CDR (form)))))
    {
      error ("malformed select", form, NULL);
    }
  branches = CDR (CDR (CDR (form)));
  while (! NULLP (branches))
    {
      branch = CAR (branches);
      if (! PAIRP (branch))
	{
	  error ("select: malformed branch", branch, NULL);
	}
      match_list = CAR (branch);
      if ((match_list == true_object) || (match_list == else_keyword))
	{
	  consequents = CDR (branch);
	  while (! NULLP (consequents))
	    {
	      ret = eval (CAR (consequents));
	      consequents = CDR (consequents);
	    }
	  return (ret);
	}
      if (! PAIRP (match_list))
	{
	  error ("select: malformed test expression", match_list, NULL);
	}
      while (! NULLP (match_list))
	{
	  ret = false_object;
	  if (apply (test, listem (target_form, eval (CAR (match_list)),
				   NULL)) != false_object)
	    {
	      consequents = CDR (branch);
	      while (! NULLP (consequents))
		{
		  ret = eval (CAR (consequents));
		  consequents = CDR (consequents);
		}
	      return (ret);
	    }
	  match_list = CDR (match_list);
	}
      branches = CDR (branches);
    }
  return (false_object);
}

static Object 
set_eval (Object form)
{
  Object sym, val, setter_sym;

  if (NULLP (CDR (form)))
    {
      error ("set!: missing forms", form, NULL);
    }
  sym = SECOND (form);
  
  if (PAIRP(sym))
    {
      setter_sym = make_setter_symbol (CAR (sym));
      eval (cons (setter_sym, 
		  append (CDR (sym), CDR (CDR (form)))));
      return (unspecified_object);
      
    }
  if (NULLP (CDR (CDR (form))))
    {
      error ("set!: missing forms", form, NULL);
    }
  val = eval (THIRD (form));
  modify_value (sym, val);
  return (val);
}

static Object 
unless_eval (Object form)
{
  Object test, body, ret;

  if (NULLP (CDR (form)))
    {
      error ("unless: missing forms", form, NULL);
    }
  test = SECOND (form);
  body = CDR (CDR (form));
  if (eval (test) == false_object) {
      ret = false_object;
      while (! NULLP (body)) {
	  ret = eval (CAR (body));
	  body = CDR (body);
      }
      return (ret);
  }
  return (false_object);
}

static Object 
until_eval (Object form)
{
  Object test, body, forms;

  if (NULLP (CDR (form)))
    {
      error ("malformed until statment", form, NULL);
    }
  test = CAR (CDR (form));
  body = CDR (CDR (form));

  while (eval (test) == false_object)
    {
      forms = body;
      while (! NULLP (forms))
	{
	  eval (CAR (forms));
	  forms = CDR (forms);
	}
    }
  return (false_object);
}

static Object 
unwind_protect_eval (Object form)
{
  Object protected, cleanups, unwind, ret;

  if (NULLP (CDR (form)))
    {
      error ("unwind-protect: missing forms", form, NULL);
    }
  protected = SECOND (form);
  cleanups = CDR (CDR (form));
  unwind = make_unwind (cleanups);

  push_scope ();
  add_binding (unwind_symbol, unwind);
  ret = eval (protected);
  pop_scope ();
  return (ret);
}

static Object 
when_eval (Object form)
{
  Object test, body, ret;

  if (NULLP (CDR (form)))
    {
      error ("when: missing forms", form, NULL);
    }
  test = SECOND (form);
  body = CDR (CDR (form));
  if (eval (test) != false_object) {
      ret = false_object;
      while (! NULLP (body)) {
	  ret = eval (CAR (body));
	  body = CDR (body);
      }
      return (ret);
  }
  return (false_object);
}

static Object 
while_eval (Object form)
{
  Object test, body, forms;

  if (NULLP (CDR (form)))
    {
      error ("malformed while statment", form, NULL);
    }
  test = CAR (CDR (form));
  body = CDR (CDR (form));

  while (eval (test) != false_object)
    {
      forms = body;
      while (! NULLP (forms))
	{
	  eval (CAR (forms));
	  forms = CDR (forms);
      }
  }
  return (false_object);
}

static Object ___passed_test_list;
static Object ___failed_test_list;
static Object ___disabled_test_list;
static Object ___failure_format_string;
static Object ___success_format_string;
static Object ___disabled_format_string;
static Object ___fail_symbol;
static Object ___pass_symbol;
static Object ___disabled_symbol;
static Object ___disabled_keyword;
static Object ___no_handler_keyword;
static Object ___signal_keyword;

static Object
define_test_eval(Object form)
{
    Object test_name, test_options, doc_string, test_form;
    Object exit_obj, ret;
    
    if (___passed_test_list == NULL) {
	___passed_test_list = make_symbol ("*passed-test-list*");
	add_top_level_binding (___passed_test_list, make_empty_list());
	
	___failed_test_list = make_symbol ("*failed-test-list*");
	add_top_level_binding (___failed_test_list, make_empty_list());

    	___disabled_test_list = make_symbol ("*disabled-test-list*");
	add_top_level_binding (___disabled_test_list, make_empty_list());

	___failure_format_string =
	    make_byte_string ("~%Failed:  ~A ~A with result ~A");
	___success_format_string =
	    make_byte_string ("~%Passed:  ~A ~A with result ~A");
	___disabled_format_string =
	    make_byte_string ("~%Disabled:  ~A ~A");
	___disabled_symbol = make_symbol ("disabled");
	___fail_symbol = make_symbol ("fail");
	___pass_symbol = make_symbol ("pass");

	___disabled_keyword = make_keyword ("disabled:");
	___no_handler_keyword = make_keyword ("no-handler:");
	___signal_keyword = make_keyword ("signal:");
}
    
    if (list_length (form) != 5) {
	error ("define-test: bad argument list", form);
    }
    form = CDR (form);
    test_name = CAR (form);
    form = CDR (form);
    test_options = CAR (form);
    form = CDR (form);
    doc_string = CAR (form);
    form = CDR (form);
    test_form = CAR (form);
    
    if (!SYMBOLP (test_name)) {
	error("define-test: first argument must be the test name",
	      test_name, NULL);
    }
    if (!LISTP (test_options)) {
	error("define-test: second argument must be a list of options",
	      test_options, NULL);
    }
    if (!BYTESTRP (doc_string)) {
	error("define-test: third argument must be documenting string",
	      doc_string, NULL);
    }
    
    if (member (___disabled_keyword, test_options)) {
	return record_disabled (test_name, doc_string);
    } else if ( member (___no_handler_keyword, test_options)) {
	return process_test_result ( test_name, map (eval, test_options),
				     doc_string, eval (test_form));
    } else {
	exit_obj = make_exit (signal_symbol);
	ret = (Object) setjmp (*EXITRET(exit_obj));
	push_scope ();
	add_binding (signal_symbol, exit_obj);
	if (! ret ) {
	    ret = eval (test_form);
	    pop_scope ();
	} else {
	    pop_scope ();
	}
	return process_test_result ( test_name, map (eval, test_options),
				     doc_string, ret);
    }
}

static Object
process_test_result (Object name, Object options, Object doc_string,
		     Object result)
{
    if (result == true_object &&
	       (EMPTYLISTP (options) ||
		member (___no_handler_keyword, options))) {
	return record_success (name, doc_string, result);
    } else if (list_length(options) > 0
	       && CAR (options) == ___signal_keyword
	       && instance (result, SECOND (options))) {
	return record_success (name, doc_string, result);
    } else {
	return record_failure (name, doc_string, result);
    }
}


static Object
record_failure (Object name, Object doc_string, Object result)
{
    format (true_object, ___failure_format_string,
	    listem (name, doc_string, result, NULL));
    modify_value (___failed_test_list,
		  cons (name, symbol_value (___failed_test_list)));
    return ___fail_symbol;
}

static Object
record_success (Object name, Object doc_string, Object test_result)
{
    format (true_object, ___success_format_string,
	    listem (name, doc_string, test_result, NULL));
    modify_value (___passed_test_list,
		  cons (name, symbol_value (___passed_test_list)));
    return ___pass_symbol;
}

static Object
record_disabled (Object name, Object doc_string)
{
    format (true_object, ___disabled_format_string,
	    listem (name, doc_string, NULL));
    modify_value (___disabled_test_list,
		  cons (name, symbol_value (___disabled_test_list)));
    return ___disabled_symbol;
}
