/*
   
   symbol.c

   Copyright, 1993, Brent Benson.  All Rights Reserved.
   0.4 Revisions Copyright 1994, Joseph N. Wilson.  All Rights Reserved.
   
   Permission to use, copy, and modify this software and its
   documentation is hereby granted only under the following terms and
   conditions.  Both the above copyright notice and this permission
   notice must appear in all copies of the software, derivative works
   or modified version, and both notices must appear in supporting
   documentation.  Users of this software agree to the terms and
   conditions set forth in this notice.

*/

#include <string.h>
#include <ctype.h>
#include "symbol.h"
#include "alloc.h"
#include "string.h"

/* local function prototypes 
*/
static Object intern_symbol (char *name);
#ifdef NO_STRCASECMP 
static int strcasecmp (unsigned char *s1, unsigned char *s2);
#endif
static char *tolowerstr (char *str);
/* local data 
*/

/* If SYMTAB_SIZE is not a power of 2, see change required below. */
#define SYMTAB_SIZE 1024
struct symtab *symbol_table[SYMTAB_SIZE];
unsigned char chartable[1 << sizeof(char)*8];

/* function definitions 
*/
void
init_symbol_prims(void)
{
    int c;
    
    for(c=0; c < (1 << sizeof(char)*8); c++){
	if (c >= 'A' && c <= 'Z') {
	    chartable[c] = c + 'a' - 'A';
	} else {
	    chartable[c] = c;
	}
    }
}

Object 
make_symbol (char *name)
{
  Object obj;

  obj = intern_symbol (name);
  return (obj);
}

Object 
make_keyword (char *name)
{
  Object obj;

  obj = intern_symbol (name);
  SYMBOLTYPE (obj) = Keyword;
  return (obj);
}


Object 
make_setter_symbol (Object sym)
{
  char *name;

  name = allocate_string (sizeof(char)*(1+strlen(SYMBOLNAME(sym)) +
					strlen("-stter")));
  strcpy (name, SYMBOLNAME(sym));
  strcat (name, "-setter");
  return (make_symbol (name));
}

#ifdef NO_STRCASECMP
static int
strcasecmp (unsigned char *s1, unsigned char *s2)
{
    while((chartable[*s1] == chartable[*s2++])) {
	if (!chartable[*++s1]) return 0;
    }
    return (chartable[*s1] - chartable[*--s2]);
}
#endif

static char*
tolowerstr(char *str)
{
    char *s = str;
    while (*s) { *s = tolower(*s); ++s; }
    return str;
}

static Object 
intern_symbol (char *name)
{
    int i;
    unsigned h;
    struct symtab *entry;
    Object sym;
    
    h = i = 0;
    while (name[i]) { h += tolower(name[i++]); }
/*
    h = h % SYMTAB_SIZE;
 */

    /* Works only if SYMTAB_SIZE is a power of 2 */
    h &= (SYMTAB_SIZE - 1);
    
    entry = symbol_table[h];
    while ( entry ) {
#ifdef NO_STRCASECMP
	if (strcasecmp ((unsigned char *)name,
			(unsigned char *)SYMBOLNAME(entry->sym)) == 0) {
#else
	if (strcasecmp (name, SYMBOLNAME(entry->sym)) == 0) {
#endif
	    return (entry->sym);
	}
	entry = entry->next;
    }
    
    /* not found, create new entry for it. */
    sym = allocate_object (sizeof (struct symbol));
    SYMBOLTYPE (sym) = Symbol;
    SYMBOLNAME (sym) = checking_strdup (name);
    entry = (struct symtab *) allocate_symtab ();
    entry->sym = sym;
    entry->next = symbol_table[h];
    symbol_table[h] = entry;
    return (sym);
}

