/*

   print.c

   Copyright, 1993, Brent Benson.  All Rights Reserved.
   0.4 Revisions Copyright 1994, Joseph N. Wilson.  All Rights Reserved.
   
   Permission to use, copy, and modify this software and its
   documentation is hereby granted only under the following terms and
   conditions.  Both the above copyright notice and this permission
   notice must appear in all copies of the software, derivative works
   or modified version, and both notices must appear in supporting
   documentation.  Users of this software agree to the terms and
   conditions set forth in this notice.

*/

#include <stdio.h>
#include "print.h"
#include "list.h"
#include "eval.h"
#include "error.h"
#include "slot.h"

/* local function prototypes */

static Object print_obj_escaped (Object obj);
static void print_pair (FILE *fp, Object pair, int escaped);
static void print_character (FILE *fp, Object c, int escaped);
static void print_vector (FILE *fp, Object vec, int escaped);
static void print_values (FILE *fp, Object vals, int escaped);
static void print_string (FILE *fp, Object str, int escaped);
static void print_instance (FILE *fp, Object inst, int escaped);
static void print_generic_function (FILE *fp, Object gf, int escaped);
static void print_method (FILE *fp, Object method, int escaped);
static void print_slot_descriptor(FILE *fp, Object slotd, int escaped);
static void print_class (FILE *fp, Object class, int escaped);
static void print_array (FILE *fp, Object array, int escaped);
static void print_stream (FILE *fp, Object stream);
static Object write_char (Object ch, Object stream_list);
static void print_type_name(FILE *fp, Object class, int escaped);

/* primitives */

static struct primitive print_prims[] =
{
  {"%print", prim_1, print_obj},
  {"%princ", prim_1, print_obj_escaped},
  {"%format", prim_3, format},
  {"%write-char", prim_2, write_char},
};

/* function definitions */

void
init_print_prims (void)
{
  int num;

  num = sizeof (print_prims) / sizeof (struct primitive);
  init_prims (num, print_prims);
}

void 
print_object (FILE *fp, Object obj, int escaped)
{
  switch (TYPE (obj))
    {
    case True:
      fprintf (fp, "#t");
      break;
    case False:
      fprintf (fp, "#f");
      break;
    case EmptyList:
      fprintf (fp, "()");
      break;
    case Integer:
      fprintf (fp, "%d", INTVAL(obj));
      break;
    case DoubleFloat:
      fprintf (fp, "%f", DFLOATVAL(obj));
      break;
    case Symbol:
      fprintf (fp, "%s", SYMBOLNAME(obj));
      break;
    case Keyword:
      fprintf (fp, "%s", KEYNAME(obj));
      break;
    case Pair:
      print_pair (fp, obj, escaped);
      break;
    case Character:
      print_character (fp, obj, escaped);
      break;
    case SimpleObjectVector:
      print_vector (fp, obj, escaped);
      break;
    case ByteString:
      print_string (fp, obj, escaped);
      break;
    case Table:
      fprintf (fp, "{table}");
      break;
    case Deque:
      fprintf (fp, "{deque}");
      break;
    case Array:
      print_array (fp, obj, escaped);
      break;
    case Primitive:
      fprintf (fp, "{primitive function %s}", PRIMNAME(obj));
      break;
    case GenericFunction:
      print_generic_function (fp, obj, escaped);
      break;
    case Method:
      print_method (fp, obj, escaped);
      break;
    case Class:
      print_class (fp, obj, escaped);
      break;
    case Instance:
      print_instance (fp, obj, escaped);
      break;
    case Singleton:
      fprintf (fp, "{the singleton ");
      print_object (fp, SINGLEVAL (obj), escaped);
      fprintf (fp, "}");
      break;
    case LimitedIntType:
      fprintf (fp, "{limited <integer>");
      if (LIMINTHASMIN (obj)) {
	  fprintf (fp, " min: %d", LIMINTMIN (obj));
      }
      if (LIMINTHASMAX (obj)) {
	  fprintf (fp, " max: %d", LIMINTMAX (obj));
      }
      fprintf(fp, "}");
      break;
    case UnionType:
      fprintf (fp, "{union");
      {
	  Object ptr;
	  for (ptr = UNIONLIST (obj); PAIRP (ptr); ptr = CDR (ptr)) {
	      fprintf (fp, " ");
	      print_object (fp, CAR (ptr), escaped);
	  }
	  fprintf (fp, "}");
      }
      break;
    case SlotDescriptor:
      print_slot_descriptor(fp, obj, escaped);
      break;
    case EndOfFile:
      fprintf (fp, "{end of file}");
      break;
    case Values:
      print_values (fp, obj, escaped);
      break;
    case Unspecified:
      break;
    case Exit:
      fprintf (fp, "{exit procedure}");
      break;
    case Unwind:
      fprintf (fp, "{unwind protect}");
      break;
    case Stream:
      print_stream (fp, obj);
      break;
    case TableEntry:
      fprintf (fp, "{table entry}");
      break;
    case UninitializedSlotValue:
      fprintf (fp, "{uninitialized slot value}");
      break;
    case DequeEntry:
      fprintf (fp, "{deque entry ", DEVALUE (obj));
      print_object (fp, DEVALUE (obj), escaped);
      fprintf (fp, "}");
      break;
    default:
      error ("print: unknown object type", NULL);
    }
}

Object
print_obj (Object obj)
{
  print_object (stdout, obj, 1);
  if (obj != unspecified_object)
    {
      printf ("\n");
    }
  return (unspecified_object);
}

static Object
print_obj_escaped (Object obj)
{
  print_object (stdout, obj, 0);
  if (obj != unspecified_object)
    {
      printf ("\n");
    }
  return (unspecified_object);
}

void
print_err (Object obj)
{
  print_object (stderr, obj, 1);
  fflush (stderr);
}

static void
print_pair (FILE *fp, Object pair, int escaped)
{
  Object cdr;

  fprintf (fp, "(");
  print_object (fp, CAR (pair), escaped);
  cdr = CDR (pair);
  while (PAIRP(cdr))
    {
      fprintf (fp, " ");
      print_object (fp, CAR (cdr), escaped);
      cdr = CDR (cdr);
    }
  if (!EMPTYLISTP (cdr))
    {
      fprintf (fp, " . ");
      print_object (fp, cdr, escaped);
    }
  fprintf (fp, ")");
}

static void 
print_character (FILE *fp, Object c, int escaped)
{
  char ch;

  ch = CHARVAL(c);
  if ( escaped )
    {
      switch (ch)
	{
	case '\n':
	  fprintf (fp, "#\\newline");
	  break;
	case ' ':
	  fprintf (fp, "#\\space");
	  break;
	case 0x7f:
	  fprintf (fp, "#\\rubout");
	  break;
	case '\f':
	  fprintf (fp, "#\\page");
	  break;
	case '\t':
	  fprintf (fp, "#\\tab");
	  break;
	case '\b':
	  fprintf (fp, "#\\backspace");
	  break;
	case '\r':
	  fprintf (fp, "#\\return");
	  break;
	default:
	  fprintf (fp, "#\\%c", ch);
	}
    }
  else
    {
      fprintf (fp, "%c", ch);
    }
}

static void 
print_vector (FILE *fp, Object vec, int escaped)
{
  int i;

  fprintf (fp, "#(");
  for ( i=0 ; i < SOVSIZE(vec) ; ++i )
    {
      print_object (fp, SOVELS(vec)[i], escaped);
      if (i < (SOVSIZE(vec) - 1))
	{
	  fprintf (fp, " ");
	}
    }
  fprintf (fp, ")");
}

static void
print_slot_values (FILE *fp, Object instance, Object slotds, int escaped)
{
    Object slot_value, slotd;
    int i;
    
    if (EMPTYLISTP (slotds)) return;
    
    for  (i = 0;
	  PAIRP (slotds);
	  i++, slotds = CDR (slotds)) {
	fprintf (fp, ", ");
	print_object (fp, SLOTDGETTER (CAR (slotds)), escaped);
	fprintf (fp, " = ");
	print_object (fp, CAR (INSTSLOTS (instance)[i]), escaped);
    }
}

static void
print_constant_slot_values (FILE *fp, Object const_slotds, int escaped)
{
    Object slotd;
    int i;
    
    if (EMPTYLISTP (const_slotds)) return;
    
    for  (i = 0;
	  PAIRP (const_slotds);
	  i++, const_slotds = CDR (const_slotds)) {
	slotd = CAR (const_slotds);
	fprintf (fp, ", ");
	print_object (fp, SLOTDGETTER (slotd), escaped);
	fprintf (fp, " = ");
	print_object (fp, SLOTDINIT (slotd), escaped);
    }
}

static void
print_virtual_slot_values (FILE *fp, Object instance, Object slotds,
			   int escaped)
{
    Object slotd;

    if (EMPTYLISTP (slotds)) return;
    
    for (slotd = CAR (slotds);
	 !EMPTYLISTP (slotds);
	 slotds = CDR (slotds)) {
	fprintf (fp, ", ");
	print_object (fp, SLOTDGETTER (slotd), escaped);
	fprintf (fp, " = ");
	print_object (fp, eval (listem (SLOTDGETTER (slotd), instance, NULL)),
		      escaped);
    }
}

static void
print_class_slot_values (FILE *fp, Object class, int escaped, int first)
{
    Object slotds, slots, supers;
    int i;


    print_slot_values (fp, CLASSCSLOTS (class),
		       (first ? append (CLASSCSLOTDS (class),
					CLASSESSLOTDS (class))
		              : CLASSCSLOTDS (class)),
		       escaped);
    
    for (supers = CLASSSUPERS (class);
	 PAIRP (supers);
	 supers = CDR (supers)) {
	print_class_slot_values (fp, CAR (supers), escaped, 0);
    }
}

static void 
print_instance (FILE *fp, Object inst, int escaped)
{
  Object slots, slot, class, instslotds;

  fprintf (fp, "{instance of class %s",
	   SYMBOLNAME(CLASSNAME(INSTCLASS(inst))));
  class = INSTCLASS (inst);

  instslotds = append (CLASSINSLOTDS (class), CLASSSLOTDS (class));
  print_slot_values (fp, inst, instslotds, escaped);
  print_virtual_slot_values (fp, inst, CLASSVSLOTDS (class), escaped);
  print_class_slot_values (fp, class, escaped, 1);
  print_constant_slot_values (fp, CLASSCONSTSLOTDS (class), escaped);
  fprintf (fp, "}");
}

static void 
print_values (FILE *fp, Object vals, int escaped)
{
  int i, num;

  num = VALUESNUM (vals);
  for ( i=0 ; i < num ; ++i )
    {
      print_object (fp, VALUESELS(vals)[i], escaped);
      if (i < (num - 1))
	{
	  fprintf (fp, "\n");
	}
    }
}

static void
print_param (FILE *fp, Object param, int escaped)
{
    if (SECOND (param) != object_class) {
	fprintf(fp, "(");
	print_object (fp, CAR (param), escaped);
	fprintf(fp, " ");
	print_type_name (fp, SECOND (param), escaped);
	fprintf(fp, ")");
    } else {
	print_object (fp, CAR (param), escaped);
    }
    
}
static void
print_param_list (FILE *fp, Object params, int escaped)
{
    if (PAIRP (params)){
	print_param (fp, CAR (params), escaped);
	params = CDR (params);
	while (PAIRP(params)) {
	    fprintf (fp, " ");
	    print_param (fp, CAR (params), escaped);
	    params = CDR (params);
	}
    }
}

static void
print_unparenthesized_list(FILE *fp, Object pair, int escaped)
{
    if (PAIRP (pair)){
	print_object (fp, CAR (pair), escaped);
	pair = CDR (pair);
	while (PAIRP(pair)) {
	    fprintf (fp, " ");
	    print_object (fp, CAR (pair), escaped);
	    pair = CDR (pair);
	}
    }
}

static void 
print_generic_function (FILE *fp, Object gf, int escaped)
{
    int some_args_printed = 0;

    fprintf (fp, "{the generic function %s (", SYMBOLNAME(GFNAME(gf)));

    if (PAIRP (GFREQPARAMS (gf))) {
	print_param_list (fp, GFREQPARAMS (gf), escaped);
	some_args_printed = 1;
    }
    if (GFRESTPARAM (gf)) {
	if (some_args_printed) {
	    fprintf(fp, " #rest %s", SYMBOLNAME (GFRESTPARAM (gf)));
	} else {
	    fprintf(fp, "#rest %s", SYMBOLNAME (GFRESTPARAM (gf)));
	}
	some_args_printed = 1;
    }
    if (PAIRP (GFKEYPARAMS (gf))) {
	if (some_args_printed) {
	    fprintf(fp, " #key ");
	} else {
	    fprintf(fp, "#key ");
	}
	print_unparenthesized_list(fp, GFKEYPARAMS (gf), escaped);
	if (GFALLKEYS (gf)) {
	    fprintf(fp, " #all-keys");
	}
    }
    fprintf (fp, ")}");
}

static void 
print_method (FILE *fp, Object method, int escaped)
{
    int some_args_printed = 0;

    if (METHNAME (method)) {
	fprintf (fp, "{method %s (", SYMBOLNAME(METHNAME(method)));
    } else {
	fprintf (fp, "{an anonymous method (");
    }
    if (PAIRP (METHREQPARAMS (method))) {
	print_param_list (fp, METHREQPARAMS (method), escaped);
	some_args_printed = 1;
    }
    if (METHRESTPARAM (method)) {
	if (some_args_printed) {
	    fprintf(fp, " #rest %s", SYMBOLNAME (METHRESTPARAM (method)));
	} else {
	    fprintf(fp, "#rest %s", SYMBOLNAME (METHRESTPARAM (method)));
	}
	some_args_printed = 1;
    }
    if (PAIRP (METHKEYPARAMS (method)) || METHALLKEYS (method)) {
	if (some_args_printed) {
	    fprintf(fp, " #key ");
	} else {
	    fprintf(fp, "#key ");
	}
	print_unparenthesized_list(fp, METHKEYPARAMS (method), escaped);
	if (METHALLKEYS (method)) {
	    fprintf(fp, " #all-keys");
	}
    }
    fprintf (fp, ")");

/*
  print_unparenthesized_list(fp, METHBODY (method), escaped);
  */

    fprintf (fp, "}");
}

static void
print_class (FILE *fp, Object class, int escaped)
{
    Object slots, slot;
    
    if (!SYMBOLNAME(CLASSNAME(class))) {
	fprintf(fp, "{an anonymous class");
    } else {
	fprintf (fp, "{the class %s", SYMBOLNAME(CLASSNAME(class)));
    }

    /*
    print_slot_values (fp, CLASSCSLOTS(class), append (CLASSCSLOTDS (class),
						       CLASSESSLOTDS (class)),
		       escaped);
    */
    
    fprintf (fp, "}");

}

static void
print_slot_descriptor(FILE *fp, Object slotd, int escaped)
{
    fprintf(fp, "{slot descriptor ");
    print_object (fp, SLOTDGETTER (slotd), escaped);
    if (SLOTDALLOCATION (slotd) != instance_symbol) {
	fprintf (fp, " allocation: ");
	print_object (fp, SLOTDALLOCATION (slotd), escaped);
    }
    if (SLOTDSETTER (slotd)) {
	fprintf(fp, " setter: ");
	print_object (fp, SLOTDSETTER (slotd), escaped);
    }
    if (SLOTDSLOTTYPE (slotd) != object_class) {
	fprintf(fp, " type: ");
	print_object (fp, SLOTDSLOTTYPE (slotd), escaped);
    }
    if (SLOTDINIT (slotd) != uninit_slot_object) {
	fprintf(fp, " init: ");
	print_object (fp, SLOTDINIT (slotd), escaped);
    }
    if (SLOTDINITKEYWORD (slotd)) {
	if (SLOTDKEYREQ (slotd)) {
	    fprintf(fp, " required-init-keyword: ");
	} else {
	    fprintf (fp, " init-keyword: ");
	}
	print_object (fp, SLOTDINITKEYWORD (slotd), escaped);
    }
    fprintf (fp, "}");
}

#if 0
static void 
print_array (FILE *fp, Object array, int escaped)
{
  fprintf (fp, "{array ");
  print_object (fp, ARRDIMS(array), escaped);
  fprintf (fp, "}");
}
#endif

static int cur_el;
static void print_array_help (FILE *fp, Object dims, Object *els, int escaped);

static void
print_array (FILE *fp, Object array, int escaped)
{
  Object dims, *els;

  dims = ARRDIMS (array);
  els = ARRELS (array);

  cur_el = 0;
  fprintf (fp, "#%da", list_length (dims));
  print_array_help (fp, dims, els, escaped);
}

static void
print_array_help (FILE *fp, Object dims, Object *els, int escaped)
{
  int dim_val, i;

  fprintf (fp, "(");
  dim_val = INTVAL (CAR (dims));
  if (NULLP (CDR (dims)))
    {
      for ( i=0 ; i < dim_val ; ++i )
	{
	  print_object (fp, els[cur_el++], escaped);
	  if (i < (dim_val - 1))
	    {
	      fprintf (fp, " ");
	    }
	}
    }
  else
    {
      for ( i=0 ; i < dim_val ; ++i )
	{
	  print_array_help (fp, CDR (dims), els, escaped);
	}
    }
  fprintf (fp, ")");
}



#if 0
static void 
print_array (FILE *fp, Object array, int escaped)
{
  Object dims;
  unsigned int dim_val, offset, i;
  int rank;

  dims = ARRDIMS (array);
  rank = list_length (dims);
  fprintf (fp, "#%da", rank);
  offset = 0;
  while (! NULLP (dims))
    {
      fprintf (fp, "(");
      if (NULLP (CDR (dims)))
	{
	  dim_val = INTVAL (CAR (dims));
	  for ( i=0 ; i < dim_val ; ++i )
	    {
	      print_object (fp, ARRELS(array)[offset], escaped);
	      offset++;
	    }
	}
     dims = CDR (dims);
    }

  while (! NULLP (dims))
    {
      fprintf (fp, "(");
      if (NULLP (CDR (dims)))
	{
	  dim_val = INTVAL (CAR (dims));
	  for ( i=0 ; i < dim_val ; ++i )
	    {
	      print_object (fp, ARRELS(array)[offset], escaped);
	      offset++;
	    }
	}
     dims = CDR (dims);
    }
}
#endif

static void 
print_string (FILE *fp, Object str, int escaped)
{
  if ( escaped )
    {
      fprintf (fp, "\"%s\"", BYTESTRVAL(str));
    }
  else
    {
      fprintf (fp, "%s", BYTESTRVAL(str));
    }
}

Object
format (Object stream, Object str, Object rest)
{
  Object obj;
  FILE *fp;
  char *cstr;
  int i;

  if (stream == true_object)
    {
      fp = stdout;
    }
  else if (OUTPUTSTREAMP (stream))
    {
      fp = STREAMFP (stream);
    }
  else
    {
      error ("format: cannot send output to non-stream", stream, NULL);
    }
  cstr = BYTESTRVAL (str);

  i = 0;
  while ( cstr[i] )
    {
      if ( cstr[i] == '~' )
	{
	  i++;
	  switch ( cstr[i] )
	    {
	    case 'a':
	    case 'A':
	      if (NULLP (rest))
		{
		  error ("format: not enough args for format string", str, NULL);
		}
	      obj = CAR (rest);
	      rest = CDR (rest);
	      print_object (fp, obj, 0);
	      break;
	    case 'd':
	    case 'D':
	      if (NULLP (rest))
		{
		  error ("format: not enough args for format string", str, NULL);
		}
	      obj = CAR (rest);
	      if (! INTEGERP (obj))
		{
		  error ("format: argument to ~d must be an integer", obj, NULL); 
		}
	      rest = CDR (rest);
#if 0
	      if (isdigit (cstr[i-1]))
		{
		  j = i - 1;
		  while (isdigit(cstr[j]))
		    {
		      j--;
		    }
		  j++;
		  sscanf (cstr[j], "%d", &arg);
		  fprintf (fp, "%");
		}
	      else
#endif
		{
		  fprintf (fp, "%d", INTVAL(obj));
		}
	      break;
	    case 's':
	    case 'S':
	      if (NULLP (rest))
		{
		  error ("format: not enough args for format string", str, NULL);
		}
	      obj = CAR (rest);
	      rest = CDR (rest);
	      print_object (fp, obj, 1);
	      break;
	    case '%':
	      fprintf (fp, "\n");
	      break;
	    case '~':
	      fprintf (fp, "~");
	      break;
	    default:
	      /* skip over digits.  individuals branches
		 handle there own arguments. */
	      if (isdigit (cstr[i]))
		{
		  while (isdigit(cstr[i]))
		    {
		      i++;
		    }
		  break;
		}
	      error ("format: bad escape character", make_character (cstr[i]), NULL);
	    }
	}
      else
	{
	  fprintf (fp, "%c", cstr[i]);
	}
      i++;
    }
  if (! NULLP (rest))
    {
      error ("format: too many arguments for format string", CAR (rest), NULL);
    }
  return (unspecified_object);
}

static void 
print_stream (FILE *fp, Object stream)
{
  switch (STREAMSTYPE (stream))
    {
    case Input:
      fprintf (fp, "{input stream}");
      break;
    case Output:
      fprintf (fp, "{output stream}");
      break;
    default:
      error ("trying to print stream of unknown type", NULL);
    }
}

static Object
write_char (Object ch, Object stream_list)
{
  char the_char;
  FILE *fp;

  if (NULLP (stream_list))
    {
      fp = stdout;
    }
  else
    {
      fp = STREAMFP (CAR (stream_list));
    }
  the_char = CHARVAL (ch);
  fwrite (&the_char, 1, sizeof (char), fp);
  return (unspecified_object);
}

static void
print_type_name (FILE *fp, Object obj, int escaped)
{
    switch (TYPE(obj)) {
    case Class:
	fprintf( fp, "%s", SYMBOLNAME (CLASSNAME (obj)));
	break;
    case LimitedIntType:
    case UnionType:
	print_object(fp, obj, escaped);
	break;
    default:
	error("print_type_name: object is not a type", obj);
    }
}
