/*

   main.c

   Copyright, 1993, Brent Benson.  All Rights Reserved.
   0.4 Revision Copyright 1994, Joseph N. Wilson.  All Rights Reserved.
   
   Permission to use, copy, and modify this software and its
   documentation is hereby granted only under the following terms and
   conditions.  Both the above copyright notice and this permission
   notice must appear in all copies of the software, derivative works
   or modified version, and both notices must appear in supporting
   documentation.  Users of this software agree to the terms and
   conditions set forth in this notice.

*/

#include "object.h"
#include "alloc.h"
#include "read.h"
#include "eval.h"
#include "print.h"
#include "boolean.h"
#include "symbol.h"
#include "list.h"
#include "class.h"
#include "slot.h"
#include "file.h"
#include "function.h"
#include "values.h"
#include "number.h"
#include "apply.h"
#include "string.h"
#include "keyword.h"
#include "vector.h"
#include "error.h"
#include "syntax.h"
#include "stream.h"
#include "table.h"
#include "deque.h"
#include "array.h"
#include "misc.h"

#include "globals.h"

void initialize_marlais (void);

#ifndef INIT_FILE
#define INIT_FILE "init.dyl"
#endif 

#ifndef VERSION
#define VERSION "0.4"
#endif

#ifdef THINK_C
#define main marlais_main
extern void init_mac_prims(void);
#endif

static int do_not_load_init_file = 0;
static char *optstring = "n";

main(int argc, char *argv[])
{
  Object obj, eobj;
  char *init_file;
  int err, c;
  extern char *optarg;
  extern int optind;

  /* banner */
  printf ("Marlais %s\n", VERSION);

  /* initialization */
  initialize_marlais ();

  /* process command line parameters except source files */
  while ((c = getopt (argc, argv, optstring)) != EOF)
    {
      switch (c)
	{
	case 'n':
	  do_not_load_init_file = 1;
	  break;
	default:
	  fatal ("fatal: unrecognized option");
	}
    } 

  /* error catch for initialization code */
  err = setjmp (error_return);
  if ( err )
    {
      printf ("; error in initialization code -- exiting.\n");
      exit (1);
    }

  /* load initialization code */
  if (! do_not_load_init_file)
    {
      init_file = getenv ("MARLAIS_INIT");
      if (init_file)
	{
	  load (make_byte_string (init_file));
	}
      else
	{
	  load (make_byte_string (INIT_FILE));
	}
    }

  /* load any source files specified on command line */
  while (optind < argc)
    {
      load (make_byte_string (argv[optind]));
      optind++;
    }

  /* errors reset to here */
  err = setjmp (error_return);
  /* things to do on an error reset */ 
  if ( err )
    {
      if (trace_functions)
	{
	  printf ("; reset\n");
	  trace_level = 0;
	}
    }

  /* read-eval-print loop */
  printf ("? ");
  while ((obj = read_object (stdin)) && (obj != eof_object))
    {
      eobj = eval (obj);
      print_obj (eobj);
      printf ("? ");
    }
}

void 
initialize_marlais (void)
{
  /* intialize garbage collector
   */
  initialize_gc ();

  /* initialize symbol table primitives -- MUST BE DONE EARLY!! */
  init_symbol_prims();

  /* intialize global objects 
   */
  initialize_empty_list ();

  true_object = make_true ();
  false_object = make_false ();
  
  key_symbol = make_symbol ("#key");
  allkeys_symbol = make_symbol ("#all-keys");
  rest_symbol = make_symbol ("#rest");
  next_symbol = make_symbol ("#next");
  values_symbol = make_symbol ("#values");
  all_symbol = make_symbol ("all");
  quote_symbol = make_symbol ("quote");
  eof_object = make_eof_object ();
  unspecified_object = make_unspecified_object ();
  unwind_symbol = make_symbol ("%unwind");
  next_method_symbol = make_symbol ("next-method");
  initialize_symbol = make_symbol ("initialize");
  equal_hash_symbol = make_symbol ("=hash");
  uninit_slot_object = make_uninit_slot ();
  standard_input_stream = make_stream (Input, stdin);
  standard_output_stream = make_stream (Output, stdout);
  standard_error_stream = make_stream (Output, stderr);
  quasiquote_symbol = make_symbol ("quasiquote");
  unquote_symbol = make_symbol ("unquote");
  unquote_splicing_symbol = make_symbol ("unquote-splicing");
  element_symbol = make_symbol("element");
  element_setter_symbol = make_symbol("element-setter");
  signal_symbol = make_symbol("signal");
  
  /* make the unspecified object available
   */
  add_top_level_binding (make_symbol ("%unspecified"), unspecified_object);

  /* make the uninitialize slot value available
   */
  add_top_level_binding (make_symbol ("%uninitialized-slot-value"),
			 uninit_slot_object);

  /* make default object
   */

  default_object = cons (false_object, false_object);
  add_top_level_binding (make_symbol ("%default-object"), default_object);


  /* often used keywords
   */
  getter_keyword = make_keyword ("getter:");
  setter_keyword = make_keyword ("setter:");
  else_keyword = make_keyword ("else:");
  type_keyword = make_keyword ("type:");
  init_value_keyword = make_keyword ("init-value:");
  init_function_keyword = make_keyword ("init-function:");
  init_keyword_keyword = make_keyword ("init-keyword:");
  required_init_keyword_keyword = make_keyword ("required-init-keyword:");
  allocation_keyword = make_keyword ("allocation:");
  super_classes_keyword = make_keyword ("superclasses:");
  slots_keyword = make_keyword("slots:");
  debug_name_keyword = make_keyword("debug-name:");
  size_keyword = make_keyword ("size:");
  fill_keyword = make_keyword ("fill:");
  dim_keyword = make_keyword ("dimensions:");
  min_keyword = make_keyword ("min:");
  max_keyword = make_keyword ("max:");

  /* often used symbols
   */
  instance_symbol = make_symbol("instance");
  class_symbol = make_symbol("class");
  each_subclass_symbol = make_symbol("each-subclass");
  constant_symbol = make_symbol("constant");
  virtual_symbol = make_symbol("virtual");
  object_class_symbol = make_symbol ("object-class");

  obj_sym = make_symbol("obj");
  slot_val_sym = make_symbol ("slot-value");
  set_slot_value_sym = make_symbol("set-slot-value!");
  val_sym = make_symbol ("val");
  initial_state_sym = make_symbol ("initial-state");
  next_state_sym = make_symbol ("next-state");
  current_element_sym = make_symbol ("current-element");

  /* infrequently used, but important symbols */
  instance_slots_symbol = make_symbol ("%instance-slots");
  class_slots_symbol = make_symbol ("%class-slots");
  each_subclass_slots_symbol = make_symbol ("%each-subclass-slots");

  /* initialize table of syntax operators and functions
   */
  init_syntax_table ();

  /* initialize builtin classes
   */
  init_class_hierarchy ();

  /* initialize primitives
   */
  init_list_prims ();
  init_class_prims ();
  init_slot_prims ();
  init_file_prims ();
  init_function_prims ();
  init_values_prims ();
  init_print_prims ();
  init_number_prims ();
  init_apply_prims ();
  init_boolean_prims ();
  init_keyword_prims ();
  init_string_prims ();
  init_vector_prims ();
  init_error_prims ();
  init_stream_prims ();
  init_read_prims ();
  init_table_prims ();
  init_misc_prims ();
  init_char_prims ();
  init_deque_prims ();
  init_array_prims ();
  init_sys_prims ();
#ifdef THINK_C
  init_mac_prims ();
#endif
}
