/*

   error.c

   Copyright, 1993, Brent Benson.  All Rights Reserved.
   0.4 Revisions Copyright 1994, Joseph N. Wilson.  All Rights Reserved.
   
   Permission to use, copy, and modify this software and its
   documentation is hereby granted only under the following terms and
   conditions.  Both the above copyright notice and this permission
   notice must appear in all copies of the software, derivative works
   or modified version, and both notices must appear in supporting
   documentation.  Users of this software agree to the terms and
   conditions set forth in this notice.

*/


#include <stdio.h>
#include <signal.h>
#include "class.h"
#include "env.h"
#include "error.h"
#include "print.h"
#include "string.h"
#include "list.h"

#define NUMSIGNALS 32
#define IGNORE  0
#define ERROR   1
#define DEFAULT 2

#ifndef __NetBSD__
extern char *sys_siglist[];
#endif
extern Object signal_symbol;
extern Object simple_error_class;

/*
 *  For those who don't have an example like sys_siglist
 */

/*
char *signal_strings[32] =
{"",
 "hangup",
 "interrupt",
 "quit",
 "illegal instruction (not reset when caught)",
 "trace trap (not reset when caught)",
 "IOT instruction",
 "EMT instruction",
 "floating point exception",
 "kill (cannot be caught or ignored)",
 "bus error",
 "segmentation violation",
 "bad argument to system call",
 "write on a pipe with no one to read it",
 "alarm clock",
 "software termination signal from kill",
 "urgent condition on IO channel",
 "sendable stop signal not from tty",
 "stop signal from tty",
 "continue a stopped process",
 "to parent on child stop or exit",
 "to readers pgrp upon background tty read",
 "like TTIN for output if (tp->t_local&LTOSTOP)",
 "input/output possible signal",
 "exceeded CPU time limit",
 "exceeded file size limit",
 "virtual time alarm",
 "profiling time alarm",
 "window changed",
 "resource lost (eg, record-lock lost)",
 "user defined signal 1",
 "user defined signal 2",
};
*/

int signal_response[32] =
{ IGNORE,
  ERROR,  /* hangup */
  DEFAULT, /* interrupt */
  DEFAULT, /* quit */
  DEFAULT, /* illegal instruction (not reset when caught) */
  DEFAULT, /* trace trap (not reset when caught) */
  DEFAULT, /* IOT instruction */
  DEFAULT, /* EMT instruction */
  ERROR,   /* floating point exception */
  IGNORE,  /* kill (cannot be caught or ignored) */
  DEFAULT, /* bus error */
  DEFAULT, /* segmentation violation */
  DEFAULT, /* bad argument to system call */
  ERROR,   /* write on a pipe with no one to read it */
  IGNORE,  /* alarm clock */
  ERROR,   /* software termination signal from kill */
  DEFAULT, /* urgent condition on IO channel */
  DEFAULT,   /* sendable stop signal not from tty */
  DEFAULT,   /* stop signal from tty */
  DEFAULT, /* continue a stopped process */
  DEFAULT, /* to parent on child stop or exit */
  DEFAULT, /* to readers pgrp upon background tty read */
  DEFAULT, /* like TTIN for output if (tp->t_local&LTOSTOP) */
  ERROR, /* input/output possible signal */
  ERROR, /* exceeded CPU time limit */
  ERROR, /* exceeded file size limit */
  ERROR, /* virtual time alarm */
  ERROR, /* profiling time alar */
  ERROR, /* window changed */
  DEFAULT, /* resource lost (eg, record-lock lost) */
  ERROR, /* user defined signal 1 */
  ERROR /* user defined signal 2 */
};

static void signal_handler(int sig);

/* primitives */

void signal_handler_init();

static Object dylan_error (Object msg_str, Object rest);
static Object dylan_warning (Object msg_str, Object rest);
static Object signal_error_jump();
static struct primitive error_prims[] =
{
  {"%error", prim_1_rest, dylan_error},
  {"%warning", prim_1_rest, dylan_warning},
  {"%signal-error-jump", prim_0, signal_error_jump},
  
};

/* function definitions */

void
init_error_prims (void)
{
    int num;

    num = sizeof (error_prims) / sizeof (struct primitive);
    init_prims (num, error_prims);
    signal_handler_init ();
}

void
signal_handler_init ()
{
    int i;
    for (i = 0; i < NUMSIGNALS; i++){
	switch (signal_response[i]){
	case IGNORE:
	    signal (i, SIG_IGN);
	    break;
	case ERROR:
	    signal (i, signal_handler);
	    break;
	case DEFAULT:
	    ;
	}
    }
}

static Object 
dylan_error (Object msg_str, Object rest)
{
  fprintf (stderr, "error: %s", BYTESTRVAL (msg_str));
  if (! NULLP (rest))
    {
      fprintf (stderr, ": ");
    }
  while (! NULLP (rest))
    {
      print_object (stderr, CAR (rest), 0);
      rest = CDR (rest);
      if (! NULLP (rest))
	{
	  fprintf (stderr, ", ");
	}
    }
  fprintf (stderr, ".\n");
  longjmp (error_return, 1);
}

static Object 
dylan_warning (Object msg_str, Object rest)
{
  fprintf (stderr, "warning: %s", BYTESTRVAL (msg_str));
  if (! NULLP (rest))
    {
      fprintf (stderr, ": ");
    }
  while (! NULLP (rest))
    {
      print_object (stderr, CAR (rest), 0);
      rest = CDR (rest);
      if (! NULLP (rest))
	{
	  fprintf (stderr, ", ");
	}
    }
  fprintf (stderr, ".\n");
}

void 
fatal (char *msg)
{
  fprintf (stderr, "%s.\n", msg);
  exit (-1);
}

void 
error (char *msg, ...)
{
  va_list args;
  Object obj, signal_value;

  va_start (args, msg);
  fprintf (stderr, "error: %s", msg);
  obj = va_arg (args, Object);
  if ( obj )
    {
      fprintf (stderr, ": ");
    }
  while ( obj )
    {
      print_object (stderr, obj, 0);
      obj = va_arg (args, Object);
      if ( obj )
	{
	  fprintf (stderr, ", ");
	}
    }
  fprintf (stderr, ".\n");

  signal_value = symbol_value (signal_symbol);
  if (signal_value) {
      apply (signal_value,
	     cons (make(simple_error_class, make_empty_list()),
		   make_empty_list()));
  } else {
      longjmp (error_return, 1);
  }
}

static Object
signal_error_jump ()
{
    longjmp (error_return, 1);
}

void 
warning (char *msg, ...)
{
  va_list args;
  Object obj;

  va_start (args, msg);
  fprintf (stderr, "warning: %s", msg);
  obj = va_arg (args, Object);
  if ( obj )
    {
      fprintf (stderr, ": ");
    }
  while ( obj )
    {
      print_object (stderr, obj, 0);
      obj = va_arg (args, Object);
      if ( obj )
	{
	  fprintf (stderr, ", ");
	}
    }
  fprintf (stderr, ".\n");
}

static void
signal_handler(int sig)
{
    error (sys_siglist[sig], NULL);
}

