/*

   apply.c

   Copyright, 1993, Brent Benson.  All Rights Reserved.
   0.4 Revisions Copyright 1994, Joseph N. Wilson.  All Rights Reserved.
   
   Permission to use, copy, and modify this software and its
   documentation is hereby granted only under the following terms and
   conditions.  Both the above copyright notice and this permission
   notice must appear in all copies of the software, derivative works
   or modified version, and both notices must appear in supporting
   documentation.  Users of this software agree to the terms and
   conditions set forth in this notice.

*/

#include "apply.h"
#include "eval.h"
#include "list.h"
#include "print.h"
#include "prim.h"
#include "env.h"
#include "class.h"
#include "function.h"
#include "keyword.h"
#include "values.h"
#include "symbol.h"
#include "error.h"

/* global data */
int trace_functions = 0;
int trace_only_user_funs = 0;
int trace_level = 0;

/* local function prototypes and data */

Object apply_generic (Object gen, Object args);
static Object apply_exit (Object exit_proc, Object args);
static Object apply_next_method (Object next_method, Object args);
static Object set_trace (Object bool);
static Object user_keyword;

/* primitives */

static struct primitive apply_prims[] =
{
  {"%apply", prim_2, apply},
  {"%trace", prim_1, set_trace},
};

/* function definitions */

void
init_apply_prims (void)
{
  int num;

  num = sizeof (apply_prims) / sizeof (struct primitive);
  init_prims (num, apply_prims);

  user_keyword = make_keyword("user:");

}

Object 
apply (Object fun, Object args)
{
  Object ret;

#ifdef THINK_C
  check_stack();
#endif

  if ( trace_functions )
    {
      int i;
      
      if ((! trace_only_user_funs) || (! PRIMP(fun)))
	{
	  printf ("; ");
	  for ( i=0 ; i < trace_level ; ++i )
	    {
	      printf ("-");
	    }
	  print_object (stdout, fun, 1);
	  printf (" called with ");
	  print_object (stdout, args, 1);
	  printf ("\n");
	  trace_level++;
	}
    }
#ifdef SMALL_OBJECTS
  if (! POINTERP (fun))
    {
      error ("apply: cannot apply this object", fun, NULL);
    }
#endif
  switch (POINTERTYPE(fun))
    {
    case Primitive:
      ret = apply_prim (fun, args);
      break;
    case Method:
      ret = apply_method (fun, args, make_empty_list());
      break;
    case GenericFunction:
      ret = apply_generic (fun, args);
      break;
    case NextMethod:
      ret = apply_next_method (fun, args);
      break;
    case Exit:
      ret = apply_exit (fun, args);
      break;
    default:
      error ("apply: cannot apply this object", fun, NULL);
    }
  if ( trace_functions && trace_level )
    {
      int i;

      if ((! trace_only_user_funs) || (! PRIMP(fun)))
	{
	  trace_level--;
	  printf ("; ");
	  for ( i=0 ; i < trace_level ; ++i )
	    {
	      printf ("-");
	    }
	  printf ("returned: ");
	  print_object (stdout, ret, 1);
	  printf ("\n");
	}
    }
  return (ret);
}

/* local functions */

/*
 * It seems to me that apply method has gotten a little big.
 * It could benefit from modularizing in a rewrite.
 *		-jnw
 */
Object
apply_method (Object meth, Object args, Object rest_methods)
{
    Object params, param, sym, val, body, ret, newret;
    Object ret_types, tmp, dup_list;
    Object rest_var, class, keyword, keys, key_decl;
    Object *tmp_ptr, old;
    int hit_rest, hit_key, hit_values;
    struct frame *old_env;
    int i, j;
    
    ret = unspecified_object;
    params = METHREQPARAMS (meth);
    body = METHBODY (meth);

    
    /* remember current environment and subsitute with
       environment present at method creation time */
    old_env = the_env;
    the_env = METHENV (meth);
    
    push_scope ();
    
    /* install of next method object if there are next methods */
    if ( PAIRP(rest_methods)) { /* check use of empty_list vs. NULL!!*/
	Object next_method;
	
	next_method = make_next_method (rest_methods, args);
	push_scope ();
	add_binding (METHNEXTMETH (meth), next_method);
    }
    
    hit_rest = hit_key = hit_values = 0;
    
    /* first process required parameters */
    while ((PAIRP (params) && PAIRP (args))
	   && (! hit_rest) && (! hit_key) && !( hit_values)) {
	param = CAR (params);
	if (param == rest_symbol) {
	    hit_rest = 1;
	} else if (param == key_symbol) {
	    hit_key = 1;
	} else if (param == values_symbol) {
	    hit_values = 1;
	} else {
	    val = CAR (args);
	    if (SYMBOLP (param)) {
		sym = param;
	    } else {
		sym = FIRST (param);
		class = SECOND (param);
		if ( ! instance (val, class)) {
		    error ("apply: argument doesn't match method specializer", 
			   val, class, meth, NULL);
		}
	    }
	    add_binding (sym, val);
	    args = CDR (args);
	    params = CDR (params);
	}
    }
    /* now process #rest and #key parameters */
	
    if ((rest_var = METHRESTPARAM(meth)) != NULL) {
	add_binding (rest_var, args);
    }
    if (PAIRP (METHKEYPARAMS (meth))) {
	/* copy keys */
	keys = copy_list(METHKEYPARAMS (meth));

	dup_list = make_empty_list(); /* For duplicate keywords */

	/* Bind each of the keyword args that is present. */
	while (! NULLP (args)) {
	    keyword = FIRST (args);
	    if (!KEYWORDP (keyword)) {
		/* jnw -- check this out! */
		if (!rest_var) {
		    error ("apply: argument to method must be keyword", meth, keyword, NULL);
		} else {
		    args = CDR (args);
		    continue;
		}
	    }
	    val = SECOND (args);
	    
	    /* if keyword is in the keys list then
	     * 1) add a binding for keyword to val
	     * 2) remove the keyword entry from keys
	     */

	    for (tmp_ptr = &keys;
		 PAIRP(*tmp_ptr);
		 tmp_ptr = &CDR (*tmp_ptr)) {
		if (symbol_to_keyword (CAR (CAR (*tmp_ptr))) == keyword) {
		    break;
		}
	    }
	    if (EMPTYLISTP (*tmp_ptr)){
		if (member (keyword, dup_list)) {
		    warning("Duplicate keyword value ignored",
			    keyword, val, NULL);
		} else {
		    error("Keyword argument not in parameter list or given twice",
			  keyword, NULL);
		}
	    } else {
		add_binding (CAR (CAR (*tmp_ptr)), val);
		dup_list = cons (keyword, dup_list);
		*tmp_ptr = CDR (*tmp_ptr);
	    }
	    args = CDR (CDR (args));
	}
	/* Bind the missing keyword args to default_object */
	while (PAIRP (keys)) {
	    add_binding (CAR (CAR (keys)), eval(SECOND (CAR (keys))));
	    keys = CDR (keys);
	}
	
    }

    if (PAIRP (args) && !rest_var) {
	if (METHALLKEYS (meth)) {
	    /* skip rest of parameters if they are keywords */
	    while (PAIRP (args)) {
		if (!KEYWORDP (CAR (args))) {
		    error ("apply: keyword argument expected", CAR (args),
			   NULL);
		} else if (!PAIRP (CDR (args))) {
		    error("apply: keyword has no associated argument value",
			  CAR (args), NULL);
		}
		args = CDR (CDR (args));
	    }
	} else {
	    error("Arguments have no matching parameters", args, NULL);
	}
    }
    if (PAIRP (params)) {
	error("Required parameters have no matching arguments", params,
	      NULL);
    }
    
    while (! NULLP (body)) {
	ret = eval (CAR (body));
	body = CDR (body);
    }
    pop_scope();
    
    /* re-assert environment present at the beginning of this function
     */
    the_env = old_env;

    /* To save effort, I make sure the return is a VALUES object.
     * This is a waste of effort and really ought to be fixed.
     */
    
    if (!VALUESP (ret)) {
	ret = make_values( listem( ret, NULL));
    }
    ret_types = METHREQVALUES (meth);
    
    /* check return values (not done for non VALUESTYPE values yet */
    for (i=0;
	 i<VALUESNUM(ret) && PAIRP (ret_types);
	 i++, ret_types = CDR (ret_types)){
	if ( ! instance (VALUESELS(ret)[i],CAR (ret_types))){
	    error ("apply: return value is not of correct type",
		   VALUESELS (ret)[i], CAR (ret_types), NULL);
	}
    }
    if (i < VALUESNUM (ret)) {
	/* We have more return values than specific return types.
	 * Check them against the #rest value return type
	 */
	if (METHRESTVALUES (meth) != NULL) {
	    for (;i<VALUESNUM (ret);i++) {
		if ( ! instance (VALUESELS (ret)[i],
				 METHRESTVALUES (meth))) {
		    error("apply: return value is not of correct type",
			  VALUESELS (ret)[i], METHRESTVALUES (meth), NULL);
		}
	    }
	} else {
	    /* Discard the extra values by ignoring them. */
	    VALUESNUM (ret) = i;
	}
    } else if (PAIRP (ret_types))  {
	/* Add default values */
	for (j = 0; PAIRP (ret_types); j++, ret_types = CDR (ret_types)){
	    if ( ! instance (false_object, CAR (ret_types))) {
		error("apply: default value doesn't match return type",
		      CAR (ret_types), NULL);
	    }
	}
	newret = allocate_object (sizeof (struct values));
	VALUESTYPE (newret) = Values;
	VALUESNUM (newret) = i+j;
	VALUESELS (newret) = (Object *)
	    checking_malloc (VALUESNUM (newret) * sizeof (Object));
	for (i=0; i<VALUESNUM(ret); i++){
	    VALUESELS (newret)[i] = VALUESELS (ret) [i];
	}
	for (; i<VALUESNUM (newret); i++){
	    VALUESELS (newret)[i] = false_object;
	}
	ret = newret;
    }


    /* turn stupid multiple value into single value */
    if (VALUESNUM(ret) == 1) {
	ret = VALUESELS(ret)[0];
    }
    return (ret);
}

Object
apply_generic (Object gen, Object args)
{
  Object methods, sorted_methods;

  methods = GFMETHODS (gen);
  sorted_methods = FIRSTVAL (sorted_applicable_methods (gen, args));
  if (EMPTYLISTP (sorted_methods)) {
      error("Ambiguous methods in apply generic function", gen, args, NULL);
  } else {
      return (apply_method (CAR (sorted_methods), args, CDR (sorted_methods)));
  }
}

static Object 
apply_exit (Object exit_proc, Object args)
{
  Object vals;

  unwind_to_exit (EXITSYM (exit_proc));
  switch (list_length (args)) {
  case 0: longjmp (*EXITRET (exit_proc), (int)(unspecified_object));
  case 1: longjmp (*EXITRET (exit_proc), (int)FIRST (args));
  default: longjmp (*EXITRET(exit_proc), (int)(values (args)));
  }
}

static Object
apply_next_method (Object next_method, Object args)
{
  Object rest_methods, real_args;

  rest_methods = NMREST (next_method);
  if (NULLP (args))
    {
      real_args = NMARGS (next_method);
    }
  else
    {
      real_args = args;
    }
  return (apply_method (CAR (rest_methods), real_args, CDR (rest_methods)));
}

static Object 
set_trace (Object flag)
{
  if (flag == false_object)
    {
      trace_functions = 0;
      trace_only_user_funs = 0;
    }
  else
    {
      trace_functions = 1;
      if (flag == user_keyword)
	{
	  trace_only_user_funs = 1;
	}
    }
  return (flag);
}
