;;; -*- Mode: LISP; Syntax: Common-lisp; Package: YY; Base: 10 -*-
;;;
;;; yy-image.lisp
;;;
;;;  Copyright (C) 1989,1990,1991 Aoyama Gakuin University
;;;  graphic-primitive.lisp
;;;
;;;		All Rights Reserved
;;;
;;; This software is developed for the YY project of Aoyama Gakuin University.
;;; Permission to use, copy, modify, and distribute this software
;;; and its documentation for any purpose and without fee is hereby granted,
;;; provided that the above copyright notices appear in all copies and that
;;; both that copyright notice and this permission notice appear in 
;;; supporting documentation, and that the name of Aoyama Gakuin
;;; not be used in advertising or publicity pertaining to distribution of
;;; the software without specific, written prior permission.
;;;
;;; This software is made available AS IS, and Aoyama Gakuin makes no
;;; warranty about the software, its performance or its conformity to
;;; any specification. 
;;;
;;; To make a contact: Send E-mail to ida@csrl.aoyama.ac.jp for overall
;;; issues. To ask specific questions, send to the individual authors at
;;; csrl.aoyama.ac.jp. To request a mailing list, send E-mail to 
;;; yyonx-request@csrl.aoyama.ac.jp.
;;;
;;; Authors:
;;;   version 1.3 91/01/12 by t.kosaka (kosaka@csrl.aoyama.ac.jp)

;;; $@%$%a!<%8=hM}$N4pK\%W%j%_%F%#%V!"%a%=%C%I(J
(in-package :yy)


;;; print-object $@%a%=%C%I(J
(defmethod print-object ((object image) stream)
  (format stream "\#<Image size ~ax~a format ~a >" (image-width object)
	  (image-height object) (image-format object))
  )

;==========================================================================
; /*	@(#)rasterfile.h 1.9 88/02/07 SMI	*/
;
; /*
;  * Description of header for files containing raster images
;  */
; struct rasterfile {
; 	int	ras_magic;		/* magic number */
; 	int	ras_width;		/* width (pixels) of image */
; 	int	ras_height;		/* height (pixels) of image */
; 	int	ras_depth;		/* depth (1, 8, or 24 bits) of pixel */
; 	int	ras_length;		/* length (bytes) of image */
; 	int	ras_type;		/* type of file; see RT_* below */
; 	int	ras_maptype;		/* type of colormap; see RMT_* below */
; 	int	ras_maplength;		/* length (bytes) of following map */
; 	/* color map follows for ras_maplength bytes, followed by image */
; };
; #define	RAS_MAGIC	0x59a66a95
; 
; 	/* Sun supported ras_type's */
; #define RT_OLD		0   /* Raw pixrect image in 68000 byte order */
; #define RT_STANDARD	1	/* Raw pixrect image in 68000 byte order */
; #define RT_BYTE_ENCODED	2	/* Run-length compression of bytes */
; #define RT_EXPERIMENTAL 0xffff	/* Reserved for testing */
; 
; 	/* Sun registered ras_maptype's */
; #define RMT_RAW		2
; 	/* Sun supported ras_maptype's */
; #define RMT_NONE	0	/* ras_maplength is expected to be 0 */
; #define RMT_EQUAL_RGB	1	/* red[ras_maplength/3],green[],blue[] */
;
; /*
;  * NOTES:
;  * 	Each line of the image is rounded out to a multiple of 16 bits.
;  *   This corresponds to the rounding convention used by the memory pixrect
;  *   package (/usr/include/pixrect/memvar.h) of the SunWindows system.
;  *	The ras_encoding field (always set to 0 by Sun's supported software)
;  *   was renamed to ras_length in release 2.0.  As a result, rasterfiles
;  *   of type 0 generated by the old software claim to have 0 length; for
;  *   compatibility, code reading rasterfiles must be prepared to compute the
;  *   true length from the width, height, and depth fields.
;  */
;----------------------------------------------------------------------------

(defun load-raster-from-file ( path )
  (declare (function draw-prompt (string) t))
;;  (format t "~%;Loading raster file")
  (draw-prompt (format nil "Loding Image FIle : ~a" path))
  (with-open-file (pixel-stream path :direction :input :element-type 
		   '(unsigned-byte 8) :if-does-not-exist nil)
     (make-raster-and-read-arrays pixel-stream)
      ))

;===========================================================================
; lower functions for rasterfile
;---------------------------------------------------------------------------


(defun make-raster-and-read-arrays (rasterfile-stream)
  (declare 
   #-CMU
   (inline / = > incf)
   (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((raster-magic (read-long-w-680x0-order rasterfile-stream)))
    (cond
     ((= raster-magic #16r59a66a95)
      (load-sun-raster-file rasterfile-stream))
     ((= raster-magic 14876)  ;;; YY format
      (let ((F-type (read-long-w-680x0-order rasterfile-stream))
	    (F-width (read-long-w-680x0-order rasterfile-stream))
	    (F-height (read-long-w-680x0-order rasterfile-stream)))
	(case F-type
	   (1   ;;; $@%+%i!<(J
	    (let ((image (make-array (* F-width F-height 3)
				     :element-type 'fixnum))
		  (count 0))
	      (do ((i 0 (incf i 4)))
		  ((= i (* F-width F-height 12)))
		 (setf (aref image count)
		       (logior (ash (lisp:read-byte rasterfile-stream) 24)
			       (ash (lisp:read-byte rasterfile-stream) 16)
			       (ash (lisp:read-byte rasterfile-stream) 8)
			       (lisp:read-byte rasterfile-stream)))
		 (incf count)
		 )
	      (values 3 F-width F-height image nil nil nil nil)))
	   (3   ;;; $@%b%N%/%m(J
	    (let ((image (make-array (* F-height (ceiling F-width 4))
				     :element-type '(unsigned-byte 8))))
	      (dotimes (i (length image))
		    (setf (aref image i)
                       (lisp:read-byte rasterfile-stream)))
              (values 4 F-width F-height image nil nil nil nil)))
	   (T
	    (format t "Not Suport Gray scale !!!~%")))))
     (t
	;;; Commo  Windows
      (format t "NOt imprement !!~%")))
    ))

(defun load-sun-raster-file (rasterfile-stream)
  (declare 
   #-CMU
   (inline / ash = > incf)
   (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((F-width (read-long-w-680x0-order rasterfile-stream))
	(F-height (read-long-w-680x0-order rasterfile-stream))
	(F-depth (read-long-w-680x0-order rasterfile-stream))
	(F-length (read-long-w-680x0-order rasterfile-stream))
	(F-type (read-long-w-680x0-order rasterfile-stream))
	(F-maptype (read-long-w-680x0-order rasterfile-stream))
	(F-maplength (read-long-w-680x0-order rasterfile-stream)))
    (declare (ignore F-maptype F-type))
    (cond 
     ((= F-depth 8)			;$@%+%i!<$N>l9g(J
      (if (> F-maplength 0)
	  (let ((red-lut-array (make-array (floor F-maplength 3)  
					   :element-type 'fixnum))
		(green-lut-array (make-array (floor F-maplength 3)
					     :element-type 'fixnum))
		(blue-lut-array (make-array (floor F-maplength 3) 
					    :element-type 'fixnum))
		(raster-array (make-array (list F-height F-width) 
					  :element-type '(unsigned-byte 8))))
	    ;;; $@%+%i!<%^%C%W$rFI$_9~$`(J   
	    (dotimes (lut-index (/ F-maplength 3))
	      (setf (aref red-lut-array lut-index)
		(ash (lisp:read-byte rasterfile-stream) 8)))

	    (dotimes (lut-index (/ F-maplength 3))
	      (setf (aref green-lut-array lut-index)
		(ash (lisp:read-byte rasterfile-stream ) 8)))

	    (dotimes (lut-index (/ F-maplength 3))
	      (setf (aref blue-lut-array lut-index)
		(ash (lisp:read-byte rasterfile-stream) 8)))

	    ;;;$@%i%9%?!<%$%a!<%8$NFI$_9~$_(J      
	    (dotimes (scan-line F-height)
	      (dotimes (pixel F-width)
		(setf (aref raster-array scan-line pixel)
		  (lisp:read-byte rasterfile-stream))))
	    
	    (values 1 F-width F-height raster-array
		    red-lut-array green-lut-array blue-lut-array))
	nil))
     ((= F-depth 1) ;;; $@%b%N%/%m(J
      (let ((raster-array (make-array 
			   (list F-height (round (/ F-length F-height)))
			   :element-type '(unsigned-byte 8))))

        ;;;$@%i%9%?!<%$%a!<%8$NFI$_9~$_(J
	(dotimes (scan-line F-height)
	  (dotimes (pixel (round (/ F-length F-height)))
	    (setf (aref raster-array scan-line pixel)
	      (lisp:read-byte rasterfile-stream))))
	(values 2 F-width F-height raster-array nil nil nil nil)))
     (t
      nil)))
  )
      
	   
;------------------------------------------------------------------------------
;-----------------------------------------------------------------------------

(defun write-long-w-680x0-order (long write-stream)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (lisp:write-byte (logand 255 (ash long -24)) write-stream)
  (lisp:write-byte (logand 255 (ash long -16)) write-stream)
  (lisp:write-byte (logand 255 (ash long -8))  write-stream)
  (lisp:write-byte (logand 255 long)           write-stream)
  )

;----------------------------------------------------------------------------

(defun read-long-w-680x0-order (read-stream)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (prog (long )
	(setq long (ash (logand 255 (lisp:read-byte read-stream )) 24))
    (setq long (logior long (ash (logand 255 (lisp:read-byte read-stream )) 16)))
    (setq long (logior long (ash (logand 255 (lisp:read-byte read-stream )) 8)))
    (setq long (logior long (logand 255 (lisp:read-byte read-stream ))))
    (return long)
    )
  )

;;; YY$@$N%b%N%/%m%$%a!<%8%G!<%?$N>uBV$rF@$k(J
(defun yy-image-bit (yy-image-data line-bytes x y)
  (declare 
   #-CMU
   (inline logand logxor floor * + mod - ash)
   (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let* ((data (elt yy-image-data (floor (+ (* y line-bytes)
					   (floor (/ x 4))))))
	 (shiht (- 0 (* (mod x 4) 2))))
    (ash (logand data (ash 192 shiht)) (- 0 (+ 6 shiht)))
    ))


;;; YY$@$N%b%N%/%m%$%a!<%8%G!<%?$KCM$r@_Dj(J
(defun set-yy-image-bit (yy-image-data line-bytes x y set)
  (declare 
   #-CMU
   (inline logand logxor floor * + mod - ash)
   (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let* ((count (floor (+ (* y line-bytes)
			  (floor (/ x 4)))))
	 (data (elt yy-image-data count))
	 (shiht (- 0 (* (mod x 4) 2)))
	 (temp (logand data 
		       (logxor (ash 192 shiht) #xFF))))

    (setf (elt yy-image-data count) 
	  (logior temp (ash set shiht)))
    ))

;;; setf $@4X?t$NDj5A(J
(eval-when (compile load eval)
(defsetf yy-image-bit set-yy-image-bit))


(defun make-image-form-raster-file (file-name &key (start-x 0) (start-y 0)
					      (width 0) (height 0)
					      (default-color *white-color*))
  (declare (function draw-prompt (string) t)
	   (special *wihte-color*)
	   (optimize (compilation-speed 0) (speed 3) (safety 0)))

  (let ((color *white-color*)
	(count 0))

    (multiple-value-bind (file-type r-width r-height raster 
			   red-map
			   green-map blue-map)

        (load-raster-from-file file-name)
	(if (zerop width)
	    (setf width r-width))
	(if (zerop height)
	    (setf height r-height))

	(case file-type
	  (1 ;;; $@%i%9%?!<%U%!%$%k$G%+%i!<$N>l9g(J
	   (let ((image-data (make-array (* width height 4) :element-type 
					 '(unsigned-byte 8)))
		 (ret nil))
	     (do ((y start-y (incf y)))
		 ((= y (+ height start-y)) nil)
		 (do ((x start-x (incf x)))
		     ((= x (+ width start-x)) nil)
		     (setf color (find-color :red (aref red-map 
							(aref raster y x))
				     :green (aref green-map 
						  (aref raster y x))
				     :blue (aref blue-map 
						 (aref raster y x))))
		     (when (null color)
			   (setf color (make-color 
				 :red (aref red-map (aref raster y x))
				 :green (aref green-map (aref raster y x))
				 :blue (aref blue-map (aref raster y x)))))

		     (when (null color)
       		       (if (null (setf color (near-from-rgb (aref red-map
							  (aref raster y x))
						      (aref green-map
							  (aref raster y x))
						      (aref blue-map
							  (aref raster y x)))))
			   (progn
			    ;;; $@%+%i!<$,:n$l$J$$(J
			     (setf color default-color)
			     (push (list x y (aref red-map (aref raster y x))
					 (aref green-map (aref raster y x))
					 (aref blue-map (aref raster y x))) 
				   ret))))
		     
		     (setf (aref image-data count)    
			   (get-one-byte (color-no color) 3)
			   (aref image-data (incf count)) 
			   (get-one-byte (color-no color) 2)
			   (aref image-data (incf count)) 
			   (get-one-byte (color-no color) 1)
			   (aref image-data (incf count)) 
			   (get-one-byte (color-no color) 0))
		     (incf count))
	       (draw-prompt 
		(format nil "Making Image ~a%"
			(ceiling (* (/ y (+ height start-y)) 100))))
	       )
	     (draw-prompt "")
	     (values (make-instance 'color-image :image-width width 
				    :line-bytes (* width 4)
				    :image-height height
				    :image-type :color :image-format :yy
				    :image-data image-data)
		     ret))
	   )
	  (2 ;;; $@%i%9%?!<%U%!%$%k$G%b%N%/%m(J
	   (let* ((line-bytes (ceiling (/ width 4)))
		  (image-data (make-array (* height line-bytes 2)
					  :element-type '(unsigned-byte 8))))
	     (do ((y start-y (incf y)))
		 ((= y (+ height start-y)))
		 (do ((x start-y (incf x)))
		     ((= x (+ width start-x)))
		     
		     (unless (zerop (sun-raster-bit raster r-width x y))
			     (setf (yy-image-bit image-data line-bytes
						 x y) #b01))
		     )
	       (draw-prompt 
		(format nil "Making Image ~a%"
			(ceiling (* (/ y (+ height start-y)) 100))))
	       )
	     (draw-prompt "")
	     (values (make-instance 'mono-image :image-width width 
				    :line-bytes (ceiling (/ width 4))
				    :image-height height
				    :image-type :mono :image-format :yy
				    :image-data image-data)
		     nil)))
	  (3 ;;; YY format$@$G%+%i!<(J
	   (let ((image-data (make-array (* width height 4)
					 :element-type '(unsigned-byte 8)))
		 (ret nil)
		 (color *white-color*)
		 (point 0))

	     (do ((y start-y (incf y)))
		 ((= y (+ height start-y)))
		 (do ((x start-x (incf x)))
		     ((= x (+ width start-x)))

		     (setf point (+ (* y 3 r-width) (* x 3)))

		     (setf color (find-color :red (aref raster point)
				     :green (aref raster (+ point 1))
				     :blue (aref raster (+ point 2))))

		     (when (null color)
			   (setf color (make-color 
				 :red (aref raster point)
				 :green (aref raster (+ point 1))
				 :blue (aref raster (+ point 2)))))

		     (when (null color)
			    ;;; $@%+%i!<$,:n$l$J$$(J
			   (setf color default-color)
			   (push (list x y (aref raster point)
				       (aref raster (+ point 1))
				       (aref raster (+ point 2))) ret))
		     
		     (setf (aref image-data count)    
			   (get-one-byte (color-no color) 3)
			   (aref image-data (incf count)) 
			   (get-one-byte (color-no color) 2)
			   (aref image-data (incf count)) 
			   (get-one-byte (color-no color) 1)
			   (aref image-data (incf count)) 
			   (get-one-byte (color-no color) 0))
		     (incf count))
	       (draw-prompt 
		(format nil "Making Image ~a%"
			(ceiling (* (/ y (+ height start-y)) 100))))
	       )
	     (draw-prompt "")
	     (values (make-instance 'color-image :image-width width 
				    :line-bytes (* width 4)
				    :image-height height
				    :image-type :color :image-format :yy
				    :image-data image-data))))
	  (4 ;;; YY format $@%b%N%/%m(J
	   (let* ((line-bytes (ceiling (/ width 4)))
		  (image-data (make-array (* height line-bytes)
					  :element-type '(unsigned-byte 8)))
		  (old-line-bytes (ceiling (/ r-width 4))))
	     (do ((y start-y (incf y)))
		 ((= y (+ height start-y)))
		 (do ((x start-y  (incf x)))
		     ((= x (+ width start-x)))
		     
		     (setf (yy-image-bit image-data line-bytes x y)
			   (yy-image-bit raster old-line-bytes (+ start-x x)
					 (+ start-y y))))
	       (draw-prompt 
		(format nil "Making Image ~a%"
			(ceiling (* (/ y (+ height start-y)) 100))))
	       )
	     (draw-prompt "")
	     (values (make-instance 'mono-image :image-width width 
    				    :line-bytes line-bytes
				    :image-height height
				    :image-type :mono :image-format :yy
				    :image-data image-data)
		     nil)))
	)
   )))
	
		 
	     
;;;  SUN$@$N%i%9%?!<%G!<%?(Jon/off$@$rCN$k(J
(defun sun-raster-bit (sun-raster-data width x y)
  (declare 
   #-CMU
   (inline foor + * mod logand ash -))
  (let ((data (elt sun-raster-data (floor (+ (* y width) x) 8))))
    (logand data (ash 1 (- 8 (mod x 8)))))
  )


;;; $@%P%$%H%"%l!<$+$i%$%a!<%8$r:n$k(J
;;; $@%"%l!<$N9=B$$O(Jwidth(bytes) * height$@$N%"%l!<$G$"$k!#(J
;;; $@Nc(J (make-array '(10 3) :element-type '(unsogned-byte 8))
;;; $@$3$N>l9g$G$"$k$HI}$O(J24$@!"9b$5$,(J10$@$N%$%a!<%8$,:n$i$l$k(J
;;; make-image-from-array array
(defun make-image-from-array (array)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let* ((dimension (array-dimensions array))
	 (width (second dimension))
	 (height (car dimension))
	 (image (make-array (* width height 2)
                           :element-type '(unsigned-byte 8)))
	 (count 0))
    (dotimes (y height)
      (dotimes (x width)
	(multiple-value-bind (f l) (make-yy-bit (aref array y x))
			    (setf (elt image count) f
				  (elt image (incf count)) l))
	(incf count)))

    (make-instance 'mono-image :image-width (* width 8)
		   :line-bytes (* width 2)
		   :image-height height
		   :image-type :mono :image-format :yy
		   :image-data image)
	))

(defun make-yy-bit (byte)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((mae (ash (logand #xF0 byte) -4))
	(usiro (logand #x0F byte)))
    (values (half-to-byte mae) (half-to-byte usiro))))
	

(defun half-to-byte (h-byte)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((temp 0) (mask #x08) (count 3)
	(ret 0))
    (dotimes (i 4)
      (setf temp (logand (ash mask (- 0 i)) h-byte)
	    ret (logior ret (ash temp count)))
      (decf count))
    ret))


;;; $@%U%!%$%k$+$i%/%j%C%T%s%0%j!<%8%g%s$G@Z$j$H$C$?%$%a!<%8%G!<%?$r@8@.(Jh
;;; load-image path-name &key clip-region background-color
;;; ARG.
;;;        path-name   =  $@%$%a!<%8%U%!%$%k$N%Q%9L>(J
;;;        clip-region =  $@%/%j%C%W%j!<%8%g%s(J
;;;        background-color = $@%+%i!<$,$J$$;~$N%G%U%)%k%H%+%i!<(J
;;; RET.
;;;       $@Bh0l(J : $@%$%a!<%8%$%s%9%?%s%9(J or NIL ($@%U%!%$%k$,$J$$(J)
;;;       $@BhFs(J : $@@8@.$G$-$J$+$C$?%+%i!<$N(JRGB$@$H:BI8CM(J
(defun load-image (path-name &key (clip-region (make-region))
			    (background-color *white-color*))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  
  (make-image-form-raster-file path-name 
			       :start-x (region-left clip-region)
			       :start-y (region-bottom clip-region)
			       :width (region-width clip-region)
			       :height (region-height clip-region)
			       :default-color background-color)
  )

;;; $@%$%a!<%8$r:n$k(J
;;; make-image width height &key color
;;; ARG.
;;;       width   =  $@%$%a!<%8$NI}(J
;;;       height  =  $@%$%a!<%8$N9b$5(J
;;;       color   =  $@%G%U%)%k%H%+%i!<(J
;;; RET.
;;;       $@%$%a!<%8%$%s%9%?%s%9(J ($@@8@.$5$l$k%+%i!<$N%?%$%W$O>o$K%+%i!<$G$"$k!#(J)
(defun make-image (width height &key (color *white-color*))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((image (make-array (* width height 4) 
			   :element-type '(unsigned-byte 8)))
	(count 0))
    (dotimes (i height)
       (dotimes (j width)
	   (setf (aref image count) (get-one-byte (color-no color) 3)
		 (aref image (incf count)) (get-one-byte (color-no color) 2)
		 (aref image (incf count)) (get-one-byte (color-no color) 1)
		 (aref image (incf count)) (get-one-byte (color-no color) 0))
	   (incf count)))

    (make-instance 'color-image :image-width width 
		   :line-bytes (* width 4)
		   :image-height height
		   :image-type :color :image-format :yy
		   :image-data image)))

	
;;; $@%$%a!<%8$NBg$-$5$r5a$a$k(J
;;; image-size image
;;; ARG.
;;;        image  =  $@%$%a!<%8%$%s%9%?%s%9(J
;;; RET
;;;        values width height
(defmethod image-size ((image image))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (values (image-width image ) (image-height image)))


;;; $@%$%a!<%8$N%+%i!<$r5a$a$k(J
;;; image-color image position
;;; ARG.
;;;       image    = $@%$%a!<%8%$%s%9%?%s%9(J
;;;       position = $@%]%8%7%g%s%$%s%9%?%s%9(J
;;; RET.
;;;       $@%+%i!<%$%s%9%?%s%9(J or NIL
(defmethod image-color ((image color-image) (position position))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (image-color-xy-internal-color image (position-x position) 
			    (position-y position)))
;;; $@%b%N%/%mMQ(J
(defmethod image-color ((image mono-image) (position position))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (image-color-xy-internal-mono image (position-x position) 
			    (position-y position)))
;;; $@%0%l!<%9%1!<%kMQ(J
(defmethod image-color ((image gray-image) (position position))
  (image-color-xy-internal-color image (position-x position) 
			    (position-y position)))
;--------------------------------------------------------------

;;; $@%$%a!<%8$N%+%i!<$r%;%C%H$9$k(J setf$@%a%=%C%I(J
;;; (setf (image-color image position) color)
;;; ARG.
;;;       image    = $@%$%a!<%8%$%s%9%?%s%9(J
;;;       position = $@%]%8%7%g%s%$%s%9%?%s%9(J
;;; RET.
;;;       $@%+%i!<%$%s%9%?%s%9(J
(defmethod (setf image-color) ((color color) (image color-image) 
			       (position position))
  (set-image-color-xy-internal-color image (position-x position) 
			    (position-y position) (color-no color)))
;;; $@%b%N%/%mMQ(J
(defmethod (setf image-color) ((color color) (image mono-image) 
			       (position position))
  (set-image-color-xy-internal-mono image (position-x position) 
			    (position-y position) color))
;;;$@%0%l!<%9%1!<%kMQ(J
(defmethod (setf image-color) ((color color) (image gray-image) 
			       (position position))
  (set-image-color-xy-internal-color image (position-x position) 
			    (position-y position) (color-no color)))
;------------------------------------------------------------------

;;; $@%$%a!<%8$N%+%i!<$r5a$a$k(J
;;; image-color image x y
;;; ARG.
;;;       image    = $@%$%a!<%8%$%s%9%?%s%9(J
;;;       x        = $@0LCV(J x
;;;       y        = $@0LCV(J y
;;; RET.
;;;       $@%+%i!<%$%s%9%?%s%9(J
(defmethod image-color-xy ((image color-image) (x integer) (y integer))
  (image-color-xy-internal-color image x y))
;;; $@%b%N%/%m(J
(defmethod image-color-xy ((image mono-image) (x integer) (y integer))
  (image-color-xy-internal-mono image x y))
;;; $@%0%l!<%9%1!<%k(J
(defmethod image-color-xy ((image gray-image) (x integer) (y integer))
  (image-color-xy-internal-color image x y))
;---------------------------------------------------------------------

;;; $@%$%a!<%8$N%+%i!<$r%;%C%H$9$k(J setf$@%a%=%C%I(J
;;;  (setf (image-color image x y) color)
;;; ARG.
;;;       image    = $@%$%a!<%8%$%s%9%?%s%9(J
;;;       x        = $@0LCV(J x
;;;       y        = $@0LCV(J y
;;; RET.
;;;       $@%+%i!<%$%s%9%?%s%9(J
(defmethod (setf image-color-xy) ((color color) (image color-image)
				  (x integer) (y integer))
  (set-image-color-xy-internal-color image x y (color-no color)))
;;; $@%b%N%/%m(J
(defmethod (setf image-color-xy) ((color color) (image mono-image)
				  (x integer) (y integer))
  (set-image-color-xy-internal-mono image x y color))
;;; $@%0%l!<%9%1!<%k(J
(defmethod (setf image-color-xy) ((color color) (image gray-image)
				  (x integer) (y integer))
  (set-image-color-xy-internal-color image x y (color-no color)))
;----------------------------------------------------------------

;;; image-color-xy-internal-color
;;; $@%$%s%?!<%J%k4X?t(J
(defun image-color-xy-internal-color (image x y)

  (declare 
   #-CMU
   (inline + * / incf logior ash mod <)  
   (special *black-color* *white-color* *transparent*)
   (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (if (and (< x (image-width image)) (< y (image-height image)))
      (let* ((count(+ (* y (line-bytes image)) (* x 4)))
	     (data (image-data image)))
	(code-color
	 (logior (ash (elt data count) 24)
		 (ash (elt data (incf count)) 16)
		 (ash (elt data (incf count)) 8)
		 (elt data (incf count)))))
    nil))
;;;image-color-xy-internal-mono
(defun image-color-xy-internal-mono (image x y)
  (declare (special *black-color* *white-color* *transparent*)
	   (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (if (and (< x (image-width image)) (< y (image-height image)))
      (case (yy-image-bit (image-data image) (line-bytes image) x y)
	    (1 *black-color*)
	    (0 *white-color*)
	    (2 *transparent*))
    nil))

;;; set-image-color-xy-internal-color
;;; $@@_Dj%$%J%?!<%J%k4X?t(J
(defun set-image-color-xy-internal-color (image x y cl-no)
    (declare 
   #-CMU
	 (inline + * / incf logior ash mod <)  
	 (special *white-color* *transparent*)
	 (optimize (compilation-speed 0) (speed 3) (safety 0)))
    (if (and (< x (image-width image)) (< y (image-height image)))
	(let* ((count(+ (* y (line-bytes image)) (* x 4))))
	  (setf (aref (image-data image) count) (get-one-byte cl-no 3)
		(aref (image-data image) (incf count)) (get-one-byte cl-no 2)
		(aref (image-data image) (incf count)) (get-one-byte cl-no 1)
		(aref (image-data image) (incf count)) (get-one-byte cl-no 0))
	  image)
      nil))

;;; set-image-color-xy-internal-mono
;;; $@@_Dj%$%J%?!<%J%k4X?t(J
(defun set-image-color-xy-internal-mono (image x y color)
  (declare (special *white-color* *transparent*)
	   (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (if (and (< x (image-width image)) (< y (image-height image)))
      (cond 
       ((eq color *white-color*)
	(setf (yy-image-bit (image-data image) (line-bytes image)
			    x y) #b00)
	image)
       ((eq color *transparent*)
	(setf (yy-image-bit (image-data image) (line-bytes image)
			    x y) #b10)
	image)
       (t
	(setf (yy-image-bit (image-data image) (line-bytes image)
			    x y) #b01)
	image))
    nil))


;;; $@%$%a!<%8$NGK2u(J
;;; flush-image image
;;; ARG.
;;;         image  =  $@%$%a!<%8%$%s%9%?%s%9(J
;;; RET.
;;;         nil
(defmethod flush-image ((image image))
  (setf image nil))


;;; $@%$%a!<%8$N%/%j%C%W(J
;;; clip-image image clip-region &key (create T)
;;; ARG.
;;;           image        = $@%/%j%C%W$5$l$k%$%a!<%8(J
;;;           clip-region  = $@%/%j%C%W$9$Y$-%j!<%8%g%s%$%s%9%?%s%9(J
;;;           create       = T or NIL 
;;;                          T   -> $@@8@.(J
;;;                          NIL -> $@GK2uE*$KJQ99(J
;;; RET.
;;;   $@%$%a!<%8%$%s%9%?%s%9(J
(defmethod clip-image ((image color-image) (clip-region region) 
			&key (create T))
  (clip-image-xy-internal-color image (region-left clip-region)
				(region-bottom clip-region)
				(region-width clip-region)
				(region-height clip-region)
				create))
;;; $@%b%N%/%mMQ(J
(defmethod clip-image ((image mono-image) (clip-region region) 
			&key (create T))

  (clip-image-internal-xy-mono 
                image (region-left clip-region)
		(region-bottom clip-region)
		(region-width clip-region)
		(region-height clip-region)
		create))


;;; $@%$%a!<%8$N%/%j%C%W(J
;;; clip-image-xy image x y width height &key (create T)
;;; ARG.
;;;           image        = $@%/%j%C%W$5$l$k%$%a!<%8(J
;;;       x y width height = $@%/%j%C%W$9$Y$-%j!<%8%g%s(J
;;;           create       = T or NIL 
;;;                          T   -> $@@8@.(J
;;;                          NIL -> $@GK2uE*$KJQ99(J
;;; RET.
;;;   $@%$%a!<%8%$%s%9%?%s%9(J
(defmethod clip-image-xy ((image color-image) 
			   (x integer) (y integer)
			   (width integer) (height integer)
			   &key (create T))
  (clip-image-xy-internal-color image x y width height create))

;;; $@%b%N%/%mMQ(J
(defmethod clip-image-xy ((image mono-image) (x integer) (y integer)
			(width integer) (height integer)
			&key (create T))

  (clip-image-internal-xy-mono 
                image x y width height create))

;;; $@%$%s%?!<%J%k4X?t(J
(defun clip-image-xy-internal-color (image x y width height create)
  (declare 
   #-CMU
   (inline > + = incf * -)
   (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (if (and (= x 0)
	   (= y 0)
	   (= width (image-width image))
	   (= height (image-height image)))
      (if create
	  (let ((new-data (make-array (* (image-width image)
					 (image-height image) 4)
				      :element-type '(unsigned-byte 8))))
	    (dotimes (i (length new-data))
	      (setf (elt new-data i) (elt (image-data image) i)))
	    (make-instance 'color-image :image-width 
			   (image-width image)
			   :image-height (image-height image)
			   :line-bytes (line-bytes image)
			   :image-data new-data))
	image)
    (let* ((data (image-data image))
	   (new-width (if (> (+ x width)
			     (image-width image))
		      (- (image-width image) x)
		    width))
	   (new-height (if (> (+ y height)
			     (image-height image))
		      (- (image-height image) y)
		      height))
	   (count 0)
	   (newcount 0)
	   (new-data (make-array (* new-width new-height 4) 
				 :element-type '(unsigned-byte 8))))
      (do ((yy y (incf yy)))
	  ((= yy (+ new-height y)))
	(do ((xx x (incf xx)))
	    ((= xx (+ new-width x)))

	    (setf count (+ (* (line-bytes image) yy) (* xx 4))
		  (elt new-data newcount) (elt data count)
		  (elt new-data (incf newcount)) (elt data (+ count 1))
		  (elt new-data (incf newcount)) (elt data (+ count 2))
		  (elt new-data (incf newcount)) (elt data (+ count 3)))
	    (incf newcount)))
      (if create
	  (make-instance 'color-image :image-width new-width 
			 :line-bytes (* new-width 4)
			 :image-height new-height :image-data new-data)
	(progn (setf (slot-value image 'image-width)  new-width
		     (line-bytes image) (* new-width 4)
		     (slot-value image 'image-height) new-height
		     (slot-value image 'image-data) new-data)
	       image))
      )))

;;; $@%$%s%?!<%J%k4X?t(J
(defun clip-image-internal-xy-mono (image x y width height create)
  (declare 
   #-CMU
   (inline > + = incf  -)
   (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (if (and (= x 0)
	   (= y 0)
	   (= width (image-width image))
	   (= height (image-height image)))
      (if create
	  (let ((new-data (make-array (length (image-data image))
				      :element-type '(unsigned-byte 8))))
	    (dotimes (i (length new-data))
	      (setf (elt new-data i) (elt (image-data image) i)))
	    (make-instance 'mono-image :image-width 
			   (image-width image)
			   :image-height (image-height image)
			   :line-bytes (line-bytes image)
			   :image-data new-data))
	image)
    (let* ((data (image-data image))
	   (new-width (if (> (+ x width)
			     (image-width image))
		      (- (image-width image) x)
		      width))
	   (new-height (if (> (+ y height)
			     (image-height image))
			  (- (image-height image) y)
			height))
	   (yy 0)
	   (xx 0)
	   (line-bytes (ceiling (/ new-width 4)))
	   (new-data (make-array (* line-bytes new-height) 
				 :element-type '(unsigned-byte 8))))
    
      (do ((y1 y (incf y1)))
	  ((= y1 (+ new-height y)))
	(do ((x1 x (incf x1)))
	    ((= x1 (+ new-width x)))
	  (setf (yy-image-bit new-data line-bytes xx yy)
	    (yy-image-bit data (line-bytes image) x1 y1))
	  (incf xx))
	(incf yy))

      (if create
	  (make-instance 'mono-image :image-width new-width 
			 :line-bytes (ceiling (/ new-width 4))
			 :image-height new-height :image-data new-data)
	(progn (setf (slot-value image 'image-width)  new-width
		     (slot-value image 'image-height) new-height
		     (line-bytes image) (ceiling (/ new-width 4))
		     (slot-value image 'image-data) new-data)
	       image))
      )))


;;; $@%$%a!<%8$r%U%!%$%k$NJ]B8(J
;;; save-image image path-name &key clip-region if-exist
;;; ARG.
;;;         image          = $@%$%a!<%8%$%s%9%?%s%9(J
;;;         path-name      = $@=q$-9~$`%U%!%$%k$N%Q%9L>(J
;;;         clip-region    = $@%/%j%C%W$9$kNN0h(J
;;;         if-exist       = Lisp$@$N(Jopen$@$K=>$&(J
;;; RET.
;;          Lisp$@$N(Jopen$@$HF1$8(J
(defmethod save-image ((image image) path-name 
		       &key (clip-region (make-region :width
						      (image-width image)
						      :height 
						      (image-height image)))
		       (if-exists :overwrite))
  (declare (special *white-color*)
	   (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-open-file (stream path-name :if-exists if-exists 
		   :direction :output
		   :if-does-not-exist :create :element-type '(unsigned-byte 8))
    (let ((new-image (if clip-region
			 (clip-image image clip-region :create T)
		       image))
	  (color *white-color*)
	  (no 0))
		  
    (write-long-w-680x0-order 14876 stream)
    (case (image-type image)
       (:color
	(write-long-w-680x0-order 1 stream))
       (:mono 
	(write-long-w-680x0-order 3 stream))
       (t
	(write-long-w-680x0-order 2 stream)))
    (write-long-w-680x0-order (image-width new-image) stream)
    (write-long-w-680x0-order (image-height new-image) stream)
    (case (image-type image)
      (:color
       (do ((i 0 (incf i 4)))
	   ((= i (length (image-data new-image))))
	   
	 (setf no (logior (ash (elt (image-data new-image) i) 24)
			  (ash (elt (image-data new-image) (+ 1 i)) 16)
			  (ash (elt (image-data new-image) (+ i 2)) 8)
			  (elt (image-data new-image) (+ i 3)))
	       color (code-color no))
	 (write-long-w-680x0-order (red color) stream)
	 (write-long-w-680x0-order (green color) stream)
	 (write-long-w-680x0-order (blue color) stream)))
      (t
       (dotimes (i (length (image-data new-image)))
	 (lisp:write-byte (elt (image-data new-image) i) stream))))
       
      
    )))

;;; End of file


   
